<?php
/**
 * Neue Sidebar - Komplett neuer Aufbau
 * Basierend auf Figma Design mit neuer Logik
 */

// Session sicherstellen
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Aktuelle Seite ermitteln
$current_page = basename($_SERVER['PHP_SELF'], '.php');
$current_user_name = $current_user_name ?? 'Administrator';
$current_user_role = $current_user_role ?? 'admin';

// User-Daten aus Datenbank laden (für Avatar/Initialien)
$current_user_avatar = null;
$current_user_email = null;
$current_user_full_name = null;
$user_initials = 'AD'; // Default

if (isset($_SESSION['admin_user_id'])) {
    try {
        require_once __DIR__ . '/../../config/database.php';
        $db = new Database();
        $conn = $db->getConnection();
        
        $stmt = $conn->prepare("SELECT username, email, avatar, full_name FROM admin_users WHERE id = ?");
        $stmt->execute([$_SESSION['admin_user_id']]);
        $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($user_data) {
            $avatar_path = $user_data['avatar'] ?? null;
            
            // Avatar-Pfad korrigieren und prüfen
            if ($avatar_path && !empty(trim($avatar_path))) {
                // Entferne nur führende ../ für file_exists Prüfung
                $check_path = ltrim($avatar_path, './');
                if (strpos($check_path, '../') === 0) {
                    $check_path = substr($check_path, 3); // Entferne ../
                }
                
                // Vollständiger Pfad für Existenzprüfung (vom admin/includes-Verzeichnis aus)
                // __DIR__ ist admin/includes, also muss ich ein Verzeichnis zurück
                $full_check_path = __DIR__ . '/../' . $check_path;
                
                // Prüfe ob Datei existiert
                if (file_exists($full_check_path)) {
                    // Pfad für HTML - normalisiere den Pfad
                    // Der Pfad sollte relativ zum admin-Verzeichnis sein
                    if (strpos($avatar_path, '../') === 0) {
                        // Pfad beginnt bereits mit ../, verwende direkt
                        $current_user_avatar = $avatar_path;
                    } else {
                        // Pfad beginnt nicht mit ../, füge es hinzu
                        $current_user_avatar = '../' . $avatar_path;
                    }
                } else {
                    // Datei existiert nicht, setze auf null
                    $current_user_avatar = null;
                }
            } else {
                $current_user_avatar = null;
            }
            
            $current_user_email = $user_data['email'] ?? ($current_user_name . '@boxenstop.de');
            $current_user_full_name = $user_data['full_name'] ?? $current_user_name;
            
            // Initialien generieren
            if ($current_user_full_name) {
                $name_parts = explode(' ', trim($current_user_full_name));
                if (count($name_parts) >= 2) {
                    $user_initials = strtoupper(substr($name_parts[0], 0, 1) . substr($name_parts[count($name_parts) - 1], 0, 1));
                } else {
                    $user_initials = strtoupper(substr($current_user_full_name, 0, 2));
                }
            } else {
                $user_initials = strtoupper(substr($current_user_name, 0, 2));
            }
        }
    } catch (Exception $e) {
        // Fehler ignorieren, verwende Defaults
    }
}

/**
 * Zentrale Funktion zum Laden aller Badge-Zähler (Volvo-konform)
 * Erweiterbar für alle Sidebar-Menüpunkte
 */
function loadSidebarBadgeCounts($conn, $current_user_id, $current_user_role, $current_user_autohaus_id) {
    $badge_counts = [
        'feedback' => 0,
        'appointments' => 0, // Anzahl der Termine im nächsten Event
        'appointments_all' => 0,
        'appointments_today' => 0,
        'appointments_week' => 0,
        'appointments_upcoming' => 0,
        'appointments_past' => 0,
        'appointments_completed' => 0,
        'appointments_cancelled' => 0,
        'appointments_pending' => 0,
        'appointments_approved' => 0,
        'notifications' => 0,
        'events' => 0,
        'whatsapp' => 0,
        'users_pending' => 0
    ];
    
    if (!$conn || !$current_user_id) {
        return $badge_counts;
    }
    
    // Autohaus-Filter für Queries
    $autohaus_filter = "";
    $autohaus_params = [];
    if ($current_user_autohaus_id && $current_user_role !== 'super_admin') {
        $autohaus_filter = " AND a.autohaus_id = ?";
        $autohaus_params[] = $current_user_autohaus_id;
    }
    
    try {
        // 1. Feedback-Badge: Neue Feedbacks
        try {
            $checkTable = $conn->query("SHOW TABLES LIKE 'feedback'");
            if ($checkTable->rowCount() > 0) {
                if ($current_user_autohaus_id && $current_user_role !== 'super_admin') {
                    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM feedback WHERE status = 'new' AND autohaus_id = ?");
                    $stmt->execute([$current_user_autohaus_id]);
                } else {
                    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM feedback WHERE status = 'new'");
                    $stmt->execute();
                }
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                $badge_counts['feedback'] = intval($result['count'] ?? 0);
            }
        } catch (Exception $e) {
            error_log("Fehler beim Laden der Feedback-Anzahl: " . $e->getMessage());
        }
        
        // 2. Termine-Badge: Anzahl der Termine im nächsten Event
        try {
            $checkTable = $conn->query("SHOW TABLES LIKE 'appointments'");
            if ($checkTable->rowCount() > 0) {
                // Finde das nächste Event
                $nextEventQuery = "
                    SELECT id 
                    FROM events 
                    WHERE event_date >= CURDATE() 
                    AND is_active = 1 
                    ORDER BY event_date ASC 
                    LIMIT 1
                ";
                $nextEventParams = [];
                
                // Prüfe ob Events-Tabelle autohaus_id hat
                try {
                    $checkEventsAutohaus = $conn->query("SHOW COLUMNS FROM events LIKE 'autohaus_id'");
                    if ($checkEventsAutohaus->rowCount() > 0 && $current_user_autohaus_id && $current_user_role !== 'super_admin') {
                        $nextEventQuery = "
                            SELECT id 
                            FROM events 
                            WHERE event_date >= CURDATE() 
                            AND is_active = 1 
                            AND autohaus_id = ?
                            ORDER BY event_date ASC 
                            LIMIT 1
                        ";
                        $nextEventParams[] = $current_user_autohaus_id;
                    }
                } catch (Exception $e) {
                    // Spalte existiert nicht
                }
                
                $stmt = $conn->prepare($nextEventQuery);
                $stmt->execute($nextEventParams);
                $nextEvent = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($nextEvent && isset($nextEvent['id'])) {
                    // Zähle Termine im nächsten Event
                    $appointmentsInNextEventQuery = "
                        SELECT COUNT(*) as count 
                        FROM appointments a
                        WHERE a.event_id = ?
                    " . $autohaus_filter;
                    
                    $params = [$nextEvent['id']];
                    if (!empty($autohaus_params)) {
                        $params = array_merge($params, $autohaus_params);
                    }
                    
                    $stmt = $conn->prepare($appointmentsInNextEventQuery);
                    $stmt->execute($params);
                    $result = $stmt->fetch(PDO::FETCH_ASSOC);
                    $badge_counts['appointments'] = intval($result['count'] ?? 0);
                }
                
                // Alle Termine
                $allQuery = "SELECT COUNT(*) as count FROM appointments a WHERE 1=1" . $autohaus_filter;
                $stmt = $conn->prepare($allQuery);
                $stmt->execute($autohaus_params);
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                $badge_counts['appointments_all'] = intval($result['count'] ?? 0);
                
                // Heute
                $todayQuery = "SELECT COUNT(*) as count FROM appointments a WHERE a.slot_date = CURDATE()" . $autohaus_filter;
                $stmt = $conn->prepare($todayQuery);
                $stmt->execute($autohaus_params);
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                $badge_counts['appointments_today'] = intval($result['count'] ?? 0);
                
                // Diese Woche
                $weekQuery = "
                    SELECT COUNT(*) as count 
                    FROM appointments a 
                    WHERE a.slot_date >= DATE_SUB(CURDATE(), INTERVAL WEEKDAY(CURDATE()) DAY)
                    AND a.slot_date < DATE_ADD(CURDATE(), INTERVAL 7 - WEEKDAY(CURDATE()) DAY)
                " . $autohaus_filter;
                $stmt = $conn->prepare($weekQuery);
                $stmt->execute($autohaus_params);
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                $badge_counts['appointments_week'] = intval($result['count'] ?? 0);
                
                // Kommende Termine
                $upcomingQuery = "SELECT COUNT(*) as count FROM appointments a WHERE a.slot_date > CURDATE()" . $autohaus_filter;
                $stmt = $conn->prepare($upcomingQuery);
                $stmt->execute($autohaus_params);
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                $badge_counts['appointments_upcoming'] = intval($result['count'] ?? 0);
                
                // Vergangene Termine
                $pastQuery = "SELECT COUNT(*) as count FROM appointments a WHERE a.slot_date < CURDATE()" . $autohaus_filter;
                $stmt = $conn->prepare($pastQuery);
                $stmt->execute($autohaus_params);
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                $badge_counts['appointments_past'] = intval($result['count'] ?? 0);
                
                // Abgeschlossen
                $completedQuery = "SELECT COUNT(*) as count FROM appointments a WHERE a.status = 'completed'" . $autohaus_filter;
                $stmt = $conn->prepare($completedQuery);
                $stmt->execute($autohaus_params);
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                $badge_counts['appointments_completed'] = intval($result['count'] ?? 0);
                
                // Storniert
                $cancelledQuery = "SELECT COUNT(*) as count FROM appointments a WHERE a.status = 'cancelled'" . $autohaus_filter;
                $stmt = $conn->prepare($cancelledQuery);
                $stmt->execute($autohaus_params);
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                $badge_counts['appointments_cancelled'] = intval($result['count'] ?? 0);
                
                // Wartend
                $pendingQuery = "SELECT COUNT(*) as count FROM appointments a WHERE a.status = 'pending'" . $autohaus_filter;
                $stmt = $conn->prepare($pendingQuery);
                $stmt->execute($autohaus_params);
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                $badge_counts['appointments_pending'] = intval($result['count'] ?? 0);
                
                // Genehmigt
                $approvedQuery = "SELECT COUNT(*) as count FROM appointments a WHERE a.status = 'approved'" . $autohaus_filter;
                $stmt = $conn->prepare($approvedQuery);
                $stmt->execute($autohaus_params);
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                $badge_counts['appointments_approved'] = intval($result['count'] ?? 0);
            }
        } catch (Exception $e) {
            error_log("Fehler beim Laden der Termine-Anzahl: " . $e->getMessage());
        }
        
        // 3. Benachrichtigungen-Badge: Neue Benachrichtigungen
        try {
            $checkTable = $conn->query("SHOW TABLES LIKE 'notifications'");
            if ($checkTable->rowCount() > 0) {
                $stmt = $conn->prepare("
                    SELECT COUNT(*) as count 
                    FROM notifications 
                    WHERE (is_read = 0 OR is_read IS NULL)
                    AND user_id = ?
                ");
                $stmt->execute([$current_user_id]);
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                $badge_counts['notifications'] = intval($result['count'] ?? 0);
            }
        } catch (Exception $e) {
            // Tabelle existiert möglicherweise nicht - kein Fehler
        }
        
        // 4. Events-Badge: Kommende Events (nächste 7 Tage)
        try {
            $checkTable = $conn->query("SHOW TABLES LIKE 'events'");
            if ($checkTable->rowCount() > 0) {
                $stmt = $conn->prepare("
                    SELECT COUNT(*) as count 
                    FROM events 
                    WHERE event_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 7 DAY)
                    AND is_active = 1
                ");
                $stmt->execute();
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                $badge_counts['events'] = intval($result['count'] ?? 0);
            }
        } catch (Exception $e) {
            error_log("Fehler beim Laden der Events-Anzahl: " . $e->getMessage());
        }
        
        // 5. WhatsApp-Badge: Fehlgeschlagene Nachrichten oder neue Logs
        try {
            $checkTable = $conn->query("SHOW TABLES LIKE 'whatsapp_logs'");
            if ($checkTable->rowCount() > 0) {
                // Zähle fehlgeschlagene Nachrichten der letzten 24 Stunden
                $stmt = $conn->prepare("
                    SELECT COUNT(*) as count 
                    FROM whatsapp_logs 
                    WHERE status = 'failed' 
                    AND created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
                ");
                $stmt->execute();
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                $badge_counts['whatsapp'] = intval($result['count'] ?? 0);
            }
        } catch (Exception $e) {
            // Tabelle existiert möglicherweise nicht - kein Fehler
        }
        
        // 6. Benutzer-Badge: Ausstehende Benutzer (Status = 'pending')
        try {
            $checkTable = $conn->query("SHOW TABLES LIKE 'users'");
            if ($checkTable->rowCount() > 0) {
                $stmt = $conn->prepare("
                    SELECT COUNT(*) as count 
                    FROM users 
                    WHERE status = 'pending'
                ");
                $stmt->execute();
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                $badge_counts['users_pending'] = intval($result['count'] ?? 0);
            }
        } catch (Exception $e) {
            error_log("Fehler beim Laden der ausstehenden Benutzer-Anzahl: " . $e->getMessage());
        }
        
    } catch (Exception $e) {
        error_log("Fehler beim Laden der Badge-Zähler: " . $e->getMessage());
    }
    
    return $badge_counts;
}

// Badge-Zähler laden
$badge_counts = [
    'feedback' => 0,
    'appointments' => 0, // Anzahl der Termine im nächsten Event
    'appointments_all' => 0,
    'appointments_today' => 0,
    'appointments_week' => 0,
    'appointments_upcoming' => 0,
    'appointments_past' => 0,
    'appointments_completed' => 0,
    'appointments_cancelled' => 0,
    'appointments_pending' => 0,
    'appointments_approved' => 0,
    'notifications' => 0,
    'events' => 0,
    'whatsapp' => 0,
    'users_pending' => 0
];
$current_user_autohaus_id = null;
$notifications_count = 0;
$all_notifications = [];

if (isset($_SESSION['admin_user_id'])) {
    try {
        require_once __DIR__ . '/../../config/database.php';
        $db = new Database();
        $conn = $db->getConnection();
        
        // Lade Autohaus-ID des aktuellen Admins
        $stmt = $conn->prepare("SELECT autohaus_id, role FROM admin_users WHERE id = ?");
        $stmt->execute([$_SESSION['admin_user_id']]);
        $admin_data = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($admin_data) {
            $current_user_autohaus_id = $admin_data['autohaus_id'];
            $current_user_role = $admin_data['role'] ?? $current_user_role;
        }
        
        // Lade alle Badge-Zähler
        $badge_counts = loadSidebarBadgeCounts($conn, $_SESSION['admin_user_id'], $current_user_role, $current_user_autohaus_id);
        
        // Lade Benachrichtigungen für Tooltip
        try {
            // Get upcoming appointments in next 24 hours - gefiltert nach Autohaus
            $appointments_query = "
                SELECT COUNT(*) as count
                FROM appointments 
                WHERE slot_date = DATE_ADD(CURDATE(), INTERVAL 1 DAY)
                AND status IN ('booked', 'vorbereitet', 'completed')
            ";
            $appointments_params = [];
            
            if ($current_user_autohaus_id && $current_user_role !== 'super_admin') {
                $appointments_query .= " AND autohaus_id = ?";
                $appointments_params[] = $current_user_autohaus_id;
            }
            
            $stmt = $conn->prepare($appointments_query);
            $stmt->execute($appointments_params);
            $upcoming_appointments_result = $stmt->fetch(PDO::FETCH_ASSOC);
            $upcoming_appointments_count = intval($upcoming_appointments_result['count'] ?? 0);
            $notifications_count += $upcoming_appointments_count;
            
            // Get detailed appointment data
            $upcoming_appointments_data = [];
            if ($upcoming_appointments_count > 0) {
                $appointments_detail_query = "
                    SELECT 
                        id,
                        customer_name, 
                        license_plate, 
                        slot_time,
                        slot_date,
                        status,
                        'appointment' as notification_type
                    FROM appointments 
                    WHERE slot_date = DATE_ADD(CURDATE(), INTERVAL 1 DAY)
                    AND status IN ('booked', 'vorbereitet', 'completed')
                ";
                $appointments_detail_params = [];
                
                if ($current_user_autohaus_id && $current_user_role !== 'super_admin') {
                    $appointments_detail_query .= " AND autohaus_id = ?";
                    $appointments_detail_params[] = $current_user_autohaus_id;
                }
                
                $appointments_detail_query .= " ORDER BY slot_time ASC";
                $stmt = $conn->prepare($appointments_detail_query);
                $stmt->execute($appointments_detail_params);
                $upcoming_appointments_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
            }
            
            // Get events starting soon
            $events_query = "
                SELECT COUNT(*) as count
                FROM events 
                WHERE event_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 7 DAY)
                AND is_active = 1
            ";
            $events_params = [];
            
            // Prüfe ob Events-Tabelle autohaus_id hat
            try {
                $check_events_autohaus = $conn->query("SHOW COLUMNS FROM events LIKE 'autohaus_id'");
                if ($check_events_autohaus->rowCount() > 0 && $current_user_autohaus_id && $current_user_role !== 'super_admin') {
                    $events_query .= " AND autohaus_id = ?";
                    $events_params[] = $current_user_autohaus_id;
                }
            } catch (Exception $e) {
                // Spalte existiert nicht, keine Filterung
            }
            
            $stmt = $conn->prepare($events_query);
            $stmt->execute($events_params);
            $events_soon_result = $stmt->fetch(PDO::FETCH_ASSOC);
            $events_soon_count = intval($events_soon_result['count'] ?? 0);
            $notifications_count += $events_soon_count;
            
            // Get detailed event data
            $events_soon_data = [];
            if ($events_soon_count > 0) {
                $events_detail_query = "
                    SELECT 
                        id, 
                        name, 
                        event_date,
                        'event' as notification_type
                    FROM events 
                    WHERE event_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 7 DAY)
                    AND is_active = 1
                ";
                $events_detail_params = [];
                
                try {
                    $check_events_autohaus = $conn->query("SHOW COLUMNS FROM events LIKE 'autohaus_id'");
                    if ($check_events_autohaus->rowCount() > 0 && $current_user_autohaus_id && $current_user_role !== 'super_admin') {
                        $events_detail_query .= " AND autohaus_id = ?";
                        $events_detail_params[] = $current_user_autohaus_id;
                    }
                } catch (Exception $e) {
                    // Spalte existiert nicht
                }
                
                $events_detail_query .= " ORDER BY event_date ASC";
                $stmt = $conn->prepare($events_detail_query);
                $stmt->execute($events_detail_params);
                $events_soon_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
            }
            
            // Kombiniere alle Benachrichtigungen
            $all_notifications = [];
            if (!empty($upcoming_appointments_data)) {
                foreach ($upcoming_appointments_data as $apt) {
                    $all_notifications[] = $apt;
                }
            }
            if (!empty($events_soon_data)) {
                foreach ($events_soon_data as $evt) {
                    $all_notifications[] = $evt;
                }
            }
            
            // Sortiere nach Datum/Zeit
            usort($all_notifications, function($a, $b) {
                $dateA = $a['notification_type'] === 'appointment' 
                    ? ($a['slot_date'] . ' ' . $a['slot_time']) 
                    : $a['event_date'];
                $dateB = $b['notification_type'] === 'appointment' 
                    ? ($b['slot_date'] . ' ' . $b['slot_time']) 
                    : $b['event_date'];
                return strtotime($dateA) - strtotime($dateB);
            });
        } catch (Exception $e) {
            error_log("Fehler beim Laden der Benachrichtigungen: " . $e->getMessage());
        }
        
    } catch (Exception $e) {
        error_log("Fehler beim Initialisieren der Badge-Zähler: " . $e->getMessage());
    }
}

// Seiten-spezifische Sidebar-Klasse
$sidebar_class = isset($sidebar_class) ? $sidebar_class : '';

/**
 * Navigation Items mit Submenüs
 * 
 * BADGE-SYSTEM VERWENDUNG:
 * ========================
 * Um ein Badge zu einem Menüpunkt hinzuzufügen, fügen Sie einfach 'badge_count' hinzu:
 * 
 * Beispiel 1: Badge aus zentraler Funktion
 * [
 *     'icon' => 'fa-comment-dots',
 *     'text' => 'Feedback',
 *     'href' => 'feedback.php',
 *     'page' => 'feedback',
 *     'tooltip' => 'Feedback verwalten',
 *     'badge_count' => $badge_counts['feedback']  // Verwendet zentrale Badge-Zähler
 * ]
 * 
 * Beispiel 2: Direkter Wert
 * [
 *     'icon' => 'fa-bell',
 *     'text' => 'Benachrichtigungen',
 *     'badge_count' => 5  // Fester Wert
 * ]
 * 
 * Beispiel 3: Custom-Berechnung
 * [
 *     'icon' => 'fa-custom',
 *     'text' => 'Custom',
 *     'badge_count' => getBadgeCount('custom', function() {
 *         // Ihre Custom-Logik hier
 *         return 10;
 *     })
 * ]
 * 
 * WICHTIG:
 * - Badge wird nur angezeigt, wenn badge_count > 0
 * - Bei Werten > 99 wird "99+" angezeigt
 * - Funktioniert auch in Submenüs
 * - Volvo-konformes Design wird automatisch angewendet
 */
// Navigation Items mit Submenüs
$nav_items = [
    'hauptbereich' => [
        'title' => 'HAUPTBEREICH',
        'items' => [
            [
                'icon' => 'fa-home',
                'text' => 'Dashboard',
                'href' => 'index.php',
                'page' => 'index',
                'tooltip' => 'Dashboard'
            ],
            [
                'icon' => 'fa-users',
                'text' => 'Kunden',
                'href' => 'admin_dashboard.php',
                'page' => ['admin_dashboard', 'p2_customers'],
                'tooltip' => 'Kunden',
                'badge_count' => $badge_counts['users_pending'],
                'submenu' => [
                    ['icon' => 'fa-users', 'text' => 'Registrierte Benutzer', 'href' => 'admin_dashboard.php', 'page' => 'admin_dashboard', 'badge_count' => $badge_counts['users_pending']],
                    ['icon' => 'fa-user-friends', 'text' => 'P2 Kunden', 'href' => 'p2_customers.php', 'page' => 'p2_customers']
                ]
            ],
            [
                'icon' => 'fa-file-alt',
                'text' => 'Events',
                'href' => 'events.php',
                'page' => 'events',
                'tooltip' => 'Events',
                'badge_count' => $badge_counts['events']
            ],
            [
                'icon' => 'fa-calendar',
                'text' => 'Termine',
                'href' => 'appointments.php',
                'page' => 'appointments',
                'tooltip' => 'Termine',
                'badge_count' => $badge_counts['appointments'], // Anzahl der Termine im nächsten Event
                'submenu' => [
                    ['icon' => 'fa-list', 'text' => 'Alle Termine', 'href' => 'appointments.php', 'page' => 'appointments', 'badge_count' => $badge_counts['appointments_all']],
                    ['icon' => 'fa-calendar-day', 'text' => 'Heute', 'href' => 'appointments.php?filter=today', 'page' => 'appointments', 'badge_count' => $badge_counts['appointments_today']],
                    ['icon' => 'fa-calendar-week', 'text' => 'Diese Woche', 'href' => 'appointments.php?filter=week', 'page' => 'appointments', 'badge_count' => $badge_counts['appointments_week']],
                    ['icon' => 'fa-calendar-alt', 'text' => 'Kommende Termine', 'href' => 'appointments.php?filter=upcoming', 'page' => 'appointments', 'badge_count' => $badge_counts['appointments_upcoming']],
                    ['icon' => 'fa-history', 'text' => 'Vergangene Termine', 'href' => 'appointments.php?filter=past', 'page' => 'appointments', 'badge_count' => $badge_counts['appointments_past']],
                    ['icon' => 'fa-check-circle', 'text' => 'Abgeschlossen', 'href' => 'appointments.php?filter=completed', 'page' => 'appointments', 'badge_count' => $badge_counts['appointments_completed']],
                    ['icon' => 'fa-times-circle', 'text' => 'Storniert', 'href' => 'appointments.php?filter=cancelled', 'page' => 'appointments', 'badge_count' => $badge_counts['appointments_cancelled']],
                    ['icon' => 'fa-clock', 'text' => 'Wartend', 'href' => 'appointments.php?filter=pending', 'page' => 'appointments', 'badge_count' => $badge_counts['appointments_pending']],
                    ['icon' => 'fa-check', 'text' => 'Genehmigt', 'href' => 'appointments.php?filter=approved', 'page' => 'appointments', 'badge_count' => $badge_counts['appointments_approved']]
                ]
            ],
            [
                'icon' => 'fa-chart-bar',
                'text' => 'Reifen & Lager',
                'href' => 'storage.php',
                'page' => ['storage', 'tire_offers'],
                'tooltip' => 'Reifen & Lager',
                'submenu' => [
                    ['icon' => 'fa-warehouse', 'text' => 'RESY-Reifenlager', 'href' => 'storage.php', 'page' => 'storage'],
                    ['icon' => 'fa-tags', 'text' => 'Reifenangebote', 'href' => 'tire_offers.php', 'page' => 'tire_offers']
                ]
            ]
        ]
    ],
    'whatsapp' => [
        'title' => 'WHATSAPP',
        'items' => [
            [
                'icon' => 'fab fa-whatsapp',
                'text' => 'WhatsApp',
                'href' => 'hellomateo.php',
                'page' => 'hellomateo',
                'tooltip' => 'WhatsApp Management',
                'badge_count' => $badge_counts['whatsapp'],
                'submenu' => [
                    ['icon' => 'fab fa-whatsapp', 'text' => 'HelloMateo', 'href' => 'hellomateo.php', 'page' => 'hellomateo'],
                    ['icon' => 'fa-bell', 'text' => 'Template-Zuweisungen', 'href' => 'whatsapp_reminders_admin.php', 'page' => 'whatsapp_reminders_admin'],
                    ['icon' => 'fa-list-alt', 'text' => 'Logs', 'href' => 'whatsapp_logs.php', 'page' => 'whatsapp_logs'],
                    ['icon' => 'fa-vial', 'text' => 'Aktionen testen', 'href' => 'test_whatsapp_actions_ui.php', 'page' => 'test_whatsapp_actions_ui']
                ]
            ]
        ]
    ],
    'einstellungen' => [
        'title' => 'EINSTELLUNGEN',
        'items' => [
            [
                'icon' => 'fa-bell',
                'text' => 'Benachrichtigungen',
                'href' => 'notifications.php',
                'page' => 'notifications',
                'tooltip' => 'Benachrichtigungen',
                'badge_count' => $badge_counts['notifications']
            ],
            [
                'icon' => 'fa-comment-dots',
                'text' => 'Feedback',
                'href' => 'feedback.php',
                'page' => 'feedback',
                'tooltip' => 'Feedback verwalten',
                'badge_count' => $badge_counts['feedback']
            ],
            [
                'icon' => 'fa-cog',
                'text' => 'Einstellungen',
                'href' => 'settings.php',
                'page' => ['settings', 'api_manager', 'users'],
                'tooltip' => 'Einstellungen',
                'submenu' => [
                    ['icon' => 'fa-cog', 'text' => 'Allgemein', 'href' => 'settings.php', 'page' => 'settings'],
                    ['icon' => 'fa-key', 'text' => 'API Manager', 'href' => 'api_manager.php', 'page' => 'api_manager'],
                    ['icon' => 'fa-users-cog', 'text' => 'Mitarbeiter', 'href' => 'users.php', 'page' => 'users']
                ]
            ]
        ]
    ]
];

// Funktion zum Prüfen ob Seite aktiv ist
function isPageActive($page, $current_page) {
    if (is_array($page)) {
        return in_array($current_page, $page);
    }
    return $page === $current_page;
}

/**
 * Hilfsfunktion: Badge-Count für Nav-Item setzen
 * 
 * Verwendung in $nav_items:
 * 'badge_count' => getBadgeCount('feedback') // Verwendet $badge_counts['feedback']
 * 'badge_count' => 5 // Direkter Wert
 * 'badge_count' => getBadgeCount('custom', function() { return 10; }) // Custom Funktion
 * 
 * @param string $key - Schlüssel aus $badge_counts Array
 * @param callable|null $customFunction - Optionale Custom-Funktion für spezielle Zähler
 * @return int
 */
function getBadgeCount($key, $customFunction = null) {
    global $badge_counts;
    
    if ($customFunction && is_callable($customFunction)) {
        try {
            return intval($customFunction());
        } catch (Exception $e) {
            error_log("Fehler in Badge-Count-Funktion: " . $e->getMessage());
            return 0;
        }
    }
    
    return intval($badge_counts[$key] ?? 0);
}

// Funktion zum Prüfen ob Submenu offen sein soll
function isSubmenuOpen($item, $current_page) {
    if (!isset($item['submenu'])) {
        return false;
    }
    foreach ($item['submenu'] as $subitem) {
        if (isPageActive($subitem['page'], $current_page)) {
            return true;
        }
    }
    return false;
}
?>

<!-- Neue Sidebar -->
<aside class="new-sidebar <?php echo htmlspecialchars($sidebar_class); ?>" id="newSidebar">
    <!-- Sidebar Header -->
    <div class="sidebar-header">
        <div class="logo-container">
            <img src="../../assets/images/BE-Logo-FINAL_Schwarz-1.png" alt="Boxenstop" class="sidebar-logo">
            <span class="logo-text">Boxenstop</span>
        </div>
        <div class="sidebar-header-actions">
            <!-- Benachrichtigungen Glocke -->
            <button class="sidebar-notifications-btn" id="sidebarNotificationsBtn" data-tooltip="Benachrichtigungen" aria-label="Benachrichtigungen">
                <i class="fas fa-bell"></i>
                <?php if ($notifications_count > 0): ?>
                    <span class="sidebar-notification-badge"><?php echo $notifications_count > 99 ? '99+' : $notifications_count; ?></span>
                <?php endif; ?>
            </button>
            <button class="sidebar-toggle-btn" id="sidebarToggle" aria-label="Sidebar ein-/ausklappen">
                <i class="fas fa-chevron-left"></i>
            </button>
        </div>
    </div>

    <!-- Navigation -->
    <nav class="sidebar-navigation">
        <?php foreach ($nav_items as $section_key => $section): ?>
            <div class="nav-section">
                <div class="nav-section-header">
                    <span class="nav-section-title"><?php echo htmlspecialchars($section['title']); ?></span>
                </div>
                <div class="nav-items-list">
                    <?php foreach ($section['items'] as $item): ?>
                        <?php 
                        $is_active = isPageActive($item['page'], $current_page);
                        $has_submenu = isset($item['submenu']);
                        $submenu_open = isSubmenuOpen($item, $current_page);
                        ?>
                        <div class="nav-item-wrapper <?php echo $has_submenu ? 'has-submenu' : ''; ?>">
                            <a href="<?php echo htmlspecialchars($item['href']); ?>" 
                               class="nav-item <?php echo $is_active ? 'active' : ''; ?> <?php echo $submenu_open ? 'submenu-open' : ''; ?>" 
                               data-tooltip="<?php echo htmlspecialchars($item['tooltip']); ?>"
                               <?php if ($has_submenu): ?>data-has-submenu="true"<?php endif; ?>>
                                <i class="fas <?php echo htmlspecialchars($item['icon']); ?> nav-item-icon"></i>
                                <span class="nav-item-text"><?php echo htmlspecialchars($item['text']); ?></span>
                                <?php if (isset($item['badge_count']) && $item['badge_count'] > 0): ?>
                                    <span class="nav-item-badge <?php echo ($item['page'] === 'appointments' && isset($item['submenu'])) ? 'info-badge' : ''; ?>" data-count="<?php echo intval($item['badge_count']); ?>">
                                        <?php echo $item['badge_count'] > 99 ? '99+' : intval($item['badge_count']); ?>
                                    </span>
                                <?php endif; ?>
                                <?php if ($has_submenu): ?>
                                    <i class="fas <?php echo $submenu_open ? 'fa-chevron-up' : 'fa-chevron-down'; ?> nav-item-chevron"></i>
                                <?php endif; ?>
                            </a>
                            <?php if ($has_submenu): ?>
                                <div class="nav-submenu <?php echo $submenu_open ? 'is-open' : ''; ?>">
                                    <?php foreach ($item['submenu'] as $subitem): ?>
                                        <a href="<?php echo htmlspecialchars($subitem['href']); ?>" 
                                           class="nav-submenu-item <?php echo isPageActive($subitem['page'], $current_page) ? 'active' : ''; ?>">
                                            <?php if (isset($subitem['icon'])): ?>
                                                <i class="fas <?php echo htmlspecialchars($subitem['icon']); ?> nav-submenu-icon"></i>
                                            <?php endif; ?>
                                            <span><?php echo htmlspecialchars($subitem['text']); ?></span>
                                            <?php if (isset($subitem['badge_count']) && $subitem['badge_count'] > 0): ?>
                                                <span class="nav-submenu-badge" data-count="<?php echo intval($subitem['badge_count']); ?>">
                                                    <?php echo $subitem['badge_count'] > 99 ? '99+' : intval($subitem['badge_count']); ?>
                                                </span>
                                            <?php endif; ?>
                                        </a>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php endforeach; ?>
    </nav>

    <!-- Sidebar Footer - Neue Struktur -->
    <div class="sidebar-footer">
        <div class="user-profile-wrapper">
            <button class="user-profile-button" id="userProfileBtn" type="button" aria-expanded="false" aria-haspopup="true">
                <div class="user-avatar-wrapper">
                    <div class="user-avatar-circle">
                        <?php if ($current_user_avatar): ?>
                            <img src="<?php echo htmlspecialchars($current_user_avatar); ?>" alt="<?php echo htmlspecialchars($current_user_name); ?>" class="user-avatar-image">
                        <?php else: ?>
                            <span class="user-avatar-initials"><?php echo htmlspecialchars($user_initials); ?></span>
                        <?php endif; ?>
                        <div class="user-status-indicator"></div>
                    </div>
                </div>
                <div class="user-info-wrapper">
                    <div class="user-name-container">
                        <span class="user-display-name"><?php echo htmlspecialchars($current_user_full_name ?: $current_user_name); ?></span>
                    </div>
                    <div class="user-role-container">
                        <span class="user-display-role"><?php echo htmlspecialchars($current_user_role === 'admin' ? 'Administrator' : ucfirst($current_user_role)); ?></span>
                    </div>
                </div>
                <div class="user-dropdown-indicator">
                    <i class="fas fa-chevron-down user-dropdown-chevron"></i>
                </div>
            </button>
            
            <!-- Internes Dropdown (für expanded state) -->
            <div class="user-profile-dropdown" id="userProfileDropdown" role="menu">
                <div class="dropdown-header">
                    <div class="dropdown-user-info">
                        <div class="dropdown-avatar">
                            <?php if ($current_user_avatar): ?>
                                <img src="<?php echo htmlspecialchars($current_user_avatar); ?>" alt="<?php echo htmlspecialchars($current_user_name); ?>" class="dropdown-avatar-image">
                            <?php else: ?>
                                <span class="dropdown-avatar-initials"><?php echo htmlspecialchars($user_initials); ?></span>
                            <?php endif; ?>
                        </div>
                        <div class="dropdown-user-details">
                            <div class="dropdown-user-name"><?php echo htmlspecialchars($current_user_full_name ?: $current_user_name); ?></div>
                            <div class="dropdown-user-email"><?php echo htmlspecialchars($current_user_email ?: ($current_user_name . '@boxenstop.de')); ?></div>
                        </div>
                    </div>
                </div>
                <div class="dropdown-divider"></div>
                <div class="dropdown-menu-list">
                    <a href="profile.php" class="dropdown-menu-item" role="menuitem">
                        <div class="menu-item-icon">
                            <i class="fas fa-user"></i>
                        </div>
                        <div class="menu-item-content">
                            <span class="menu-item-title">Mein Profil</span>
                        </div>
                    </a>
                    <a href="settings.php" class="dropdown-menu-item" role="menuitem">
                        <div class="menu-item-icon">
                            <i class="fas fa-cog"></i>
                        </div>
                        <div class="menu-item-content">
                            <span class="menu-item-title">Einstellungen</span>
                        </div>
                    </a>
                    <a href="users.php" class="dropdown-menu-item" role="menuitem">
                        <div class="menu-item-icon">
                            <i class="fas fa-users"></i>
                        </div>
                        <div class="menu-item-content">
                            <span class="menu-item-title">Benutzer verwalten</span>
                        </div>
                    </a>
                    <a href="notifications.php" class="dropdown-menu-item" role="menuitem">
                        <div class="menu-item-icon">
                            <i class="fas fa-bell"></i>
                        </div>
                        <div class="menu-item-content">
                            <span class="menu-item-title">Benachrichtigungen</span>
                        </div>
                    </a>
                    <a href="audit_log.php" class="dropdown-menu-item" role="menuitem">
                        <div class="menu-item-icon">
                            <i class="fas fa-history"></i>
                        </div>
                        <div class="menu-item-content">
                            <span class="menu-item-title">Aktivitätsprotokoll</span>
                        </div>
                    </a>
                    <div class="dropdown-divider"></div>
                    <a href="logout.php" class="dropdown-menu-item logout-item" role="menuitem">
                        <div class="menu-item-icon">
                            <i class="fas fa-sign-out-alt"></i>
                        </div>
                        <div class="menu-item-content">
                            <span class="menu-item-title">Abmelden</span>
                        </div>
                    </a>
                </div>
            </div>
        </div>
    </div>
</aside>

<script>
(function() {
    'use strict';
    
    const sidebar = document.getElementById('newSidebar');
    const sidebarToggle = document.getElementById('sidebarToggle');
    const adminMain = document.querySelector('.admin-main');
    
    // Sidebar State Management
    const SidebarState = {
        init: function() {
            const savedState = localStorage.getItem('newSidebarCollapsed');
            if (savedState === 'true' && sidebar) {
                sidebar.classList.add('collapsed');
                if (adminMain) {
                    adminMain.classList.add('sidebar-collapsed');
                }
            }
        },
        
        toggle: function() {
            if (!sidebar) return;
            
            sidebar.classList.toggle('collapsed');
            const isCollapsed = sidebar.classList.contains('collapsed');
            localStorage.setItem('newSidebarCollapsed', isCollapsed ? 'true' : 'false');
            
            if (adminMain) {
                if (isCollapsed) {
                    adminMain.classList.add('sidebar-collapsed');
                } else {
                    adminMain.classList.remove('sidebar-collapsed');
                }
            }
            
            // Schließe alle offenen Menüs
            SubmenuManager.closeAll();
            TooltipManager.hide();
            FlyoutManager.hide();
            UserProfileManager.hideAll();
            
            // Verstecke alle Submenüs im collapsed state
            if (isCollapsed) {
                document.querySelectorAll('.nav-submenu').forEach(function(submenu) {
                    submenu.classList.remove('is-open');
                });
            }
        }
    };
    
    // Submenu Management
    const SubmenuManager = {
        init: function() {
            const navItemGroups = document.querySelectorAll('.nav-item-wrapper.has-submenu');
            
            navItemGroups.forEach(function(group) {
                const navItem = group.querySelector('.nav-item');
                const submenu = group.querySelector('.nav-submenu');
                
                if (navItem && submenu) {
                    navItem.addEventListener('click', function(e) {
                        if (!sidebar || !sidebar.classList.contains('collapsed')) {
                            e.preventDefault();
                            const isOpen = submenu.classList.contains('is-open');
                            SubmenuManager.closeAll();
                            if (!isOpen) {
                                submenu.classList.add('is-open');
                                navItem.classList.add('submenu-open');
                                // Update Chevron
                                const chevron = navItem.querySelector('.nav-item-chevron');
                                if (chevron) {
                                    chevron.classList.remove('fa-chevron-down');
                                    chevron.classList.add('fa-chevron-up');
                                }
                            }
                        }
                    });
                }
            });
            
            // Close on outside click
            document.addEventListener('click', function(e) {
                if (!sidebar || sidebar.classList.contains('collapsed')) return;
                if (!e.target.closest('.nav-item-wrapper')) {
                    SubmenuManager.closeAll();
                }
            });
        },
        
        closeAll: function() {
            document.querySelectorAll('.nav-submenu').forEach(function(submenu) {
                submenu.classList.remove('is-open');
            });
            document.querySelectorAll('.nav-item').forEach(function(item) {
                item.classList.remove('submenu-open');
                const chevron = item.querySelector('.nav-item-chevron');
                if (chevron) {
                    chevron.classList.remove('fa-chevron-up');
                    chevron.classList.add('fa-chevron-down');
                }
            });
        }
    };
    
    // Tooltip Management
    const TooltipManager = {
        tooltip: null,
        
        create: function() {
            if (!this.tooltip) {
                this.tooltip = document.createElement('div');
                this.tooltip.className = 'sidebar-tooltip';
                document.body.appendChild(this.tooltip);
            }
            return this.tooltip;
        },
        
        show: function(element, text) {
            if (!sidebar || !sidebar.classList.contains('collapsed')) return;
            
            const tooltip = this.create();
            const rect = element.getBoundingClientRect();
            
            tooltip.textContent = text;
            tooltip.style.left = (rect.right + 12) + 'px';
            tooltip.style.top = (rect.top + (rect.height / 2)) + 'px';
            tooltip.classList.add('show');
        },
        
        hide: function() {
            if (this.tooltip) {
                this.tooltip.classList.remove('show');
            }
        },
        
        init: function() {
            const navItems = document.querySelectorAll('.nav-item[data-tooltip]');
            const self = this;
            
            navItems.forEach(function(item) {
                item.addEventListener('mouseenter', function() {
                    const tooltipText = this.getAttribute('data-tooltip');
                    self.show(this, tooltipText);
                });
                
                item.addEventListener('mouseleave', function() {
                    self.hide();
                });
            });
        }
    };
    
    // Notifications Tooltip Manager (ähnlich wie TooltipManager)
    const NotificationsTooltipManager = {
        tooltip: null,
        hideTimeout: null,
        
        create: function() {
            if (!this.tooltip) {
                this.tooltip = document.createElement('div');
                this.tooltip.className = 'sidebar-notifications-tooltip';
                this.tooltip.id = 'sidebarNotificationsTooltip';
                document.body.appendChild(this.tooltip);
                
                // Verhindere, dass Tooltip verschwindet wenn man darauf hovert
                this.tooltip.addEventListener('mouseenter', function() {
                    if (NotificationsTooltipManager.hideTimeout) {
                        clearTimeout(NotificationsTooltipManager.hideTimeout);
                        NotificationsTooltipManager.hideTimeout = null;
                    }
                });
                
                this.tooltip.addEventListener('mouseleave', function() {
                    NotificationsTooltipManager.hide();
                });
            }
            return this.tooltip;
        },
        
        show: function(element) {
            const tooltip = this.create();
            const rect = element.getBoundingClientRect();
            const isCollapsed = sidebar && sidebar.classList.contains('collapsed');
            
            // Setze Content
            const notificationsData = <?php echo json_encode([
                'count' => $notifications_count,
                'notifications' => $all_notifications
            ]); ?>;
            
            let content = `
                <div class="notifications-tooltip-header">
                    <h3>
                        <i class="fas fa-bell"></i>
                        Benachrichtigungen
                        ${notificationsData.count > 0 ? `<span class="notifications-count-badge">${notificationsData.count}</span>` : ''}
                    </h3>
                    <div class="notifications-filter-buttons">
                        <button class="notification-filter-btn active" data-filter="all">Alle</button>
                        <button class="notification-filter-btn" data-filter="appointment">Termine</button>
                        <button class="notification-filter-btn" data-filter="event">Events</button>
                    </div>
                </div>
                <div class="notifications-tooltip-body" id="notificationsTooltipBody">
            `;
            
            if (notificationsData.count > 0 && notificationsData.notifications.length > 0) {
                notificationsData.notifications.forEach(function(notification) {
                    const isAppointment = notification.notification_type === 'appointment';
                    const iconBg = isAppointment ? '#ef4444' : '#3b82f6';
                    const icon = isAppointment ? 'fa-calendar-day' : 'fa-calendar-alt';
                    const href = isAppointment ? `appointments.php?id=${notification.id}` : `events.php?id=${notification.id}`;
                    
                    content += `
                        <div class="notification-tooltip-item" data-notification-type="${notification.notification_type}" onclick="window.location.href='${href}'">
                            <div class="notification-icon" style="background: ${iconBg};">
                                <i class="fas ${icon}"></i>
                            </div>
                            <div class="notification-content">
                                ${isAppointment ? `
                                    <div class="notification-title">Termin: ${notification.customer_name || 'Unbekannt'}${notification.license_plate ? ' (' + notification.license_plate + ')' : ''}</div>
                                    <div class="notification-description">
                                        <i class="fas fa-clock"></i>
                                        ${notification.slot_date ? new Date(notification.slot_date).toLocaleDateString('de-DE') : ''} um ${notification.slot_time || ''}
                                        <span class="notification-status">${notification.status || ''}</span>
                                    </div>
                                ` : `
                                    <div class="notification-title">Event: ${notification.name || 'Unbekannt'}</div>
                                    <div class="notification-description">
                                        <i class="fas fa-calendar"></i>
                                        ${notification.event_date ? new Date(notification.event_date).toLocaleDateString('de-DE') : ''}
                                    </div>
                                `}
                            </div>
                        </div>
                    `;
                });
            } else {
                content += `
                    <div class="notifications-empty">
                        <i class="fas fa-bell-slash"></i>
                        <p>Keine neuen Benachrichtigungen</p>
                        <p class="notifications-empty-subtitle">Alles ist auf dem neuesten Stand</p>
                    </div>
                `;
            }
            
            content += `</div>`;
            
            if (notificationsData.count > 0) {
                content += `
                    <div class="notifications-tooltip-footer">
                        <button class="mark-all-read-btn" onclick="markAllNotificationsRead()">
                            <i class="fas fa-check-double"></i>
                            Alle als gelesen markieren
                        </button>
                    </div>
                `;
            }
            
            tooltip.innerHTML = content;
            
            // Position berechnen
            if (isCollapsed) {
                tooltip.style.left = (rect.right + 12) + 'px';
                tooltip.style.top = rect.top + 'px';
            } else {
                tooltip.style.left = (rect.right + 12) + 'px';
                tooltip.style.top = rect.top + 'px';
            }
            
            tooltip.classList.add('show');
            
            // Filter-Buttons Event Listeners
            const filterBtns = tooltip.querySelectorAll('.notification-filter-btn');
            filterBtns.forEach(function(btn) {
                btn.addEventListener('click', function(e) {
                    e.stopPropagation();
                    const filterType = this.dataset.filter;
                    
                    // Update active button
                    filterBtns.forEach(b => b.classList.remove('active'));
                    this.classList.add('active');
                    
                    // Filter items
                    const items = tooltip.querySelectorAll('.notification-tooltip-item');
                    items.forEach(function(item) {
                        if (filterType === 'all') {
                            item.style.display = 'flex';
                        } else if (filterType === item.dataset.notificationType) {
                            item.style.display = 'flex';
                        } else {
                            item.style.display = 'none';
                        }
                    });
                });
            });
        },
        
        hide: function() {
            const self = this;
            this.hideTimeout = setTimeout(function() {
                if (self.tooltip) {
                    self.tooltip.classList.remove('show');
                }
                self.hideTimeout = null;
            }, 100);
        },
        
        init: function() {
            const notificationsBtn = document.getElementById('sidebarNotificationsBtn');
            if (!notificationsBtn) return;
            
            const self = this;
            
            notificationsBtn.addEventListener('mouseenter', function() {
                self.show(this);
            });
            
            notificationsBtn.addEventListener('mouseleave', function(e) {
                // Prüfe ob Maus zum Tooltip geht
                const tooltip = self.tooltip;
                if (tooltip && tooltip.classList.contains('show')) {
                    const tooltipRect = tooltip.getBoundingClientRect();
                    const mouseX = e.clientX;
                    const mouseY = e.clientY;
                    
                    if (mouseX >= tooltipRect.left - 20 && mouseX <= tooltipRect.right + 20 &&
                        mouseY >= tooltipRect.top - 20 && mouseY <= tooltipRect.bottom + 20) {
                        return;
                    }
                }
                self.hide();
            });
        }
    };
    
    // Flyout Menu Management
    const FlyoutManager = {
        flyout: null,
        activeGroup: null,
        hideTimeout: null,
        
        create: function() {
            if (!this.flyout) {
                this.flyout = document.createElement('div');
                this.flyout.className = 'sidebar-flyout';
                document.body.appendChild(this.flyout);
                
                // Verhindere, dass Flyout verschwindet wenn man darauf hovert
                this.flyout.addEventListener('mouseenter', function() {
                    if (FlyoutManager.hideTimeout) {
                        clearTimeout(FlyoutManager.hideTimeout);
                        FlyoutManager.hideTimeout = null;
                    }
                });
                
                this.flyout.addEventListener('mouseleave', function() {
                    FlyoutManager.hide();
                });
            }
            return this.flyout;
        },
        
        show: function(group) {
            if (!sidebar || !sidebar.classList.contains('collapsed')) return;
            
            // Verhindere Hide wenn bereits gezeigt
            if (this.hideTimeout) {
                clearTimeout(this.hideTimeout);
                this.hideTimeout = null;
            }
            
            const submenu = group.querySelector('.nav-submenu');
            if (!submenu) return;
            
            const flyout = this.create();
            const navItem = group.querySelector('.nav-item');
            const rect = navItem.getBoundingClientRect();
            
            flyout.innerHTML = submenu.innerHTML;
            flyout.style.left = (rect.right + 12) + 'px';
            flyout.style.top = rect.top + 'px';
            flyout.classList.add('show');
            
            this.activeGroup = group;
            
            // Event Listeners für Flyout Items
            const flyoutItems = flyout.querySelectorAll('.nav-submenu-item');
            flyoutItems.forEach(function(item) {
                item.addEventListener('click', function() {
                    // Warte kurz bevor Hide, damit der Link funktioniert
                    setTimeout(function() {
                        FlyoutManager.hide();
                    }, 100);
                });
            });
        },
        
        hide: function() {
            // Verwende Timeout, damit man vom Nav-Item zum Flyout wechseln kann
            const self = this;
            this.hideTimeout = setTimeout(function() {
                if (self.flyout) {
                    self.flyout.classList.remove('show');
                }
                self.activeGroup = null;
                self.hideTimeout = null;
            }, 100);
        },
        
        init: function() {
            const navItemGroups = document.querySelectorAll('.nav-item-wrapper.has-submenu');
            const self = this;
            
            navItemGroups.forEach(function(group) {
                const navItem = group.querySelector('.nav-item[data-has-submenu="true"]');
                
                if (navItem) {
                    navItem.addEventListener('mouseenter', function() {
                        if (sidebar && sidebar.classList.contains('collapsed')) {
                            self.show(group);
                        }
                    });
                    
                    // Verzögertes Hide, damit man zum Flyout wechseln kann
                    group.addEventListener('mouseleave', function(e) {
                        if (sidebar && sidebar.classList.contains('collapsed')) {
                            // Prüfe ob Maus zum Flyout geht
                            const flyout = self.flyout;
                            if (flyout && flyout.classList.contains('show')) {
                                const flyoutRect = flyout.getBoundingClientRect();
                                const mouseX = e.clientX;
                                const mouseY = e.clientY;
                                
                                // Wenn Maus in Richtung Flyout geht, nicht verstecken
                                if (mouseX >= flyoutRect.left - 20 && mouseX <= flyoutRect.right + 20 &&
                                    mouseY >= flyoutRect.top - 20 && mouseY <= flyoutRect.bottom + 20) {
                                    return;
                                }
                            }
                            self.hide();
                        }
                    });
                }
            });
        }
    };
    
    // User Profile Manager - Neue erweiterte Logik mit mehr Features
    const UserProfileManager = {
        externalDropdown: null,
        internalDropdown: null,
        userProfileBtn: null,
        isInitialized: false,
        animationDuration: 200,
        hideTimeout: null,
        
        init: function() {
            if (this.isInitialized) return;
            
            this.userProfileBtn = document.getElementById('userProfileBtn');
            this.internalDropdown = document.getElementById('userProfileDropdown');
            
            if (!this.userProfileBtn || !this.internalDropdown) return;
            
            this.isInitialized = true;
            
            // Click Handler für Button
            this.userProfileBtn.addEventListener('click', function(e) {
                e.stopPropagation();
                UserProfileManager.handleClick();
            });
            
            // Keyboard Navigation
            this.userProfileBtn.addEventListener('keydown', function(e) {
                if (e.key === 'Enter' || e.key === ' ') {
                    e.preventDefault();
                    UserProfileManager.handleClick();
                } else if (e.key === 'Escape') {
                    UserProfileManager.hideAll();
                }
            });
            
            // Click Outside Handler
            document.addEventListener('click', function(e) {
                UserProfileManager.handleClickOutside(e);
            });
            
            // Escape Key Handler
            document.addEventListener('keydown', function(e) {
                if (e.key === 'Escape') {
                    UserProfileManager.hideAll();
                }
            });
            
            // Sidebar State Change Observer
            this.initSidebarObserver();
            
            // Window Resize Handler
            window.addEventListener('resize', function() {
                UserProfileManager.handleResize();
            });
        },
        
        initSidebarObserver: function() {
            if (!sidebar) return;
            
            const observer = new MutationObserver(function(mutations) {
                mutations.forEach(function(mutation) {
                    if (mutation.type === 'attributes' && mutation.attributeName === 'class') {
                        UserProfileManager.handleSidebarStateChange();
                    }
                });
            });
            
            observer.observe(sidebar, {
                attributes: true,
                attributeFilter: ['class']
            });
        },
        
        handleClick: function() {
            const isCollapsed = sidebar && sidebar.classList.contains('collapsed');
            
            if (isCollapsed) {
                this.toggleExternal();
            } else {
                this.toggleInternal();
            }
        },
        
        toggleInternal: function() {
            if (!this.internalDropdown || !this.userProfileBtn) return;
            
            const isOpen = this.internalDropdown.classList.contains('show');
            
            // Schließe externes Dropdown falls offen
            this.hideExternal();
            
            if (isOpen) {
                this.hideInternal();
            } else {
                this.showInternal();
            }
        },
        
        showInternal: function() {
            if (!this.internalDropdown || !this.userProfileBtn) return;
            
            this.internalDropdown.classList.add('show');
            this.userProfileBtn.setAttribute('aria-expanded', 'true');
            
            // Focus Management
            const firstMenuItem = this.internalDropdown.querySelector('.dropdown-menu-item');
            if (firstMenuItem) {
                setTimeout(function() {
                    firstMenuItem.focus();
                }, 100);
            }
        },
        
        hideInternal: function() {
            if (!this.internalDropdown || !this.userProfileBtn) return;
            
            this.internalDropdown.classList.remove('show');
            this.userProfileBtn.setAttribute('aria-expanded', 'false');
        },
        
        toggleExternal: function() {
            if (!sidebar || !sidebar.classList.contains('collapsed')) return;
            
            const external = this.getOrCreateExternal();
            
            if (!external || !this.userProfileBtn) return;
            
            const isOpen = external.classList.contains('show');
            
            // Schließe internes Dropdown falls offen
            this.hideInternal();
            
            if (isOpen) {
                this.hideExternal();
            } else {
                this.showExternal(external);
            }
        },
        
        getOrCreateExternal: function() {
            if (!this.externalDropdown) {
                const userProfileDropdown = document.getElementById('userProfileDropdown');
                if (userProfileDropdown) {
                    this.externalDropdown = userProfileDropdown.cloneNode(true);
                    this.externalDropdown.id = 'user-profile-dropdown-external';
                    this.externalDropdown.className = 'user-profile-dropdown-external';
                    document.body.appendChild(this.externalDropdown);
                    
                    // Event Listeners für externe Dropdown-Items
                    this.attachExternalItemListeners();
                }
            }
            return this.externalDropdown;
        },
        
        attachExternalItemListeners: function() {
            if (!this.externalDropdown) return;
            
            const dropdownItems = this.externalDropdown.querySelectorAll('.dropdown-menu-item');
            const self = this;
            
            dropdownItems.forEach(function(item) {
                item.addEventListener('click', function() {
                    // Warte kurz bevor Hide, damit Link funktioniert
                    setTimeout(function() {
                        self.hideExternal();
                    }, 150);
                });
                
                // Keyboard Navigation
                item.addEventListener('keydown', function(e) {
                    if (e.key === 'Enter') {
                        item.click();
                    }
                });
            });
        },
        
        showExternal: function(external) {
            if (!external || !this.userProfileBtn) return;
            
            // Warte kurz damit DOM bereit ist
            setTimeout(function() {
                const rect = UserProfileManager.userProfileBtn.getBoundingClientRect();
                const dropdownHeight = external.offsetHeight || 350;
                const viewportHeight = window.innerHeight;
                const viewportWidth = window.innerWidth;
                
                // Berechne optimale Position
                let topPosition = rect.top - dropdownHeight - 8;
                let leftPosition = rect.right + 12;
                
                // Prüfe ob Dropdown über Viewport hinausgeht
                if (topPosition < 10) {
                    // Zeige Dropdown unter dem Button
                    topPosition = rect.bottom + 8;
                }
                
                // Prüfe ob Dropdown unter Viewport hinausgeht
                if (topPosition + dropdownHeight > viewportHeight - 10) {
                    topPosition = Math.max(10, viewportHeight - dropdownHeight - 10);
                }
                
                // Prüfe ob Dropdown rechts über Viewport hinausgeht
                const dropdownWidth = external.offsetWidth || 280;
                if (leftPosition + dropdownWidth > viewportWidth - 10) {
                    // Zeige Dropdown links vom Button
                    leftPosition = rect.left - dropdownWidth - 12;
                }
                
                external.style.top = topPosition + 'px';
                external.style.left = leftPosition + 'px';
                external.classList.add('show');
                UserProfileManager.userProfileBtn.setAttribute('aria-expanded', 'true');
                
                // Focus Management
                const firstMenuItem = external.querySelector('.dropdown-menu-item');
                if (firstMenuItem) {
                    setTimeout(function() {
                        firstMenuItem.focus();
                    }, 100);
                }
            }, 10);
        },
        
        hideExternal: function() {
            if (this.externalDropdown && this.userProfileBtn) {
                this.externalDropdown.classList.remove('show');
                this.userProfileBtn.setAttribute('aria-expanded', 'false');
            }
        },
        
        hideInternal: function() {
            if (this.internalDropdown && this.userProfileBtn) {
                this.internalDropdown.classList.remove('show');
                this.userProfileBtn.setAttribute('aria-expanded', 'false');
            }
        },
        
        hideAll: function() {
            this.hideExternal();
            this.hideInternal();
        },
        
        handleClickOutside: function(e) {
            const isCollapsed = sidebar && sidebar.classList.contains('collapsed');
            
            if (isCollapsed) {
                // Prüfe externes Dropdown
                if (this.externalDropdown && 
                    !this.externalDropdown.contains(e.target) && 
                    this.userProfileBtn && 
                    !this.userProfileBtn.contains(e.target)) {
                    this.hideExternal();
                }
            } else {
                // Prüfe internes Dropdown
                const container = this.userProfileBtn ? this.userProfileBtn.closest('.user-profile-wrapper') : null;
                if (container && !container.contains(e.target)) {
                    this.hideInternal();
                }
            }
        },
        
        handleSidebarStateChange: function() {
            const isCollapsed = sidebar && sidebar.classList.contains('collapsed');
            
            // Schließe alle Dropdowns beim State-Wechsel
            this.hideAll();
            
            // Wenn Sidebar eingeklappt wird, entferne internes Dropdown falls vorhanden
            if (isCollapsed && this.internalDropdown) {
                this.internalDropdown.classList.remove('show');
            }
        },
        
        handleResize: function() {
            // Repositioniere externes Dropdown bei Resize
            if (this.externalDropdown && this.externalDropdown.classList.contains('show')) {
                const external = this.externalDropdown;
                const userProfileBtn = this.userProfileBtn;
                
                if (external && userProfileBtn) {
                    // Temporär verstecken, neu positionieren, dann wieder zeigen
                    external.classList.remove('show');
                    setTimeout(function() {
                        UserProfileManager.showExternal(external);
                    }, 50);
                }
            }
        }
    };
    
    // Initialize alles
    document.addEventListener('DOMContentLoaded', function() {
        SidebarState.init();
        
        if (sidebarToggle) {
            sidebarToggle.addEventListener('click', function(e) {
                e.stopPropagation();
                SidebarState.toggle();
            });
        }
        
        SubmenuManager.init();
        TooltipManager.init();
        FlyoutManager.init();
        UserProfileManager.init();
        NotificationsTooltipManager.init();
        
        // Expose functions for external use
        window.hideExternalUserDropdown = function() {
            UserProfileManager.hideAll();
        };
        
        // Mark all notifications as read
        window.markAllNotificationsRead = function() {
            const tooltip = document.getElementById('sidebarNotificationsTooltip');
            const body = tooltip ? tooltip.querySelector('.notifications-tooltip-body') : null;
            
            if (body) {
                body.style.opacity = '0.5';
                setTimeout(function() {
                    body.innerHTML = `
                        <div class="notifications-empty">
                            <i class="fas fa-check-circle" style="color: #10b981;"></i>
                            <p>Alle Benachrichtigungen als gelesen markiert</p>
                            <p class="notifications-empty-subtitle">Alles ist auf dem neuesten Stand</p>
                        </div>
                    `;
                    body.style.opacity = '1';
                    
                    // Entferne Badge
                    const badge = document.querySelector('.sidebar-notification-badge');
                    if (badge) {
                        badge.remove();
                    }
                    
                    // Update count in header
                    const countBadge = tooltip.querySelector('.notifications-count-badge');
                    if (countBadge) {
                        countBadge.remove();
                    }
                }, 300);
            }
        };
    });
})();
</script>


