<?php
/**
 * HelloMateo API Wrapper
 * Zentrale Funktionen für HelloMateo API-Interaktionen
 */

class HelloMateoAPI {
    private $apiKey;
    private $baseUrl = 'https://integration.getmateo.com/api/v1';
    private $autohausId;
    
    public function __construct($apiKey = null, $autohausId = null) {
        // Stelle sicher, dass autohausId als Integer gespeichert wird
        $this->autohausId = $autohausId !== null ? intval($autohausId) : null;
        $this->apiKey = $apiKey ?? $this->getApiKey();
        
        // Debug-Logging
        error_log("HelloMateoAPI Constructor: autohausId=" . ($this->autohausId ?? 'NULL') . " (type: " . gettype($this->autohausId) . "), has_api_key=" . (!empty($this->apiKey) ? 'true' : 'false'));
    }
    
    /**
     * Holt API-Key aus Datenbank (Autohaus-spezifisch oder global)
     */
    private function getApiKey() {
        require_once __DIR__ . '/../../config/database.php';
        $db = new Database();
        $conn = $db->getConnection();
        
        // Prüfe ob whatsapp_api_key Spalte existiert
        try {
            $stmt = $conn->query("SHOW COLUMNS FROM autohaus LIKE 'whatsapp_api_key'");
            $hasColumn = $stmt->rowCount() > 0;
        } catch (Exception $e) {
            $hasColumn = false;
        }
        
        // Versuche Autohaus-spezifischen API-Key zu laden
        if ($this->autohausId && $hasColumn) {
            try {
                $stmt = $conn->prepare("SELECT whatsapp_api_key FROM autohaus WHERE id = ?");
                $stmt->execute([$this->autohausId]);
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($result && isset($result['whatsapp_api_key']) && !empty(trim($result['whatsapp_api_key']))) {
                    $apiKey = $result['whatsapp_api_key'];
                    
                    // ROBUSTE Bereinigung: Entferne ALLE Whitespace-Zeichen
                    $apiKey = trim($apiKey); // Entferne führende/trailing Whitespace
                    
                    // WICHTIG: JWT-Token haben 3 Teile getrennt durch Punkte
                    // Entferne Whitespace NUR zwischen den Teilen, nicht innerhalb der Base64-Strings
                    $parts = explode('.', $apiKey);
                    if (count($parts) === 3) {
                        // JWT-Format: Bereinige jeden Teil einzeln
                        $parts[0] = trim($parts[0]);
                        $parts[1] = trim($parts[1]);
                        $parts[2] = trim($parts[2]);
                        // Entferne Whitespace nur am Anfang/Ende jedes Teils
                        $parts[0] = preg_replace('/\s+/', '', $parts[0]);
                        $parts[1] = preg_replace('/\s+/', '', $parts[1]);
                        $parts[2] = preg_replace('/\s+/', '', $parts[2]);
                        $apiKey = implode('.', $parts);
                    } else {
                        // Nicht-JWT-Format: Entferne alle Whitespace
                        $apiKey = preg_replace('/\s+/', '', $apiKey);
                    }
                    
                    $apiKey = preg_replace('/[\x00-\x1F\x7F]/', '', $apiKey); // Entferne alle Control-Zeichen
                    
                    // Prüfe ob API-Key nach Bereinigung noch gültig ist
                    if (empty($apiKey)) {
                        error_log("HelloMateoAPI: API-Key wurde nach Bereinigung leer für Autohaus ID " . $this->autohausId);
                        return null;
                    }
                    
                    error_log("HelloMateoAPI: API-Key gefunden für Autohaus ID " . $this->autohausId . " (Länge vor Bereinigung: " . strlen($result['whatsapp_api_key']) . ", nach Bereinigung: " . strlen($apiKey) . ")");
                    error_log("HelloMateoAPI: API-Key Präfix: " . substr($apiKey, 0, 30) . "... Suffix: ..." . substr($apiKey, -20));
                    error_log("HelloMateoAPI: JWT-Teile: " . count(explode('.', $apiKey)));
                    return $apiKey;
                } else {
                    error_log("HelloMateoAPI: Kein API-Key für Autohaus ID " . $this->autohausId . " gefunden (Wert: " . ($result['whatsapp_api_key'] ?? 'NULL') . ")");
                }
            } catch (Exception $e) {
                error_log("HelloMateoAPI: Fehler beim Laden des Autohaus API-Keys: " . $e->getMessage());
            }
        } else {
            if (!$hasColumn) {
                error_log("HelloMateoAPI: Spalte 'whatsapp_api_key' existiert nicht in autohaus-Tabelle");
            }
            if (!$this->autohausId) {
                error_log("HelloMateoAPI: Keine Autohaus-ID gesetzt");
            }
        }
        
        // Fallback auf globale Einstellungen
        try {
            $stmt = $conn->prepare("SELECT setting_value FROM settings WHERE setting_key = 'whatsapp_api_key'");
            $stmt->execute();
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($result && isset($result['setting_value']) && !empty(trim($result['setting_value']))) {
                $apiKey = $result['setting_value'];
                
                // ROBUSTE Bereinigung: Entferne ALLE Whitespace-Zeichen
                $apiKey = trim($apiKey);
                
                // WICHTIG: JWT-Token haben 3 Teile getrennt durch Punkte
                $parts = explode('.', $apiKey);
                if (count($parts) === 3) {
                    // JWT-Format: Bereinige jeden Teil einzeln
                    $parts[0] = trim(preg_replace('/\s+/', '', $parts[0]));
                    $parts[1] = trim(preg_replace('/\s+/', '', $parts[1]));
                    $parts[2] = trim(preg_replace('/\s+/', '', $parts[2]));
                    $apiKey = implode('.', $parts);
                } else {
                    // Nicht-JWT-Format: Entferne alle Whitespace
                    $apiKey = preg_replace('/\s+/', '', $apiKey);
                }
                
                $apiKey = preg_replace('/[\x00-\x1F\x7F]/', '', $apiKey); // Entferne Control-Zeichen
                
                if (empty($apiKey)) {
                    error_log("HelloMateoAPI: Globaler API-Key wurde nach Bereinigung leer");
                    return null;
                }
                
                error_log("HelloMateoAPI: Globaler API-Key gefunden (Länge: " . strlen($apiKey) . ")");
                return $apiKey;
            } else {
                error_log("HelloMateoAPI: Kein globaler API-Key gefunden");
            }
        } catch (Exception $e) {
            error_log("HelloMateoAPI: Fehler beim Laden des globalen API-Keys: " . $e->getMessage());
        }
        
        error_log("HelloMateoAPI: Kein API-Key gefunden (weder Autohaus-spezifisch noch global)");
        return null;
    }
    
    /**
     * Führt einen API-Request aus
     */
    private function makeRequest($endpoint, $method = 'GET', $data = null) {
        if (!$this->apiKey) {
            return ['success' => false, 'error' => 'API-Key nicht gefunden'];
        }
        
        // API-Key bereinigen - entferne ALLE Whitespace-Zeichen und Control-Zeichen
        $apiKey = trim($this->apiKey);
        
        // WICHTIG: JWT-Token haben 3 Teile getrennt durch Punkte
        $parts = explode('.', $apiKey);
        if (count($parts) === 3) {
            // JWT-Format: Bereinige jeden Teil einzeln
            $parts[0] = trim(preg_replace('/\s+/', '', $parts[0]));
            $parts[1] = trim(preg_replace('/\s+/', '', $parts[1]));
            $parts[2] = trim(preg_replace('/\s+/', '', $parts[2]));
            $apiKey = implode('.', $parts);
        } else {
            // Nicht-JWT-Format: Entferne alle Whitespace
            $apiKey = preg_replace('/\s+/', '', $apiKey);
        }
        
        $apiKey = preg_replace('/[\x00-\x1F\x7F]/', '', $apiKey); // Entferne Control-Zeichen
        
        if (empty($apiKey)) {
            error_log("HelloMateoAPI: API-Key ist leer nach Bereinigung");
            return ['success' => false, 'error' => 'API-Key ist leer nach Bereinigung'];
        }
        
        // Prüfe API-Key Format (sollte JWT sein - 3 Teile getrennt durch Punkte)
        $jwtParts = explode('.', $apiKey);
        if (count($jwtParts) !== 3) {
            error_log("HelloMateoAPI: API-Key hat ungültiges Format (kein JWT - erwartet 3 Teile, gefunden: " . count($jwtParts) . ")");
            error_log("HelloMateoAPI: API-Key Preview: " . substr($apiKey, 0, 50) . "...");
        } else {
            error_log("HelloMateoAPI: API-Key ist im JWT-Format (3 Teile)");
        }
        
        $url = $this->baseUrl . $endpoint;
        $ch = curl_init();
        
        // WICHTIG: Authorization Header gemäß HelloMateo Docs: "Bearer {API_KEY}"
        // Dokumentation: https://docs.getmateo.com/
        // Format: Authorization: Bearer 123456789
        $authHeader = 'Bearer ' . trim($apiKey);
        $headers = [
            'Authorization: ' . $authHeader,
            'Content-Type: application/json',
            'Accept: application/json'
        ];
        
        // Detailliertes Logging für Debugging
        error_log("=== HelloMateoAPI Request ===");
        error_log("URL: {$url}");
        error_log("Method: {$method}");
        error_log("Endpoint: {$endpoint}");
        error_log("Base URL: {$this->baseUrl}");
        error_log("API-Key Länge: " . strlen($apiKey) . ", Prefix: " . substr($apiKey, 0, 30) . "... Suffix: ..." . substr($apiKey, -20));
        error_log("Authorization Header Länge: " . strlen($authHeader));
        if ($data !== null) {
            error_log("Request Data (JSON): " . json_encode($data, JSON_PRETTY_PRINT));
            error_log("Request Data Type: " . gettype($data));
            if (is_array($data)) {
                error_log("Request Data Keys: " . implode(', ', array_keys($data)));
                error_log("Request Data Count: " . count($data));
            }
        } else {
            error_log("Request Data: NULL (keine Daten)");
        }
        error_log("Headers: " . implode(', ', $headers));
        
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            if ($data) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            }
        } elseif ($method === 'PATCH') {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PATCH');
            if ($data) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            }
        } elseif ($method === 'DELETE') {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        $curlInfo = curl_getinfo($ch);
        curl_close($ch);
        
        if ($error) {
            error_log("HelloMateoAPI: cURL Fehler: " . $error);
            return ['success' => false, 'error' => 'cURL Fehler: ' . $error, 'http_code' => $httpCode];
        }
        
        // Logge die rohe Antwort für Debugging
        error_log("HelloMateoAPI: HTTP Code: {$httpCode}, Response Length: " . strlen($response));
        if ($httpCode >= 400) {
            error_log("HelloMateoAPI: Fehler-Response (erste 500 Zeichen): " . substr($response, 0, 500));
        }
        
        $responseData = json_decode($response, true);
        
        // Debug-Logging für ALLE Requests (auch erfolgreiche)
        error_log("HelloMateoAPI: HTTP {$httpCode} für {$endpoint}");
        error_log("HelloMateoAPI: Raw Response (erste 2000 Zeichen): " . substr($response, 0, 2000));
        if ($responseData) {
            error_log("HelloMateoAPI: Decoded Response: " . json_encode($responseData, JSON_PRETTY_PRINT));
        } else {
            error_log("HelloMateoAPI: WARNUNG - Response konnte nicht als JSON dekodiert werden!");
            error_log("HelloMateoAPI: Response Type: " . gettype($response) . ", Length: " . strlen($response));
        }
        error_log("HelloMateoAPI: cURL Info - URL: " . ($curlInfo['url'] ?? 'N/A') . ", Content-Type: " . ($curlInfo['content_type'] ?? 'N/A'));
        
        if ($httpCode >= 200 && $httpCode < 300) {
            return ['success' => true, 'data' => $responseData, 'http_code' => $httpCode];
        } else {
            $errorMsg = 'API Fehler';
            
            // Spezielle Fehlermeldungen für bekannte HTTP-Codes
            if ($httpCode === 429) {
                $errorMsg = 'Zu viele Anfragen (Rate Limit erreicht). Dies kann mehrere Ursachen haben:';
                $errorMsg .= ' 1) Zu viele Anfragen in kurzer Zeit, 2) Rate Limit auf API-Key-Ebene, 3) Rate Limit auf IP-Adresse.';
                $errorMsg .= ' Bitte warten Sie mindestens 1 Stunde oder kontaktieren Sie HelloMateo Support, wenn das Problem länger anhält.';
                error_log("HelloMateoAPI: RATE LIMIT ERREICHT (HTTP 429) - Dies könnte ein Problem mit dem API-Key oder der IP-Adresse sein.");
                error_log("HelloMateoAPI: Wenn das Problem nach mehreren Stunden weiterhin besteht, könnte es ein Problem mit der API-Key-Konfiguration geben.");
            } elseif ($httpCode === 405) {
                $errorMsg = 'HTTP-Methode nicht erlaubt. Der Endpunkt akzeptiert diese Anfrage nicht.';
            } elseif ($httpCode === 400) {
                $errorMsg = 'Ungültige Anfrage. Bitte überprüfen Sie die übermittelten Daten.';
            } elseif ($httpCode === 401 || $httpCode === 403) {
                $errorMsg = 'Invalid token - Der API-Key ist ungültig oder abgelaufen. Bitte überprüfen Sie den API-Key in den Einstellungen.';
            } elseif ($httpCode === 404) {
                $errorMsg = 'Endpunkt nicht gefunden. Der angeforderte Endpunkt existiert nicht.';
            } elseif ($httpCode >= 500) {
                $errorMsg = 'Server-Fehler. Der API-Server hat einen Fehler verursacht. Bitte versuchen Sie es später erneut.';
            }
            
            if ($responseData) {
                if (isset($responseData['message'])) {
                    $errorMsg = $responseData['message'] . ($errorMsg !== 'API Fehler' ? ' (' . $errorMsg . ')' : '');
                } elseif (isset($responseData['error'])) {
                    $errorMsg = $responseData['error'] . ($errorMsg !== 'API Fehler' ? ' (' . $errorMsg . ')' : '');
                }
                if (isset($responseData['hint'])) {
                    $errorMsg .= ' Hinweis: ' . $responseData['hint'];
                }
                if (isset($responseData['code'])) {
                    $errorMsg = $responseData['code'] . ': ' . $errorMsg;
                }
            }
            
            return ['success' => false, 'error' => $errorMsg, 'http_code' => $httpCode, 'data' => $responseData];
        }
    }
    
    /**
     * Sendet eine Nachricht
     */
    public function sendMessage($to, $message, $templateId = null, $channelId = null, $placeholderValues = null) {
        // HelloMateo API erwartet spezifisches Format
        $data = [];
        
        // Telefonnummer formatieren und extrahieren
        $phone = null;
        if (is_array($to)) {
            // Wenn Array, nimm die erste Telefonnummer
            $phone = $to[0]['handle'] ?? $to[0] ?? null;
            $data['to'] = $to;
        } else {
            // Telefonnummer formatieren
            $phone = trim($to);
            
            // Entferne Leerzeichen und Sonderzeichen, behalte nur Zahlen und +
            $phone = preg_replace('/[^0-9+]/', '', $phone);
            
            // Wenn keine + am Anfang, füge +49 für Deutschland hinzu (wenn deutsche Nummer)
            if (!empty($phone) && !str_starts_with($phone, '+')) {
                // Wenn mit 0 beginnt, ersetze durch +49
                if (str_starts_with($phone, '0')) {
                    $phone = '+49' . substr($phone, 1);
                } else {
                    // Wenn nur Zahlen, füge +49 hinzu
                    $phone = '+49' . $phone;
                }
            }
            
            error_log("HelloMateoAPI sendMessage: Telefonnummer formatiert von '{$to}' zu '{$phone}'");
            $data['to'] = [['handle' => $phone]];
        }
        
        // WICHTIG: custom_fields müssen über contact_id verknüpft werden
        // HelloMateo erwartet, dass custom_fields im Contact-Kontext vorhanden sind
        // Daher müssen wir:
        // 1. Den Contact finden (über Telefonnummer)
        // 2. Die custom_fields über contact_custom_field Endpunkt setzen
        // 3. Dann die Nachricht mit contact_id senden
        
        // Erweiterte Debug-Logs
        $logDir = __DIR__ . '/../../logs';
        if (!is_dir($logDir)) {
            @mkdir($logDir, 0755, true);
        }
        $debugLogFile = $logDir . '/whatsapp_contact_debug.log';
        $logMessage = function($message) use ($debugLogFile) {
            $timestamp = date('Y-m-d H:i:s');
            $logEntry = "[{$timestamp}] {$message}\n";
            @file_put_contents($debugLogFile, $logEntry, FILE_APPEND);
            error_log($message); // Auch in PHP Error Log
        };
        
        $logMessage("=== HelloMateoAPI sendMessage START ===");
        $logMessage("templateId: " . ($templateId ?? 'NULL'));
        $logMessage("placeholderValues Type: " . gettype($placeholderValues));
        $logMessage("placeholderValues: " . json_encode($placeholderValues, JSON_PRETTY_PRINT));
        $logMessage("phone: " . ($phone ?? 'NULL'));
        $logMessage("templateId check: " . ($templateId ? 'TRUE' : 'FALSE'));
        $logMessage("placeholderValues check: " . ($placeholderValues ? 'TRUE' : 'FALSE'));
        $logMessage("is_array check: " . (is_array($placeholderValues) ? 'TRUE' : 'FALSE'));
        $logMessage("!empty check: " . (!empty($placeholderValues) ? 'TRUE' : 'FALSE'));
        $logMessage("phone check: " . ($phone ? 'TRUE' : 'FALSE'));
        
        $customFieldsData = [];
        $contactId = null;
        
        if ($templateId && $placeholderValues && is_array($placeholderValues) && !empty($placeholderValues) && $phone) {
            $logMessage("✅ Bedingung erfüllt, starte custom_fields Verarbeitung");
            // Extrahiere custom_fields aus placeholderValues
            foreach ($placeholderValues as $key => $value) {
                if (is_string($key) && strpos($key, 'custom_fields.') === 0) {
                    $fieldName = substr($key, strlen('custom_fields.'));
                    $customFieldsData[$fieldName] = $value;
                }
            }
            
            // NEUE LOGIK: Vereinfachter Ansatz ohne vorherige Contact-Suche
            // 1. Versuche contact_id aus lokalem Cache zu holen (Telefonnummer -> contact_id Mapping)
            // 2. Wenn nicht im Cache, sende Nachricht direkt (HelloMateo findet/erstellt Contact automatisch)
            // 3. Hole contact_id aus Conversation-Antwort
            // 4. Setze custom_fields NACH dem Senden
            // 5. Speichere contact_id im Cache für zukünftige Verwendung
            
            if (!empty($customFieldsData)) {
                $logMessage("✅ Custom Fields gefunden: " . json_encode($customFieldsData));
                
                // 1. Versuche contact_id aus lokalem Cache zu holen
                $logMessage("Versuche contact_id aus lokalem Cache zu holen für Telefonnummer: " . $phone);
                $contactId = $this->getContactIdFromCache($phone);
                
                if ($contactId) {
                    $logMessage("✅ contact_id aus Cache geholt: " . $contactId);
                    // Setze custom_fields VOR dem Senden (wenn contact_id bekannt)
                    // ABER: Prüfe zuerst auf Rate Limit
                    $checkResult = $this->makeRequest("/contact_custom_field?contact_id=eq.{$contactId}&key=eq.last_name&limit=1");
                    $checkHttpCode = $checkResult['http_code'] ?? 0;
                    
                    if ($checkHttpCode !== 429) {
                        // Kein Rate Limit - setze custom_fields VOR dem Senden
                        $this->setCustomFieldsOnContact($contactId, $customFieldsData, $logMessage);
                    } else {
                        $logMessage("⚠️ Rate Limit (429) - custom_fields werden NACH dem Senden gesetzt");
                        $contactId = null; // Setze auf null, damit sie nach dem Senden geholt wird
                    }
                } else {
                    $logMessage("⚠️ contact_id nicht im Cache - versuche Contact zu finden/erstellen...");
                    
                    // WICHTIG: Finde oder erstelle Contact VOR dem Senden, damit custom_fields gesetzt werden können
                    // Prüfe zuerst auf Rate Limit
                    $rateLimitCheck = $this->makeRequest("/contact?limit=1");
                    $rateLimitHttpCode = $rateLimitCheck['http_code'] ?? 0;
                    
                    if ($rateLimitHttpCode !== 429) {
                        // Kein Rate Limit - finde oder erstelle Contact
                        $foundContactId = $this->findOrCreateContactByPhone($phone);
                        
                        if ($foundContactId) {
                            $contactId = $foundContactId; // Setze die Variable
                            $logMessage("✅ Contact gefunden/erstellt: " . $contactId);
                            
                            // Speichere contact_id im Cache
                            $this->saveContactIdToCache($phone, $contactId);
                            $logMessage("✅ contact_id im Cache gespeichert: " . $contactId);
                            
                            // Setze custom_fields VOR dem Senden
                            $this->setCustomFieldsOnContact($contactId, $customFieldsData, $logMessage);
                            
                            // Verwende contact_id im 'to' Objekt
                            if (isset($data['to'][0])) {
                                $data['to'][0]['contact_id'] = $contactId;
                                if (!isset($data['to'][0]['handle'])) {
                                    $data['to'][0]['handle'] = $phone;
                                }
                                $logMessage("✅ contact_id im 'to' Objekt gesetzt: " . $contactId);
                            }
                        } else {
                            $logMessage("⚠️ Konnte Contact nicht finden/erstellen - sende Nachricht ohne contact_id");
                            $logMessage("⚠️ custom_fields werden NACH dem Senden gesetzt (für zukünftige Nachrichten)");
                        }
                    } else {
                        $logMessage("⚠️ Rate Limit (429) - überspringe Contact-Suche");
                        $logMessage("⚠️ sende Nachricht direkt - custom_fields werden NACH dem Senden gesetzt");
                    }
                }
                
                // Wenn contact_id vorhanden ist (aus Cache oder neu gefunden), verwende sie im 'to' Objekt
                if ($contactId && isset($data['to'][0]) && !isset($data['to'][0]['contact_id'])) {
                    $data['to'][0]['contact_id'] = $contactId;
                    if (!isset($data['to'][0]['handle'])) {
                        $data['to'][0]['handle'] = $phone;
                    }
                    $logMessage("✅ contact_id im 'to' Objekt gesetzt (Fallback): " . $contactId);
                }
            } else {
                $logMessage("⚠️ Keine custom_fields in placeholderValues gefunden");
            }
        } else {
            $logMessage("❌ Bedingung NICHT erfüllt für custom_fields Verarbeitung");
        }
        
        $logMessage("=== HelloMateoAPI sendMessage ENDE ===");
        
        // WICHTIG: Warte länger, damit HelloMateo die custom_fields verarbeiten kann
        // Dies gibt HelloMateo Zeit, die custom_fields im Contact-Kontext zu speichern und zu indizieren
        if (!empty($customFieldsData) && $contactId) {
            $logMessage("Warte 2 Sekunden, damit HelloMateo custom_fields verarbeiten kann...");
            usleep(2000000); // 2 Sekunden warten
            
            // Optional: Verifiziere, dass das Field tatsächlich gesetzt wurde
            // Verwende custom_field_id für Verifizierung (genauer als key)
            foreach ($customFieldsData as $verifyFieldName => $verifyFieldValue) {
                $customFieldId = null;
                $customFieldCheck = $this->makeRequest("/custom_field?key=eq.{$verifyFieldName}&limit=1");
                if ($customFieldCheck['success'] && !empty($customFieldCheck['data'])) {
                    $customField = is_array($customFieldCheck['data']) && isset($customFieldCheck['data'][0]) ? $customFieldCheck['data'][0] : $customFieldCheck['data'];
                    $customFieldId = $customField['id'] ?? null;
                }
                
                if ($customFieldId) {
                    $verifyResult = $this->makeRequest("/contact_custom_field?contact_id=eq.{$contactId}&custom_field_id=eq.{$customFieldId}&limit=1");
                } else {
                    $verifyResult = $this->makeRequest("/contact_custom_field?contact_id=eq.{$contactId}&key=eq.{$verifyFieldName}&limit=1");
                }
                
                if ($verifyResult['success'] && !empty($verifyResult['data'])) {
                    $verifyFieldData = is_array($verifyResult['data']) && isset($verifyResult['data'][0]) ? $verifyResult['data'][0] : $verifyResult['data'];
                    $verifyValue = $verifyFieldData['string_value'] ?? null;
                    if ($verifyValue === $verifyFieldValue) {
                        $logMessage("✅ Verifizierung: custom_field '{$verifyFieldName}' ist korrekt gesetzt: '{$verifyValue}'");
                    } else {
                        $logMessage("⚠️ Verifizierung: custom_field '{$verifyFieldName}' hat unerwarteten Wert: '{$verifyValue}' (erwartet: '{$verifyFieldValue}')");
                        $logMessage("⚠️ Das Field existiert, aber der Wert stimmt nicht überein - möglicherweise ein Timing-Problem");
                    }
                } else {
                    $logMessage("⚠️ Verifizierung: Konnte custom_field '{$verifyFieldName}' nicht verifizieren");
                }
            }
            
            $logMessage("Wartezeit abgeschlossen, sende jetzt Nachricht...");
        }
        
        $logMessage("=== VOR send_message API-Call ===");
        $logMessage("Template-ID: " . ($templateId ?? 'NULL'));
        $logMessage("Channel-ID (from): " . ($data['from'] ?? 'NULL'));
        $logMessage("To-Objekt: " . json_encode($data['to'] ?? [], JSON_PRETTY_PRINT));
        $logMessage("Data-Keys: " . implode(', ', array_keys($data)));
        
        // WICHTIG: Wenn ein Template verwendet wird, darf KEIN Text gesendet werden!
        // Die HelloMateo API erlaubt nicht, Template und Text gleichzeitig zu senden
        if ($templateId) {
            // Template-Nachricht: Nur Template-ID, kein Text
            $data['template_id'] = $templateId;
            
            // WICHTIG: Für custom_fields Platzhalter werden die Werte aus dem Contact-Kontext verwendet!
            // Die Fehlermeldung "custom_fields.last_name not found in context" bedeutet,
            // dass HelloMateo die Werte aus dem Contact erwartet, nicht aus placeholder_values
            // Wir haben den Contact bereits oben mit custom_fields erstellt/aktualisiert
            // Daher müssen wir placeholder_values NICHT für custom_fields übergeben
            
            // WICHTIG: Für custom_fields Platzhalter müssen wir die Werte möglicherweise über placeholder_values übergeben
            // ABER in einem speziellen Format, das HelloMateo erwartet
            
            // Prüfe ob es nur custom_fields Platzhalter gibt
            $onlyCustomFields = true;
            $nonCustomFieldsValues = [];
            if ($placeholderValues && is_array($placeholderValues) && !empty($placeholderValues)) {
                foreach ($placeholderValues as $key => $value) {
                    if (is_string($key) && strpos($key, 'custom_fields.') !== 0) {
                        // Nicht-custom_fields Platzhalter
                        $onlyCustomFields = false;
                        $nonCustomFieldsValues[$key] = $value;
                    }
                }
            }
            
            // WICHTIG: Für custom_fields Platzhalter werden die Werte AUSSCHLIESSLICH aus dem Contact-Kontext verwendet!
            // HelloMateo liest custom_fields automatisch aus dem Contact, wenn contact_id im 'to' Objekt vorhanden ist
            // Daher dürfen wir placeholder_values NICHT für custom_fields übergeben!
            // Die Fehlermeldung "custom_fields.last_name not found in context" bedeutet,
            // dass HelloMateo die Werte aus dem Contact erwartet, nicht aus placeholder_values
            // Wir haben den Contact bereits oben mit custom_fields erstellt/aktualisiert
            // Daher müssen wir placeholder_values NICHT für custom_fields übergeben
            
            // Nur wenn es nicht-custom_fields Platzhalter gibt, füge sie zu placeholder_values hinzu
            if (!$onlyCustomFields && !empty($nonCustomFieldsValues)) {
                // Füge nur nicht-custom_fields Werte hinzu
                $data['placeholder_values'] = array_values($nonCustomFieldsValues);
                error_log("HelloMateoAPI sendMessage: Template-Platzhalter-Werte (nur nicht-custom_fields): " . json_encode($data['placeholder_values']));
            } else {
                // Nur custom_fields Platzhalter - Werte kommen AUSSCHLIESSLICH aus Contact-Kontext
                // KEINE placeholder_values für custom_fields übergeben!
                error_log("HelloMateoAPI sendMessage: Template verwendet nur custom_fields - Werte kommen AUSSCHLIESSLICH aus Contact-Kontext, KEINE placeholder_values für custom_fields!");
                error_log("HelloMateoAPI sendMessage: contact_id im 'to' Objekt: " . ($data['to'][0]['contact_id'] ?? 'NICHT GESETZT'));
            }
        } else {
            // Normale Nachricht: Nur Text, kein Template
            if ($message) {
                $data['text'] = $message;
            }
        }
        
        // WICHTIG: Stelle sicher, dass 'to' im richtigen Format ist
        // Die API erwartet ein Array von Objekten mit 'handle' und optional 'contact_id'
        // WICHTIG: Wenn contact_id gefunden wurde, MUSS sie im 'to' Objekt verwendet werden!
        // HelloMateo liest custom_fields aus dem Contact-Kontext, wenn contact_id vorhanden ist
        if (isset($data['to']) && is_array($data['to']) && !empty($data['to'])) {
            // Stelle sicher, dass jedes Element ein Objekt mit 'handle' ist
            foreach ($data['to'] as &$recipient) {
                if (is_string($recipient)) {
                    $recipient = ['handle' => $recipient];
                } elseif (is_array($recipient) && !isset($recipient['handle'])) {
                    $recipient = ['handle' => $recipient[0] ?? $recipient];
                }
                
                // WICHTIG: Wenn contact_id gefunden wurde, MUSS sie verwendet werden!
                // HelloMateo liest custom_fields aus dem Contact-Kontext, wenn contact_id vorhanden ist
                if ($contactId && !isset($recipient['contact_id'])) {
                    $recipient['contact_id'] = $contactId;
                    $logMessage("✅ contact_id im 'to' Objekt gesetzt: " . $contactId);
                    error_log("HelloMateoAPI sendMessage: contact_id im 'to' Objekt gesetzt: " . $contactId);
                }
                
                // WICHTIG: custom_fields werden NICHT direkt im 'to' Objekt übergeben
                // HelloMateo liest custom_fields aus dem Contact-Kontext über contact_id
                // Daher müssen custom_fields VORHER auf dem Contact gesetzt werden (über /contact_custom_field)
            }
            unset($recipient);
        }
        
        // Channel ID (from) - ERFORDERLICH für HelloMateo API!
        // Die API benötigt das "from" Feld, um zu wissen, von welchem Kanal gesendet werden soll
        if ($channelId) {
            $data['from'] = $channelId;
            error_log("HelloMateoAPI sendMessage: Channel-ID (from) gesetzt: " . $channelId);
        } else {
            error_log("HelloMateoAPI sendMessage: WARNUNG - Keine Channel-ID (from) angegeben! Versuche automatisch zu holen...");
            // Versuche Channel automatisch zu holen
            $channelsResult = $this->listChannels();
            if ($channelsResult['success'] && !empty($channelsResult['data'])) {
                $channels = $channelsResult['data'];
                if (is_array($channels)) {
                    if (isset($channels[0])) {
                        $firstChannel = is_array($channels[0]) ? $channels[0] : $channels;
                        $autoChannelId = $firstChannel['id'] ?? $firstChannel['handle'] ?? $firstChannel['channel_id'] ?? null;
                        if ($autoChannelId) {
                            $data['from'] = $autoChannelId;
                            $channelId = $autoChannelId;
                            error_log("HelloMateoAPI sendMessage: Channel-ID automatisch geholt: " . $autoChannelId);
                        }
                    } elseif (isset($channels['id'])) {
                        // Direktes Channel-Objekt
                        $autoChannelId = $channels['id'] ?? $channels['handle'] ?? null;
                        if ($autoChannelId) {
                            $data['from'] = $autoChannelId;
                            $channelId = $autoChannelId;
                            error_log("HelloMateoAPI sendMessage: Channel-ID aus direktem Objekt: " . $autoChannelId);
                        }
                    }
                }
            }
            
            // Wenn immer noch keine Channel-ID, ist das ein Fehler
            if (empty($data['from'])) {
                error_log("HelloMateoAPI sendMessage: FEHLER - Konnte keine Channel-ID finden!");
                return ['success' => false, 'error' => 'Channel-ID (from) ist erforderlich. Bitte konfigurieren Sie die Channel-ID in den HelloMateo-Einstellungen oder stellen Sie sicher, dass Channels verfügbar sind.'];
            }
        }
        
        // Prüfe ob alle erforderlichen Felder vorhanden sind
        if (empty($data['to'])) {
            error_log("HelloMateoAPI sendMessage: FEHLER - 'to' Feld ist leer!");
            return ['success' => false, 'error' => 'Empfänger (to) ist erforderlich'];
        }
        
        if (empty($templateId) && empty($data['text'])) {
            error_log("HelloMateoAPI sendMessage: FEHLER - Weder Template noch Text vorhanden!");
            return ['success' => false, 'error' => 'Entweder Template oder Text ist erforderlich'];
        }
        
        // Prüfe ob 'from' Feld vorhanden ist (wurde oben gesetzt)
        if (empty($data['from'])) {
            error_log("HelloMateoAPI sendMessage: FEHLER - 'from' Feld ist leer!");
            return ['success' => false, 'error' => 'Channel-ID (from) ist erforderlich'];
        }
        
        // Detailliertes Logging der Request-Daten
        error_log("HelloMateoAPI sendMessage Request Data: " . json_encode($data, JSON_PRETTY_PRINT));
        error_log("HelloMateoAPI sendMessage: 'to' Objekt Details: " . json_encode($data['to'] ?? [], JSON_PRETTY_PRINT));
        
        // WICHTIG: Prüfe ob custom_fields im 'to' Objekt vorhanden sind
        if (isset($data['to'][0]['custom_fields'])) {
            error_log("HelloMateoAPI sendMessage: ✅ custom_fields im 'to' Objekt gefunden: " . json_encode($data['to'][0]['custom_fields']));
        } else {
            error_log("HelloMateoAPI sendMessage: ⚠️ KEINE custom_fields im 'to' Objekt gefunden!");
        }
        
        // WICHTIG: Prüfe ob contact_id im 'to' Objekt vorhanden ist
        if (isset($data['to'][0]['contact_id'])) {
            error_log("HelloMateoAPI sendMessage: ✅ contact_id im 'to' Objekt gefunden: " . $data['to'][0]['contact_id']);
        } else {
            error_log("HelloMateoAPI sendMessage: ⚠️ KEINE contact_id im 'to' Objekt gefunden!");
        }
        
        // HelloMateo API Dokumentation: https://docs.getmateo.com/api-reference/send-message/send-a-message
        // Der korrekte Endpunkt ist /send_message (nicht /message oder /rpc/send_message)
        // Format: Direktes Objekt mit "from", "to" (Array), "text" oder "template_id"
        $endpoint = '/send_message';
        $method = 'POST';
        $requestData = $data; // Direktes Objekt, nicht Array
        
        $logMessage("=== Sende Nachricht an HelloMateo API ===");
        $logMessage("Endpoint: {$endpoint}");
        $logMessage("Method: {$method}");
        $logMessage("Request Data: " . json_encode($requestData, JSON_PRETTY_PRINT));
        
        error_log("HelloMateoAPI sendMessage: Sende an Endpunkt: {$endpoint} mit Method: {$method}");
        error_log("HelloMateoAPI sendMessage: Request Data Format: " . gettype($requestData) . " (direktes Objekt)");
        error_log("HelloMateoAPI sendMessage: Request Data Keys: " . implode(', ', array_keys($requestData)));
        error_log("HelloMateoAPI sendMessage: Vollständige Request Data: " . json_encode($requestData, JSON_PRETTY_PRINT));
        
        $result = $this->makeRequest($endpoint, $method, $requestData);
        
        $logMessage("=== API-Antwort erhalten ===");
        $logMessage("Success: " . ($result['success'] ? 'true' : 'false'));
        $logMessage("HTTP Code: " . ($result['http_code'] ?? 'N/A'));
        $logMessage("Error: " . ($result['error'] ?? 'N/A'));
        $logMessage("Response Data: " . json_encode($result['data'] ?? null, JSON_PRETTY_PRINT));
        
        // Bei Rate Limit (429) sofort stoppen - KEINE weiteren Versuche!
        if (!$result['success'] && ($result['http_code'] ?? 0) === 429) {
            error_log("HelloMateoAPI sendMessage: HTTP 429 (Rate Limit) erkannt - KEINE weiteren Versuche um Rate Limit nicht zu verschlimmern!");
        }
        
        // WICHTIG: Wenn Contact nicht gefunden wurde VOR dem Senden (z.B. wegen Rate Limit),
        // aber die Nachricht erfolgreich gesendet wurde, versuche die contact_id aus der Conversation zu holen
        // und setze custom_fields NACH dem Senden
        // WICHTIG: Diese Nachricht wird custom_fields.last_name NICHT haben, aber zukünftige Nachrichten schon
        if (empty($contactId) && !empty($customFieldsData) && $result['success'] && !empty($result['data'])) {
            $logMessage("⚠️ Contact nicht gefunden VOR dem Senden - versuche contact_id aus Conversation zu holen...");
            
            // Extrahiere contact_id aus der Conversation
            $conversation = null;
            if (is_array($result['data'])) {
                if (isset($result['data']['conversation'])) {
                    $conversation = $result['data']['conversation'];
                } elseif (isset($result['data'][0]['conversation'])) {
                    $conversation = $result['data'][0]['conversation'];
                } elseif (isset($result['data']['contact_id'])) {
                    // Direkt contact_id in data
                    $contactIdFromConversation = $result['data']['contact_id'];
                    $logMessage("✅ contact_id direkt aus data geholt: " . $contactIdFromConversation);
                }
            }
            
            if ($conversation && isset($conversation['contact_id'])) {
                $contactIdFromConversation = $conversation['contact_id'];
                $logMessage("✅ contact_id aus Conversation geholt: " . $contactIdFromConversation);
            }
            
            // Setze custom_fields NACH dem Senden (für zukünftige Nachrichten)
            if (isset($contactIdFromConversation)) {
                $logMessage("Setze custom_fields NACH dem Senden der Nachricht (für zukünftige Nachrichten)...");
                $logMessage("⚠️ WICHTIG: Diese Nachricht wird custom_fields.last_name NICHT haben, aber zukünftige Nachrichten schon!");
                
                // Speichere contact_id im Cache für zukünftige Verwendung
                $this->saveContactIdToCache($phone, $contactIdFromConversation);
                $logMessage("✅ contact_id im Cache gespeichert: " . $contactIdFromConversation);
                
                // Setze custom_fields
                $this->setCustomFieldsOnContact($contactIdFromConversation, $customFieldsData, $logMessage);
                
                // Alte Logik entfernt - verwende jetzt setCustomFieldsOnContact Methode
                /*
                foreach ($customFieldsData as $fieldName => $fieldValue) {
                    // Prüfe zuerst, ob das Custom Field bereits existiert
                    $checkResult = $this->makeRequest("/contact_custom_field?contact_id=eq.{$contactIdFromConversation}&key=eq.{$fieldName}&limit=1");
                    $fieldHttpCode = $checkResult['http_code'] ?? 0;
                    
                    if ($fieldHttpCode === 429) {
                        $logMessage("⚠️ Rate Limit (429) beim Prüfen von '{$fieldName}' - überspringe (Rate Limit noch aktiv)");
                        continue;
                    }
                    
                    $fieldExists = false;
                    if ($checkResult['success'] && !empty($checkResult['data'])) {
                        $existingField = is_array($checkResult['data']) && isset($checkResult['data'][0]) ? $checkResult['data'][0] : $checkResult['data'];
                        if ($existingField && isset($existingField['string_value']) && $existingField['string_value'] === $fieldValue) {
                            $logMessage("✅ Custom Field '{$fieldName}' ist bereits gesetzt mit korrektem Wert");
                            $fieldExists = true;
                        } elseif ($existingField && isset($existingField['id'])) {
                            // Field existiert, aber Wert ist anders - Update verwenden
                            $logMessage("Custom Field '{$fieldName}' existiert bereits, aktualisiere Wert...");
                            $updateData = ['string_value' => $fieldValue];
                            $updateResult = $this->makeRequest("/contact_custom_field?id=eq.{$existingField['id']}", 'PATCH', $updateData);
                            if ($updateResult['success']) {
                                $logMessage("✅ Custom Field '{$fieldName}' erfolgreich aktualisiert");
                                $fieldExists = true;
                            } else {
                                $updateHttpCode = $updateResult['http_code'] ?? 0;
                                if ($updateHttpCode === 429) {
                                    $logMessage("⚠️ Rate Limit (429) beim Aktualisieren von '{$fieldName}' - überspringe (Rate Limit noch aktiv)");
                                } else {
                                    $logMessage("❌ Fehler beim Aktualisieren von '{$fieldName}': " . ($updateResult['error'] ?? 'Unbekannt'));
                                }
                            }
                        }
                    }
                    
                    // Wenn Field nicht existiert, erstelle es
                    if (!$fieldExists) {
                        $customFieldData = [
                            'contact_id' => $contactIdFromConversation,
                            'key' => $fieldName,
                            'string_value' => $fieldValue
                        ];
                        
                        $createResult = $this->makeRequest('/contact_custom_field', 'POST', $customFieldData);
                        if ($createResult['success']) {
                            $logMessage("✅ Custom Field '{$fieldName}' erfolgreich gesetzt (nach dem Senden)");
                        } else {
                            $createHttpCode = $createResult['http_code'] ?? 0;
                            if ($createHttpCode === 429) {
                                $logMessage("⚠️ Rate Limit (429) beim Setzen von '{$fieldName}' - überspringe (Rate Limit noch aktiv)");
                            } else {
                                $logMessage("❌ Fehler beim Setzen von '{$fieldName}': " . ($createResult['error'] ?? 'Unbekannt'));
                            }
                        }
                    }
                }
                */
            } else {
                $logMessage("⚠️ Konnte contact_id nicht aus Conversation extrahieren");
                $logMessage("⚠️ custom_fields können nicht gesetzt werden - diese und zukünftige Nachrichten werden custom_fields.last_name nicht haben");
            }
        }
        
        // Detailliertes Logging der Antwort
        error_log("HelloMateoAPI sendMessage Response: " . json_encode($result, JSON_PRETTY_PRINT));
        
        // Prüfe ob die Antwort wirklich erfolgreich ist
        if ($result['success']) {
            // Prüfe ob die Antwort Daten enthält
            if (isset($result['data'])) {
                // Prüfe verschiedene mögliche Antwortstrukturen
                $hasMessageId = false;
                $hasStatus = false;
                $messageId = null;
                
                if (is_array($result['data'])) {
                    // Prüfe ob es ein Array mit Nachrichten-Objekten ist
                    if (!empty($result['data'])) {
                        // Prüfe ob es ein numerisch indiziertes Array ist
                        if (isset($result['data'][0]) && is_array($result['data'][0])) {
                            $firstItem = $result['data'][0];
                        } else {
                            // Es ist ein assoziatives Array (z.B. mit 'message', 'conversation', etc.)
                            $firstItem = $result['data'];
                        }
                        $hasMessageId = isset($firstItem['id']) || isset($firstItem['message_id']) || (isset($result['data']['message']) && isset($result['data']['message']['id']));
                        $hasStatus = isset($firstItem['status']) || (isset($result['data']['message']) && isset($result['data']['message']['status']));
                        
                        // Extrahiere Message-ID
                        if (isset($firstItem['id'])) {
                            $messageId = $firstItem['id'];
                        } elseif (isset($firstItem['message_id'])) {
                            $messageId = $firstItem['message_id'];
                        } elseif (isset($result['data']['message']['id'])) {
                            $messageId = $result['data']['message']['id'];
                        }
                        
                        error_log("HelloMateoAPI sendMessage: Antwort enthält Daten: " . json_encode($result['data'], JSON_PRETTY_PRINT));
                        error_log("HelloMateoAPI sendMessage: Hat Message-ID: " . ($hasMessageId ? 'JA (' . $messageId . ')' : 'NEIN') . ", Hat Status: " . ($hasStatus ? 'JA' : 'NEIN'));
                    } else {
                        error_log("HelloMateoAPI sendMessage: WARNUNG - Antwort ist 'erfolgreich' aber data-Array ist leer!");
                    }
                } else if (is_object($result['data']) || (is_array($result['data']) && isset($result['data']['id']))) {
                    // Prüfe ob es ein einzelnes Nachrichten-Objekt ist
                    $dataObj = is_array($result['data']) ? $result['data'] : (array)$result['data'];
                    $hasMessageId = isset($dataObj['id']) || isset($dataObj['message_id']);
                    $hasStatus = isset($dataObj['status']);
                    
                    // Extrahiere Message-ID
                    if (isset($dataObj['id'])) {
                        $messageId = $dataObj['id'];
                    } elseif (isset($dataObj['message_id'])) {
                        $messageId = $dataObj['message_id'];
                    }
                    
                    error_log("HelloMateoAPI sendMessage: Antwort enthält Nachrichten-Objekt: " . json_encode($result['data'], JSON_PRETTY_PRINT));
                    error_log("HelloMateoAPI sendMessage: Hat Message-ID: " . ($hasMessageId ? 'JA (' . $messageId . ')' : 'NEIN') . ", Hat Status: " . ($hasStatus ? 'JA' : 'NEIN'));
                } else {
                    error_log("HelloMateoAPI sendMessage: Antwort enthält Daten (unbekanntes Format): " . json_encode($result['data'], JSON_PRETTY_PRINT));
                }
                
                // WICHTIG: Wenn keine Message-ID vorhanden ist, wurde die Nachricht möglicherweise nicht wirklich gesendet
                if (!$hasMessageId) {
                    error_log("HelloMateoAPI sendMessage: ❌ FEHLER - Keine Message-ID in der Antwort gefunden! Die Nachricht wurde möglicherweise NICHT wirklich gesendet.");
                    error_log("HelloMateoAPI sendMessage: Vollständige Antwort: " . json_encode($result, JSON_PRETTY_PRINT));
                    // Setze success auf false, wenn keine Message-ID vorhanden ist
                    $result['success'] = false;
                    $result['error'] = 'Keine Message-ID in der API-Antwort gefunden. Die Nachricht wurde möglicherweise nicht wirklich gesendet. Antwort: ' . json_encode($result['data']);
                } else {
                    error_log("HelloMateoAPI sendMessage: ✅ Message-ID gefunden: " . $messageId . " - Nachricht wurde erfolgreich gesendet.");
                    $result['message_id'] = $messageId;
                }
            } else {
                error_log("HelloMateoAPI sendMessage: ❌ FEHLER - Antwort ist 'erfolgreich' aber enthält kein 'data' Feld!");
                error_log("HelloMateoAPI sendMessage: Vollständige Antwort: " . json_encode($result, JSON_PRETTY_PRINT));
                // Setze success auf false, wenn kein data-Feld vorhanden ist
                $result['success'] = false;
                $result['error'] = 'Keine Daten in der API-Antwort gefunden. Die Nachricht wurde möglicherweise nicht wirklich gesendet.';
            }
        } else {
            error_log("HelloMateoAPI sendMessage: FEHLER - " . ($result['error'] ?? 'Unbekannter Fehler'));
        }
        
        return $result;
    }
    
    /**
     * Sendet eine Broadcast-Nachricht
     */
    public function sendBroadcast($to, $message, $templateId = null) {
        $data = [
            'to' => is_array($to) ? $to : [['handle' => $to]],
            'text' => $message
        ];
        
        if ($templateId) {
            $data['template_id'] = $templateId;
        }
        
        return $this->makeRequest('/send_broadcast', 'POST', $data);
    }
    
    /**
     * Listet alle Nachrichten
     */
    public function listMessages($limit = 50, $offset = 0) {
        return $this->makeRequest("/message?limit={$limit}&offset={$offset}");
    }
    
    /**
     * Listet alle Kontakte
     */
    public function listContacts($limit = 50, $offset = 0) {
        // HelloMateo verwendet PostgREST - limit und offset als Query-Parameter
        return $this->makeRequest("/contact?limit={$limit}&offset={$offset}");
    }
    
    /**
     * Findet oder erstellt einen Contact anhand der Telefonnummer
     * Gibt die contact_id zurück
     */
    public function findOrCreateContactByPhone($phone) {
        // Normalisiere Telefonnummer
        $normalizedPhone = preg_replace('/[^0-9+]/', '', $phone);
        
        // Versuche zuerst, den Contact zu finden
        $findResult = $this->makeRequest("/contact?handle=eq.{$normalizedPhone}&limit=1");
        
        if ($findResult['success'] && !empty($findResult['data'])) {
            $contact = is_array($findResult['data']) && isset($findResult['data'][0]) 
                ? $findResult['data'][0] 
                : $findResult['data'];
            
            if (isset($contact['id'])) {
                error_log("HelloMateoAPI findOrCreateContactByPhone: Contact gefunden: " . $contact['id']);
                return $contact['id'];
            }
        }
        
        // Contact nicht gefunden - erstelle neuen Contact
        error_log("HelloMateoAPI findOrCreateContactByPhone: Contact nicht gefunden, erstelle neuen Contact für: " . $normalizedPhone);
        $createResult = $this->createContact([
            'handle' => $normalizedPhone
        ]);
        
        if ($createResult['success'] && !empty($createResult['data'])) {
            $newContact = is_array($createResult['data']) && isset($createResult['data'][0]) 
                ? $createResult['data'][0] 
                : $createResult['data'];
            
            if (isset($newContact['id'])) {
                error_log("HelloMateoAPI findOrCreateContactByPhone: Neuer Contact erstellt: " . $newContact['id']);
                return $newContact['id'];
            }
        }
        
        error_log("HelloMateoAPI findOrCreateContactByPhone: FEHLER - Konnte Contact weder finden noch erstellen");
        return null;
    }
    
    /**
     * Erstellt einen Kontakt
     */
    public function createContact($contactData) {
        return $this->makeRequest('/contact', 'POST', $contactData);
    }
    
    /**
     * Aktualisiert einen Kontakt
     */
    public function updateContact($contactId, $contactData) {
        return $this->makeRequest("/contact?id=eq.{$contactId}", 'PATCH', $contactData);
    }
    
    /**
     * Löscht einen Kontakt
     */
    public function deleteContact($contactId) {
        return $this->makeRequest("/contact?id=eq.{$contactId}", 'DELETE');
    }
    
    /**
     * Listet alle Templates
     */
    public function listTemplates($limit = 50, $offset = 0) {
        // HelloMateo verwendet PostgREST - limit und offset als Query-Parameter
        // Versuche auch Components zu laden mit select Parameter
        return $this->makeRequest("/template?limit={$limit}&offset={$offset}&select=*");
    }
    
    /**
     * Holt ein einzelnes Template nach ID mit allen Details
     */
    public function getTemplateById($templateId) {
        // Versuche verschiedene Endpunkte
        $endpoints = [
            "/template?id=eq.{$templateId}",
            "/template/{$templateId}",
            "/templates/{$templateId}"
        ];
        
        foreach ($endpoints as $endpoint) {
            $result = $this->makeRequest($endpoint);
            if ($result['success'] && !empty($result['data'])) {
                return $result;
            }
        }
        
        // Wenn keiner funktioniert, gib den letzten Fehler zurück
        return $result ?? ['success' => false, 'error' => 'Template nicht gefunden'];
    }
    
    /**
     * Listet alle Konversationen
     */
    public function listConversations($limit = 50, $offset = 0) {
        // HelloMateo verwendet PostgREST - limit und offset als Query-Parameter
        return $this->makeRequest("/conversation?limit={$limit}&offset={$offset}");
    }
    
    /**
     * Holt eine einzelne Konversation nach ID
     */
    public function getConversationById($conversationId) {
        // Versuche verschiedene Endpunkte
        $endpoints = [
            "/conversation?id=eq.{$conversationId}",
            "/conversation/{$conversationId}",
            "/conversations/{$conversationId}"
        ];
        
        foreach ($endpoints as $endpoint) {
            $result = $this->makeRequest($endpoint);
            if ($result['success'] && !empty($result['data'])) {
                return $result;
            }
        }
        
        // Wenn keiner funktioniert, gib den letzten Fehler zurück
        return $result ?? ['success' => false, 'error' => 'Konversation nicht gefunden'];
    }
    
    /**
     * Listet alle Kanäle
     */
    public function listChannels() {
        // Versuche verschiedene Endpunkte für Kanäle
        $endpoints = ['/channel', '/channels'];
        foreach ($endpoints as $endpoint) {
            $result = $this->makeRequest($endpoint);
            if ($result['success']) {
                return $result;
            }
        }
        // Wenn keiner funktioniert, gib den letzten Fehler zurück
        return $result;
    }
    
    /**
     * Testet die API-Verbindung
     */
    public function testConnection() {
        // Prüfe zuerst ob API-Key vorhanden ist
        if (!$this->apiKey) {
            return ['success' => false, 'error' => 'API-Key nicht gefunden'];
        }
        
        // Versuche Kanäle abzurufen als Test
        $result = $this->listChannels();
        return $result;
    }
    
    /**
     * Gibt den aktuellen API-Key zurück (für Debugging)
     */
    public function getApiKeyInfo() {
        $jwtInfo = null;
        if ($this->apiKey) {
            $parts = explode('.', $this->apiKey);
            if (count($parts) === 3) {
                // Versuche JWT zu dekodieren
                try {
                    $payload = json_decode(base64_decode(str_pad(strtr($parts[1], '-_', '+/'), strlen($parts[1]) % 4, '=', STR_PAD_RIGHT)), true);
                    if ($payload) {
                        $jwtInfo = [
                            'is_jwt' => true,
                            'payload' => $payload,
                            'exp' => $payload['exp'] ?? null,
                            'exp_formatted' => isset($payload['exp']) ? date('Y-m-d H:i:s', $payload['exp']) : null,
                            'is_expired' => isset($payload['exp']) && $payload['exp'] < time(),
                            'iat' => $payload['iat'] ?? null,
                            'iat_formatted' => isset($payload['iat']) ? date('Y-m-d H:i:s', (int)$payload['iat']) : null,
                            'iss' => $payload['iss'] ?? null,
                            'sub' => $payload['sub'] ?? null
                        ];
                    }
                } catch (Exception $e) {
                    error_log("HelloMateoAPI: Fehler beim Dekodieren des JWT: " . $e->getMessage());
                }
            }
        }
        
        return [
            'has_key' => !empty($this->apiKey),
            'key_length' => $this->apiKey ? strlen($this->apiKey) : 0,
            'autohaus_id' => $this->autohausId,
            'key_prefix' => $this->apiKey ? substr($this->apiKey, 0, 20) . '...' : null,
            'jwt_info' => $jwtInfo
        ];
    }
    
    /**
     * Holt contact_id aus lokalem Cache (Datenbank)
     */
    private function getContactIdFromCache($phone) {
        try {
            require_once __DIR__ . '/../../config/database.php';
            $db = new Database();
            $conn = $db->getConnection();
            
            if (!$conn) {
                return null;
            }
            
            $normalizedPhone = preg_replace('/[^0-9+]/', '', $phone);
            
            // Prüfe ob Tabelle existiert, wenn nicht erstelle sie
            $stmt = $conn->query("SELECT 1 FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name = 'whatsapp_contact_cache'");
            if ($stmt->rowCount() === 0) {
                // Erstelle Tabelle
                $conn->exec("
                    CREATE TABLE IF NOT EXISTS whatsapp_contact_cache (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        phone VARCHAR(50) UNIQUE NOT NULL,
                        contact_id VARCHAR(255) NOT NULL,
                        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                        INDEX idx_phone (phone)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
                ");
            }
            
            // Suche in Cache
            $stmt = $conn->prepare("SELECT contact_id FROM whatsapp_contact_cache WHERE phone = ? LIMIT 1");
            $stmt->execute([$normalizedPhone]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($result && !empty($result['contact_id'])) {
                return $result['contact_id'];
            }
            
            return null;
        } catch (Exception $e) {
            error_log("HelloMateoAPI getContactIdFromCache Fehler: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Speichert contact_id im lokalen Cache
     */
    private function saveContactIdToCache($phone, $contactId) {
        try {
            require_once __DIR__ . '/../../config/database.php';
            $db = new Database();
            $conn = $db->getConnection();
            
            if (!$conn || empty($contactId)) {
                return false;
            }
            
            $normalizedPhone = preg_replace('/[^0-9+]/', '', $phone);
            
            // Prüfe ob Tabelle existiert
            $stmt = $conn->query("SELECT 1 FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name = 'whatsapp_contact_cache'");
            if ($stmt->rowCount() === 0) {
                // Erstelle Tabelle
                $conn->exec("
                    CREATE TABLE IF NOT EXISTS whatsapp_contact_cache (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        phone VARCHAR(50) UNIQUE NOT NULL,
                        contact_id VARCHAR(255) NOT NULL,
                        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                        INDEX idx_phone (phone)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
                ");
            }
            
            // Speichere oder aktualisiere
            $stmt = $conn->prepare("
                INSERT INTO whatsapp_contact_cache (phone, contact_id) 
                VALUES (?, ?) 
                ON DUPLICATE KEY UPDATE contact_id = ?, updated_at = CURRENT_TIMESTAMP
            ");
            $stmt->execute([$normalizedPhone, $contactId, $contactId]);
            
            return true;
        } catch (Exception $e) {
            error_log("HelloMateoAPI saveContactIdToCache Fehler: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Setzt custom_fields auf einem Contact
     */
    private function setCustomFieldsOnContact($contactId, $customFieldsData, $logMessage) {
        if (empty($contactId) || empty($customFieldsData)) {
            return false;
        }
        
        $logMessage("Setze custom_fields über /contact_custom_field Endpunkt für Contact: " . $contactId);
        $allFieldsSet = true;
        
        foreach ($customFieldsData as $fieldName => $fieldValue) {
            // SCHRITT 1: Prüfe ob custom_field existiert (nicht contact_custom_field!)
            $customFieldCheck = $this->makeRequest("/custom_field?key=eq.{$fieldName}&limit=1");
            $customFieldId = null;
            
            if ($customFieldCheck['success'] && !empty($customFieldCheck['data'])) {
                $existingCustomField = is_array($customFieldCheck['data']) && isset($customFieldCheck['data'][0]) ? $customFieldCheck['data'][0] : $customFieldCheck['data'];
                if ($existingCustomField && isset($existingCustomField['id'])) {
                    $customFieldId = $existingCustomField['id'];
                    $logMessage("✅ Custom Field '{$fieldName}' existiert bereits (ID: {$customFieldId})");
                }
            }
            
            // SCHRITT 2: Wenn custom_field nicht existiert, erstelle es
            if (!$customFieldId) {
                $createCustomFieldResult = $this->makeRequest('/custom_field', 'POST', [
                    'key' => $fieldName,
                    'name' => ucfirst(str_replace('_', ' ', $fieldName)), // z.B. "last_name" -> "Last Name"
                    'type' => 'string', // String-Feld
                    'entity' => 'contact' // Contact-Entity
                ]);
                
                if ($createCustomFieldResult['success'] && !empty($createCustomFieldResult['data'])) {
                    $newCustomField = is_array($createCustomFieldResult['data']) && isset($createCustomFieldResult['data'][0]) ? $createCustomFieldResult['data'][0] : $createCustomFieldResult['data'];
                    if ($newCustomField && isset($newCustomField['id'])) {
                        $customFieldId = $newCustomField['id'];
                        $logMessage("✅ Custom Field '{$fieldName}' erstellt (ID: {$customFieldId})");
                    }
                } else {
                    $createCustomFieldHttpCode = $createCustomFieldResult['http_code'] ?? 0;
                    if ($createCustomFieldHttpCode === 429) {
                        $logMessage("⚠️ Rate Limit (429) beim Erstellen von Custom Field '{$fieldName}' - überspringe");
                        $allFieldsSet = false;
                        continue;
                    } else {
                        $logMessage("❌ Fehler beim Erstellen von Custom Field '{$fieldName}': " . ($createCustomFieldResult['error'] ?? 'Unbekannt'));
                        $allFieldsSet = false;
                        continue;
                    }
                }
            }
            
            // SCHRITT 3: Prüfe ob contact_custom_field bereits existiert (mit contact_id UND custom_field_id)
            if ($customFieldId) {
                $logMessage("Prüfe ob contact_custom_field existiert (contact_id: {$contactId}, custom_field_id: {$customFieldId})...");
                
                // Versuche verschiedene PostgREST-Filter-Varianten
                // WICHTIG: Die Tabelle hat KEINE 'id' Spalte - Primary Key ist vermutlich Composite (contact_id + custom_field_id)
                $checkVariants = [
                    "/contact_custom_field?contact_id=eq.{$contactId}&custom_field_id=eq.{$customFieldId}&limit=1",
                    "/contact_custom_field?contact_id=eq.{$contactId}&custom_field_id=eq.{$customFieldId}",
                    "/contact_custom_field?contact_id={$contactId}&custom_field_id={$customFieldId}&limit=1"
                ];
                
                $checkResult = null;
                $fieldHttpCode = 0;
                
                foreach ($checkVariants as $variantIndex => $variant) {
                    $logMessage("Prüfe Variante " . ($variantIndex + 1) . ": {$variant}");
                    $checkResult = $this->makeRequest($variant);
                    $fieldHttpCode = $checkResult['http_code'] ?? 0;
                    
                    if ($fieldHttpCode === 429) {
                        $logMessage("⚠️ Rate Limit (429) beim Prüfen von '{$fieldName}' - überspringe");
                        $allFieldsSet = false;
                        continue 2; // Breche beide Schleifen ab
                    }
                    
                    if ($checkResult['success'] && !empty($checkResult['data'])) {
                        $logMessage("✅ Variante " . ($variantIndex + 1) . " erfolgreich - contact_custom_field gefunden");
                        break; // Erfolgreich gefunden
                    } else {
                        $logMessage("❌ Variante " . ($variantIndex + 1) . " fehlgeschlagen: " . ($checkResult['error'] ?? 'Keine Daten') . " (HTTP: {$fieldHttpCode}, success: " . ($checkResult['success'] ? 'true' : 'false') . ", data empty: " . (empty($checkResult['data']) ? 'true' : 'false') . ")");
                    }
                }
                
                // FALLBACK: Wenn direkte Suche fehlschlägt, hole alle contact_custom_fields für diesen Contact
                // WICHTIG: Prüfe auch, ob checkResult zwar success=true hat, aber keine Daten zurückgegeben wurden
                if (!$checkResult || !$checkResult['success'] || empty($checkResult['data'])) {
                    $logMessage("⚠️ Direkte Suche fehlgeschlagen (checkResult: " . ($checkResult ? 'vorhanden' : 'null') . ", success: " . ($checkResult['success'] ?? 'N/A') . ", data empty: " . (empty($checkResult['data']) ? 'true' : 'false') . "), versuche Fallback...");
                    $logMessage("⚠️ Direkte Suche fehlgeschlagen, versuche Fallback: Hole alle contact_custom_fields für diesen Contact...");
                    $fallbackResult = $this->makeRequest("/contact_custom_field?contact_id=eq.{$contactId}&limit=100");
                    
                    if ($fallbackResult['success'] && !empty($fallbackResult['data'])) {
                        $allFields = is_array($fallbackResult['data']) ? $fallbackResult['data'] : [$fallbackResult['data']];
                        $logMessage("ℹ️ Fallback: Gefunden " . count($allFields) . " contact_custom_fields für diesen Contact");
                        foreach ($allFields as $field) {
                            $fieldCustomFieldId = $field['custom_field_id'] ?? null;
                            $logMessage("ℹ️ Prüfe Field: custom_field_id=" . ($fieldCustomFieldId ?? 'N/A') . " (erwartet: {$customFieldId})");
                            if ($fieldCustomFieldId && $fieldCustomFieldId === $customFieldId) {
                                $checkResult = ['success' => true, 'data' => [$field]];
                                $logMessage("✅ Contact Custom Field '{$fieldName}' über Fallback gefunden (ID: " . ($field['id'] ?? 'N/A') . ")");
                                // WICHTIG: Speichere die 'id' aus dem Fallback
                                if (isset($field['id'])) {
                                    $existingContactCustomFieldId = $field['id'];
                                    $logMessage("✅ 'id' aus Fallback gespeichert: {$existingContactCustomFieldId}");
                                }
                                break;
                            }
                        }
                        if (!$checkResult || !$checkResult['success'] || empty($checkResult['data'])) {
                            $logMessage("⚠️ Fallback: Contact Custom Field '{$fieldName}' nicht in den gefundenen Fields gefunden");
                        }
                    } else {
                        $logMessage("❌ Fallback fehlgeschlagen: " . ($fallbackResult['error'] ?? 'Unbekannt'));
                    }
                }
                
                $fieldExists = false;
                // WICHTIG: existingContactCustomFieldId muss VORHER gesetzt werden, falls es im Fallback gefunden wurde
                $existingContactCustomFieldId = null;
                
                if ($checkResult && $checkResult['success'] && !empty($checkResult['data'])) {
                    $existingField = is_array($checkResult['data']) && isset($checkResult['data'][0]) ? $checkResult['data'][0] : $checkResult['data'];
                    $logMessage("ℹ️ Prüfe gefundenes Field: " . json_encode($existingField));
                    
                    // WICHTIG: Die Tabelle hat KEINE 'id' Spalte - Primary Key ist Composite (contact_id + custom_field_id)
                    $fieldContactId = $existingField['contact_id'] ?? null;
                    $fieldCustomFieldId = $existingField['custom_field_id'] ?? null;
                    
                    if ($existingField && $fieldContactId && $fieldCustomFieldId) {
                        $logMessage("ℹ️ Contact Custom Field '{$fieldName}' gefunden (contact_id: {$fieldContactId}, custom_field_id: {$fieldCustomFieldId})");
                        
                        $currentValue = $existingField['string_value'] ?? null;
                        $logMessage("ℹ️ Aktueller Wert: " . ($currentValue ?? 'N/A') . ", erwarteter Wert: {$fieldValue}");
                        
                        if ($currentValue !== null && $currentValue === $fieldValue) {
                            $logMessage("✅ Contact Custom Field '{$fieldName}' ist bereits gesetzt mit korrektem Wert");
                            $fieldExists = true;
                        } else {
                            // Field existiert, aber Wert ist anders oder fehlt - Update verwenden
                            $logMessage("Contact Custom Field '{$fieldName}' existiert bereits (aktueller Wert: '{$currentValue}'), aktualisiere auf '{$fieldValue}'...");
                            // WICHTIG: Setze beide Felder - 'value' und 'string_value'
                            // In erfolgreichen Logs sind beide gesetzt: "value":"Schöttle","string_value":"Schöttle"
                            $updateData = [
                                'string_value' => $fieldValue,
                                'value' => $fieldValue  // Auch 'value' setzen
                            ];
                            
                            // WICHTIG: PostgREST benötigt möglicherweise die 'id' für Updates
                            // Versuche zuerst, die 'id' zu holen, indem wir alle Fields für den Contact abrufen
                            if (!$existingContactCustomFieldId) {
                                $logMessage("⚠️ Keine 'id' im Response - versuche 'id' über Fallback zu holen...");
                                $fallbackResult = $this->makeRequest("/contact_custom_field?contact_id=eq.{$fieldContactId}&custom_field_id=eq.{$fieldCustomFieldId}&select=id&limit=1");
                                if ($fallbackResult['success'] && !empty($fallbackResult['data'])) {
                                    $fallbackField = is_array($fallbackResult['data']) && isset($fallbackResult['data'][0]) ? $fallbackResult['data'][0] : $fallbackResult['data'];
                                    $existingContactCustomFieldId = $fallbackField['id'] ?? null;
                                    if ($existingContactCustomFieldId) {
                                        $logMessage("✅ 'id' über Fallback geholt: {$existingContactCustomFieldId}");
                                    }
                                }
                            }
                            
                            // Verwende id für Update, falls vorhanden, sonst contact_id + custom_field_id
                            if ($existingContactCustomFieldId) {
                                $updateUrl = "/contact_custom_field?id=eq.{$existingContactCustomFieldId}";
                                $logMessage("Verwende Update-URL mit id: {$updateUrl}");
                            } else {
                                $updateUrl = "/contact_custom_field?contact_id=eq.{$fieldContactId}&custom_field_id=eq.{$fieldCustomFieldId}";
                                $logMessage("Verwende Update-URL mit contact_id + custom_field_id: {$updateUrl}");
                            }
                            
                            $logMessage("Update Data: " . json_encode($updateData));
                            $updateResult = $this->makeRequest($updateUrl, 'PATCH', $updateData);
                            $logMessage("Update Response: " . json_encode($updateResult));
                            $logMessage("Update HTTP Code: " . ($updateResult['http_code'] ?? 'N/A'));
                            
                            if ($updateResult['success']) {
                                // Verifiziere sofort, ob das Update wirklich funktioniert hat
                                $verifyUpdate = $this->makeRequest("/contact_custom_field?contact_id=eq.{$fieldContactId}&custom_field_id=eq.{$fieldCustomFieldId}&limit=1");
                                if ($verifyUpdate['success'] && !empty($verifyUpdate['data'])) {
                                    $verifyFieldData = is_array($verifyUpdate['data']) && isset($verifyUpdate['data'][0]) ? $verifyUpdate['data'][0] : $verifyUpdate['data'];
                                    $verifyValue = $verifyFieldData['string_value'] ?? null;
                                    if ($verifyValue === $fieldValue) {
                                        $logMessage("✅ Contact Custom Field '{$fieldName}' erfolgreich aktualisiert und verifiziert: '{$verifyValue}'");
                                        $fieldExists = true;
                                    } else {
                                        $logMessage("⚠️ Update gemeldet als erfolgreich, aber Wert ist noch nicht gesetzt (aktuell: '{$verifyValue}', erwartet: '{$fieldValue}')");
                                        $logMessage("⚠️ PostgREST unterstützt möglicherweise keine Updates mit Composite Keys - verwende Fallback: Lösche und erstelle neu");
                                        
                                        // FALLBACK: Lösche das Field und erstelle es neu
                                        $deleteResult = $this->makeRequest("/contact_custom_field?contact_id=eq.{$fieldContactId}&custom_field_id=eq.{$fieldCustomFieldId}", 'DELETE');
                                        if ($deleteResult['success']) {
                                            $logMessage("✅ Field gelöscht - erstelle es neu...");
                                            usleep(2000000); // 2 Sekunden warten nach Löschen
                                            
                                            // WICHTIG: Setze beide Felder - 'value' und 'string_value'
                                            // In erfolgreichen Logs sind beide gesetzt: "value":"Schöttle","string_value":"Schöttle"
                                            $createData = [
                                                'contact_id' => $fieldContactId,
                                                'custom_field_id' => $fieldCustomFieldId,
                                                'string_value' => $fieldValue,
                                                'value' => $fieldValue  // Auch 'value' setzen
                                            ];
                                            $logMessage("Erstelle Field mit Daten: " . json_encode($createData));
                                            $createResult = $this->makeRequest('/contact_custom_field', 'POST', $createData);
                                            $logMessage("Create Response: " . json_encode($createResult));
                                            
                                            // Warte länger nach dem Erstellen, damit PostgREST die Werte verarbeiten kann
                                            usleep(2000000); // 2 Sekunden warten
                                            
                                            if ($createResult['success']) {
                                                $logMessage("✅ Contact Custom Field '{$fieldName}' erfolgreich neu erstellt mit Wert: '{$fieldValue}'");
                                                
                                                // WICHTIG: Prüfe die Response-Daten, ob das Field mit dem Wert erstellt wurde
                                                if (!empty($createResult['data'])) {
                                                    $createdField = is_array($createResult['data']) && isset($createResult['data'][0]) ? $createResult['data'][0] : $createResult['data'];
                                                    $createdValue = $createdField['string_value'] ?? null;
                                                    $logMessage("ℹ️ Erstelltes Field aus Response: " . json_encode($createdField));
                                                    if ($createdValue === $fieldValue) {
                                                        $logMessage("✅ Field wurde mit korrektem Wert erstellt (aus Response): '{$createdValue}'");
                                                        $fieldExists = true;
                                                    } else {
                                                        $logMessage("⚠️ Field wurde erstellt, aber Response zeigt anderen Wert: '{$createdValue}' (erwartet: '{$fieldValue}')");
                                                    }
                                                }
                                                
                                                // Verifiziere das neu erstellte Field nach Wartezeit
                                                usleep(2000000); // 2 Sekunden warten nach Erstellen
                                                $verifyCreate = $this->makeRequest("/contact_custom_field?contact_id=eq.{$fieldContactId}&custom_field_id=eq.{$fieldCustomFieldId}&limit=1");
                                                if ($verifyCreate['success'] && !empty($verifyCreate['data'])) {
                                                    $verifyCreateField = is_array($verifyCreate['data']) && isset($verifyCreate['data'][0]) ? $verifyCreate['data'][0] : $verifyCreate['data'];
                                                    $verifyCreateValue = $verifyCreateField['string_value'] ?? null;
                                                    $logMessage("ℹ️ Verifiziertes Field: " . json_encode($verifyCreateField));
                                                    if ($verifyCreateValue === $fieldValue) {
                                                        $logMessage("✅ Verifizierung nach Neuerstellung: Field hat korrekten Wert: '{$verifyCreateValue}'");
                                                        $fieldExists = true;
                                                    } else {
                                                        $logMessage("⚠️ Verifizierung nach Neuerstellung: Field hat unerwarteten Wert: '{$verifyCreateValue}' (erwartet: '{$fieldValue}')");
                                                        $logMessage("⚠️ Field wurde erstellt, aber Wert stimmt nicht überein - möglicherweise ein Timing-Problem oder PostgREST-Problem");
                                                        $fieldExists = true; // Trotzdem als erfolgreich markieren, da Field erstellt wurde
                                                    }
                                                } else {
                                                    $logMessage("⚠️ Verifizierung nach Neuerstellung nicht möglich - Field wurde aber erstellt");
                                                    $fieldExists = true;
                                                }
                                            } else {
                                                $logMessage("❌ Fehler beim Neuerstellen von '{$fieldName}': " . ($createResult['error'] ?? 'Unbekannt'));
                                                $logMessage("❌ Create Data: " . json_encode($createData));
                                                $allFieldsSet = false;
                                            }
                                        } else {
                                            $logMessage("❌ Fehler beim Löschen von '{$fieldName}': " . ($deleteResult['error'] ?? 'Unbekannt'));
                                            $allFieldsSet = false;
                                        }
                                    }
                                } else {
                                    $logMessage("✅ Contact Custom Field '{$fieldName}' Update erfolgreich (Verifizierung nicht möglich)");
                                    $fieldExists = true;
                                }
                            } else {
                                $updateHttpCode = $updateResult['http_code'] ?? 0;
                                if ($updateHttpCode === 429) {
                                    $logMessage("⚠️ Rate Limit (429) beim Aktualisieren von '{$fieldName}' - überspringe");
                                    $allFieldsSet = false;
                                } else {
                                    $logMessage("❌ Fehler beim Aktualisieren von '{$fieldName}': " . ($updateResult['error'] ?? 'Unbekannt'));
                                    $logMessage("❌ Update URL: {$updateUrl}");
                                    $logMessage("❌ Update Data: " . json_encode($updateData));
                                    $allFieldsSet = false;
                                }
                            }
                        }
                    } else {
                        $logMessage("⚠️ Gefundenes Field hat keine contact_id oder custom_field_id: " . json_encode($existingField));
                    }
                } else {
                    if ($checkResult && $checkResult['success'] && empty($checkResult['data'])) {
                        $logMessage("ℹ️ Prüfung erfolgreich, aber keine Daten zurückgegeben - contact_custom_field existiert noch nicht");
                    } elseif ($checkResult && !$checkResult['success']) {
                        $logMessage("⚠️ Prüfung fehlgeschlagen: " . ($checkResult['error'] ?? 'Unbekannt'));
                    } else {
                        $logMessage("⚠️ checkResult ist leer oder fehlgeschlagen: success=" . ($checkResult['success'] ?? 'N/A') . ", data empty=" . (empty($checkResult['data']) ? 'true' : 'false'));
                        $logMessage("ℹ️ Contact Custom Field '{$fieldName}' existiert noch nicht - wird erstellt");
                    }
                }
                
                // SCHRITT 4: Wenn contact_custom_field nicht existiert, erstelle es
                if (!$fieldExists) {
                    // WICHTIG: Setze beide Felder - 'value' und 'string_value'
                    // In erfolgreichen Logs sind beide gesetzt: "value":"Schöttle","string_value":"Schöttle"
                    $contactCustomFieldData = [
                        'contact_id' => $contactId,
                        'custom_field_id' => $customFieldId,
                        'string_value' => $fieldValue,
                        'value' => $fieldValue  // Auch 'value' setzen
                    ];
                    
                    $createResult = $this->makeRequest('/contact_custom_field', 'POST', $contactCustomFieldData);
                    if ($createResult['success']) {
                        $logMessage("✅ Contact Custom Field '{$fieldName}' erfolgreich gesetzt");
                    } else {
                        $createHttpCode = $createResult['http_code'] ?? 0;
                        if ($createHttpCode === 429) {
                            $logMessage("⚠️ Rate Limit (429) beim Setzen von '{$fieldName}' - überspringe");
                            $allFieldsSet = false;
                        } else {
                            // Wenn "duplicate key" Fehler, bedeutet das, dass es bereits existiert - versuche zu aktualisieren
                            if (strpos($createResult['error'] ?? '', 'duplicate') !== false || strpos($createResult['error'] ?? '', 'unique') !== false) {
                                $logMessage("⚠️ Contact Custom Field '{$fieldName}' existiert bereits (duplicate key) - versuche zu aktualisieren...");
                                
                                // Versuche verschiedene Suchvarianten
                                $retryVariants = [
                                    "/contact_custom_field?contact_id=eq.{$contactId}&custom_field_id=eq.{$customFieldId}&limit=1",
                                    "/contact_custom_field?contact_id=eq.{$contactId}&custom_field_id=eq.{$customFieldId}",
                                    "/contact_custom_field?contact_id={$contactId}&custom_field_id={$customFieldId}&limit=1"
                                ];
                                
                                $retryField = null;
                                foreach ($retryVariants as $retryVariant) {
                                    $retryCheck = $this->makeRequest($retryVariant);
                                    if ($retryCheck['success'] && !empty($retryCheck['data'])) {
                                        $retryField = is_array($retryCheck['data']) && isset($retryCheck['data'][0]) ? $retryCheck['data'][0] : $retryCheck['data'];
                                        if ($retryField && isset($retryField['id'])) {
                                            break;
                                        }
                                    }
                                }
                                
                                if ($retryField && isset($retryField['id'])) {
                                    $updateData = ['string_value' => $fieldValue];
                                    $updateResult = $this->makeRequest("/contact_custom_field?id=eq.{$retryField['id']}", 'PATCH', $updateData);
                                    if ($updateResult['success']) {
                                        $logMessage("✅ Contact Custom Field '{$fieldName}' erfolgreich aktualisiert (nach duplicate key Fehler)");
                                        $fieldExists = true; // Markiere als erfolgreich
                                    } else {
                                        $logMessage("❌ Fehler beim Aktualisieren von '{$fieldName}' (nach duplicate key): " . ($updateResult['error'] ?? 'Unbekannt'));
                                        $allFieldsSet = false;
                                    }
                                } else {
                                    $logMessage("❌ Konnte contact_custom_field nicht finden (nach duplicate key) - möglicherweise ein Timing-Problem");
                                    $logMessage("ℹ️ Das contact_custom_field existiert, aber konnte nicht abgerufen werden. Es wird bei der nächsten Nachricht funktionieren.");
                                    // Betrachte als teilweise erfolgreich, da das Field existiert
                                    $allFieldsSet = false;
                                }
                            } else {
                                $logMessage("❌ Fehler beim Setzen von '{$fieldName}': " . ($createResult['error'] ?? 'Unbekannt'));
                                $allFieldsSet = false;
                            }
                        }
                    }
                }
            }
        }
        
        return $allFieldsSet;
    }
}
?>

