<?php
/**
 * HelloMateo / WhatsApp Management
 * Zentrale Verwaltung für WhatsApp-Funktionen
 */

// Fehlerbehandlung: Alle Fehler in Logs schreiben, nicht ausgeben (wichtig für AJAX)
ini_set('display_errors', 0);
ini_set('log_errors', 1);
error_reporting(E_ALL);

session_start();

// Fehlerbehandlung für require_once
try {
    require_once '../config/database.php';
    require_once 'includes/helpers.php';
    require_once 'includes/hellomateo_api.php';
    require_once '../api/autohaus_whatsapp_notifications.php';
    require_once 'includes/template_config_helper.php';
} catch (Exception $e) {
    // Wenn AJAX-Request, JSON-Fehler zurückgeben
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajax_action'])) {
        header('Content-Type: application/json');
        echo json_encode(['success' => false, 'error' => 'Fehler beim Laden der Abhängigkeiten: ' . $e->getMessage()]);
        exit;
    }
    // Sonst Fehler weiterwerfen
    throw $e;
}

// Check if user is logged in
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

$db = new Database();
$conn = $db->getConnection();

// Get current user data with autohaus information
$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_name = 'Administrator';
$current_user_role = 'admin';
$current_user_autohaus_id = null;
$current_user_autohaus_name = null;
$can_switch_autohaus = false;

if ($current_user_id) {
    $stmt = $conn->prepare("
        SELECT u.role, u.autohaus_id, u.can_switch_autohaus, a.name as autohaus_name, u.username
        FROM admin_users u
        LEFT JOIN autohaus a ON u.autohaus_id = a.id
        WHERE u.id = ?
    ");
    $stmt->execute([$current_user_id]);
    $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($user_data) {
        $current_user_role = $user_data['role'] ?? 'admin';
        // Autohaus-ID kann NULL sein - das ist OK
        $current_user_autohaus_id = !empty($user_data['autohaus_id']) ? intval($user_data['autohaus_id']) : null;
        $current_user_autohaus_name = $user_data['autohaus_name'] ?? null;
        $current_user_name = $user_data['username'] ?? 'Administrator';
        $can_switch_autohaus = ($user_data['can_switch_autohaus'] == 1) || ($current_user_role === 'admin');
        
        // Debug-Logging
        error_log("hellomateo.php: User geladen - ID: {$current_user_id}, Role: {$current_user_role}, Autohaus ID: " . ($current_user_autohaus_id ?? 'NULL') . ", Can Switch: " . ($can_switch_autohaus ? 'true' : 'false'));
    }
}

// Get user theme preference
$current_theme = 'light';
if ($current_user_id) {
    $stmt = $conn->prepare("SELECT theme FROM admin_users WHERE id = ?");
    $stmt->execute([$current_user_id]);
    $theme_result = $stmt->fetch();
    $current_theme = $theme_result['theme'] ?? 'light';
}

// Aktiver Tab
$active_tab = $_GET['tab'] ?? 'messages';

// Autohaus-Auswahl für Admins (Session-basiert)
$selected_autohaus_id = null;
if ($can_switch_autohaus || $current_user_role === 'admin') {
    // Prüfe ob Autohaus in Session gespeichert ist
    if (isset($_GET['autohaus_id']) && !empty($_GET['autohaus_id'])) {
        $_SESSION['selected_autohaus_id'] = intval($_GET['autohaus_id']);
        $selected_autohaus_id = $_SESSION['selected_autohaus_id'];
    } elseif (isset($_SESSION['selected_autohaus_id'])) {
        $selected_autohaus_id = $_SESSION['selected_autohaus_id'];
    }
}

// Verwende ausgewähltes Autohaus (für Admins) oder Benutzer-Autohaus
$active_autohaus_id = $selected_autohaus_id ?? $current_user_autohaus_id;

// Lade alle Autohäuser für Auswahl (nur für Admins)
// WICHTIG: Nur Autohäuser anzeigen die WhatsApp aktiviert haben und einen API-Key haben
$all_autohauser = [];
if ($can_switch_autohaus || $current_user_role === 'admin') {
    // Prüfe ob whatsapp_notifications_enabled Spalte existiert
    $stmt = $conn->query("SHOW COLUMNS FROM autohaus LIKE 'whatsapp_notifications_enabled'");
    $hasWhatsAppColumn = $stmt->rowCount() > 0;
    
    if ($hasWhatsAppColumn) {
        // Prüfe ob Test-Status-Spalte existiert
        $stmt = $conn->query("SHOW COLUMNS FROM autohaus LIKE 'whatsapp_api_key_test_successful'");
        $hasTestColumn = $stmt->rowCount() > 0;
        
        if ($hasTestColumn) {
            // Nur Autohäuser mit aktiviertem WhatsApp, API-Key UND erfolgreichem Test
            $stmt = $conn->prepare("
                SELECT id, name, short_name, whatsapp_notifications_enabled, whatsapp_api_key, whatsapp_api_key_test_successful
                FROM autohaus 
                WHERE is_active = 1 
                AND whatsapp_notifications_enabled = 1 
                AND whatsapp_api_key IS NOT NULL 
                AND whatsapp_api_key != '' 
                AND TRIM(whatsapp_api_key) != ''
                AND whatsapp_api_key_test_successful = 1
                ORDER BY name ASC
            ");
        } else {
            // Nur Autohäuser mit aktiviertem WhatsApp und API-Key (Test-Status-Spalte existiert noch nicht)
            $stmt = $conn->prepare("
                SELECT id, name, short_name, whatsapp_notifications_enabled, whatsapp_api_key 
                FROM autohaus 
                WHERE is_active = 1 
                AND whatsapp_notifications_enabled = 1 
                AND whatsapp_api_key IS NOT NULL 
                AND whatsapp_api_key != '' 
                AND TRIM(whatsapp_api_key) != ''
                ORDER BY name ASC
            ");
        }
    } else {
        // Fallback: Nur aktive Autohäuser (wenn Spalte noch nicht existiert)
        $stmt = $conn->prepare("SELECT id, name, short_name FROM autohaus WHERE is_active = 1 ORDER BY name ASC");
    }
    $stmt->execute();
    $all_autohauser = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Für jeden Autohaus: Teste ob API-Key gültig ist (optional - kann langsam sein)
    // Wir zeigen alle mit API-Key an, auch wenn nicht getestet
    
    // Lade Name des aktiven Autohauses
    if ($active_autohaus_id) {
        $stmt = $conn->prepare("SELECT name FROM autohaus WHERE id = ?");
        $stmt->execute([$active_autohaus_id]);
        $active_autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
        $current_user_autohaus_name = $active_autohaus['name'] ?? $current_user_autohaus_name;
    }
}

// HelloMateo API initialisieren
// WICHTIG: Aktive Autohaus-ID übergeben, damit der richtige API-Key geladen wird
$hellomateo = new HelloMateoAPI(null, $active_autohaus_id);

// Debug: Prüfe ob API-Key geladen wurde
$apiKeyTest = $hellomateo->testConnection();
if (!$apiKeyTest['success'] && strpos($apiKeyTest['error'] ?? '', 'API-Key nicht gefunden') !== false) {
    error_log("hellomateo.php: API-Key nicht gefunden für Autohaus ID: " . ($active_autohaus_id ?? 'NULL'));
}

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajax_action'])) {
    // Output-Buffering aktivieren, um sicherzustellen, dass keine Ausgaben vor dem JSON-Header erfolgen
    ob_start();
    
    // Fehlerbehandlung: Alle Fehler in Logs schreiben, nicht ausgeben
    ini_set('display_errors', 0);
    ini_set('log_errors', 1);
    
    // JSON-Header setzen
    header('Content-Type: application/json');
    
    // Alle bisherigen Ausgaben löschen
    ob_clean();
    
    $action = $_POST['ajax_action'];
    $response = ['success' => false, 'error' => 'Unbekannte Aktion'];
    
    switch ($action) {
        case 'send_message':
            $to = $_POST['to'] ?? '';
            $message = $_POST['message'] ?? '';
            $templateId = $_POST['template_id'] ?? null;
            $channelId = $_POST['channel_id'] ?? null;
            $placeholderValuesJson = $_POST['placeholder_values'] ?? null;
            $placeholderValues = null;
            
            // Parse Platzhalter-Werte
            if ($placeholderValuesJson) {
                $placeholderValues = json_decode($placeholderValuesJson, true);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    error_log("hellomateo.php send_message: Fehler beim Parsen von placeholder_values: " . json_last_error_msg());
                    $placeholderValues = null;
                }
            }
            
            error_log("hellomateo.php send_message: to='{$to}', message length=" . strlen($message) . ", templateId=" . ($templateId ?? 'null') . ", channelId=" . ($channelId ?? 'null') . ", placeholderValues=" . ($placeholderValues ? json_encode($placeholderValues) : 'null'));
            
            // Validierung: Wenn Template verwendet wird, ist Nachricht optional
            // Wenn kein Template verwendet wird, ist Nachricht erforderlich
            if (empty($to)) {
                $response = ['success' => false, 'error' => 'Telefonnummer ist erforderlich'];
            } elseif (empty($templateId) && empty($message)) {
                $response = ['success' => false, 'error' => 'Nachricht oder Template ist erforderlich'];
            } else {
                // Wenn Template verwendet wird, Nachricht nicht mitsenden (wird in sendMessage() gehandhabt)
                $result = $hellomateo->sendMessage($to, $message, $templateId, $channelId, $placeholderValues);
                
                // Detailliertes Logging
                error_log("hellomateo.php send_message Result: " . json_encode($result, JSON_PRETTY_PRINT));
                
                // Prüfe die Antwort genauer
                if ($result['success']) {
                    // Prüfe ob die API wirklich eine Nachricht zurückgibt
                    if (isset($result['data']) && !empty($result['data'])) {
                        $response = $result;
                    } else {
                        // API gibt success zurück, aber keine Daten - das könnte ein Problem sein
                        error_log("hellomateo.php send_message: WARNUNG - API gibt success zurück, aber keine Daten!");
                        // Trotzdem als Erfolg behandeln, aber mit Warnung
                        $response = [
                            'success' => true,
                            'warning' => 'Die API hat die Anfrage akzeptiert, aber keine Bestätigungsdaten zurückgegeben. Bitte prüfen Sie, ob die Nachricht angekommen ist.',
                            'data' => $result['data'] ?? null,
                            'http_code' => $result['http_code'] ?? null
                        ];
                    }
                } else {
                    $response = $result;
                }
            }
            break;
            
        case 'send_broadcast':
            $to = $_POST['to'] ?? [];
            $message = $_POST['message'] ?? '';
            $templateId = $_POST['template_id'] ?? null;
            
            if (empty($to) || empty($message)) {
                $response = ['success' => false, 'error' => 'Empfänger und Nachricht sind erforderlich'];
            } else {
                $result = $hellomateo->sendBroadcast($to, $message, $templateId);
                $response = $result;
            }
            break;
            
        case 'load_contacts':
            $limit = intval($_POST['limit'] ?? 50);
            $offset = intval($_POST['offset'] ?? 0);
            $result = $hellomateo->listContacts($limit, $offset);
            $response = $result;
            break;
            
        case 'create_contact':
            $contactData = [
                'handle' => $_POST['handle'] ?? '',
                'full_name' => $_POST['full_name'] ?? '',
                'country_codes' => $_POST['country_codes'] ?? ['DE']
            ];
            $result = $hellomateo->createContact($contactData);
            $response = $result;
            break;
            
        case 'update_contact':
            $contactId = $_POST['contact_id'] ?? null;
            if (!$contactId) {
                $response = ['success' => false, 'error' => 'Kontakt-ID fehlt'];
            } else {
                $contactData = [
                    'full_name' => $_POST['full_name'] ?? '',
                    'handle' => $_POST['handle'] ?? '',
                    'country_codes' => $_POST['country_codes'] ?? ['DE']
                ];
                $result = $hellomateo->updateContact($contactId, $contactData);
                $response = $result;
            }
            break;
            
        case 'delete_contact':
            $contactId = $_POST['contact_id'] ?? null;
            if (!$contactId) {
                $response = ['success' => false, 'error' => 'Kontakt-ID fehlt'];
            } else {
                $result = $hellomateo->deleteContact($contactId);
                $response = $result;
            }
            break;
            
        case 'get_message_details':
            $messageId = $_POST['message_id'] ?? null;
            if (!$messageId) {
                $response = ['success' => false, 'error' => 'Nachricht-ID fehlt'];
            } else {
                // Lade alle Nachrichten und finde die spezifische
                $messagesResult = $hellomateo->listMessages(1000, 0);
                if ($messagesResult['success'] && is_array($messagesResult['data'])) {
                    $message = null;
                    foreach ($messagesResult['data'] as $msg) {
                        if (($msg['id'] ?? $msg['message_id'] ?? '') == $messageId) {
                            $message = $msg;
                            break;
                        }
                    }
                    if ($message) {
                        $response = ['success' => true, 'data' => $message];
                    } else {
                        $response = ['success' => false, 'error' => 'Nachricht nicht gefunden'];
                    }
                } else {
                    $response = ['success' => false, 'error' => 'Nachrichten konnten nicht geladen werden'];
                }
            }
            break;
            
        case 'get_contact_details':
            $contactId = $_POST['contact_id'] ?? null;
            if (!$contactId) {
                $response = ['success' => false, 'error' => 'Kontakt-ID fehlt'];
            } else {
                // Lade Kontakt direkt über ID
                $contactResult = $hellomateo->getContactById($contactId);
                if ($contactResult['success'] && !empty($contactResult['data'])) {
                    $contact = is_array($contactResult['data']) && isset($contactResult['data'][0]) 
                        ? $contactResult['data'][0] 
                        : $contactResult['data'];
                    if ($contact) {
                        // Lade verknüpfte Daten
                        $linkedData = [
                            'appointments' => [],
                            'storage' => [],
                            'messages' => [],
                            'conversations' => []
                        ];
                        
                        // Suche Appointments mit dieser Telefonnummer
                        if (!empty($contact['handle'])) {
                            $phone = preg_replace('/[^0-9+]/', '', $contact['handle']);
                            $stmt = $conn->prepare("SELECT id, customer_name, license_plate, slot_date, slot_time, status FROM appointments WHERE phone = ? OR phone LIKE ? ORDER BY slot_date DESC, slot_time DESC LIMIT 10");
                            $stmt->execute([$phone, '%' . $phone . '%']);
                            $linkedData['appointments'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
                            
                            // Suche Storage-Einträge (falls vorhanden)
                            // TODO: Storage-Verknüpfung implementieren wenn Storage-Tabelle Telefonnummern hat
                        }
                        
                        // Suche Nachrichten mit diesem Kontakt
                        $messagesResult = $hellomateo->listMessages(100, 0);
                        if ($messagesResult['success'] && is_array($messagesResult['data'])) {
                            $linkedData['messages'] = array_filter($messagesResult['data'], function($msg) use ($contact) {
                                $recipients = $msg['recipients'] ?? [];
                                foreach ($recipients as $r) {
                                    if (($r['handle'] ?? '') === ($contact['handle'] ?? '')) {
                                        return true;
                                    }
                                }
                                return false;
                            });
                        }
                        
                        $response = ['success' => true, 'data' => $contact, 'linked' => $linkedData];
                    } else {
                        $response = ['success' => false, 'error' => 'Kontakt nicht gefunden'];
                    }
                } else {
                    $response = ['success' => false, 'error' => 'Kontakte konnten nicht geladen werden'];
                }
            }
            break;
            
        case 'get_template_details':
            $templateId = $_POST['template_id'] ?? null;
            if (!$templateId) {
                $response = ['success' => false, 'error' => 'Template-ID fehlt'];
            } else {
                // Lade alle Templates und finde das spezifische
                $templatesResult = $hellomateo->listTemplates(1000, 0);
                if ($templatesResult['success'] && is_array($templatesResult['data'])) {
                    $template = null;
                    foreach ($templatesResult['data'] as $t) {
                        if (($t['id'] ?? '') == $templateId) {
                            $template = $t;
                            break;
                        }
                    }
                    if ($template) {
                        // Zähle Verwendung (Nachrichten mit diesem Template)
                        $messagesResult = $hellomateo->listMessages(1000, 0);
                        $usageCount = 0;
                        if ($messagesResult['success'] && is_array($messagesResult['data'])) {
                            foreach ($messagesResult['data'] as $msg) {
                                if (($msg['template_id'] ?? '') == $templateId) {
                                    $usageCount++;
                                }
                            }
                        }
                        $template['usage_count'] = $usageCount;
                        $response = ['success' => true, 'data' => $template];
                    } else {
                        $response = ['success' => false, 'error' => 'Template nicht gefunden'];
                    }
                } else {
                    $response = ['success' => false, 'error' => 'Templates konnten nicht geladen werden'];
                }
            }
            break;
            
        case 'get_conversation_details':
            $conversationId = $_POST['conversation_id'] ?? null;
            if (!$conversationId) {
                $response = ['success' => false, 'error' => 'Konversations-ID fehlt'];
            } else {
                // Versuche zuerst direkten API-Aufruf
                $conversationResult = $hellomateo->getConversationById($conversationId);
                
                if ($conversationResult['success'] && !empty($conversationResult['data'])) {
                    // API gibt möglicherweise Array zurück
                    $conversation = is_array($conversationResult['data']) && isset($conversationResult['data'][0]) 
                        ? $conversationResult['data'][0] 
                        : $conversationResult['data'];
                } else {
                    // Fallback: Rekursiv alle Konversationen durchsuchen
                    $conversation = null;
                    $offset = 0;
                    $limit = 100;
                    
                    do {
                        $conversationsResult = $hellomateo->listConversations($limit, $offset);
                        if ($conversationsResult['success'] && is_array($conversationsResult['data'])) {
                            $conversations = $conversationsResult['data'];
                            
                            // Suche in diesem Batch
                            foreach ($conversations as $c) {
                                if (($c['id'] ?? '') == $conversationId) {
                                    $conversation = $c;
                                    break 2; // Breche beide Schleifen
                                }
                            }
                            
                            // Wenn wir weniger als $limit erhalten haben, sind wir fertig
                            if (count($conversations) < $limit) {
                                break;
                            }
                            
                            $offset += $limit;
                        } else {
                            // Fehler beim Laden
                            break;
                        }
                    } while ($conversation === null);
                }
                
                if ($conversation) {
                    // Lade ALLE Nachrichten für diese Konversation rekursiv
                    $conversationMessages = [];
                    $messagesOffset = 0;
                    $messagesLimit = 100;
                    do {
                        $messagesResult = $hellomateo->listMessages($messagesLimit, $messagesOffset);
                        if ($messagesResult['success'] && is_array($messagesResult['data'])) {
                            $batchMessages = array_filter($messagesResult['data'], function($msg) use ($conversation) {
                                return ($msg['conversation_id'] ?? '') == ($conversation['id'] ?? '');
                            });
                            $conversationMessages = array_merge($conversationMessages, $batchMessages);
                            
                            // Wenn wir weniger als $messagesLimit erhalten haben, sind wir fertig
                            if (count($messagesResult['data']) < $messagesLimit) {
                                break;
                            }
                            $messagesOffset += $messagesLimit;
                        } else {
                            break;
                        }
                    } while (true);
                    
                    // Sortiere nach Datum (neueste zuerst)
                    usort($conversationMessages, function($a, $b) {
                        $dateA = $a['created_at'] ?? '';
                        $dateB = $b['created_at'] ?? '';
                        return strcmp($dateB, $dateA);
                    });
                    
                    $conversation['messages'] = array_values($conversationMessages);
                    $response = ['success' => true, 'data' => $conversation];
                } else {
                    $response = ['success' => false, 'error' => 'Konversation nicht gefunden'];
                }
            }
            break;
            
        case 'get_appointments_by_phone':
            $phone = $_POST['phone'] ?? '';
            if (empty($phone)) {
                $response = ['success' => false, 'error' => 'Telefonnummer fehlt'];
            } else {
                $phone = preg_replace('/[^0-9+]/', '', $phone);
                $stmt = $conn->prepare("SELECT id, customer_name, license_plate, slot_date, slot_time, status, event_id FROM appointments WHERE phone = ? OR phone LIKE ? ORDER BY slot_date DESC, slot_time DESC LIMIT 20");
                $stmt->execute([$phone, '%' . $phone . '%']);
                $appointments = $stmt->fetchAll(PDO::FETCH_ASSOC);
                $response = ['success' => true, 'data' => $appointments];
            }
            break;
            
        case 'get_storage_by_phone':
            $phone = $_POST['phone'] ?? '';
            if (empty($phone)) {
                $response = ['success' => false, 'error' => 'Telefonnummer fehlt'];
            } else {
                // Suche Storage-Einträge über Kennzeichen (falls vorhanden)
                // Da Storage keine direkte Telefonnummer hat, suchen wir über appointments
                $phone = preg_replace('/[^0-9+]/', '', $phone);
                $stmt = $conn->prepare("
                    SELECT DISTINCT a.storage_number, a.license_plate, a.customer_name, a.slot_date
                    FROM appointments a
                    WHERE a.phone = ? OR a.phone LIKE ?
                    AND a.storage_number IS NOT NULL
                    AND a.storage_number != ''
                    ORDER BY a.slot_date DESC
                    LIMIT 20
                ");
                $stmt->execute([$phone, '%' . $phone . '%']);
                $storage = $stmt->fetchAll(PDO::FETCH_ASSOC);
                $response = ['success' => true, 'data' => $storage];
            }
            break;
            
        case 'sync_contact_from_appointment':
            $appointmentId = $_POST['appointment_id'] ?? $_GET['appointment_id'] ?? null;
            if (!$appointmentId) {
                $response = ['success' => false, 'error' => 'Appointment-ID fehlt'];
            } else {
                $stmt = $conn->prepare("SELECT phone, customer_name FROM appointments WHERE id = ?");
                $stmt->execute([$appointmentId]);
                $appointment = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if (!$appointment || empty($appointment['phone'])) {
                    $response = ['success' => false, 'error' => 'Appointment nicht gefunden oder keine Telefonnummer'];
                } else {
                    // Prüfe ob Kontakt bereits existiert
                    $contactsResult = $hellomateo->listContacts(1000, 0);
                    $existingContact = null;
                    if ($contactsResult['success'] && is_array($contactsResult['data'])) {
                        $phone = preg_replace('/[^0-9+]/', '', $appointment['phone']);
                        foreach ($contactsResult['data'] as $contact) {
                            $contactPhone = preg_replace('/[^0-9+]/', '', $contact['handle'] || '');
                            if ($contactPhone === $phone) {
                                $existingContact = $contact;
                                break;
                            }
                        }
                    }
                    
                    if ($existingContact) {
                        $response = ['success' => true, 'data' => $existingContact, 'created' => false];
                    } else {
                        // Erstelle neuen Kontakt
                        $contactData = [
                            'handle' => $appointment['phone'],
                            'full_name' => $appointment['customer_name'] || 'Unbekannt',
                            'country_codes' => ['DE']
                        ];
                        $result = $hellomateo->createContact($contactData);
                        $response = $result;
                        if ($result['success']) {
                            $response['created'] = true;
                        }
                    }
                }
            }
            break;
            
        case 'send_message_from_appointment':
            $appointmentId = $_POST['appointment_id'] ?? null;
            $message = $_POST['message'] ?? '';
            $templateId = $_POST['template_id'] ?? null;
            $action = $_POST['action'] ?? 'confirmation'; // Standard: confirmation
            $autohausId = $_POST['autohaus_id'] ?? $active_autohaus_id ?? null;
            
            if (!$appointmentId) {
                $response = ['success' => false, 'error' => 'Appointment-ID fehlt'];
            } elseif (!$autohausId) {
                $response = ['success' => false, 'error' => 'Autohaus-ID fehlt'];
            } else {
                // Lade Appointment mit vollständigen Daten
                $stmt = $conn->prepare("
                    SELECT a.*, e.name as event_name, e.event_date, e.start_time, e.end_time,
                           ah.name as autohaus_name, ah.short_name as autohaus_short_name,
                           u.phone_number as user_phone_number, u.whatsapp_opt_out
                    FROM appointments a 
                    LEFT JOIN events e ON a.event_id = e.id 
                    LEFT JOIN autohaus ah ON a.autohaus_id = ah.id
                    LEFT JOIN users u ON a.user_id = u.id
                    WHERE a.id = ?
                ");
                $stmt->execute([$appointmentId]);
                $appointment = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if (!$appointment) {
                    $response = ['success' => false, 'error' => 'Appointment nicht gefunden'];
                } else {
                    // Verwende das neue Template-Konfigurationssystem
                    try {
                        $whatsappNotifications = new AutohausWhatsAppNotifications($autohausId);
                        
                        // Wenn Template-ID explizit übergeben wurde, verwende direkten Send
                        if ($templateId && !empty($message)) {
                            // Direkter Send mit expliziter Template-ID und Nachricht
                            $phoneNumber = $appointment['phone'] ?? $appointment['user_phone_number'] ?? null;
                            if (empty($phoneNumber)) {
                                $response = ['success' => false, 'error' => 'Keine gültige Telefonnummer für Termin vorhanden'];
                            } else {
                                // Hole Channel ID
                                $channelId = null;
                                $stmt = $conn->prepare("SELECT whatsapp_channel_id FROM autohaus WHERE id = ?");
                                $stmt->execute([$autohausId]);
                                $autohausData = $stmt->fetch(PDO::FETCH_ASSOC);
                                $channelId = $autohausData['whatsapp_channel_id'] ?? null;
                                
                                // Extrahiere Custom Fields aus Appointment
                                $customerName = $appointment['customer_name'] ?? '';
                                $nameParts = explode(' ', trim($customerName));
                                $firstName = !empty($nameParts) ? $nameParts[0] : $customerName;
                                $lastName = !empty($nameParts) ? end($nameParts) : $customerName;
                                
                                // Standard Custom Fields Mapping
                                $placeholderValues = [
                                    'custom_fields.last_name' => $lastName,
                                    'custom_fields.first_name' => $firstName
                                ];
                                
                                $result = $hellomateo->sendMessage($phoneNumber, $message, $templateId, $channelId, $placeholderValues);
                                $response = $result;
                            }
                        } else {
                            // Verwende das neue Template-Konfigurationssystem mit Aktion
                            // Mappe Aktion zu action_type
                            $actionTypeMap = [
                                'reminder' => 'reminder_24h',
                                'confirmation' => 'confirmation',
                                'approved' => 'approved',
                                'rejected' => 'rejected',
                                'cancellation' => 'cancellation',
                                'rescheduled' => 'rescheduled'
                            ];
                            
                            $actionType = $actionTypeMap[$action] ?? $action;
                            $result = $whatsappNotifications->sendAppointmentNotification($appointmentId, $actionType);
                            $response = $result;
                        }
                    } catch (Exception $e) {
                        error_log("hellomateo.php send_message_from_appointment Fehler: " . $e->getMessage());
                        $response = ['success' => false, 'error' => 'Fehler beim Senden: ' . $e->getMessage()];
                    }
                }
            }
            break;
            
        case 'resend_message':
            $messageId = $_POST['message_id'] ?? null;
            if (!$messageId) {
                $response = ['success' => false, 'error' => 'Nachricht-ID fehlt'];
            } else {
                // Lade Nachricht
                $messagesResult = $hellomateo->listMessages(1000, 0);
                if ($messagesResult['success'] && is_array($messagesResult['data'])) {
                    $message = null;
                    foreach ($messagesResult['data'] as $msg) {
                        if (($msg['id'] ?? $msg['message_id'] ?? '') == $messageId) {
                            $message = $msg;
                            break;
                        }
                    }
                    if ($message) {
                        // Extrahiere Empfänger und Nachricht
                        $to = $message['recipients'][0]['handle'] ?? $message['to'] ?? '';
                        $text = $message['text'] ?? $message['blurb'] ?? '';
                        $templateId = $message['template_id'] ?? null;
                        
                        if (empty($to) || empty($text)) {
                            $response = ['success' => false, 'error' => 'Nachrichtendaten unvollständig'];
                        } else {
                            $result = $hellomateo->sendMessage($to, $text, $templateId);
                            $response = $result;
                        }
                    } else {
                        $response = ['success' => false, 'error' => 'Nachricht nicht gefunden'];
                    }
                } else {
                    $response = ['success' => false, 'error' => 'Nachrichten konnten nicht geladen werden'];
                }
            }
            break;
            
        case 'load_templates':
            $limit = intval($_POST['limit'] ?? 50);
            $offset = intval($_POST['offset'] ?? 0);
            $result = $hellomateo->listTemplates($limit, $offset);
            $response = $result;
            break;
            
        case 'load_conversations':
            $limit = intval($_POST['limit'] ?? 50);
            $offset = intval($_POST['offset'] ?? 0);
            $result = $hellomateo->listConversations($limit, $offset);
            $response = $result;
            break;
            
        case 'load_messages':
            $limit = intval($_POST['limit'] ?? 50);
            $offset = intval($_POST['offset'] ?? 0);
            $result = $hellomateo->listMessages($limit, $offset);
            $response = $result;
            break;
            
        case 'load_channels':
            $result = $hellomateo->listChannels();
            $response = $result;
            break;
            
        case 'test_api_key':
            // Teste API-Key direkt aus der Datenbank
            $testAutohausId = $_POST['autohaus_id'] ?? $active_autohaus_id;
            if ($testAutohausId) {
                $stmt = $conn->prepare("SELECT whatsapp_api_key FROM autohaus WHERE id = ?");
                $stmt->execute([$testAutohausId]);
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($result && !empty($result['whatsapp_api_key'])) {
                    $rawApiKey = $result['whatsapp_api_key'];
                    $apiKey = trim($rawApiKey);
                    $apiKey = preg_replace('/\s+/', '', $apiKey);
                    $apiKey = preg_replace('/[\x00-\x1F\x7F]/', '', $apiKey);
                    
                    // Teste mit test_hellomateo_api.php
                    $testData = json_encode(['api_key' => $apiKey, 'autohaus_id' => $testAutohausId]);
                    $ch = curl_init();
                    curl_setopt($ch, CURLOPT_URL, 'http://' . $_SERVER['HTTP_HOST'] . dirname($_SERVER['PHP_SELF']) . '/test_hellomateo_api.php');
                    curl_setopt($ch, CURLOPT_POST, true);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, $testData);
                    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($ch, CURLOPT_COOKIE, session_name() . '=' . session_id());
                    $testResponse = curl_exec($ch);
                    curl_close($ch);
                    
                    $testResult = json_decode($testResponse, true);
                    
                    // Füge Debug-Informationen hinzu
                    $testResult['debug'] = [
                        'raw_key_length' => strlen($rawApiKey),
                        'cleaned_key_length' => strlen($apiKey),
                        'raw_key_preview' => substr($rawApiKey, 0, 50) . '...' . substr($rawApiKey, -20),
                        'cleaned_key_preview' => substr($apiKey, 0, 50) . '...' . substr($apiKey, -20),
                        'has_whitespace' => preg_match('/\s/', $rawApiKey),
                        'has_control_chars' => preg_match('/[\x00-\x1F\x7F]/', $rawApiKey),
                        'is_jwt_format' => (count(explode('.', $apiKey)) === 3),
                        'jwt_parts_count' => count(explode('.', $apiKey))
                    ];
                    
                    $response = $testResult ?: ['success' => false, 'error' => 'Test konnte nicht durchgeführt werden'];
                } else {
                    $response = ['success' => false, 'error' => 'Kein API-Key in Datenbank gefunden'];
                }
            } else {
                $response = ['success' => false, 'error' => 'Keine Autohaus-ID übergeben'];
            }
            break;
            
        case 'get_error_logs':
            // Lade die letzten Error-Log-Einträge
            $logEntries = [];
            $maxLines = intval($_POST['max_lines'] ?? 100);
            
            // Mögliche Log-Pfade
            $possibleLogPaths = [
                ini_get('error_log'), // PHP error_log Einstellung
                'C:\\xampp\\php\\logs\\php_error_log',
                'C:\\xampp\\apache\\logs\\error.log',
                __DIR__ . '/../logs/php_error.log',
                __DIR__ . '/../error.log',
                sys_get_temp_dir() . '/php_errors.log'
            ];
            
            $logPath = null;
            foreach ($possibleLogPaths as $path) {
                if ($path && file_exists($path) && is_readable($path)) {
                    $logPath = $path;
                    break;
                }
            }
            
            if ($logPath) {
                // Lade die letzten Zeilen der Log-Datei
                $lines = file($logPath);
                if ($lines) {
                    $logEntries = array_slice($lines, -$maxLines);
                    // Filtere nur HelloMateo-relevante Einträge
                    $filteredEntries = array_filter($logEntries, function($line) {
                        return stripos($line, 'HelloMateo') !== false || 
                               stripos($line, 'hellomateo') !== false ||
                               stripos($line, 'sendMessage') !== false ||
                               stripos($line, 'send_message') !== false;
                    });
                    
                    if (!empty($filteredEntries)) {
                        $logEntries = array_values($filteredEntries);
                    } else {
                        // Wenn keine gefilterten Einträge, zeige die letzten allgemeinen Einträge
                        $logEntries = array_slice($lines, -50);
                    }
                }
            }
            
            $response = [
                'success' => true,
                'log_path' => $logPath,
                'log_entries' => $logEntries,
                'entries_count' => count($logEntries),
                'possible_paths' => $possibleLogPaths
            ];
            break;
            
        case 'get_debug_info':
            // Lade Debug-Informationen
            $debugInfo = [];
            
            if ($active_autohaus_id) {
                $stmt = $conn->prepare("SELECT whatsapp_api_key, whatsapp_notifications_enabled FROM autohaus WHERE id = ?");
                $stmt->execute([$active_autohaus_id]);
                $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($autohaus) {
                    $rawKey = $autohaus['whatsapp_api_key'] ?? '';
                    $cleanedKey = $rawKey;
                    $cleanedKey = trim($cleanedKey);
                    $cleanedKey = preg_replace('/\s+/', '', $cleanedKey);
                    $cleanedKey = preg_replace('/[\x00-\x1F\x7F]/', '', $cleanedKey);
                    
                    $jwtParts = explode('.', $cleanedKey);
                    $jwtPartsCount = count($jwtParts);
                    
                    // Prüfe Datenbank-Spalten-Größe
                    $columnInfo = null;
                    $needsMigration = false;
                    try {
                        $stmt = $conn->query("SHOW COLUMNS FROM autohaus WHERE Field = 'whatsapp_api_key'");
                        $columnInfo = $stmt->fetch(PDO::FETCH_ASSOC);
                        if ($columnInfo && (stripos($columnInfo['Type'], 'varchar(255)') !== false)) {
                            $needsMigration = true;
                            // Versuche automatisch zu migrieren
                            try {
                                $conn->exec("ALTER TABLE autohaus MODIFY COLUMN whatsapp_api_key TEXT DEFAULT NULL");
                                error_log("hellomateo.php: whatsapp_api_key Spalte automatisch von VARCHAR(255) auf TEXT erweitert");
                                // Lade Spalten-Info neu
                                $stmt = $conn->query("SHOW COLUMNS FROM autohaus WHERE Field = 'whatsapp_api_key'");
                                $columnInfo = $stmt->fetch(PDO::FETCH_ASSOC);
                            } catch (Exception $e) {
                                error_log("hellomateo.php: Fehler beim automatischen Erweitern der Spalte: " . $e->getMessage());
                            }
                        }
                    } catch (Exception $e) {
                        error_log("Fehler beim Abrufen der Spalten-Info: " . $e->getMessage());
                    }
                    
                    $debugInfo = [
                        'autohaus_id' => $active_autohaus_id,
                        'whatsapp_enabled' => $autohaus['whatsapp_notifications_enabled'] ?? 0,
                        'raw_key_length' => strlen($rawKey),
                        'cleaned_key_length' => strlen($cleanedKey),
                        'raw_key_preview' => substr($rawKey, 0, 50) . '...' . substr($rawKey, -20),
                        'cleaned_key_preview' => substr($cleanedKey, 0, 50) . '...' . substr($cleanedKey, -20),
                        'raw_key_full' => $rawKey, // VOLLSTÄNDIGER KEY für Debugging
                        'has_whitespace' => preg_match('/\s/', $rawKey),
                        'has_control_chars' => preg_match('/[\x00-\x1F\x7F]/', $rawKey),
                        'is_jwt_format' => ($jwtPartsCount === 3),
                        'jwt_parts_count' => $jwtPartsCount,
                        'jwt_part_lengths' => array_map('strlen', $jwtParts),
                        'column_type' => $columnInfo['Type'] ?? 'N/A',
                        'column_max_length' => $columnInfo ? (int)preg_replace('/[^0-9]/', '', $columnInfo['Type']) : null,
                        'is_truncated' => ($jwtPartsCount < 3),
                        'needs_migration' => $needsMigration,
                        'migration_applied' => ($needsMigration && $columnInfo && stripos($columnInfo['Type'], 'text') !== false),
                        'api_key_info' => $hellomateo->getApiKeyInfo()
                    ];
                    
                    // Teste API-Key direkt
                    $testApi = new HelloMateoAPI($cleanedKey, $active_autohaus_id);
                    $testResult = $testApi->listChannels();
                    
                    // Erstelle Response-Preview (inkl. Fehler)
                    $responsePreview = null;
                    if (isset($testResult['data'])) {
                        $responsePreview = substr(json_encode($testResult['data'], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), 0, 1000);
                    } elseif (isset($testResult['error'])) {
                        $responsePreview = $testResult['error'];
                        if (isset($testResult['data'])) {
                            $responsePreview .= "\n\n" . substr(json_encode($testResult['data'], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), 0, 500);
                        }
                    }
                    
                    $debugInfo['api_test'] = [
                        'success' => $testResult['success'] ?? false,
                        'error' => $testResult['error'] ?? null,
                        'http_code' => $testResult['http_code'] ?? null,
                        'response_preview' => $responsePreview,
                        'full_response' => json_encode($testResult, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)
                    ];
                }
            }
            
            $response = ['success' => true, 'debug' => $debugInfo];
            break;
    }
    
    // Alle Ausgaben löschen und nur JSON ausgeben
    ob_clean();
    echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    ob_end_flush();
    exit;
}

// Lade initiale Daten
$channels = [];
$channelsResult = $hellomateo->listChannels();
if ($channelsResult['success'] && isset($channelsResult['data'])) {
    $channels = is_array($channelsResult['data']) ? $channelsResult['data'] : [];
}

$current_page = 'hellomateo';
$page_title = 'WhatsApp Management';

?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>WhatsApp Management - Boxenstop Admin</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/volvo-theme.css">
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="<?php echo asset_url('assets/css/unified-sidebar.css'); ?>">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&family=SF+Pro+Display:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body class="admin-body" data-user-id="<?php echo htmlspecialchars($current_user_id); ?>">
    <div class="admin-container">
        <!-- Include Unified Sidebar -->
        <?php include 'includes/sidebar.php'; ?>

        <!-- Main Content -->
        <main class="admin-main">

<style>
.hellomateo-container {
    padding: 2rem;
    max-width: 1400px;
    margin: 0 auto;
}

.hellomateo-tabs {
    display: flex;
    gap: 0.5rem;
    border-bottom: 2px solid #dee2e6;
    margin-bottom: 2rem;
    flex-wrap: wrap;
}

.hellomateo-tab {
    padding: 1rem 1.5rem;
    background: transparent;
    border: none;
    border-bottom: 3px solid transparent;
    cursor: pointer;
    font-size: 1rem;
    font-weight: 500;
    color: #6c757d;
    transition: all 0.3s;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.hellomateo-tab:hover {
    color: #0056b3;
    background: #f8f9fa;
}

.hellomateo-tab.active {
    color: #0056b3;
    border-bottom-color: #0056b3;
    background: #f8f9fa;
}

.hellomateo-tab-content {
    display: none;
}

.hellomateo-tab-content.active {
    display: block;
}

.content-card {
    background: white;
    border-radius: 8px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    padding: 1.5rem;
    margin-bottom: 1.5rem;
}

.card-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1.5rem;
    padding-bottom: 1rem;
    border-bottom: 1px solid #dee2e6;
}

.card-header h3 {
    margin: 0;
    font-size: 1.25rem;
    color: #212529;
}

.form-group {
    margin-bottom: 1.5rem;
}

.form-group label {
    display: block;
    margin-bottom: 0.5rem;
    font-weight: 600;
    color: #495057;
}

.form-group input,
.form-group textarea,
.form-group select {
    width: 100%;
    padding: 0.75rem;
    border: 2px solid #dee2e6;
    border-radius: 6px;
    font-size: 0.95rem;
    transition: border-color 0.3s;
}

.form-group input:focus,
.form-group textarea:focus,
.form-group select:focus {
    outline: none;
    border-color: #0056b3;
}

.btn {
    padding: 0.75rem 1.5rem;
    border: none;
    border-radius: 6px;
    font-size: 0.95rem;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
}

.btn-primary {
    background: #0056b3;
    color: white;
}

.btn-primary:hover {
    background: #004085;
}

.btn-success {
    background: #28a745;
    color: white;
}

.btn-success:hover {
    background: #218838;
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-secondary:hover {
    background: #5a6268;
}

.alert {
    padding: 1rem;
    border-radius: 6px;
    margin-bottom: 1rem;
}

.alert-success {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.alert-error {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.badge {
    display: inline-block;
    padding: 0.25rem 0.5rem;
    font-size: 0.75rem;
    font-weight: 600;
    border-radius: 4px;
    text-transform: uppercase;
}

.badge-success {
    background: #d4edda;
    color: #155724;
}

.badge-error {
    background: #f8d7da;
    color: #721c24;
}

.badge-warning {
    background: #fff3cd;
    color: #856404;
}

.badge-info {
    background: #d1ecf1;
    color: #0c5460;
}

/* SCHLICHTES, MINIMALISTISCHES DESIGN */
:root {
    --whatsapp-green: #25D366;
    --primary-color: #0056b3;
    --border-color: #dee2e6;
    --bg-light: #f8f9fa;
}

/* Schlichtes Nachrichten-Formular */
.modern-message-form {
    background: white;
    border-radius: 8px;
    border: 1px solid var(--border-color);
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
    padding: 2rem;
    margin-bottom: 2rem;
}

.form-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 2rem;
    padding-bottom: 1.5rem;
    border-bottom: 2px solid rgba(0, 0, 0, 0.05);
}

.header-content h2 {
    font-size: 1.5rem;
    font-weight: 600;
    color: #212529;
    margin: 0 0 0.5rem 0;
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.header-subtitle {
    color: #6b7280;
    font-size: 0.95rem;
    margin: 0;
}

.header-actions {
    display: flex;
    gap: 0.5rem;
}

.btn-icon {
    width: 36px;
    height: 36px;
    border-radius: 6px;
    border: 1px solid var(--border-color);
    background: white;
    color: #6c757d;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: all 0.2s;
}

.btn-icon:hover {
    background: var(--bg-light);
    border-color: var(--primary-color);
    color: var(--primary-color);
}

/* Formular-Schritte */
.modern-form {
    display: flex;
    flex-direction: column;
    gap: 2rem;
}

.form-step {
    transition: opacity 0.2s;
}

/* Alle Schritte sind immer aktiv - kein Ausgrauen */
.form-step {
    opacity: 1;
    pointer-events: all;
}

.step-header {
    display: flex;
    align-items: flex-start;
    gap: 1rem;
    margin-bottom: 1.5rem;
}

.step-number {
    width: 32px;
    height: 32px;
    border-radius: 6px;
    background: var(--primary-color);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 0.9rem;
    font-weight: 600;
    flex-shrink: 0;
}

.step-title h3 {
    margin: 0 0 0.25rem 0;
    font-size: 1.1rem;
    font-weight: 600;
    color: #212529;
}

.step-title p {
    margin: 0;
    color: #6c757d;
    font-size: 0.85rem;
}

.form-section {
    padding-left: 48px;
}

/* Moderne Eingabefelder */
.modern-input {
    margin-bottom: 1.5rem;
}

.modern-input label {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-weight: 600;
    color: #374151;
    margin-bottom: 0.75rem;
    font-size: 0.95rem;
}

.required {
    color: #ef4444;
}

.input-wrapper {
    display: flex;
    gap: 0.5rem;
    align-items: stretch;
}

.country-selector {
    position: relative;
}

.country-code-select {
    padding: 0.75rem;
    border: 1px solid var(--border-color);
    border-radius: 6px 0 0 6px;
    background: white;
    font-weight: 500;
    cursor: pointer;
    transition: border-color 0.2s;
    min-width: 100px;
    font-size: 0.9rem;
}

.country-code-select:focus {
    outline: none;
    border-color: var(--primary-color);
}

.phone-input {
    flex: 1;
    padding: 0.75rem;
    border: 1px solid var(--border-color);
    border-left: none;
    border-radius: 0 6px 6px 0;
    font-size: 0.95rem;
    transition: border-color 0.2s;
}

.phone-input:focus {
    outline: none;
    border-color: var(--primary-color);
}

.input-action {
    width: 40px;
    border: 1px solid var(--border-color);
    border-left: none;
    border-radius: 0 6px 6px 0;
    background: var(--bg-light);
    color: #6c757d;
    cursor: pointer;
    transition: all 0.2s;
    display: flex;
    align-items: center;
    justify-content: center;
}

.input-action:hover {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
}

.input-feedback {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    margin-top: 0.5rem;
    font-size: 0.85rem;
    color: #6b7280;
}

.phone-preview {
    margin-top: 0.75rem;
    padding: 0.75rem 1rem;
    background: #d1fae5;
    border-radius: 8px;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #065f46;
    font-size: 0.9rem;
}

/* Bulk-Send Toggle */
.bulk-toggle-wrapper {
    margin: 1.5rem 0;
}

.modern-toggle {
    display: flex;
    align-items: center;
    gap: 1rem;
    cursor: pointer;
    padding: 0.75rem 1rem;
    background: var(--bg-light);
    border-radius: 6px;
    border: 1px solid var(--border-color);
    transition: all 0.2s;
}

.modern-toggle:hover {
    border-color: var(--primary-color);
    background: white;
}

.modern-toggle input {
    display: none;
}

.toggle-slider {
    width: 44px;
    height: 24px;
    background: #d1d5db;
    border-radius: 12px;
    position: relative;
    transition: background 0.2s;
}

.toggle-slider::before {
    content: '';
    position: absolute;
    width: 20px;
    height: 20px;
    background: white;
    border-radius: 50%;
    top: 2px;
    left: 2px;
    transition: transform 0.2s;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.2);
}

.modern-toggle input:checked + .toggle-slider {
    background: var(--primary-color);
}

.modern-toggle input:checked + .toggle-slider::before {
    transform: translateX(20px);
}

.toggle-label {
    font-weight: 500;
    color: #374151;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.bulk-container {
    margin-top: 1rem;
    animation: slideDown 0.3s ease-out;
}

@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.bulk-textarea {
    width: 100%;
    padding: 1rem;
    border: 2px solid #e5e7eb;
    border-radius: 12px;
    font-family: 'Monaco', 'Courier New', monospace;
    font-size: 0.9rem;
    resize: vertical;
    transition: all 0.3s;
}

.bulk-textarea:focus {
    outline: none;
    border-color: #6366f1;
    box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.1);
}

/* Kanal-Auswahl */
.channel-selector {
    margin-top: 1rem;
}

.channel-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
    gap: 1rem;
}

.channel-card {
    position: relative;
    cursor: pointer;
}

.channel-card input[type="radio"] {
    position: absolute;
    opacity: 0;
    pointer-events: none;
}

.channel-card-content {
    padding: 1rem;
    background: white;
    border: 1px solid var(--border-color);
    border-radius: 8px;
    display: flex;
    align-items: center;
    gap: 1rem;
    transition: all 0.2s;
}

.channel-card:hover .channel-card-content {
    border-color: var(--primary-color);
    background: var(--bg-light);
}

.channel-card input[type="radio"]:checked + .channel-card-content {
    border-color: var(--primary-color);
    background: rgba(0, 86, 179, 0.05);
    box-shadow: 0 0 0 2px rgba(0, 86, 179, 0.1);
}

.channel-icon {
    width: 40px;
    height: 40px;
    border-radius: 6px;
    background: var(--bg-light);
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.25rem;
    color: #6c757d;
    transition: all 0.2s;
    flex-shrink: 0;
}

.channel-icon.active {
    background: var(--whatsapp-green);
    color: white;
}

.channel-card input[type="radio"]:checked + .channel-card-content .channel-icon {
    background: var(--whatsapp-green);
    color: white;
}

.channel-info {
    flex: 1;
}

.channel-name {
    font-weight: 600;
    color: #1f2937;
    margin-bottom: 0.25rem;
}

.channel-details {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    font-size: 0.85rem;
}

.channel-handle {
    color: #6b7280;
}

.channel-status {
    padding: 0.25rem 0.5rem;
    border-radius: 6px;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
}

.channel-status.success {
    background: #d1fae5;
    color: #065f46;
}

.channel-status.warning {
    background: #fef3c7;
    color: #92400e;
}

.channel-check {
    width: 24px;
    height: 24px;
    border-radius: 50%;
    background: var(--primary-color);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 0.75rem;
    opacity: 0;
    transform: scale(0);
    transition: all 0.2s;
}

.channel-card input[type="radio"]:checked + .channel-card-content .channel-check {
    opacity: 1;
    transform: scale(1);
}

/* Nachricht-Typ-Auswahl */
.message-type-selector {
    margin-bottom: 1.5rem;
}

.type-tabs {
    display: flex;
    gap: 0.5rem;
    background: var(--bg-light);
    padding: 0.25rem;
    border-radius: 6px;
    border: 1px solid var(--border-color);
}

.type-tab {
    flex: 1;
    padding: 0.75rem 1rem;
    border: none;
    background: transparent;
    border-radius: 4px;
    font-weight: 500;
    color: #6c757d;
    cursor: pointer;
    transition: all 0.2s;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
    font-size: 0.9rem;
}

.type-tab:hover {
    color: var(--primary-color);
}

.type-tab.active {
    background: white;
    color: var(--primary-color);
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
}

.message-section {
    display: none;
    animation: fadeIn 0.3s ease-out;
}

.message-section.active {
    display: block;
}

@keyframes fadeIn {
    from {
        opacity: 0;
        transform: translateY(10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

/* Template-Auswahl */
.template-selector-wrapper {
    margin-bottom: 1.5rem;
}

.template-search-wrapper {
    position: relative;
    margin-bottom: 1rem;
}

.template-search-wrapper .fa-search {
    position: absolute;
    left: 1rem;
    top: 50%;
    transform: translateY(-50%);
    color: #9ca3af;
}

.template-search {
    width: 100%;
    padding: 0.75rem 1rem 0.75rem 2.5rem;
    border: 1px solid var(--border-color);
    border-radius: 6px;
    font-size: 0.9rem;
    transition: border-color 0.2s;
}

.template-search:focus {
    outline: none;
    border-color: var(--primary-color);
}

.modern-select {
    width: 100%;
    padding: 0.75rem 1rem;
    border: 1px solid var(--border-color);
    border-radius: 6px;
    background: white;
    font-size: 0.9rem;
    cursor: pointer;
    transition: border-color 0.2s;
    appearance: none;
    background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='12' height='12' viewBox='0 0 12 12'%3E%3Cpath fill='%236c757d' d='M6 9L1 4h10z'/%3E%3C/svg%3E");
    background-repeat: no-repeat;
    background-position: right 0.75rem center;
    padding-right: 2.25rem;
}

.modern-select:focus {
    outline: none;
    border-color: var(--primary-color);
}

/* Template-Vorschau */
.template-preview-card {
    margin-top: 1.5rem;
    padding: 1.25rem;
    background: white;
    border: 1px solid var(--border-color);
    border-radius: 8px;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.05);
}

.preview-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1rem;
    padding-bottom: 1rem;
    border-bottom: 1px solid #e5e7eb;
}

.preview-header h4 {
    margin: 0;
    font-size: 1.1rem;
    font-weight: 600;
    color: #1f2937;
}

.template-meta {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    margin-top: 0.5rem;
}

.template-text-preview {
    white-space: pre-wrap;
    color: #374151;
    line-height: 1.6;
    font-size: 0.95rem;
}

/* Platzhalter-Eingabe */
.placeholders-container {
    margin-top: 1.5rem;
    padding-top: 1.5rem;
    border-top: 2px solid #e5e7eb;
}

.placeholders-header {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    margin-bottom: 1rem;
    font-weight: 600;
    color: #374151;
}

.placeholders-fields {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.placeholder-field {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.placeholder-field label {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-weight: 500;
    color: #374151;
    font-size: 0.9rem;
}

.placeholder-field label code {
    background: #f3f4f6;
    padding: 0.25rem 0.5rem;
    border-radius: 6px;
    font-family: 'Monaco', 'Courier New', monospace;
    font-size: 0.85rem;
    color: #6366f1;
}

.placeholder-input {
    width: 100%;
    padding: 0.75rem;
    border: 1px solid var(--border-color);
    border-radius: 6px;
    font-size: 0.9rem;
    transition: border-color 0.2s;
}

.placeholder-input:focus {
    outline: none;
    border-color: var(--primary-color);
}

.placeholder-input.valid {
    border-color: #28a745;
}

.placeholder-input.invalid {
    border-color: #dc3545;
}

/* Live-Vorschau */
.live-preview-card {
    margin-top: 1.5rem;
    padding: 1.25rem;
    background: #e3f2fd;
    border: 1px solid #90caf9;
    border-radius: 8px;
    color: #1565c0;
}

.live-preview-card .preview-header {
    border-bottom-color: #90caf9;
    color: #1565c0;
}

.live-preview-content {
    white-space: pre-wrap;
    line-height: 1.6;
    font-size: 0.95rem;
    color: #1565c0;
    padding: 1rem;
    background: white;
    border-radius: 6px;
    margin-top: 0.75rem;
}

/* Text-Eingabe */
.text-input-wrapper {
    margin-bottom: 1.5rem;
}

.char-count {
    float: right;
    font-size: 0.85rem;
    color: #9ca3af;
    font-weight: normal;
}

.modern-textarea {
    width: 100%;
    padding: 0.75rem;
    border: 1px solid var(--border-color);
    border-radius: 6px;
    font-size: 0.9rem;
    font-family: inherit;
    resize: vertical;
    transition: border-color 0.2s;
    line-height: 1.6;
}

.modern-textarea:focus {
    outline: none;
    border-color: var(--primary-color);
}

.modern-input {
    width: 100%;
    padding: 0.75rem;
    border: 1px solid var(--border-color);
    border-radius: 6px;
    font-size: 0.9rem;
    transition: border-color 0.2s;
}

.modern-input:focus {
    outline: none;
    border-color: var(--primary-color);
}

.message-preview-card {
    margin-top: 1rem;
    padding: 1.25rem;
    background: #f9fafb;
    border: 2px solid #e5e7eb;
    border-radius: 12px;
}

/* Formular-Aktionen */
.form-actions {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-top: 2rem;
    padding-top: 2rem;
    border-top: 2px solid rgba(0, 0, 0, 0.05);
}

.btn-send {
    position: relative;
    min-width: 160px;
    padding: 0.75rem 1.5rem;
    font-size: 0.95rem;
    font-weight: 600;
    border-radius: 6px;
    background: var(--primary-color);
    color: white;
    border: none;
    cursor: pointer;
    transition: background 0.2s;
}

.btn-send:hover {
    background: #004085;
}

.btn-send:active {
    background: #003366;
}

.btn-loader {
    position: absolute;
    left: 50%;
    top: 50%;
    transform: translate(-50%, -50%);
}

.btn-send.loading span {
    opacity: 0;
}

.btn-send.loading .btn-loader {
    display: block;
}

.result-container {
    margin-top: 1.5rem;
    animation: slideDown 0.3s ease-out;
}

/* Responsive Design */
@media (max-width: 768px) {
    .modern-message-form {
        padding: 1.5rem;
    }
    
    .form-section {
        padding-left: 0;
    }
    
    .step-header {
        flex-direction: column;
        gap: 0.75rem;
    }
    
    .channel-grid {
        grid-template-columns: 1fr;
    }
    
    .form-actions {
        flex-direction: column;
        gap: 1rem;
    }
    
    .btn-send {
        width: 100%;
    }
}

/* Erweiterte Badge-Styles */
.badge {
    display: inline-flex;
    align-items: center;
    gap: 0.25rem;
    padding: 0.35rem 0.75rem;
    font-size: 0.8rem;
    font-weight: 600;
    border-radius: 12px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    transition: all 0.2s ease;
    cursor: default;
}

.badge:hover {
    transform: translateY(-1px);
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.badge i {
    font-size: 0.7rem;
}

/* Nachrichten-Badges */
.badge-message-sent {
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
    color: white;
}

.badge-message-delivered {
    background: linear-gradient(135deg, #007bff 0%, #0056b3 100%);
    color: white;
}

.badge-message-read {
    background: linear-gradient(135deg, #0056b3 0%, #004085 100%);
    color: white;
}

.badge-message-failed {
    background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
    color: white;
}

.badge-message-pending {
    background: linear-gradient(135deg, #ffc107 0%, #ff9800 100%);
    color: #212529;
}

/* Template-Badges */
.badge-template-approved {
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
    color: white;
}

.badge-template-pending {
    background: linear-gradient(135deg, #ffc107 0%, #ff9800 100%);
    color: #212529;
}

.badge-template-rejected {
    background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
    color: white;
}

.badge-template-draft {
    background: linear-gradient(135deg, #6c757d 0%, #5a6268 100%);
    color: white;
}

/* Konversations-Badges */
.badge-conversation-active {
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
    color: white;
}

.badge-conversation-open {
    background: linear-gradient(135deg, #007bff 0%, #0056b3 100%);
    color: white;
}

.badge-conversation-closed {
    background: linear-gradient(135deg, #6c757d 0%, #5a6268 100%);
    color: white;
}

.badge-conversation-pending {
    background: linear-gradient(135deg, #ffc107 0%, #ff9800 100%);
    color: #212529;
}

/* Appointments-Badges */
.badge-appointment-booked {
    background: linear-gradient(135deg, #007bff 0%, #0056b3 100%);
    color: white;
}

.badge-appointment-pending {
    background: linear-gradient(135deg, #ffc107 0%, #ff9800 100%);
    color: #212529;
}

.badge-appointment-completed {
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
    color: white;
}

.badge-appointment-cancelled {
    background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
    color: white;
}

/* Modal Styles */
.modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0,0,0,0.6);
    display: flex;
    justify-content: center;
    align-items: center;
    z-index: 10000;
    animation: fadeIn 0.2s ease;
}

@keyframes fadeIn {
    from { opacity: 0; }
    to { opacity: 1; }
}

.modal-content {
    background: white;
    border-radius: 12px;
    max-width: 800px;
    max-height: 90vh;
    overflow-y: auto;
    box-shadow: 0 10px 40px rgba(0,0,0,0.2);
    animation: slideUp 0.3s ease;
    position: relative;
}

@keyframes slideUp {
    from {
        transform: translateY(20px);
        opacity: 0;
    }
    to {
        transform: translateY(0);
        opacity: 1;
    }
}

.modal-header {
    padding: 1.5rem;
    border-bottom: 2px solid #dee2e6;
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
    border-radius: 12px 12px 0 0;
}

.modal-header h3 {
    margin: 0;
    font-size: 1.5rem;
    color: #212529;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.modal-close {
    background: none;
    border: none;
    font-size: 1.5rem;
    color: #6c757d;
    cursor: pointer;
    padding: 0.5rem;
    border-radius: 6px;
    transition: all 0.2s;
}

.modal-close:hover {
    background: #dee2e6;
    color: #212529;
}

.modal-body {
    padding: 1.5rem;
}

.modal-footer {
    padding: 1rem 1.5rem;
    border-top: 1px solid #dee2e6;
    display: flex;
    justify-content: flex-end;
    gap: 0.5rem;
    background: #f8f9fa;
    border-radius: 0 0 12px 12px;
}

.detail-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1rem;
    margin-bottom: 1.5rem;
}

.detail-item {
    padding: 0.75rem;
    background: #f8f9fa;
    border-radius: 6px;
    border-left: 3px solid #0056b3;
}

.detail-label {
    font-size: 0.75rem;
    font-weight: 600;
    color: #6c757d;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    margin-bottom: 0.25rem;
}

.detail-value {
    font-size: 1rem;
    color: #212529;
    font-weight: 500;
}

.detail-section {
    margin-bottom: 2rem;
}

.detail-section-title {
    font-size: 1.1rem;
    font-weight: 600;
    color: #212529;
    margin-bottom: 1rem;
    padding-bottom: 0.5rem;
    border-bottom: 2px solid #dee2e6;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.link-button {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    background: #0056b3;
    color: white;
    text-decoration: none;
    border-radius: 6px;
    font-size: 0.9rem;
    transition: all 0.2s;
}

.link-button:hover {
    background: #004085;
    transform: translateY(-1px);
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.btn-sm {
    padding: 0.375rem 0.75rem;
    font-size: 0.875rem;
}

.btn-danger {
    background: #dc3545;
    color: white;
}

.btn-danger:hover {
    background: #c82333;
}

.data-table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 1rem;
}

.data-table th,
.data-table td {
    padding: 0.75rem;
    text-align: left;
    border-bottom: 1px solid #dee2e6;
}

.data-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
}

.data-table tr:hover {
    background: #f8f9fa;
}

.empty-state {
    text-align: center;
    padding: 3rem;
    color: #6c757d;
}

.empty-state i {
    font-size: 3rem;
    margin-bottom: 1rem;
    opacity: 0.5;
}

.loading {
    text-align: center;
    padding: 2rem;
    color: #6c757d;
}

.loading i {
    animation: spin 1s linear infinite;
}

@keyframes spin {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}
</style>

    <div class="hellomateo-container">
        <div class="content-card">
            <div class="card-header">
                <h3><i class="fab fa-whatsapp"></i> WhatsApp Management</h3>
                <div style="display: flex; align-items: center; gap: 1rem; flex-wrap: wrap;">
                    <!-- Autohaus-Auswahl für Admins -->
                    <?php if (($can_switch_autohaus || $current_user_role === 'admin') && !empty($all_autohauser)): ?>
                        <div style="display: flex; align-items: center; gap: 0.5rem;">
                            <label for="autohaus-select" style="font-size: 0.9rem; color: #6c757d; font-weight: 500;">
                                <i class="fas fa-building"></i> Autohaus:
                            </label>
                            <select id="autohaus-select" onchange="switchAutohaus(this.value)" style="padding: 0.5rem 1rem; border: 2px solid #dee2e6; border-radius: 6px; font-size: 0.9rem; min-width: 200px;">
                                <?php foreach ($all_autohauser as $autohaus): ?>
                                    <option value="<?php echo $autohaus['id']; ?>" 
                                            <?php echo ($active_autohaus_id == $autohaus['id']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($autohaus['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    <?php elseif ($current_user_autohaus_name): ?>
                        <span style="color: #6c757d; font-size: 0.9rem;">
                            <i class="fas fa-building"></i> <?php echo htmlspecialchars($current_user_autohaus_name); ?>
                        </span>
                    <?php endif; ?>
                    
                    <?php 
                    // Prüfe API-Key Status
                    $apiKeyInfo = $hellomateo->getApiKeyInfo();
                    if (!$apiKeyInfo['has_key']): 
                    ?>
                        <div style="background: #fff3cd; color: #856404; padding: 0.5rem 1rem; border-radius: 6px; font-size: 0.85rem; border: 1px solid #ffc107;">
                            <i class="fas fa-exclamation-triangle"></i> 
                            <strong>API-Key nicht gefunden</strong>
                            <br>
                            <small>Autohaus ID: <?php echo htmlspecialchars($active_autohaus_id ?? 'NULL'); ?> | 
                            Bitte in <a href="settings.php#whatsapp" style="color: #856404; text-decoration: underline; font-weight: 600;">Einstellungen</a> konfigurieren</small>
                        </div>
                    <?php else: ?>
                        <div style="background: #d4edda; color: #155724; padding: 0.5rem 1rem; border-radius: 6px; font-size: 0.85rem; border: 1px solid #28a745;">
                            <i class="fas fa-check-circle"></i> 
                            <strong>API-Key gefunden</strong>
                            <br>
                            <small>Länge: <?php echo $apiKeyInfo['key_length']; ?> Zeichen | Autohaus ID: <?php echo htmlspecialchars($apiKeyInfo['autohaus_id'] ?? 'NULL'); ?></small>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        
        <!-- Tabs -->
        <div class="hellomateo-tabs">
            <button class="hellomateo-tab <?php echo $active_tab === 'messages' ? 'active' : ''; ?>" data-tab="messages">
                <i class="fas fa-paper-plane"></i> Nachrichten
            </button>
            <button class="hellomateo-tab <?php echo $active_tab === 'contacts' ? 'active' : ''; ?>" data-tab="contacts">
                <i class="fas fa-address-book"></i> Kontakte
            </button>
            <button class="hellomateo-tab <?php echo $active_tab === 'templates' ? 'active' : ''; ?>" data-tab="templates">
                <i class="fas fa-file-alt"></i> Templates
            </button>
            <button class="hellomateo-tab <?php echo $active_tab === 'conversations' ? 'active' : ''; ?>" data-tab="conversations">
                <i class="fas fa-comments"></i> Konversationen
            </button>
        </div>
        
        <!-- Tab Content: Nachrichten -->
        <div class="hellomateo-tab-content <?php echo $active_tab === 'messages' ? 'active' : ''; ?>" id="tab-messages">
            <!-- 🚀 MODERNES NACHRICHTEN-FORMULAR -->
            <div class="modern-message-form">
                <div class="form-header">
                    <div class="header-content">
                        <h2><i class="fas fa-paper-plane"></i> Neue Nachricht senden</h2>
                        <p class="header-subtitle">Senden Sie WhatsApp-Nachrichten mit Templates und Platzhaltern</p>
                    </div>
                    <div class="header-actions">
                        <button type="button" class="btn-icon" onclick="showErrorLogs()" title="Error-Logs anzeigen">
                            <i class="fas fa-bug"></i>
                        </button>
                    </div>
                </div>
                
                <form id="send-message-form" class="modern-form">
                    <!-- Schritt 1: Empfänger -->
                    <div class="form-step" data-step="1">
                        <div class="step-header">
                            <div class="step-number">1</div>
                            <div class="step-title">
                                <h3>Empfänger</h3>
                                <p>Telefonnummer des Empfängers</p>
                            </div>
                        </div>
                        
                        <div class="form-section">
                            <div class="input-group modern-input">
                                <label for="message-to">
                                    <i class="fas fa-phone"></i> Telefonnummer
                                    <span class="required">*</span>
                                </label>
                                <div class="input-wrapper">
                                    <div class="country-selector">
                                        <select id="country-code" class="country-code-select">
                                            <option value="+49" data-flag="🇩🇪">🇩🇪 +49</option>
                                            <option value="+43" data-flag="🇦🇹">🇦🇹 +43</option>
                                            <option value="+41" data-flag="🇨🇭">🇨🇭 +41</option>
                                            <option value="+1" data-flag="🇺🇸">🇺🇸 +1</option>
                                            <option value="+44" data-flag="🇬🇧">🇬🇧 +44</option>
                                        </select>
                                    </div>
                                    <input 
                                        type="tel" 
                                        id="message-to" 
                                        name="to" 
                                        class="phone-input"
                                        placeholder="123456789" 
                                        pattern="[0-9\s\-]+"
                                        required
                                        autocomplete="tel"
                                    >
                                    <button type="button" class="input-action" id="validate-phone" title="Nummer validieren">
                                        <i class="fas fa-check-circle"></i>
                                    </button>
                                </div>
                                <div class="input-feedback" id="phone-feedback">
                                    <i class="fas fa-info-circle"></i>
                                    <span>Geben Sie die Nummer ohne Ländercode ein (z.B. 1782465658)</span>
                                </div>
                                <div class="phone-preview" id="phone-preview" style="display: none;">
                                    <i class="fas fa-check text-success"></i>
                                    <span>Formatierte Nummer: <strong id="formatted-phone"></strong></span>
                                </div>
                            </div>
                            
                            <!-- Bulk-Send Toggle -->
                            <div class="bulk-toggle-wrapper">
                                <label class="modern-toggle">
                                    <input type="checkbox" id="bulk-send-mode" onchange="toggleBulkSendMode()">
                                    <span class="toggle-slider"></span>
                                    <span class="toggle-label">
                                        <i class="fas fa-users"></i> Mehrere Empfänger (Bulk-Send)
                                    </span>
                                </label>
                            </div>
                            
                            <div id="bulk-recipients-container" class="bulk-container" style="display: none;">
                                <label for="bulk-recipients">
                                    <i class="fas fa-list"></i> Empfänger-Liste
                                    <span class="badge-info" id="bulk-count">0</span>
                                </label>
                                <textarea 
                                    id="bulk-recipients" 
                                    rows="6" 
                                    class="bulk-textarea"
                                    placeholder="Eine Nummer pro Zeile (ohne Ländercode):&#10;1782465658&#10;1234567890&#10;..."
                                ></textarea>
                                <div class="input-feedback">
                                    <i class="fas fa-info-circle"></i>
                                    <span>Geben Sie eine Nummer pro Zeile ein. Das Ländercode wird automatisch hinzugefügt.</span>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Schritt 2: Kanal -->
                    <div class="form-step" data-step="2">
                        <div class="step-header">
                            <div class="step-number">2</div>
                            <div class="step-title">
                                <h3>Kanal auswählen</h3>
                                <p>Wählen Sie den WhatsApp-Kanal für den Versand</p>
                            </div>
                        </div>
                        
                        <div class="form-section">
                            <div class="channel-selector">
                                <div class="channel-grid">
                                    <label class="channel-card" for="channel-none">
                                        <input type="radio" name="channel_id" id="channel-none" value="" checked>
                                        <div class="channel-card-content">
                                            <div class="channel-icon">
                                                <i class="fas fa-question-circle"></i>
                                            </div>
                                            <div class="channel-info">
                                                <div class="channel-name">Standard-Kanal</div>
                                                <div class="channel-status warning">Nicht empfohlen</div>
                                            </div>
                                        </div>
                                    </label>
                                    
                                    <?php foreach ($channels as $channel): ?>
                                    <label class="channel-card" for="channel-<?php echo htmlspecialchars($channel['id'] ?? ''); ?>">
                                        <input type="radio" name="channel_id" id="channel-<?php echo htmlspecialchars($channel['id'] ?? ''); ?>" value="<?php echo htmlspecialchars($channel['id'] ?? ''); ?>">
                                        <div class="channel-card-content">
                                            <div class="channel-icon active">
                                                <i class="fab fa-whatsapp"></i>
                                            </div>
                                            <div class="channel-info">
                                                <div class="channel-name"><?php echo htmlspecialchars($channel['name'] ?? $channel['id'] ?? 'Unbekannt'); ?></div>
                                                <div class="channel-details">
                                                    <span class="channel-handle"><?php echo htmlspecialchars($channel['handle'] ?? ''); ?></span>
                                                    <span class="channel-status success"><?php echo htmlspecialchars($channel['status'] ?? 'active'); ?></span>
                                                </div>
                                            </div>
                                            <div class="channel-check">
                                                <i class="fas fa-check-circle"></i>
                                            </div>
                                        </div>
                                    </label>
                                    <?php endforeach; ?>
                                </div>
                                <input type="hidden" id="message-channel" name="channel_id" value="">
                            </div>
                        </div>
                    </div>
                    
                    <!-- Schritt 3: Template oder Text -->
                    <div class="form-step" data-step="3">
                        <div class="step-header">
                            <div class="step-number">3</div>
                            <div class="step-title">
                                <h3>Nachricht</h3>
                                <p>Wählen Sie ein Template oder schreiben Sie eine freie Nachricht</p>
                            </div>
                        </div>
                        
                        <div class="form-section">
                            <!-- Template-Auswahl -->
                            <div class="message-type-selector">
                                <div class="type-tabs">
                                    <button type="button" class="type-tab active" data-type="template" onclick="switchMessageType('template')">
                                        <i class="fas fa-file-alt"></i> Template verwenden
                                    </button>
                                    <button type="button" class="type-tab" data-type="text" onclick="switchMessageType('text')">
                                        <i class="fas fa-keyboard"></i> Freie Nachricht
                                    </button>
                                </div>
                            </div>
                            
                            <!-- Template-Bereich -->
                            <div id="template-section" class="message-section active">
                                <div class="template-selector-wrapper">
                                    <div class="template-search-wrapper">
                                        <i class="fas fa-search"></i>
                                        <input 
                                            type="text" 
                                            id="template-search" 
                                            class="template-search"
                                            placeholder="Template suchen..."
                                            onkeyup="filterTemplates()"
                                        >
                                        <button type="button" class="btn-icon" onclick="invalidateCache('templates'); loadTemplates(false);" title="Templates aktualisieren">
                                            <i class="fas fa-sync"></i>
                                        </button>
                                    </div>
                                    
                                    <select id="message-template" name="template_id" class="modern-select" onchange="loadTemplatePreview()">
                                        <option value="">Kein Template ausgewählt</option>
                                    </select>
                                </div>
                                
                                <!-- Template-Vorschau -->
                                <div id="template-preview" class="template-preview-card" style="display: none;">
                                    <div class="preview-header">
                                        <div>
                                            <h4 id="template-preview-name">Template-Vorschau</h4>
                                            <div class="template-meta">
                                                <span class="badge" id="template-preview-status"></span>
                                                <span class="template-language" id="template-preview-language" style="display: none;"></span>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="preview-content">
                                        <div id="template-preview-content" class="template-text-preview"></div>
                                    </div>
                                    
                                    <!-- Platzhalter-Eingabefelder -->
                                    <div id="template-placeholders-container" class="placeholders-container" style="display: none;">
                                        <div class="placeholders-header">
                                            <i class="fas fa-edit"></i>
                                            <span>Platzhalter ausfüllen</span>
                                            <span class="badge-info" id="placeholders-count">0</span>
                                        </div>
                                        <div id="template-placeholders-fields" class="placeholders-fields"></div>
                                    </div>
                                    
                                    <!-- Live-Vorschau -->
                                    <div id="template-live-preview" class="live-preview-card" style="display: none;">
                                        <div class="preview-header">
                                            <i class="fas fa-eye"></i>
                                            <span>Live-Vorschau</span>
                                        </div>
                                        <div id="template-live-preview-content" class="live-preview-content"></div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Text-Bereich -->
                            <div id="text-section" class="message-section" style="display: none;">
                                <div class="text-input-wrapper">
                                    <label for="message-text">
                                        <i class="fas fa-keyboard"></i> Nachricht
                                        <span id="message-required-indicator" class="required">*</span>
                                        <span class="char-count" id="char-count">0 Zeichen</span>
                                    </label>
                                    <textarea 
                                        id="message-text" 
                                        name="message" 
                                        rows="8" 
                                        class="modern-textarea"
                                        placeholder="Schreiben Sie Ihre Nachricht hier..."
                                        required
                                        oninput="updateMessagePreview(); updateCharCount()"
                                    ></textarea>
                                    <div id="message-hint" class="input-feedback">
                                        <i class="fas fa-info-circle"></i>
                                        <span>Bei Verwendung eines Templates wird dieser Text ignoriert</span>
                                    </div>
                                </div>
                                
                                <!-- Text-Vorschau -->
                                <div id="message-preview" class="message-preview-card" style="display: none;">
                                    <div class="preview-header">
                                        <i class="fas fa-eye"></i>
                                        <span>Vorschau</span>
                                    </div>
                                    <div id="message-preview-content" class="preview-content"></div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Ergebnis-Anzeige -->
                    <div id="message-result" class="result-container"></div>
                    
                    <!-- Formular-Aktionen -->
                    <div class="form-actions">
                        <button type="button" class="btn-secondary" onclick="resetForm()">
                            <i class="fas fa-redo"></i> Zurücksetzen
                        </button>
                        <button type="submit" class="btn-primary btn-send" id="send-button">
                            <i class="fas fa-paper-plane"></i>
                            <span>Nachricht senden</span>
                            <div class="btn-loader" style="display: none;">
                                <i class="fas fa-spinner fa-spin"></i>
                            </div>
                        </button>
                    </div>
                </form>
            </div>
            
            <div class="content-card">
                <div class="card-header">
                    <h3>Gesendete Nachrichten</h3>
                    <div style="display: flex; gap: 0.5rem; align-items: center; flex-wrap: wrap;">
                        <input type="text" id="messages-search" placeholder="Suche nach Text oder Empfänger..." style="padding: 0.5rem; border: 1px solid #ddd; border-radius: 4px; width: 250px;" onkeyup="filterMessages()">
                        <select id="messages-status-filter" style="padding: 0.5rem; border: 1px solid #ddd; border-radius: 4px;" onchange="filterMessages()">
                            <option value="">Alle Status</option>
                            <option value="sent">Gesendet</option>
                            <option value="delivered">Zugestellt</option>
                            <option value="read">Gelesen</option>
                            <option value="failed">Fehlgeschlagen</option>
                            <option value="pending">Ausstehend</option>
                        </select>
                        <input type="date" id="messages-date-from" placeholder="Von" style="padding: 0.5rem; border: 1px solid #ddd; border-radius: 4px;" onchange="filterMessages()">
                        <input type="date" id="messages-date-to" placeholder="Bis" style="padding: 0.5rem; border: 1px solid #ddd; border-radius: 4px;" onchange="filterMessages()">
                        <select id="messages-sort" style="padding: 0.5rem; border: 1px solid #ddd; border-radius: 4px;" onchange="filterMessages()">
                            <option value="date-desc">Datum (neueste zuerst)</option>
                            <option value="date-asc">Datum (älteste zuerst)</option>
                            <option value="status">Status</option>
                            <option value="recipient">Empfänger</option>
                        </select>
                        <button class="btn btn-secondary" onclick="exportMessages()" title="Nachrichten exportieren">
                            <i class="fas fa-download"></i> Export
                        </button>
                        <button class="btn btn-secondary" onclick="invalidateCache('messages'); loadMessages(false);">
                            <i class="fas fa-sync"></i> Aktualisieren
                        </button>
                    </div>
                </div>
                
                <div id="messages-pagination" style="margin-bottom: 1rem; display: flex; justify-content: space-between; align-items: center;">
                    <div>
                        <span id="messages-count">0 Nachrichten</span>
                    </div>
                    <div style="display: flex; gap: 0.5rem;">
                        <button class="btn btn-sm" onclick="changeMessagesPage(-1)" id="messages-prev" disabled>
                            <i class="fas fa-chevron-left"></i> Zurück
                        </button>
                        <span id="messages-page-info">Seite 1</span>
                        <button class="btn btn-sm" onclick="changeMessagesPage(1)" id="messages-next">
                            <i class="fas fa-chevron-right"></i> Weiter
                        </button>
                    </div>
                </div>
                
                <div id="messages-list">
                    <div class="loading">
                        <i class="fas fa-spinner"></i> Lade Nachrichten...
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Tab Content: Kontakte -->
        <div class="hellomateo-tab-content <?php echo $active_tab === 'contacts' ? 'active' : ''; ?>" id="tab-contacts">
            <div class="content-card">
                <div class="card-header">
                    <h3>Kontakte verwalten</h3>
                    <div style="display: flex; gap: 0.5rem;">
                        <button class="btn btn-secondary" onclick="showImportContactsModal()" title="Kontakte importieren">
                            <i class="fas fa-upload"></i> Import
                        </button>
                        <button class="btn btn-secondary" onclick="exportContacts()" title="Kontakte exportieren">
                            <i class="fas fa-download"></i> Export
                        </button>
                    </div>
                </div>
                
                <form id="create-contact-form">
                    <div class="form-group">
                        <label for="contact-handle">Telefonnummer</label>
                        <input type="tel" id="contact-handle" name="handle" placeholder="+49123456789" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="contact-name">Name</label>
                        <input type="text" id="contact-name" name="full_name" placeholder="Max Mustermann" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="contact-country">Ländercode</label>
                        <select id="contact-country" name="country_codes">
                            <option value="DE">Deutschland (DE)</option>
                            <option value="AT">Österreich (AT)</option>
                            <option value="CH">Schweiz (CH)</option>
                        </select>
                    </div>
                    
                    <div id="contact-result"></div>
                    
                    <button type="submit" class="btn btn-success">
                        <i class="fas fa-user-plus"></i> Kontakt erstellen
                    </button>
                </form>
            </div>
            
            <div class="content-card">
                <div class="card-header">
                    <h3>Alle Kontakte</h3>
                    <div style="display: flex; gap: 0.5rem; align-items: center; flex-wrap: wrap;">
                        <input type="text" id="contacts-search" placeholder="Suche nach Name oder Telefonnummer..." style="padding: 0.5rem; border: 1px solid #ddd; border-radius: 4px; width: 250px;" onkeyup="filterContacts()">
                        <select id="contacts-country-filter" style="padding: 0.5rem; border: 1px solid #ddd; border-radius: 4px;" onchange="filterContacts()">
                            <option value="">Alle Länder</option>
                            <option value="DE">Deutschland</option>
                            <option value="AT">Österreich</option>
                            <option value="CH">Schweiz</option>
                        </select>
                        <button class="btn btn-secondary" onclick="exportContacts()" title="Kontakte exportieren">
                            <i class="fas fa-download"></i> Export
                        </button>
                        <button class="btn btn-secondary" onclick="invalidateCache('contacts'); loadContacts(false);">
                            <i class="fas fa-sync"></i> Aktualisieren
                        </button>
                    </div>
                </div>
                
                <div id="contacts-pagination" style="margin-bottom: 1rem; display: flex; justify-content: space-between; align-items: center;">
                    <div>
                        <span id="contacts-count">0 Kontakte</span>
                    </div>
                    <div style="display: flex; gap: 0.5rem;">
                        <button class="btn btn-sm" onclick="changeContactsPage(-1)" id="contacts-prev" disabled>
                            <i class="fas fa-chevron-left"></i> Zurück
                        </button>
                        <span id="contacts-page-info">Seite 1</span>
                        <button class="btn btn-sm" onclick="changeContactsPage(1)" id="contacts-next">
                            <i class="fas fa-chevron-right"></i> Weiter
                        </button>
                    </div>
                </div>
                
                <div id="contacts-list">
                    <div class="loading">
                        <i class="fas fa-spinner"></i> Lade Kontakte...
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Tab Content: Templates -->
        <div class="hellomateo-tab-content <?php echo $active_tab === 'templates' ? 'active' : ''; ?>" id="tab-templates">
            <div class="content-card">
                <div class="card-header">
                    <h3>Verfügbare Templates</h3>
                    <div style="display: flex; gap: 0.5rem; align-items: center; flex-wrap: wrap;">
                        <input type="text" id="templates-search" placeholder="Suche nach Name oder ID..." style="padding: 0.5rem; border: 1px solid #ddd; border-radius: 4px; width: 250px;" onkeyup="renderTemplates()">
                        <select id="templates-status-filter" style="padding: 0.5rem; border: 1px solid #ddd; border-radius: 4px;" onchange="renderTemplates()">
                            <option value="">Alle Status</option>
                            <option value="approved">Genehmigt</option>
                            <option value="pending">Ausstehend</option>
                            <option value="rejected">Abgelehnt</option>
                            <option value="draft">Entwurf</option>
                        </select>
                        <select id="templates-language-filter" style="padding: 0.5rem; border: 1px solid #ddd; border-radius: 4px;" onchange="renderTemplates()">
                            <option value="">Alle Sprachen</option>
                            <option value="de">Deutsch</option>
                            <option value="en">Englisch</option>
                        </select>
                        <button class="btn btn-secondary" onclick="exportTemplates()" title="Templates exportieren">
                            <i class="fas fa-download"></i> Export
                        </button>
                        <button class="btn btn-secondary" onclick="invalidateCache('templates'); loadTemplates(false);">
                            <i class="fas fa-sync"></i> Aktualisieren
                        </button>
                    </div>
                </div>
                
                <div id="templates-list">
                    <div class="loading">
                        <i class="fas fa-spinner"></i> Lade Templates...
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Tab Content: Konversationen -->
        <div class="hellomateo-tab-content <?php echo $active_tab === 'conversations' ? 'active' : ''; ?>" id="tab-conversations">
            <div class="content-card">
                <div class="card-header">
                    <h3>Konversationen</h3>
                    <div style="display: flex; gap: 0.5rem; align-items: center; flex-wrap: wrap;">
                        <input type="text" id="conversations-search" placeholder="Suche nach Betreff oder Kontakt..." style="padding: 0.5rem; border: 1px solid #ddd; border-radius: 4px; width: 250px;" onkeyup="renderConversations()">
                        <select id="conversations-status-filter" style="padding: 0.5rem; border: 1px solid #ddd; border-radius: 4px;" onchange="renderConversations()">
                            <option value="">Alle Status</option>
                            <option value="active">Aktiv</option>
                            <option value="open">Offen</option>
                            <option value="closed">Geschlossen</option>
                            <option value="pending">Ausstehend</option>
                        </select>
                        <button class="btn btn-secondary" onclick="invalidateCache('conversations'); loadConversations(false);">
                            <i class="fas fa-sync"></i> Aktualisieren
                        </button>
                    </div>
                </div>
                
                <div id="conversations-pagination" style="margin-bottom: 1rem; display: flex; justify-content: space-between; align-items: center;">
                    <div>
                        <span id="conversations-count">0 Konversationen</span>
                    </div>
                    <div style="display: flex; gap: 0.5rem;">
                        <button class="btn btn-sm" onclick="changeConversationsPage(-1)" id="conversations-prev" disabled>
                            <i class="fas fa-chevron-left"></i> Zurück
                        </button>
                        <span id="conversations-page-info">Seite 1</span>
                        <button class="btn btn-sm" onclick="changeConversationsPage(1)" id="conversations-next">
                            <i class="fas fa-chevron-right"></i> Weiter
                        </button>
                    </div>
                </div>
                
                <div id="conversations-list">
                    <div class="loading">
                        <i class="fas fa-spinner"></i> Lade Konversationen...
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// ============================================
// API CACHING SYSTEM - Reduziert API-Aufrufe
// ============================================

const API_CACHE = {
    contacts: { data: null, timestamp: 0, ttl: 5 * 60 * 1000 }, // 5 Minuten
    conversations: { data: null, timestamp: 0, ttl: 5 * 60 * 1000 }, // 5 Minuten
    messages: { data: null, timestamp: 0, ttl: 2 * 60 * 1000 }, // 2 Minuten
    templates: { data: null, timestamp: 0, ttl: 10 * 60 * 1000 }, // 10 Minuten
    channels: { data: null, timestamp: 0, ttl: 30 * 60 * 1000 } // 30 Minuten
};

// Prüfe ob Cache gültig ist
function isCacheValid(cacheKey) {
    const cache = API_CACHE[cacheKey];
    if (!cache || !cache.data) return false;
    const age = Date.now() - cache.timestamp;
    return age < cache.ttl;
}

// Hole Daten aus Cache oder lade neu
function getCachedData(cacheKey, loadFunction, forceReload = false) {
    if (!forceReload && isCacheValid(cacheKey)) {
        console.log(`✅ Cache HIT für ${cacheKey}`);
        return Promise.resolve(API_CACHE[cacheKey].data);
    }
    
    console.log(`🔄 Cache MISS für ${cacheKey} - lade neu...`);
    return loadFunction().then(data => {
        if (data && data.success !== false) {
            API_CACHE[cacheKey].data = data;
            API_CACHE[cacheKey].timestamp = Date.now();
        }
        return data;
    });
}

// Cache invalidieren
function invalidateCache(cacheKey = null) {
    if (cacheKey) {
        API_CACHE[cacheKey].data = null;
        API_CACHE[cacheKey].timestamp = 0;
        console.log(`🗑️ Cache invalidiert: ${cacheKey}`);
    } else {
        // Alle Caches invalidieren
        Object.keys(API_CACHE).forEach(key => {
            API_CACHE[key].data = null;
            API_CACHE[key].timestamp = 0;
        });
        console.log('🗑️ Alle Caches invalidiert');
    }
}

// Cache-Status anzeigen (für Debugging)
function getCacheStatus() {
    const status = {};
    Object.keys(API_CACHE).forEach(key => {
        const cache = API_CACHE[key];
        const age = cache.timestamp ? Math.floor((Date.now() - cache.timestamp) / 1000) : null;
        status[key] = {
            cached: !!cache.data,
            age: age ? `${age}s` : 'never',
            valid: isCacheValid(key),
            count: cache.data ? (Array.isArray(cache.data) ? cache.data.length : 'object') : 0
        };
    });
    return status;
}

// Tab Switching
document.querySelectorAll('.hellomateo-tab').forEach(tab => {
    tab.addEventListener('click', function() {
        const tabName = this.getAttribute('data-tab');
        
        // Update URL
        const url = new URL(window.location);
        url.searchParams.set('tab', tabName);
        window.history.pushState({}, '', url);
        
        // Switch tabs
        document.querySelectorAll('.hellomateo-tab').forEach(t => t.classList.remove('active'));
        document.querySelectorAll('.hellomateo-tab-content').forEach(c => c.classList.remove('active'));
        
        this.classList.add('active');
        document.getElementById('tab-' + tabName).classList.add('active');
        
        // Load data for active tab (mit Cache)
        switch(tabName) {
            case 'messages':
                loadMessages(1, true); // true = Cache verwenden
                loadTemplates(1, true); // Load templates for select
                break;
            case 'contacts':
                loadContacts(1, true); // true = Cache verwenden
                break;
            case 'templates':
                loadTemplates(1, true); // true = Cache verwenden
                break;
            case 'conversations':
                loadConversations(1, true); // true = Cache verwenden
                break;
        }
    });
});

// Extrahiere Platzhalter aus Template-Content
function extractPlaceholders(template) {
    const placeholders = [];
    
    if (!template) return placeholders;
    
    // Methode 1: Prüfe content-Objekt (Slate-Format) für handlebar-Objekte
    if (template.content && typeof template.content === 'object') {
        function findHandlebars(obj) {
            if (!obj) return;
            
            // Wenn es ein handlebar-Objekt ist
            if (obj.type === 'handlebar' && obj.key) {
                if (!placeholders.includes(obj.key)) {
                    placeholders.push(obj.key);
                }
            }
            
            // Rekursiv durch children suchen
            if (obj.children && Array.isArray(obj.children)) {
                obj.children.forEach(child => {
                    findHandlebars(child);
                });
            }
        }
        
        findHandlebars(template.content);
    }
    
    // Methode 2: Prüfe body/text für {{placeholder}} Format
    let contentText = '';
    if (template.body) {
        contentText = template.body;
    } else if (template.text) {
        contentText = template.text;
    } else if (template.content && typeof template.content === 'string') {
        contentText = template.content;
    } else if (template.content && typeof template.content === 'object') {
        // Extrahiere Text aus Slate-Format für zusätzliche Suche
        contentText = extractTextFromSlate(template.content);
    }
    
    // Suche nach Platzhaltern im Format {{placeholder}} oder {{placeholder_name}}
    if (contentText) {
        const placeholderRegex = /\{\{(\w+)\}\}/g;
        const matches = contentText.matchAll(placeholderRegex);
        
        for (const match of matches) {
            const placeholderName = match[1];
            if (!placeholders.includes(placeholderName)) {
                placeholders.push(placeholderName);
            }
        }
    }
    
    return placeholders;
}

// Erstelle Template-Text aus Content (für Anzeige)
function getTemplateText(template) {
    if (!template) return '';
    
    if (template.body) {
        return template.body;
    }
    
    if (template.content) {
        // Wenn content ein Slate-Objekt ist, extrahiere Text
        if (typeof template.content === 'object') {
            return extractTextFromSlate(template.content);
        }
        return template.content;
    }
    
    if (template.text) {
        return template.text;
    }
    
    // Fallback: Versuche aus snippet zu extrahieren
    if (template.snippet) {
        return template.snippet;
    }
    
    return '';
}

// Extrahiere Text aus Slate-Content-Objekt
function extractTextFromSlate(content) {
    if (!content || typeof content !== 'object') return '';
    
    let text = '';
    
    function processNode(node) {
        if (!node) return;
        
        // Text-Node
        if (node.type === 'text' && node.content) {
            text += node.content;
        }
        // Handlebar-Node (Platzhalter)
        else if (node.type === 'handlebar' && node.key) {
            text += '{{' + node.key + '}}';
        }
        // Paragraph oder andere Container
        else if (node.children && Array.isArray(node.children)) {
            node.children.forEach(child => {
                processNode(child);
            });
            // Füge Zeilenumbruch nach Paragraph hinzu
            if (node.type === 'paragraph') {
                text += '\n';
            }
        }
    }
    
    // Starte mit root oder direkt mit content
    if (content.type === 'root' && content.children) {
        content.children.forEach(child => {
            processNode(child);
        });
    } else {
        processNode(content);
    }
    
    return text.trim();
}

// Validiere Platzhalter-Eingaben
function validatePlaceholders() {
    const templateSelect = document.getElementById('message-template');
    const templateId = templateSelect.value;
    
    if (!templateId) return true;
    
    const template = allTemplates.find(t => t.id === templateId);
    if (!template) return true;
    
    const placeholders = extractPlaceholders(template);
    if (placeholders.length === 0) return true;
    
    let allFilled = true;
    placeholders.forEach(placeholder => {
        const input = document.getElementById(`template-placeholder-${placeholder}`);
        if (input && !input.value.trim()) {
            allFilled = false;
            input.style.borderColor = '#dc3545';
        } else if (input) {
            input.style.borderColor = '#28a745';
        }
    });
    
    // Aktiviere/Deaktiviere Senden-Button basierend auf Validierung
    const sendButton = document.querySelector('#send-message-form button[type="submit"]');
    if (sendButton) {
        if (allFilled) {
            sendButton.disabled = false;
            sendButton.style.opacity = '1';
        } else {
            sendButton.disabled = true;
            sendButton.style.opacity = '0.6';
        }
    }
    
    return allFilled;
}

// Update Live-Vorschau
function updateTemplateLivePreview() {
    const templateSelect = document.getElementById('message-template');
    const templateId = templateSelect.value;
    const livePreviewDiv = document.getElementById('template-live-preview');
    const livePreviewContent = document.getElementById('template-live-preview-content');
    
    if (!templateId) {
        livePreviewDiv.style.display = 'none';
        return;
    }
    
    // Verwende loadFullTemplateDetails um sicherzustellen, dass vollständige Daten vorhanden sind
    loadFullTemplateDetails(templateId, function(template) {
        if (!template) {
            livePreviewDiv.style.display = 'none';
            return;
        }
        
        const placeholders = extractPlaceholders(template);
        if (placeholders.length === 0) {
            livePreviewDiv.style.display = 'none';
            return;
        }
        
        // Sammle Platzhalter-Werte
        const placeholderValues = {};
        placeholders.forEach(placeholder => {
            const input = document.getElementById(`template-placeholder-${placeholder}`);
            if (input) {
                placeholderValues[placeholder] = input.value || '';
            }
        });
        
        // Erstelle Live-Vorschau
        let templateText = getTemplateText(template);
        
        // Ersetze Platzhalter durch Werte
        placeholders.forEach(placeholder => {
            const value = placeholderValues[placeholder] || `[${placeholder}]`;
            templateText = templateText.replace(new RegExp(`\\{\\{${placeholder}\\}\\}`, 'g'), value);
        });
        
        livePreviewContent.textContent = templateText;
        livePreviewDiv.style.display = 'block';
    });
}

// Lade vollständige Template-Details (falls nicht vorhanden)
function loadFullTemplateDetails(templateId, callback) {
    const template = allTemplates.find(t => t.id === templateId);
    
    // Wenn Template bereits vollständige content-Daten hat, rufe Callback direkt auf
    if (template && template.content) {
        callback(template);
        return;
    }
    
    // Lade Template-Details von API
    const formData = new FormData();
    formData.append('ajax_action', 'get_template_details');
    formData.append('template_id', templateId);
    
    fetch('hellomateo.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success && data.data) {
            // Aktualisiere Template in allTemplates
            const index = allTemplates.findIndex(t => t.id === templateId);
            if (index !== -1) {
                allTemplates[index] = { ...allTemplates[index], ...data.data };
            }
            callback(data.data);
        } else {
            // Fallback auf vorhandenes Template
            callback(template);
        }
    })
    .catch(error => {
        console.error('Fehler beim Laden der Template-Details:', error);
        callback(template);
    });
}

// Load Template Preview
function loadTemplatePreview() {
    const templateSelect = document.getElementById('message-template');
    const templateId = templateSelect.value;
    const previewDiv = document.getElementById('template-preview');
    const previewContent = document.getElementById('template-preview-content');
    const placeholdersContainer = document.getElementById('template-placeholders-container');
    const placeholdersFields = document.getElementById('template-placeholders-fields');
    const livePreviewDiv = document.getElementById('template-live-preview');
    const messageTextarea = document.getElementById('message-text');
    const messageRequiredIndicator = document.getElementById('message-required-indicator');
    const messageHint = document.getElementById('message-hint');
    
    if (!templateId) {
        previewDiv.style.display = 'none';
        placeholdersContainer.style.display = 'none';
        livePreviewDiv.style.display = 'none';
        // Nachricht ist erforderlich wenn kein Template
        if (messageTextarea) {
            messageTextarea.disabled = false;
            messageTextarea.required = true;
            messageTextarea.style.background = 'white';
            messageTextarea.style.cursor = 'text';
            messageTextarea.placeholder = 'Ihre Nachricht...';
        }
        if (messageRequiredIndicator) {
            messageRequiredIndicator.style.display = 'inline';
        }
        if (messageHint) {
            messageHint.textContent = 'Nachricht ist erforderlich';
            messageHint.style.color = '#6c757d';
        }
        
        // Aktiviere Senden-Button
        const sendButton = document.querySelector('#send-message-form button[type="submit"]');
        if (sendButton) {
            sendButton.disabled = false;
            sendButton.style.opacity = '1';
        }
        
        return;
    }
    
    // Lade vollständige Template-Details
    loadFullTemplateDetails(templateId, function(template) {
        if (!template) {
            previewDiv.style.display = 'none';
            placeholdersContainer.style.display = 'none';
            livePreviewDiv.style.display = 'none';
            return;
        }
        
        previewDiv.style.display = 'block';
        
        // Modernes Template-Vorschau-Design
        const templateName = document.getElementById('template-preview-name');
        const templateStatus = document.getElementById('template-preview-status');
        const templateLanguage = document.getElementById('template-preview-language');
        const templateTextPreview = document.getElementById('template-preview-content');
        
        if (templateName) {
            templateName.textContent = template.name || 'Unbekannt';
        }
        
        if (templateStatus) {
            templateStatus.innerHTML = getTemplateStatusBadge(template.status || 'unknown');
        }
        
        if (templateLanguage && template.language) {
            templateLanguage.textContent = template.language.toUpperCase();
            templateLanguage.style.display = 'inline-block';
        } else if (templateLanguage) {
            templateLanguage.style.display = 'none';
        }
        
        const templateText = getTemplateText(template);
        if (templateTextPreview && templateText) {
            templateTextPreview.textContent = templateText;
        }
        
        // Prüfe auf Platzhalter
        const placeholders = extractPlaceholders(template);
        
        // Debug-Logging
        console.log('loadTemplatePreview - Template:', {
            id: template.id,
            name: template.name,
            hasContent: !!template.content,
            hasBody: !!template.body,
            contentType: typeof template.content,
            contentPreview: template.content ? JSON.stringify(template.content).substring(0, 200) : 'none',
            placeholders: placeholders
        });
        
        if (placeholders.length > 0) {
            // Zeige Platzhalter-Eingabefelder
            placeholdersContainer.style.display = 'block';
            let fieldsHtml = '';
            
            // Aktualisiere Platzhalter-Zähler
            const placeholdersCount = document.getElementById('placeholders-count');
            if (placeholdersCount) {
                placeholdersCount.textContent = placeholders.length;
            }
            
            placeholders.forEach((placeholder, index) => {
                fieldsHtml += `
                    <div class="placeholder-field">
                        <label for="template-placeholder-${placeholder}">
                            <code>{{${placeholder}}}</code>
                            <span class="required">*</span>
                        </label>
                        <input 
                            type="text" 
                            id="template-placeholder-${placeholder}" 
                            class="template-placeholder-input placeholder-input"
                            placeholder="Wert für ${placeholder} eingeben..."
                            required
                            oninput="updateTemplateLivePreview(); validatePlaceholders()"
                        >
                    </div>
                `;
            });
            
            placeholdersFields.innerHTML = fieldsHtml;
            
            // DEAKTIVIERE Nachricht-Textfeld wenn Platzhalter vorhanden
            if (messageTextarea) {
                messageTextarea.disabled = true;
                messageTextarea.required = false;
                messageTextarea.value = ''; // Leere das Feld
                messageTextarea.style.background = '#f5f5f5';
                messageTextarea.style.cursor = 'not-allowed';
                messageTextarea.placeholder = 'Deaktiviert - Bitte füllen Sie die Platzhalter aus';
            }
            
            // Validiere Platzhalter nach kurzer Verzögerung (damit DOM bereit ist)
            setTimeout(() => {
                validatePlaceholders();
            }, 100);
            
            // Zeige Live-Vorschau
            updateTemplateLivePreview();
        } else {
            placeholdersContainer.style.display = 'none';
            livePreviewDiv.style.display = 'none';
            
            // AKTIVIERE Nachricht-Textfeld wenn keine Platzhalter
            if (messageTextarea) {
                messageTextarea.disabled = false;
                messageTextarea.required = false;
                messageTextarea.style.background = 'white';
                messageTextarea.style.cursor = 'text';
                messageTextarea.placeholder = 'Optionale Nachricht (wird ignoriert wenn Template verwendet wird)';
            }
        }
        
        // Wenn Template verwendet wird, ist Nachricht optional
        if (messageTextarea && placeholders.length === 0) {
            messageTextarea.required = false;
        }
        if (messageRequiredIndicator) {
            messageRequiredIndicator.style.display = placeholders.length > 0 ? 'none' : 'inline';
        }
        if (messageHint) {
            if (placeholders.length > 0) {
                messageHint.innerHTML = '<strong style="color: #2196f3;">ℹ️ Hinweis:</strong> Das Nachricht-Textfeld ist deaktiviert. Bitte füllen Sie alle Platzhalter aus. Die Nachricht wird erst versendet, wenn alle Platzhalter-Werte angegeben sind.';
                messageHint.style.color = '#2196f3';
            } else {
                messageHint.innerHTML = '<strong style="color: #ff9800;">⚠️ Hinweis:</strong> Bei Verwendung eines Templates wird nur das Template gesendet, der Text wird ignoriert.';
                messageHint.style.color = '#ff9800';
            }
        }
    });
}

// Toggle Bulk Send Mode (aktualisiert für neues Design)
function toggleBulkSendMode() {
    const bulkMode = document.getElementById('bulk-send-mode').checked;
    const bulkContainer = document.getElementById('bulk-recipients-container');
    const toInput = document.getElementById('message-to');
    const countryCode = document.getElementById('country-code');
    const phoneWrapper = toInput ? toInput.closest('.input-wrapper') : null;
    
    if (bulkMode) {
        bulkContainer.style.display = 'block';
        if (toInput) {
            toInput.required = false;
            toInput.style.display = 'none';
        }
        if (countryCode) countryCode.style.display = 'none';
        if (phoneWrapper) phoneWrapper.style.display = 'none';
    } else {
        bulkContainer.style.display = 'none';
        if (toInput) {
            toInput.required = true;
            toInput.style.display = 'block';
        }
        if (countryCode) countryCode.style.display = 'block';
        if (phoneWrapper) phoneWrapper.style.display = 'flex';
    }
}

// Update Message Preview
function updateMessagePreview() {
    const messageText = document.getElementById('message-text').value;
    const previewDiv = document.getElementById('message-preview');
    const previewContent = document.getElementById('message-preview-content');
    
    if (messageText && messageText.length > 0) {
        previewDiv.style.display = 'block';
        previewContent.textContent = messageText;
    } else {
        previewDiv.style.display = 'none';
    }
}

// Send Message - Erweitert
document.getElementById('send-message-form').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const templateId = document.getElementById('message-template').value;
    const message = document.getElementById('message-text').value;
    const to = document.getElementById('message-to').value;
    
    // Validierung: Wenn Template verwendet wird, ist Nachricht optional
    // Wenn kein Template verwendet wird, ist Nachricht erforderlich
    if (!to) {
        document.getElementById('message-result').innerHTML = '<div class="alert alert-error"><i class="fas fa-exclamation-circle"></i> Bitte geben Sie eine Telefonnummer ein</div>';
        return;
    }
    
    // Prüfe ob Template mit Platzhaltern verwendet wird
    const template = allTemplates.find(t => t.id === templateId);
    const placeholders = template ? extractPlaceholders(template) : [];
    
    if (templateId && placeholders.length > 0) {
        // Wenn Template mit Platzhaltern: Validiere Platzhalter
        const allPlaceholdersFilled = validatePlaceholders();
        if (!allPlaceholdersFilled) {
            document.getElementById('message-result').innerHTML = '<div class="alert alert-error"><i class="fas fa-exclamation-circle"></i> Bitte füllen Sie alle Platzhalter aus</div>';
            return;
        }
    } else if (!templateId && !message) {
        document.getElementById('message-result').innerHTML = '<div class="alert alert-error"><i class="fas fa-exclamation-circle"></i> Bitte geben Sie eine Nachricht ein oder wählen Sie ein Template</div>';
        return;
    }
    
    const bulkMode = document.getElementById('bulk-send-mode').checked;
    const resultDiv = document.getElementById('message-result');
    
    if (bulkMode) {
        const bulkRecipients = document.getElementById('bulk-recipients').value.split('\n').filter(r => r.trim());
        if (bulkRecipients.length === 0) {
            resultDiv.innerHTML = '<div class="alert alert-error"><i class="fas fa-exclamation-circle"></i> Bitte geben Sie mindestens einen Empfänger ein</div>';
            return;
        }
        
        const message = document.getElementById('message-text').value;
        const templateId = document.getElementById('message-template').value || null;
        
        resultDiv.innerHTML = '<div class="loading"><i class="fas fa-spinner"></i> Sende ' + bulkRecipients.length + ' Nachrichten...</div>';
        
        let successCount = 0;
        let failCount = 0;
        let promises = [];
        
        // Sammle Platzhalter-Werte für Bulk-Send
        const template = allTemplates.find(t => t.id === templateId);
        let placeholderValuesJson = null;
        if (template && templateId) {
            const placeholders = extractPlaceholders(template);
            if (placeholders.length > 0) {
                const placeholderValues = {};
                placeholders.forEach(placeholder => {
                    const input = document.getElementById(`template-placeholder-${placeholder}`);
                    if (input && input.value) {
                        placeholderValues[placeholder] = input.value;
                    }
                });
                if (Object.keys(placeholderValues).length > 0) {
                    placeholderValuesJson = JSON.stringify(placeholderValues);
                }
            }
        }
        
        bulkRecipients.forEach((recipient, index) => {
            const formData = new FormData();
            formData.append('ajax_action', 'send_message');
            formData.append('to', recipient.trim());
            // WICHTIG: Wenn Template verwendet wird, KEINEN Text mitsenden
            if (templateId) {
                formData.append('template_id', templateId);
                // Platzhalter-Werte mitsenden
                if (placeholderValuesJson) {
                    formData.append('placeholder_values', placeholderValuesJson);
                }
                // Kein message-Feld bei Template
            } else {
                formData.append('message', message);
            }
            
            promises.push(
                fetch('hellomateo.php', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        successCount++;
                    } else {
                        failCount++;
                    }
                })
                .catch(error => {
                    failCount++;
                })
            );
        });
        
        Promise.all(promises).then(() => {
            resultDiv.innerHTML = `<div class="alert ${failCount === 0 ? 'alert-success' : 'alert-warning'}">
                <i class="fas fa-${failCount === 0 ? 'check-circle' : 'exclamation-triangle'}"></i> 
                ${successCount} Nachrichten erfolgreich gesendet${failCount > 0 ? ', ' + failCount + ' fehlgeschlagen' : ''}
            </div>`;
            if (failCount === 0) {
                // Cache invalidieren nach erfolgreichem Senden
                invalidateCache('messages');
                invalidateCache('conversations');
                this.reset();
                loadMessages(false); // false = Cache verwenden (wird gerade invalidiert, also neu laden)
            }
        });
    } else {
        const formData = new FormData();
        formData.append('ajax_action', 'send_message');
        formData.append('to', to);
        
        // WICHTIG: Wenn Template verwendet wird, KEINEN Text mitsenden
        const currentTemplateId = document.getElementById('message-template').value;
        if (currentTemplateId) {
            formData.append('template_id', currentTemplateId);
            
            // Sammle Platzhalter-Werte
            // Prüfe ob Platzhalter-Eingabefelder vorhanden sind
            const placeholderInputs = document.querySelectorAll('.template-placeholder-input');
            if (placeholderInputs.length > 0) {
                const placeholderValues = {};
                let allFilled = true;
                
                placeholderInputs.forEach(input => {
                    const placeholderName = input.id.replace('template-placeholder-', '');
                    if (input.value.trim()) {
                        placeholderValues[placeholderName] = input.value.trim();
                    } else {
                        allFilled = false;
                    }
                });
                
                // Sende Platzhalter-Werte als JSON
                if (allFilled && Object.keys(placeholderValues).length > 0) {
                    formData.append('placeholder_values', JSON.stringify(placeholderValues));
                    console.log('Sending placeholder_values:', placeholderValues);
                } else {
                    console.warn('Nicht alle Platzhalter ausgefüllt!', placeholderValues);
                }
            } else {
                // Fallback: Versuche Platzhalter aus Template zu extrahieren
                const template = allTemplates.find(t => t.id === currentTemplateId);
                if (template) {
                    const placeholders = extractPlaceholders(template);
                    if (placeholders.length > 0) {
                        const placeholderValues = {};
                        placeholders.forEach(placeholder => {
                            const input = document.getElementById(`template-placeholder-${placeholder}`);
                            if (input && input.value.trim()) {
                                placeholderValues[placeholder] = input.value.trim();
                            }
                        });
                        
                        if (Object.keys(placeholderValues).length > 0) {
                            formData.append('placeholder_values', JSON.stringify(placeholderValues));
                            console.log('Sending placeholder_values (fallback):', placeholderValues);
                        }
                    }
                }
            }
            
            // Kein message-Feld bei Template - wird in sendMessage() gehandhabt
        } else {
            formData.append('message', message);
        }
        
        const channelId = document.getElementById('message-channel').value;
        if (channelId) {
            formData.append('channel_id', channelId);
        }
        
        resultDiv.innerHTML = '<div class="loading"><i class="fas fa-spinner"></i> Sende Nachricht...</div>';
        
        fetch('hellomateo.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            console.log('Send Message API Response:', data);
            
            if (data.success) {
                // Cache invalidieren nach erfolgreichem Senden
                invalidateCache('messages');
                invalidateCache('conversations');
                
                let message = '<div class="alert alert-success"><i class="fas fa-check-circle"></i> Nachricht erfolgreich gesendet!</div>';
                
                // Zeige Warnung falls vorhanden
                if (data.warning) {
                    message += '<div class="alert alert-warning" style="margin-top: 0.5rem;"><i class="fas fa-exclamation-triangle"></i> ' + escapeHtml(data.warning) + '</div>';
                }
                
                // Zeige zusätzliche Debug-Informationen in der Konsole
                if (data.data) {
                    console.log('API Response Data:', data.data);
                    
                    // Prüfe Message-Status
                    let messageStatus = null;
                    let messageId = null;
                    let recipientStatus = null;
                    
                    if (data.data.message) {
                        messageStatus = data.data.message.status;
                        messageId = data.data.message.id;
                    } else if (data.data.status) {
                        messageStatus = data.data.status;
                        messageId = data.data.id || data.data.message_id;
                    }
                    
                    if (data.data.recipients && Array.isArray(data.data.recipients)) {
                        const toRecipient = data.data.recipients.find(r => r.role === 'to');
                        if (toRecipient) {
                            recipientStatus = toRecipient.status;
                        }
                    }
                    
                    // Prüfe ob Template-Platzhalter fehlen
                    let hasPlaceholders = false;
                    let missingPlaceholders = [];
                    if (data.data.message && data.data.message.content) {
                        // Prüfe ob Platzhalter im Content vorhanden sind
                        const contentStr = JSON.stringify(data.data.message.content);
                        const placeholderMatches = contentStr.match(/\{\{(\w+)\}\}/g);
                        if (placeholderMatches) {
                            hasPlaceholders = true;
                            placeholderMatches.forEach(match => {
                                const placeholderName = match.replace(/\{\{|\}\}/g, '');
                                if (!data.data.message.placeholder_values || !data.data.message.placeholder_values[placeholderName]) {
                                    missingPlaceholders.push(placeholderName);
                                }
                            });
                        }
                    }
                    
                    // Zeige Status-Information
                    if (messageStatus === 'scheduled' || recipientStatus === 'scheduled') {
                        // Bei WhatsApp-Templates ist "scheduled" normal - die Nachricht wird sofort versendet
                        // Der Status bedeutet nur, dass sie in die Warteschlange eingereiht wurde
                        message += '<div class="alert alert-success" style="margin-top: 0.5rem;">';
                        message += '<i class="fas fa-check-circle"></i> <strong>Status: Erfolgreich versendet</strong><br>';
                        
                        if (hasPlaceholders && missingPlaceholders.length > 0) {
                            message += '<strong style="color: #856404;">⚠️ WICHTIG:</strong> Die Nachricht wurde erstellt, aber möglicherweise nicht versendet, weil Template-Platzhalter fehlen!<br>';
                            message += 'Fehlende Platzhalter: <code>' + missingPlaceholders.join(', ') + '</code><br>';
                            message += 'Die HelloMateo API benötigt Werte für alle Platzhalter im Template, bevor die Nachricht versendet werden kann.<br>';
                            message += '<small>Bitte verwenden Sie ein Template ohne Platzhalter oder geben Sie die Platzhalter-Werte an.</small><br>';
                        } else {
                            message += 'Die Nachricht wurde erfolgreich in die Warteschlange eingereiht und wird sofort versendet.<br>';
                            message += '<small style="color: #6c757d;">Hinweis: Bei WhatsApp-Templates zeigt die API den Status "scheduled" an, auch wenn die Nachricht bereits versendet wurde. ';
                            message += 'Dies ist normal und bedeutet, dass die Nachricht erfolgreich verarbeitet wurde.</small><br>';
                        }
                        
                        if (messageId) {
                            message += '<small>Message-ID: ' + escapeHtml(messageId) + '</small>';
                        }
                        message += '</div>';
                    } else if (messageStatus === 'sent' || recipientStatus === 'sent') {
                        message += '<div class="alert alert-success" style="margin-top: 0.5rem;">';
                        message += '<i class="fas fa-check-circle"></i> <strong>Status: Versendet</strong><br>';
                        message += 'Die Nachricht wurde erfolgreich versendet.';
                        message += '</div>';
                    } else if (messageStatus === 'failed' || recipientStatus === 'failed') {
                        message += '<div class="alert alert-error" style="margin-top: 0.5rem;">';
                        message += '<i class="fas fa-exclamation-circle"></i> <strong>Status: Fehlgeschlagen</strong><br>';
                        message += 'Die Nachricht konnte nicht versendet werden.';
                        if (data.data.message && data.data.message.error) {
                            message += '<br>Fehler: ' + escapeHtml(data.data.message.error);
                        }
                        message += '</div>';
                    } else if (!messageId) {
                        message += '<div class="alert alert-warning" style="margin-top: 0.5rem;"><i class="fas fa-exclamation-triangle"></i> <strong>Warnung:</strong> Die API hat keine Message-ID zurückgegeben. Die Nachricht wurde möglicherweise nicht wirklich gesendet. Bitte prüfen Sie die PHP Error-Logs für Details.</div>';
                    }
                } else {
                    message += '<div class="alert alert-warning" style="margin-top: 0.5rem;"><i class="fas fa-exclamation-triangle"></i> <strong>Warnung:</strong> Die API hat keine Daten zurückgegeben. Bitte prüfen Sie die PHP Error-Logs für Details.</div>';
                }
                if (data.http_code) {
                    console.log('HTTP Status Code:', data.http_code);
                }
                
                // Zeige vollständige API-Antwort in der Konsole für Debugging
                console.log('Vollständige API-Antwort:', JSON.stringify(data, null, 2));
                
                resultDiv.innerHTML = message;
                
                // Button-Status zurücksetzen
                const sendButton = document.getElementById('send-button');
                if (sendButton) {
                    sendButton.classList.remove('loading');
                    sendButton.disabled = false;
                }
                
                // Formular zurücksetzen bei Erfolg
                if (data.success) {
                    setTimeout(() => {
                        resetForm();
                        loadMessages();
                    }, 2000);
                }
            } else {
                let errorMsg = data.error || 'Unbekannter Fehler';
                let alertClass = 'alert-error';
                let icon = 'fa-exclamation-circle';
                
                // Spezielle Behandlung für HTTP 429 (Rate Limit)
                if (data.http_code === 429) {
                    alertClass = 'alert-warning';
                    icon = 'fa-clock';
                    errorMsg = 'Zu viele Anfragen (Rate Limit erreicht).';
                    errorMsg += '<br><br><strong>Mögliche Ursachen:</strong>';
                    errorMsg += '<br>1. Zu viele Anfragen in kurzer Zeit';
                    errorMsg += '<br>2. Rate Limit auf API-Key-Ebene';
                    errorMsg += '<br>3. Rate Limit auf IP-Adresse';
                    errorMsg += '<br><br><strong>Lösung:</strong>';
                    errorMsg += '<br>• Warten Sie mindestens 1 Stunde';
                    errorMsg += '<br>• Prüfen Sie Ihren API-Key in den Einstellungen';
                    errorMsg += '<br>• Kontaktieren Sie HelloMateo Support, wenn das Problem länger als 24 Stunden anhält';
                } else if (data.http_code) {
                    errorMsg += ' (HTTP ' + data.http_code + ')';
                }
                
                if (data.data) {
                    console.error('API Error Data:', data.data);
                    // Zeige zusätzliche Fehlerdetails falls vorhanden
                    if (typeof data.data === 'object') {
                        const errorDetails = JSON.stringify(data.data, null, 2);
                        errorMsg += '<br><small style="font-family: monospace; font-size: 0.85rem; display: block; margin-top: 0.5rem; color: #666;">' + escapeHtml(errorDetails) + '</small>';
                    }
                }
                resultDiv.innerHTML = '<div class="alert ' + alertClass + '"><i class="fas ' + icon + '"></i> Fehler: ' + escapeHtml(errorMsg) + '</div>';
                
                // Button-Status zurücksetzen
                const sendButton = document.getElementById('send-button');
                if (sendButton) {
                    sendButton.classList.remove('loading');
                    sendButton.disabled = false;
                }
            }
        })
        .catch(error => {
            console.error('Send Message Error:', error);
            resultDiv.innerHTML = '<div class="alert alert-error"><i class="fas fa-exclamation-circle"></i> Fehler: ' + escapeHtml(error.message) + '</div>';
            
            // Button-Status zurücksetzen
            const sendButton = document.getElementById('send-button');
            if (sendButton) {
                sendButton.classList.remove('loading');
                sendButton.disabled = false;
            }
        });
    }
});

// Update preview on input
document.getElementById('message-text').addEventListener('input', updateMessagePreview);

// 🚀 NEUE FUNKTIONEN FÜR MODERNES DESIGN

// Nachricht-Typ wechseln (Template/Text)
function switchMessageType(type) {
    const templateSection = document.getElementById('template-section');
    const textSection = document.getElementById('text-section');
    const templateTab = document.querySelector('.type-tab[data-type="template"]');
    const textTab = document.querySelector('.type-tab[data-type="text"]');
    const templateSelect = document.getElementById('message-template');
    const messageText = document.getElementById('message-text');
    
    if (type === 'template') {
        templateSection.classList.add('active');
        textSection.classList.remove('active');
        templateTab.classList.add('active');
        textTab.classList.remove('active');
        messageText.required = false;
    } else {
        templateSection.classList.remove('active');
        textSection.classList.add('active');
        templateTab.classList.remove('active');
        textTab.classList.add('active');
        templateSelect.value = '';
        loadTemplatePreview(); // Verstecke Template-Vorschau
        messageText.required = true;
    }
}

// Zeichenzähler für Text-Eingabe
function updateCharCount() {
    const textarea = document.getElementById('message-text');
    const charCount = document.getElementById('char-count');
    if (textarea && charCount) {
        const count = textarea.value.length;
        charCount.textContent = count + ' Zeichen';
        if (count > 1000) {
            charCount.style.color = '#ef4444';
        } else if (count > 500) {
            charCount.style.color = '#f59e0b';
        } else {
            charCount.style.color = '#9ca3af';
        }
    }
}

// Telefonnummer-Validierung und Formatierung
document.addEventListener('DOMContentLoaded', function() {
    const phoneInput = document.getElementById('message-to');
    const countryCode = document.getElementById('country-code');
    const validateBtn = document.getElementById('validate-phone');
    const phoneFeedback = document.getElementById('phone-feedback');
    const phonePreview = document.getElementById('phone-preview');
    const formattedPhone = document.getElementById('formatted-phone');
    
    if (phoneInput && countryCode) {
        // Auto-Formatierung beim Tippen
        phoneInput.addEventListener('input', function(e) {
            let value = e.target.value.replace(/\D/g, ''); // Nur Zahlen
            e.target.value = value;
            
            // Formatierung anzeigen
            if (value.length > 0) {
                const code = countryCode.value;
                const formatted = code + ' ' + value;
                if (formattedPhone) {
                    formattedPhone.textContent = formatted;
                    phonePreview.style.display = 'flex';
                }
            } else {
                if (phonePreview) phonePreview.style.display = 'none';
            }
        });
        
        // Validierung
        if (validateBtn) {
            validateBtn.addEventListener('click', function() {
                const value = phoneInput.value.replace(/\D/g, '');
                const code = countryCode.value;
                
                if (value.length >= 7 && value.length <= 15) {
                    phoneInput.style.borderColor = '#10b981';
                    phoneFeedback.innerHTML = '<i class="fas fa-check-circle text-success"></i> <span>Nummer ist gültig</span>';
                    phoneFeedback.style.color = '#065f46';
                } else {
                    phoneInput.style.borderColor = '#ef4444';
                    phoneFeedback.innerHTML = '<i class="fas fa-exclamation-circle text-error"></i> <span>Nummer muss 7-15 Ziffern haben</span>';
                    phoneFeedback.style.color = '#991b1b';
                }
            });
        }
        
        // Ländercode-Änderung
        countryCode.addEventListener('change', function() {
            const value = phoneInput.value.replace(/\D/g, '');
            if (value.length > 0 && formattedPhone) {
                formattedPhone.textContent = countryCode.value + ' ' + value;
            }
        });
    }
    
    // Bulk-Send Zeichenzähler
    const bulkTextarea = document.getElementById('bulk-recipients');
    const bulkCount = document.getElementById('bulk-count');
    if (bulkTextarea && bulkCount) {
        bulkTextarea.addEventListener('input', function() {
            const lines = this.value.split('\n').filter(line => line.trim().length > 0);
            bulkCount.textContent = lines.length;
        });
    }
    
    // Kanal-Auswahl Handler
    const channelRadios = document.querySelectorAll('input[name="channel_id"]');
    const channelHidden = document.getElementById('message-channel');
    
    channelRadios.forEach(radio => {
        radio.addEventListener('change', function() {
            if (channelHidden) {
                channelHidden.value = this.value;
            }
        });
    });
    
    // Template-Suche
    const templateSearch = document.getElementById('template-search');
    if (templateSearch) {
        templateSearch.addEventListener('input', filterTemplates);
    }
});

// Template-Filterung
function filterTemplates() {
    const searchTerm = document.getElementById('template-search').value.toLowerCase();
    const templateSelect = document.getElementById('message-template');
    const options = templateSelect.querySelectorAll('option');
    
    options.forEach(option => {
        if (option.value === '') return;
        const text = option.textContent.toLowerCase();
        if (text.includes(searchTerm)) {
            option.style.display = '';
        } else {
            option.style.display = 'none';
        }
    });
}

// Formular zurücksetzen
function resetForm() {
    const form = document.getElementById('send-message-form');
    if (form) {
        form.reset();
        document.getElementById('message-channel').value = '';
        document.getElementById('bulk-send-mode').checked = false;
        toggleBulkSendMode();
        switchMessageType('text');
        loadTemplatePreview();
        updateCharCount();
        document.getElementById('message-result').innerHTML = '';
        
        // Alle Vorschauen verstecken
        document.getElementById('template-preview').style.display = 'none';
        document.getElementById('message-preview').style.display = 'none';
        document.getElementById('phone-preview').style.display = 'none';
    }
}

// Verbesserte Bulk-Send Toggle
function toggleBulkSendMode() {
    const bulkMode = document.getElementById('bulk-send-mode').checked;
    const bulkContainer = document.getElementById('bulk-recipients-container');
    const toInput = document.getElementById('message-to');
    const countryCode = document.getElementById('country-code');
    
    if (bulkMode) {
        bulkContainer.style.display = 'block';
        if (toInput) toInput.required = false;
        if (countryCode) countryCode.style.display = 'none';
    } else {
        bulkContainer.style.display = 'none';
        if (toInput) toInput.required = true;
        if (countryCode) countryCode.style.display = 'block';
    }
}

// Create Contact
document.getElementById('create-contact-form').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    formData.append('ajax_action', 'create_contact');
    
    const resultDiv = document.getElementById('contact-result');
    resultDiv.innerHTML = '<div class="loading"><i class="fas fa-spinner"></i> Erstelle Kontakt...</div>';
    
    fetch('hellomateo.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Cache invalidieren nach erfolgreichem Erstellen
            invalidateCache('contacts');
            resultDiv.innerHTML = '<div class="alert alert-success"><i class="fas fa-check-circle"></i> Kontakt erfolgreich erstellt!</div>';
            this.reset();
            loadContacts(false); // false = Cache verwenden (wird gerade invalidiert, also neu laden)
        } else {
            resultDiv.innerHTML = '<div class="alert alert-error"><i class="fas fa-exclamation-circle"></i> Fehler: ' + (data.error || 'Unbekannter Fehler') + '</div>';
        }
    })
    .catch(error => {
        resultDiv.innerHTML = '<div class="alert alert-error"><i class="fas fa-exclamation-circle"></i> Fehler: ' + error.message + '</div>';
    });
});

// Messages Pagination
let messagesCurrentPage = 1;
let messagesPageSize = 20;
let allMessages = [];

// Load Messages (mit Cache)
function loadMessages(page = 1, useCache = true) {
    messagesCurrentPage = page;
    const listDiv = document.getElementById('messages-list');
    
    // Prüfe Cache
    if (useCache && isCacheValid('messages') && API_CACHE.messages.data) {
        console.log('✅ Verwende gecachte Nachrichten');
        allMessages = API_CACHE.messages.data;
        window.allMessages = allMessages;
        filterMessages();
        return;
    }
    
    listDiv.innerHTML = '<div class="loading"><i class="fas fa-spinner"></i> Lade Nachrichten...</div>';
    
    const formData = new FormData();
    formData.append('ajax_action', 'load_messages');
    formData.append('limit', 100); // Lade mehr für Filterung
    formData.append('offset', 0);
    
    fetch('hellomateo.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success && data.data) {
            allMessages = Array.isArray(data.data) ? data.data : [];
            window.allMessages = allMessages;
            // Cache speichern
            API_CACHE.messages.data = allMessages;
            API_CACHE.messages.timestamp = Date.now();
            console.log(`✅ Nachrichten geladen: ${allMessages.length} (gecacht)`);
            filterMessages();
        } else {
            listDiv.innerHTML = '<div class="alert alert-error"><i class="fas fa-exclamation-circle"></i> Fehler: ' + (data.error || 'Nachrichten konnten nicht geladen werden') + '</div>';
        }
    })
    .catch(error => {
        listDiv.innerHTML = '<div class="alert alert-error"><i class="fas fa-exclamation-circle"></i> Fehler: ' + error.message + '</div>';
    });
}

// Filter Messages - Erweitert
function filterMessages() {
    const searchTerm = (document.getElementById('messages-search')?.value || '').toLowerCase();
    const statusFilter = document.getElementById('messages-status-filter')?.value || '';
    const dateFrom = document.getElementById('messages-date-from')?.value || '';
    const dateTo = document.getElementById('messages-date-to')?.value || '';
    const sortBy = document.getElementById('messages-sort')?.value || 'date-desc';
    
    let filtered = allMessages.filter(msg => {
        const matchesSearch = !searchTerm || 
            (msg.text && msg.text.toLowerCase().includes(searchTerm)) ||
            (msg.blurb && msg.blurb.toLowerCase().includes(searchTerm)) ||
            (msg.recipients && msg.recipients.some(r => r.handle && r.handle.toLowerCase().includes(searchTerm))) ||
            (msg.author_id && msg.author_id.toLowerCase().includes(searchTerm)) ||
            (msg.from && msg.from.toLowerCase().includes(searchTerm));
        const matchesStatus = !statusFilter || (msg.status && msg.status.toLowerCase() === statusFilter.toLowerCase());
        
        // Datum-Filter
        let matchesDate = true;
        if (dateFrom || dateTo) {
            const msgDate = msg.created_at ? new Date(msg.created_at).toISOString().split('T')[0] : '';
            if (dateFrom && msgDate < dateFrom) matchesDate = false;
            if (dateTo && msgDate > dateTo) matchesDate = false;
        }
        
        return matchesSearch && matchesStatus && matchesDate;
    });
    
    // Sortierung
    filtered.sort((a, b) => {
        switch(sortBy) {
            case 'date-asc':
                const dateA = a.created_at ? new Date(a.created_at) : new Date(0);
                const dateB = b.created_at ? new Date(b.created_at) : new Date(0);
                return dateA - dateB;
            case 'date-desc':
                const dateA2 = a.created_at ? new Date(a.created_at) : new Date(0);
                const dateB2 = b.created_at ? new Date(b.created_at) : new Date(0);
                return dateB2 - dateA2;
            case 'status':
                const statusA = (a.status || '').toLowerCase();
                const statusB = (b.status || '').toLowerCase();
                return statusA.localeCompare(statusB);
            case 'recipient':
                const recipientA = a.recipients ? (a.recipients[0]?.handle || '') : (a.to || '');
                const recipientB = b.recipients ? (b.recipients[0]?.handle || '') : (b.to || '');
                return recipientA.localeCompare(recipientB);
            default:
                return 0;
        }
    });
    
    // Pagination
    const totalPages = Math.ceil(filtered.length / messagesPageSize);
    const startIndex = (messagesCurrentPage - 1) * messagesPageSize;
    const endIndex = startIndex + messagesPageSize;
    const paginatedMessages = filtered.slice(startIndex, endIndex);
    
    // Update Pagination Controls
    document.getElementById('messages-count').textContent = `${filtered.length} Nachrichten`;
    document.getElementById('messages-page-info').textContent = `Seite ${messagesCurrentPage} von ${Math.max(1, totalPages)}`;
    document.getElementById('messages-prev').disabled = messagesCurrentPage <= 1;
    document.getElementById('messages-next').disabled = messagesCurrentPage >= totalPages;
    
    // Render Messages
    const listDiv = document.getElementById('messages-list');
    if (paginatedMessages.length === 0) {
        listDiv.innerHTML = '<div class="empty-state"><i class="fas fa-inbox"></i><p>Keine Nachrichten gefunden</p></div>';
    } else {
        let html = '<table class="data-table"><thead><tr><th>Datum</th><th>Von</th><th>An</th><th>Nachricht</th><th>Status</th><th>Aktionen</th></tr></thead><tbody>';
        paginatedMessages.forEach(msg => {
            const date = msg.created_at ? new Date(msg.created_at).toLocaleString('de-DE') : '-';
            const from = msg.author_id || msg.from || '-';
            const to = msg.recipients ? msg.recipients.map(r => r.handle || r).join(', ') : (msg.to || '-');
            const text = msg.text || msg.blurb || '-';
            const status = msg.status || 'unknown';
            const statusBadge = getMessageStatusBadge(status);
            const msgId = msg.id || msg.message_id || '';
            
            // Timestamp-Badges für Status
            let statusTimestamps = '';
            if (msg.delivered_at) {
                statusTimestamps += `<div style="font-size: 0.75rem; color: #6c757d; margin-top: 0.25rem;"><i class="fas fa-check-double"></i> Zugestellt: ${new Date(msg.delivered_at).toLocaleString('de-DE')}</div>`;
            }
            if (msg.read_at) {
                statusTimestamps += `<div style="font-size: 0.75rem; color: #6c757d; margin-top: 0.25rem;"><i class="fas fa-eye"></i> Gelesen: ${new Date(msg.read_at).toLocaleString('de-DE')}</div>`;
            }
            
            html += `<tr>
                <td>${date}</td>
                <td>${from}</td>
                <td>${to}</td>
                <td><span title="${escapeHtml(text)}">${escapeHtml(text.substring(0, 50))}${text.length > 50 ? '...' : ''}</span></td>
                <td>
                    ${statusBadge}
                    ${statusTimestamps}
                </td>
                <td>
                    <button class="btn btn-sm" onclick="showMessageDetails('${msgId}')" title="Details anzeigen"><i class="fas fa-eye"></i></button>
                    ${status === 'failed' ? `<button class="btn btn-sm btn-warning" onclick="resendMessage('${msgId}')" title="Erneut senden"><i class="fas fa-redo"></i></button>` : ''}
                </td>
            </tr>`;
        });
        html += '</tbody></table>';
        listDiv.innerHTML = html;
    }
}

// Change Messages Page
function changeMessagesPage(direction) {
    const searchTerm = (document.getElementById('messages-search')?.value || '').toLowerCase();
    const statusFilter = document.getElementById('messages-status-filter')?.value || '';
    
    let filtered = allMessages.filter(msg => {
        const matchesSearch = !searchTerm || 
            (msg.text && msg.text.toLowerCase().includes(searchTerm)) ||
            (msg.blurb && msg.blurb.toLowerCase().includes(searchTerm)) ||
            (msg.recipients && msg.recipients.some(r => r.handle && r.handle.toLowerCase().includes(searchTerm)));
        const matchesStatus = !statusFilter || (msg.status && msg.status.toLowerCase() === statusFilter.toLowerCase());
        return matchesSearch && matchesStatus;
    });
    
    const totalPages = Math.ceil(filtered.length / messagesPageSize);
    const newPage = messagesCurrentPage + direction;
    
    if (newPage >= 1 && newPage <= totalPages) {
        messagesCurrentPage = newPage;
        filterMessages();
    }
}

// Show Message Details - Erweitert
function showMessageDetails(messageId) {
    // Lade vollständige Details vom Server
    const formData = new FormData();
    formData.append('ajax_action', 'get_message_details');
    formData.append('message_id', messageId);
    
    fetch('hellomateo.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (!data.success || !data.data) {
            alert('Nachricht nicht gefunden');
            return;
        }
        
        const message = data.data;
        const status = (message.status || 'unknown').toLowerCase();
        const statusBadge = getMessageStatusBadge(status);
        
        let html = `
            <div class="modal-overlay" onclick="if(event.target === this) this.remove()">
                <div class="modal-content">
                    <div class="modal-header">
                        <h3><i class="fas fa-paper-plane"></i> Nachricht-Details</h3>
                        <button class="modal-close" onclick="this.closest('.modal-overlay').remove()">&times;</button>
                    </div>
                    <div class="modal-body">
                        <div class="detail-section">
                            <div class="detail-section-title"><i class="fas fa-info-circle"></i> Grundinformationen</div>
                            <div class="detail-grid">
                                <div class="detail-item">
                                    <div class="detail-label">Nachricht-ID</div>
                                    <div class="detail-value">${escapeHtml(message.id || message.message_id || '-')}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Status</div>
                                    <div class="detail-value">${statusBadge}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Erstellt am</div>
                                    <div class="detail-value">${message.created_at ? new Date(message.created_at).toLocaleString('de-DE') : '-'}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Aktualisiert am</div>
                                    <div class="detail-value">${message.updated_at ? new Date(message.updated_at).toLocaleString('de-DE') : '-'}</div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="detail-section">
                            <div class="detail-section-title"><i class="fas fa-users"></i> Teilnehmer</div>
                            <div class="detail-grid">
                                <div class="detail-item">
                                    <div class="detail-label">Von / Absender</div>
                                    <div class="detail-value">${escapeHtml(message.author_id || message.from || message.sender || '-')}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">An / Empfänger</div>
                                    <div class="detail-value">${message.recipients ? message.recipients.map(r => escapeHtml(r.handle || r.name || r)).join(', ') : (message.to || '-')}</div>
                                </div>
                                ${message.channel_id ? `
                                <div class="detail-item">
                                    <div class="detail-label">Kanal</div>
                                    <div class="detail-value">${escapeHtml(message.channel_id)}</div>
                                </div>
                                ` : ''}
                            </div>
                        </div>
                        
                        <div class="detail-section">
                            <div class="detail-section-title"><i class="fas fa-envelope"></i> Nachricht</div>
                            <div class="detail-item" style="grid-column: 1 / -1;">
                                <div class="detail-label">Inhalt</div>
                                <div class="detail-value" style="white-space: pre-wrap; background: white; padding: 1rem; border-radius: 6px; border: 1px solid #dee2e6;">${escapeHtml(message.text || message.blurb || message.body || '-')}</div>
                            </div>
                        </div>
                        
                        ${message.template_id ? `
                        <div class="detail-section">
                            <div class="detail-section-title"><i class="fas fa-file-alt"></i> Template</div>
                            <div class="detail-grid">
                                <div class="detail-item">
                                    <div class="detail-label">Template-ID</div>
                                    <div class="detail-value">${escapeHtml(message.template_id)}</div>
                                </div>
                            </div>
                        </div>
                        ` : ''}
                        
                        ${message.delivery_status || message.read_status ? `
                        <div class="detail-section">
                            <div class="detail-section-title"><i class="fas fa-check-double"></i> Zustellungsstatus</div>
                            <div class="detail-grid">
                                ${message.delivery_status ? `
                                <div class="detail-item">
                                    <div class="detail-label">Zugestellt</div>
                                    <div class="detail-value">${message.delivered_at ? new Date(message.delivered_at).toLocaleString('de-DE') : (message.delivery_status === 'delivered' ? 'Ja' : 'Nein')}</div>
                                </div>
                                ` : ''}
                                ${message.read_status ? `
                                <div class="detail-item">
                                    <div class="detail-label">Gelesen</div>
                                    <div class="detail-value">${message.read_at ? new Date(message.read_at).toLocaleString('de-DE') : (message.read_status === 'read' ? 'Ja' : 'Nein')}</div>
                                </div>
                                ` : ''}
                            </div>
                        </div>
                        ` : ''}
                        
                        ${message.error || message.error_message ? `
                        <div class="detail-section">
                            <div class="detail-section-title"><i class="fas fa-exclamation-triangle"></i> Fehler-Details</div>
                            <div class="detail-item" style="grid-column: 1 / -1; background: #f8d7da; border-left-color: #dc3545;">
                                <div class="detail-label">Fehlermeldung</div>
                                <div class="detail-value" style="color: #721c24;">${escapeHtml(message.error || message.error_message || '-')}</div>
                            </div>
                        </div>
                        ` : ''}
                        
                        ${message.metadata || message.raw_data ? `
                        <div class="detail-section">
                            <div class="detail-section-title"><i class="fas fa-code"></i> Technische Details</div>
                            <div class="detail-item" style="grid-column: 1 / -1;">
                                <div class="detail-label">Vollständige Daten</div>
                                <div class="detail-value" style="font-family: monospace; font-size: 0.85rem; background: #f8f9fa; padding: 1rem; border-radius: 6px; overflow-x: auto;">
                                    <pre>${JSON.stringify(message, null, 2)}</pre>
                                </div>
                            </div>
                        </div>
                        ` : ''}
                    </div>
                    <div class="modal-footer">
                        ${status === 'failed' ? `
                        <button class="btn btn-warning" onclick="resendMessage('${messageId}'); this.closest('.modal-overlay').remove();">
                            <i class="fas fa-redo"></i> Erneut senden
                        </button>
                        ` : ''}
                        <button class="btn btn-secondary" onclick="this.closest('.modal-overlay').remove()">
                            <i class="fas fa-times"></i> Schließen
                        </button>
                    </div>
                </div>
            </div>
        `;
        
        const modal = document.createElement('div');
        modal.innerHTML = html;
        document.body.appendChild(modal);
    })
    .catch(error => {
        alert('Fehler beim Laden der Nachrichtendetails: ' + error.message);
    });
}

// Helper: Message Status Badge
function getMessageStatusBadge(status) {
    const statusLower = status.toLowerCase();
    const badges = {
        'sent': '<span class="badge badge-message-sent"><i class="fas fa-check"></i> Gesendet</span>',
        'delivered': '<span class="badge badge-message-delivered"><i class="fas fa-check-double"></i> Zugestellt</span>',
        'read': '<span class="badge badge-message-read"><i class="fas fa-check-double"></i> Gelesen</span>',
        'failed': '<span class="badge badge-message-failed"><i class="fas fa-times"></i> Fehlgeschlagen</span>',
        'pending': '<span class="badge badge-message-pending"><i class="fas fa-clock"></i> Ausstehend</span>'
    };
    return badges[statusLower] || `<span class="badge badge-info">${escapeHtml(status)}</span>`;
}

// Contacts Pagination
let contactsCurrentPage = 1;
let contactsPageSize = 20;
let allContacts = [];

// ============================================
// NEUE KONTAKTE-LOGIK - REKURSIVES LADEN
// ============================================

// Load Contacts - Rekursiv ALLE Kontakte laden (mit Cache)
function loadContacts(page = 1, useCache = true) {
    contactsCurrentPage = page;
    const listDiv = document.getElementById('contacts-list');
    
    // Prüfe Cache
    if (useCache && isCacheValid('contacts') && API_CACHE.contacts.data) {
        console.log('✅ Verwende gecachte Kontakte');
        window.allContacts = API_CACHE.contacts.data;
        filterContacts();
        return;
    }
    
    listDiv.innerHTML = '<div class="loading"><i class="fas fa-spinner"></i> Lade alle Kontakte...</div>';
    
    // Lade ALLE Kontakte rekursiv
    loadAllContactsRecursive(0, []);
}

// Rekursive Funktion zum Laden ALLER Kontakte
function loadAllContactsRecursive(offset, accumulatedContacts) {
    const formData = new FormData();
    formData.append('ajax_action', 'load_contacts');
    formData.append('limit', 100); // Lade in Batches von 100
    formData.append('offset', offset);
    
    console.log(`Lade Kontakte: offset=${offset}, bereits geladen: ${accumulatedContacts.length}`);
    
    fetch('hellomateo.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        console.log(`API Response: success=${data.success}, data length=${data.data ? (Array.isArray(data.data) ? data.data.length : 'not array') : 'no data'}`);
        
        if (data.success && data.data) {
            const newContacts = Array.isArray(data.data) ? data.data : [];
            const allContacts = [...accumulatedContacts, ...newContacts];
            
            console.log(`Neue Kontakte: ${newContacts.length}, Gesamt: ${allContacts.length}`);
            
            // Wenn wir 0 Kontakte erhalten haben, sind wir fertig
            if (newContacts.length === 0) {
                // Keine weiteren Kontakte - fertig
                window.allContacts = allContacts;
                // Cache speichern
                API_CACHE.contacts.data = allContacts;
                API_CACHE.contacts.timestamp = Date.now();
                console.log(`✅ Alle Kontakte geladen: ${allContacts.length} Kontakte (gecacht)`);
                filterContacts();
            } else {
                // Wir haben Kontakte erhalten - lade IMMER den nächsten Batch
                // Nur wenn wir 0 Kontakte erhalten, wissen wir, dass wir fertig sind
                console.log(`Lade nächsten Batch (offset=${offset + newContacts.length})...`);
                loadAllContactsRecursive(offset + newContacts.length, allContacts);
            }
        } else {
            // Fehler oder keine weiteren Kontakte
            if (accumulatedContacts.length > 0) {
                window.allContacts = accumulatedContacts;
                // Cache speichern
                API_CACHE.contacts.data = accumulatedContacts;
                API_CACHE.contacts.timestamp = Date.now();
                console.log(`⚠️ Kontakte geladen (mit Fehler): ${accumulatedContacts.length} Kontakte (gecacht)`);
                filterContacts();
            } else {
                const listDiv = document.getElementById('contacts-list');
                if (listDiv) {
                    listDiv.innerHTML = '<div class="alert alert-error"><i class="fas fa-exclamation-circle"></i> Fehler: ' + (data.error || 'Kontakte konnten nicht geladen werden') + '</div>';
                }
            }
        }
    })
    .catch(error => {
        console.error('Error loading contacts:', error);
        // Verwende bereits geladene Kontakte falls vorhanden
        if (accumulatedContacts.length > 0) {
            window.allContacts = accumulatedContacts;
            // Cache speichern
            API_CACHE.contacts.data = accumulatedContacts;
            API_CACHE.contacts.timestamp = Date.now();
            console.log(`⚠️ Kontakte geladen (mit Fehler): ${accumulatedContacts.length} Kontakte (gecacht)`);
            filterContacts();
        } else {
            const listDiv = document.getElementById('contacts-list');
            if (listDiv) {
                listDiv.innerHTML = '<div class="alert alert-error"><i class="fas fa-exclamation-circle"></i> Fehler beim Laden der Kontakte: ' + error.message + '</div>';
            }
        }
    });
}

// Show Import Contacts Modal
function showImportContactsModal() {
    let html = `
        <div class="modal-overlay" onclick="if(event.target === this) this.remove()">
            <div class="modal-content" style="max-width: 600px;">
                <div class="modal-header">
                    <h3><i class="fas fa-upload"></i> Kontakte importieren</h3>
                    <button class="modal-close" onclick="this.closest('.modal-overlay').remove()">&times;</button>
                </div>
                <div class="modal-body">
                    <p style="margin-bottom: 1rem; color: #6c757d;">Importieren Sie Kontakte aus einer CSV-Datei. Erwartetes Format:</p>
                    <div style="background: #f8f9fa; padding: 1rem; border-radius: 6px; margin-bottom: 1rem; font-family: monospace; font-size: 0.85rem;">
                        Name,Telefonnummer,Ländercode<br>
                        Max Mustermann,+49123456789,DE<br>
                        ...
                    </div>
                    <div class="form-group">
                        <label for="import-file">CSV-Datei auswählen</label>
                        <input type="file" id="import-file" accept=".csv" onchange="handleFileSelect(event)">
                    </div>
                    <div id="import-preview" style="display: none; margin-top: 1rem;">
                        <div style="font-weight: 600; margin-bottom: 0.5rem;">Vorschau:</div>
                        <div id="import-preview-content" style="max-height: 200px; overflow-y: auto; background: #f8f9fa; padding: 0.75rem; border-radius: 6px;"></div>
                    </div>
                    <div id="import-result"></div>
                </div>
                <div class="modal-footer">
                    <button class="btn btn-primary" onclick="importContacts()" id="import-btn" disabled>
                        <i class="fas fa-upload"></i> Importieren
                    </button>
                    <button class="btn btn-secondary" onclick="this.closest('.modal-overlay').remove()">
                        <i class="fas fa-times"></i> Abbrechen
                    </button>
                </div>
            </div>
        </div>
    `;
    
    const modal = document.createElement('div');
    modal.innerHTML = html;
    document.body.appendChild(modal);
    
    window.importContactsData = null;
}

// Handle File Select
function handleFileSelect(event) {
    const file = event.target.files[0];
    if (!file) return;
    
    const reader = new FileReader();
    reader.onload = function(e) {
        const text = e.target.result;
        const lines = text.split('\n').filter(l => l.trim());
        
        if (lines.length === 0) {
            alert('Datei ist leer');
            return;
        }
        
        // Parse CSV
        const contacts = [];
        for (let i = 1; i < lines.length; i++) { // Skip header
            const parts = lines[i].split(',').map(p => p.trim().replace(/^"|"$/g, ''));
            if (parts.length >= 2) {
                contacts.push({
                    full_name: parts[0] || '',
                    handle: parts[1] || '',
                    country_codes: parts[2] ? [parts[2]] : ['DE']
                });
            }
        }
        
        if (contacts.length === 0) {
            alert('Keine gültigen Kontakte gefunden');
            return;
        }
        
        window.importContactsData = contacts;
        
        // Show preview
        const previewDiv = document.getElementById('import-preview');
        const previewContent = document.getElementById('import-preview-content');
        const importBtn = document.getElementById('import-btn');
        
        previewDiv.style.display = 'block';
        previewContent.innerHTML = contacts.slice(0, 10).map(c => 
            `<div style="padding: 0.5rem; border-bottom: 1px solid #dee2e6;">${escapeHtml(c.full_name)} - ${escapeHtml(c.handle)}</div>`
        ).join('') + (contacts.length > 10 ? `<div style="padding: 0.5rem; color: #6c757d; font-style: italic;">... und ${contacts.length - 10} weitere</div>` : '');
        
        importBtn.disabled = false;
    };
    
    reader.readAsText(file);
}

// Import Contacts
function importContacts() {
    if (!window.importContactsData || window.importContactsData.length === 0) {
        alert('Keine Kontakte zum Importieren');
        return;
    }
    
    const resultDiv = document.getElementById('import-result');
    const importBtn = document.getElementById('import-btn');
    importBtn.disabled = true;
    importBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Importiere...';
    
    let successCount = 0;
    let failCount = 0;
    let promises = [];
    
    window.importContactsData.forEach(contact => {
        const formData = new FormData();
        formData.append('ajax_action', 'create_contact');
        formData.append('handle', contact.handle);
        formData.append('full_name', contact.full_name);
        formData.append('country_codes', contact.country_codes[0] || 'DE');
        
        promises.push(
            fetch('hellomateo.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    successCount++;
                } else {
                    failCount++;
                }
            })
            .catch(error => {
                failCount++;
            })
        );
    });
    
    Promise.all(promises).then(() => {
        resultDiv.innerHTML = `<div class="alert ${failCount === 0 ? 'alert-success' : 'alert-warning'}">
            <i class="fas fa-${failCount === 0 ? 'check-circle' : 'exclamation-triangle'}"></i> 
            ${successCount} Kontakte erfolgreich importiert${failCount > 0 ? ', ' + failCount + ' fehlgeschlagen' : ''}
        </div>`;
        importBtn.innerHTML = '<i class="fas fa-upload"></i> Importieren';
        
        if (failCount === 0) {
            setTimeout(() => {
                document.querySelector('.modal-overlay').remove();
                loadContacts();
            }, 2000);
        } else {
            importBtn.disabled = false;
        }
    });
}

// Export Contacts
function exportContacts() {
    const searchTerm = (document.getElementById('contacts-search')?.value || '').toLowerCase();
    const countryFilter = document.getElementById('contacts-country-filter')?.value || '';
    
    let filtered = allContacts.filter(contact => {
        if (searchTerm) {
            const name = (contact.full_name || '').toLowerCase();
            const handle = (contact.handle || '').toLowerCase();
            if (!name.includes(searchTerm) && !handle.includes(searchTerm)) return false;
        }
        if (countryFilter) {
            const countries = Array.isArray(contact.country_codes) ? contact.country_codes : [contact.country_codes || ''];
            if (!countries.includes(countryFilter)) return false;
        }
        return true;
    });
    
    // Erstelle CSV
    let csv = 'Name,Telefonnummer,Ländercodes,Erstellt am\n';
    filtered.forEach(contact => {
        const name = (contact.full_name || '').replace(/"/g, '""');
        const handle = (contact.handle || '').replace(/"/g, '""');
        const countries = Array.isArray(contact.country_codes) ? contact.country_codes.join('; ') : (contact.country_codes || '');
        const created = contact.created_at ? new Date(contact.created_at).toLocaleDateString('de-DE') : '-';
        csv += `"${name}","${handle}","${countries}","${created}"\n`;
    });
    
    // Download
    const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    link.setAttribute('href', url);
    link.setAttribute('download', `whatsapp-kontakte-${new Date().toISOString().split('T')[0]}.csv`);
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

// Filter Contacts - Erweitert
function filterContacts() {
    const searchTerm = (document.getElementById('contacts-search')?.value || '').toLowerCase();
    const countryFilter = document.getElementById('contacts-country-filter')?.value || '';
    
    // Verwende window.allContacts falls verfügbar, sonst allContacts
    const contactsToFilter = window.allContacts || allContacts || [];
    if (!contactsToFilter || contactsToFilter.length === 0) {
        const listDiv = document.getElementById('contacts-list');
        if (listDiv) {
            listDiv.innerHTML = '<div class="empty-state"><i class="fas fa-address-book"></i><p>Keine Kontakte gefunden</p></div>';
        }
        return;
    }
    
    let filtered = contactsToFilter.filter(contact => {
        if (searchTerm) {
            const name = (contact.full_name || '').toLowerCase();
            const handle = (contact.handle || '').toLowerCase();
            if (!name.includes(searchTerm) && !handle.includes(searchTerm)) return false;
        }
        if (countryFilter) {
            const countries = Array.isArray(contact.country_codes) ? contact.country_codes : [contact.country_codes || ''];
            if (!countries.includes(countryFilter)) return false;
        }
        return true;
    });
    
    // Sortiere nach Name
    filtered.sort((a, b) => {
        const nameA = (a.full_name || '').toLowerCase();
        const nameB = (b.full_name || '').toLowerCase();
        return nameA.localeCompare(nameB);
    });
    
    // Pagination
    const totalPages = Math.ceil(filtered.length / contactsPageSize);
    const startIndex = (contactsCurrentPage - 1) * contactsPageSize;
    const endIndex = startIndex + contactsPageSize;
    const paginatedContacts = filtered.slice(startIndex, endIndex);
    
    // Update Pagination Controls
    document.getElementById('contacts-count').textContent = `${filtered.length} Kontakte`;
    document.getElementById('contacts-page-info').textContent = `Seite ${contactsCurrentPage} von ${Math.max(1, totalPages)}`;
    document.getElementById('contacts-prev').disabled = contactsCurrentPage <= 1;
    document.getElementById('contacts-next').disabled = contactsCurrentPage >= totalPages;
    
    // Render Contacts
    const listDiv = document.getElementById('contacts-list');
    if (paginatedContacts.length === 0) {
        listDiv.innerHTML = '<div class="empty-state"><i class="fas fa-address-book"></i><p>Keine Kontakte gefunden</p></div>';
    } else {
        // Zähle Nachrichten und Konversationen für jeden Kontakt
        const allMessagesForStats = window.allMessages || allMessages || [];
        const allConversationsForStats = window.allConversations || allConversations || [];
        const contactsWithStats = paginatedContacts.map(contact => {
            const contactPhone = contact.handle || '';
            const messagesCount = allMessagesForStats.filter(msg => {
                const recipients = msg.recipients || [];
                return recipients.some(r => (r.handle || '') === contactPhone);
            }).length;
            const conversationsCount = allConversationsForStats.filter(conv => {
                return (conv.contact_id || '') === (contact.id || '');
            }).length;
            
            // Finde letzte Nachricht
            const lastMessage = allMessagesForStats
                .filter(msg => {
                    const recipients = msg.recipients || [];
                    return recipients.some(r => (r.handle || '') === contactPhone);
                })
                .sort((a, b) => {
                    const dateA = a.created_at ? new Date(a.created_at) : new Date(0);
                    const dateB = b.created_at ? new Date(b.created_at) : new Date(0);
                    return dateB - dateA;
                })[0];
            
            return {
                ...contact,
                messagesCount,
                conversationsCount,
                lastMessage: lastMessage ? {
                    text: (lastMessage.text || lastMessage.blurb || '').substring(0, 50),
                    date: lastMessage.created_at
                } : null
            };
        });
        
        let html = '<table class="data-table"><thead><tr><th>Name</th><th>Telefonnummer</th><th>Ländercode</th><th>Nachrichten</th><th>Konversationen</th><th>Letzte Nachricht</th><th>Erstellt</th><th>Aktionen</th></tr></thead><tbody>';
        contactsWithStats.forEach(contact => {
            const name = contact.full_name || '-';
            const handle = contact.handle || '';
            // Formatiere Telefonnummer
            const formattedPhone = (() => {
                if (!handle || handle === '-') return '-';
                const phoneMatch = handle.match(/^(\+?\d{1,4})(.*)$/);
                if (phoneMatch) {
                    const countryCode = phoneMatch[1];
                    const number = phoneMatch[2].trim().replace(/\s+/g, ' ');
                    return countryCode + ' ' + number;
                }
                return handle;
            })();
            // Extrahiere Ländercode
            const countryCode = (() => {
                if (contact.country_codes) {
                    if (Array.isArray(contact.country_codes) && contact.country_codes.length > 0) {
                        return contact.country_codes.join(', ');
                    }
                    return contact.country_codes || '-';
                }
                // Versuche aus handle zu extrahieren
                if (handle && handle.startsWith('+')) {
                    const countryCodeMap = {
                        '+49': 'DE',
                        '+43': 'AT',
                        '+41': 'CH',
                        '+1': 'US',
                        '+44': 'GB'
                    };
                    for (const [code, country] of Object.entries(countryCodeMap)) {
                        if (handle.startsWith(code)) {
                            return country;
                        }
                    }
                }
                return '-';
            })();
            const created = contact.created_at ? new Date(contact.created_at).toLocaleDateString('de-DE') : '-';
            const contactId = contact.id || '';
            const lastMsg = contact.lastMessage ? 
                `<div style="font-size: 0.85rem; color: #6c757d;" title="${escapeHtml(contact.lastMessage.text)}">
                    ${escapeHtml(contact.lastMessage.text)}${contact.lastMessage.text.length > 50 ? '...' : ''}
                    <br><small>${new Date(contact.lastMessage.date).toLocaleString('de-DE')}</small>
                </div>` : '-';
            
            html += `<tr>
                <td>${escapeHtml(name)}</td>
                <td>${escapeHtml(formattedPhone)}</td>
                <td>${escapeHtml(countryCode)}</td>
                <td><span class="badge badge-info">${contact.messagesCount}</span></td>
                <td><span class="badge badge-info">${contact.conversationsCount}</span></td>
                <td>${lastMsg}</td>
                <td>${created}</td>
                <td>
                    <button class="btn btn-sm" onclick="editContact('${contactId}')" title="Bearbeiten"><i class="fas fa-edit"></i></button>
                    <button class="btn btn-sm btn-danger" onclick="deleteContact('${contactId}')" title="Löschen"><i class="fas fa-trash"></i></button>
                    <button class="btn btn-sm" onclick="showContactDetails('${contactId}')" title="Details"><i class="fas fa-eye"></i></button>
                    ${handle ? `<a href="appointments.php?phone=${encodeURIComponent(handle)}" class="btn btn-sm btn-success" title="Termine anzeigen" target="_blank"><i class="fas fa-calendar"></i></a>` : ''}
                </td>
            </tr>`;
        });
        html += '</tbody></table>';
        listDiv.innerHTML = html;
    }
}

// Change Contacts Page
function changeContactsPage(direction) {
    const searchTerm = (document.getElementById('contacts-search')?.value || '').toLowerCase();
    
    let filtered = allContacts.filter(contact => {
        if (!searchTerm) return true;
        const name = (contact.full_name || '').toLowerCase();
        const handle = (contact.handle || '').toLowerCase();
        return name.includes(searchTerm) || handle.includes(searchTerm);
    });
    
    const totalPages = Math.ceil(filtered.length / contactsPageSize);
    const newPage = contactsCurrentPage + direction;
    
    if (newPage >= 1 && newPage <= totalPages) {
        contactsCurrentPage = newPage;
        filterContacts();
    }
}

// Edit Contact Modal
function editContactModal(contactId) {
    const contact = allContacts.find(c => c.id === contactId);
    if (!contact) {
        alert('Kontakt nicht gefunden');
        return;
    }
    
    let html = `
        <div class="modal-overlay" onclick="if(event.target === this) this.remove()">
            <div class="modal-content" style="max-width: 600px;">
                <div class="modal-header">
                    <h3><i class="fas fa-edit"></i> Kontakt bearbeiten</h3>
                    <button class="modal-close" onclick="this.closest('.modal-overlay').remove()">&times;</button>
                </div>
                <div class="modal-body">
                    <form id="edit-contact-form">
                        <input type="hidden" name="contact_id" value="${contactId}">
                        <div class="form-group">
                            <label for="edit-contact-name">Name</label>
                            <input type="text" id="edit-contact-name" name="full_name" value="${escapeHtml(contact.full_name || '')}" required>
                        </div>
                        <div class="form-group">
                            <label for="edit-contact-handle">Telefonnummer</label>
                            <input type="tel" id="edit-contact-handle" name="handle" value="${escapeHtml(contact.handle || '')}" required>
                        </div>
                        <div class="form-group">
                            <label for="edit-contact-country">Ländercode</label>
                            <select id="edit-contact-country" name="country_codes">
                                <option value="DE" ${(contact.country_codes && (Array.isArray(contact.country_codes) ? contact.country_codes.includes('DE') : contact.country_codes === 'DE')) ? 'selected' : ''}>Deutschland (DE)</option>
                                <option value="AT" ${(contact.country_codes && (Array.isArray(contact.country_codes) ? contact.country_codes.includes('AT') : contact.country_codes === 'AT')) ? 'selected' : ''}>Österreich (AT)</option>
                                <option value="CH" ${(contact.country_codes && (Array.isArray(contact.country_codes) ? contact.country_codes.includes('CH') : contact.country_codes === 'CH')) ? 'selected' : ''}>Schweiz (CH)</option>
                            </select>
                        </div>
                        <div id="edit-contact-result"></div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button class="btn btn-primary" onclick="saveContactEdit('${contactId}')">
                        <i class="fas fa-save"></i> Speichern
                    </button>
                    <button class="btn btn-secondary" onclick="this.closest('.modal-overlay').remove()">
                        <i class="fas fa-times"></i> Abbrechen
                    </button>
                </div>
            </div>
        </div>
    `;
    
    const modal = document.createElement('div');
    modal.innerHTML = html;
    document.body.appendChild(modal);
}

// Save Contact Edit
function saveContactEdit(contactId) {
    const form = document.getElementById('edit-contact-form');
    if (!form) return;
    
    const formData = new FormData(form);
    formData.append('ajax_action', 'update_contact');
    
    const resultDiv = document.getElementById('edit-contact-result');
    resultDiv.innerHTML = '<div class="loading"><i class="fas fa-spinner"></i> Speichere...</div>';
    
    fetch('hellomateo.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            resultDiv.innerHTML = '<div class="alert alert-success"><i class="fas fa-check-circle"></i> Kontakt erfolgreich aktualisiert!</div>';
            setTimeout(() => {
                document.querySelector('.modal-overlay').remove();
                loadContacts(contactsCurrentPage);
            }, 1000);
        } else {
            resultDiv.innerHTML = '<div class="alert alert-error"><i class="fas fa-exclamation-circle"></i> Fehler: ' + (data.error || 'Unbekannter Fehler') + '</div>';
        }
    })
    .catch(error => {
        resultDiv.innerHTML = '<div class="alert alert-error"><i class="fas fa-exclamation-circle"></i> Fehler: ' + error.message + '</div>';
    });
}

// Edit Contact (Legacy - ruft jetzt Modal auf)
function editContact(contactId) {
    editContactModal(contactId);
}

// Delete Contact
function deleteContact(contactId) {
    if (!confirm('Möchten Sie diesen Kontakt wirklich löschen?')) return;
    
    const formData = new FormData();
    formData.append('ajax_action', 'delete_contact');
    formData.append('contact_id', contactId);
    
    fetch('hellomateo.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Cache invalidieren nach erfolgreichem Löschen
            invalidateCache('contacts');
            alert('Kontakt erfolgreich gelöscht');
            loadContacts(contactsCurrentPage, false); // false = Cache verwenden (wird gerade invalidiert, also neu laden)
        } else {
            alert('Fehler: ' + (data.error || 'Unbekannter Fehler'));
        }
    })
    .catch(error => {
        alert('Fehler: ' + error.message);
    });
}

// Show Contact Details - Erweitert mit Verknüpfungen
function showContactDetails(contactId) {
    // Lade vollständige Details vom Server
    const formData = new FormData();
    formData.append('ajax_action', 'get_contact_details');
    formData.append('contact_id', contactId);
    
    fetch('hellomateo.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (!data.success || !data.data) {
            alert('Kontakt nicht gefunden');
            return;
        }
        
        const contact = data.data;
        const linked = data.linked || {};
        
        let html = `
            <div class="modal-overlay" onclick="if(event.target === this) this.remove()">
                <div class="modal-content" style="max-width: 900px;">
                    <div class="modal-header">
                        <h3><i class="fas fa-user"></i> Kontakt-Details</h3>
                        <button class="modal-close" onclick="this.closest('.modal-overlay').remove()">&times;</button>
                    </div>
                    <div class="modal-body">
                        <div class="detail-section">
                            <div class="detail-section-title"><i class="fas fa-info-circle"></i> Kontaktinformationen</div>
                            <div class="detail-grid">
                                <div class="detail-item">
                                    <div class="detail-label">Kontakt-ID</div>
                                    <div class="detail-value">${escapeHtml(contact.id || '-')}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Name</div>
                                    <div class="detail-value">${escapeHtml(contact.full_name || '-')}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Telefonnummer</div>
                                    <div class="detail-value">${(() => {
                                        const handle = contact.handle || '';
                                        if (!handle || handle === '-') return '-';
                                        // Formatiere Telefonnummer für bessere Lesbarkeit
                                        const phoneMatch = handle.match(/^(\+?\d{1,4})(.*)$/);
                                        if (phoneMatch) {
                                            const countryCode = phoneMatch[1];
                                            const number = phoneMatch[2].trim().replace(/\s+/g, ' ');
                                            return escapeHtml(countryCode + ' ' + number);
                                        }
                                        return escapeHtml(handle);
                                    })()}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Ländercodes</div>
                                    <div class="detail-value">${(() => {
                                        if (contact.country_codes) {
                                            if (Array.isArray(contact.country_codes)) {
                                                return contact.country_codes.length > 0 ? contact.country_codes.join(', ') : '-';
                                            }
                                            return contact.country_codes;
                                        }
                                        // Versuche Ländercode aus handle zu extrahieren
                                        const handle = contact.handle || '';
                                        if (handle && handle.startsWith('+')) {
                                            const countryCodeMap = {
                                                '+49': 'DE',
                                                '+43': 'AT',
                                                '+41': 'CH',
                                                '+1': 'US',
                                                '+44': 'GB'
                                            };
                                            for (const [code, country] of Object.entries(countryCodeMap)) {
                                                if (handle.startsWith(code)) {
                                                    return country;
                                                }
                                            }
                                        }
                                        return '-';
                                    })()}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Erstellt am</div>
                                    <div class="detail-value">${contact.created_at ? new Date(contact.created_at).toLocaleString('de-DE') : '-'}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Aktualisiert am</div>
                                    <div class="detail-value">${contact.updated_at ? new Date(contact.updated_at).toLocaleString('de-DE') : '-'}</div>
                                </div>
                            </div>
                        </div>
                        
                        ${linked.messages && linked.messages.length > 0 ? `
                        <div class="detail-section">
                            <div class="detail-section-title"><i class="fas fa-paper-plane"></i> Nachrichten (${linked.messages.length})</div>
                            <div style="max-height: 200px; overflow-y: auto;">
                                ${linked.messages.slice(0, 10).map(msg => `
                                    <div style="padding: 0.75rem; margin-bottom: 0.5rem; background: #f8f9fa; border-radius: 6px; border-left: 3px solid #0056b3;">
                                        <div style="font-weight: 600; margin-bottom: 0.25rem;">${escapeHtml(msg.text || msg.blurb || '-').substring(0, 100)}${(msg.text || msg.blurb || '').length > 100 ? '...' : ''}</div>
                                        <div style="font-size: 0.85rem; color: #6c757d;">
                                            ${msg.created_at ? new Date(msg.created_at).toLocaleString('de-DE') : '-'} | 
                                            ${getMessageStatusBadge(msg.status || 'unknown')}
                                        </div>
                                    </div>
                                `).join('')}
                                ${linked.messages.length > 10 ? `<div style="text-align: center; padding: 0.5rem; color: #6c757d;">... und ${linked.messages.length - 10} weitere</div>` : ''}
                            </div>
                        </div>
                        ` : ''}
                        
                        ${linked.appointments && linked.appointments.length > 0 ? `
                        <div class="detail-section">
                            <div class="detail-section-title"><i class="fas fa-calendar"></i> Termine (${linked.appointments.length})</div>
                            <div style="max-height: 200px; overflow-y: auto;">
                                ${linked.appointments.map(apt => `
                                    <div style="padding: 0.75rem; margin-bottom: 0.5rem; background: #f8f9fa; border-radius: 6px; border-left: 3px solid #28a745; display: flex; justify-content: space-between; align-items: center;">
                                        <div>
                                            <div style="font-weight: 600; margin-bottom: 0.25rem;">${escapeHtml(apt.customer_name || '-')}</div>
                                            <div style="font-size: 0.85rem; color: #6c757d;">
                                                ${apt.slot_date ? new Date(apt.slot_date).toLocaleDateString('de-DE') : '-'} 
                                                ${apt.slot_time ? apt.slot_time : ''} | 
                                                ${escapeHtml(apt.license_plate || '-')}
                                            </div>
                                        </div>
                                        <div style="display: flex; gap: 0.5rem;">
                                            <button class="btn btn-sm btn-success" onclick="sendMessageFromAppointment(${apt.id}, 'confirmation')" title="Bestätigung senden">
                                                <i class="fas fa-check"></i>
                                            </button>
                                            <button class="btn btn-sm btn-info" onclick="sendMessageFromAppointment(${apt.id}, 'reminder')" title="Erinnerung senden">
                                                <i class="fas fa-bell"></i>
                                            </button>
                                            <a href="appointments.php?id=${apt.id}" class="btn btn-sm" target="_blank" title="Termin öffnen">
                                                <i class="fas fa-external-link-alt"></i>
                                            </a>
                                        </div>
                                    </div>
                                `).join('')}
                            </div>
                        </div>
                        ` : ''}
                        
                        ${linked.storage && linked.storage.length > 0 ? `
                        <div class="detail-section">
                            <div class="detail-section-title"><i class="fas fa-warehouse"></i> Reifenlager (${linked.storage.length})</div>
                            <div style="max-height: 200px; overflow-y: auto;">
                                ${linked.storage.map(st => `
                                    <div style="padding: 0.75rem; margin-bottom: 0.5rem; background: #f8f9fa; border-radius: 6px; border-left: 3px solid #ffc107;">
                                        <div style="font-weight: 600; margin-bottom: 0.25rem;">${escapeHtml(st.storage_number || '-')}</div>
                                        <div style="font-size: 0.85rem; color: #6c757d;">
                                            ${st.created_at ? new Date(st.created_at).toLocaleDateString('de-DE') : '-'}
                                        </div>
                                    </div>
                                `).join('')}
                            </div>
                        </div>
                        ` : ''}
                        
                        ${contact.metadata || contact.raw_data ? `
                        <div class="detail-section">
                            <div class="detail-section-title"><i class="fas fa-code"></i> Technische Details</div>
                            <div class="detail-item" style="grid-column: 1 / -1;">
                                <div class="detail-label">Vollständige Daten</div>
                                <div class="detail-value" style="font-family: monospace; font-size: 0.85rem; background: #f8f9fa; padding: 1rem; border-radius: 6px; overflow-x: auto;">
                                    <pre>${JSON.stringify(contact, null, 2)}</pre>
                                </div>
                            </div>
                        </div>
                        ` : ''}
                    </div>
                    <div class="modal-footer">
                        <button class="btn btn-primary" onclick="editContactModal('${contactId}'); this.closest('.modal-overlay').remove();">
                            <i class="fas fa-edit"></i> Bearbeiten
                        </button>
                        <button class="btn btn-success" onclick="sendMessageToContact('${contact.handle || ''}'); this.closest('.modal-overlay').remove();">
                            <i class="fas fa-paper-plane"></i> Nachricht senden
                        </button>
                        <button class="btn btn-secondary" onclick="this.closest('.modal-overlay').remove()">
                            <i class="fas fa-times"></i> Schließen
                        </button>
                    </div>
                </div>
            </div>
        `;
        
        const modal = document.createElement('div');
        modal.innerHTML = html;
        document.body.appendChild(modal);
    })
    .catch(error => {
        alert('Fehler beim Laden der Kontaktdetails: ' + error.message);
    });
}

// Export Messages
function exportMessages() {
    const searchTerm = (document.getElementById('messages-search')?.value || '').toLowerCase();
    const statusFilter = document.getElementById('messages-status-filter')?.value || '';
    const dateFrom = document.getElementById('messages-date-from')?.value || '';
    const dateTo = document.getElementById('messages-date-to')?.value || '';
    
    let filtered = allMessages.filter(msg => {
        const matchesSearch = !searchTerm || 
            (msg.text && msg.text.toLowerCase().includes(searchTerm)) ||
            (msg.blurb && msg.blurb.toLowerCase().includes(searchTerm)) ||
            (msg.recipients && msg.recipients.some(r => r.handle && r.handle.toLowerCase().includes(searchTerm)));
        const matchesStatus = !statusFilter || (msg.status && msg.status.toLowerCase() === statusFilter.toLowerCase());
        
        let matchesDate = true;
        if (dateFrom || dateTo) {
            const msgDate = msg.created_at ? new Date(msg.created_at).toISOString().split('T')[0] : '';
            if (dateFrom && msgDate < dateFrom) matchesDate = false;
            if (dateTo && msgDate > dateTo) matchesDate = false;
        }
        
        return matchesSearch && matchesStatus && matchesDate;
    });
    
    // Erstelle CSV
    let csv = 'Datum,Zeit,Von,An,Nachricht,Status,Template-ID\n';
    filtered.forEach(msg => {
        const date = msg.created_at ? new Date(msg.created_at) : null;
        const dateStr = date ? date.toLocaleDateString('de-DE') : '-';
        const timeStr = date ? date.toLocaleTimeString('de-DE') : '-';
        const from = msg.author_id || msg.from || '-';
        const to = msg.recipients ? msg.recipients.map(r => r.handle || r).join('; ') : (msg.to || '-');
        const text = (msg.text || msg.blurb || '-').replace(/"/g, '""');
        const status = msg.status || '-';
        const templateId = msg.template_id || '-';
        csv += `"${dateStr}","${timeStr}","${from}","${to}","${text}","${status}","${templateId}"\n`;
    });
    
    // Download
    const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    link.setAttribute('href', url);
    link.setAttribute('download', `whatsapp-nachrichten-${new Date().toISOString().split('T')[0]}.csv`);
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

// Helper: Send Message to Contact
// Send Message from Appointment - Verwendet das neue Template-Konfigurationssystem
function sendMessageFromAppointment(appointmentId, action = 'confirmation') {
    if (!appointmentId) {
        alert('Termin-ID fehlt');
        return;
    }
    
    // Hole Autohaus-ID aus dem aktuellen Kontext
    const autohausId = <?php echo json_encode($active_autohaus_id ?? null); ?>;
    if (!autohausId) {
        alert('Autohaus-ID fehlt. Bitte wählen Sie ein Autohaus aus.');
        return;
    }
    
    const actionLabels = {
        'confirmation': 'Bestätigung',
        'reminder': 'Erinnerung',
        'approved': 'Genehmigung',
        'rejected': 'Ablehnung',
        'cancellation': 'Stornierung',
        'rescheduled': 'Verschoben'
    };
    
    const actionLabel = actionLabels[action] || action;
    
    if (!confirm(`Möchten Sie wirklich eine ${actionLabel} für diesen Termin senden?`)) {
        return;
    }
    
    const formData = new FormData();
    formData.append('ajax_action', 'send_message_from_appointment');
    formData.append('appointment_id', appointmentId);
    formData.append('action', action);
    formData.append('autohaus_id', autohausId);
    
    // Zeige Ladeanzeige
    const loadingDiv = document.createElement('div');
    loadingDiv.className = 'loading-overlay';
    loadingDiv.innerHTML = '<div class="loading"><i class="fas fa-spinner fa-spin"></i> Sende Nachricht...</div>';
    document.body.appendChild(loadingDiv);
    
    fetch('hellomateo.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        document.body.removeChild(loadingDiv);
        
        if (data.success) {
            alert(`✅ ${actionLabel} erfolgreich gesendet!`);
            // Cache invalidieren
            invalidateCache('messages');
            invalidateCache('conversations');
        } else {
            alert(`❌ Fehler beim Senden: ${data.error || 'Unbekannter Fehler'}`);
        }
    })
    .catch(error => {
        document.body.removeChild(loadingDiv);
        alert(`❌ Fehler: ${error.message}`);
    });
}

function sendMessageToContact(phone) {
    if (!phone) {
        alert('Keine Telefonnummer verfügbar');
        return;
    }
    
    // Wechsle zum Nachrichten-Tab und fülle Empfänger
    document.querySelectorAll('.hellomateo-tab').forEach(t => t.classList.remove('active'));
    document.querySelectorAll('.hellomateo-tab-content').forEach(c => c.classList.remove('active'));
    
    const messagesTab = document.querySelector('[data-tab="messages"]');
    const messagesContent = document.getElementById('tab-messages');
    messagesTab.classList.add('active');
    messagesContent.classList.add('active');
    
    // Fülle Empfänger-Feld
    const toInput = document.getElementById('message-to');
    if (toInput) {
        toInput.value = phone;
        toInput.focus();
    }
    
    // Scroll zum Formular
    messagesContent.scrollIntoView({ behavior: 'smooth' });
}

// Templates Pagination
let templatesCurrentPage = 1;
let templatesPageSize = 20;
let allTemplates = [];

// Load Templates
function loadTemplates(page = 1, useCache = true) {
    templatesCurrentPage = page;
    const listDiv = document.getElementById('templates-list');
    
    // Prüfe Cache
    if (useCache && isCacheValid('templates') && API_CACHE.templates.data) {
        console.log('✅ Verwende gecachte Templates');
        allTemplates = API_CACHE.templates.data;
        renderTemplates();
        
        // Update template select in messages tab
        const templateSelect = document.getElementById('message-template');
        if (templateSelect) {
            templateSelect.innerHTML = '<option value="">Kein Template</option>';
            allTemplates.forEach(template => {
                const option = document.createElement('option');
                option.value = template.id || '';
                option.textContent = template.name || template.id || '';
                templateSelect.appendChild(option);
            });
        }
        return;
    }
    
    listDiv.innerHTML = '<div class="loading"><i class="fas fa-spinner"></i> Lade Templates...</div>';
    
    const formData = new FormData();
    formData.append('ajax_action', 'load_templates');
    formData.append('limit', 100); // Lade mehr für Filterung
    formData.append('offset', 0);
    
    fetch('hellomateo.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success && data.data) {
            allTemplates = Array.isArray(data.data) ? data.data : [];
            // Cache speichern
            API_CACHE.templates.data = allTemplates;
            API_CACHE.templates.timestamp = Date.now();
            console.log(`✅ Templates geladen: ${allTemplates.length} (gecacht)`);
            renderTemplates();
            
            // Update template select in messages tab
            const templateSelect = document.getElementById('message-template');
            if (templateSelect) {
                templateSelect.innerHTML = '<option value="">Kein Template</option>';
                allTemplates.forEach(template => {
                    const option = document.createElement('option');
                    option.value = template.id || '';
                    option.textContent = template.name || template.id || '';
                    templateSelect.appendChild(option);
                });
            }
        } else {
            listDiv.innerHTML = '<div class="alert alert-error"><i class="fas fa-exclamation-circle"></i> Fehler: ' + (data.error || 'Templates konnten nicht geladen werden') + '</div>';
        }
    })
    .catch(error => {
        listDiv.innerHTML = '<div class="alert alert-error"><i class="fas fa-exclamation-circle"></i> Fehler: ' + error.message + '</div>';
    });
}

// Export Templates
function exportTemplates() {
    const searchTerm = (document.getElementById('templates-search')?.value || '').toLowerCase();
    const statusFilter = document.getElementById('templates-status-filter')?.value || '';
    const languageFilter = document.getElementById('templates-language-filter')?.value || '';
    
    let filtered = allTemplates.filter(template => {
        const matchesSearch = !searchTerm || 
            (template.name && template.name.toLowerCase().includes(searchTerm)) ||
            (template.id && template.id.toLowerCase().includes(searchTerm));
        const matchesStatus = !statusFilter || (template.status && template.status.toLowerCase() === statusFilter.toLowerCase());
        const matchesLanguage = !languageFilter || (template.language && template.language.toLowerCase() === languageFilter.toLowerCase());
        return matchesSearch && matchesStatus && matchesLanguage;
    });
    
    // Erstelle CSV
    let csv = 'ID,Name,Status,Sprache,Kategorie,Erstellt am,Verwendungen\n';
    filtered.forEach(template => {
        const id = (template.id || '').replace(/"/g, '""');
        const name = (template.name || '').replace(/"/g, '""');
        const status = (template.status || '').replace(/"/g, '""');
        const language = (template.language || '').replace(/"/g, '""');
        const category = (template.category || '').replace(/"/g, '""');
        const created = template.created_at ? new Date(template.created_at).toLocaleDateString('de-DE') : '-';
        const usage = template.usage_count || 0;
        csv += `"${id}","${name}","${status}","${language}","${category}","${created}","${usage}"\n`;
    });
    
    // Download
    const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    link.setAttribute('href', url);
    link.setAttribute('download', `whatsapp-templates-${new Date().toISOString().split('T')[0]}.csv`);
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

// Render Templates with Filters - Erweitert
function renderTemplates() {
    const searchTerm = (document.getElementById('templates-search')?.value || '').toLowerCase();
    const statusFilter = document.getElementById('templates-status-filter')?.value || '';
    const languageFilter = document.getElementById('templates-language-filter')?.value || '';
    
    let filtered = allTemplates.filter(template => {
        const matchesSearch = !searchTerm || 
            (template.name && template.name.toLowerCase().includes(searchTerm)) ||
            (template.id && template.id.toLowerCase().includes(searchTerm));
        const matchesStatus = !statusFilter || (template.status && template.status.toLowerCase() === statusFilter.toLowerCase());
        const matchesLanguage = !languageFilter || (template.language && template.language.toLowerCase() === languageFilter.toLowerCase());
        return matchesSearch && matchesStatus && matchesLanguage;
    });
    
    // Pagination
    const totalPages = Math.ceil(filtered.length / templatesPageSize);
    const startIndex = (templatesCurrentPage - 1) * templatesPageSize;
    const endIndex = startIndex + templatesPageSize;
    const paginatedTemplates = filtered.slice(startIndex, endIndex);
    
    const listDiv = document.getElementById('templates-list');
    if (paginatedTemplates.length === 0) {
        listDiv.innerHTML = '<div class="empty-state"><i class="fas fa-file-alt"></i><p>Keine Templates gefunden</p></div>';
    } else {
        // Berechne Verwendungsstatistik für jeden Template
        const templatesWithStats = paginatedTemplates.map(template => {
            const usageCount = allMessages.filter(msg => (msg.template_id || '') === (template.id || '')).length;
            return { ...template, usage_count: usageCount };
        });
        
        let html = '<table class="data-table"><thead><tr><th>Name</th><th>Status</th><th>Sprache</th><th>Kategorie</th><th>Verwendungen</th><th>Erstellt</th><th>Aktionen</th></tr></thead><tbody>';
        templatesWithStats.forEach(template => {
            const name = template.name || '-';
            const status = template.status || '-';
            const language = template.language || '-';
            const category = template.category || '-';
            const templateId = template.id || '';
            const statusBadge = getTemplateStatusBadge(status);
            const usageCount = template.usage_count || 0;
            const created = template.created_at ? new Date(template.created_at).toLocaleDateString('de-DE') : '-';
            html += `<tr>
                <td>${escapeHtml(name)}</td>
                <td>${statusBadge}</td>
                <td>${escapeHtml(language)}</td>
                <td>${escapeHtml(category)}</td>
                <td><span class="badge badge-info">${usageCount}</span></td>
                <td>${created}</td>
                <td>
                    <button class="btn btn-sm" onclick="showTemplateDetails('${templateId}')" title="Details"><i class="fas fa-eye"></i></button>
                    <button class="btn btn-sm btn-success" onclick="useTemplate('${templateId}')" title="Verwenden"><i class="fas fa-check"></i></button>
                    <button class="btn btn-sm btn-warning" onclick="testTemplate('${templateId}')" title="Test senden"><i class="fas fa-vial"></i></button>
                </td>
            </tr>`;
        });
        html += '</tbody></table>';
        listDiv.innerHTML = html;
    }
}

// Helper: Template Status Badge
function getTemplateStatusBadge(status) {
    const statusLower = status.toLowerCase();
    const badges = {
        'approved': '<span class="badge badge-template-approved"><i class="fas fa-check-circle"></i> Genehmigt</span>',
        'pending': '<span class="badge badge-template-pending"><i class="fas fa-clock"></i> Ausstehend</span>',
        'rejected': '<span class="badge badge-template-rejected"><i class="fas fa-times-circle"></i> Abgelehnt</span>',
        'draft': '<span class="badge badge-template-draft"><i class="fas fa-edit"></i> Entwurf</span>'
    };
    return badges[statusLower] || `<span class="badge badge-info">${escapeHtml(status)}</span>`;
}

// Show Template Details
function showTemplateDetails(templateId) {
    const formData = new FormData();
    formData.append('ajax_action', 'get_template_details');
    formData.append('template_id', templateId);
    
    fetch('hellomateo.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (!data.success || !data.data) {
            alert('Template nicht gefunden');
            return;
        }
        
        const template = data.data;
        const statusBadge = getTemplateStatusBadge(template.status || 'unknown');
        
        let html = `
            <div class="modal-overlay" onclick="if(event.target === this) this.remove()">
                <div class="modal-content" style="max-width: 900px;">
                    <div class="modal-header">
                        <h3><i class="fas fa-file-alt"></i> Template-Details</h3>
                        <button class="modal-close" onclick="this.closest('.modal-overlay').remove()">&times;</button>
                    </div>
                    <div class="modal-body">
                        <div class="detail-section">
                            <div class="detail-section-title"><i class="fas fa-info-circle"></i> Grundinformationen</div>
                            <div class="detail-grid">
                                <div class="detail-item">
                                    <div class="detail-label">Template-ID</div>
                                    <div class="detail-value">${escapeHtml(template.id || '-')}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Name</div>
                                    <div class="detail-value">${escapeHtml(template.name || '-')}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Status</div>
                                    <div class="detail-value">${statusBadge}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Sprache</div>
                                    <div class="detail-value">${escapeHtml(template.language || '-')}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Kategorie</div>
                                    <div class="detail-value">${escapeHtml(template.category || '-')}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Verwendungen</div>
                                    <div class="detail-value">${template.usage_count || 0}</div>
                                </div>
                            </div>
                        </div>
                        
                        ${template.components || template.body ? `
                        <div class="detail-section">
                            <div class="detail-section-title"><i class="fas fa-envelope"></i> Template-Inhalt</div>
                            ${template.components ? `
                            <div class="detail-item" style="grid-column: 1 / -1;">
                                <div class="detail-label">Komponenten</div>
                                <div class="detail-value" style="font-family: monospace; font-size: 0.85rem; background: #f8f9fa; padding: 1rem; border-radius: 6px; overflow-x: auto;">
                                    <pre>${JSON.stringify(template.components, null, 2)}</pre>
                                </div>
                            </div>
                            ` : ''}
                            ${template.body ? `
                            <div class="detail-item" style="grid-column: 1 / -1;">
                                <div class="detail-label">Nachricht</div>
                                <div class="detail-value" style="white-space: pre-wrap; background: white; padding: 1rem; border-radius: 6px; border: 1px solid #dee2e6;">${escapeHtml(template.body)}</div>
                            </div>
                            ` : ''}
                        </div>
                        ` : ''}
                    </div>
                    <div class="modal-footer">
                        <button class="btn btn-success" onclick="useTemplate('${templateId}'); this.closest('.modal-overlay').remove();">
                            <i class="fas fa-check"></i> Verwenden
                        </button>
                        <button class="btn btn-secondary" onclick="this.closest('.modal-overlay').remove()">
                            <i class="fas fa-times"></i> Schließen
                        </button>
                    </div>
                </div>
            </div>
        `;
        
        const modal = document.createElement('div');
        modal.innerHTML = html;
        document.body.appendChild(modal);
    })
    .catch(error => {
        alert('Fehler beim Laden der Templatedetails: ' + error.message);
    });
}

// Test Template
function testTemplate(templateId) {
    const template = allTemplates.find(t => t.id === templateId);
    if (!template) {
        alert('Template nicht gefunden');
        return;
    }
    
    const testPhone = prompt('Test-Empfänger (Telefonnummer):', '+49123456789');
    if (!testPhone) return;
    
    const testMessage = prompt('Test-Nachricht (optional, wird durch Template ersetzt):', '');
    
    const formData = new FormData();
    formData.append('ajax_action', 'send_message');
    formData.append('to', testPhone);
    formData.append('message', testMessage || 'Test-Nachricht');
    formData.append('template_id', templateId);
    
    fetch('hellomateo.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('Test-Nachricht erfolgreich gesendet!');
            loadMessages();
        } else {
            alert('Fehler: ' + (data.error || 'Unbekannter Fehler'));
        }
    })
    .catch(error => {
        alert('Fehler: ' + error.message);
    });
}

// Use Template
function useTemplate(templateId) {
    const template = allTemplates.find(t => t.id === templateId);
    if (!template) {
        alert('Template nicht gefunden');
        return;
    }
    
    // Wechsle zum Nachrichten-Tab
    document.querySelectorAll('.hellomateo-tab').forEach(t => t.classList.remove('active'));
    document.querySelectorAll('.hellomateo-tab-content').forEach(c => c.classList.remove('active'));
    
    const messagesTab = document.querySelector('[data-tab="messages"]');
    const messagesContent = document.getElementById('tab-messages');
    messagesTab.classList.add('active');
    messagesContent.classList.add('active');
    
    // Setze Template
    const templateSelect = document.getElementById('message-template');
    if (templateSelect) {
        templateSelect.value = templateId;
        templateSelect.dispatchEvent(new Event('change'));
    }
    
    // Scroll zum Formular
    messagesContent.scrollIntoView({ behavior: 'smooth' });
}

// Conversations Pagination
let conversationsCurrentPage = 1;
let conversationsPageSize = 20;
let allConversations = [];

// ============================================
// NEUE KONVERSATIONEN-LOGIK - REKURSIVES LADEN
// ============================================

// Load Conversations - Rekursiv ALLE Konversationen laden (mit Cache)
function loadConversations(page = 1, useCache = true) {
    conversationsCurrentPage = page;
    const listDiv = document.getElementById('conversations-list');
    
    // Prüfe Cache
    if (useCache && isCacheValid('conversations') && API_CACHE.conversations.data) {
        console.log('✅ Verwende gecachte Konversationen');
        window.allConversations = API_CACHE.conversations.data;
        renderConversations();
        return;
    }
    
    listDiv.innerHTML = '<div class="loading"><i class="fas fa-spinner"></i> Lade alle Konversationen...</div>';
    
    // Lade ALLE Konversationen rekursiv
    loadAllConversationsRecursive(0, []);
}

// Rekursive Funktion zum Laden ALLER Konversationen
function loadAllConversationsRecursive(offset, accumulatedConversations) {
    const formData = new FormData();
    formData.append('ajax_action', 'load_conversations');
    formData.append('limit', 100); // Lade in Batches von 100
    formData.append('offset', offset);
    
    console.log(`Lade Konversationen: offset=${offset}, bereits geladen: ${accumulatedConversations.length}`);
    
    fetch('hellomateo.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        console.log(`API Response: success=${data.success}, data length=${data.data ? (Array.isArray(data.data) ? data.data.length : 'not array') : 'no data'}`);
        
        if (data.success && data.data) {
            const newConversations = Array.isArray(data.data) ? data.data : [];
            const allConversations = [...accumulatedConversations, ...newConversations];
            
            console.log(`Neue Konversationen: ${newConversations.length}, Gesamt: ${allConversations.length}`);
            
            // Wenn wir 0 Konversationen erhalten haben, sind wir fertig
            if (newConversations.length === 0) {
                // Keine weiteren Konversationen - fertig
                window.allConversations = allConversations;
                // Cache speichern
                API_CACHE.conversations.data = allConversations;
                API_CACHE.conversations.timestamp = Date.now();
                console.log(`✅ Alle Konversationen geladen: ${allConversations.length} Konversationen (gecacht)`);
                renderConversations();
            } else {
                // Wir haben Konversationen erhalten - lade IMMER den nächsten Batch
                // Nur wenn wir 0 Konversationen erhalten, wissen wir, dass wir fertig sind
                console.log(`Lade nächsten Batch (offset=${offset + newConversations.length})...`);
                loadAllConversationsRecursive(offset + newConversations.length, allConversations);
            }
        } else {
            // Fehler oder keine weiteren Konversationen
            if (accumulatedConversations.length > 0) {
                window.allConversations = accumulatedConversations;
                // Cache speichern
                API_CACHE.conversations.data = accumulatedConversations;
                API_CACHE.conversations.timestamp = Date.now();
                console.log(`⚠️ Konversationen geladen (mit Fehler): ${accumulatedConversations.length} Konversationen (gecacht)`);
                renderConversations();
            } else {
                const listDiv = document.getElementById('conversations-list');
                if (listDiv) {
                    listDiv.innerHTML = '<div class="alert alert-error"><i class="fas fa-exclamation-circle"></i> Fehler: ' + (data.error || 'Konversationen konnten nicht geladen werden') + '</div>';
                }
            }
        }
    })
    .catch(error => {
        console.error('Error loading conversations:', error);
        // Verwende bereits geladene Konversationen falls vorhanden
        if (accumulatedConversations.length > 0) {
            window.allConversations = accumulatedConversations;
            // Cache speichern
            API_CACHE.conversations.data = accumulatedConversations;
            API_CACHE.conversations.timestamp = Date.now();
            console.log(`⚠️ Konversationen geladen (mit Fehler): ${accumulatedConversations.length} Konversationen (gecacht)`);
            renderConversations();
        } else {
            const listDiv = document.getElementById('conversations-list');
            if (listDiv) {
                listDiv.innerHTML = '<div class="alert alert-error"><i class="fas fa-exclamation-circle"></i> Fehler beim Laden der Konversationen: ' + error.message + '</div>';
            }
        }
    });
}

// Render Conversations with Filters
function renderConversations() {
    const searchTerm = (document.getElementById('conversations-search')?.value || '').toLowerCase();
    const statusFilter = document.getElementById('conversations-status-filter')?.value || '';
    
    // Verwende window.allConversations falls verfügbar, sonst allConversations
    const conversationsToFilter = window.allConversations || allConversations || [];
    if (!conversationsToFilter || conversationsToFilter.length === 0) {
        const listDiv = document.getElementById('conversations-list');
        if (listDiv) {
            listDiv.innerHTML = '<div class="empty-state"><i class="fas fa-comments"></i><p>Keine Konversationen gefunden</p></div>';
        }
        return;
    }
    
    let filtered = conversationsToFilter.filter(conv => {
        const matchesSearch = !searchTerm || 
            (conv.subject && conv.subject.toLowerCase().includes(searchTerm)) ||
            (conv.contact_id && conv.contact_id.toLowerCase().includes(searchTerm));
        const matchesStatus = !statusFilter || (conv.status && conv.status.toLowerCase() === statusFilter.toLowerCase());
        return matchesSearch && matchesStatus;
    });
    
    // Sortiere nach letzter Nachricht
    filtered.sort((a, b) => {
        const dateA = a.latest_message_created_at ? new Date(a.latest_message_created_at) : new Date(0);
        const dateB = b.latest_message_created_at ? new Date(b.latest_message_created_at) : new Date(0);
        return dateB - dateA;
    });
    
    // Pagination
    const totalPages = Math.ceil(filtered.length / conversationsPageSize);
    const startIndex = (conversationsCurrentPage - 1) * conversationsPageSize;
    const endIndex = startIndex + conversationsPageSize;
    const paginatedConversations = filtered.slice(startIndex, endIndex);
    
    // Update Pagination Controls
    document.getElementById('conversations-count').textContent = `${filtered.length} Konversationen`;
    document.getElementById('conversations-page-info').textContent = `Seite ${conversationsCurrentPage} von ${Math.max(1, totalPages)}`;
    document.getElementById('conversations-prev').disabled = conversationsCurrentPage <= 1;
    document.getElementById('conversations-next').disabled = conversationsCurrentPage >= totalPages;
    
    // Berechne Statistiken für jede Konversation
    const allMessagesForStats = window.allMessages || allMessages || [];
    const conversationsWithStats = paginatedConversations.map(conv => {
        const messagesCount = allMessagesForStats.filter(msg => {
            return (msg.conversation_id || '') === (conv.id || '') ||
                   (msg.contact_id || '') === (conv.contact_id || '');
        }).length;
        return { ...conv, messages_count: messagesCount };
    });
    
    const listDiv = document.getElementById('conversations-list');
    if (paginatedConversations.length === 0) {
        listDiv.innerHTML = '<div class="empty-state"><i class="fas fa-comments"></i><p>Keine Konversationen gefunden</p></div>';
    } else {
        let html = '<table class="data-table"><thead><tr><th>Kontakt</th><th>Betreff</th><th>Status</th><th>Nachrichten</th><th>Letzte Aktivität</th><th>Aktionen</th></tr></thead><tbody>';
        conversationsWithStats.forEach(conv => {
            const contact = conv.contact_id || conv.contact_name || '-';
            const subject = conv.subject || '-';
            const status = conv.status || '-';
            const lastMessage = conv.latest_message_created_at ? new Date(conv.latest_message_created_at).toLocaleString('de-DE') : '-';
            const convId = conv.id || '';
            const statusBadge = getConversationStatusBadge(status);
            const messagesCount = conv.messages_count || 0;
            html += `<tr>
                <td>${escapeHtml(contact)}</td>
                <td>${escapeHtml(subject)}</td>
                <td>${statusBadge}</td>
                <td><span class="badge badge-info">${messagesCount}</span></td>
                <td>${lastMessage}</td>
                <td>
                    <button class="btn btn-sm" onclick="showConversationDetails('${convId}')" title="Details"><i class="fas fa-eye"></i></button>
                    <button class="btn btn-sm btn-success" onclick="replyToConversation('${convId}')" title="Antworten"><i class="fas fa-reply"></i></button>
                    ${status === 'closed' ? `<button class="btn btn-sm btn-warning" onclick="reopenConversation('${convId}')" title="Wiedereröffnen"><i class="fas fa-unlock"></i></button>` : ''}
                    ${status !== 'closed' ? `<button class="btn btn-sm btn-secondary" onclick="closeConversation('${convId}')" title="Schließen"><i class="fas fa-lock"></i></button>` : ''}
                </td>
            </tr>`;
        });
        html += '</tbody></table>';
        listDiv.innerHTML = html;
    }
}

// Change Conversations Page
function changeConversationsPage(direction) {
    const searchTerm = (document.getElementById('conversations-search')?.value || '').toLowerCase();
    const statusFilter = document.getElementById('conversations-status-filter')?.value || '';
    
    let filtered = allConversations.filter(conv => {
        const matchesSearch = !searchTerm || 
            (conv.subject && conv.subject.toLowerCase().includes(searchTerm)) ||
            (conv.contact_id && conv.contact_id.toLowerCase().includes(searchTerm));
        const matchesStatus = !statusFilter || (conv.status && conv.status.toLowerCase() === statusFilter.toLowerCase());
        return matchesSearch && matchesStatus;
    });
    
    const totalPages = Math.ceil(filtered.length / conversationsPageSize);
    const newPage = conversationsCurrentPage + direction;
    
    if (newPage >= 1 && newPage <= totalPages) {
        conversationsCurrentPage = newPage;
        renderConversations();
    }
}

// Close Conversation
function closeConversation(conversationId) {
    if (!confirm('Möchten Sie diese Konversation wirklich schließen?')) return;
    // TODO: Implementiere API-Call zum Schließen
    alert('Funktion wird implementiert...');
    loadConversations();
}

// Reopen Conversation
function reopenConversation(conversationId) {
    if (!confirm('Möchten Sie diese Konversation wiedereröffnen?')) return;
    // TODO: Implementiere API-Call zum Wiedereröffnen
    alert('Funktion wird implementiert...');
    loadConversations();
}

// Helper: Conversation Status Badge
function getConversationStatusBadge(status) {
    const statusLower = status.toLowerCase();
    const badges = {
        'active': '<span class="badge badge-conversation-active"><i class="fas fa-circle"></i> Aktiv</span>',
        'open': '<span class="badge badge-conversation-open"><i class="fas fa-folder-open"></i> Offen</span>',
        'closed': '<span class="badge badge-conversation-closed"><i class="fas fa-folder"></i> Geschlossen</span>',
        'pending': '<span class="badge badge-conversation-pending"><i class="fas fa-clock"></i> Ausstehend</span>'
    };
    return badges[statusLower] || `<span class="badge badge-info">${escapeHtml(status)}</span>`;
}

// Show Conversation Details - NEUE VERSION MIT ALLEN NACHRICHTEN
function showConversationDetails(conversationId) {
    // Zeige Loading-Modal
    const loadingModal = document.createElement('div');
    loadingModal.className = 'modal-overlay';
    loadingModal.innerHTML = `
        <div class="modal-content" style="max-width: 600px;">
            <div class="modal-body" style="text-align: center; padding: 3rem;">
                <i class="fas fa-spinner fa-spin" style="font-size: 2rem; color: #0056b3; margin-bottom: 1rem;"></i>
                <p>Lade Konversations-Details...</p>
            </div>
        </div>
    `;
    document.body.appendChild(loadingModal);
    
    const formData = new FormData();
    formData.append('ajax_action', 'get_conversation_details');
    formData.append('conversation_id', conversationId);
    
    fetch('hellomateo.php', {
        method: 'POST',
        body: formData
    })
    .then(response => {
        // Prüfe ob Response OK ist
        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        // Prüfe Content-Type
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            return response.text().then(text => {
                console.error('Nicht-JSON Antwort:', text);
                throw new Error('Server hat keine JSON-Antwort zurückgegeben: ' + text.substring(0, 200));
            });
        }
        return response.json();
    })
    .then(data => {
        if (loadingModal && loadingModal.parentNode) {
            loadingModal.remove();
        }
        
        if (!data || !data.success || !data.data) {
            const errorMsg = data?.error || 'Konversation nicht gefunden';
            alert('Fehler: ' + errorMsg);
            console.error('Konversations-Details Fehler:', data);
            return;
        }
        
        const conversation = data.data;
        const messages = conversation.messages || [];
        const statusBadge = getConversationStatusBadge(conversation.status || 'unknown');
        
        console.log(`✅ Konversation geladen: ${messages.length} Nachrichten`);
        
        let html = `
            <div class="modal-overlay" onclick="if(event.target === this) this.remove()">
                <div class="modal-content" style="max-width: 900px;">
                    <div class="modal-header">
                        <h3><i class="fas fa-comments"></i> Konversations-Details</h3>
                        <button class="modal-close" onclick="this.closest('.modal-overlay').remove()">&times;</button>
                    </div>
                    <div class="modal-body">
                        <div class="detail-section">
                            <div class="detail-section-title"><i class="fas fa-info-circle"></i> Grundinformationen</div>
                            <div class="detail-grid">
                                <div class="detail-item">
                                    <div class="detail-label">Konversations-ID</div>
                                    <div class="detail-value">${escapeHtml(conversation.id || '-')}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Status</div>
                                    <div class="detail-value">${statusBadge}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Kontakt</div>
                                    <div class="detail-value">${escapeHtml(conversation.contact_id || conversation.contact_name || '-')}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Betreff</div>
                                    <div class="detail-value">${escapeHtml(conversation.subject || '-')}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Erstellt am</div>
                                    <div class="detail-value">${conversation.created_at ? new Date(conversation.created_at).toLocaleString('de-DE') : '-'}</div>
                                </div>
                                <div class="detail-item">
                                    <div class="detail-label">Letzte Aktivität</div>
                                    <div class="detail-value">${conversation.latest_message_created_at ? new Date(conversation.latest_message_created_at).toLocaleString('de-DE') : '-'}</div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="detail-section">
                            <div class="detail-section-title"><i class="fas fa-envelope"></i> Nachrichtenverlauf (${messages.length})</div>
                            <div style="max-height: 400px; overflow-y: auto;">
                                ${messages.length > 0 ? messages.map(msg => {
                            const isFromUs = msg.author_id || msg.from;
                            const avatar = isFromUs ? '<i class="fas fa-user-circle" style="font-size: 2rem; color: #2196f3;"></i>' : '<i class="fas fa-user-circle" style="font-size: 2rem; color: #9e9e9e;"></i>';
                            return `
                                    <div style="padding: 1rem; margin-bottom: 0.75rem; background: ${isFromUs ? '#e3f2fd' : '#f5f5f5'}; border-radius: 8px; border-left: 4px solid ${isFromUs ? '#2196f3' : '#9e9e9e'}; display: flex; gap: 1rem;">
                                        <div style="flex-shrink: 0;">${avatar}</div>
                                        <div style="flex: 1;">
                                            <div style="display: flex; justify-content: space-between; margin-bottom: 0.5rem;">
                                                <div style="font-weight: 600; color: #212529;">${isFromUs ? 'Von uns' : 'Von Kontakt'}</div>
                                                <div style="font-size: 0.85rem; color: #6c757d;">${msg.created_at ? new Date(msg.created_at).toLocaleString('de-DE') : '-'}</div>
                                            </div>
                                            <div style="color: #212529; white-space: pre-wrap;">${escapeHtml(msg.text || msg.blurb || '-')}</div>
                                            ${msg.status ? `<div style="margin-top: 0.5rem;">${getMessageStatusBadge(msg.status)}</div>` : ''}
                                        </div>
                                    </div>
                                `;
                        }).join('') : '<div style="text-align: center; padding: 2rem; color: #6c757d;">Keine Nachrichten</div>'}
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button class="btn btn-success" onclick="replyToConversation('${conversationId}'); this.closest('.modal-overlay').remove();">
                            <i class="fas fa-reply"></i> Antworten
                        </button>
                        <button class="btn btn-secondary" onclick="this.closest('.modal-overlay').remove()">
                            <i class="fas fa-times"></i> Schließen
                        </button>
                    </div>
                </div>
            </div>
        `;
        
        const modal = document.createElement('div');
        modal.innerHTML = html;
        document.body.appendChild(modal);
    })
    .catch(error => {
        if (loadingModal && loadingModal.parentNode) {
            loadingModal.remove();
        }
        console.error('Fehler beim Laden der Konversationsdetails:', error);
        alert('Fehler beim Laden der Konversationsdetails: ' + error.message);
    });
}

// Reply to Conversation
function replyToConversation(conversationId) {
    // Verwende window.allConversations falls verfügbar
    const conversationsToSearch = window.allConversations || allConversations || [];
    const conversation = conversationsToSearch.find(c => c.id === conversationId);
    
    if (!conversation) {
        // Versuche Konversation über API zu laden
        const formData = new FormData();
        formData.append('ajax_action', 'get_conversation_details');
        formData.append('conversation_id', conversationId);
        
        fetch('hellomateo.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && data.data) {
                openReplyForm(data.data);
            } else {
                alert('Konversation nicht gefunden');
            }
        })
        .catch(error => {
            console.error('Fehler beim Laden der Konversation:', error);
            alert('Fehler beim Laden der Konversation: ' + error.message);
        });
        return;
    }
    
    openReplyForm(conversation);
}

// Öffne Antwort-Formular
function openReplyForm(conversation) {
    // Wechsle zum Nachrichten-Tab
    document.querySelectorAll('.hellomateo-tab').forEach(t => t.classList.remove('active'));
    document.querySelectorAll('.hellomateo-tab-content').forEach(c => c.classList.remove('active'));
    
    const messagesTab = document.querySelector('[data-tab="messages"]');
    const messagesContent = document.getElementById('tab-messages');
    if (messagesTab) messagesTab.classList.add('active');
    if (messagesContent) messagesContent.classList.add('active');
    
    // Finde Telefonnummer des Kontakts
    const contactId = conversation.contact_id || conversation.external_id || '';
    let phoneNumber = '';
    
    // Versuche Telefonnummer aus Konversation zu extrahieren
    if (conversation.external_id) {
        phoneNumber = conversation.external_id;
    } else if (contactId) {
        // Suche Kontakt in allContacts
        const contactsToSearch = window.allContacts || allContacts || [];
        const contact = contactsToSearch.find(c => c.id === contactId);
        if (contact && contact.handle) {
            phoneNumber = contact.handle;
        }
    }
    
    // Setze Empfänger
    const toInput = document.getElementById('message-to');
    const countryCodeSelect = document.getElementById('country-code');
    
    if (toInput && phoneNumber) {
        // Extrahiere Ländercode und Nummer
        const phone = phoneNumber.trim();
        
        if (phone.startsWith('+')) {
            // Finde Ländercode
            const countryCodes = {
                '+49': '+49',
                '+43': '+43',
                '+41': '+41',
                '+1': '+1',
                '+44': '+44'
            };
            
            let foundCode = null;
            let number = phone;
            
            for (const [code, value] of Object.entries(countryCodes)) {
                if (phone.startsWith(code)) {
                    foundCode = value;
                    number = phone.substring(code.length).trim();
                    break;
                }
            }
            
            if (foundCode && countryCodeSelect) {
                countryCodeSelect.value = foundCode;
                if (typeof updateCountryCodeDisplay === 'function') {
                    updateCountryCodeDisplay();
                }
            }
            
            toInput.value = number;
        } else {
            toInput.value = phone;
        }
        
        // Validiere Telefonnummer
        if (typeof validatePhoneNumber === 'function') {
            validatePhoneNumber();
        }
    }
    
    // Scroll zum Formular
    if (messagesContent) {
        messagesContent.scrollIntoView({ behavior: 'smooth' });
    }
}

// Resend Message
function resendMessage(messageId) {
    if (!confirm('Möchten Sie diese Nachricht wirklich erneut senden?')) return;
    
    const formData = new FormData();
    formData.append('ajax_action', 'resend_message');
    formData.append('message_id', messageId);
    
    fetch('hellomateo.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('Nachricht erfolgreich erneut gesendet');
            loadMessages();
        } else {
            alert('Fehler: ' + (data.error || 'Unbekannter Fehler'));
        }
    })
    .catch(error => {
        alert('Fehler: ' + error.message);
    });
}

// Autohaus wechseln
function switchAutohaus(autohausId) {
    if (!autohausId) return;
    
    const url = new URL(window.location);
    url.searchParams.set('autohaus_id', autohausId);
    // Aktuellen Tab beibehalten
    const currentTab = url.searchParams.get('tab') || 'messages';
    url.searchParams.set('tab', currentTab);
    window.location.href = url.toString();
}

// HTML-Escape-Funktion
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Error-Logs anzeigen
function showErrorLogs() {
    const formData = new FormData();
    formData.append('ajax_action', 'get_error_logs');
    formData.append('max_lines', 100);
    
    // Erstelle Modal für Logs
    const modal = document.createElement('div');
    modal.className = 'modal';
    modal.style.display = 'block';
    modal.innerHTML = `
        <div class="modal-content" style="max-width: 90%; max-height: 90vh; overflow: auto;">
            <div class="modal-header">
                <h3>PHP Error-Logs (HelloMateo)</h3>
                <button type="button" class="modal-close" onclick="this.closest('.modal').remove()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="modal-body">
                <div id="error-logs-content" style="padding: 1rem;">
                    <div class="loading"><i class="fas fa-spinner"></i> Lade Logs...</div>
                </div>
            </div>
        </div>
    `;
    document.body.appendChild(modal);
    
    fetch('hellomateo.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        const contentDiv = document.getElementById('error-logs-content');
        
        if (data.success) {
            let html = '';
            
            if (data.log_path) {
                html += `<div style="margin-bottom: 1rem; padding: 0.75rem; background: #e3f2fd; border-radius: 6px;">
                    <strong>Log-Datei gefunden:</strong><br>
                    <code style="font-size: 0.9rem; word-break: break-all;">${escapeHtml(data.log_path)}</code>
                </div>`;
            } else {
                html += `<div style="margin-bottom: 1rem; padding: 0.75rem; background: #fff3cd; border-radius: 6px;">
                    <strong>⚠️ Keine Log-Datei gefunden!</strong><br>
                    <p>Mögliche Log-Pfade:</p>
                    <ul style="margin: 0.5rem 0; padding-left: 1.5rem;">
                        ${data.possible_paths.map(path => `<li><code style="font-size: 0.85rem;">${escapeHtml(path || 'N/A')}</code></li>`).join('')}
                    </ul>
                    <p style="margin-top: 0.5rem;"><strong>Tipp:</strong> Prüfen Sie die PHP-Konfiguration (php.ini) für den Wert von <code>error_log</code></p>
                </div>`;
            }
            
            if (data.log_entries && data.log_entries.length > 0) {
                html += `<div style="margin-bottom: 1rem;">
                    <strong>Letzte ${data.entries_count} HelloMateo-relevante Log-Einträge:</strong>
                </div>`;
                html += `<pre style="background: #f5f5f5; padding: 1rem; border-radius: 6px; overflow-x: auto; font-size: 0.85rem; line-height: 1.5; max-height: 60vh; overflow-y: auto;">`;
                data.log_entries.forEach(entry => {
                    html += escapeHtml(entry);
                });
                html += `</pre>`;
            } else {
                html += `<div style="padding: 1rem; background: #f5f5f5; border-radius: 6px;">
                    <p>Keine HelloMateo-relevanten Log-Einträge gefunden.</p>
                    <p>Mögliche Gründe:</p>
                    <ul style="margin: 0.5rem 0; padding-left: 1.5rem;">
                        <li>Es wurden noch keine Nachrichten gesendet</li>
                        <li>Die Log-Datei ist an einem anderen Ort</li>
                        <li>PHP-Fehler werden nicht geloggt (prüfen Sie php.ini)</li>
                    </ul>
                </div>`;
            }
            
            contentDiv.innerHTML = html;
        } else {
            contentDiv.innerHTML = `<div class="alert alert-error">Fehler beim Laden der Logs: ${escapeHtml(data.error || 'Unbekannter Fehler')}</div>`;
        }
    })
    .catch(error => {
        const contentDiv = document.getElementById('error-logs-content');
        contentDiv.innerHTML = `<div class="alert alert-error">Fehler: ${escapeHtml(error.message)}</div>`;
    });
}

// Load initial data for active tab (mit Cache)
document.addEventListener('DOMContentLoaded', function() {
    const activeTab = '<?php echo $active_tab; ?>';
    
    // Prüfe URL-Parameter für vorausgefüllte Werte
    const urlParams = new URLSearchParams(window.location.search);
    const toParam = urlParams.get('to');
    const messageParam = urlParams.get('message');
    
    switch(activeTab) {
        case 'messages':
            loadMessages(1, true); // true = Cache verwenden
            loadTemplates(1, true); // Load templates for select
            
            // Fülle Formular wenn URL-Parameter vorhanden
            setTimeout(() => {
                if (toParam) {
                    const toInput = document.getElementById('message-to');
                    if (toInput) {
                        toInput.value = decodeURIComponent(toParam);
                    }
                }
                if (messageParam) {
                    const messageTextarea = document.getElementById('message-text');
                    if (messageTextarea) {
                        messageTextarea.value = decodeURIComponent(messageParam);
                    }
                }
            }, 500);
            break;
        case 'contacts':
            loadContacts(1, true); // true = Cache verwenden
            break;
        case 'templates':
            loadTemplates(1, true); // true = Cache verwenden
            break;
        case 'conversations':
            loadConversations(1, true); // true = Cache verwenden
            break;
    }
});

// HTML-Escape-Funktion
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

</script>
        </main>
    </div>
</body>
</html>

