<?php
/**
 * Feedback Management - Admin Seite (Komplett überarbeitet)
 */

session_start();
require_once '../config/database.php';
require_once 'includes/helpers.php';

// Prüfe ob Admin eingeloggt ist
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

$db = new Database();
$conn = $db->getConnection();

// Aktueller Admin-User
$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_name = 'Administrator';
$current_user_role = 'admin';
$current_user_autohaus_id = null;

if ($current_user_id) {
    $stmt = $conn->prepare("
        SELECT u.role, u.autohaus_id, u.username
        FROM admin_users u
        WHERE u.id = ?
    ");
    $stmt->execute([$current_user_id]);
    $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
    $current_user_role = $user_data['role'] ?? 'admin';
    $current_user_autohaus_id = $user_data['autohaus_id'];
    $current_user_name = $user_data['username'] ?? 'Administrator';
}

// Prüfe ob Feedback-Tabelle existiert, falls nicht erstelle sie
try {
    $checkTable = $conn->query("SHOW TABLES LIKE 'feedback'");
    if ($checkTable->rowCount() == 0) {
        $conn->exec("
            CREATE TABLE feedback (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT NULL,
                autohaus_id INT NULL,
                rating INT NOT NULL,
                feedback_text TEXT NOT NULL,
                feedback_type ENUM('service', 'booking', 'general') DEFAULT 'general',
                page_url VARCHAR(500) NULL,
                is_anonymous BOOLEAN DEFAULT FALSE,
                status ENUM('new', 'read', 'archived') DEFAULT 'new',
                admin_response TEXT NULL,
                admin_response_by INT NULL,
                admin_response_at DATETIME NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                INDEX idx_user_id (user_id),
                INDEX idx_autohaus_id (autohaus_id),
                INDEX idx_status (status),
                INDEX idx_created_at (created_at)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");
    }
} catch (PDOException $e) {
    // Tabelle existiert bereits oder Fehler
}

// Filter und Suche
$filter_status = $_GET['status'] ?? 'all';
$filter_autohaus = $_GET['autohaus_id'] ?? 'all';
$filter_type = $_GET['type'] ?? 'all';
$filter_rating = $_GET['rating'] ?? 'all';
$search_query = $_GET['search'] ?? '';

// Lade Feedback mit Filtern
$where_conditions = [];
$params = [];

if ($filter_status !== 'all') {
    $where_conditions[] = "f.status = ?";
    $params[] = $filter_status;
}

if ($filter_autohaus !== 'all') {
    $where_conditions[] = "f.autohaus_id = ?";
    $params[] = $filter_autohaus;
}

if ($filter_type !== 'all') {
    $where_conditions[] = "f.feedback_type = ?";
    $params[] = $filter_type;
}

if ($filter_rating !== 'all') {
    $where_conditions[] = "f.rating = ?";
    $params[] = $filter_rating;
}

if (!empty($search_query)) {
    $where_conditions[] = "(f.feedback_text LIKE ? OR u.first_name LIKE ? OR u.last_name LIKE ? OR u.email LIKE ?)";
    $search_param = "%{$search_query}%";
    $params[] = $search_param;
    $params[] = $search_param;
    $params[] = $search_param;
    $params[] = $search_param;
}

// Wenn Admin nur für ein Autohaus zuständig ist
if ($current_user_autohaus_id && $current_user_role !== 'super_admin') {
    $where_conditions[] = "f.autohaus_id = ?";
    $params[] = $current_user_autohaus_id;
}

$where_clause = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";

// Debug: Teste zuerst einfache Abfrage
try {
    $testStmt = $conn->query("SELECT COUNT(*) as count FROM feedback");
    $testResult = $testStmt->fetch(PDO::FETCH_ASSOC);
    $totalCount = $testResult['count'] ?? 0;
    
    if ($totalCount == 0) {
        error_log("Feedback Debug: Keine Feedbacks in der Datenbank gefunden!");
        $feedback_list = [];
    } else {
        // Lade Feedbacks mit vollständiger Abfrage
        $sql = "
            SELECT f.*, 
                   u.first_name, u.last_name, u.email,
                   a.name as autohaus_name,
                   admin.username as admin_response_by_name
            FROM feedback f
            LEFT JOIN users u ON f.user_id = u.id
            LEFT JOIN autohaus a ON f.autohaus_id = a.id
            LEFT JOIN admin_users admin ON f.admin_response_by = admin.id
            $where_clause
            ORDER BY f.created_at DESC
            LIMIT 100
        ";
        
        $stmt = $conn->prepare($sql);
        $stmt->execute($params);
        $feedback_list = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Wenn keine Feedbacks geladen wurden, aber welche in der DB sind, versuche ohne JOINs
        if (empty($feedback_list) && $totalCount > 0) {
            error_log("Feedback Debug: Feedbacks mit JOINs nicht geladen, versuche ohne JOINs");
            $simpleStmt = $conn->prepare("
                SELECT * FROM feedback 
                $where_clause
                ORDER BY created_at DESC
                LIMIT 100
            ");
            $simpleStmt->execute($params);
            $feedback_list = $simpleStmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Füge leere Werte für JOIN-Felder hinzu
            foreach ($feedback_list as &$fb) {
                $fb['first_name'] = null;
                $fb['last_name'] = null;
                $fb['email'] = null;
                $fb['autohaus_name'] = null;
                $fb['admin_response_by_name'] = null;
            }
        }
    }
} catch (PDOException $e) {
    error_log("Fehler beim Laden der Feedbacks: " . $e->getMessage());
    error_log("SQL Query: " . ($sql ?? 'N/A'));
    error_log("Params: " . print_r($params, true));
    $feedback_list = [];
    
    // Fallback: Lade Feedbacks ohne JOINs
    try {
        $fallbackStmt = $conn->query("SELECT * FROM feedback ORDER BY created_at DESC LIMIT 100");
        $feedback_list = $fallbackStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Füge leere Werte für JOIN-Felder hinzu
        foreach ($feedback_list as &$fb) {
            $fb['first_name'] = null;
            $fb['last_name'] = null;
            $fb['email'] = null;
            $fb['autohaus_name'] = null;
            $fb['admin_response_by_name'] = null;
        }
    } catch (PDOException $e2) {
        error_log("Fallback-Abfrage fehlgeschlagen: " . $e2->getMessage());
        $feedback_list = [];
    }
}

// Lade Autohäuser für Filter
$stmt = $conn->prepare("SELECT id, name FROM autohaus WHERE is_active = 1 ORDER BY name");
$stmt->execute();
$autohauser = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Statistiken
$stats_where = ($current_user_autohaus_id && $current_user_role !== 'super_admin') ? "WHERE autohaus_id = ?" : "";
$stats_params = ($current_user_autohaus_id && $current_user_role !== 'super_admin') ? [$current_user_autohaus_id] : [];

$stmt = $conn->prepare("
    SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN status = 'new' THEN 1 ELSE 0 END) as new_count,
        SUM(CASE WHEN status = 'read' THEN 1 ELSE 0 END) as read_count,
        SUM(CASE WHEN status = 'archived' THEN 1 ELSE 0 END) as archived_count,
        AVG(rating) as avg_rating,
        SUM(CASE WHEN rating = 5 THEN 1 ELSE 0 END) as rating_5,
        SUM(CASE WHEN rating = 4 THEN 1 ELSE 0 END) as rating_4,
        SUM(CASE WHEN rating = 3 THEN 1 ELSE 0 END) as rating_3,
        SUM(CASE WHEN rating = 2 THEN 1 ELSE 0 END) as rating_2,
        SUM(CASE WHEN rating = 1 THEN 1 ELSE 0 END) as rating_1
    FROM feedback
    $stats_where
");
$stmt->execute($stats_params);
$stats = $stmt->fetch(PDO::FETCH_ASSOC);

?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Feedback verwalten - Boxenstop Admin</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/volvo-theme.css">
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="<?php echo asset_url('assets/css/unified-sidebar.css'); ?>">
    <link rel="stylesheet" href="assets/css/admin-zoom.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&family=SF+Pro+Display:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #003057;
            --primary-light: #0056b3;
            --success: #28a745;
            --warning: #ffc107;
            --danger: #dc3545;
            --info: #17a2b8;
            --light: #f8f9fa;
            --dark: #343a40;
            --border: #dee2e6;
            --shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
            --shadow-lg: 0 8px 24px rgba(0, 0, 0, 0.15);
        }

        /* Stats Grid */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .stat-card {
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            display: flex;
            align-items: center;
            gap: 1rem;
            box-shadow: var(--shadow);
            transition: transform 0.2s, box-shadow 0.2s;
            border: 1px solid var(--border);
        }

        .stat-card:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }

        .stat-icon {
            width: 60px;
            height: 60px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 1.5rem;
            flex-shrink: 0;
        }

        .stat-content {
            flex: 1;
            min-width: 0;
        }

        .stat-value {
            font-size: 2rem;
            font-weight: 700;
            color: var(--dark);
            line-height: 1.2;
        }

        .stat-label {
            color: #6c757d;
            font-size: 0.9rem;
            margin-top: 0.25rem;
        }

        /* Filters */
        .filters-section {
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            margin-bottom: 2rem;
            box-shadow: var(--shadow);
            border: 1px solid var(--border);
        }

        .filters-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-bottom: 1rem;
        }

        .filter-group {
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        }

        .filter-group label {
            font-weight: 600;
            color: var(--dark);
            font-size: 0.9rem;
        }

        .filter-group select,
        .filter-group input {
            padding: 0.75rem;
            border: 1px solid var(--border);
            border-radius: 8px;
            font-size: 0.9rem;
            transition: border-color 0.2s;
        }

        .filter-group select:focus,
        .filter-group input:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(0, 48, 87, 0.1);
        }

        .search-box {
            position: relative;
        }

        .search-box i {
            position: absolute;
            left: 1rem;
            top: 50%;
            transform: translateY(-50%);
            color: #6c757d;
        }

        .search-box input {
            padding-left: 2.75rem;
        }

        /* Feedback Cards */
        .feedback-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 1.5rem;
            margin-top: 2rem;
        }
        
        @media (max-width: 768px) {
            .feedback-grid {
                grid-template-columns: 1fr;
            }
        }

        .feedback-card {
            background: white;
            border-radius: 12px;
            padding: 1.5rem;
            box-shadow: var(--shadow);
            border: 1px solid var(--border);
            transition: all 0.2s;
            cursor: pointer;
        }

        .feedback-card:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
            border-color: var(--primary);
        }

        .feedback-card.new {
            border-left: 4px solid var(--warning);
        }

        .feedback-card-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 1rem;
        }

        .feedback-rating {
            display: flex;
            gap: 0.25rem;
            align-items: center;
        }

        .feedback-rating .fa-star {
            color: #dee2e6;
            font-size: 1.1rem;
        }

        .feedback-rating .fa-star.active {
            color: var(--warning);
        }

        .feedback-status {
            padding: 0.375rem 0.75rem;
            border-radius: 6px;
            font-size: 0.8rem;
            font-weight: 600;
            text-transform: uppercase;
        }

        .status-new {
            background: #fff3cd;
            color: #856404;
        }

        .status-read {
            background: #d1ecf1;
            color: #0c5460;
        }

        .status-archived {
            background: #e2e3e5;
            color: #383d41;
        }

        .feedback-body {
            margin-bottom: 1rem;
        }

        .feedback-text {
            color: var(--dark);
            line-height: 1.6;
            margin-bottom: 0.75rem;
            display: -webkit-box;
            -webkit-line-clamp: 3;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }

        .feedback-meta {
            display: flex;
            flex-wrap: wrap;
            gap: 1rem;
            font-size: 0.85rem;
            color: #6c757d;
        }

        .feedback-meta-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .feedback-meta-item i {
            color: var(--primary);
        }

        .feedback-actions {
            display: flex;
            gap: 0.5rem;
            margin-top: 1rem;
            padding-top: 1rem;
            border-top: 1px solid var(--border);
        }

        .btn-action {
            padding: 0.5rem 1rem;
            border: none;
            border-radius: 6px;
            font-size: 0.85rem;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.2s;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }

        .btn-view {
            background: var(--primary);
            color: white;
        }

        .btn-view:hover {
            background: var(--primary-light);
        }

        .btn-respond {
            background: var(--success);
            color: white;
        }

        .btn-respond:hover {
            background: #218838;
        }

        /* Modal Styles */
        .modal {
            display: none;
            position: fixed;
            z-index: 10000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.7);
            backdrop-filter: blur(4px);
            animation: fadeIn 0.3s;
        }

        .modal.active {
            display: flex;
            align-items: center;
            justify-content: center;
        }

        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }

        .modal-content {
            background: white;
            border-radius: 16px;
            width: 90%;
            max-width: 800px;
            max-height: 90vh;
            overflow-y: auto;
            box-shadow: var(--shadow-lg);
            animation: slideUp 0.3s;
            position: relative;
        }

        @keyframes slideUp {
            from {
                transform: translateY(50px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }

        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 1.5rem 2rem;
            border-bottom: 1px solid var(--border);
            background: var(--light);
            border-radius: 16px 16px 0 0;
            position: sticky;
            top: 0;
            z-index: 1;
        }

        .modal-header h3 {
            margin: 0;
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--dark);
        }

        .modal-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            color: #6c757d;
            cursor: pointer;
            width: 36px;
            height: 36px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 8px;
            transition: all 0.2s;
        }

        .modal-close:hover {
            background: var(--border);
            color: var(--dark);
        }

        .modal-body {
            padding: 2rem;
        }

        .detail-section {
            margin-bottom: 2rem;
        }

        .detail-section:last-child {
            margin-bottom: 0;
        }

        .detail-section-title {
            font-size: 1.1rem;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 1rem;
            padding-bottom: 0.5rem;
            border-bottom: 2px solid var(--border);
        }

        .detail-grid {
            display: grid;
            grid-template-columns: 150px 1fr;
            gap: 1rem;
            margin-bottom: 1rem;
        }

        .detail-label {
            font-weight: 600;
            color: #6c757d;
        }

        .detail-value {
            color: var(--dark);
        }

        .feedback-text-full {
            background: var(--light);
            padding: 1.5rem;
            border-radius: 12px;
            line-height: 1.8;
            color: var(--dark);
            white-space: pre-wrap;
        }

        .admin-response-box {
            background: #e3f2fd;
            padding: 1.5rem;
            border-radius: 12px;
            border-left: 4px solid var(--primary);
            margin-top: 1rem;
        }

        .admin-response-box small {
            display: block;
            margin-top: 0.75rem;
            color: #6c757d;
            font-size: 0.85rem;
        }

        .form-group {
            margin-bottom: 1.5rem;
        }

        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
            color: var(--dark);
        }

        .form-group textarea,
        .form-group select {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid var(--border);
            border-radius: 8px;
            font-size: 0.9rem;
            font-family: inherit;
            transition: border-color 0.2s;
        }

        .form-group textarea {
            min-height: 120px;
            resize: vertical;
        }

        .form-group textarea:focus,
        .form-group select:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(0, 48, 87, 0.1);
        }

        .form-actions {
            display: flex;
            gap: 1rem;
            justify-content: flex-end;
            margin-top: 2rem;
            padding-top: 2rem;
            border-top: 1px solid var(--border);
        }

        .btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 8px;
            font-size: 0.9rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.2s;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--primary-light);
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
        }

        .btn-secondary:hover {
            background: #5a6268;
        }

        .btn-success {
            background: var(--success);
            color: white;
        }

        .btn-success:hover {
            background: #218838;
        }

        .empty-state {
            text-align: center;
            padding: 4rem 2rem;
            color: #6c757d;
        }

        .empty-state i {
            font-size: 4rem;
            color: #dee2e6;
            margin-bottom: 1rem;
        }

        .empty-state h3 {
            color: var(--dark);
            margin-bottom: 0.5rem;
        }

        /* Audit Log Styles */
        .audit-log-list {
            display: flex;
            flex-direction: column;
            gap: 1rem;
            margin-top: 1rem;
        }

        .audit-log-item {
            display: flex;
            gap: 1rem;
            padding: 1rem;
            background: var(--light);
            border-radius: 8px;
            border-left: 3px solid var(--primary);
            transition: all 0.2s;
        }

        .audit-log-item:hover {
            background: #e9ecef;
            transform: translateX(4px);
        }

        .audit-log-icon {
            width: 40px;
            height: 40px;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.1rem;
            flex-shrink: 0;
        }

        .audit-log-content {
            flex: 1;
            min-width: 0;
        }

        .audit-log-action {
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 0.5rem;
        }

        .audit-log-change {
            margin: 0.5rem 0;
            font-size: 0.9rem;
        }

        .audit-log-meta {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.85rem;
            color: #6c757d;
            margin-top: 0.5rem;
        }

        /* Responsive */
        @media (max-width: 768px) {
            .stats-grid {
                grid-template-columns: 1fr;
            }

            .filters-grid {
                grid-template-columns: 1fr;
            }

            .detail-grid {
                grid-template-columns: 1fr;
            }

            .modal-content {
                width: 95%;
                max-height: 95vh;
            }

            .modal-body {
                padding: 1.5rem;
            }
        }
    </style>
</head>
<body class="admin-body">
    <div class="admin-container">
        <?php include 'includes/sidebar.php'; ?>
        
        <main class="admin-main">
            <div class="content-header">
                <h1>
                    <i class="fas fa-comment-dots"></i>
                    Feedback-Verwaltung
                </h1>
                <div class="header-actions">
                    <button class="btn btn-secondary" onclick="exportFeedback()">
                        <i class="fas fa-download"></i>
                        Export
                    </button>
                </div>
            </div>

            <!-- Statistiken -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon" style="background: var(--primary);">
                        <i class="fas fa-comments"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo $stats['total'] ?? 0; ?></div>
                        <div class="stat-label">Gesamt Feedback</div>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon" style="background: var(--warning);">
                        <i class="fas fa-exclamation-circle"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo $stats['new_count'] ?? 0; ?></div>
                        <div class="stat-label">Neu</div>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon" style="background: var(--info);">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo $stats['read_count'] ?? 0; ?></div>
                        <div class="stat-label">Gelesen</div>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon" style="background: var(--success);">
                        <i class="fas fa-star"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo number_format($stats['avg_rating'] ?? 0, 1); ?></div>
                        <div class="stat-label">Ø Bewertung</div>
                    </div>
                </div>
            </div>

            <!-- Filter -->
            <div class="filters-section">
                <form method="GET" id="filter-form">
                    <div class="filters-grid">
                        <div class="filter-group">
                            <label>Status</label>
                            <select name="status" onchange="this.form.submit()">
                                <option value="all" <?php echo $filter_status === 'all' ? 'selected' : ''; ?>>Alle Status</option>
                                <option value="new" <?php echo $filter_status === 'new' ? 'selected' : ''; ?>>Neu</option>
                                <option value="read" <?php echo $filter_status === 'read' ? 'selected' : ''; ?>>Gelesen</option>
                                <option value="archived" <?php echo $filter_status === 'archived' ? 'selected' : ''; ?>>Archiviert</option>
                            </select>
                        </div>
                        <div class="filter-group">
                            <label>Autohaus</label>
                            <select name="autohaus_id" onchange="this.form.submit()">
                                <option value="all" <?php echo $filter_autohaus === 'all' ? 'selected' : ''; ?>>Alle Autohäuser</option>
                                <?php foreach ($autohauser as $ah): ?>
                                    <option value="<?php echo $ah['id']; ?>" <?php echo $filter_autohaus == $ah['id'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($ah['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="filter-group">
                            <label>Typ</label>
                            <select name="type" onchange="this.form.submit()">
                                <option value="all" <?php echo $filter_type === 'all' ? 'selected' : ''; ?>>Alle Typen</option>
                                <option value="service" <?php echo $filter_type === 'service' ? 'selected' : ''; ?>>Service</option>
                                <option value="booking" <?php echo $filter_type === 'booking' ? 'selected' : ''; ?>>Buchung</option>
                                <option value="general" <?php echo $filter_type === 'general' ? 'selected' : ''; ?>>Allgemein</option>
                            </select>
                        </div>
                        <div class="filter-group">
                            <label>Bewertung</label>
                            <select name="rating" onchange="this.form.submit()">
                                <option value="all" <?php echo $filter_rating === 'all' ? 'selected' : ''; ?>>Alle Bewertungen</option>
                                <option value="5" <?php echo $filter_rating === '5' ? 'selected' : ''; ?>>5 Sterne</option>
                                <option value="4" <?php echo $filter_rating === '4' ? 'selected' : ''; ?>>4 Sterne</option>
                                <option value="3" <?php echo $filter_rating === '3' ? 'selected' : ''; ?>>3 Sterne</option>
                                <option value="2" <?php echo $filter_rating === '2' ? 'selected' : ''; ?>>2 Sterne</option>
                                <option value="1" <?php echo $filter_rating === '1' ? 'selected' : ''; ?>>1 Stern</option>
                            </select>
                        </div>
                        <div class="filter-group search-box">
                            <label>Suche</label>
                            <input type="text" name="search" placeholder="Suche nach Text, Name oder E-Mail..." value="<?php echo htmlspecialchars($search_query); ?>">
                            <i class="fas fa-search"></i>
                        </div>
                    </div>
                    <input type="hidden" name="status" value="<?php echo htmlspecialchars($filter_status); ?>">
                    <input type="hidden" name="autohaus_id" value="<?php echo htmlspecialchars($filter_autohaus); ?>">
                    <input type="hidden" name="type" value="<?php echo htmlspecialchars($filter_type); ?>">
                    <input type="hidden" name="rating" value="<?php echo htmlspecialchars($filter_rating); ?>">
                </form>
            </div>

            <!-- Feedback Liste -->
            <div class="feedback-grid">
                <?php 
                // Debug: Zeige Anzahl der Feedbacks
                if (isset($_GET['debug'])) {
                    echo "<!-- Debug: Anzahl Feedbacks: " . count($feedback_list) . " -->";
                }
                if (empty($feedback_list)): ?>
                    <div class="empty-state">
                        <i class="fas fa-inbox"></i>
                        <h3>Kein Feedback gefunden</h3>
                        <p>Es wurden keine Feedback-Einträge gefunden, die den Filterkriterien entsprechen.</p>
                        <?php if (isset($_GET['debug'])): ?>
                            <p style="margin-top: 1rem; font-size: 0.9rem; color: #6c757d;">
                                Debug: Filter Status: <?php echo htmlspecialchars($filter_status); ?>, 
                                Autohaus: <?php echo htmlspecialchars($filter_autohaus); ?>, 
                                Type: <?php echo htmlspecialchars($filter_type); ?>, 
                                Rating: <?php echo htmlspecialchars($filter_rating); ?>
                            </p>
                        <?php endif; ?>
                    </div>
                <?php else: ?>
                    <?php foreach ($feedback_list as $fb): ?>
                        <div class="feedback-card <?php echo $fb['status'] === 'new' ? 'new' : ''; ?>" onclick="openDetailModal(<?php echo $fb['id']; ?>)">
                            <div class="feedback-card-header">
                                <div class="feedback-rating">
                                    <?php for ($i = 1; $i <= 5; $i++): ?>
                                        <i class="fas fa-star <?php echo $i <= $fb['rating'] ? 'active' : ''; ?>"></i>
                                    <?php endfor; ?>
                                    <span style="margin-left: 0.5rem; font-weight: 600;"><?php echo $fb['rating']; ?>/5</span>
                                </div>
                                <span class="feedback-status status-<?php echo $fb['status']; ?>">
                                    <?php 
                                        echo $fb['status'] === 'new' ? 'Neu' : 
                                            ($fb['status'] === 'read' ? 'Gelesen' : 'Archiviert'); 
                                    ?>
                                </span>
                            </div>
                            <div class="feedback-body">
                                <div class="feedback-text">
                                    <?php echo htmlspecialchars($fb['feedback_text']); ?>
                                </div>
                                <div class="feedback-meta">
                                    <div class="feedback-meta-item">
                                        <i class="fas fa-user"></i>
                                        <span>
                                            <?php if ($fb['is_anonymous']): ?>
                                                Anonym
                                            <?php elseif ($fb['first_name']): ?>
                                                <?php echo htmlspecialchars($fb['first_name'] . ' ' . $fb['last_name']); ?>
                                            <?php else: ?>
                                                Unbekannt
                                            <?php endif; ?>
                                        </span>
                                    </div>
                                    <?php if ($fb['autohaus_name']): ?>
                                    <div class="feedback-meta-item">
                                        <i class="fas fa-building"></i>
                                        <span><?php echo htmlspecialchars($fb['autohaus_name']); ?></span>
                                    </div>
                                    <?php endif; ?>
                                    <div class="feedback-meta-item">
                                        <i class="fas fa-calendar"></i>
                                        <span><?php echo date('d.m.Y H:i', strtotime($fb['created_at'])); ?></span>
                                    </div>
                                    <div class="feedback-meta-item">
                                        <i class="fas fa-tag"></i>
                                        <span><?php echo ucfirst($fb['feedback_type']); ?></span>
                                    </div>
                                </div>
                            </div>
                            <div class="feedback-actions" onclick="event.stopPropagation();">
                                <button class="btn-action btn-view" onclick="openDetailModal(<?php echo $fb['id']; ?>)">
                                    <i class="fas fa-eye"></i>
                                    Details
                                </button>
                                <?php if (!$fb['admin_response']): ?>
                                <button class="btn-action btn-respond" onclick="openResponseModal(<?php echo $fb['id']; ?>)">
                                    <i class="fas fa-reply"></i>
                                    Antworten
                                </button>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </main>
    </div>

    <!-- Detail Modal -->
    <div id="detail-modal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3><i class="fas fa-comment-dots"></i> Feedback Details</h3>
                <button class="modal-close" onclick="closeDetailModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="modal-body" id="detail-modal-body">
                <div style="text-align: center; padding: 2rem;">
                    <i class="fas fa-spinner fa-spin" style="font-size: 2rem; color: var(--primary);"></i>
                    <p>Lade Details...</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Response Modal -->
    <div id="response-modal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3><i class="fas fa-reply"></i> Admin-Antwort</h3>
                <button class="modal-close" onclick="closeResponseModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="modal-body">
                <form id="response-form">
                    <input type="hidden" id="response-feedback-id" name="feedback_id">
                    <div class="form-group">
                        <label>Ihre Antwort an den Benutzer:</label>
                        <textarea id="response-text" name="admin_response" required placeholder="Geben Sie hier Ihre Antwort ein..."></textarea>
                    </div>
                    <div class="form-actions">
                        <button type="button" class="btn btn-secondary" onclick="closeResponseModal()">
                            <i class="fas fa-times"></i>
                            Abbrechen
                        </button>
                        <button type="submit" class="btn btn-success">
                            <i class="fas fa-paper-plane"></i>
                            Antwort senden
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        // Detail Modal
        async function openDetailModal(feedbackId) {
            const modal = document.getElementById('detail-modal');
            const body = document.getElementById('detail-modal-body');
            
            modal.classList.add('active');
            body.innerHTML = '<div style="text-align: center; padding: 2rem;"><i class="fas fa-spinner fa-spin" style="font-size: 2rem; color: var(--primary);"></i><p>Lade Details...</p></div>';
            
            try {
                const response = await fetch(`api/get_feedback.php?id=${feedbackId}`);
                const data = await response.json();
                
                if (data.success) {
                    const fb = data.feedback;
                    let stars = '';
                    for (let i = 1; i <= 5; i++) {
                        stars += `<i class="fas fa-star ${i <= fb.rating ? 'active' : ''}" style="color: ${i <= fb.rating ? 'var(--warning)' : '#dee2e6'};"></i>`;
                    }
                    
                    body.innerHTML = `
                        <div class="detail-section">
                            <div class="detail-grid">
                                <div class="detail-label">Bewertung:</div>
                                <div class="detail-value">
                                    <div class="feedback-rating">${stars} <span style="margin-left: 0.5rem; font-weight: 600;">${fb.rating}/5</span></div>
                                </div>
                            </div>
                            <div class="detail-grid">
                                <div class="detail-label">Status:</div>
                                <div class="detail-value">
                                    <span class="feedback-status status-${fb.status}">
                                        ${fb.status === 'new' ? 'Neu' : (fb.status === 'read' ? 'Gelesen' : 'Archiviert')}
                                    </span>
                                </div>
                            </div>
                            <div class="detail-grid">
                                <div class="detail-label">Typ:</div>
                                <div class="detail-value">${fb.feedback_type.charAt(0).toUpperCase() + fb.feedback_type.slice(1)}</div>
                            </div>
                            ${fb.autohaus_name ? `
                            <div class="detail-grid">
                                <div class="detail-label">Autohaus:</div>
                                <div class="detail-value">${fb.autohaus_name}</div>
                            </div>
                            ` : ''}
                            <div class="detail-grid">
                                <div class="detail-label">Benutzer:</div>
                                <div class="detail-value">
                                    ${fb.is_anonymous ? '<em>Anonym</em>' : (fb.first_name ? `${fb.first_name} ${fb.last_name} (${fb.email})` : '<em>Nicht verfügbar</em>')}
                                </div>
                            </div>
                            <div class="detail-grid">
                                <div class="detail-label">Erstellt am:</div>
                                <div class="detail-value">${new Date(fb.created_at).toLocaleString('de-DE')}</div>
                            </div>
                            ${fb.page_url ? `
                            <div class="detail-grid">
                                <div class="detail-label">Seite:</div>
                                <div class="detail-value">${fb.page_url}</div>
                            </div>
                            ` : ''}
                        </div>
                        
                        <div class="detail-section">
                            <div class="detail-section-title">Feedback-Text</div>
                            <div class="feedback-text-full">${fb.feedback_text.replace(/\n/g, '<br>')}</div>
                        </div>
                        
                        ${fb.admin_response ? `
                        <div class="detail-section">
                            <div class="detail-section-title">Admin-Antwort</div>
                            <div class="admin-response-box">
                                ${fb.admin_response.replace(/\n/g, '<br>')}
                                <small>
                                    von ${fb.admin_response_by_name || 'Unbekannt'} 
                                    am ${new Date(fb.admin_response_at).toLocaleString('de-DE')}
                                </small>
                            </div>
                        </div>
                        ` : ''}
                        
                        ${data.audit_log && data.audit_log.length > 0 ? `
                        <div class="detail-section">
                            <div class="detail-section-title">
                                <i class="fas fa-history"></i>
                                Aktivitätsprotokoll
                            </div>
                            <div class="audit-log-list">
                                ${data.audit_log.map(log => {
                                    const actionIcons = {
                                        'status_changed': 'fas fa-exchange-alt',
                                        'response_added': 'fas fa-reply',
                                        'response_updated': 'fas fa-edit',
                                        'email_sent': 'fas fa-envelope',
                                        'email_failed': 'fas fa-exclamation-triangle',
                                        'viewed': 'fas fa-eye',
                                        'archived': 'fas fa-archive',
                                        'restored': 'fas fa-undo'
                                    };
                                    const actionColors = {
                                        'status_changed': '#0056b3',
                                        'response_added': '#28a745',
                                        'response_updated': '#ffc107',
                                        'email_sent': '#17a2b8',
                                        'email_failed': '#dc3545',
                                        'viewed': '#6c757d',
                                        'archived': '#6c757d',
                                        'restored': '#28a745'
                                    };
                                    const icon = actionIcons[log.action_type] || 'fas fa-circle';
                                    const color = actionColors[log.action_type] || '#6c757d';
                                    const date = new Date(log.created_at).toLocaleString('de-DE');
                                    let metadata = '';
                                    if (log.metadata) {
                                        try {
                                            const meta = JSON.parse(log.metadata);
                                            if (meta.error) {
                                                metadata = `<div style="font-size: 0.85rem; color: #dc3545; margin-top: 0.25rem;">Fehler: ${meta.error}</div>`;
                                            }
                                        } catch (e) {}
                                    }
                                    return `
                                        <div class="audit-log-item">
                                            <div class="audit-log-icon" style="background: ${color}20; color: ${color};">
                                                <i class="${icon}"></i>
                                            </div>
                                            <div class="audit-log-content">
                                                <div class="audit-log-action">${log.action_description}</div>
                                                ${log.old_value && log.new_value ? `
                                                <div class="audit-log-change">
                                                    <span style="color: #6c757d;">${log.old_value}</span>
                                                    <i class="fas fa-arrow-right" style="margin: 0 0.5rem; color: #6c757d;"></i>
                                                    <span style="font-weight: 600;">${log.new_value}</span>
                                                </div>
                                                ` : ''}
                                                <div class="audit-log-meta">
                                                    <span style="font-weight: 600;">${log.admin_username || 'System'}</span>
                                                    ${log.admin_role ? `<span style="color: #6c757d;">(${log.admin_role})</span>` : ''}
                                                    <span style="color: #6c757d; margin-left: 0.5rem;">• ${date}</span>
                                                </div>
                                                ${metadata}
                                            </div>
                                        </div>
                                    `;
                                }).join('')}
                            </div>
                        </div>
                        ` : ''}
                        
                        <div class="form-actions">
                            <button type="button" class="btn btn-secondary" onclick="closeDetailModal()">
                                <i class="fas fa-times"></i>
                                Schließen
                            </button>
                            <select id="status-select" class="btn" onchange="updateStatus(${fb.id}, this.value)" style="padding: 0.75rem 1.5rem; border: 1px solid var(--border); background: white; color: var(--dark);">
                                <option value="new" ${fb.status === 'new' ? 'selected' : ''}>Neu</option>
                                <option value="read" ${fb.status === 'read' ? 'selected' : ''}>Gelesen</option>
                                <option value="archived" ${fb.status === 'archived' ? 'selected' : ''}>Archiviert</option>
                            </select>
                            ${!fb.admin_response ? `
                            <button type="button" class="btn btn-success" onclick="openResponseModal(${fb.id})">
                                <i class="fas fa-reply"></i>
                                Antworten
                            </button>
                            ` : ''}
                        </div>
                    `;
                } else {
                    body.innerHTML = '<div style="text-align: center; padding: 2rem; color: var(--danger);"><i class="fas fa-exclamation-circle"></i><p>Fehler beim Laden der Details</p></div>';
                }
            } catch (error) {
                body.innerHTML = '<div style="text-align: center; padding: 2rem; color: var(--danger);"><i class="fas fa-exclamation-circle"></i><p>Fehler beim Laden der Details</p></div>';
            }
        }

        function closeDetailModal() {
            document.getElementById('detail-modal').classList.remove('active');
        }

        // Response Modal
        function openResponseModal(feedbackId) {
            document.getElementById('response-feedback-id').value = feedbackId;
            document.getElementById('response-modal').classList.add('active');
            document.getElementById('response-text').focus();
        }

        function closeResponseModal() {
            document.getElementById('response-modal').classList.remove('active');
            document.getElementById('response-form').reset();
        }

        // Status Update
        async function updateStatus(feedbackId, status) {
            try {
                const response = await fetch('api/update_feedback_status.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        feedback_id: feedbackId,
                        status: status
                    })
                });
                
                const data = await response.json();
                if (data.success) {
                    location.reload();
                } else {
                    alert('Fehler beim Aktualisieren des Status');
                }
            } catch (error) {
                alert('Fehler beim Aktualisieren des Status');
            }
        }

        // Response Form
        document.getElementById('response-form').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            const data = {
                feedback_id: formData.get('feedback_id'),
                admin_response: formData.get('admin_response')
            };
            
            const submitBtn = e.target.querySelector('button[type="submit"]');
            const originalText = submitBtn.innerHTML;
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Wird gesendet...';
            
            try {
                const response = await fetch('api/add_feedback_response.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(data)
                });
                
                // Prüfe ob Response OK ist
                if (!response.ok) {
                    const errorText = await response.text();
                    let errorMessage = 'Fehler beim Senden der Antwort';
                    try {
                        const errorJson = JSON.parse(errorText);
                        errorMessage = errorJson.error || errorMessage;
                    } catch (e) {
                        errorMessage = `Server-Fehler (${response.status})`;
                    }
                    alert(errorMessage);
                    submitBtn.disabled = false;
                    submitBtn.innerHTML = originalText;
                    return;
                }
                
                // Prüfe Content-Type
                const contentType = response.headers.get('content-type');
                if (!contentType || !contentType.includes('application/json')) {
                    const errorText = await response.text();
                    console.error('Ungültige Antwort (kein JSON):', errorText.substring(0, 200));
                    alert('Server-Fehler: Die Antwort ist kein gültiges JSON. Bitte prüfen Sie die Server-Logs.');
                    submitBtn.disabled = false;
                    submitBtn.innerHTML = originalText;
                    return;
                }
                
                const result = await response.json();
                
                if (result.success) {
                    // Erfolg - auch wenn E-Mail-Warnung vorhanden ist
                    if (result.email_warning) {
                        // Zeige Info, aber nicht als Fehler
                        console.warn('E-Mail-Warnung:', result.email_warning);
                    }
                    closeResponseModal();
                    // Kurze Verzögerung für besseres UX
                    setTimeout(() => {
                        location.reload();
                    }, 500);
                } else {
                    // Nur bei echten Fehlern
                    alert(result.error || 'Fehler beim Senden der Antwort');
                    submitBtn.disabled = false;
                    submitBtn.innerHTML = originalText;
                }
            } catch (error) {
                console.error('Feedback-Antwort Fehler:', error);
                
                // Prüfe ob es ein JSON-Parse-Fehler ist
                if (error instanceof SyntaxError && error.message.includes('JSON')) {
                    console.error('JSON-Parse-Fehler - Server hat wahrscheinlich HTML statt JSON zurückgegeben');
                    alert('Server-Fehler: Die Antwort konnte nicht verarbeitet werden. Bitte prüfen Sie die Server-Logs oder kontaktieren Sie den Administrator.');
                } else if (error instanceof TypeError && error.message.includes('fetch')) {
                    // Netzwerkfehler - möglicherweise wurde die Antwort trotzdem gespeichert
                    alert('Verbindungsfehler. Bitte prüfen Sie, ob die Antwort gespeichert wurde.');
                } else {
                    alert('Fehler beim Senden der Antwort: ' + error.message);
                }
                submitBtn.disabled = false;
                submitBtn.innerHTML = originalText;
            }
        });

        // Modal schließen bei Klick außerhalb
        document.querySelectorAll('.modal').forEach(modal => {
            modal.addEventListener('click', function(e) {
                if (e.target === this) {
                    this.classList.remove('active');
                }
            });
        });

        // Export
        function exportFeedback() {
            const params = new URLSearchParams(window.location.search);
            window.location.href = 'api/export_feedback.php?' + params.toString();
        }
    </script>
</body>
</html>
