<?php
/**
 * Update User Data API
 * Allows admins to update user information
 */

// Disable error display and ensure only JSON is output
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Set content type to JSON
header('Content-Type: application/json');

// Start output buffering to catch any unwanted output
ob_start();

// Track if we've already sent a response
$responseSent = false;

// Register error handler to catch all errors
register_shutdown_function(function() use (&$responseSent) {
    // Only handle fatal errors if we haven't sent a response yet
    if ($responseSent) {
        return;
    }
    
    $error = error_get_last();
    if ($error !== NULL && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        ob_clean();
        http_response_code(500);
        echo json_encode([
            'success' => false, 
            'message' => 'Fatal error: ' . $error['message'] . ' in ' . basename($error['file']) . ' on line ' . $error['line']
        ]);
        ob_end_flush();
        exit;
    }
});

// Set error handler for non-fatal errors
set_error_handler(function($errno, $errstr, $errfile, $errline) {
    error_log("PHP Error [$errno]: $errstr in $errfile on line $errline");
    // Don't output errors, just log them
    return true;
});

session_start();

// Check if user is logged in as admin
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    ob_clean();
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized - Please log in as admin']);
    $responseSent = true;
    ob_end_flush();
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

if (!$input || !isset($input['user_id'])) {
    ob_clean();
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid input']);
    $responseSent = true;
    ob_end_flush();
    exit;
}

try {
    // Check if database config file exists
    $dbConfigPath = __DIR__ . '/../../config/database.php';
    if (!file_exists($dbConfigPath)) {
        throw new Exception("Database config file not found: " . $dbConfigPath);
    }
    
    require_once $dbConfigPath;
    
    // Check if Database class exists
    if (!class_exists('Database')) {
        throw new Exception("Database class not found in config file");
    }
    
    $db = new Database();
    
    // Check if getConnection method exists
    if (!method_exists($db, 'getConnection')) {
        throw new Exception("getConnection method not found in Database class");
    }
    
    $conn = $db->getConnection();
    
    if (!$conn) {
        throw new Exception("Database connection failed - getConnection returned null");
    }
    
    // Verify it's a PDO connection
    if (!($conn instanceof PDO)) {
        throw new Exception("Database connection is not a PDO instance");
    }
    
    $userId = $input['user_id'];
    
    // Build update query dynamically based on provided fields
    $updateFields = [];
    $params = [':user_id' => $userId];
    
    if (isset($input['first_name'])) {
        $updateFields[] = 'first_name = :first_name';
        $params[':first_name'] = $input['first_name'];
    }
    
    if (isset($input['last_name'])) {
        $updateFields[] = 'last_name = :last_name';
        $params[':last_name'] = $input['last_name'];
    }
    
    if (isset($input['email'])) {
        $updateFields[] = 'email = :email';
        $params[':email'] = $input['email'];
    }
    
    if (isset($input['phone_number'])) {
        $updateFields[] = 'phone_number = :phone_number';
        $params[':phone_number'] = $input['phone_number'];
    }
    
    if (isset($input['license_plate'])) {
        $updateFields[] = 'license_plate = :license_plate';
        $params[':license_plate'] = $input['license_plate'];
    }
    
    if (isset($input['preferred_dealership_id'])) {
        $updateFields[] = 'preferred_dealership_id = :preferred_dealership_id';
        // Convert empty string to null, otherwise use the value or null
        $dealershipId = $input['preferred_dealership_id'];
        $params[':preferred_dealership_id'] = ($dealershipId === '' || $dealershipId === null) ? null : (int)$dealershipId;
    }
    
    if (empty($updateFields)) {
        ob_clean();
        echo json_encode(['success' => false, 'message' => 'No fields to update']);
        $responseSent = true;
        ob_end_flush();
        exit;
    }
    
    $sql = "UPDATE users SET " . implode(', ', $updateFields) . " WHERE id = :user_id";
    
    $stmt = $conn->prepare($sql);
    
    // Log for debugging (remove in production)
    error_log("Update SQL: " . $sql);
    error_log("Update Params: " . print_r($params, true));
    
    $result = $stmt->execute($params);
    
    // Check for SQL errors
    if (!$result) {
        $errorInfo = $stmt->errorInfo();
        error_log("SQL Error: " . print_r($errorInfo, true));
        throw new Exception("SQL execution failed: " . ($errorInfo[2] ?? 'Unknown error'));
    }
    
    // If we get here, the update was successful
    // Get updated user data with autohaus name
    try {
        $selectSql = "SELECT u.first_name, u.last_name, u.email, u.phone_number, u.license_plate, u.preferred_dealership_id, a.name as dealership_name 
                      FROM users u 
                      LEFT JOIN autohaus a ON u.preferred_dealership_id = a.id 
                      WHERE u.id = :user_id";
        $selectStmt = $conn->prepare($selectSql);
        $selectResult = $selectStmt->execute([':user_id' => $userId]);
        
        if (!$selectResult) {
            $errorInfo = $selectStmt->errorInfo();
            error_log("SELECT Error: " . print_r($errorInfo, true));
            throw new Exception("Failed to fetch updated user data: " . ($errorInfo[2] ?? 'Unknown error'));
        }
        
        $userData = $selectStmt->fetch(PDO::FETCH_ASSOC);
        
        // Clear any output buffer
        ob_clean();
        
        // Return data directly (not nested in 'data' key) for easier JavaScript access
        echo json_encode(array_merge([
            'success' => true, 
            'message' => 'User data updated successfully'
        ], $userData ?: []));
        $responseSent = true;
        ob_end_flush();
    } catch (Exception $selectError) {
        // Even if SELECT fails, the UPDATE was successful, so return success with available data
        error_log("Error fetching updated data (but update was successful): " . $selectError->getMessage());
        ob_clean();
        echo json_encode([
            'success' => true, 
            'message' => 'User data updated successfully (but could not fetch updated data)',
            'warning' => $selectError->getMessage()
        ]);
        $responseSent = true;
        ob_end_flush();
    }
    
} catch (PDOException $e) {
    $errorInfo = $e->errorInfo();
    error_log("PDO Error updating user data: " . $e->getMessage());
    error_log("PDO Error Code: " . $e->getCode());
    error_log("PDO Error Info: " . print_r($errorInfo, true));
    ob_clean();
    http_response_code(500);
    $errorMessage = 'Database error: ' . ($errorInfo[2] ?? $e->getMessage());
    echo json_encode(['success' => false, 'message' => $errorMessage]);
    $responseSent = true;
    ob_end_flush();
    exit;
} catch (Exception $e) {
    error_log("Error updating user data: " . $e->getMessage());
    error_log("Stack trace: " . $e->getTraceAsString());
    ob_clean();
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Internal server error: ' . $e->getMessage()]);
    $responseSent = true;
    ob_end_flush();
    exit;
}
?>
