<?php
/**
 * Tasks & Reminders Manager API
 * Verwaltet Aufgaben und Erinnerungen für das Dashboard
 */

session_start();
require_once '../../config/database_optimized.php';

// Check if user is logged in
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'error' => 'Nicht angemeldet']);
    exit;
}

$db = DatabaseOptimized::getInstance();
$conn = $db->getConnection();
$current_user_id = $_SESSION['admin_user_id'] ?? null;

header('Content-Type: application/json');

$action = $_POST['action'] ?? $_GET['action'] ?? 'list';

try {
    switch ($action) {
        case 'create':
            $title = $_POST['title'] ?? '';
            $description = $_POST['description'] ?? '';
            $priority = $_POST['priority'] ?? 'medium';
            $due_date = $_POST['due_date'] ?? null;
            $due_time = $_POST['due_time'] ?? null;
            $reminder_date = $_POST['reminder_date'] ?? null;
            $reminder_time = $_POST['reminder_time'] ?? null;
            $assigned_to = $_POST['assigned_to'] ?? null;
            $category = $_POST['category'] ?? 'general';
            $related_id = $_POST['related_id'] ?? null;
            $related_type = $_POST['related_type'] ?? null;
            $is_reminder = isset($_POST['is_reminder']) ? 1 : 0;
            
            if (empty($title)) {
                throw new Exception('Titel ist erforderlich');
            }
            
            $stmt = $conn->prepare("
                INSERT INTO dashboard_tasks 
                (user_id, title, description, priority, status, due_date, due_time, 
                 reminder_date, reminder_time, assigned_to, created_by, category, 
                 related_id, related_type, is_reminder, is_active)
                VALUES (?, ?, ?, ?, 'pending', ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 1)
            ");
            
            $stmt->execute([
                null, // user_id - NULL = für alle
                $title,
                $description,
                $priority,
                $due_date ?: null,
                $due_time ?: null,
                $reminder_date ?: null,
                $reminder_time ?: null,
                $assigned_to ?: null,
                $current_user_id,
                $category,
                $related_id ?: null,
                $related_type ?: null,
                $is_reminder
            ]);
            
            echo json_encode(['success' => true, 'message' => 'Aufgabe erfolgreich erstellt', 'id' => $conn->lastInsertId()]);
            break;
            
        case 'update':
            $task_id = $_POST['id'] ?? null;
            if (!$task_id) {
                throw new Exception('Aufgaben-ID erforderlich');
            }
            
            $updates = [];
            $params = [];
            
            if (isset($_POST['title'])) {
                $updates[] = 'title = ?';
                $params[] = $_POST['title'];
            }
            if (isset($_POST['description'])) {
                $updates[] = 'description = ?';
                $params[] = $_POST['description'];
            }
            if (isset($_POST['priority'])) {
                $updates[] = 'priority = ?';
                $params[] = $_POST['priority'];
            }
            if (isset($_POST['status'])) {
                $updates[] = 'status = ?';
                $params[] = $_POST['status'];
                if ($_POST['status'] === 'completed') {
                    $updates[] = 'completed_at = NOW()';
                }
            }
            if (isset($_POST['due_date'])) {
                $updates[] = 'due_date = ?';
                $params[] = $_POST['due_date'] ?: null;
            }
            if (isset($_POST['due_time'])) {
                $updates[] = 'due_time = ?';
                $params[] = $_POST['due_time'] ?: null;
            }
            if (isset($_POST['assigned_to'])) {
                $updates[] = 'assigned_to = ?';
                $params[] = $_POST['assigned_to'] ?: null;
            }
            
            if (empty($updates)) {
                throw new Exception('Keine Änderungen angegeben');
            }
            
            $params[] = $task_id;
            $sql = "UPDATE dashboard_tasks SET " . implode(', ', $updates) . " WHERE id = ?";
            $stmt = $conn->prepare($sql);
            $stmt->execute($params);
            
            echo json_encode(['success' => true, 'message' => 'Aufgabe erfolgreich aktualisiert']);
            break;
            
        case 'delete':
            $task_id = $_POST['id'] ?? $_GET['id'] ?? null;
            if (!$task_id) {
                throw new Exception('Aufgaben-ID erforderlich');
            }
            
            $stmt = $conn->prepare("UPDATE dashboard_tasks SET is_active = 0 WHERE id = ?");
            $stmt->execute([$task_id]);
            
            echo json_encode(['success' => true, 'message' => 'Aufgabe erfolgreich gelöscht']);
            break;
            
        case 'complete':
            $task_id = $_POST['id'] ?? $_GET['id'] ?? null;
            if (!$task_id) {
                throw new Exception('Aufgaben-ID erforderlich');
            }
            
            $stmt = $conn->prepare("UPDATE dashboard_tasks SET status = 'completed', completed_at = NOW() WHERE id = ?");
            $stmt->execute([$task_id]);
            
            echo json_encode(['success' => true, 'message' => 'Aufgabe als erledigt markiert']);
            break;
            
        case 'list':
        default:
            $user_id = $_GET['user_id'] ?? $current_user_id;
            $status = $_GET['status'] ?? null;
            $priority = $_GET['priority'] ?? null;
            $category = $_GET['category'] ?? null;
            $search = $_GET['search'] ?? null;
            
            $sql = "
                SELECT t.*, 
                       u1.username as assigned_to_name,
                       u1.full_name as assigned_to_full_name,
                       u2.username as created_by_name,
                       u2.full_name as created_by_full_name
                FROM dashboard_tasks t
                LEFT JOIN admin_users u1 ON t.assigned_to = u1.id
                LEFT JOIN admin_users u2 ON t.created_by = u2.id
                WHERE t.is_active = 1
            ";
            
            $params = [];
            
            if ($status) {
                $sql .= " AND t.status = ?";
                $params[] = $status;
            }
            
            if ($priority) {
                $sql .= " AND t.priority = ?";
                $params[] = $priority;
            }
            
            if ($category) {
                $sql .= " AND t.category = ?";
                $params[] = $category;
            }
            
            if ($search) {
                $sql .= " AND (t.title LIKE ? OR t.description LIKE ?)";
                $searchTerm = "%{$search}%";
                $params[] = $searchTerm;
                $params[] = $searchTerm;
            }
            
            $sql .= " AND (t.user_id IS NULL OR t.user_id = ? OR t.assigned_to = ?)";
            $params[] = $user_id;
            $params[] = $user_id;
            
            $sql .= " ORDER BY 
                CASE t.priority 
                    WHEN 'urgent' THEN 1
                    WHEN 'high' THEN 2
                    WHEN 'medium' THEN 3
                    WHEN 'low' THEN 4
                END,
                t.due_date ASC,
                t.due_time ASC";
            
            $stmt = $conn->prepare($sql);
            $stmt->execute($params);
            $tasks = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo json_encode(['success' => true, 'tasks' => $tasks]);
            break;
    }
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}
