<?php
/**
 * Suspend User API
 * Allows admins to suspend/unsuspend users
 */

// Disable error display to prevent HTML in JSON response
ini_set('display_errors', 0);
error_reporting(E_ALL);
ini_set('log_errors', 1);

// Set JSON header BEFORE any output
header('Content-Type: application/json');

// Start output buffering to catch any unexpected output
ob_start();

session_start();

// Debug session info
error_log("Suspend API - Session admin_logged_in: " . (isset($_SESSION['admin_logged_in']) ? 'true' : 'false'));
error_log("Suspend API - Session admin_user_id: " . (isset($_SESSION['admin_user_id']) ? $_SESSION['admin_user_id'] : 'not set'));

// Check if user is logged in as admin
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    error_log("Suspend API - Unauthorized access attempt");
    ob_clean();
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized - Please log in as admin']);
    ob_end_flush();
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

if (!$input || !isset($input['user_id'])) {
    ob_clean();
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid input']);
    ob_end_flush();
    exit;
}

try {
    $dbPath = __DIR__ . '/../../config/database.php';
    if (!file_exists($dbPath)) {
        throw new Exception("Database config file not found: $dbPath");
    }
    require_once $dbPath;
    
    if (!class_exists('Database')) {
        throw new Exception("Database class not found");
    }
    
    $db = new Database();
    $conn = $db->getConnection();
    
    if (!$conn) {
        throw new Exception("Failed to get database connection");
    }
    
    $userId = $input['user_id'];
    $action = $input['action'] ?? 'suspend'; // suspend or unsuspend
    $reason = $input['reason'] ?? '';
    $admin_password = $input['admin_password'] ?? '';
    $admin_id = $_SESSION['admin_user_id'] ?? null;
    
    // Check if password is provided
    if (empty($admin_password)) {
        ob_clean();
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Passwort zur Bestätigung erforderlich']);
        ob_end_flush();
        exit;
    }
    
    // Verify admin password and role
    $stmt = $conn->prepare("SELECT id, password_hash, role FROM admin_users WHERE id = ?");
    $stmt->execute([$admin_id]);
    $admin = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$admin) {
        ob_clean();
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Admin-Benutzer nicht gefunden']);
        ob_end_flush();
        exit;
    }
    
    // Check if role is admin
    if ($admin['role'] !== 'admin') {
        ob_clean();
        http_response_code(403);
        echo json_encode(['success' => false, 'message' => 'Nur Administratoren können Benutzer sperren/entsperren']);
        ob_end_flush();
        exit;
    }
    
    // Verify password
    if (!password_verify($admin_password, $admin['password_hash'])) {
        ob_clean();
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Falsches Passwort. Bitte versuchen Sie es erneut.']);
        ob_end_flush();
        exit;
    }
    
    error_log("Suspend API - Processing user_id: $userId, action: $action");
    
    // Get current user status
    $checkSql = "SELECT status FROM users WHERE id = :user_id";
    $checkStmt = $conn->prepare($checkSql);
    $checkStmt->execute([':user_id' => $userId]);
    $user = $checkStmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        error_log("Suspend API - User not found: $userId");
        ob_clean();
        echo json_encode(['success' => false, 'message' => 'User not found']);
        ob_end_flush();
        exit;
    }
    
    $newStatus = ($action === 'suspend') ? 'suspended' : 'pending';
    
    // Update user status
    $updateSql = "UPDATE users SET status = :status WHERE id = :user_id";
    $updateStmt = $conn->prepare($updateSql);
    $result = $updateStmt->execute([
        ':status' => $newStatus,
        ':user_id' => $userId
    ]);
    
    if ($result) {
        // Log the action (only if admin_user_id is available)
        if (isset($_SESSION['admin_user_id'])) {
            try {
                // Check if admin_actions table exists
                $tableCheck = $conn->query("SHOW TABLES LIKE 'admin_actions'");
                if ($tableCheck->rowCount() > 0) {
                    $logSql = "INSERT INTO admin_actions (admin_id, user_id, action, reason, created_at) VALUES (:admin_id, :user_id, :action, :reason, NOW())";
                    $logStmt = $conn->prepare($logSql);
                    $actionName = ($action === 'suspend') ? 'suspend' : 'unsuspend';
                    $logReason = $reason ? ($action === 'suspend' ? 'Sperrgrund: ' : 'Entsperrgrund: ') . $reason : '';
                    $logStmt->execute([
                        ':admin_id' => $admin_id,
                        ':user_id' => $userId,
                        ':action' => $actionName,
                        ':reason' => $logReason
                    ]);
                    error_log("Suspend API - Admin action logged successfully");
                } else {
                    error_log("Suspend API - admin_actions table does not exist, skipping log");
                }
            } catch (Exception $logError) {
                error_log("Suspend API - Failed to log admin action: " . $logError->getMessage());
                error_log("Suspend API - Log error details: " . $logError->getTraceAsString());
                // Continue execution even if logging fails
            }
        } else {
            error_log("Suspend API - No admin_user_id in session, skipping admin action log");
        }
        
        // Create notification for user
        try {
            $helperPath = __DIR__ . '/../../includes/push_notification_helper.php';
            if (file_exists($helperPath)) {
                require_once $helperPath;
                
                if ($action === 'suspend') {
                    $title = 'Account gesperrt';
                    $message = 'Ihr Account wurde von einem Administrator gesperrt.' . ($reason ? ' Grund: ' . $reason : '');
                    $type = 'warning';
                } else {
                    $title = 'Account entsperrt';
                    $message = 'Ihr Account wurde von einem Administrator entsperrt.' . ($reason ? ' Grund: ' . $reason : '');
                    $type = 'success';
                }
                
                if (function_exists('createNotificationWithPush')) {
                    $notificationResult = createNotificationWithPush($conn, $userId, $title, $message, $type, '/profile.php');
                    error_log("Suspend API - User notification created successfully. Push sent: " . ($notificationResult['push_sent'] ?? false ? 'Yes' : 'No'));
                } else {
                    error_log("Suspend API - createNotificationWithPush function not found, creating notification manually");
                    // Fallback: Create notification without push
                    $stmt = $conn->prepare("
                        INSERT INTO user_notifications (user_id, type, title, message, is_read, created_at)
                        VALUES (?, ?, ?, ?, 0, NOW())
                    ");
                    $stmt->execute([$userId, $type, $title, $message]);
                    error_log("Suspend API - Notification created manually (without push)");
                }
            } else {
                error_log("Suspend API - push_notification_helper.php not found at: $helperPath, creating notification manually");
                // Fallback: Create notification without push
                if ($action === 'suspend') {
                    $title = 'Account gesperrt';
                    $message = 'Ihr Account wurde von einem Administrator gesperrt.' . ($reason ? ' Grund: ' . $reason : '');
                    $type = 'warning';
                } else {
                    $title = 'Account entsperrt';
                    $message = 'Ihr Account wurde von einem Administrator entsperrt.' . ($reason ? ' Grund: ' . $reason : '');
                    $type = 'success';
                }
                $stmt = $conn->prepare("
                    INSERT INTO user_notifications (user_id, type, title, message, is_read, created_at)
                    VALUES (?, ?, ?, ?, 0, NOW())
                ");
                $stmt->execute([$userId, $type, $title, $message]);
                error_log("Suspend API - Notification created manually (without push)");
            }
        } catch (Exception $notificationError) {
            error_log("Suspend API - Failed to create user notification: " . $notificationError->getMessage());
            error_log("Suspend API - Notification error details: " . $notificationError->getTraceAsString());
            // Continue execution even if notification fails
        }
        
        ob_clean();
        echo json_encode([
            'success' => true, 
            'message' => 'User ' . $action . 'ed successfully',
            'new_status' => $newStatus
        ]);
        ob_end_flush();
    } else {
        error_log("Suspend API - Failed to update user status");
        ob_clean();
        echo json_encode(['success' => false, 'message' => 'Failed to ' . $action . ' user']);
        ob_end_flush();
    }
    
} catch (Exception $e) {
    // Clear any output buffer
    ob_clean();
    
    error_log("Suspend API - Error " . ($action ?? 'processing') . " user: " . $e->getMessage());
    error_log("Suspend API - Stack trace: " . $e->getTraceAsString());
    error_log("Suspend API - File: " . $e->getFile() . " Line: " . $e->getLine());
    
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'message' => 'Internal server error: ' . $e->getMessage(),
        'debug' => [
            'action' => $action ?? 'unknown',
            'user_id' => $userId ?? 'unknown',
            'error' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine()
        ]
    ]);
} catch (Error $e) {
    // Catch PHP 7+ errors (TypeError, ParseError, etc.)
    ob_clean();
    
    error_log("Suspend API - PHP Error: " . $e->getMessage());
    error_log("Suspend API - Stack trace: " . $e->getTraceAsString());
    
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'message' => 'Internal server error: ' . $e->getMessage(),
        'debug' => [
            'error' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine()
        ]
    ]);
    ob_end_flush();
}

// End output buffering only if still active
if (ob_get_level() > 0) {
    ob_end_flush();
}
?>
