<?php
/**
 * Bulk Retry WhatsApp Messages API
 * Erlaubt das erneute Versenden mehrerer fehlgeschlagener WhatsApp-Nachrichten
 */

session_start();
require_once '../../config/database.php';
require_once '../includes/helpers.php';
require_once '../includes/hellomateo_api.php';

header('Content-Type: application/json');

// Check if user is logged in
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    echo json_encode(['success' => false, 'error' => 'Nicht angemeldet']);
    exit;
}

$db = new Database();
$conn = $db->getConnection();

// Get current user autohaus
$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_autohaus_id = null;

if ($current_user_id) {
    $stmt = $conn->prepare("SELECT autohaus_id FROM admin_users WHERE id = ?");
    $stmt->execute([$current_user_id]);
    $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
    $current_user_autohaus_id = $user_data['autohaus_id'] ?? null;
}

// Get selected autohaus from session (for admins)
$selected_autohaus_id = $_SESSION['selected_autohaus_id'] ?? null;
$active_autohaus_id = $selected_autohaus_id ?? $current_user_autohaus_id;

$input = json_decode(file_get_contents('php://input'), true);
$filters = $input['filters'] ?? [];

// Baue Query für fehlgeschlagene Logs
$where = ["wl.status = 'error'"];
$params = [];

if (!empty($filters['date_from'])) {
    $where[] = 'DATE(wl.created_at) >= ?';
    $params[] = $filters['date_from'];
}

if (!empty($filters['date_to'])) {
    $where[] = 'DATE(wl.created_at) <= ?';
    $params[] = $filters['date_to'];
}

if (!empty($filters['autohaus_id'])) {
    $where[] = 'wl.autohaus_id = ?';
    $params[] = $filters['autohaus_id'];
} elseif ($current_user_autohaus_id) {
    // Filter nach Benutzer-Autohaus, wenn kein spezifisches Autohaus gewählt
    $where[] = 'wl.autohaus_id = ?';
    $params[] = $current_user_autohaus_id;
}

$whereClause = implode(' AND ', $where);

// Lade fehlgeschlagene Logs
$stmt = $conn->prepare("
    SELECT wl.*
    FROM whatsapp_logs wl
    WHERE $whereClause
    ORDER BY wl.created_at DESC
    LIMIT 100
");
$stmt->execute($params);
$failedLogs = $stmt->fetchAll(PDO::FETCH_ASSOC);

if (empty($failedLogs)) {
    echo json_encode(['success' => false, 'error' => 'Keine fehlgeschlagenen Nachrichten gefunden', 'count' => 0]);
    exit;
}

$successCount = 0;
$errorCount = 0;
$processedLogs = [];

foreach ($failedLogs as $log) {
    $logAutohausId = $log['autohaus_id'] ?? $active_autohaus_id;
    $hellomateo = new HelloMateoAPI(null, $logAutohausId);
    
    $phone = $log['phone_number'];
    $message = $log['message'];
    $templateId = $log['template_id'] ?? null;
    $actionType = $log['action_type'];
    
    try {
        $result = $hellomateo->sendMessage($phone, $message, $templateId);
        
        $newStatus = $result['success'] ? 'success' : 'error';
        $errorMessage = $result['success'] ? null : ($result['error'] ?? $result['message'] ?? 'Unbekannter Fehler');
        $responseData = json_encode($result);
        
        // Erstelle neuen Log-Eintrag
        $insertStmt = $conn->prepare("
            INSERT INTO whatsapp_logs (
                autohaus_id, appointment_id, customer_name, phone_number,
                action_type, message, template_id, status, error_message,
                response_data, created_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        
        $insertStmt->execute([
            $log['autohaus_id'],
            $log['appointment_id'],
            $log['customer_name'],
            $phone,
            $actionType . '_retry',
            $message,
            $templateId,
            $newStatus,
            $errorMessage,
            $responseData
        ]);
        
        if ($result['success']) {
            $successCount++;
        } else {
            $errorCount++;
        }
        
        $processedLogs[] = [
            'log_id' => $log['id'],
            'phone' => $phone,
            'success' => $result['success']
        ];
        
    } catch (Exception $e) {
        $errorCount++;
        
        // Erstelle Log-Eintrag für Fehler
        $insertStmt = $conn->prepare("
            INSERT INTO whatsapp_logs (
                autohaus_id, appointment_id, customer_name, phone_number,
                action_type, message, template_id, status, error_message,
                response_data, created_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, 'error', ?, ?, NOW())
        ");
        
        $errorMessage = $e->getMessage();
        $insertStmt->execute([
            $log['autohaus_id'],
            $log['appointment_id'],
            $log['customer_name'],
            $phone,
            $actionType . '_retry',
            $message,
            $templateId,
            $errorMessage,
            json_encode(['exception' => $errorMessage])
        ]);
        
        $processedLogs[] = [
            'log_id' => $log['id'],
            'phone' => $phone,
            'success' => false,
            'error' => $errorMessage
        ];
    }
    
    // Kleine Pause zwischen Versendungen, um API-Limits zu vermeiden
    usleep(100000); // 0.1 Sekunde
}

echo json_encode([
    'success' => true,
    'count' => count($failedLogs),
    'success_count' => $successCount,
    'error_count' => $errorCount,
    'processed_logs' => $processedLogs
]);
?>




