<?php
/**
 * Quick Actions API
 * API für Quick Actions Management (für Modal)
 */

session_start();
require_once '../../config/database_optimized.php';

// Check if user is logged in
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'error' => 'Nicht angemeldet']);
    exit;
}

$db = DatabaseOptimized::getInstance();
$conn = $db->getConnection();
$current_user_id = $_SESSION['admin_user_id'] ?? null;

header('Content-Type: application/json');

$action = $_POST['action'] ?? $_GET['action'] ?? 'list';

try {
    switch ($action) {
        case 'list':
            $stmt = $conn->prepare("
                SELECT * FROM quick_actions 
                WHERE (user_id = ? OR user_id IS NULL) AND is_active = 1
                ORDER BY user_id DESC, position_order ASC, id ASC
            ");
            $stmt->execute([$current_user_id]);
            $quick_actions = $stmt->fetchAll(PDO::FETCH_ASSOC);
            echo json_encode(['success' => true, 'actions' => $quick_actions]);
            break;
            
        case 'create':
            $title = $_POST['title'] ?? '';
            $url = $_POST['url'] ?? '';
            $icon = $_POST['icon'] ?? 'fas fa-circle';
            $color = $_POST['color'] ?? '#007AFF';
            
            if (empty($title) || empty($url)) {
                throw new Exception('Titel und URL sind erforderlich');
            }
            
            // Get max position_order
            $stmt = $conn->prepare("SELECT MAX(position_order) as max_pos FROM quick_actions WHERE user_id = ?");
            $stmt->execute([$current_user_id]);
            $max_pos = $stmt->fetch(PDO::FETCH_ASSOC)['max_pos'] ?? 0;
            
            $stmt = $conn->prepare("
                INSERT INTO quick_actions (title, url, icon, background_color, user_id, position_order, created_at) 
                VALUES (?, ?, ?, ?, ?, ?, NOW())
            ");
            $stmt->execute([$title, $url, $icon, $color, $current_user_id, $max_pos + 1]);
            echo json_encode(['success' => true, 'message' => 'Schnellaktion erfolgreich erstellt', 'id' => $conn->lastInsertId()]);
            break;
            
        case 'update':
            $id = $_POST['id'] ?? null;
            $title = $_POST['title'] ?? null;
            $url = $_POST['url'] ?? null;
            $icon = $_POST['icon'] ?? null;
            $color = $_POST['color'] ?? null;
            
            if (!$id) {
                throw new Exception('ID ist erforderlich');
            }
            
            $updates = [];
            $params = [];
            
            if ($title !== null) { $updates[] = 'title = ?'; $params[] = $title; }
            if ($url !== null) { $updates[] = 'url = ?'; $params[] = $url; }
            if ($icon !== null) { $updates[] = 'icon = ?'; $params[] = $icon; }
            if ($color !== null) { $updates[] = 'background_color = ?'; $params[] = $color; }
            
            if (empty($updates)) {
                throw new Exception('Keine Änderungen angegeben');
            }
            
            $updates[] = 'updated_at = NOW()';
            $params[] = $id;
            $params[] = $current_user_id;
            
            $sql = "UPDATE quick_actions SET " . implode(', ', $updates) . " WHERE id = ? AND user_id = ?";
            $stmt = $conn->prepare($sql);
            $stmt->execute($params);
            
            echo json_encode(['success' => true, 'message' => 'Schnellaktion erfolgreich aktualisiert']);
            break;
            
        case 'delete':
            $id = $_POST['id'] ?? $_GET['id'] ?? null;
            if (!$id) {
                throw new Exception('ID ist erforderlich');
            }
            
            $stmt = $conn->prepare("DELETE FROM quick_actions WHERE id = ? AND user_id = ?");
            $stmt->execute([$id, $current_user_id]);
            echo json_encode(['success' => true, 'message' => 'Schnellaktion erfolgreich gelöscht']);
            break;
            
        case 'update_positions':
            $positions = $_POST['positions'] ?? [];
            if (!is_array($positions)) {
                throw new Exception('Ungültige Positionsdaten');
            }
            
            foreach ($positions as $position => $id) {
                $stmt = $conn->prepare("UPDATE quick_actions SET position_order = ?, updated_at = NOW() WHERE id = ? AND user_id = ?");
                $stmt->execute([$position, $id, $current_user_id]);
            }
            
            echo json_encode(['success' => true, 'message' => 'Reihenfolge aktualisiert']);
            break;
            
        default:
            throw new Exception('Ungültige Aktion');
    }
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}

