<?php
/**
 * API-Endpunkt zum Abrufen der Login-Historie eines Benutzers
 * Nur für Admin-Benutzer zugänglich
 */

session_start();
header('Content-Type: application/json');

// Prüfe ob Admin eingeloggt ist
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Nicht autorisiert']);
    exit;
}

require_once __DIR__ . '/../../config/database.php';

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    $user_id = intval($_GET['user_id'] ?? 0);
    
    if (!$user_id) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Benutzer-ID erforderlich']);
        exit;
    }
    
    // Prüfe ob Benutzer existiert
    $stmt = $conn->prepare("SELECT id, first_name, last_name, email FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'Benutzer nicht gefunden']);
        exit;
    }
    
    // Hole Login-Historie aus user_activity_log
    $stmt = $conn->prepare("
        SELECT 
            id,
            action_type,
            description,
            metadata,
            created_at
        FROM user_activity_log 
        WHERE user_id = ? AND action_type = 'login'
        ORDER BY created_at DESC
        LIMIT 100
    ");
    $stmt->execute([$user_id]);
    $logins = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Hole auch zusätzliche Informationen aus users-Tabelle
    $stmt = $conn->prepare("
        SELECT 
            last_login_source,
            last_login_device,
            registration_ip
        FROM users 
        WHERE id = ?
    ");
    $stmt->execute([$user_id]);
    $userInfo = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Erweitere Login-Daten mit zusätzlichen Informationen
    $enrichedLogins = [];
    foreach ($logins as $login) {
        $metadata = $login['metadata'] ? json_decode($login['metadata'], true) : [];
        
        // Versuche IP-Adresse aus verschiedenen Quellen zu holen
        $ipAddress = $metadata['ip_address'] ?? 
                    $metadata['ip'] ?? 
                    $_SERVER['REMOTE_ADDR'] ?? 
                    'Unbekannt';
        
        // Versuche Standort-Informationen zu holen
        $location_data = $metadata['location_data'] ?? null;
        $location = $metadata['location'] ?? 
                   ($location_data && isset($location_data['location']) ? $location_data['location'] : null) ??
                   ($location_data && isset($location_data['city']) ? $location_data['city'] : null) ??
                   ($location_data && isset($location_data['country']) ? $location_data['country'] : null) ??
                   'Unbekannt';
        
        // Koordinaten für Karte
        $latitude = null;
        $longitude = null;
        if ($location_data) {
            $latitude = $location_data['latitude'] ?? null;
            $longitude = $location_data['longitude'] ?? null;
        }
        
        // Versuche Geräte-Informationen zu holen
        $device = $metadata['device'] ?? 
                 $metadata['device_info'] ?? 
                 $userInfo['last_login_device'] ?? 
                 'Unbekannt';
        
        // Versuche Quelle zu holen
        $source = $metadata['source'] ?? 
                 $metadata['login_source'] ?? 
                 $userInfo['last_login_source'] ?? 
                 'web';
        
        // Hole location_source und accuracy aus Metadaten
        $location_source = $metadata['location_source'] ?? ($location_data['source'] ?? 'geoip');
        $accuracy = isset($location_data['accuracy']) ? floatval($location_data['accuracy']) : null;
        
        $enrichedLogins[] = [
            'id' => $login['id'],
            'created_at' => $login['created_at'],
            'description' => $login['description'],
            'ip_address' => $ipAddress,
            'location' => $location,
            'latitude' => $latitude,
            'longitude' => $longitude,
            'location_data' => $location_data,
            'location_source' => $location_source,
            'accuracy' => $accuracy,
            'device' => $device,
            'source' => $source,
            'metadata' => $metadata,
            'last_login_source' => $userInfo['last_login_source'] ?? null,
            'last_login_device' => $userInfo['last_login_device'] ?? null
        ];
    }
    
    // Gesamtanzahl der Logins
    $stmt = $conn->prepare("SELECT COUNT(*) FROM user_activity_log WHERE user_id = ? AND action_type = 'login'");
    $stmt->execute([$user_id]);
    $total_logins = $stmt->fetchColumn();
    
    // Letzter Login
    $last_login = null;
    if (!empty($enrichedLogins)) {
        $last_login = $enrichedLogins[0]['created_at'];
    }
    
    echo json_encode([
        'success' => true,
        'user_id' => $user_id,
        'user_name' => $user['first_name'] . ' ' . $user['last_name'],
        'user_email' => $user['email'],
        'logins' => $enrichedLogins,
        'total_logins' => intval($total_logins),
        'last_login' => $last_login
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Fehler beim Laden der Login-Historie: ' . $e->getMessage()
    ]);
}

