<?php
/**
 * API-Endpunkt zum Abrufen aller Aktivitäten eines Benutzers
 * Nur für Admin-Benutzer zugänglich
 */

session_start();
header('Content-Type: application/json');

// Prüfe ob Admin eingeloggt ist
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Nicht autorisiert']);
    exit;
}

require_once __DIR__ . '/../../config/database.php';

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    $user_id = intval($_GET['user_id'] ?? 0);
    
    if (!$user_id) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Benutzer-ID erforderlich']);
        exit;
    }
    
    // Prüfe ob Benutzer existiert
    $stmt = $conn->prepare("SELECT id, first_name, last_name, email FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'Benutzer nicht gefunden']);
        exit;
    }
    
    // Hole alle Aktivitäten aus user_activity_log
    $stmt = $conn->prepare("
        SELECT 
            id,
            action_type,
            description,
            metadata,
            created_at
        FROM user_activity_log 
        WHERE user_id = ?
        ORDER BY created_at DESC
        LIMIT 200
    ");
    $stmt->execute([$user_id]);
    $activities = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Erweitere Aktivitäten mit zusätzlichen Informationen
    $enrichedActivities = [];
    foreach ($activities as $activity) {
        $metadata = $activity['metadata'] ? json_decode($activity['metadata'], true) : [];
        
        // Hole Event-Name aus Datenbank falls appointment_id vorhanden
        if (isset($metadata['appointment_id']) && !isset($metadata['event_name'])) {
            try {
                $stmt = $conn->prepare("
                    SELECT e.name as event_name, e.id as event_id
                    FROM appointments a
                    LEFT JOIN events e ON a.event_id = e.id
                    WHERE a.id = ?
                ");
                $stmt->execute([$metadata['appointment_id']]);
                $eventInfo = $stmt->fetch(PDO::FETCH_ASSOC);
                if ($eventInfo && $eventInfo['event_name']) {
                    $metadata['event_name'] = $eventInfo['event_name'];
                    $metadata['event_id'] = $eventInfo['event_id'];
                }
            } catch (Exception $e) {
                // Fehler ignorieren
            }
        }
        
        // Hole Event-Name aus Datenbank falls event_id vorhanden
        if (isset($metadata['event_id']) && !isset($metadata['event_name'])) {
            try {
                $stmt = $conn->prepare("SELECT name FROM events WHERE id = ?");
                $stmt->execute([$metadata['event_id']]);
                $eventName = $stmt->fetchColumn();
                if ($eventName) {
                    $metadata['event_name'] = $eventName;
                }
            } catch (Exception $e) {
                // Fehler ignorieren
            }
        }
        
        $enrichedActivities[] = [
            'id' => $activity['id'],
            'action_type' => $activity['action_type'],
            'description' => $activity['description'],
            'created_at' => $activity['created_at'],
            'metadata' => $metadata
        ];
    }
    
    // Gesamtanzahl der Aktivitäten
    $stmt = $conn->prepare("SELECT COUNT(*) FROM user_activity_log WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $total_count = $stmt->fetchColumn();
    
    echo json_encode([
        'success' => true,
        'user_id' => $user_id,
        'user_name' => $user['first_name'] . ' ' . $user['last_name'],
        'user_email' => $user['email'],
        'activities' => $enrichedActivities,
        'total_count' => intval($total_count)
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Fehler beim Laden der Aktivitäten: ' . $e->getMessage()
    ]);
}

