<?php
/**
 * API: Admin-Antwort zu Feedback hinzufügen
 */

// Output-Buffering aktivieren, um sicherzustellen, dass nur JSON ausgegeben wird
ob_start();

// Fehlerbehandlung: Alle Fehler als JSON ausgeben
set_error_handler(function($severity, $message, $file, $line) {
    if (error_reporting() === 0) return false;
    // Nur kritische Fehler werfen, Warnungen und Notices ignorieren
    if ($severity === E_ERROR || $severity === E_PARSE || $severity === E_CORE_ERROR || $severity === E_COMPILE_ERROR) {
        throw new ErrorException($message, 0, $severity, $file, $line);
    }
    // Warnungen und Notices loggen, aber nicht werfen
    error_log("PHP Warning/Notice in $file:$line: $message");
    return true; // Fehler wurde behandelt
});

// Alle PHP-Warnungen und Notices unterdrücken (werden trotzdem geloggt)
error_reporting(E_ALL & ~E_WARNING & ~E_NOTICE & ~E_DEPRECATED & ~E_STRICT);
ini_set('display_errors', '0');

session_start();
require_once '../../config/database.php';

// Alle bisherigen Outputs löschen
ob_clean();

header('Content-Type: application/json; charset=utf-8');

if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Nicht autorisiert']);
    exit;
}

$current_user_id = $_SESSION['admin_user_id'] ?? null;

if (!$current_user_id) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Benutzer nicht gefunden']);
    exit;
}

$input = json_decode(file_get_contents('php://input'), true);

if (!isset($input['feedback_id']) || !isset($input['admin_response'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Fehlende Parameter']);
    exit;
}

$feedback_id = (int)$input['feedback_id'];
$admin_response = trim($input['admin_response']);

if (empty($admin_response)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Antwort darf nicht leer sein']);
    exit;
}

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    // Hole alten Status für Audit-Log
    $oldStmt = $conn->prepare("SELECT status, admin_response FROM feedback WHERE id = ?");
    $oldStmt->execute([$feedback_id]);
    $oldData = $oldStmt->fetch(PDO::FETCH_ASSOC);
    $oldStatus = $oldData['status'] ?? null;
    $hadResponse = !empty($oldData['admin_response']);
    
    $stmt = $conn->prepare("
        UPDATE feedback 
        SET admin_response = ?, 
            admin_response_by = ?, 
            admin_response_at = NOW(),
            status = 'read'
        WHERE id = ?
    ");
    $result = $stmt->execute([$admin_response, $current_user_id, $feedback_id]);
    
    if ($result) {
        // Audit-Log: Antwort hinzugefügt
        try {
            $checkTable = $conn->query("SHOW TABLES LIKE 'feedback_audit_log'");
            if ($checkTable->rowCount() > 0) {
                $actionType = $hadResponse ? 'response_updated' : 'response_added';
                $actionDesc = $hadResponse 
                    ? "Admin-Antwort aktualisiert" 
                    : "Admin-Antwort hinzugefügt";
                
                $auditStmt = $conn->prepare("
                    INSERT INTO feedback_audit_log 
                    (feedback_id, admin_user_id, action_type, action_description, old_value, new_value, metadata)
                    VALUES (?, ?, ?, ?, ?, ?, ?)
                ");
                $auditStmt->execute([
                    $feedback_id,
                    $current_user_id,
                    $actionType,
                    $actionDesc,
                    $oldStatus,
                    'read',
                    json_encode(['response_length' => strlen($admin_response)])
                ]);
            }
        } catch (PDOException $auditError) {
            error_log("Fehler beim Erstellen des Audit-Logs: " . $auditError->getMessage());
        }
        
        // Antwort wurde erfolgreich gespeichert - das ist das Wichtigste
        $response = [
            'success' => true,
            'message' => 'Antwort erfolgreich gespeichert'
        ];
        
        // E-Mail-Benachrichtigung an Benutzer senden (nicht kritisch)
        // Wird in separatem try-catch behandelt, damit Fehler die Antwort nicht beeinträchtigen
        $emailSent = false;
        $emailError = null;
        try {
            if (file_exists(__DIR__ . '/../../api/email_notifications.php')) {
                require_once __DIR__ . '/../../api/email_notifications.php';
                
                if (class_exists('EmailNotifications')) {
                    $emailNotifier = new EmailNotifications();
                    $emailResult = $emailNotifier->sendFeedbackResponseNotification($feedback_id);
                    
                    // Log E-Mail-Ergebnis
                    if (isset($emailResult['success'])) {
                        if ($emailResult['success']) {
                            error_log("Feedback-Antwort E-Mail erfolgreich gesendet für Feedback ID: " . $feedback_id);
                            $emailSent = true;
                            $response['email_sent'] = true;
                            
                            // Audit-Log: E-Mail gesendet
                            try {
                                $checkTable = $conn->query("SHOW TABLES LIKE 'feedback_audit_log'");
                                if ($checkTable->rowCount() > 0) {
                                    $auditStmt = $conn->prepare("
                                        INSERT INTO feedback_audit_log 
                                        (feedback_id, admin_user_id, action_type, action_description)
                                        VALUES (?, ?, 'email_sent', 'E-Mail-Benachrichtigung an Benutzer gesendet')
                                    ");
                                    $auditStmt->execute([$feedback_id, $current_user_id]);
                                }
                            } catch (PDOException $e) {
                                // Nicht kritisch
                            }
                        } else {
                            $emailError = $emailResult['error'] ?? 'Unbekannter Fehler';
                            error_log("Feedback-Antwort E-Mail Fehler für Feedback ID: " . $feedback_id . ": " . $emailError);
                            $response['email_sent'] = false;
                            
                            // Audit-Log: E-Mail fehlgeschlagen
                            try {
                                $checkTable = $conn->query("SHOW TABLES LIKE 'feedback_audit_log'");
                                if ($checkTable->rowCount() > 0) {
                                    $auditStmt = $conn->prepare("
                                        INSERT INTO feedback_audit_log 
                                        (feedback_id, admin_user_id, action_type, action_description, metadata)
                                        VALUES (?, ?, 'email_failed', 'E-Mail-Benachrichtigung fehlgeschlagen', ?)
                                    ");
                                    $auditStmt->execute([
                                        $feedback_id, 
                                        $current_user_id,
                                        json_encode(['error' => $emailError])
                                    ]);
                                }
                            } catch (PDOException $e) {
                                // Nicht kritisch
                            }
                        }
                    } else {
                        error_log("Feedback-Antwort E-Mail: Unbekanntes Ergebnis für Feedback ID: " . $feedback_id);
                        $response['email_sent'] = false;
                    }
                } else {
                    error_log("EmailNotifications Klasse nicht gefunden für Feedback ID: " . $feedback_id);
                    $response['email_sent'] = false;
                }
            } else {
                error_log("email_notifications.php nicht gefunden für Feedback ID: " . $feedback_id);
                $response['email_sent'] = false;
            }
        } catch (Throwable $emailException) {
            // Alle Fehler abfangen (Exception, Error, etc.)
            $emailError = $emailException->getMessage();
            error_log("Feedback-Antwort E-Mail Exception für Feedback ID: " . $feedback_id . ": " . $emailError);
            error_log("Exception Type: " . get_class($emailException));
            error_log("Stack Trace: " . $emailException->getTraceAsString());
            
            // Prüfe ob E-Mail möglicherweise trotzdem gesendet wurde (z.B. bei SMTP Debug Output)
            // Wenn die Exception nur Debug-Output betrifft, könnte die E-Mail trotzdem gesendet worden sein
            $emailSent = false;
            if (strpos($emailError, 'SMTP') !== false || strpos($emailError, 'Debug') !== false) {
                // Möglicherweise nur Debug-Fehler - E-Mail könnte trotzdem gesendet worden sein
                error_log("Möglicherweise nur Debug-Fehler - E-Mail könnte trotzdem gesendet worden sein");
            }
            
            $response['email_sent'] = false;
            
            // Audit-Log: E-Mail Exception
            try {
                $checkTable = $conn->query("SHOW TABLES LIKE 'feedback_audit_log'");
                if ($checkTable->rowCount() > 0) {
                    $auditStmt = $conn->prepare("
                        INSERT INTO feedback_audit_log 
                        (feedback_id, admin_user_id, action_type, action_description, metadata)
                        VALUES (?, ?, 'email_failed', 'E-Mail-Benachrichtigung Exception', ?)
                    ");
                    $auditStmt->execute([
                        $feedback_id, 
                        $current_user_id,
                        json_encode([
                            'error' => $emailError, 
                            'exception' => get_class($emailException),
                            'message' => substr($emailException->getMessage(), 0, 200)
                        ])
                    ]);
                }
            } catch (PDOException $e) {
                // Nicht kritisch
            }
        }
        
        // Optional: Benachrichtigung in user_notifications Tabelle erstellen (nicht kritisch)
        try {
            // Hole user_id aus Feedback
            $stmt = $conn->prepare("SELECT user_id FROM feedback WHERE id = ?");
            $stmt->execute([$feedback_id]);
            $feedback_data = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($feedback_data && $feedback_data['user_id'] && !empty($feedback_data['user_id'])) {
                // Prüfe ob user_notifications Tabelle existiert
                $checkTable = $conn->query("SHOW TABLES LIKE 'user_notifications'");
                if ($checkTable->rowCount() > 0) {
                    // Verwende Helper-Funktion für Push-Benachrichtigungen
                    require_once __DIR__ . '/../../includes/push_notification_helper.php';
                    $result = createNotificationWithPush(
                        $conn,
                        $feedback_data['user_id'],
                        'Antwort auf Ihr Feedback',
                        'Sie haben eine Antwort auf Ihr Feedback erhalten.',
                        'info',
                        '/profile.php'
                    );
                }
            }
        } catch (PDOException $notifError) {
            // Fehler beim Erstellen der Benachrichtigung ist nicht kritisch
            error_log("Fehler beim Erstellen der Feedback-Benachrichtigung: " . $notifError->getMessage());
        }
        
        ob_clean(); // Alle Outputs löschen
        echo json_encode($response);
        exit;
    } else {
        ob_clean(); // Alle Outputs löschen
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => 'Fehler beim Speichern der Antwort'
        ]);
        exit;
    }
    
} catch (PDOException $e) {
    ob_clean(); // Alle Outputs löschen
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Fehler beim Speichern der Antwort: ' . $e->getMessage()
    ]);
    exit;
} catch (Throwable $e) {
    ob_clean(); // Alle Outputs löschen
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Unerwarteter Fehler: ' . $e->getMessage()
    ]);
    exit;
}

