# 🔄 Dynamisches Laden der Drucker-Informationen - Funktionsweise

## 📋 Übersicht

Das System lädt Drucker-Einstellungen **dynamisch** basierend auf dem Autohaus des Termins. Jedes Autohaus kann eigene Drucker-Einstellungen haben.

---

## 🔄 Datenfluss

### 1. **Frontend: Modal öffnen** (`appointments.php`)

```javascript
// Wenn ein Termin zum Drucken ausgewählt wird
function showPrintEtikettModal(appointment) {
  // appointment enthält: { id, customer_name, license_plate, autohaus_id, ... }

  // Lade Drucker-Informationen dynamisch (basierend auf Autohaus)
  loadPrinterInfo(appointment.autohaus_id || null);
}
```

**Was passiert:**

- Das Modal wird geöffnet
- Die `autohaus_id` wird aus dem Appointment-Objekt extrahiert
- `loadPrinterInfo()` wird mit der `autohaus_id` aufgerufen

---

### 2. **Frontend: API-Aufruf** (`appointments.php`)

```javascript
async function loadPrinterInfo(autohausId, isBulk = false) {
  if (!autohausId) {
    // Fallback: Keine autohaus_id → Standard-Drucker
    updatePrinterDisplay("PRT17_BUEK-Server", isBulk);
    return;
  }

  try {
    // AJAX-Request an API-Endpoint
    const response = await fetch(
      `api/get_autohaus_printer.php?autohaus_id=${autohausId}`
    );
    const data = await response.json();

    if (data.success && data.printer) {
      // Druckername aus DB-Einstellungen
      const printerName =
        data.printer.printer_name_local ||
        data.printer.printer_name ||
        "PRT17_BUEK-Server";
      updatePrinterDisplay(printerName, isBulk);
    } else {
      // Fallback: Standard-Drucker
      updatePrinterDisplay("PRT17_BUEK-Server", isBulk);
    }
  } catch (error) {
    // Fehler → Fallback
    updatePrinterDisplay("PRT17_BUEK-Server", isBulk);
  }
}
```

**Was passiert:**

- AJAX-Request an `api/get_autohaus_printer.php?autohaus_id=3`
- Wartet auf JSON-Antwort
- Extrahiert den Druckernamen aus der Antwort
- Aktualisiert die Anzeige im Modal

---

### 3. **Backend: API-Endpoint** (`admin/api/get_autohaus_printer.php`)

```php
<?php
// 1. Session-Prüfung
session_start();
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    echo json_encode(['success' => false, 'error' => 'Nicht autorisiert']);
    exit;
}

// 2. autohaus_id aus GET-Parameter holen
$autohausId = $_GET['autohaus_id'] ?? null;

// 3. Datenbank-Abfrage
$stmt = $conn->prepare("
    SELECT
        printer_name, printer_name_local, printer_name_server,
        printer_server_ip, printer_server_hostname,
        printer_enabled
    FROM autohaus
    WHERE id = ? AND is_active = 1
");
$stmt->execute([$autohausId]);
$printer = $stmt->fetch(PDO::FETCH_ASSOC);

// 4. JSON-Antwort zurückgeben
echo json_encode([
    'success' => true,
    'printer' => $printer
]);
?>
```

**Was passiert:**

- Prüft, ob der Benutzer eingeloggt ist
- Holt die `autohaus_id` aus dem GET-Parameter
- Lädt Drucker-Einstellungen aus der Datenbank
- Gibt JSON zurück mit den Drucker-Informationen

**Beispiel-Antwort:**

```json
{
  "success": true,
  "printer": {
    "printer_name": "PRT17_BUEK-Server",
    "printer_name_local": "PRT17_BUEK-Server",
    "printer_name_server": "BUEK-PRT17",
    "printer_server_ip": "100.82.207.72",
    "printer_server_hostname": "buek-s01",
    "printer_enabled": 1
  }
}
```

---

### 4. **Frontend: Anzeige aktualisieren** (`appointments.php`)

```javascript
function updatePrinterDisplay(printerName, isBulk = false) {
  if (isBulk) {
    // Bulk-Print-Modal
    const bulkPrinterNameDisplay = document.getElementById(
      "bulk-printer-name-display"
    );
    if (bulkPrinterNameDisplay) {
      bulkPrinterNameDisplay.textContent = printerName;
    }
  } else {
    // Single-Print-Modal
    const printerNameDisplay = document.getElementById("printer-name-display");
    if (printerNameDisplay) {
      printerNameDisplay.textContent = printerName;
    }
  }
}
```

**Was passiert:**

- Der Druckername wird im Modal angezeigt
- Entweder im Single-Print-Modal oder Bulk-Print-Modal

---

## 🖨️ Beim Drucken: Backend-Logik (`print_etikett_api.php`)

### Schritt 1: autohaus_id bestimmen

```php
function printEtikett($name, $kennzeichen, $copies = 1, $autohaus_id = null) {
    // Option 1: autohaus_id wurde direkt übergeben (aus Frontend)
    if (!$autohaus_id) {
        // Option 2: Aus Appointment-ID holen
        if (isset($_POST['appointment_id'])) {
            $stmt = $conn->prepare("SELECT autohaus_id FROM appointments WHERE id = ?");
            $stmt->execute([$_POST['appointment_id']]);
            $appointment = $stmt->fetch(PDO::FETCH_ASSOC);
            $autohaus_id = $appointment['autohaus_id'] ?? null;
        }

        // Option 3: Aus Session (Benutzer-Autohaus)
        if (!$autohaus_id && isset($_SESSION['admin_user_id'])) {
            $stmt = $conn->prepare("SELECT autohaus_id FROM admin_users WHERE id = ?");
            $stmt->execute([$_SESSION['admin_user_id']]);
            $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
            $autohaus_id = $user_data['autohaus_id'] ?? null;
        }
    }
```

**Priorität:**

1. ✅ Direkt übergeben (aus Frontend)
2. ✅ Aus Appointment-ID
3. ✅ Aus Session (Benutzer-Autohaus)
4. ❌ Fallback auf `config/printer.php`

---

### Schritt 2: Drucker-Einstellungen laden

```php
// Lade Drucker-Einstellungen aus der Datenbank
$printer_config = null;
if ($autohaus_id) {
    $stmt = $conn->prepare("
        SELECT
            printer_name, printer_server_ip, printer_server_hostname,
            printer_server_user, printer_server_pass, printer_name_server,
            printer_name_local, printer_enabled
        FROM autohaus
        WHERE id = ? AND is_active = 1 AND printer_enabled = 1
    ");
    $stmt->execute([$autohaus_id]);
    $printer_config = $stmt->fetch(PDO::FETCH_ASSOC);
}

// Fallback auf config/printer.php, falls keine DB-Einstellungen
if (!$printer_config || !$printer_config['printer_name_local']) {
    // Verwende Standard-Werte aus config/printer.php
    $printer_name_local = PRINTER_NAME_LOCAL; // 'PRT17_BUEK-Server'
    $printer_server_ip = PRINTER_SERVER_TAILSCALE_IP; // '100.82.207.72'
    // ...
} else {
    // Verwende DB-Einstellungen
    $printer_name_local = $printer_config['printer_name_local'];
    $printer_server_ip = $printer_config['printer_server_ip'];
    // ...
}
```

**Was passiert:**

- Lädt Drucker-Einstellungen aus der `autohaus` Tabelle
- Falls keine Einstellungen vorhanden → Fallback auf `config/printer.php`
- Falls Einstellungen vorhanden → Verwendet diese

---

## 📊 Datenbank-Struktur

### Tabelle: `autohaus`

```sql
CREATE TABLE autohaus (
    id INT PRIMARY KEY,
    name VARCHAR(255),
    -- ... andere Felder ...

    -- Drucker-Einstellungen (neu hinzugefügt)
    printer_name VARCHAR(255),
    printer_name_local VARCHAR(255),        -- z.B. 'PRT17_BUEK-Server'
    printer_name_server VARCHAR(255),        -- z.B. 'BUEK-PRT17'
    printer_server_ip VARCHAR(50),          -- z.B. '100.82.207.72'
    printer_server_hostname VARCHAR(255),   -- z.B. 'buek-s01'
    printer_server_user VARCHAR(255),       -- z.B. 'VOLVOKIE\Projekt'
    printer_server_pass VARCHAR(255),       -- z.B. 'start#2025'
    printer_enabled TINYINT(1) DEFAULT 1
);
```

**Beispiel-Daten:**

```sql
id | name                    | printer_name_local    | printer_server_ip
---|-------------------------|----------------------|------------------
1  | B & E Volvo Kiel        | PRT17_BUEK-Server    | 100.82.207.72
2  | B & E Volvo Norderstedt | PRT17_BUEK-Server    | 100.82.207.72
3  | B & E Volvo Glinde       | PRT17_BUEK-Server    | 100.82.207.72
```

---

## 🔄 Kompletter Ablauf (Beispiel)

### Szenario: Benutzer druckt Etikett für Termin aus Autohaus "Kiel" (ID: 1)

1. **Frontend: Modal öffnen**

   ```
   appointment = { id: 123, autohaus_id: 1, customer_name: "Max Mustermann", ... }
   → showPrintEtikettModal(appointment)
   → loadPrinterInfo(1)
   ```

2. **Frontend: API-Aufruf**

   ```
   GET api/get_autohaus_printer.php?autohaus_id=1
   ```

3. **Backend: Datenbank-Abfrage**

   ```sql
   SELECT printer_name_local, printer_server_ip, ...
   FROM autohaus
   WHERE id = 1 AND is_active = 1
   ```

   Ergebnis: `{ printer_name_local: "PRT17_BUEK-Server", ... }`

4. **Backend: JSON-Antwort**

   ```json
   {
       "success": true,
       "printer": {
           "printer_name_local": "PRT17_BUEK-Server",
           "printer_server_ip": "100.82.207.72",
           ...
       }
   }
   ```

5. **Frontend: Anzeige aktualisieren**

   ```
   updatePrinterDisplay("PRT17_BUEK-Server")
   → Modal zeigt: "Drucker: PRT17_BUEK-Server"
   ```

6. **Benutzer klickt "Drucken"**

   ```javascript
   executePrint()
   → FormData wird erstellt:
      - action: "print_etikett"
      - name: "Max Mustermann"
      - kennzeichen: "HH-AB 123"
      - copies: 6
      - autohaus_id: 1  // ✅ Wird jetzt übergeben!
      - appointment_id: 123  // ✅ Als Fallback
   → POST print_etikett_api.php
   ```

7. **Backend: Drucken**
   ```php
   printEtikett("Max Mustermann", "HH-AB 123", 6, 1)
   → Lädt Drucker-Einstellungen für autohaus_id = 1
   → Verwendet: printer_name_local = "PRT17_BUEK-Server"
   → Druckt auf diesem Drucker
   ```

---

## 🎯 Wichtige Punkte

### 1. **Autohaus-ID Bestimmung**

**Priorität beim Drucken:**

1. ✅ Direkt übergeben (`$_POST['autohaus_id']`)
2. ✅ Aus Appointment-ID (`$_POST['appointment_id']` → `appointments.autohaus_id`)
3. ✅ Aus Session (`$_SESSION['admin_user_id']` → `admin_users.autohaus_id`)
4. ❌ Fallback auf `config/printer.php`

### 2. **Fallback-Mechanismus**

**Frontend:**

- Keine `autohaus_id` → Zeigt Standard-Drucker: `PRT17_BUEK-Server`
- API-Fehler → Zeigt Standard-Drucker: `PRT17_BUEK-Server`

**Backend:**

- Keine DB-Einstellungen → Verwendet `config/printer.php`
- `printer_enabled = 0` → Verwendet `config/printer.php`

### 3. **Admin vs. Normaler Benutzer**

**Normaler Benutzer:**

- Hat `autohaus_id` in `admin_users` Tabelle
- Verwendet automatisch das Autohaus-Drucker

**Admin:**

- Hat `autohaus_id = NULL` in `admin_users` Tabelle
- Kann Drucker pro Termin wählen (wenn implementiert)
- Fallback auf `config/printer.php`

---

## 🔍 Debugging

### Frontend (Browser-Konsole)

```javascript
// Prüfe, ob autohaus_id vorhanden ist
console.log("Appointment:", appointment);
console.log("Autohaus ID:", appointment.autohaus_id);

// Prüfe API-Antwort
fetch(`api/get_autohaus_printer.php?autohaus_id=1`)
  .then((r) => r.json())
  .then((data) => console.log("API Response:", data));
```

### Backend (PHP)

```php
// In print_etikett_api.php
error_log("autohaus_id: " . $autohaus_id);
error_log("printer_config: " . print_r($printer_config, true));
```

---

## 📝 Zusammenfassung

1. **Frontend lädt Drucker-Info** → AJAX-Request an API
2. **API lädt aus DB** → `autohaus` Tabelle
3. **Frontend zeigt Druckername** → Im Modal
4. **Beim Drucken** → Backend verwendet DB-Einstellungen
5. **Fallback** → `config/printer.php` wenn keine DB-Einstellungen

**Vorteile:**

- ✅ Jedes Autohaus kann eigenen Drucker haben
- ✅ Zentrale Verwaltung in Settings
- ✅ Automatische Erkennung basierend auf Termin
- ✅ Fallback-Mechanismus für Kompatibilität

---

**Letzte Aktualisierung:** 2025-01-XX
