<?php
/**
 * Test-Script für Cookie-Validierung und automatische Aktualisierung
 */

header('Content-Type: application/json; charset=utf-8');

// Debug-Logging-Funktion
function debugLog($message) {
    $logFile = __DIR__ . '/../logs/cookie_test.log';
    $timestamp = date('Y-m-d H:i:s');
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

// Cookie-Validierung testen
function testCookieValidation($baseUrl, $cookies) {
    debugLog("=== COOKIE-VALIDIERUNG TEST START ===");
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    curl_setopt($ch, CURLOPT_COOKIE, implode('; ', $cookies));
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    
    // Teste mit einer einfachen RESY-Seite
    $testUrl = $baseUrl . '/resyweb/index.php';
    curl_setopt($ch, CURLOPT_URL, $testUrl);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    debugLog("Cookie-Test: HTTP $httpCode, Length: " . strlen($response));
    
    // Prüfe ob die Cookies gültig sind
    $hasLoginForm = strpos($response, 'Anmeldename') !== false || 
                    strpos($response, 'UserName') !== false;
    $hasResyContent = strpos($response, 'RESY') !== false ||
                     strpos($response, 'resyweb') !== false;
    
    $isValid = !$hasLoginForm && $hasResyContent;
    
    debugLog("Cookie-Validierung: " . ($isValid ? "GÜLTIG" : "UNGÜLTIG"));
    debugLog("Login-Form gefunden: " . ($hasLoginForm ? "JA" : "NEIN"));
    debugLog("RESY-Content gefunden: " . ($hasResyContent ? "JA" : "NEIN"));
    
    return [
        'valid' => $isValid,
        'http_code' => $httpCode,
        'response_length' => strlen($response),
        'has_login_form' => $hasLoginForm,
        'has_resy_content' => $hasResyContent
    ];
}

// Teste verschiedene Cookie-Sets
function runCookieTests() {
    $baseUrl = 'https://resy.dtm-reifen.de';
    
    // Aktuelle Cookies aus der API
    $currentCookies = [
        "PHPSESSID=a4pe0htf62vbtvoo3ivti484vs",
        "HideWheelsetSearchResultBEZEICHNUNG=0",
        "HideWheelsetSearchResultKDNR=0",
        "HideWheelsetSearchResultORT=0",
        "HideWheelsetSearchResultSTRASSE1=0",
        "SortWheelsetSearchResultCol=6",
        "SortWheelsetSearchResultDesc=-1"
    ];
    
    // Alte Cookies (zum Vergleich)
    $oldCookies = [
        "PHPSESSID=0b8ctmiebkumf369hsh8pk4004",
        "Autohaus_LastChange=1758873482",
        "Autohaus_Usage[453]=14",
        "HideWheelsetSearchResultFLGFEHL_FLG=1",
        "HideWheelsetSearchResultMINPROFIL=0",
        "HideWheelsetSearchResultPRUFDOC_ID=1",
        "HideWheelsetSearchResultRADKAPFEHL_FLG=1",
        "HideWheelsetSearchResultRFFEHL_FLG=1",
        "HideWheelsetSearchResultSTATUS_ID=1",
        "Ladeplatz_LastChange=1758873482",
        "Ladeplatz_Usage[]=1",
        "Ladeplatz_Usage[0]=0",
        "Ladeplatz_Usage[1]=1",
        "SortConfirmTableCol=2",
        "SortConfirmTableDesc=1",
        "SortWheelsetSearchResultCol=11",
        "SortWheelsetSearchResultDesc=-1"
    ];
    
    $results = [];
    
    // Test 1: Aktuelle Cookies
    debugLog("--- TEST 1: AKTUELLE COOKIES ---");
    $results['current'] = testCookieValidation($baseUrl, $currentCookies);
    
    // Test 2: Alte Cookies
    debugLog("--- TEST 2: ALTE COOKIES ---");
    $results['old'] = testCookieValidation($baseUrl, $oldCookies);
    
    // Test 3: Keine Cookies
    debugLog("--- TEST 3: KEINE COOKIES ---");
    $results['none'] = testCookieValidation($baseUrl, []);
    
    return $results;
}

try {
    debugLog("=== COOKIE-VALIDIERUNG TEST GESTARTET ===");
    
    $results = runCookieTests();
    
    echo json_encode([
        'success' => true,
        'results' => $results,
        'message' => 'Cookie-Validierung abgeschlossen',
        'timestamp' => date('Y-m-d H:i:s')
    ], JSON_PRETTY_PRINT);
    
} catch (Exception $e) {
    debugLog("Fehler: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ], JSON_PRETTY_PRINT);
}

debugLog("=== COOKIE-VALIDIERUNG TEST BEENDET ===");
?>
