<?php
/**
 * Performance Test Page - Shows before/after optimization results
 */

require_once 'config/database_optimized.php';
require_once 'includes/performance_monitor.php';

$db = DatabaseOptimized::getInstance();
$conn = $db->getConnection();

// Test different types of queries
$tests = [];

// Test 1: Simple COUNT query
$start = microtime(true);
$result = $db->fetchColumn("SELECT COUNT(*) FROM events WHERE is_active = 1");
$tests['simple_count'] = (microtime(true) - $start) * 1000;

// Test 2: Complex JOIN query
$start = microtime(true);
$result = $db->fetchAll("
    SELECT a.*, e.name as event_name, ah.name as autohaus_name
    FROM appointments a
    JOIN events e ON a.event_id = e.id
    LEFT JOIN autohaus ah ON a.autohaus_id = ah.id
    ORDER BY a.created_at DESC
    LIMIT 10
");
$tests['complex_join'] = (microtime(true) - $start) * 1000;

// Test 3: Dashboard stats query
$start = microtime(true);
$stats = $db->getDashboardStats();
$tests['dashboard_stats'] = (microtime(true) - $start) * 1000;

// Test 4: User data query (cached)
$start = microtime(true);
$user_data = $db->getUserData(1);
$tests['user_data_cached'] = (microtime(true) - $start) * 1000;

$performance_data = PerformanceMonitor::end();
?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Performance Test - Boxenstop</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        body { font-family: 'Inter', sans-serif; margin: 0; padding: 20px; background: #f8f9fa; }
        .container { max-width: 1200px; margin: 0 auto; }
        .header { background: white; padding: 20px; border-radius: 8px; margin-bottom: 20px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        .test-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px; }
        .test-card { background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        .test-title { font-size: 18px; font-weight: 600; margin-bottom: 10px; color: #2c3e50; }
        .test-time { font-size: 24px; font-weight: 700; margin-bottom: 5px; }
        .test-time.fast { color: #28a745; }
        .test-time.medium { color: #ffc107; }
        .test-time.slow { color: #dc3545; }
        .test-description { color: #6c757d; font-size: 14px; }
        .overall-stats { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; border-radius: 8px; margin-bottom: 20px; }
        .stat-item { display: inline-block; margin-right: 30px; }
        .stat-value { font-size: 24px; font-weight: 700; }
        .stat-label { font-size: 14px; opacity: 0.9; }
        .recommendations { background: #e3f2fd; padding: 20px; border-radius: 8px; margin-top: 20px; }
        .btn { display: inline-block; padding: 10px 20px; background: #007bff; color: white; text-decoration: none; border-radius: 5px; margin: 5px; }
        .btn:hover { background: #0056b3; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1><i class="fas fa-tachometer-alt"></i> Performance Test Results</h1>
            <p>Messung der Datenbankabfragen nach Optimierung</p>
        </div>

        <div class="overall-stats">
            <div class="stat-item">
                <div class="stat-value"><?php echo round($performance_data['total_time'], 3); ?>s</div>
                <div class="stat-label">Gesamtladezeit</div>
            </div>
            <div class="stat-item">
                <div class="stat-value"><?php echo $performance_data['query_count']; ?></div>
                <div class="stat-label">DB-Abfragen</div>
            </div>
            <div class="stat-item">
                <div class="stat-value"><?php echo round($performance_data['memory_used'] / 1024 / 1024, 2); ?>MB</div>
                <div class="stat-label">Speicherverbrauch</div>
            </div>
            <div class="stat-item">
                <div class="stat-value"><?php echo $db->getQueryCount(); ?></div>
                <div class="stat-label">Optimierte Queries</div>
            </div>
        </div>

        <div class="test-grid">
            <div class="test-card">
                <div class="test-title">Simple COUNT Query</div>
                <div class="test-time <?php echo $tests['simple_count'] < 10 ? 'fast' : ($tests['simple_count'] < 50 ? 'medium' : 'slow'); ?>">
                    <?php echo round($tests['simple_count'], 2); ?>ms
                </div>
                <div class="test-description">Einfache Zählabfrage mit Index</div>
            </div>

            <div class="test-card">
                <div class="test-title">Complex JOIN Query</div>
                <div class="test-time <?php echo $tests['complex_join'] < 20 ? 'fast' : ($tests['complex_join'] < 100 ? 'medium' : 'slow'); ?>">
                    <?php echo round($tests['complex_join'], 2); ?>ms
                </div>
                <div class="test-description">Komplexe JOIN-Abfrage mit mehreren Tabellen</div>
            </div>

            <div class="test-card">
                <div class="test-title">Dashboard Stats</div>
                <div class="test-time <?php echo $tests['dashboard_stats'] < 30 ? 'fast' : ($tests['dashboard_stats'] < 100 ? 'medium' : 'slow'); ?>">
                    <?php echo round($tests['dashboard_stats'], 2); ?>ms
                </div>
                <div class="test-description">Dashboard-Statistiken (gecacht)</div>
            </div>

            <div class="test-card">
                <div class="test-title">User Data (Cached)</div>
                <div class="test-time <?php echo $tests['user_data_cached'] < 5 ? 'fast' : ($tests['user_data_cached'] < 20 ? 'medium' : 'slow'); ?>">
                    <?php echo round($tests['user_data_cached'], 2); ?>ms
                </div>
                <div class="test-description">Benutzerdaten aus Cache</div>
            </div>
        </div>

        <div class="recommendations">
            <h3><i class="fas fa-lightbulb"></i> Performance-Bewertung</h3>
            <?php
            $avg_time = array_sum($tests) / count($tests);
            if ($avg_time < 20) {
                echo "<p style='color: #28a745; font-weight: 600;'>🎉 Ausgezeichnete Performance! Alle Abfragen sind sehr schnell.</p>";
            } elseif ($avg_time < 50) {
                echo "<p style='color: #ffc107; font-weight: 600;'>⚠️ Gute Performance, aber noch Verbesserungspotential.</p>";
            } else {
                echo "<p style='color: #dc3545; font-weight: 600;'>❌ Langsame Performance - weitere Optimierung erforderlich.</p>";
            }
            ?>
            
            <h4>Nächste Schritte:</h4>
            <a href="admin/apply_database_indexes.php" class="btn">Datenbankindizes anwenden</a>
            <a href="admin/performance.php" class="btn">Performance Dashboard</a>
            <a href="admin/index.php" class="btn">Admin Dashboard</a>
        </div>
    </div>
</body>
</html>
