<?php
/**
 * Login/Registrierungsseite für Boxenstop
 */

session_start();
require_once 'config/database.php';

// Prüfe ob User bereits eingeloggt ist
if (isset($_SESSION['user_logged_in']) && $_SESSION['user_logged_in'] === true) {
    // Redirect zu gewünschter Seite oder index.php
    $redirect = $_GET['redirect'] ?? 'index.php';
    if (isset($_GET['event'])) {
        $redirect .= '?event=' . $_GET['event'];
    }
    header('Location: ' . $redirect);
    exit;
}

// Redirect-Parameter speichern
$redirect_after_login = $_GET['redirect'] ?? 'index.php';
if (isset($_GET['event'])) {
    $redirect_after_login .= '?event=' . $_GET['event'];
}
$_SESSION['redirect_after_login'] = $redirect_after_login;

// Unternehmensdaten laden
$db = new Database();
$conn = $db->getConnection();

$stmt = $conn->prepare("SELECT setting_key, setting_value FROM settings WHERE setting_key IN ('company_name', 'company_logo', 'app_name')");
$stmt->execute();
$companySettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

$companyName = $companySettings['company_name'] ?? 'B&E Gruppe';
$companyLogo = $companySettings['company_logo'] ?? 'assets/images/BE-Logo-FINAL_Schwarz-1.png';
$companyLogo = ltrim($companyLogo, '/');
$appName = $companySettings['app_name'] ?? 'B&E Gruppe';
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Anmelden oder Registrieren - <?php echo htmlspecialchars($appName); ?></title>
    <link rel="stylesheet" href="assets/css/style.css">
    <link rel="stylesheet" href="assets/css/volvo-theme.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://appleid.cdn-apple.com/appleauth/static/jsapi/appleid/1/en_US/appleid.auth.js"></script>
    <script src="https://accounts.google.com/gsi/client" async defer></script>
    <style>
        body {
            position: relative;
            min-height: 100vh;
            background: url('https://www.volvocars-haendler.de/assets/support/img/home_campaign_teaser/1369/backgroundimage2x_processed.jpg') center center / cover no-repeat;
        }
        
        body::before {
            content: '';
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: url('https://www.volvocars-haendler.de/assets/support/img/home_campaign_teaser/1369/backgroundimage2x_processed.jpg') center center / cover no-repeat;
            filter: blur(8px);
            z-index: -1;
        }
        
        body::after {
            content: '';
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.3);
            z-index: -1;
        }
        
        .auth-container {
            min-height: calc(100vh - 80px);
            padding: 2rem;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-top: 80px;
        }
        
        .auth-card {
            background: rgba(255, 255, 255, 0.98);
            backdrop-filter: blur(20px);
            border-radius: 20px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.2);
            max-width: 480px;
            width: 100%;
            overflow: hidden;
            border: 1px solid rgba(255, 255, 255, 0.3);
        }
        
        .auth-header {
            text-align: center;
            padding: 2.5rem 2rem 2rem;
            background: linear-gradient(135deg, #003057 0%, #0056b3 100%);
        }
        
        .auth-logo {
            height: 55px;
            margin-bottom: 1rem;
            filter: drop-shadow(0 2px 8px rgba(0, 0, 0, 0.2));
        }
        
        .auth-title {
            font-size: 1.75rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
            color: white;
        }
        
        .auth-subtitle {
            font-size: 0.95rem;
            opacity: 0.95;
            color: white;
        }
        
        .auth-body {
            padding: 2.5rem;
        }
        
        .auth-tabs {
            display: flex;
            gap: 0.5rem;
            margin-bottom: 2rem;
        }
        
        .auth-tab {
            flex: 1;
            padding: 0.875rem 1rem;
            text-align: center;
            background: #f8f9fa;
            border: 2px solid transparent;
            font-size: 0.95rem;
            font-weight: 600;
            color: #6c757d;
            cursor: pointer;
            transition: all 0.3s;
            border-radius: 10px;
        }
        
        .auth-tab.active {
            background: #003057;
            color: white;
            border-color: #003057;
        }
        
        .auth-tab:hover:not(.active) {
            background: #e9ecef;
            color: #495057;
        }
        
        .auth-content {
            display: none;
        }
        
        .auth-content.active {
            display: block;
        }
        
        .form-group {
            margin-bottom: 1.5rem;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            color: #495057;
            font-weight: 500;
            font-size: 0.95rem;
        }
        
        .form-group input {
            width: 100%;
            padding: 1rem 1.25rem;
            border: 2px solid #e9ecef;
            border-radius: 12px;
            font-size: 0.95rem;
            transition: all 0.3s;
        }
        
        .form-group input:focus {
            outline: none;
            border-color: #003057;
            box-shadow: 0 0 0 3px rgba(0, 48, 87, 0.1);
        }
        
        .btn {
            width: 100%;
            padding: 1rem;
            border: none;
            border-radius: 8px;
            font-size: 1.1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.75rem;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #003057 0%, #0056b3 100%);
            color: white;
            box-shadow: 0 4px 12px rgba(0, 48, 87, 0.3);
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(0, 48, 87, 0.4);
        }
        
        .btn-apple {
            background: #000;
            color: white;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.2);
        }
        
        .btn-apple:hover {
            background: #1a1a1a;
        }
        
        .btn-google {
            background: #fff;
            color: #4285f4;
            border: 2px solid #dadce0;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
            position: relative;
            overflow: hidden;
        }
        
        .btn-google:hover {
            background: #f8f9fa;
            border-color: #4285f4;
        }
        
        /* Google Logo mit vier Farben */
        .google-logo {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            width: 20px;
            height: 20px;
            margin-right: 0.5rem;
            position: relative;
        }
        
        .google-logo svg {
            width: 20px;
            height: 20px;
        }
        
        /* Google Logo - vierfarbiges "G" */
        .google-g {
            font-weight: 500;
            font-size: 18px;
            line-height: 1;
            display: inline-block;
            background: linear-gradient(135deg, #4285f4 0%, #4285f4 25%, #ea4335 25%, #ea4335 50%, #fbbc04 50%, #fbbc04 75%, #34a853 75%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            font-family: 'Product Sans', 'Roboto', 'Arial', sans-serif;
        }
        
        .divider {
            display: flex;
            align-items: center;
            margin: 2rem 0;
            color: #6c757d;
        }
        
        .divider::before,
        .divider::after {
            content: '';
            flex: 1;
            height: 1px;
            background: #dee2e6;
        }
        
        .social-buttons {
            display: flex;
            gap: 1rem;
            margin-bottom: 1rem;
        }
        
        .social-buttons .btn {
            flex: 1;
        }
        
        .divider span {
            padding: 0 1rem;
            font-size: 0.9rem;
        }
        
        .alert {
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
            display: none;
        }
        
        .alert.show {
            display: block;
        }
        
        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .password-toggle {
            position: relative;
        }
        
        .password-toggle input {
            padding-right: 3rem;
        }
        
        .password-toggle-btn {
            position: absolute;
            right: 1rem;
            top: 50%;
            transform: translateY(-50%);
            background: none;
            border: none;
            color: #6c757d;
            cursor: pointer;
            padding: 0.5rem;
        }
        
        @media (max-width: 768px) {
            .auth-container {
                padding: 1rem;
            }
            
            .auth-body {
                padding: 1.5rem;
            }
            
            .auth-header {
                padding: 2rem 1.5rem;
            }
        }
        /* Navigation - Volvo Style */
        .navbar {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-bottom: 1px solid rgba(0, 0, 0, 0.1);
        }
        
        .nav-brand h1 {
            font-size: 1.2rem;
            margin: 0;
        }
        
        .nav-brand .brand-subtitle {
            font-size: 0.85rem;
        }
        
        .nav-link {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.625rem 1.25rem;
            border-radius: 8px;
            text-decoration: none;
            font-weight: 500;
            font-size: 0.95rem;
            transition: all 0.3s ease;
            border: 1px solid #dee2e6;
            background: #6c757d;
            color: white;
        }
        
        .nav-link:hover {
            background: #5a6268;
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(108, 117, 125, 0.2);
        }
        
        .nav-link i {
            font-size: 1rem;
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar">
        <div class="nav-container">
            <div class="nav-brand">
                <div class="logo-wrapper">
                    <img src="<?php echo htmlspecialchars($companyLogo); ?>" alt="<?php echo htmlspecialchars($companyName); ?>" class="nav-logo">
                    <div class="brand-text">
                        <h1><?php echo htmlspecialchars($appName); ?></h1>
                        <span class="brand-subtitle">B&E Gruppe - Volvo Service</span>
                    </div>
                </div>
            </div>
            <div class="nav-menu">
                <a href="index.php" class="nav-link">
                    <i class="fas fa-arrow-left"></i>
                    <span>Zurück</span>
                </a>
            </div>
        </div>
    </nav>
    
    <div class="auth-container">
        <div class="auth-card">
            <div class="auth-body">
                <!-- Alert Messages -->
                <div id="alert-message" class="alert"></div>
                
                <!-- Tabs -->
                <div class="auth-tabs">
                    <button class="auth-tab active" onclick="switchTab('login')">Anmelden</button>
                    <button class="auth-tab" onclick="switchTab('register')">Registrieren</button>
                </div>
                
                <!-- Login Form -->
                <div id="login-content" class="auth-content active">
                    <form id="login-form">
                        <div class="form-group">
                            <label for="login-email">E-Mail-Adresse</label>
                            <input type="email" id="login-email" name="email" required placeholder="ihr@email.de">
                        </div>
                        
                        <div class="form-group">
                            <label for="login-password">Passwort</label>
                            <div class="password-toggle">
                                <input type="password" id="login-password" name="password" required placeholder="Ihr Passwort">
                                <button type="button" class="password-toggle-btn" onclick="togglePassword('login-password')">
                                    <i class="fas fa-eye"></i>
                                </button>
                            </div>
                        </div>
                        
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-sign-in-alt"></i>
                            <span>Anmelden</span>
                        </button>
                    </form>
                </div>
                
                <!-- Register Form -->
                <div id="register-content" class="auth-content">
                    <form id="register-form">
                        <div class="form-group">
                            <label for="register-firstname">Vorname</label>
                            <input type="text" id="register-firstname" name="first_name" required placeholder="Max">
                        </div>
                        
                        <div class="form-group">
                            <label for="register-lastname">Nachname</label>
                            <input type="text" id="register-lastname" name="last_name" required placeholder="Mustermann">
                        </div>
                        
                        <div class="form-group">
                            <label for="register-email">E-Mail-Adresse</label>
                            <input type="email" id="register-email" name="email" required placeholder="ihr@email.de">
                        </div>
                        
                        <div class="form-group">
                            <label for="register-phone">Telefonnummer</label>
                            <input type="tel" id="register-phone" name="phone_number" required placeholder="+49 123 456789">
                        </div>
                        
                        <div class="form-group">
                            <label for="register-license">Kennzeichen</label>
                            <input type="text" id="register-license" name="license_plate" required placeholder="M-AB 123">
                        </div>
                        
                        <div class="form-group">
                            <label for="register-password">Passwort</label>
                            <div class="password-toggle">
                                <input type="password" id="register-password" name="password" required placeholder="Mindestens 8 Zeichen">
                                <button type="button" class="password-toggle-btn" onclick="togglePassword('register-password')">
                                    <i class="fas fa-eye"></i>
                                </button>
                            </div>
                        </div>
                        
                        
                        <button type="submit" id="register-submit" class="btn btn-primary">
                            <i class="fas fa-user-plus"></i>
                            <span>Registrieren</span>
                        </button>
                    </form>
                </div>
                
                <!-- Apple Sign In -->
                <div class="divider">
                    <span>oder</span>
                </div>
                
                <!-- Einheitliche Datenschutz-Checkbox für alle Register-Methoden -->
                <div id="global-datenschutz" style="display: none; margin-bottom: 1.5rem; padding: 1.25rem; background: #fff; border-radius: 12px; border: 2px solid #e9ecef; transition: all 0.3s ease;" class="datenschutz-container">
                    <label style="display: flex; align-items: flex-start; gap: 0.75rem; font-size: 0.95rem; margin: 0;">
                        <input type="checkbox" id="global-datenschutz-agreement" disabled style="width: auto; margin-top: 0.25rem; flex-shrink: 0; cursor: not-allowed;" onchange="toggleAllRegisterButtons()">
                        <span style="line-height: 1.6;">
                            <a href="#" onclick="openDatenschutzModal(); return false;" style="color: #0056b3; text-decoration: underline; font-weight: 600;">Datenschutzerklärung lesen und akzeptieren</a> ist erforderlich. *
                        </span>
                    </label>
                    <p style="margin-top: 0.75rem; color: #6c757d; font-size: 0.85rem; text-align: center;">
                        <i class="fas fa-lock"></i> Öffnen Sie die Datenschutzerklärung und bestätigen Sie diese im Modal.
                    </p>
                </div>
                
                <div class="social-buttons">
                    <button id="apple-signin-button" class="btn btn-apple" onclick="handleAppleSignIn()">
                        <i class="fab fa-apple"></i>
                        <span id="apple-button-text">Mit Apple anmelden</span>
                    </button>
                    
                    <button id="google-signin-button" class="btn btn-google" onclick="handleGoogleSignIn()">
                        <span class="google-logo">
                            <svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                <path fill="#4285F4" d="M22.56 12.25c0-.78-.07-1.53-.2-2.25H12v4.26h5.92c-.26 1.37-1.04 2.53-2.21 3.31v2.77h3.57c2.08-1.92 3.28-4.74 3.28-8.09z"/>
                                <path fill="#34A853" d="M12 23c2.97 0 5.46-.98 7.28-2.66l-3.57-2.77c-.98.66-2.23 1.06-3.71 1.06-2.86 0-5.29-1.93-6.16-4.53H2.18v2.84C3.99 20.53 7.7 23 12 23z"/>
                                <path fill="#FBBC05" d="M5.84 14.09c-.22-.66-.35-1.36-.35-2.09s.13-1.43.35-2.09V7.07H2.18C1.43 8.55 1 10.22 1 12s.43 3.45 1.18 4.93l2.85-2.22.81-.62z"/>
                                <path fill="#EA4335" d="M12 5.38c1.62 0 3.06.56 4.21 1.64l3.15-3.15C17.45 2.09 14.97 1 12 1 7.7 1 3.99 3.47 2.18 7.07l3.66 2.84c.87-2.6 3.3-4.53 6.16-4.53z"/>
                            </svg>
                        </span>
                        <span id="google-button-text">Mit Google anmelden</span>
                    </button>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Datenschutz Modal -->
    <div id="datenschutz-modal" class="modal" style="display: none; position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.7); z-index: 9999; overflow-y: auto; padding: 1rem;">
        <div class="modal-content" style="background: white; max-width: 800px; margin: 1rem auto; border-radius: 16px; padding: 1.5rem; box-shadow: 0 8px 32px rgba(0,0,0,0.3); max-height: 95vh; display: flex; flex-direction: column;">
            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 2rem;">
                <h2 style="margin: 0; color: #003057;">Datenschutzerklärung</h2>
                <button onclick="closeDatenschutzModal()" style="background: none; border: none; font-size: 2rem; cursor: pointer; color: #6c757d;">&times;</button>
            </div>
            
            <div style="flex: 1; overflow-y: auto; line-height: 1.8; margin-bottom: 1.5rem;">
                <h3 style="color: #003057; margin-top: 0;">1. Verantwortlicher</h3>
                <p>
                    B&E Vertriebsgesellschaft mbH<br>
                    B&E Gruppe
                </p>
                
                <h3 style="color: #003057;">2. Erhebung und Verarbeitung personenbezogener Daten</h3>
                <p>
                    Wir erheben und verarbeiten folgende personenbezogene Daten:
                </p>
                <ul>
                    <li><strong>Anmeldedaten:</strong> Vorname, Nachname, E-Mail-Adresse, Passwort (verschlüsselt)</li>
                    <li><strong>Kontaktdaten:</strong> Telefonnummer, Adresse, Postleitzahl, Stadt</li>
                    <li><strong>Fahrzeugdaten:</strong> Kennzeichen, Marke, Modell, Baujahr</li>
                    <li><strong>Buchungsdaten:</strong> Termine, Autohaus, bevorzugtes Datum</li>
                    <li><strong>Technische Daten:</strong> IP-Adresse, Gerätetyp, Browser-Version</li>
                    <li><strong>Registrierungsquelle:</strong> Web, iOS App, Android App, Apple Sign-In, Google Sign-In</li>
                </ul>
                
                <h3 style="color: #003057;">3. Zweck der Datenverarbeitung</h3>
                <p>
                    Ihre Daten werden für folgende Zwecke verwendet:
                </p>
                <ul>
                    <li>Erstellung und Verwaltung Ihres Benutzerkontos</li>
                    <li>Terminbuchungen und -verwaltung bei unseren Partner-Autohäusern</li>
                    <li>Kommunikation bezüglich Ihrer Termine</li>
                    <li>Verifikation Ihrer Identität zum Schutz vor Missbrauch</li>
                    <li>Erfüllung gesetzlicher Anforderungen</li>
                    <li>Verbesserung unserer Dienstleistungen</li>
                </ul>
                
                <h3 style="color: #003057;">4. Speicherdauer</h3>
                <p>
                    Ihre Daten werden so lange gespeichert, wie es für die genannten Zwecke erforderlich ist oder gesetzlich vorgeschrieben ist. Nach Beendigung der Geschäftsbeziehung werden die Daten gelöscht, sofern keine gesetzlichen Aufbewahrungspflichten bestehen.
                </p>
                
                <h3 style="color: #003057;">5. Ihre Rechte</h3>
                <p>
                    Sie haben folgende Rechte bezüglich Ihrer Daten:
                </p>
                <ul>
                    <li><strong>Auskunft:</strong> Sie können jederzeit Auskunft über Ihre gespeicherten Daten verlangen</li>
                    <li><strong>Berichtigung:</strong> Sie können falsche Daten korrigieren lassen</li>
                    <li><strong>Löschung:</strong> Sie können die Löschung Ihrer Daten verlangen (außer bei gesetzlichen Aufbewahrungspflichten)</li>
                    <li><strong>Einschränkung:</strong> Sie können die Verarbeitung Ihrer Daten einschränken lassen</li>
                    <li><strong>Widerspruch:</strong> Sie können der Verarbeitung Ihrer Daten widersprechen</li>
                    <li><strong>Datenübertragbarkeit:</strong> Sie können Ihre Daten in einem strukturierten, gängigen Format erhalten</li>
                </ul>
                
                <h3 style="color: #003057;">6. Datensicherheit</h3>
                <p>
                    Wir setzen technische und organisatorische Maßnahmen ein, um Ihre Daten vor unberechtigtem Zugriff, Verlust oder Zerstörung zu schützen. Alle Übertragungen erfolgen verschlüsselt (HTTPS/SSL).
                </p>
                
                <h3 style="color: #003057;">7. Weitergabe von Daten</h3>
                <p>
                    Ihre Daten werden nur an unsere Partner-Autohäuser weitergegeben, soweit dies für die Terminbuchung erforderlich ist. Eine Weitergabe an Dritte erfolgt nicht ohne Ihre ausdrückliche Einwilligung, es sei denn, gesetzliche Bestimmungen verlangen dies.
                </p>
                
                <h3 style="color: #003057;">8. Einwilligung</h3>
                <p>
                    Durch die Registrierung erklären Sie sich mit der Erhebung, Verarbeitung und Nutzung Ihrer Daten gemäß dieser Datenschutzerklärung einverstanden. Diese Einwilligung kann jederzeit widerrufen werden.
                </p>
                
                <h3 style="color: #003057;">9. Änderungen der Datenschutzerklärung</h3>
                <p>
                    Wir behalten uns vor, diese Datenschutzerklärung anzupassen. Aktuelle Version: Oktober 2025.
                </p>
                
                <p style="margin-top: 2rem; font-weight: 600; color: #003057;">
                    Bei Fragen zum Datenschutz kontaktieren Sie uns bitte unter:<br>
                    E-Mail: info@be-automobile.de
                </p>
            </div>
            
            <div style="text-align: center; padding-top: 1rem; border-top: 2px solid #e9ecef; flex-shrink: 0;">
                <button onclick="acceptDatenschutzAndClose()" class="btn btn-primary" style="padding: 1rem 3rem; font-size: 1.1rem; width: 100%; max-width: 400px;">
                    <i class="fas fa-check-circle"></i>
                    Verstanden und Akzeptieren
                </button>
            </div>
        </div>
    </div>
    
    <style>
        .modal {
            animation: fadeIn 0.3s;
        }
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        
        /* Mobile Optimierung */
        @media (max-width: 768px) {
            #datenschutz-modal .modal-content {
                margin: 0.5rem !important;
                padding: 1rem !important;
                max-height: 98vh !important;
            }
            
            #datenschutz-modal h2 {
                font-size: 1.3rem !important;
            }
            
            #datenschutz-modal h3 {
                font-size: 1rem !important;
            }
            
            #datenschutz-modal p, #datenschutz-modal li {
                font-size: 0.9rem !important;
                line-height: 1.5 !important;
            }
            
            #datenschutz-modal button[onclick="acceptDatenschutzAndClose()"] {
                padding: 0.875rem 1.5rem !important;
                font-size: 1rem !important;
            }
            
            .close-modal {
                font-size: 1.5rem !important;
            }
        }
    </style>
    
        <script>
        // Redirect nach Login speichern
        const urlParams = new URLSearchParams(window.location.search);
        const redirect = urlParams.get('redirect');
        const event = urlParams.get('event');
        if (redirect) {
            let redirectUrl = redirect;
            if (event) {
                redirectUrl += '?event=' + event;
            }
            localStorage.setItem('redirect_after_login', redirectUrl);
        }
        
        // Tab Switcher
        function switchTab(tab) {
            document.querySelectorAll('.auth-tab').forEach(t => t.classList.remove('active'));
            document.querySelectorAll('.auth-content').forEach(c => c.classList.remove('active'));
            
            const globalDatenschutz = document.getElementById('global-datenschutz');
            const checkbox = document.getElementById('global-datenschutz-agreement');
            
            if (tab === 'login') {
                document.querySelector('.auth-tab:first-child').classList.add('active');
                document.getElementById('login-content').classList.add('active');
                // Button Texts ändern
                document.getElementById('apple-button-text').textContent = 'Mit Apple anmelden';
                document.getElementById('google-button-text').textContent = 'Mit Google anmelden';
                // Datenschutz für Login ausblenden
                if (globalDatenschutz) globalDatenschutz.style.display = 'none';
                // Zusätzlich die alte Checkbox ausblenden
                const oldCheckbox = document.getElementById('datenschutz-agreement');
                if (oldCheckbox && oldCheckbox.closest('div')) {
                    oldCheckbox.closest('div').style.display = 'none';
                }
            } else {
                document.querySelector('.auth-tab:last-child').classList.add('active');
                document.getElementById('register-content').classList.add('active');
                // Button Texts ändern
                document.getElementById('apple-button-text').textContent = 'Mit Apple registrieren';
                document.getElementById('google-button-text').textContent = 'Mit Google registrieren';
                // Datenschutz für Registrierung anzeigen
                if (globalDatenschutz) globalDatenschutz.style.display = 'block';
                // Die alte Checkbox im Form ausblenden (wird durch global ersetzt)
                const oldCheckbox = document.getElementById('datenschutz-agreement');
                if (oldCheckbox && oldCheckbox.closest('div')) {
                    oldCheckbox.closest('div').style.display = 'none';
                }
                
                // Stelle sicher dass Checkbox disabled ist beim ersten Anzeigen
                if (checkbox) {
                    checkbox.disabled = true;
                    checkbox.checked = false;
                }
            }
            
            // Clear alerts
            hideAlert();
        }
        
        // Password Toggle
        function togglePassword(inputId) {
            const input = document.getElementById(inputId);
            const icon = input.nextElementSibling.querySelector('i');
            if (input.type === 'password') {
                input.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                input.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        }
        
        // Datenschutz Modal
        function openDatenschutzModal() {
            document.getElementById('datenschutz-modal').style.display = 'block';
        }
        
        function closeDatenschutzModal() {
            document.getElementById('datenschutz-modal').style.display = 'none';
        }
        
        function acceptDatenschutzAndClose() {
            // Setze die Checkbox automatisch und aktiviere sie
            const checkbox = document.getElementById('global-datenschutz-agreement');
            const container = document.getElementById('global-datenschutz');
            
            if (checkbox && container) {
                checkbox.checked = true;
                checkbox.disabled = false;
                
                // Dezentes visuelles Feedback
                container.style.borderColor = '#28a745';
                container.style.background = '#f8fff9';
                
                // Ändere Text nach Akzeptieren
                const label = container.querySelector('label');
                if (label) {
                    label.querySelector('span').innerHTML = 'Ich stimme der <a href="#" onclick="openDatenschutzModal(); return false;" style="color: #0056b3; text-decoration: underline; font-weight: 600;">Datenschutzerklärung</a> zu und bin mit der Verarbeitung meiner Daten einverstanden. *';
                    label.style.cursor = 'pointer';
                }
                
                const hint = container.querySelector('p');
                if (hint) {
                    hint.innerHTML = '<i class="fas fa-check-circle" style="color: #28a745;"></i> Datenschutz akzeptiert';
                }
                
                toggleAllRegisterButtons();
            }
            closeDatenschutzModal();
        }
        
        function toggleAllRegisterButtons() {
            const checkbox = document.getElementById('global-datenschutz-agreement');
            const container = document.getElementById('global-datenschutz');
            const registerBtn = document.getElementById('register-submit');
            const appleBtn = document.getElementById('apple-signin-button');
            const googleBtn = document.getElementById('google-signin-button');
            
            if (!checkbox) return;
            
            const enable = checkbox.checked;
            
            // Visuelles Feedback für Container
            if (container) {
                if (enable) {
                    container.style.borderColor = '#28a745';
                    container.style.background = '#f0fff4';
                } else {
                    container.style.borderColor = '#e9ecef';
                    container.style.background = '#fff';
                }
            }
            
            // Register Button
            if (registerBtn) {
                registerBtn.disabled = !enable;
                registerBtn.style.opacity = enable ? '1' : '0.5';
                registerBtn.style.cursor = enable ? 'pointer' : 'not-allowed';
            }
            
            // Apple Button
            if (appleBtn) {
                appleBtn.disabled = !enable;
                appleBtn.style.opacity = enable ? '1' : '0.5';
                appleBtn.style.cursor = enable ? 'pointer' : 'not-allowed';
            }
            
            // Google Button
            if (googleBtn) {
                googleBtn.disabled = !enable;
                googleBtn.style.opacity = enable ? '1' : '0.5';
                googleBtn.style.cursor = enable ? 'pointer' : 'not-allowed';
            }
        }
        
        // Warnung zeigen wenn versucht wird ohne Checkbox zu registrieren
        function warnDatenschutzNotAccepted() {
            const container = document.getElementById('global-datenschutz');
            if (container) {
                container.style.borderColor = '#dc3545';
                container.style.background = '#fff5f5';
                container.style.boxShadow = '0 0 0 4px rgba(220, 53, 69, 0.2)';
                
                setTimeout(() => {
                    container.style.boxShadow = '';
                }, 2000);
            }
        }
        
        // Close modal on outside click
        window.onclick = function(event) {
            const modal = document.getElementById('datenschutz-modal');
            if (event.target === modal) {
                closeDatenschutzModal();
            }
        }
        
        // Show Alert
        function showAlert(message, type = 'error') {
            const alert = document.getElementById('alert-message');
            alert.textContent = message;
            alert.className = `alert alert-${type} show`;
            setTimeout(() => {
                hideAlert();
            }, 5000);
        }
        
        function hideAlert() {
            const alert = document.getElementById('alert-message');
            alert.className = 'alert';
        }
        
        // Login Form Handler
        document.getElementById('login-form').addEventListener('submit', async function(e) {
            e.preventDefault();
            hideAlert();
            
            const formData = {
                email: document.getElementById('login-email').value,
                password: document.getElementById('login-password').value
            };
            
            try {
                const response = await fetch('api/login_user.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(formData)
                });
                
                const data = await response.json();
                
                if (data.success) {
                    // Save to session via PHP
                    await saveSession(data.user, data.token);
                    showAlert('Erfolgreich angemeldet!', 'success');
                    setTimeout(() => {
                        const redirect = localStorage.getItem('redirect_after_login') || 'index.php';
                        localStorage.removeItem('redirect_after_login');
                        window.location.href = redirect;
                    }, 1000);
                } else {
                    showAlert(data.message || 'Anmeldung fehlgeschlagen');
                }
            } catch (error) {
                showAlert('Verbindungsfehler. Bitte versuchen Sie es erneut.');
                console.error(error);
            }
        });
        
        // Register Form Handler
        document.getElementById('register-form').addEventListener('submit', async function(e) {
            e.preventDefault();
            hideAlert();
            
            // Prüfe Datenschutz-Checkbox
            const checkbox = document.getElementById('global-datenschutz-agreement');
            if (!checkbox || !checkbox.checked) {
                showAlert('Bitte stimmen Sie der Datenschutzerklärung zu.', 'error');
                warnDatenschutzNotAccepted();
                return;
            }
            
            const formData = {
                first_name: document.getElementById('register-firstname').value,
                last_name: document.getElementById('register-lastname').value,
                email: document.getElementById('register-email').value,
                phone_number: document.getElementById('register-phone').value,
                license_plate: document.getElementById('register-license').value,
                password: document.getElementById('register-password').value,
                datenschutz_accepted: true // Wurde client-seitig geprüft
            };
            
            try {
                const response = await fetch('api/register_user.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(formData)
                });
                
                const data = await response.json();
                
                if (data.success) {
                    // Save to session via PHP
                    await saveSession(data.user, data.token);
                    showAlert('Registrierung erfolgreich!', 'success');
                    setTimeout(() => {
                        const redirect = localStorage.getItem('redirect_after_login') || 'index.php';
                        localStorage.removeItem('redirect_after_login');
                        window.location.href = redirect;
                    }, 1000);
                } else {
                    showAlert(data.message || 'Registrierung fehlgeschlagen');
                }
            } catch (error) {
                showAlert('Verbindungsfehler. Bitte versuchen Sie es erneut.');
                console.error(error);
            }
        });
        
        // Save Session
        async function saveSession(user, token) {
            const response = await fetch('api/save_user_session.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    user: user,
                    token: token
                })
            });
            return await response.json();
        }
        
        // Apple Sign In Handler
        function handleAppleSignIn() {
            // Prüfe ob Datenschutz akzeptiert wurde
            const checkbox = document.getElementById('global-datenschutz-agreement');
            const tab = document.querySelector('.auth-tab.active')?.textContent;
            
            // Nur bei Registrierung (Register-Tab aktiv) prüfen
            if (tab && tab.includes('Registrieren')) {
                if (!checkbox || !checkbox.checked) {
                    showAlert('Bitte stimmen Sie der Datenschutzerklärung zu.', 'error');
                    warnDatenschutzNotAccepted();
                    return;
                }
            }
            
            if (typeof AppleID !== 'undefined') {
                AppleID.auth.init({
                    clientId: 'be.automobileapps.boxenstop',
                    scope: 'name email',
                    redirectURI: 'https://be-automobileapps.de/api/apple_callback.php',
                    usePopup: true
                });
                
                // WICHTIG: requeste AUF JEDEN FALL 'name' UND 'email' um die Namen zu bekommen
                AppleID.auth.signIn({
                    requestedScopes: ['name', 'email']
                }).then(function(response) {
                console.log('=== APPLE SIGN IN RESPONSE ===');
                console.log('Full Response:', JSON.stringify(response, null, 2));
                console.log('Response Type:', typeof response);
                console.log('Response Keys:', Object.keys(response));
                console.log('=======================');
                
                
                // Handle response
                handleAppleCallback(response);
                }).catch(function(error) {
                    console.error('Apple Sign In Error:', error);
                    console.error('Error Details:', JSON.stringify(error, null, 2));
                    showAlert('Apple Anmeldung fehlgeschlagen');
                });
            } else {
                showAlert('Apple Sign In nicht verfügbar');
            }
        }
        
        async function handleAppleCallback(response) {
            try {
                console.log('=== APPLE CALLBACK - FULL RESPONSE ===');
                console.log('Response:', JSON.stringify(response, null, 2));
                console.log('Response Keys:', Object.keys(response));
                console.log('Has response.user?', !!response.user);
                console.log('response.user:', response.user);
                console.log('response.authorization:', response.authorization);
                console.log('========================================');
                
                // Extrahiere User-Informationen aus dem Apple Response
                const appleUserId = response.user || response.authorization?.id_token || 'apple_user_' + Date.now();
                const appleIdentityToken = response.authorization?.id_token;
                
                // Email kann im JWT Token sein (im Backend dekodiert)
                let email = null;
                if (response.user && response.user.email) {
                    email = response.user.email;
                    console.log('Email from response.user:', email);
                } else if (response.email) {
                    email = response.email;
                    console.log('Email from response.email:', email);
                }
                // Falls nicht vorhanden, wird im Backend aus Token extrahiert
                
                // Name-Daten werden nur beim ersten Login geliefert (response.user.name)
                let firstName = null;
                let lastName = null;
                
                // PRIORITÄT 1: response.user.name (nur beim ersten Login)
                if (response.user && response.user.name) {
                    console.log('✅ Found response.user.name:', response.user.name);
                    firstName = response.user.name.givenName || response.user.name.firstName;
                    lastName = response.user.name.familyName || response.user.name.lastName;
                }
                
                // PRIORITÄT 2: Alternative Strukturen
                else if (response.user) {
                    console.log('Checking alternative user structures');
                    firstName = response.user.firstName || response.user.givenName;
                    lastName = response.user.lastName || response.user.familyName;
                }
                
                // PRIORITÄT 3: Direkt im Response
                else if (response.fullName) {
                    console.log('Found fullName directly in response');
                    firstName = response.fullName.givenName || response.fullName.firstName;
                    lastName = response.fullName.familyName || response.fullName.lastName;
                }
                
                console.log('Final extracted names - first:', firstName, 'last:', lastName);
                
                // WICHTIG: Apple liefert user.name NUR beim ersten Login
                // Bei nachfolgenden Logins ist response.user = undefined
                // Fallback: Wenn keine Namen vorhanden, verwende 'Unknown' damit der User weiß dass er editieren muss
                if (!firstName && !lastName) {
                    console.warn('⚠️ No name data from Apple - user needs to edit profile');
                }
                
                console.log('=== EXTRACTED DATA ===');
                console.log('Apple User ID:', appleUserId);
                console.log('Email:', email);
                console.log('First Name:', firstName, firstName ? '(from Apple)' : '(MUST EDIT)');
                console.log('Last Name:', lastName, lastName ? '(from Apple)' : '(MUST EDIT)');
                console.log('=========================');
                
                
                const apiResponse = await fetch('api/login_user.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        apple_user_id: appleUserId,
                        apple_identity_token: appleIdentityToken,
                        email: email,
                        first_name: firstName,
                        last_name: lastName,
                        phone_number: '', // Apple liefert keine Telefonnummer
                        registration_device: navigator.userAgent,
                        registration_ip: await getClientIP(),
                        datenschutz_accepted: true // Wurde client-seitig geprüft
                    })
                });
                
                const data = await apiResponse.json();
                console.log('API Response:', data);
                
                
                if (data.success) {
                    await saveSession(data.user, data.token);
                    showAlert('Erfolgreich mit Apple angemeldet!', 'success');
                    console.log('✅ Login successful - User:', data.user);
                    
                    setTimeout(() => {
                        const redirect = localStorage.getItem('redirect_after_login') || 'profile.php';
                        localStorage.removeItem('redirect_after_login');
                        console.log('🔄 Redirecting to:', redirect);
                        window.location.href = redirect;
                    }, 1500);
                } else {
                    showAlert(data.message || 'Apple Anmeldung fehlgeschlagen');
                }
            } catch (error) {
                showAlert('Verbindungsfehler. Bitte versuchen Sie es erneut.');
                console.error(error);
            }
        }
        
        // Google Sign In Handler
        async function handleGoogleSignIn() {
            // Prüfe ob Datenschutz akzeptiert wurde
            const checkbox = document.getElementById('global-datenschutz-agreement');
            const tab = document.querySelector('.auth-tab.active')?.textContent;
            
            // Nur bei Registrierung (Register-Tab aktiv) prüfen
            if (tab && tab.includes('Registrieren')) {
                if (!checkbox || !checkbox.checked) {
                    showAlert('Bitte stimmen Sie der Datenschutzerklärung zu.', 'error');
                    warnDatenschutzNotAccepted();
                    return;
                }
            }
            
            try {
                // Warte bis Google SDK geladen ist
                if (typeof google === 'undefined' || !google.accounts) {
                    showAlert('Google Sign-In wird geladen... Bitte versuchen Sie es in einem Moment erneut.', 'info');
                    // Warte kurz und versuche es erneut
                    setTimeout(() => {
                        if (typeof google !== 'undefined' && google.accounts) {
                            handleGoogleSignIn();
                        }
                    }, 1000);
                    return;
                }
                
                // Google Client ID - TODO: Ersetzen Sie dies mit Ihrer Google Client ID
                const GOOGLE_CLIENT_ID = '104117899033-jobelrs9lg173bvr3amka2ffql5qicvh.apps.googleusercontent.com';
                
                // Google OAuth mit Popup
                const client = google.accounts.oauth2.initTokenClient({
                    client_id: GOOGLE_CLIENT_ID,
                    scope: 'openid email profile',
                    callback: async (response) => {
                        if (response.error) {
                            console.error('Google OAuth Error:', response);
                            showAlert('Google Anmeldung fehlgeschlagen: ' + (response.error_description || response.error));
                            return;
                        }
                        
                        // Token verwenden um User-Info zu bekommen
                        await fetchGoogleUserInfo(response.access_token);
                    }
                });
                
                // Fordere Access Token an (öffnet Popup)
                client.requestAccessToken();
                
            } catch (error) {
                showAlert('Google Sign-In Fehler');
                console.error('Google Sign-In Error:', error);
            }
        }
        
        // Google User Info mit Access Token abrufen
        async function fetchGoogleUserInfo(accessToken) {
            try {
                const response = await fetch('https://www.googleapis.com/oauth2/v2/userinfo', {
                    headers: {
                        'Authorization': `Bearer ${accessToken}`
                    }
                });
                
                if (!response.ok) {
                    throw new Error('Failed to fetch user info');
                }
                
                const userInfo = await response.json();
                console.log('=== GOOGLE USER INFO ===');
                console.log('Full User Info:', JSON.stringify(userInfo, null, 2));
                console.log('========================');
                
                const googleUserId = userInfo.id;
                const email = userInfo.email;
                const firstName = userInfo.given_name || null;
                const lastName = userInfo.family_name || null;
                const fullName = userInfo.name || null;
                
                // Falls Vorname/Nachname nicht direkt vorhanden, aus fullName extrahieren
                let extractedFirstName = firstName;
                let extractedLastName = lastName;
                
                if (!extractedFirstName && fullName) {
                    const nameParts = fullName.split(' ');
                    extractedFirstName = nameParts[0] || null;
                    extractedLastName = nameParts.slice(1).join(' ') || null;
                }
                
                console.log('=== EXTRACTED DATA ===');
                console.log('Google User ID:', googleUserId);
                console.log('Email:', email);
                console.log('First Name:', extractedFirstName, extractedFirstName ? '(from Google)' : '(MUST EDIT)');
                console.log('Last Name:', extractedLastName, extractedLastName ? '(from Google)' : '(MUST EDIT)');
                console.log('=======================');
                
                const apiResponse = await fetch('api/login_user.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        google_user_id: googleUserId,
                        email: email,
                        first_name: extractedFirstName,
                        last_name: extractedLastName,
                        phone_number: '', // Google liefert keine Telefonnummer
                        registration_device: navigator.userAgent,
                        registration_ip: await getClientIP(),
                        datenschutz_accepted: true // Wurde client-seitig geprüft
                    })
                });
                
                const data = await apiResponse.json();
                console.log('API Response:', data);
                
                if (data.success) {
                    await saveSession(data.user, data.token);
                    showAlert('Erfolgreich mit Google angemeldet!', 'success');
                    console.log('✅ Login successful - User:', data.user);
                    
                    setTimeout(() => {
                        const redirect = localStorage.getItem('redirect_after_login') || 'profile.php';
                        localStorage.removeItem('redirect_after_login');
                        console.log('🔄 Redirecting to:', redirect);
                        window.location.href = redirect;
                    }, 1500);
                } else {
                    showAlert(data.message || 'Google Anmeldung fehlgeschlagen');
                }
            } catch (error) {
                console.error('Error fetching Google user info:', error);
                showAlert('Fehler beim Abrufen der Google-Daten');
            }
        }
        
        
        // Client IP Helper
        async function getClientIP() {
            try {
                const response = await fetch('https://api.ipify.org?format=json');
                const data = await response.json();
                return data.ip || 'Unknown';
            } catch (error) {
                return 'Unknown';
            }
        }
    </script>
</body>
</html>

