<?php
/**
 * Fix Script für wheelset_offers Tabelle
 * Fügt fehlende Spalten hinzu
 */

require_once '../config/database.php';

try {
    $database = new Database();
    $conn = $database->getConnection();
    
    if (!$conn) {
        throw new Exception("Keine Datenbankverbindung möglich");
    }
    
    // Liste der Spalten die hinzugefügt werden sollen
    $columns = [
        'customer_name' => "VARCHAR(255) DEFAULT NULL",
        'customer_no' => "VARCHAR(50) DEFAULT NULL",
        'license_plate' => "VARCHAR(20) DEFAULT NULL",
        'vin' => "VARCHAR(50) DEFAULT NULL",
        'offer_data' => "TEXT DEFAULT NULL COMMENT 'JSON mit Reifendaten, Services, etc.'",
        'letter_content' => "TEXT DEFAULT NULL COMMENT 'Brieftext'",
        'employee_name' => "VARCHAR(255) DEFAULT NULL COMMENT 'Name des Mitarbeiters der das Angebot erstellt'",
        'token' => "VARCHAR(64) DEFAULT NULL COMMENT 'Token für Kundenauswahl'",
        'token_expires_at' => "DATETIME DEFAULT NULL COMMENT 'Token-Ablaufzeit (24h)'",
        'token_used_at' => "DATETIME DEFAULT NULL COMMENT 'Wann wurde Token verwendet'",
        'email_sent' => "TINYINT(1) DEFAULT 0 COMMENT 'Wurde Email versendet'",
        'email_sent_at' => "DATETIME DEFAULT NULL",
        'pdf_filename' => "VARCHAR(255) DEFAULT NULL",
        'status' => "ENUM('draft', 'sent', 'viewed', 'accepted', 'declined') DEFAULT 'draft'",
        'customer_selection' => "TEXT DEFAULT NULL COMMENT 'JSON mit Kundenauswahl'",
        'total_amount' => "DECIMAL(10,2) DEFAULT NULL",
        'created_by' => "INT(11) DEFAULT NULL COMMENT 'User ID'",
        'updated_at' => "DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"
    ];
    
    // Prüfe welche Spalten bereits existieren
    $stmt = $conn->query("SHOW COLUMNS FROM wheelset_offers");
    $existingColumns = [];
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $existingColumns[] = $row['Field'];
    }
    
    $added = [];
    $errors = [];
    
    // Füge fehlende Spalten hinzu
    foreach ($columns as $columnName => $definition) {
        if (!in_array($columnName, $existingColumns)) {
            try {
                // Bestimme Position (nach autohaus_id wenn möglich)
                $afterColumn = 'autohaus_id';
                
                // Spezielle Positionen
                if ($columnName === 'customer_name') {
                    $afterColumn = 'autohaus_id';
                } elseif ($columnName === 'customer_no') {
                    $afterColumn = 'customer_name';
                } elseif ($columnName === 'license_plate') {
                    $afterColumn = 'customer_no';
                } elseif ($columnName === 'vin') {
                    $afterColumn = 'license_plate';
                } elseif ($columnName === 'updated_at') {
                    $afterColumn = 'created_at';
                }
                
                // Prüfe ob AFTER-Spalte existiert
                if (!in_array($afterColumn, $existingColumns)) {
                    $afterColumn = null;
                }
                
                $sql = "ALTER TABLE wheelset_offers ADD COLUMN `$columnName` $definition";
                if ($afterColumn) {
                    $sql .= " AFTER `$afterColumn`";
                }
                
                $conn->exec($sql);
                $added[] = $columnName;
                echo "✓ Spalte '$columnName' hinzugefügt\n";
                
            } catch (PDOException $e) {
                // Ignoriere Fehler wenn Spalte bereits existiert
                if (strpos($e->getMessage(), 'Duplicate column name') === false) {
                    $errors[] = "Fehler bei '$columnName': " . $e->getMessage();
                }
            }
        } else {
            echo "- Spalte '$columnName' existiert bereits\n";
        }
    }
    
    // Füge Indizes hinzu falls nicht vorhanden
    $indexes = [
        'idx_wheelset_autohaus' => "(`wheelset_number`, `autohaus_id`)",
        'idx_token' => "(`token`)",
        'idx_status' => "(`status`)",
        'idx_created_at' => "(`created_at`)"
    ];
    
    $stmt = $conn->query("SHOW INDEXES FROM wheelset_offers");
    $existingIndexes = [];
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $existingIndexes[] = $row['Key_name'];
    }
    
    foreach ($indexes as $indexName => $columns) {
        if (!in_array($indexName, $existingIndexes)) {
            try {
                $conn->exec("ALTER TABLE wheelset_offers ADD INDEX $indexName $columns");
                echo "✓ Index '$indexName' hinzugefügt\n";
            } catch (PDOException $e) {
                echo "⚠ Fehler bei Index '$indexName': " . $e->getMessage() . "\n";
            }
        }
    }
    
    echo "\n=== Zusammenfassung ===\n";
    echo "Hinzugefügte Spalten: " . count($added) . "\n";
    if (count($added) > 0) {
        echo "  - " . implode("\n  - ", $added) . "\n";
    }
    
    if (count($errors) > 0) {
        echo "\nFehler:\n";
        foreach ($errors as $error) {
            echo "  ⚠ $error\n";
        }
    }
    
    if (count($added) > 0 || count($errors) > 0) {
        echo "\n✅ Fertig!\n";
    } else {
        echo "\n✅ Alle Spalten sind bereits vorhanden.\n";
    }
    
} catch (Exception $e) {
    echo "❌ Fehler: " . $e->getMessage() . "\n";
    exit(1);
}

