<?php
/**
 * Cron-Job für automatische WhatsApp-Terminerinnerungen
 * Läuft stündlich und sendet Erinnerungen für Termine am nächsten Tag
 */

require_once '../config/database.php';
require_once '../api/autohaus_whatsapp_notifications.php';

// Log-Datei für Cron-Jobs
$logFile = '../logs/whatsapp_reminders.log';

function logMessage($message) {
    global $logFile;
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] $message" . PHP_EOL;
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
}

try {
    logMessage("WhatsApp-Erinnerungen Cron-Job gestartet");
    
    $db = new Database();
    $conn = $db->getConnection();
    
    // Hole alle aktiven Autohäuser
    $stmt = $conn->prepare("SELECT id, name FROM autohaus WHERE is_active = 1");
    $stmt->execute();
    $autohauser = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $totalReminders = 0;
    $totalErrors = 0;
    
    foreach ($autohauser as $autohaus) {
        logMessage("Verarbeite Autohaus: " . $autohaus['name'] . " (ID: " . $autohaus['id'] . ")");
        
        try {
            // Erstelle WhatsApp-Instanz für dieses Autohaus
            $whatsapp = new AutohausWhatsAppNotifications($autohaus['id']);
            
            // Hole anstehende Termine für morgen
            $stmt = $conn->prepare("
                SELECT a.*, e.name as event_name, e.event_date, e.start_time, e.end_time
                FROM appointments a 
                LEFT JOIN events e ON a.event_id = e.id 
                WHERE a.status = 'approved' 
                AND a.slot_date = DATE_ADD(CURDATE(), INTERVAL 1 DAY)
                AND a.autohaus_id = ?
                AND a.whatsapp_reminder_sent = 0
            ");
            $stmt->execute([$autohaus['id']]);
            $appointments = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            logMessage("Gefundene Termine für morgen: " . count($appointments));
            
            foreach ($appointments as $appointment) {
                try {
                    // Sende Erinnerung
                    $result = $whatsapp->sendAppointmentNotification($appointment['id'], 'reminder');
                    
                    if ($result['success']) {
                        // Markiere Erinnerung als gesendet
                        $updateStmt = $conn->prepare("
                            UPDATE appointments 
                            SET whatsapp_reminder_sent = 1, whatsapp_reminder_sent_at = NOW() 
                            WHERE id = ?
                        ");
                        $updateStmt->execute([$appointment['id']]);
                        
                        $totalReminders++;
                        logMessage("Erinnerung gesendet für Termin ID " . $appointment['id'] . " (" . $appointment['customer_name'] . ")");
                    } else {
                        $totalErrors++;
                        logMessage("Fehler beim Senden der Erinnerung für Termin ID " . $appointment['id'] . ": " . $result['error']);
                    }
                    
                } catch (Exception $e) {
                    $totalErrors++;
                    logMessage("Exception beim Senden der Erinnerung für Termin ID " . $appointment['id'] . ": " . $e->getMessage());
                }
            }
            
        } catch (Exception $e) {
            $totalErrors++;
            logMessage("Fehler beim Verarbeiten von Autohaus " . $autohaus['name'] . ": " . $e->getMessage());
        }
    }
    
    logMessage("WhatsApp-Erinnerungen Cron-Job beendet. Gesendet: $totalReminders, Fehler: $totalErrors");
    
    // Optional: Sende Zusammenfassung per E-Mail an Admin
    if ($totalReminders > 0 || $totalErrors > 0) {
        $summaryMessage = "WhatsApp-Erinnerungen Cron-Job Zusammenfassung:\n\n";
        $summaryMessage .= "Gesendete Erinnerungen: $totalReminders\n";
        $summaryMessage .= "Fehler: $totalErrors\n";
        $summaryMessage .= "Zeit: " . date('Y-m-d H:i:s') . "\n";
        
        // Hier könnte eine E-Mail-Benachrichtigung implementiert werden
        logMessage("Zusammenfassung: " . $summaryMessage);
    }
    
} catch (Exception $e) {
    logMessage("Kritischer Fehler im WhatsApp-Erinnerungen Cron-Job: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    exit;
}

echo json_encode([
    'success' => true, 
    'reminders_sent' => $totalReminders,
    'errors' => $totalErrors,
    'message' => "WhatsApp-Erinnerungen verarbeitet"
]);
?>
