<?php
/**
 * Automatische RESY-Integration
 * Lädt Daten vollautomatisch aus dem RESY-System
 */

require_once 'config/database.php';

// Lade WebDriver-Bibliothek (falls nicht vorhanden, nutze cURL-Fallback)
if (!class_exists('Facebook\WebDriver\Remote\RemoteWebDriver')) {
    require_once 'api/resy_selenium_simple.php';
    $useSimple = true;
} else {
    require_once 'vendor/autoload.php';
    $useSimple = false;
}

class AutomaticResyIntegration {
    private $autohausId;
    private $username;
    private $password;
    private $dealerId;
    private $pdo;
    
    public function __construct($autohausId) {
        $this->autohausId = $autohausId;
        $db = new Database();
        $this->pdo = $db->getConnection();
        $this->loadCredentials();
    }
    
    private function loadCredentials() {
        $stmt = $this->pdo->prepare("SELECT resy_username, resy_password FROM autohaus WHERE id = ?");
        $stmt->execute([$this->autohausId]);
        $data = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$data) {
            throw new Exception("Autohaus nicht gefunden");
        }
        
        $this->username = $data['resy_username'];
        $this->password = $data['resy_password'];
        $this->dealerId = $this->getDealerId();
    }
    
    private function getDealerId() {
        $stmt = $this->pdo->prepare("SELECT name FROM autohaus WHERE id = ?");
        $stmt->execute([$this->autohausId]);
        $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $mapping = [
            'B & E Volvo Kiel' => '453',
            'B & E Volvo Norderstedt' => '452',
            'B & E Volvo Glinde' => '454'
        ];
        
        return $mapping[$autohaus['name']] ?? '453';
    }
    
    /**
     * Führt vollautomatischen Login und Datenabruf durch
     */
    public function execute() {
        try {
            // Prüfe ob Selenium verfügbar ist
            if (!ResySeleniumDriver::isAvailable()) {
                error_log("Selenium nicht verfügbar, nutze Datenbank-Fallback");
                return $this->fallbackToDatabase();
            }
            
            // Nutze einfache Selenium-Integration
            $driver = new ResySeleniumDriver($this->username, $this->password, $this->dealerId);
            
            if ($driver->login()) {
                $html = $driver->loadWheelsets();
                $driver->close();
                
                if ($html) {
                    $wheelsets = $this->parseHTML($html);
                    $this->saveToDatabase($wheelsets);
                    return $wheelsets;
                }
            }
            
            // Falls Login fehlschlägt, nutze Datenbank
            return $this->fallbackToDatabase();
            
        } catch (Exception $e) {
            error_log("RESY-Integration Fehler: " . $e->getMessage());
            return $this->fallbackToDatabase();
        }
    }
    
    /**
     * Parst HTML und extrahiert Reifensätze
     */
    private function parseHTML($html) {
        $wheelsets = [];
        
        // Hier würde das HTML-Parsing erfolgen
        // Vereinfachte Version
        return [];
    }
    
    /**
     * Speichert Reifensätze in der Datenbank
     */
    private function saveToDatabase($wheelsets) {
        // Speichere in stored_wheelsets
        foreach ($wheelsets as $wheelset) {
            $stmt = $this->pdo->prepare("
                INSERT INTO stored_wheelsets 
                (autohaus_id, kennzeichen, marke, modell, oe_nummer, reifengroesse, updated_at)
                VALUES (?, ?, ?, ?, ?, ?, NOW())
                ON DUPLICATE KEY UPDATE 
                updated_at = NOW()
            ");
            $stmt->execute([
                $this->autohausId,
                $wheelset['kennzeichen'] ?? '',
                $wheelset['marke'] ?? '',
                $wheelset['modell'] ?? '',
                $wheelset['oe_nummer'] ?? '',
                $wheelset['reifengroesse'] ?? ''
            ]);
        }
    }
    
    /**
     * Fallback auf Datenbank
     */
    private function fallbackToDatabase() {
        $stmt = $this->pdo->prepare("
            SELECT * FROM stored_wheelsets 
            WHERE autohaus_id = ? AND archiviert = 0
            ORDER BY updated_at DESC
        ");
        $stmt->execute([$this->autohausId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}

// Verwendung
try {
    $integration = new AutomaticResyIntegration(1);
    $data = $integration->execute();
    
    echo json_encode([
        'success' => true,
        'data' => $data,
        'count' => count($data)
    ]);
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}

