<?php
/**
 * send_push_notification.php
 * Sendet Push-Benachrichtigungen über Apple Push Notification Service (APNs)
 * 
 * Verwendung:
 * - POST /api/send_push_notification.php (mit user_id und message)
 * - Oder direkt von admin_approve_user.php aufgerufen
 */

require_once 'config.php';

/**
 * Sendet Push-Benachrichtigung an einen Benutzer
 * 
 * @param int $userId Benutzer-ID
 * @param string $title Titel der Benachrichtigung
 * @param string $body Text der Benachrichtigung
 * @param array $customData Zusätzliche Daten (optional)
 * @return bool Erfolg
 */
function sendPushNotification($userId, $title, $body, $customData = []) {
    global $pdo;
    
    if (!$pdo) {
        error_log("send_push_notification: Keine DB-Verbindung");
        return false;
    }
    
    // Hole alle Device Tokens für diesen Benutzer
    $stmt = $pdo->prepare("SELECT device_token FROM user_device_tokens WHERE user_id = ? AND device_type = 'ios'");
    $stmt->execute([$userId]);
    $tokens = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    if (empty($tokens)) {
        error_log("send_push_notification: Keine Device Tokens für User ID $userId");
        return false;
    }
    
    $successCount = 0;
    $failureCount = 0;
    
    // APNs-Konfiguration
    // WICHTIG: Diese Werte müssen angepasst werden!
    $apnsKeyPath = __DIR__ . '/config/apns_key.p8'; // APNs Auth Key (.p8 Datei)
    $apnsKeyId = ''; // Key ID aus Apple Developer Portal
    $teamId = ''; // Team ID aus Apple Developer Portal
    $bundleId = 'B-E-Autocentrum-GmbH.Test3'; // Bundle ID der App
    $apnsEnvironment = 'development'; // 'development' oder 'production'
    
    // APNs URL (development vs. production)
    $apnsUrl = $apnsEnvironment === 'production' 
        ? 'https://api.push.apple.com'
        : 'https://api.sandbox.push.apple.com';
    
    // Prüfe ob APNs-Konfiguration vorhanden ist
    if (!file_exists($apnsKeyPath) || empty($apnsKeyId) || empty($teamId)) {
        error_log("send_push_notification: APNs-Konfiguration unvollständig. Bitte config/apns_key.p8, Key ID und Team ID setzen.");
        // Fallback: Log für später
        foreach ($tokens as $token) {
            $stmt = $pdo->prepare("
                INSERT INTO push_notification_queue (user_id, device_token, title, body, custom_data, status, created_at)
                VALUES (?, ?, ?, ?, ?, 'pending', NOW())
            ");
            $stmt->execute([$userId, $token, $title, $body, json_encode($customData)]);
        }
        return false;
    }
    
    // JWT Token für APNs generieren
    $jwtToken = generateAPNsJWT($apnsKeyPath, $apnsKeyId, $teamId);
    if (!$jwtToken) {
        error_log("send_push_notification: JWT-Token-Generierung fehlgeschlagen");
        return false;
    }
    
    // Benachrichtigungs-Payload
    $payload = [
        'aps' => [
            'alert' => [
                'title' => $title,
                'body' => $body
            ],
            'sound' => 'default',
            'badge' => 1
        ]
    ];
    
    // Custom Data hinzufügen
    if (!empty($customData)) {
        foreach ($customData as $key => $value) {
            $payload[$key] = $value;
        }
    }
    
    $payloadJson = json_encode($payload);
    
    // Sende an jedes Device Token
    foreach ($tokens as $deviceToken) {
        $url = "$apnsUrl/3/device/$deviceToken";
        
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_2_0,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $payloadJson,
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $jwtToken,
                'Content-Type: application/json',
                'apns-topic: ' . $bundleId,
                'apns-priority: 10'
            ],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($httpCode === 200) {
            $successCount++;
            // Ungültige Tokens aus DB löschen (falls vorhanden)
            // APNs würde bei ungültigen Tokens 410 zurückgeben
        } else {
            $failureCount++;
            error_log("send_push_notification: Fehler für Token (erste 20 Zeichen): " . substr($deviceToken, 0, 20) . " - HTTP $httpCode - $error");
            
            // Wenn Token ungültig (410), aus DB löschen
            if ($httpCode === 410) {
                $stmt = $pdo->prepare("DELETE FROM user_device_tokens WHERE device_token = ?");
                $stmt->execute([$deviceToken]);
                error_log("send_push_notification: Ungültiges Token gelöscht");
            }
        }
    }
    
    error_log("send_push_notification: $successCount erfolgreich, $failureCount fehlgeschlagen für User ID $userId");
    return $successCount > 0;
}

/**
 * Generiert JWT Token für APNs Authentication
 */
function generateAPNsJWT($keyPath, $keyId, $teamId) {
    if (!file_exists($keyPath)) {
        return false;
    }
    
    $key = file_get_contents($keyPath);
    if (!$key) {
        return false;
    }
    
    // Header
    $header = [
        'alg' => 'ES256',
        'kid' => $keyId
    ];
    
    // Payload
    $issuedAt = time();
    $payload = [
        'iss' => $teamId,
        'iat' => $issuedAt
    ];
    
    // JWT Base64 URL Encoding
    $base64UrlEncode = function($data) {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    };
    
    $headerEncoded = $base64UrlEncode(json_encode($header));
    $payloadEncoded = $base64UrlEncode(json_encode($payload));
    
    $data = "$headerEncoded.$payloadEncoded";
    
    // Signatur mit ES256
    // WICHTIG: Für Produktion sollte eine Bibliothek wie phpseclib verwendet werden
    // Hier ein vereinfachter Ansatz - für Produktion muss dies angepasst werden!
    
    $signature = '';
    $pkey = openssl_pkey_get_private($key);
    if (!$pkey) {
        error_log("generateAPNsJWT: Fehler beim Laden des privaten Keys");
        return false;
    }
    
    if (!openssl_sign($data, $signature, $pkey, OPENSSL_ALGO_SHA256)) {
        error_log("generateAPNsJWT: Fehler bei der Signatur");
        openssl_free_key($pkey);
        return false;
    }
    
    openssl_free_key($pkey);
    $signatureEncoded = $base64UrlEncode($signature);
    
    return "$data.$signatureEncoded";
}

// Wenn direkt aufgerufen (z.B. für Tests)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && php_sapi_name() !== 'cli') {
    header('Content-Type: application/json');
    header('Access-Control-Allow-Origin: *');
    header('Access-Control-Allow-Methods: POST, OPTIONS');
    header('Access-Control-Allow-Headers: Content-Type, Authorization');
    
    if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
        exit(0);
    }
    
    try {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (empty($input['user_id']) || empty($input['title']) || empty($input['body'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'user_id, title und body sind erforderlich']);
            exit;
        }
        
        $userId = (int)$input['user_id'];
        $title = $input['title'];
        $body = $input['body'];
        $customData = $input['custom_data'] ?? [];
        
        $result = sendPushNotification($userId, $title, $body, $customData);
        
        if ($result) {
            echo json_encode(['success' => true, 'message' => 'Push-Benachrichtigung gesendet']);
        } else {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => 'Fehler beim Senden der Push-Benachrichtigung']);
        }
    } catch (Exception $e) {
        http_response_code(500);
        error_log("send_push_notification.php ERROR: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Fehler: ' . $e->getMessage()]);
    }
}
?>

