<?php
/**
 * save_user_session.php
 * Erstellt eine PHP-Session aus einem Bearer Token
 * Wird benötigt für Endpunkte, die nur Session-basierte Authentifizierung verwenden (z.B. add_vehicle.php)
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Api-Key, X-Timestamp, X-Nonce, X-Signature');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Session starten
session_start();

// Log-Funktion (optional, kann später entfernt werden)
function logToFile($message, $data = null) {
    $logDir = __DIR__ . '/../logs';
    if (!is_dir($logDir)) {
        @mkdir($logDir, 0755, true);
    }
    $logFile = $logDir . '/save_user_session_' . date('Y-m-d') . '.log';
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] $message";
    if ($data !== null) {
        $logMessage .= "\n" . (is_string($data) ? $data : json_encode($data, JSON_PRETTY_PRINT));
    }
    $logMessage .= "\n" . str_repeat('-', 80) . "\n";
    @file_put_contents($logFile, $logMessage, FILE_APPEND);
}

// Database connection helper - verwendet config.php wie upload_verification_document.php
function getDbConnection() {
    static $pdo = null;
    if ($pdo !== null) return $pdo;
    
    logToFile("getDbConnection: Versuche DB-Verbindung herzustellen");
    
    try {
        // Methode 1: Versuche config.php
        $configPath = __DIR__ . '/../config.php';
        logToFile("getDbConnection: Prüfe config.php", $configPath);
        
        if (file_exists($configPath)) {
            logToFile("getDbConnection: config.php gefunden, lade...");
            require_once $configPath;
            
            if (isset($pdo) && $pdo instanceof PDO) {
                logToFile("getDbConnection SUCCESS", "Verbindung aus config.php geladen (via \$pdo)");
                return $pdo;
            }
            
            if (isset($GLOBALS['pdo']) && $GLOBALS['pdo'] instanceof PDO) {
                logToFile("getDbConnection SUCCESS", "Verbindung aus config.php geladen (via \$GLOBALS['pdo'])");
                $pdo = $GLOBALS['pdo'];
                return $pdo;
            }
        }
        
        // Methode 2: Versuche config/database.php
        $dbConfigPath = __DIR__ . '/../config/database.php';
        if (file_exists($dbConfigPath)) {
            require_once $dbConfigPath;
            if (class_exists('Database')) {
                try {
                    $db = new Database();
                    $pdo = $db->getConnection();
                    if ($pdo instanceof PDO) {
                        logToFile("getDbConnection SUCCESS", "Verbindung via Database-Klasse hergestellt");
                        return $pdo;
                    }
                } catch (Exception $e) {
                    logToFile("getDbConnection WARNING", "Database-Klasse Fehler: " . $e->getMessage());
                }
            }
        }
        
        // Methode 3: Fallback mit Constants
        if (defined('DB_HOST') && defined('DB_NAME') && defined('DB_USER')) {
            $host = DB_HOST;
            $dbname = DB_NAME;
            $user = DB_USER;
            $pass = defined('DB_PASS') ? DB_PASS : '';
            try {
                $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $user, $pass);
                $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
                logToFile("getDbConnection SUCCESS", "Verbindung via Constants hergestellt");
                return $pdo;
            } catch (PDOException $e) {
                logToFile("getDbConnection ERROR", "Constants-Verbindung fehlgeschlagen: " . $e->getMessage());
            }
        }
        
        logToFile("getDbConnection ERROR", "Alle Verbindungsversuche fehlgeschlagen");
        return null;
        
    } catch (Exception $e) {
        logToFile("getDbConnection ERROR", "Exception: " . $e->getMessage());
        return null;
    }
}

// Bearer Token extrahieren
function getBearerToken(): ?string {
    $hdrs = getallheaders();
    if (!$hdrs) return null;
    foreach ($hdrs as $k => $v) {
        if (strtolower($k) === 'authorization' && stripos($v, 'bearer ') === 0) {
            return trim(substr($v, 7));
        }
    }
    return null;
}

// Input lesen
$input = json_decode(file_get_contents('php://input'), true);

// Token aus verschiedenen Quellen ermitteln
$token = null;

// 1. Aus Request Body
if (isset($input['auth_token'])) {
    $token = trim($input['auth_token']);
    logToFile("Token aus Request Body", substr($token, 0, 20) . '...');
} elseif (isset($input['token'])) {
    $token = trim($input['token']);
    logToFile("Token aus Request Body (key: token)", substr($token, 0, 20) . '...');
}

// 2. Aus Authorization Header (Fallback)
if (!$token) {
    $token = getBearerToken();
    if ($token) {
        logToFile("Token aus Authorization Header", substr($token, 0, 20) . '...');
    }
}

// Kein Token gefunden
if (!$token || empty($token)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Kein Token angegeben']);
    exit;
}

// User-ID aus Token ermitteln
$pdo = getDbConnection();
if (!$pdo) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Datenbankverbindung fehlgeschlagen']);
    exit;
}

try {
    $stmt = $pdo->prepare("SELECT id FROM users WHERE auth_token = ? AND token_expires_at > NOW() AND is_active = 1 LIMIT 1");
    $stmt->execute([$token]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$row) {
        logToFile("ERROR: Token ungültig oder abgelaufen", substr($token, 0, 20) . '...');
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Token ungültig oder abgelaufen']);
        exit;
    }
    
    $userId = (int)$row['id'];
    
    // Session setzen
    $_SESSION['user_id'] = $userId;
    $_SESSION['user_logged_in'] = true;
    
    logToFile("SUCCESS: Session erstellt", [
        'user_id' => $userId,
        'session_id' => session_id()
    ]);
    
    echo json_encode([
        'success' => true,
        'message' => 'Session erfolgreich erstellt',
        'session_id' => session_id()
    ]);
    
} catch (PDOException $e) {
    logToFile("ERROR: PDO Exception", $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Datenbankfehler']);
} catch (Exception $e) {
    logToFile("ERROR: Exception", $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Fehler: ' . $e->getMessage()]);
}

