<?php
/**
 * RESY Selenium Driver
 * Nutzt echten Browser für RESY-Login
 */

require_once __DIR__ . '/../vendor/autoload.php';

use Facebook\WebDriver\Remote\RemoteWebDriver;
use Facebook\WebDriver\Remote\DesiredCapabilities;
use Facebook\WebDriver\WebDriverBy;
use Facebook\WebDriver\WebDriverExpectedCondition;
use Facebook\WebDriver\Exception\TimeoutException;

class ResySeleniumDriver {
    private $driver;
    private $username;
    private $password;
    private $dealerId;
    
    public function __construct($username, $password, $dealerId) {
        $this->username = $username;
        $this->password = $password;
        $this->dealerId = $dealerId;
    }
    
    /**
     * Verbindet zum Selenium-Server
     */
    public function connect() {
        try {
            $capabilities = DesiredCapabilities::chrome();
            $capabilities->setCapability('chromeOptions', [
                'args' => [
                    '--headless',
                    '--disable-gpu',
                    '--no-sandbox',
                    '--disable-dev-shm-usage'
                ]
            ]);
            
            $this->driver = RemoteWebDriver::create(
                'http://localhost:4444/wd/hub',
                $capabilities
            );
            
            error_log("✅ Selenium-Verbindung hergestellt");
            return true;
            
        } catch (Exception $e) {
            error_log("❌ Selenium-Fehler: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Führt Login durch
     */
    public function login() {
        try {
            // 1. Navigiere zur Login-Seite
            $this->driver->get('https://resy.dtm-reifen.de/index.php?m=login&a=login');
            error_log("📄 Login-Seite geladen");
            
            // 2. Warte auf Formular
            $this->driver->wait(10)->until(
                WebDriverExpectedCondition::presenceOfElementLocated(WebDriverBy::id('InputUser'))
            );
            
            // 3. Fülle Username
            $usernameField = $this->driver->findElement(WebDriverBy::id('InputUser'));
            $usernameField->sendKeys($this->username);
            error_log("✅ Username eingegeben: " . $this->username);
            
            // 4. Fülle Password
            $passwordField = $this->driver->findElement(WebDriverBy::id('InputPass'));
            $passwordField->sendKeys($this->password);
            error_log("✅ Password eingegeben");
            
            // 5. Submit
            $submitButton = $this->driver->findElement(WebDriverBy::name('button'));
            $submitButton->click();
            error_log("✅ Login-Button geklickt");
            
            // 6. Warte auf Login-Erfolg
            $this->driver->wait(10)->until(
                WebDriverExpectedCondition::not(
                    WebDriverExpectedCondition::presenceOfElementLocated(WebDriverBy::id('InputUser'))
                )
            );
            
            error_log("✅ Login erfolgreich!");
            return true;
            
        } catch (Exception $e) {
            error_log("❌ Login-Fehler: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Lädt Reifensätze
     */
    public function loadWheelsets() {
        try {
            // Navigiere zur Suchseite
            $searchUrl = 'https://resy.dtm-reifen.de/index.php?m=wheelset&a=search&FN=WheelsetSearch&KD_ID=' . $this->dealerId . '&STATUS_ID=20';
            $this->driver->get($searchUrl);
            error_log("📄 Suchseite geladen");
            
            // Warte auf Ergebnisse
            $this->driver->wait(10)->until(
                WebDriverExpectedCondition::presenceOfElementLocated(
                    WebDriverBy::cssSelector('table[id*="WheelsetSearchResult"]')
                )
            );
            
            // Hole HTML
            $html = $this->driver->getPageSource();
            
            error_log("✅ Reifensätze geladen (" . strlen($html) . " Zeichen)");
            return $html;
            
        } catch (TimeoutException $e) {
            error_log("❌ Timeout beim Laden der Reifensätze");
            return null;
        } catch (Exception $e) {
            error_log("❌ Fehler beim Laden: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Beendet die Session
     */
    public function close() {
        if ($this->driver) {
            $this->driver->quit();
            error_log("✅ Selenium-Session beendet");
        }
    }
    
    /**
     * Prüft ob Selenium verfügbar ist
     */
    public static function isAvailable() {
        $ch = curl_init('http://localhost:4444/wd/hub/status');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 2);
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        return $httpCode === 200;
    }
}


