<?php
/**
 * RESY Data Extraction API - Mit korrektem Login
 * Führt den kompletten Login-Prozess durch und navigiert zur Detailseite
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Error handling
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Debug-Logging-Funktion
function debugLog($message) {
    $logFile = __DIR__ . '/../logs/resy_extraction.log';
    $timestamp = date('Y-m-d H:i:s');
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

// HTTP-Helper-Funktion mit Cookie-Unterstützung
function httpRequest($url, $options = []) {
    $defaultOptions = [
        'method' => 'GET',
        'headers' => [],
        'data' => null,
        'timeout' => 30,
        'cookies' => null,
        'follow_redirects' => true
    ];
    
    $options = array_merge($defaultOptions, $options);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, $options['follow_redirects']);
    curl_setopt($ch, CURLOPT_TIMEOUT, $options['timeout']);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    curl_setopt($ch, CURLOPT_COOKIEJAR, '/tmp/cookies.txt');
    curl_setopt($ch, CURLOPT_COOKIEFILE, '/tmp/cookies.txt');
    
    if ($options['cookies']) {
        curl_setopt($ch, CURLOPT_COOKIE, $options['cookies']);
    }
    
    if ($options['method'] === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($options['data']) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $options['data']);
        }
    }
    
    if (!empty($options['headers'])) {
        curl_setopt($ch, CURLOPT_HTTPHEADER, $options['headers']);
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        throw new Exception("cURL Error: $error");
    }
    
    return [
        'body' => $response,
        'http_code' => $httpCode,
        'url' => $url
    ];
}

// RESY Login durchführen
function performResyLogin($baseUrl, $username, $password) {
    debugLog("=== RESY LOGIN START ===");
    debugLog("Base URL: $baseUrl");
    debugLog("Username: $username");
    
    // Schritt 1: Login-Seite laden
    $loginUrl = "$baseUrl/index.php?m=login&a=login";
    debugLog("Lade Login-Seite: $loginUrl");
    
    $response = httpRequest($loginUrl);
    if ($response['http_code'] !== 200) {
        throw new Exception("Login-Seite nicht erreichbar: HTTP " . $response['http_code']);
    }
    
    debugLog("Login-Seite erfolgreich geladen");
    
    // Schritt 2: Login-Formular analysieren
    $dom = new DOMDocument();
    @$dom->loadHTML($response['body']);
    $xpath = new DOMXPath($dom);
    
    // Suche nach dem Login-Formular
    $form = $xpath->query('//form')->item(0);
    if (!$form) {
        throw new Exception("Login-Formular nicht gefunden");
    }
    
    $action = $form->getAttribute('action');
    if (empty($action)) {
        $action = $loginUrl; // Fallback zur aktuellen URL
    }
    
    // Stelle sicher, dass die Action-URL vollständig ist
    if (strpos($action, 'http') !== 0) {
        if (strpos($action, '/') === 0) {
            $action = $baseUrl . $action;
        } else {
            $action = $baseUrl . '/' . $action;
        }
    }
    
    debugLog("Login-Formular gefunden, Action: $action");
    
    // Schritt 3: Login-Daten senden
    $loginData = http_build_query([
        'UserName' => $username,
        'Password' => $password,
        'button' => 'submit'
    ]);
    
    debugLog("Sende Login-Daten...");
    
    $loginResponse = httpRequest($action, [
        'method' => 'POST',
        'data' => $loginData,
        'headers' => [
            'Content-Type: application/x-www-form-urlencoded',
            'Referer: ' . $loginUrl
        ]
    ]);
    
    debugLog("Login-Response Code: " . $loginResponse['http_code']);
    
    // Schritt 4: Prüfe ob Login erfolgreich war
    if (strpos($loginResponse['body'], 'Anmeldename') !== false || 
        strpos($loginResponse['body'], 'UserName') !== false) {
        throw new Exception("Login fehlgeschlagen - Anmeldeformular noch sichtbar");
    }
    
    if (strpos($loginResponse['body'], 'Fehler') !== false || 
        strpos($loginResponse['body'], 'Error') !== false) {
        throw new Exception("Login fehlgeschlagen - Fehlermeldung erkannt");
    }
    
    debugLog("✅ Login erfolgreich!");
    debugLog("=== RESY LOGIN END ===");
    
    return $loginResponse;
}

// Zur Suchseite navigieren
function navigateToSearch($baseUrl) {
    debugLog("=== NAVIGATION TO SEARCH ===");
    
    $searchUrl = "$baseUrl/index.php?m=wheelset&a=search";
    debugLog("Navigiere zur Suchseite: $searchUrl");
    
    $response = httpRequest($searchUrl);
    if ($response['http_code'] !== 200) {
        throw new Exception("Suchseite nicht erreichbar: HTTP " . $response['http_code']);
    }
    
    debugLog("Suchseite erfolgreich geladen");
    debugLog("=== NAVIGATION TO SEARCH END ===");
    
    return $response;
}

// LGS_ID suchen
function searchLgsId($baseUrl, $lgsId) {
    debugLog("=== SEARCH LGS_ID: $lgsId ===");
    
    $searchUrl = "$baseUrl/index.php?m=wheelset&a=search";
    $searchData = http_build_query([
        'LGS_ID' => $lgsId,
        'button' => 'submit'
    ]);
    
    debugLog("Suche nach LGS_ID: $lgsId");
    
    $response = httpRequest($searchUrl, [
        'method' => 'POST',
        'data' => $searchData,
        'headers' => [
            'Content-Type: application/x-www-form-urlencoded',
            'Referer: ' . $searchUrl
        ]
    ]);
    
    debugLog("Suche-Response Code: " . $response['http_code']);
    
    // Prüfe ob wir direkt zur Detailseite weitergeleitet wurden
    if (strpos($response['body'], 'WheelHALTERNAME') !== false) {
        debugLog("✅ Direkt zur Detailseite weitergeleitet!");
        return $response;
    }
    
    // Prüfe ob Suchergebnisse vorhanden sind
    if (strpos($response['body'], 'Einlagerungsnummer') !== false) {
        debugLog("Suchergebnisse gefunden, suche Detail-Link...");
        
        $dom = new DOMDocument();
        @$dom->loadHTML($response['body']);
        $xpath = new DOMXPath($dom);
        
        // Suche nach Links zu Detailseiten
        $links = $xpath->query('//a[contains(@href, "LGS_ID")]');
        foreach ($links as $link) {
            $href = $link->getAttribute('href');
            if (strpos($href, $lgsId) !== false) {
                debugLog("Detail-Link gefunden: $href");
                
                // Stelle sicher, dass die URL vollständig ist
                if (strpos($href, 'http') !== 0) {
                    if (strpos($href, '/') === 0) {
                        $detailUrl = $baseUrl . $href;
                    } else {
                        $detailUrl = $baseUrl . '/' . $href;
                    }
                } else {
                    $detailUrl = $href;
                }
                
                debugLog("Navigiere zur Detailseite: $detailUrl");
                
                $detailResponse = httpRequest($detailUrl);
                if ($detailResponse['http_code'] === 200 && strpos($detailResponse['body'], 'WheelHALTERNAME') !== false) {
                    debugLog("✅ Detailseite erfolgreich geladen!");
                    return $detailResponse;
                }
            }
        }
    }
    
    throw new Exception("Keine Detailseite für LGS_ID $lgsId gefunden");
}

// HTML-Daten extrahieren
function extractDataFromHtml($html, $field) {
    $dom = new DOMDocument();
    @$dom->loadHTML($html);
    $xpath = new DOMXPath($dom);
    
    debugLog("=== HTML-ANALYSE FÜR FELD: $field ===");
    
    // Mapping der Feldnamen zu HTML-IDs
    $fieldMapping = [
        'fahrzeughalter' => 'WheelHALTERNAME',
        'kennzeichen' => 'WheelKENNZEICHEN',
        'fin' => 'WheelFAHRGESTNR',
        'eingangsnummer' => 'WheelLGS_ID',
        'ladestelle' => 'WheelKD_ID', // Autohaus ist die Ladestelle
        'bemerkung' => 'KDBEMERK',
        'ruecklieferungsdatum' => 'WheelRUECKLIEFERTERMIN',
        'rueckgabe_bemerkung' => 'WheelRUECKAKTBEMERKUNG'
    ];
    
    $htmlId = $fieldMapping[$field] ?? null;
    if (!$htmlId) {
        throw new Exception("Unbekanntes Feld: $field");
    }
    
    debugLog("Suche nach Element mit ID: $htmlId");
    
    // Suche nach dem Element mit der entsprechenden ID
    $element = $xpath->query("//*[@id='$htmlId']")->item(0);
    
    if (!$element) {
        debugLog("Element mit ID '$htmlId' nicht gefunden für Feld '$field'");
        return '';
    }
    
    // Extrahiere den Textinhalt
    $value = trim($element->textContent);
    
    // Für Bemerkung-Felder: Suche nach textarea
    if ($field === 'bemerkung') {
        $textarea = $xpath->query("//textarea[@name='KDBEMERK']")->item(0);
        if ($textarea) {
            $value = trim($textarea->textContent);
            debugLog("Bemerkung aus textarea extrahiert: '$value'");
        }
    }
    
    debugLog("Feld '$field' (ID: $htmlId) extrahiert: '$value'");
    
    // Wenn Wert leer ist, suche nach <b> Tags innerhalb des Elements
    if (empty($value)) {
        $boldElement = $xpath->query(".//b", $element)->item(0);
        if ($boldElement) {
            $value = trim($boldElement->textContent);
            debugLog("Wert aus <b> Tag extrahiert: '$value'");
        }
    }
    
    debugLog("=== ENDE HTML-ANALYSE FÜR FELD: $field ===");
    return $value;
}

// Hauptverarbeitung
try {
    debugLog("=== RESY LOGIN API Request Started ===");
    
    // JSON-Input lesen
    $rawInput = file_get_contents('php://input');
    debugLog("Raw Input Length: " . strlen($rawInput));
    
    if (empty($rawInput)) {
        throw new Exception('Keine Eingabedaten empfangen');
    }
    
    $input = json_decode($rawInput, true);
    
    if (!$input) {
        $error = json_last_error_msg();
        debugLog("JSON Decode Error: " . $error);
        throw new Exception('Keine gültigen JSON-Daten empfangen: ' . $error);
    }
    
    $action = $input['action'] ?? '';
    debugLog("Empfangene Aktion: $action");
    
    // Datenbankverbindung
    try {
        require_once __DIR__ . '/../config/database.php';
        $db = new Database();
        $conn = $db->getConnection();
        debugLog("Datenbankverbindung erfolgreich");
    } catch (Exception $e) {
        debugLog("Datenbankfehler: " . $e->getMessage());
        throw new Exception('Datenbankfehler: ' . $e->getMessage());
    }
    
    switch ($action) {
        case 'login_and_navigate':
            $appointmentId = $input['appointment_id'] ?? null;
            $lgsId = $input['lgsId'] ?? null;
            debugLog("Login und Navigation für Appointment ID: $appointmentId, LGS_ID: $lgsId");
            
            if (!$appointmentId || !$lgsId) {
                throw new Exception('Appointment ID oder LGS_ID fehlt');
            }
            
            // Autohaus-Daten aus Appointment laden
            $stmt = $conn->prepare("
                SELECT a.resy_base_url, a.resy_username, a.resy_password 
                FROM autohaus a 
                JOIN appointments ap ON a.id = ap.autohaus_id 
                WHERE ap.id = ?
            ");
            $stmt->execute([$appointmentId]);
            $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$autohaus) {
                throw new Exception('Autohaus-Daten nicht gefunden für Appointment ID: ' . $appointmentId);
            }
            
            debugLog("Autohaus gefunden: " . json_encode($autohaus));
            
            // Schritt 1: Login durchführen
            performResyLogin($autohaus['resy_base_url'], $autohaus['resy_username'], $autohaus['resy_password']);
            
            // Schritt 2: Zur Suchseite navigieren
            navigateToSearch($autohaus['resy_base_url']);
            
            // Schritt 3: LGS_ID suchen und zur Detailseite navigieren
            $detailResponse = searchLgsId($autohaus['resy_base_url'], $lgsId);
            
            $result = [
                'success' => true,
                'detailHtml' => $detailResponse['body'],
                'url' => $detailResponse['url'],
                'approach' => 'login_and_navigate',
                'message' => 'Erfolgreich eingeloggt und zur Detailseite navigiert'
            ];
            
            echo json_encode($result);
            break;
            
        case 'extract_field':
            $field = $input['field'] ?? null;
            $html = $input['html'] ?? null;
            debugLog("Extract field: $field");
            
            if (!$field) {
                throw new Exception('Feld-Name fehlt');
            }
            
            if (!$html) {
                throw new Exception('HTML-Inhalt fehlt');
            }
            
            try {
                $value = extractDataFromHtml($html, $field);
                
                $result = [
                    'success' => true,
                    'value' => $value,
                    'field' => $field,
                    'message' => "Feld '$field' erfolgreich extrahiert"
                ];
                
                echo json_encode($result);
                
            } catch (Exception $e) {
                debugLog("Feld-Extraktion fehlgeschlagen: " . $e->getMessage());
                $result = [
                    'success' => false,
                    'error' => $e->getMessage(),
                    'field' => $field
                ];
                echo json_encode($result);
            }
            break;
            
        default:
            throw new Exception("Unbekannte Aktion: $action");
    }
    
} catch (Exception $e) {
    debugLog("API-Fehler: " . $e->getMessage());
    debugLog("Stack Trace: " . $e->getTraceAsString());
    
    // Sicherstellen, dass nur JSON ausgegeben wird
    http_response_code(500);
    $result = [
        'success' => false,
        'error' => $e->getMessage(),
        'debug' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'timestamp' => date('Y-m-d H:i:s')
        ]
    ];
    echo json_encode($result);
}

debugLog("=== RESY LOGIN API Request Ended ===");
?>
