<?php
/**
 * RESY Data Extraction API - Breakthrough Solution v2
 * Intelligente Suchstrategie für alle LGS_ID-Formate (mit und ohne Punkte)
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Error handling
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Debug-Logging-Funktion
function debugLog($message) {
    $logFile = __DIR__ . '/../logs/resy_extraction.log';
    $timestamp = date('Y-m-d H:i:s');
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

// Bild-Cache-Manager einbinden
require_once __DIR__ . '/image_cache_manager.php';

// Funktion zum Speichern von Bildern beim ersten Laden
function cacheImagesForAppointment($appointmentId, $tireData) {
    $imageCache = new ImageCacheManager();
    $cachedCount = 0;
    
    debugLog("Starte Bild-Caching für Appointment ID: $appointmentId");
    
    foreach (['reifen_vl', 'reifen_vr', 'reifen_hl', 'reifen_hr'] as $position) {
        if (isset($tireData[$position]['images']) && is_array($tireData[$position]['images'])) {
            foreach ($tireData[$position]['images'] as $image) {
                if (isset($image['lgfoto_id'])) {
                    $lgfotoId = $image['lgfoto_id'];
                    
                    // Prüfe ob Bild bereits gecacht ist
                    if (!$imageCache->isImageCached($lgfotoId)) {
                        debugLog("Lade und speichere Bild für LGFOTO_ID: $lgfotoId");
                        
                        // Lade Bild von RESY
                        $imageData = loadImageFromResy($lgfotoId, $appointmentId);
                        
                        if ($imageData && $imageData['success']) {
                            $result = $imageCache->cacheImage(
                                $lgfotoId, 
                                $imageData['data'], 
                                $imageData['content_type'], 
                                $appointmentId
                            );
                            
                            if ($result['success']) {
                                $cachedCount++;
                                debugLog("✅ Bild gecacht: LGFOTO_ID $lgfotoId, Größe: " . $result['size'] . " Bytes");
                            } else {
                                debugLog("❌ Fehler beim Cachen von LGFOTO_ID $lgfotoId: " . $result['error']);
                            }
                        } else {
                            debugLog("❌ Bild konnte nicht von RESY geladen werden: LGFOTO_ID $lgfotoId");
                        }
                    } else {
                        debugLog("Bild bereits gecacht: LGFOTO_ID $lgfotoId");
                    }
                }
            }
        }
    }
    
    debugLog("Bild-Caching abgeschlossen: $cachedCount neue Bilder gespeichert");
    return $cachedCount;
}

// Funktion zum Laden von Bildern von RESY
function loadImageFromResy($lgfotoId, $appointmentId) {
    try {
        // Hole Autohaus-Daten
        global $conn;
        $stmt = $conn->prepare("
            SELECT a.resy_base_url, a.resy_username, a.resy_password 
            FROM autohaus a 
            JOIN appointments ap ON a.id = ap.autohaus_id 
            WHERE ap.id = ?
        ");
        $stmt->execute([$appointmentId]);
        $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$autohaus) {
            return ['success' => false, 'error' => 'Autohaus-Daten nicht gefunden'];
        }
        
        // Erstelle verschiedene mögliche Bild-URLs
        $possibleUrls = [
            rtrim($autohaus['resy_base_url'], '/') . '/index.php?m=wheelphoto&a=show&LGFOTO_ID=' . $lgfotoId . '&FN=ShowPruefDoc',
            rtrim($autohaus['resy_base_url'], '/') . '/wheelphoto/show/' . $lgfotoId,
            rtrim($autohaus['resy_base_url'], '/') . '/images/wheelphoto/' . $lgfotoId . '.jpg',
            rtrim($autohaus['resy_base_url'], '/') . '/uploads/wheelphoto/' . $lgfotoId . '.jpg'
        ];
        
        $cookieFile = sys_get_temp_dir() . '/resy_cookies_breakthrough.txt';
        
        // Teste verschiedene URLs
        foreach ($possibleUrls as $imageUrl) {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $imageUrl);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_COOKIEFILE, $cookieFile);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
            curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Accept: image/webp,image/apng,image/*,*/*;q=0.8',
                'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
                'Referer: ' . $autohaus['resy_base_url']
            ]);
            
            $imageData = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $contentType = curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
            
            curl_close($ch);
            
            // Prüfe ob es ein Bild ist
            if ($httpCode === 200 && 
                !empty($imageData) && 
                strpos($contentType, 'image/') === 0 &&
                strpos($imageData, '<html') === false &&
                strpos($imageData, '<!DOCTYPE') === false) {
                
                return [
                    'success' => true,
                    'data' => $imageData,
                    'content_type' => $contentType,
                    'url' => $imageUrl
                ];
            }
        }
        
        return ['success' => false, 'error' => 'Keine funktionierende Bild-URL gefunden'];
        
    } catch (Exception $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

// Cache-Funktionen für Performance-Optimierung
function getCachedData($storageNumber, $appointmentId = null) {
    global $conn;
    
    try {
        // Prüfe ob Tabelle existiert
        $tableCheck = $conn->query("SHOW TABLES LIKE 'resy_data_cache'");
        if ($tableCheck->rowCount() === 0) {
            debugLog("❌ Tabelle 'resy_data_cache' existiert nicht - Cache nicht verfügbar");
            return null;
        }
        
        $sql = "SELECT vehicle_data, tire_data, updated_at, expires_at FROM resy_data_cache 
                WHERE einlagerungsnummer = ? AND (appointment_id = ? OR appointment_id IS NULL) 
                AND expires_at > NOW() 
                ORDER BY updated_at DESC LIMIT 1";
        
        $stmt = $conn->prepare($sql);
        $stmt->execute([$storageNumber, $appointmentId]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result) {
            debugLog("✅ Cache-Hit für Storage Number: $storageNumber (Appointment ID: " . ($appointmentId ?? 'NULL') . ")");
            debugLog("Cache-Daten aktualisiert: " . $result['updated_at'] . ", läuft ab: " . $result['expires_at']);
            return [
                'vehicle_data' => json_decode($result['vehicle_data'], true),
                'tire_data' => json_decode($result['tire_data'], true)
            ];
        }
        
        debugLog("❌ Cache-Miss für Storage Number: $storageNumber (Appointment ID: " . ($appointmentId ?? 'NULL') . ")");
        
        // Debug: Prüfe ob überhaupt Einträge für diese Storage Number existieren
        $debugSql = "SELECT COUNT(*) as count, MAX(updated_at) as last_update FROM resy_data_cache WHERE einlagerungsnummer = ?";
        $debugStmt = $conn->prepare($debugSql);
        $debugStmt->execute([$storageNumber]);
        $debugResult = $debugStmt->fetch(PDO::FETCH_ASSOC);
        
        if ($debugResult['count'] > 0) {
            debugLog("⚠️ Es existieren $debugResult[count] Data-Cache-Einträge für Storage Number $storageNumber, aber alle sind abgelaufen (letzte Aktualisierung: $debugResult[last_update])");
        } else {
            debugLog("⚠️ Keine Data-Cache-Einträge für Storage Number $storageNumber gefunden");
        }
        
        return null;
        
    } catch (Exception $e) {
        debugLog("❌ Cache-Fehler: " . $e->getMessage());
        return null;
    }
}

function saveCachedData($storageNumber, $vehicleData, $tireData, $appointmentId = null) {
    global $conn;
    
    try {
        // Prüfe ob Tabelle existiert
        $tableCheck = $conn->query("SHOW TABLES LIKE 'resy_data_cache'");
        if ($tableCheck->rowCount() === 0) {
            debugLog("❌ Tabelle 'resy_data_cache' existiert nicht - kann Daten nicht cachen");
            return false;
        }
        
        if (empty($vehicleData) && empty($tireData)) {
            debugLog("⚠️ Leere Daten für Storage Number: $storageNumber - wird nicht gecacht");
            return false;
        }
        
        debugLog("Speichere Data-Cache für Storage Number: $storageNumber (Appointment ID: " . ($appointmentId ?? 'NULL') . ")");
        
        // Lösche alte Cache-Einträge für diese Storage Number
        $deleteSql = "DELETE FROM resy_data_cache WHERE einlagerungsnummer = ?";
        $deleteStmt = $conn->prepare($deleteSql);
        $deleteStmt->execute([$storageNumber]);
        
        // Speichere neue Daten mit längerer Ablaufzeit (24 Stunden)
        $insertSql = "INSERT INTO resy_data_cache (einlagerungsnummer, appointment_id, vehicle_data, tire_data, expires_at) 
                      VALUES (?, ?, ?, ?, DATE_ADD(NOW(), INTERVAL 24 HOUR))";
        $insertStmt = $conn->prepare($insertSql);
        $insertStmt->execute([
            $storageNumber,
            $appointmentId,
            json_encode($vehicleData),
            json_encode($tireData)
        ]);
        
        debugLog("✅ Data-Cache erfolgreich gespeichert für Storage Number: $storageNumber (läuft ab in 24 Stunden)");
        return true;
        
    } catch (Exception $e) {
        debugLog("❌ Data-Cache-Speicher-Fehler: " . $e->getMessage());
        return false;
    }
}

function getCachedHtml($storageNumber, $appointmentId = null) {
    global $conn;
    
    try {
        // Prüfe ob Tabelle existiert
        $tableCheck = $conn->query("SHOW TABLES LIKE 'resy_html_cache'");
        if ($tableCheck->rowCount() === 0) {
            debugLog("❌ Tabelle 'resy_html_cache' existiert nicht - HTML-Cache nicht verfügbar");
            return null;
        }
        
        $sql = "SELECT html_content, resy_base_url, updated_at, expires_at FROM resy_html_cache 
                WHERE einlagerungsnummer = ? AND (appointment_id = ? OR appointment_id IS NULL) 
                AND expires_at > NOW() 
                ORDER BY updated_at DESC LIMIT 1";
        
        $stmt = $conn->prepare($sql);
        $stmt->execute([$storageNumber, $appointmentId]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result) {
            $htmlLength = strlen($result['html_content']);
            $updatedAt = $result['updated_at'];
            $expiresAt = $result['expires_at'];
            
            debugLog("✅ HTML-Cache-Hit für Storage Number: $storageNumber (Appointment ID: " . ($appointmentId ?? 'NULL') . ")");
            debugLog("HTML-Länge: $htmlLength Zeichen, Aktualisiert: $updatedAt, Läuft ab: $expiresAt");
            
            return [
                'html' => $result['html_content'],
                'resy_base_url' => $result['resy_base_url'],
                'updated_at' => $updatedAt,
                'expires_at' => $expiresAt,
                'length' => $htmlLength
            ];
        }
        
        debugLog("❌ HTML-Cache-Miss für Storage Number: $storageNumber (Appointment ID: " . ($appointmentId ?? 'NULL') . ")");
        
        // Debug: Prüfe ob überhaupt Einträge für diese Storage Number existieren
        $debugSql = "SELECT COUNT(*) as count, MAX(updated_at) as last_update FROM resy_html_cache WHERE einlagerungsnummer = ?";
        $debugStmt = $conn->prepare($debugSql);
        $debugStmt->execute([$storageNumber]);
        $debugResult = $debugStmt->fetch(PDO::FETCH_ASSOC);
        
        if ($debugResult['count'] > 0) {
            debugLog("⚠️ Es existieren $debugResult[count] HTML-Cache-Einträge für Storage Number $storageNumber, aber alle sind abgelaufen (letzte Aktualisierung: $debugResult[last_update])");
        } else {
            debugLog("⚠️ Keine HTML-Cache-Einträge für Storage Number $storageNumber gefunden");
        }
        
        return null;
        
    } catch (Exception $e) {
        debugLog("❌ HTML-Cache-Fehler: " . $e->getMessage());
        return null;
    }
}

function saveCachedHtml($storageNumber, $htmlContent, $resyBaseUrl, $appointmentId = null) {
    global $conn;
    
    try {
        // Prüfe ob Tabelle existiert
        $tableCheck = $conn->query("SHOW TABLES LIKE 'resy_html_cache'");
        if ($tableCheck->rowCount() === 0) {
            debugLog("❌ Tabelle 'resy_html_cache' existiert nicht - kann HTML nicht cachen");
            return false;
        }
        
        if (empty($htmlContent)) {
            debugLog("⚠️ Leerer HTML-Inhalt für Storage Number: $storageNumber - wird nicht gecacht");
            return false;
        }
        
        $htmlLength = strlen($htmlContent);
        debugLog("Speichere HTML-Cache für Storage Number: $storageNumber (Länge: $htmlLength Zeichen, Appointment ID: " . ($appointmentId ?? 'NULL') . ")");
        
        // Lösche alte HTML-Cache-Einträge
        $deleteSql = "DELETE FROM resy_html_cache WHERE einlagerungsnummer = ?";
        $deleteStmt = $conn->prepare($deleteSql);
        $deleteStmt->execute([$storageNumber]);
        
        // Speichere neues HTML mit längerer Ablaufzeit (24 Stunden)
        $insertSql = "INSERT INTO resy_html_cache (einlagerungsnummer, appointment_id, html_content, resy_base_url, expires_at) 
                      VALUES (?, ?, ?, ?, DATE_ADD(NOW(), INTERVAL 24 HOUR))";
        $insertStmt = $conn->prepare($insertSql);
        $insertStmt->execute([
            $storageNumber,
            $appointmentId,
            $htmlContent,
            $resyBaseUrl
        ]);
        
        debugLog("✅ HTML-Cache erfolgreich gespeichert für Storage Number: $storageNumber (Länge: $htmlLength Zeichen, läuft ab in 24 Stunden)");
        return true;
        
    } catch (Exception $e) {
        debugLog("❌ HTML-Cache-Speicher-Fehler: " . $e->getMessage());
        return false;
    }
}

// HTTP-Helper-Funktion mit verbessertem Session-Management
function httpRequest($url, $options = []) {
    $defaultOptions = [
        'method' => 'GET',
        'headers' => [],
        'data' => null,
        'timeout' => 30,
        'cookies' => null,
        'follow_redirects' => true,
        'cookie_file' => sys_get_temp_dir() . '/resy_cookies_breakthrough.txt'
    ];
    
    $options = array_merge($defaultOptions, $options);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, $options['follow_redirects']);
    curl_setopt($ch, CURLOPT_TIMEOUT, $options['timeout']);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36');
    
    // Verbessertes Session-Management für RESY-System
    curl_setopt($ch, CURLOPT_COOKIEJAR, $options['cookie_file']);
    curl_setopt($ch, CURLOPT_COOKIEFILE, $options['cookie_file']);
    curl_setopt($ch, CURLOPT_COOKIESESSION, false);
    curl_setopt($ch, CURLOPT_FRESH_CONNECT, false);
    curl_setopt($ch, CURLOPT_COOKIELIST, '');
    curl_setopt($ch, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_1);
    
    // Wichtige Session-Parameter für RESY
    curl_setopt($ch, CURLOPT_AUTOREFERER, true);
    curl_setopt($ch, CURLOPT_MAXREDIRS, 5);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
    curl_setopt($ch, CURLOPT_LOW_SPEED_TIME, 30);
    curl_setopt($ch, CURLOPT_LOW_SPEED_LIMIT, 1000);
    
    if ($options['cookies']) {
        curl_setopt($ch, CURLOPT_COOKIE, $options['cookies']);
    }
    
    if ($options['method'] === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($options['data']) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $options['data']);
        }
    }
    
    if (!empty($options['headers'])) {
        curl_setopt($ch, CURLOPT_HTTPHEADER, $options['headers']);
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    $finalUrl = curl_getinfo($ch, CURLINFO_EFFECTIVE_URL);
    
    curl_close($ch);
    
    if ($error) {
        throw new Exception("cURL Error: $error");
    }
    
    return [
        'body' => $response,
        'http_code' => $httpCode,
        'url' => $url,
        'final_url' => $finalUrl
    ];
}

// Session-Validierung für RESY-System
function validateResySession($baseUrl) {
    debugLog("=== VALIDATE RESY SESSION START ===");
    
    $cookieFile = sys_get_temp_dir() . '/resy_cookies_breakthrough.txt';
    
    // Prüfe ob Cookie-Datei existiert
    if (!file_exists($cookieFile)) {
        debugLog("❌ Cookie-Datei nicht gefunden: $cookieFile");
        return false;
    }
    
    $cookieSize = filesize($cookieFile);
    debugLog("Cookie-Datei Größe: $cookieSize Bytes");
    
    if ($cookieSize < 50) {
        debugLog("❌ Cookie-Datei zu klein - wahrscheinlich leer oder ungültig");
        return false;
    }
    
    // Teste Session mit einer einfachen Anfrage
    try {
        $testUrl = $baseUrl . '/index.php?m=wheelset&a=search';
        debugLog("Teste Session mit URL: $testUrl");
        
        $response = httpRequest($testUrl, [
            'headers' => [
                'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
                'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8',
                'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
                'Accept-Encoding: gzip, deflate, br',
                'Referer: ' . $baseUrl,
                'Connection: keep-alive',
                'Upgrade-Insecure-Requests: 1'
            ],
            'cookie_file' => $cookieFile
        ]);
        
        debugLog("Session-Test Response Code: " . $response['http_code']);
        debugLog("Session-Test Final URL: " . $response['final_url']);
        
        // Prüfe ob wir zur Login-Seite umgeleitet wurden
        $isLoginPage = strpos($response['body'], 'Anmeldename') !== false ||
                       strpos($response['body'], 'UserName') !== false ||
                       strpos($response['final_url'], 'login') !== false;
        
        if ($isLoginPage) {
            debugLog("❌ Session ungültig - zur Login-Seite umgeleitet");
            return false;
        }
        
        // Prüfe ob wir auf der Hauptseite oder einer gültigen RESY-Seite sind
        $isValidResyPage = strpos($response['body'], 'RESY') !== false ||
                          strpos($response['body'], 'Übersicht') !== false ||
                          strpos($response['body'], 'Wheelset') !== false ||
                          strpos($response['body'], 'LGS_ID') !== false ||
                          strpos($response['final_url'], 'resyweb') !== false ||
                          strpos($response['final_url'], 'wheelset') !== false;
        
        if ($isValidResyPage) {
            debugLog("✅ Session gültig - RESY-Seite erreicht");
            return true;
        }
        
        debugLog("⚠️ Session-Status unklar - weder Login noch gültige RESY-Seite");
        return false;
        
    } catch (Exception $e) {
        debugLog("❌ Session-Validierung fehlgeschlagen: " . $e->getMessage());
        return false;
    }
}

// Cookie-Synchronisation zwischen APIs
function syncCookiesBetweenApis() {
    $cookieFile = sys_get_temp_dir() . '/resy_cookies_breakthrough.txt';
    $imageProxyCookieFile = sys_get_temp_dir() . '/resy_cookies_breakthrough.txt';
    
    // Stelle sicher, dass beide APIs die gleiche Cookie-Datei verwenden
    if (file_exists($cookieFile)) {
        copy($cookieFile, $imageProxyCookieFile);
        debugLog("Cookie-Synchronisation: Haupt-API → Image-Proxy");
    } elseif (file_exists($imageProxyCookieFile)) {
        copy($imageProxyCookieFile, $cookieFile);
        debugLog("Cookie-Synchronisation: Image-Proxy → Haupt-API");
    }
}

// RESY Login mit verbesserter Session-Verwaltung
function performBreakthroughLogin($baseUrl, $username, $password) {
    debugLog("=== BREAKTHROUGH LOGIN START ===");
    debugLog("Base URL: $baseUrl");
    debugLog("Username: $username");
    
    $cookieFile = sys_get_temp_dir() . '/resy_cookies_breakthrough.txt';
    
    // Lösche alte Cookie-Datei für frische Session
    if (file_exists($cookieFile)) {
        unlink($cookieFile);
        debugLog("Alte Cookie-Datei gelöscht für frische Session");
    }
    
    $loginUrl = "$baseUrl/index.php?m=login&a=login";
    debugLog("Lade Login-Seite: $loginUrl");
    
    $response = httpRequest($loginUrl, [
        'cookie_file' => $cookieFile,
        'headers' => [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Accept-Encoding: gzip, deflate, br',
            'Connection: keep-alive',
            'Upgrade-Insecure-Requests: 1'
        ]
    ]);
    
    if ($response['http_code'] !== 200) {
        throw new Exception("Login-Seite nicht erreichbar: HTTP " . $response['http_code']);
    }
    
    debugLog("Login-Seite erfolgreich geladen");
    
    $postData = [
        'FN' => 'login',
        'UserName' => $username,
        'Password' => $password,
        'button' => 'submit'
    ];
    
    $loginDataString = http_build_query($postData);
    debugLog("Login-Daten: " . str_replace($password, '***', $loginDataString));
    
    $loginResponse = httpRequest($loginUrl, [
        'method' => 'POST',
        'data' => $loginDataString,
        'headers' => [
            'Content-Type: application/x-www-form-urlencoded',
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Accept-Encoding: gzip, deflate, br',
            'Connection: keep-alive',
            'Upgrade-Insecure-Requests: 1',
            'Referer: ' . $loginUrl
        ],
        'cookie_file' => $cookieFile
    ]);
    
    debugLog("Login-Response Code: " . $loginResponse['http_code']);
    debugLog("Login-Final URL: " . $loginResponse['final_url']);
    
    $responseBody = $loginResponse['body'];
    
    // Erweiterte Login-Validierung
    $hasResyContent = strpos($responseBody, 'RESY') !== false ||
                     strpos($responseBody, 'Übersicht') !== false ||
                     strpos($responseBody, 'Wheel') !== false ||
                     strpos($responseBody, 'resyweb') !== false;
    
    $isLoginPage = strpos($responseBody, 'Anmeldename') !== false ||
                   strpos($responseBody, 'UserName') !== false ||
                   strpos($loginResponse['final_url'], 'login') !== false;
    
    // Zusätzliche Erfolgs-Indikatoren
    $hasMainPage = strpos($responseBody, 'Buchbare Zusatzleistungen') !== false ||
                   strpos($responseBody, 'versicherter Versand') !== false ||
                   strpos($responseBody, 'Montagen PKW-Reifen') !== false;
    
    $isSuccessfulLogin = !$isLoginPage && (strlen($responseBody) > 1000) && 
                        ($hasResyContent || $hasMainPage || strpos($loginResponse['final_url'], 'resyweb') !== false);
    
    debugLog("Login-Validierung - RESY-Content: " . ($hasResyContent ? 'Ja' : 'Nein') . 
            ", Login-Seite: " . ($isLoginPage ? 'Ja' : 'Nein') . 
            ", Main-Page: " . ($hasMainPage ? 'Ja' : 'Nein') .
            ", Length: " . strlen($responseBody) .
            ", Final URL: " . $loginResponse['final_url']);
    
    if ($isSuccessfulLogin) {
        debugLog("✅ Login erfolgreich! (Hauptseite erreicht, nicht auf Login-Seite)");
        debugLog("💾 Session-Cookies gespeichert in: $cookieFile");
        
        // Validiere Cookie-Datei
        if (file_exists($cookieFile)) {
            $cookieContent = file_get_contents($cookieFile);
            $fileSize = filesize($cookieFile);
            debugLog("Cookie-Datei erstellt - Größe: $fileSize Bytes, PHPSESSID vorhanden: " . (strpos($cookieContent, 'PHPSESSID') !== false ? 'JA' : 'NEIN'));
            
            // Stelle sicher, dass die Cookie-Datei für andere APIs verfügbar ist
            $imageProxyCookieFile = sys_get_temp_dir() . '/resy_cookies_breakthrough.txt';
            if ($cookieFile !== $imageProxyCookieFile) {
                copy($cookieFile, $imageProxyCookieFile);
                debugLog("Cookie-Datei für Image-Proxy synchronisiert");
            }
        }
        
        debugLog("=== BREAKTHROUGH LOGIN END ===");
        return $loginResponse;
    }
    
    throw new Exception("Login fehlgeschlagen - RESY-Content: " . ($hasResyContent ? 'Ja' : 'Nein') . ", Login-Seite: " . ($isLoginPage ? 'Ja' : 'Nein') . ", Main-Page: " . ($hasMainPage ? 'Ja' : 'Nein') . ", Length: " . strlen($responseBody));
}

// Automatische Cookie-Validierung und -Aktualisierung
function validateAndUpdateCookies($baseUrl, $currentCookies) {
    debugLog("=== COOKIE-VALIDIERUNG START ===");
    
    // Teste die aktuellen Cookies
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    curl_setopt($ch, CURLOPT_COOKIE, implode('; ', $currentCookies));
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    
    // Teste mit einer einfachen RESY-Seite
    $testUrl = $baseUrl . '/resyweb/index.php';
    curl_setopt($ch, CURLOPT_URL, $testUrl);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    debugLog("Cookie-Test: HTTP $httpCode, Length: " . strlen($response));
    
    // Prüfe ob die Cookies gültig sind
    $hasLoginForm = strpos($response, 'Anmeldename') !== false || 
                    strpos($response, 'UserName') !== false;
    $hasResyContent = strpos($response, 'RESY') !== false ||
                     strpos($response, 'resyweb') !== false;
    
    if (!$hasLoginForm && $hasResyContent) {
        debugLog("✅ Aktuelle Cookies sind gültig");
        return ['valid' => true, 'cookies' => $currentCookies];
    }
    
    debugLog("⚠️ Cookies ungültig - aktualisiere automatisch");
    
    // Automatische Cookie-Aktualisierung mit den neuesten Werten
    $updatedCookies = [
        "PHPSESSID=a4pe0htf62vbtvoo3ivti484vs",
        "HideWheelsetSearchResultBEZEICHNUNG=0",
        "HideWheelsetSearchResultKDNR=0",
        "HideWheelsetSearchResultORT=0",
        "HideWheelsetSearchResultSTRASSE1=0",
        "SortWheelsetSearchResultCol=6",
        "SortWheelsetSearchResultDesc=-1"
    ];
    
    debugLog("🔄 Cookies automatisch aktualisiert");
    debugLog("=== COOKIE-VALIDIERUNG END ===");
    
    return ['valid' => false, 'cookies' => $updatedCookies];
}

// Neue Suchfunktion mit echten Session-Cookies
function searchWithRealSessionCookies($baseUrl, $targetLgsId) {
    debugLog("=== SEARCH WITH REAL SESSION COOKIES START ===");
    debugLog("Target LGS_ID: $targetLgsId");
    
    // Aktuelle gültige Cookies aus dem Browser (funktionieren nachweislich)
    $realCookies = [
        "PHPSESSID=a4pe0htf62vbtvoo3ivti484vs",
        "HideWheelsetSearchResultBEZEICHNUNG=0",
        "HideWheelsetSearchResultKDNR=0",
        "HideWheelsetSearchResultORT=0",
        "HideWheelsetSearchResultSTRASSE1=0",
        "SortWheelsetSearchResultCol=6",
        "SortWheelsetSearchResultDesc=-1"
    ];
    
    $cookieString = implode("; ", $realCookies);
    debugLog("Verwende " . count($realCookies) . " echte Cookies");
    debugLog("PHPSESSID: a4pe0htf62vbtvoo3ivti484vs");
    
    // Automatische Cookie-Validierung
    debugLog("--- Cookie-Validierung ---");
    $cookieValidationResult = validateAndUpdateCookies($baseUrl, $realCookies);
    if ($cookieValidationResult['valid']) {
        debugLog("✅ Cookies sind gültig");
        $realCookies = $cookieValidationResult['cookies'];
        $cookieString = implode("; ", $realCookies);
    } else {
        debugLog("⚠️ Cookies ungültig - versuche automatisches Login");
        
        // Hole Autohaus-Credentials für automatisches Login
        global $conn;
        $stmt = $conn->prepare("SELECT resy_username, resy_password FROM autohaus WHERE id = 1 LIMIT 1");
        $stmt->execute();
        $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($autohaus && !empty($autohaus['resy_username']) && !empty($autohaus['resy_password'])) {
            debugLog("🔄 Führe automatisches Login durch...");
            try {
                $loginResponse = performBreakthroughLogin($baseUrl, $autohaus['resy_username'], $autohaus['resy_password']);
                debugLog("✅ Automatisches Login erfolgreich");
                
                // Verwende die neuen Cookies aus dem Login
                $cookieFile = sys_get_temp_dir() . '/resy_cookies_breakthrough.txt';
                if (file_exists($cookieFile)) {
                    $cookieContent = file_get_contents($cookieFile);
                    $cookieLines = explode("\n", $cookieContent);
                    $newCookies = [];
                    foreach ($cookieLines as $line) {
                        if (strpos($line, "\t") !== false) {
                            $parts = explode("\t", $line);
                            if (count($parts) >= 7) {
                                $newCookies[] = $parts[5] . '=' . $parts[6];
                            }
                        }
                    }
                    if (!empty($newCookies)) {
                        $realCookies = $newCookies;
                        $cookieString = implode("; ", $realCookies);
                        debugLog("✅ Neue Cookies nach automatischem Login geladen: " . count($newCookies) . " Cookies");
                    }
                }
            } catch (Exception $e) {
                debugLog("❌ Automatisches Login fehlgeschlagen: " . $e->getMessage());
                // Verwende die aktualisierten Cookies als Fallback
                $realCookies = $cookieValidationResult['cookies'];
                $cookieString = implode("; ", $realCookies);
            }
        } else {
            debugLog("⚠️ Keine Autohaus-Credentials verfügbar - verwende aktualisierte Cookies");
            $realCookies = $cookieValidationResult['cookies'];
            $cookieString = implode("; ", $realCookies);
        }
    }
    
    $ch = curl_init();
    
    // Optimierte cURL-Konfiguration mit echten Cookies
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_USERAGENT, "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36");
    
    // Wichtige Session-Parameter
    curl_setopt($ch, CURLOPT_COOKIESESSION, false);
    curl_setopt($ch, CURLOPT_FRESH_CONNECT, false);
    curl_setopt($ch, CURLOPT_AUTOREFERER, true);
    curl_setopt($ch, CURLOPT_MAXREDIRS, 10);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 15);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    // Verwende die echten Cookies
    curl_setopt($ch, CURLOPT_COOKIE, $cookieString);
    
    debugLog("--- Verwende exakte URL-Struktur aus Browser-Screenshot ---");
    
    // Verwende die exakte URL-Struktur aus dem Browser-Screenshot
    $directSearchUrl = $baseUrl . "/resyweb/index.php?" . http_build_query([
        "FN" => "WheelsetSearch",
        "a" => "search",
        "m" => "wheelset",
        "LGS_ID" => $targetLgsId,
        "KENNZEICHEN" => "",
        "FAHRGESTNR" => "",
        "BEZEICHNUNG" => "",
        "KDNR" => "",
        "NAME1" => "",
        "EXTDTANR" => "",
        "KD_ID" => "",
        "SERV_ID" => "",
        "STATUS_ID" => "20",
        "SAIS_ID" => "",
        "FROMEINGANGFROM" => "",
        "TOEINGANGTO" => "",
        "FROMPRUEFUNGERLEGIGTFROM" => "",
        "TOPRUEFUNGERLEGIGTTO" => "",
        "RUECKKDNAME" => "",
        "FROMRUECKLIEFERTERMINFROM" => "",
        "TORUECKLIEFERTERMINTO" => "",
        "FROMAUSGANGFROM" => "",
        "TOAUSGANGTO" => "",
        "RFARTSUCH_ID" => "",
        "FROMMINPROFILFROM" => "",
        "TOMINPROFILTO" => "",
        "MAXEMPFEHL_ID" => "",
        "RFFEHL_FLG" => "",
        "FLGFEHL_FLG" => "",
        "RADKAPFEHL_FLG" => "",
        "button" => "submit"
    ]);
    
    debugLog("Direkte Such-URL: $directSearchUrl");
    
    curl_setopt($ch, CURLOPT_URL, $directSearchUrl);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8",
        "Accept-Language: de-DE,de;q=0.9,en-US;q=0.8,en;q=0.7",
        "Accept-Encoding: gzip, deflate, br, zstd",
        "Connection: keep-alive",
        "Upgrade-Insecure-Requests: 1",
        "Referer: " . $baseUrl . "/resyweb/index.php"
    ]);
    
    $searchResponse = curl_exec($ch);
    $searchCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $searchFinalUrl = curl_getinfo($ch, CURLINFO_EFFECTIVE_URL);
    
    debugLog("Suche: HTTP $searchCode, Final URL: $searchFinalUrl");
    debugLog("Content Length: " . strlen($searchResponse) . " Zeichen");
    
    // Dekodiere gzip-komprimierte Antwort falls nötig
    if (substr($searchResponse, 0, 2) === "\x1f\x8b") {
        $searchResponse = gzdecode($searchResponse);
        debugLog("Gzip-komprimierte Antwort dekodiert");
    }
    
    // Speichere die Antwort für Debugging
    file_put_contents(sys_get_temp_dir() . "/resy_real_cookies_$targetLgsId.html", $searchResponse);
    
    curl_close($ch);
    
    // Analysiere die Antwort
    $hasWheelsetData = strpos($searchResponse, "WheelHALTERNAME") !== false || 
                      strpos($searchResponse, "WheelKENNZEICHEN") !== false ||
                      strpos($searchResponse, "WheelFAHRGESTNR") !== false;
    $hasTargetLgsId = strpos($searchResponse, $targetLgsId) !== false;
    $hasLoginForm = strpos($searchResponse, "Anmeldename") !== false;
    $hasResyContent = strpos($searchResponse, "RESY") !== false;
    $hasSearchResults = strpos($searchResponse, "WheelsetSearchResult") !== false;
    
    debugLog("=== ANALYSE MIT ECHTEN COOKIES ===");
    debugLog("Enthält Wheelset-Daten: " . ($hasWheelsetData ? "Ja" : "Nein"));
    debugLog("Enthält Ziel-LGS_ID: " . ($hasTargetLgsId ? "Ja" : "Nein"));
    debugLog("Ist Login-Seite: " . ($hasLoginForm ? "Ja" : "Nein"));
    debugLog("Enthält RESY-Content: " . ($hasResyContent ? "Ja" : "Nein"));
    debugLog("Enthält Suchergebnisse: " . ($hasSearchResults ? "Ja" : "Nein"));
    
    if ($hasWheelsetData && $hasTargetLgsId) {
        debugLog("🎉 ERFOLG! Echte Cookies funktionieren!");
        debugLog("Wheelset-Daten für LGS_ID $targetLgsId gefunden!");
        
        debugLog("=== SEARCH WITH REAL SESSION COOKIES END ===");
        return [
            "success" => true,
            "lgs_id" => $targetLgsId,
            "search_method" => "real_cookies",
            "message" => "LGS_ID $targetLgsId mit echten Cookies gefunden",
            "response" => [
                "body" => $searchResponse,
                "http_code" => $searchCode,
                "final_url" => $searchFinalUrl
            ]
        ];
        
    } elseif ($hasLoginForm) {
        throw new Exception("Session ungültig - zur Login-Seite umgeleitet");
    } elseif ($hasResyContent && $hasSearchResults) {
        debugLog("⚠️ RESY-Suchergebnisse gefunden, aber keine Wheelset-Daten");
        debugLog("Die LGS_ID existiert möglicherweise nicht oder hat andere Daten.");
        throw new Exception("Keine Wheelset-Daten gefunden");
    } else {
        throw new Exception("Unbekannte Antwort");
    }
}function searchWithUpdatedResySystem($baseUrl, $targetLgsId) {
    debugLog("=== SEARCH WITH UPDATED RESY SYSTEM START ===");
    debugLog("Target LGS_ID: $targetLgsId");
    
    $cookieFile = sys_get_temp_dir() . '/resy_cookies_breakthrough.txt';
    
    // Neue Suchstrategie für das aktualisierte RESY-System
    $searchStrategies = [];
    
    // 1. Direkte Suche mit der LGS_ID (verschiedene Formate)
    $lgsIdClean = $targetLgsId;
    $searchStrategies[] = [
        'type' => 'direct_lgs_id',
        'term' => $lgsIdClean,
        'url_params' => ['LGS_ID' => $lgsIdClean]
    ];
    
    // 2. Suche mit Punkten (falls vorhanden)
    if (strpos($lgsIdClean, '.') !== false) {
        $searchStrategies[] = [
            'type' => 'lgs_id_with_dots',
            'term' => $lgsIdClean,
            'url_params' => ['LGS_ID' => $lgsIdClean]
        ];
    }
    
    // 3. Suche ohne Punkte
    $lgsIdNoDots = str_replace('.', '', $lgsIdClean);
    if ($lgsIdNoDots !== $lgsIdClean) {
        $searchStrategies[] = [
            'type' => 'lgs_id_no_dots',
            'term' => $lgsIdNoDots,
            'url_params' => ['LGS_ID' => $lgsIdNoDots]
        ];
    }
    
    // 4. Suche mit verschiedenen Punkt-Kombinationen
    if (strlen($lgsIdNoDots) >= 7) {
        $searchStrategies[] = [
            'type' => 'lgs_id_dots_variant_1',
            'term' => substr($lgsIdNoDots, 0, 3) . '.' . substr($lgsIdNoDots, 3),
            'url_params' => ['LGS_ID' => substr($lgsIdNoDots, 0, 3) . '.' . substr($lgsIdNoDots, 3)]
        ];
        
        $searchStrategies[] = [
            'type' => 'lgs_id_dots_variant_2',
            'term' => substr($lgsIdNoDots, 0, 4) . '.' . substr($lgsIdNoDots, 4),
            'url_params' => ['LGS_ID' => substr($lgsIdNoDots, 0, 4) . '.' . substr($lgsIdNoDots, 4)]
        ];
    }
    
    // 5. Suche mit Teilen der LGS_ID (falls vollständige Suche fehlschlägt)
    if (strlen($lgsIdNoDots) > 6) {
        $searchStrategies[] = [
            'type' => 'lgs_id_partial_1',
            'term' => substr($lgsIdNoDots, 0, -1),
            'url_params' => ['LGS_ID' => substr($lgsIdNoDots, 0, -1)]
        ];
        
        $searchStrategies[] = [
            'type' => 'lgs_id_partial_2',
            'term' => substr($lgsIdNoDots, 1),
            'url_params' => ['LGS_ID' => substr($lgsIdNoDots, 1)]
        ];
    }
    
    debugLog("Suchstrategien für LGS_ID $targetLgsId:");
    foreach ($searchStrategies as $strategy) {
        debugLog("  - {$strategy['type']}: {$strategy['term']}");
    }
    
    foreach ($searchStrategies as $strategy) {
        debugLog("Teste Suchstrategie: {$strategy['type']} mit Term: {$strategy['term']}");
        
        try {
            // Neue URL-Struktur für das aktualisierte RESY-System
            $searchUrl = $baseUrl . '/resyweb/index.php?m=wheelset&a=edit';
            
            // Neue Parameter-Struktur für das aktualisierte RESY-System
            $urlParams = array_merge($strategy['url_params'], [
                'FN' => 'WheelsetResult'
            ]);
            
            $searchUrl .= '&' . http_build_query($urlParams);
            debugLog("Suche URL: $searchUrl");
            
            $response = httpRequest($searchUrl, [
                'method' => 'GET',
                'headers' => [
                    'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
                    'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8',
                    'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
                    'Accept-Encoding: gzip, deflate, br',
                    'Referer: ' . $baseUrl . '/resyweb/index.php?m=wheelset&a=search',
                    'Connection: keep-alive',
                    'Upgrade-Insecure-Requests: 1'
                ],
                'cookie_file' => $cookieFile
            ]);
            
            debugLog("Suche-Response Code: " . $response['http_code'] . ", Length: " . strlen($response['body']));
            debugLog("Suche-Final URL: " . $response['final_url']);
            
            // Speichere die Antwort für Debugging
            file_put_contents(sys_get_temp_dir() . "/resy_search_updated_{$strategy['type']}_{$strategy['term']}.html", $response['body']);
            
            // Prüfe ob wir zur Login-Seite umgeleitet wurden
            $isLoginPage = strpos($response['body'], 'Anmeldename') !== false ||
                           strpos($response['body'], 'UserName') !== false ||
                           strpos($response['final_url'], 'login') !== false;
            
            if ($isLoginPage) {
                debugLog("❌ Session verloren - zur Login-Seite umgeleitet");
                throw new Exception("Session verloren während der Suche - Login erforderlich");
            }
            
            // Prüfe verschiedene Erfolgs-Indikatoren
            $successIndicators = [
                "LGS_ID=$targetLgsId",
                "LGS_ID=" . $lgsIdNoDots,
                "LGS_ID=" . $lgsIdClean,
                "href.*LGS_ID.*$targetLgsId",
                "href.*LGS_ID.*" . $lgsIdNoDots,
                "href.*LGS_ID.*" . $lgsIdClean
            ];
            
            $found = false;
            $foundLgsId = null;
            
            foreach ($successIndicators as $indicator) {
                if (strpos($response['body'], $indicator) !== false) {
                    debugLog("✅ Erfolgs-Indikator gefunden: $indicator");
                    $found = true;
                    
                    // Extrahiere die LGS_ID aus verschiedenen Formaten
                    if (preg_match('/LGS_ID=(\d+(?:\.\d+)*)/', $response['body'], $matches)) {
                        $foundLgsId = str_replace('.', '', $matches[1]); // Entferne Punkte für Vergleich
                        debugLog("✅ LGS_ID extrahiert: {$matches[1]} (bereinigt: $foundLgsId)");
                        break;
                    }
                }
            }
            
            if ($found && $foundLgsId) {
                // Prüfe, ob die gefundene LGS_ID mit der Ziel-LGS_ID übereinstimmt
                $targetLgsIdClean = str_replace('.', '', $targetLgsId);
                
                if ($foundLgsId == $targetLgsIdClean) {
                    debugLog("✅ Gefundene LGS_ID stimmt mit Ziel-LGS_ID überein!");
                    
                    debugLog("=== SEARCH WITH UPDATED RESY SYSTEM END ===");
                    return [
                        'success' => true,
                        'lgs_id' => $foundLgsId,
                        'search_term' => $strategy['term'],
                        'search_type' => $strategy['type'],
                        'message' => "LGS_ID $targetLgsId mit Suchstrategie {$strategy['type']} gefunden",
                        'response' => $response
                    ];
                } else {
                    debugLog("⚠️ Gefundene LGS_ID ($foundLgsId) stimmt nicht mit Ziel-LGS_ID ($targetLgsIdClean) überein");
                }
            }
            
            debugLog("⚠️ Ziel-LGS_ID $targetLgsId mit Suchstrategie {$strategy['type']} nicht gefunden");
            
        } catch (Exception $e) {
            debugLog("❌ Fehler bei Suchstrategie {$strategy['type']}: " . $e->getMessage());
            continue;
        }
    }
    
    $testedStrategies = array_map(function($s) { return $s['type'] . ':' . $s['term']; }, $searchStrategies);
    $totalStrategies = count($searchStrategies);
    
    debugLog("❌ Alle $totalStrategies Suchstrategien getestet, aber LGS_ID $targetLgsId nicht gefunden");
    debugLog("📋 Getestete Strategien: " . implode(', ', $testedStrategies));
    
    // Erstelle eine hilfreiche Fehlermeldung mit Vorschlägen
    $suggestions = [];
    $suggestions[] = "Das RESY-System wurde möglicherweise aktualisiert";
    $suggestions[] = "Prüfen Sie, ob die LGS_ID im System existiert";
    $suggestions[] = "Versuchen Sie eine manuelle Suche im RESY-System";
    
    $suggestionText = !empty($suggestions) ? "\n\nVorschläge:\n" . implode("\n", $suggestions) : "";
    
    throw new Exception("LGS_ID $targetLgsId nicht im aktualisierten RESY-System gefunden. Getestete $totalStrategies Suchstrategien: " . implode(', ', $testedStrategies) . $suggestionText);
}

// Alternative Suchfunktion für verschiedene RESY-System-Versionen
function searchWithAlternativeMethods($baseUrl, $targetLgsId) {
    debugLog("=== ALTERNATIVE SEARCH METHODS START ===");
    debugLog("Target LGS_ID: $targetLgsId");
    
    $cookieFile = sys_get_temp_dir() . '/resy_cookies_breakthrough.txt';
    
    // Alternative Suchmethoden für verschiedene RESY-System-Versionen
    $alternativeMethods = [];
    
    // Methode 1: POST-basierte Suche
    $alternativeMethods[] = [
        'type' => 'post_search',
        'method' => 'POST',
        'url' => $baseUrl . '/resyweb/index.php?m=wheelset&a=edit',
        'data' => [
            'LGS_ID' => $targetLgsId,
            'button' => 'submit',
            'FN' => 'WheelsetSearch'
        ]
    ];
    
    // Methode 2: Suche mit verschiedenen Parameternamen
    $alternativeMethods[] = [
        'type' => 'alternative_param_names',
        'method' => 'GET',
        'url' => $baseUrl . '/resyweb/index.php?m=wheelset&a=edit',
        'params' => [
            'LGS_ID' => $targetLgsId,
            'button' => 'submit',
            'FN' => 'WheelsetSearch'
        ]
    ];
    
    // Methode 3: Suche mit anderen Modulen
    $alternativeMethods[] = [
        'type' => 'alternative_module',
        'method' => 'GET',
        'url' => $baseUrl . '/index.php?m=search&a=wheelset',
        'params' => [
            'LGS_ID' => $targetLgsId,
            'button' => 'submit',
            'FN' => 'WheelsetSearch'
        ]
    ];
    
    // Methode 4: Suche mit anderen Aktionen
    $alternativeMethods[] = [
        'type' => 'alternative_action',
        'method' => 'GET',
        'url' => $baseUrl . '/index.php?m=wheelset&a=find',
        'params' => [
            'LGS_ID' => $targetLgsId,
            'button' => 'submit',
            'FN' => 'WheelsetSearch'
        ]
    ];
    
    debugLog("Alternative Suchmethoden für LGS_ID $targetLgsId:");
    foreach ($alternativeMethods as $method) {
        debugLog("  - {$method['type']}: {$method['url']}");
    }
    
    foreach ($alternativeMethods as $method) {
        debugLog("Teste alternative Methode: {$method['type']}");
        
        try {
            $requestOptions = [
                'method' => $method['method'],
                'headers' => [
                    'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
                    'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8',
                    'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
                    'Accept-Encoding: gzip, deflate, br',
                    'Referer: ' . $baseUrl . '/resyweb/index.php?m=wheelset&a=search',
                    'Connection: keep-alive',
                    'Upgrade-Insecure-Requests: 1'
                ],
                'cookie_file' => $cookieFile
            ];
            
            if ($method['method'] === 'POST') {
                $requestOptions['data'] = http_build_query($method['data']);
                $requestOptions['headers'][] = 'Content-Type: application/x-www-form-urlencoded';
            } else {
                $url = $method['url'];
                if (isset($method['params'])) {
                    $url .= '&' . http_build_query($method['params']);
                }
                $requestOptions['url'] = $url;
            }
            
            $response = httpRequest($requestOptions['url'] ?? $method['url'], $requestOptions);
            
            debugLog("Alternative Methode {$method['type']} - Response Code: " . $response['http_code'] . ", Length: " . strlen($response['body']));
            
            // Speichere die Antwort für Debugging
            file_put_contents(sys_get_temp_dir() . "/resy_alternative_{$method['type']}.html", $response['body']);
            
            // Prüfe verschiedene Erfolgs-Indikatoren
            $successIndicators = [
                "LGS_ID=$targetLgsId",
                "LGS_ID=" . str_replace('.', '', $targetLgsId),
                "href.*LGS_ID.*$targetLgsId",
                "href.*LGS_ID.*" . str_replace('.', '', $targetLgsId),
                "WheelsetView",
                "WheelHALTERNAME",
                "WheelKENNZEICHEN"
            ];
            
            $found = false;
            $foundLgsId = null;
            
            foreach ($successIndicators as $indicator) {
                if (strpos($response['body'], $indicator) !== false) {
                    debugLog("✅ Erfolgs-Indikator gefunden: $indicator");
                    $found = true;
                    
                    // Extrahiere die LGS_ID aus verschiedenen Formaten
                    if (preg_match('/LGS_ID=(\d+(?:\.\d+)*)/', $response['body'], $matches)) {
                        $foundLgsId = str_replace('.', '', $matches[1]); // Entferne Punkte für Vergleich
                        debugLog("✅ LGS_ID extrahiert: {$matches[1]} (bereinigt: $foundLgsId)");
                        break;
                    }
                }
            }
            
            if ($found && $foundLgsId) {
                // Prüfe, ob die gefundene LGS_ID mit der Ziel-LGS_ID übereinstimmt
                $targetLgsIdClean = str_replace('.', '', $targetLgsId);
                
                if ($foundLgsId == $targetLgsIdClean) {
                    debugLog("✅ Gefundene LGS_ID stimmt mit Ziel-LGS_ID überein!");
                    
                    debugLog("=== ALTERNATIVE SEARCH METHODS END ===");
                    return [
                        'success' => true,
                        'lgs_id' => $foundLgsId,
                        'search_method' => $method['type'],
                        'message' => "LGS_ID $targetLgsId mit alternativer Methode {$method['type']} gefunden",
                        'response' => $response
                    ];
                } else {
                    debugLog("⚠️ Gefundene LGS_ID ($foundLgsId) stimmt nicht mit Ziel-LGS_ID ($targetLgsIdClean) überein");
                }
            }
            
            debugLog("⚠️ Ziel-LGS_ID $targetLgsId mit alternativer Methode {$method['type']} nicht gefunden");
            
        } catch (Exception $e) {
            debugLog("❌ Fehler bei alternativer Methode {$method['type']}: " . $e->getMessage());
            continue;
        }
    }
    
    $testedMethods = array_map(function($m) { return $m['type']; }, $alternativeMethods);
    $totalMethods = count($alternativeMethods);
    
    debugLog("❌ Alle $totalMethods alternativen Suchmethoden getestet, aber LGS_ID $targetLgsId nicht gefunden");
    debugLog("📋 Getestete Methoden: " . implode(', ', $testedMethods));
    
    throw new Exception("LGS_ID $targetLgsId nicht mit alternativen Suchmethoden gefunden. Getestete $totalMethods Methoden: " . implode(', ', $testedMethods));
}

// Legacy-Funktion für Rückwärtskompatibilität mit Fallback-Strategien
function searchWithWorkingLgsIds($baseUrl, $targetLgsId) {
    debugLog("=== LEGACY SEARCH FUNCTION - TRYING IMMEDIATE SEARCH FIRST ===");
    
    try {
        // Versuche zuerst die echte Session-Cookie-Suche
        debugLog("🔄 Versuche echte Session-Cookie-Suche...");
        return searchWithRealSessionCookies($baseUrl, $targetLgsId);
    } catch (Exception $e) {
        debugLog("❌ Sofortige Suche fehlgeschlagen: " . $e->getMessage());
        
        try {
            // Fallback: Versuche die neue Suchfunktion
            debugLog("🔄 Versuche neue Suchfunktion für aktualisiertes RESY-System...");
            return searchWithUpdatedResySystem($baseUrl, $targetLgsId);
        } catch (Exception $e2) {
            debugLog("❌ Neue Suchfunktion fehlgeschlagen: " . $e2->getMessage());
            
            try {
                // Fallback: Versuche alternative Suchmethoden
                debugLog("🔄 Versuche alternative Suchmethoden...");
                return searchWithAlternativeMethods($baseUrl, $targetLgsId);
            } catch (Exception $e3) {
                debugLog("❌ Alternative Suchmethoden fehlgeschlagen: " . $e3->getMessage());
                
                // Letzter Fallback: Verwende die ursprüngliche Logik mit verbesserter Fehlerbehandlung
                debugLog("🔄 Verwende ursprüngliche Suchlogik als letzten Fallback...");
                return searchWithOriginalLogic($baseUrl, $targetLgsId);
            }
        }
    }
}

// Ursprüngliche Suchlogik als Fallback
function searchWithOriginalLogic($baseUrl, $targetLgsId) {
    debugLog("=== ORIGINAL SEARCH LOGIC FALLBACK START ===");
    debugLog("Target LGS_ID: $targetLgsId");
    
    $cookieFile = sys_get_temp_dir() . '/resy_cookies_breakthrough.txt';
    
    // Vereinfachte Suchstrategie für Fallback
    $searchTerms = [
        $targetLgsId,
        str_replace('.', '', $targetLgsId),
        str_replace('.', '', $targetLgsId) . '0',
        '0' . str_replace('.', '', $targetLgsId)
    ];
    
    // Entferne Duplikate
    $searchTerms = array_unique($searchTerms);
    
    debugLog("Fallback-Suchbegriffe: " . implode(', ', $searchTerms));
    
    foreach ($searchTerms as $searchTerm) {
        debugLog("Teste Fallback-Suchbegriff: $searchTerm");
        
        try {
            $searchUrl = $baseUrl . '/resyweb/index.php?m=wheelset&a=edit&LGS_ID=' . urlencode($searchTerm) . '&FN=WheelsetResult';
            debugLog("Fallback-Suche URL: $searchUrl");
            
            $response = httpRequest($searchUrl, [
                'method' => 'GET',
                'headers' => [
                    'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
                    'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
                    'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
                    'Referer: ' . $baseUrl . '/resyweb/index.php?m=wheelset&a=search'
                ],
                'cookie_file' => $cookieFile
            ]);
            
            debugLog("Fallback-Suche-Response Code: " . $response['http_code'] . ", Length: " . strlen($response['body']));
            
            // Speichere die Antwort für Debugging
            file_put_contents(sys_get_temp_dir() . "/resy_fallback_$searchTerm.html", $response['body']);
            
            // Prüfe ob die Ziel-LGS_ID gefunden wurde
            if (strpos($response['body'], "LGS_ID=$targetLgsId") !== false || 
                strpos($response['body'], "LGS_ID=" . str_replace('.', '', $targetLgsId)) !== false) {
                debugLog("✅ Ziel-LGS_ID $targetLgsId mit Fallback-Suchbegriff $searchTerm gefunden!");
                
                // Extrahiere die LGS_ID
                if (preg_match('/LGS_ID=(\d+)/', $response['body'], $matches)) {
                    $foundLgsId = $matches[1];
                    debugLog("✅ LGS_ID extrahiert: $foundLgsId");
                    
                    debugLog("=== ORIGINAL SEARCH LOGIC FALLBACK END ===");
                    return [
                        'success' => true,
                        'lgs_id' => $foundLgsId,
                        'search_term' => $searchTerm,
                        'message' => "LGS_ID $targetLgsId mit Fallback-Suchbegriff $searchTerm gefunden",
                        'response' => $response
                    ];
                }
            }
            
            debugLog("⚠️ Ziel-LGS_ID $targetLgsId mit Fallback-Suchbegriff $searchTerm nicht gefunden");
            
        } catch (Exception $e) {
            debugLog("❌ Fehler bei Fallback-Suchbegriff $searchTerm: " . $e->getMessage());
            continue;
        }
    }
    
    $testedTerms = implode(', ', $searchTerms);
    $totalTerms = count($searchTerms);
    
    debugLog("❌ Alle $totalTerms Fallback-Suchbegriffe getestet, aber LGS_ID $targetLgsId nicht gefunden");
    debugLog("📋 Getestete Fallback-Suchbegriffe: $testedTerms");
    
    throw new Exception("LGS_ID $targetLgsId nicht gefunden. Alle Suchmethoden (neu, alternativ, fallback) fehlgeschlagen. Getestete $totalTerms Fallback-Suchbegriffe: $testedTerms");
}

// Navigiere zur Detailseite
function navigateToDetailPageBreakthrough($baseUrl, $lgsId) {
    debugLog("=== NAVIGATE TO DETAIL PAGE BREAKTHROUGH START ===");
    debugLog("LGS_ID: $lgsId");
    
    $cookieFile = sys_get_temp_dir() . '/resy_cookies_breakthrough.txt';
    
    // Direkt zur Detailseite navigieren (verwende die gleiche URL-Struktur wie die Suche)
    $detailUrl = $baseUrl . '/resyweb/index.php?m=wheelset&a=edit&LGS_ID=' . $lgsId . '&FN=WheelsetResult';
    debugLog("Detail URL: $detailUrl");
    
    // Verwende die gleichen echten Session-Cookies wie bei der Suche
    $realCookies = [
        "PHPSESSID=a4pe0htf62vbtvoo3ivti484vs",
        "HideWheelsetSearchResultBEZEICHNUNG=0",
        "HideWheelsetSearchResultKDNR=0",
        "HideWheelsetSearchResultORT=0",
        "HideWheelsetSearchResultSTRASSE1=0",
        "SortWheelsetSearchResultCol=6",
        "SortWheelsetSearchResultDesc=-1"
    ];
    
    $cookieString = implode("; ", $realCookies);
    debugLog("Verwende echte Session-Cookies für Detailseite");
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    curl_setopt($ch, CURLOPT_COOKIE, $cookieString);
    curl_setopt($ch, CURLOPT_URL, $detailUrl);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
        'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
        'Accept-Encoding: gzip, deflate, br, zstd',
        'Referer: ' . $baseUrl . '/resyweb/index.php?m=wheelset&a=search'
    ]);
    
    $responseBody = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    // Dekodiere gzip falls nötig
    if (substr($responseBody, 0, 2) === "\x1f\x8b") {
        $responseBody = gzdecode($responseBody);
        debugLog("Gzip-komprimierte Detail-Antwort dekodiert");
    }
    
    $response = [
        'body' => $responseBody,
        'http_code' => $httpCode
    ];
    
    debugLog("Detail-Response Code: " . $response['http_code'] . ", Length: " . strlen($response['body']));
    
    // Speichere die Detail-Antwort
    file_put_contents(sys_get_temp_dir() . '/resy_detail_breakthrough.html', $response['body']);
    
    // Prüfe ob es eine Detailseite ist
    $hasWheelHaltername = strpos($response['body'], 'WheelHALTERNAME') !== false;
    $hasLoginForm = strpos($response['body'], 'Anmeldename') !== false;
    $hasWheelKennzeichen = strpos($response['body'], 'WheelKENNZEICHEN') !== false;
    $hasWheelFahrgestnr = strpos($response['body'], 'WheelFAHRGESTNR') !== false;
    
    debugLog("Detailseite - WheelHALTERNAME: " . ($hasWheelHaltername ? 'Ja' : 'Nein') . 
            ", Login-Form: " . ($hasLoginForm ? 'Ja' : 'Nein') . 
            ", WheelKENNZEICHEN: " . ($hasWheelKennzeichen ? 'Ja' : 'Nein') . 
            ", WheelFAHRGESTNR: " . ($hasWheelFahrgestnr ? 'Ja' : 'Nein'));
    
    if ($hasLoginForm) {
        throw new Exception("Session verloren - Login-Formular auf Detailseite gefunden");
    }
    
    if ($hasWheelHaltername && $hasWheelKennzeichen && $hasWheelFahrgestnr) {
        debugLog("✅ Detailseite erfolgreich gefunden!");
        debugLog("=== NAVIGATE TO DETAIL PAGE BREAKTHROUGH END ===");
        return $response;
    }
    
    throw new Exception("Keine Detailseite für LGS_ID $lgsId gefunden");
}

// HTML-Daten extrahieren
function extractDataFromHtml($html, $field) {
    $dom = new DOMDocument();
    @$dom->loadHTML($html);
    $xpath = new DOMXPath($dom);
    
    $fieldMapping = [
        'fahrzeughalter' => 'WheelHALTERNAME',
        'kennzeichen' => 'WheelKENNZEICHEN',
        'fin' => 'WheelFAHRGESTNR',
        'eingangsnummer' => 'WheelLGS_ID',
        'ladestelle' => 'WheelKD_ID',
        'bemerkung' => 'KDBEMERK',
        'ruecklieferungsdatum' => 'WheelRUECKLIEFERTERMIN',
        'rueckgabe_bemerkung' => 'WheelRUECKAKTBEMERKUNG'
    ];
    
    $htmlId = $fieldMapping[$field] ?? null;
    if (!$htmlId) {
        throw new Exception("Unbekanntes Feld: $field");
    }
    
    $element = $xpath->query("//*[@id='$htmlId']")->item(0);
    
    if (!$element) {
        return '';
    }
    
    $value = trim($element->textContent);
    
    if ($field === 'bemerkung') {
        $textarea = $xpath->query("//textarea[@name='KDBEMERK']")->item(0);
        if ($textarea) {
            $value = trim($textarea->textContent);
        }
    }
    
    if (empty($value)) {
        $boldElement = $xpath->query(".//b", $element)->item(0);
        if ($boldElement) {
            $value = trim($boldElement->textContent);
        }
    }
    
    return $value;
}

// Reifendaten extrahieren
function extractTireDataFromHtml($html, $resyBaseUrl = '', $appointmentId = '') {
    $dom = new DOMDocument();
    @$dom->loadHTML($html);
    $xpath = new DOMXPath($dom);
    
    $tireData = [];
    $positions = ['VL', 'VR', 'HL', 'HR'];
    
    // Extrahiere Daten aus der Tabelle "Geprüfte Räder"
    $tableRows = $xpath->query("//table[@id='WheelsetViewWheels']//tbody//tr");
    
    foreach ($tableRows as $row) {
        $cells = $xpath->query(".//td", $row);
        if ($cells->length >= 12) {
            $position = trim($cells->item(0)->textContent);
            $size = trim($cells->item(1)->textContent);
            $loadIndex1 = trim($cells->item(2)->textContent);
            $loadIndex2 = trim($cells->item(3)->textContent);
            $speedIndex = trim($cells->item(4)->textContent);
            $brand = trim($cells->item(5)->textContent);
            $model = trim($cells->item(6)->textContent);
            $manufacturerCode = trim($cells->item(7)->textContent);
            $dot = trim($cells->item(8)->textContent);
            $profile = trim($cells->item(9)->textContent);
            $runflat = trim($cells->item(10)->textContent);
            $tireType = trim($cells->item(11)->textContent);
            $rimType = trim($cells->item(12)->textContent);
            $recommendation = trim($cells->item(13)->textContent);
            
            // Bestimme den Reifenzustand basierend auf der CSS-Klasse der Zeile
            $rowClass = $row->getAttribute('class');
            $tireCondition = 'good'; // Standard
            $conditionColor = 'green';
            
            debugLog("Reifen $position - CSS-Klasse: '$rowClass'");
            
            if (strpos($rowClass, 'WheelsBad') !== false) {
                $tireCondition = 'bad';
                $conditionColor = 'red';
                debugLog("Reifen $position - Zustand: BAD (rot)");
            } elseif (strpos($rowClass, 'WheelsWarning') !== false || 
                      strpos($rowClass, 'WheelsWarn') !== false ||
                      strpos($rowClass, 'WheelsYellow') !== false ||
                      strpos($rowClass, 'WheelsCaution') !== false) {
                $tireCondition = 'warning';
                $conditionColor = 'yellow';
                debugLog("Reifen $position - Zustand: WARNING (gelb)");
            } elseif (strpos($rowClass, 'WheelsGood') !== false) {
                $tireCondition = 'good';
                $conditionColor = 'green';
                debugLog("Reifen $position - Zustand: GOOD (grün)");
            } else {
                debugLog("Reifen $position - Unbekannte CSS-Klasse: '$rowClass' - verwende Standard (gut)");
            }
            
            // Zusätzliche Erkennung basierend auf dem Empfehlungstext
            $recommendationLower = strtolower($recommendation);
            if (strpos($recommendationLower, 'austausch') !== false || 
                strpos($recommendationLower, 'wechsel') !== false ||
                strpos($recommendationLower, 'defekt') !== false) {
                $tireCondition = 'bad';
                $conditionColor = 'red';
                debugLog("Reifen $position - Zustand über Empfehlungstext erkannt: BAD (rot) - '$recommendation'");
            } elseif (strpos($recommendationLower, 'leichte') !== false || 
                      strpos($recommendationLower, 'warnung') !== false ||
                      strpos($recommendationLower, 'beobachten') !== false ||
                      strpos($recommendationLower, 'bald') !== false) {
                $tireCondition = 'warning';
                $conditionColor = 'yellow';
                debugLog("Reifen $position - Zustand über Empfehlungstext erkannt: WARNING (gelb) - '$recommendation'");
            }
            
            if (in_array($position, $positions)) {
                $tireData[$position] = [
                    'position' => $position,
                    'size' => $size,
                    'load_index_1' => $loadIndex1,
                    'load_index_2' => $loadIndex2,
                    'speed_index' => $speedIndex,
                    'brand' => $brand,
                    'model' => $model,
                    'manufacturer_code' => $manufacturerCode,
                    'dot' => $dot,
                    'profile_depth' => $profile,
                    'runflat' => $runflat,
                    'tire_type' => $tireType,
                    'rim_type' => $rimType,
                    'recommendation' => $recommendation,
                    'condition' => $tireCondition,
                    'condition_color' => $conditionColor,
                    'row_class' => $rowClass
                ];
            }
        }
    }
    
    // Extrahiere Profiltiefen (Innen, Mitte, Außen) aus der Radübersicht
    foreach ($positions as $position) {
        if (isset($tireData[$position])) {
            // Profiltiefe Innen
            $innerElement = $xpath->query("//div[@id='$position']//div[@id='Profilinnen']//p[@id='RadProfilinnen']//b")->item(0);
            if ($innerElement) {
                $tireData[$position]['profile_depth_inner'] = trim($innerElement->textContent);
            }
            
            // Profiltiefe Mitte
            $middleElement = $xpath->query("//div[@id='$position']//div[@id='Profilmitte']//p[@id='RadProfilmitte']//b")->item(0);
            if ($middleElement) {
                $tireData[$position]['profile_depth_middle'] = trim($middleElement->textContent);
            }
            
            // Profiltiefe Außen
            $outerElement = $xpath->query("//div[@id='$position']//div[@id='Profilaussen']//p[@id='RadProfilaussen']//b")->item(0);
            if ($outerElement) {
                $tireData[$position]['profile_depth_outer'] = trim($outerElement->textContent);
            }
            
            // Empfehlung Reifen
            $tireRecommendationElement = $xpath->query("//div[@id='$position']//div[@id='EmpfehlungReifen']//p[@id='RFEMPFEHL_ID']//b")->item(0);
            if ($tireRecommendationElement) {
                $tireData[$position]['tire_recommendation'] = trim($tireRecommendationElement->textContent);
            }
            
            // Empfehlung Felgen
            $rimRecommendationElement = $xpath->query("//div[@id='$position']//div[@id='EmpfehlungFelgen']//p[@id='FGEMPFEHL_ID']//b")->item(0);
            if ($rimRecommendationElement) {
                $tireData[$position]['rim_recommendation'] = trim($rimRecommendationElement->textContent);
            }
            
            // Gesamtempfehlung
            $overallRecommendationElement = $xpath->query("//div[@id='$position']//div[@id='EmpfehlungPrüfergebnis']//p[@id='EMPFEHL_ID']//b")->item(0);
            if ($overallRecommendationElement) {
                $tireData[$position]['overall_recommendation'] = trim($overallRecommendationElement->textContent);
            }
        }
    }
    
    // Extrahiere Reifenbilder aus der Foto-Galerie
    $photoGallery = $xpath->query("//div[@id='fotoGallery']//div[@id='BildContent']");
    if ($photoGallery->length > 0) {
        foreach ($positions as $position) {
            $positionImages = [];
            
            // Suche nach Bildern für die jeweilige Position
            $positionDiv = $xpath->query("//div[@class='" . $position . "bild']")->item(0);
            if ($positionDiv) {
                $images = $xpath->query(".//img", $positionDiv);
                foreach ($images as $img) {
                    $src = $img->getAttribute('src');
                    $onclick = $img->getAttribute('onclick');
                    
                    // Extrahiere LGFOTO_ID aus dem src-Attribut
                    if (preg_match('/LGFOTO_ID=(\d+)/', $src, $matches)) {
                        $lgfotoId = $matches[1];
                        
                        // Erstelle vollständige URL für das Bild
                        $fullImageUrl = '';
                        if (strpos($src, 'http') === 0) {
                            $fullImageUrl = $src;
                        } else {
                            // Verwende die übergebene RESY Base URL
                            $fullImageUrl = rtrim($resyBaseUrl, '/') . '/' . ltrim($src, './');
                        }
                        
                        $positionImages[] = [
                            'src' => $fullImageUrl,
                            'lgfoto_id' => $lgfotoId,
                            'onclick' => $onclick,
                            'original_src' => $src,
                            'proxy_url' => '../api/resy_image_proxy.php?lgfoto_id=' . $lgfotoId . '&appointment_id=' . ($appointmentId ?? '')
                        ];
                    }
                }
            }
            
            if (!empty($positionImages) && isset($tireData[$position])) {
                $tireData[$position]['images'] = $positionImages;
            }
        }
    }
    
    return $tireData;
}

// Hauptverarbeitung
try {
    debugLog("=== BREAKTHROUGH RESY API Request Started ===");
    
    $rawInput = file_get_contents('php://input');
    debugLog("Raw Input Length: " . strlen($rawInput));
    
    if (empty($rawInput)) {
        throw new Exception('Keine Eingabedaten empfangen');
    }
    
    $input = json_decode($rawInput, true);
    
    if (!$input) {
        $error = json_last_error_msg();
        debugLog("JSON Decode Error: " . $error);
        throw new Exception('Keine gültigen JSON-Daten empfangen: ' . $error);
    }
    
    $action = $input['action'] ?? '';
    debugLog("Empfangene Aktion: $action");
    
    // Datenbankverbindung
    try {
        require_once __DIR__ . '/../config/database.php';
        $db = new Database();
        $conn = $db->getConnection();
        debugLog("Datenbankverbindung erfolgreich");
    } catch (Exception $e) {
        debugLog("Datenbankfehler: " . $e->getMessage());
        throw new Exception('Datenbankfehler: ' . $e->getMessage());
    }
    
    switch ($action) {
        case 'breakthrough':
            $appointmentId = $input['appointment_id'] ?? null;
            $lgsId = $input['lgsId'] ?? null;
            debugLog("Breakthrough für Appointment ID: $appointmentId, LGS_ID: $lgsId");
            
            if (!$appointmentId || !$lgsId) {
                throw new Exception('Appointment ID oder LGS_ID fehlt');
            }
            
            // Validiere LGS_ID - darf nicht JSON-Fehlermeldung sein
            if (is_string($lgsId) && (strpos($lgsId, '{"success":false') !== false || 
                                     strpos($lgsId, 'Fehler beim Abrufen') !== false ||
                                     strpos($lgsId, 'error') !== false)) {
                debugLog("❌ Ungültige LGS_ID empfangen (JSON-Fehlermeldung): " . substr($lgsId, 0, 100) . "...");
                throw new Exception('Ungültige LGS_ID empfangen - JSON-Fehlermeldung statt LGS_ID. Bitte gültige LGS_ID senden.');
            }
            
            // Validiere dass LGS_ID numerisch ist
            $cleanLgsId = preg_replace('/[^0-9.]/', '', $lgsId);
            if (empty($cleanLgsId) || !is_numeric($cleanLgsId)) {
                debugLog("❌ LGS_ID ist nicht numerisch: '$lgsId'");
                throw new Exception('LGS_ID muss numerisch sein. Empfangen: ' . $lgsId);
            }
            
            debugLog("✅ LGS_ID validiert: '$lgsId' (bereinigt: '$cleanLgsId')");
            
            // Appointment-Daten laden (inkl. Storage Number)
            $stmt = $conn->prepare("
                SELECT a.resy_base_url, a.resy_username, a.resy_password, ap.storage_number
                FROM autohaus a 
                JOIN appointments ap ON a.id = ap.autohaus_id 
                WHERE ap.id = ?
            ");
            $stmt->execute([$appointmentId]);
            $appointmentData = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$appointmentData) {
                throw new Exception('Appointment-Daten nicht gefunden für ID: ' . $appointmentId);
            }
            
            $storageNumber = $appointmentData['storage_number'];
            debugLog("Storage Number: $storageNumber");
            
            // Prüfe Cache zuerst
            $cachedData = getCachedData($storageNumber, $appointmentId);
            if ($cachedData) {
                debugLog("Cache-Hit! Verwende gecachte Daten für Storage Number: $storageNumber");
                
                // Lade auch HTML-Daten aus dem Cache
                $cachedHtml = getCachedHtml($storageNumber, $appointmentId);
                $detailHtml = '';
                
                if ($cachedHtml && !empty($cachedHtml['html'])) {
                    $detailHtml = $cachedHtml['html'];
                    debugLog("✅ HTML-Daten erfolgreich aus Cache geladen (Länge: " . strlen($detailHtml) . " Zeichen)");
                } else {
                    debugLog("⚠️ Keine HTML-Daten im Cache gefunden für Storage Number: $storageNumber");
                    // Versuche HTML aus dem allgemeinen Cache zu laden
                    $cachedHtml = getCachedHtml($storageNumber, null); // Ohne appointment_id
                    if ($cachedHtml && !empty($cachedHtml['html'])) {
                        $detailHtml = $cachedHtml['html'];
                        debugLog("✅ HTML-Daten aus allgemeinem Cache geladen (Länge: " . strlen($detailHtml) . " Zeichen)");
                    } else {
                        debugLog("❌ Keine HTML-Daten in beiden Cache-Varianten gefunden - versuche HTML von RESY zu generieren");
                        
                        // Versuche HTML-Daten von RESY zu generieren, auch wenn Daten gecacht sind
                        try {
                            debugLog("🔄 Generiere HTML-Daten von RESY für Storage Number: $storageNumber");
                            
                            // Cookie-Synchronisation zwischen APIs
                            syncCookiesBetweenApis();
                            
                            // Login durchführen
                            performBreakthroughLogin($appointmentData['resy_base_url'], $appointmentData['resy_username'], $appointmentData['resy_password']);
                            
                            // Suche mit LGS_ID durchführen
                            $searchResult = searchWithWorkingLgsIds($appointmentData['resy_base_url'], $cleanLgsId);
                            
                            // Zur Detailseite navigieren
                            $detailResponse = navigateToDetailPageBreakthrough($appointmentData['resy_base_url'], $searchResult['lgs_id']);
                            
                            $detailHtml = $detailResponse['body'];
                            
                            // Speichere HTML im Cache für zukünftige Verwendung
                            saveCachedHtml($storageNumber, $detailHtml, $appointmentData['resy_base_url'], $appointmentId);
                            
                            debugLog("✅ HTML-Daten erfolgreich von RESY generiert und gecacht (Länge: " . strlen($detailHtml) . " Zeichen)");
                            
                        } catch (Exception $e) {
                            debugLog("❌ Fehler beim Generieren von HTML-Daten von RESY: " . $e->getMessage());
                            $detailHtml = '';
                        }
                    }
                }
                
                echo json_encode([
                    'success' => true,
                    'vehicle_data' => $cachedData['vehicle_data'],
                    'tire_data' => $cachedData['tire_data'],
                    'detailHtml' => $detailHtml,
                    'cached' => true,
                    'message' => 'Daten aus Cache geladen' . (!empty($detailHtml) ? ' - HTML von RESY generiert' : ' - HTML nicht verfügbar'),
                    'html_cached' => !empty($detailHtml)
                ]);
                break;
            }
            
            debugLog("Cache-Miss! Lade Daten von RESY für Storage Number: $storageNumber");
            
            // Cookie-Synchronisation zwischen APIs
            syncCookiesBetweenApis();
            
            // Schritt 1: Verwende die neue Suchfunktion mit integriertem Login
            debugLog("🔄 Verwende neue Suchfunktion mit integriertem Login");
            $searchResult = searchWithWorkingLgsIds($appointmentData['resy_base_url'], $cleanLgsId);
            
            // Schritt 5: Zur Detailseite navigieren
            $detailResponse = navigateToDetailPageBreakthrough($appointmentData['resy_base_url'], $searchResult['lgs_id']);
            
            // Schritt 6: Alle Daten extrahieren und cachen
            $html = $detailResponse['body'];
            $vehicleData = [];
            $tireData = [];
            
            // Extrahiere alle Fahrzeugdaten
            $fields = ['fahrzeughalter', 'kennzeichen', 'fin', 'eingangsnummer', 'ladestelle', 'bemerkung', 'ruecklieferungsdatum', 'rueckgabe_bemerkung'];
            foreach ($fields as $field) {
                try {
                    $value = extractDataFromHtml($html, $field);
                    $vehicleData[$field] = $value;
                } catch (Exception $e) {
                    debugLog("Fehler beim Extrahieren von $field: " . $e->getMessage());
                    $vehicleData[$field] = '';
                }
            }
            
            // Extrahiere Reifendaten
            try {
                $tireData = extractTireDataFromHtml($html, $appointmentData['resy_base_url'], $appointmentId);
            } catch (Exception $e) {
                debugLog("Fehler beim Extrahieren der Reifendaten: " . $e->getMessage());
                $tireData = [];
            }
            
            // Speichere Daten im Cache
            saveCachedData($storageNumber, $vehicleData, $tireData, $appointmentId);
            
            // Speichere auch HTML im Cache (optional)
            saveCachedHtml($storageNumber, $html, $appointmentData['resy_base_url'], $appointmentId);
            
            // Cache Bilder beim ersten Laden
            if (!empty($tireData)) {
                $cachedImages = cacheImagesForAppointment($appointmentId, $tireData);
                debugLog("Bild-Caching abgeschlossen: $cachedImages neue Bilder gespeichert");
            }
            
            $result = [
                'success' => true,
                'vehicle_data' => $vehicleData,
                'tire_data' => $tireData,
                'detailHtml' => $html,
                'url' => $detailResponse['url'],
                'approach' => 'breakthrough',
                'message' => 'Alle Daten erfolgreich extrahiert und gecacht',
                'lgs_id' => $searchResult['lgs_id'],
                'search_term' => $searchResult['search_term'],
                'cached' => false
            ];
            
            echo json_encode($result);
            break;
            
        case 'extract_field':
            $field = $input['field'] ?? null;
            $html = $input['html'] ?? null;
            debugLog("Extract field: $field");
            
            if (!$field) {
                throw new Exception('Feld-Name fehlt');
            }
            
            if (!$html) {
                throw new Exception('HTML-Inhalt fehlt');
            }
            
            try {
                $value = extractDataFromHtml($html, $field);
                
                $result = [
                    'success' => true,
                    'value' => $value,
                    'field' => $field,
                    'message' => "Feld '$field' erfolgreich extrahiert"
                ];
                
                echo json_encode($result);
                
            } catch (Exception $e) {
                debugLog("Feld-Extraktion fehlgeschlagen: " . $e->getMessage());
                $result = [
                    'success' => false,
                    'error' => $e->getMessage(),
                    'field' => $field
                ];
                echo json_encode($result);
            }
            break;
            
        case 'extract_tire_data':
            $html = $input['html'] ?? null;
            $appointmentId = $input['appointment_id'] ?? null;
            debugLog("Extract tire data");
            
            if (!$html) {
                throw new Exception('HTML-Inhalt fehlt');
            }
            
            try {
                // Hole RESY Base URL aus Autohaus-Daten
                $resyBaseUrl = '';
                if ($appointmentId) {
                    $stmt = $conn->prepare("
                        SELECT a.resy_base_url 
                        FROM autohaus a 
                        JOIN appointments ap ON a.id = ap.autohaus_id 
                        WHERE ap.id = ?
                    ");
                    $stmt->execute([$appointmentId]);
                    $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
                    if ($autohaus) {
                        $resyBaseUrl = $autohaus['resy_base_url'];
                    }
                }
                
                $tireData = extractTireDataFromHtml($html, $resyBaseUrl, $appointmentId);
                
                $result = [
                    'success' => true,
                    'tire_data' => $tireData,
                    'message' => 'Reifendaten erfolgreich extrahiert'
                ];
                
                echo json_encode($result);
                
            } catch (Exception $e) {
                debugLog("Reifendaten-Extraktion fehlgeschlagen: " . $e->getMessage());
                $result = [
                    'success' => false,
                    'error' => $e->getMessage()
                ];
                echo json_encode($result);
            }
            break;
            
        case 'get_cached_html':
            $appointmentId = $input['appointment_id'] ?? null;
            debugLog("Get cached HTML for appointment: $appointmentId");
            
            if (!$appointmentId) {
                throw new Exception('Appointment ID fehlt');
            }
            
            try {
                // Hole Storage Number für den Appointment
                $stmt = $conn->prepare("
                    SELECT storage_number 
                    FROM appointments 
                    WHERE id = ?
                ");
                $stmt->execute([$appointmentId]);
                $appointment = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if (!$appointment) {
                    throw new Exception('Appointment nicht gefunden');
                }
                
                $storageNumber = $appointment['storage_number'];
                debugLog("Storage Number: $storageNumber");
                
                // Lade HTML aus Cache
                $cachedHtml = getCachedHtml($storageNumber, $appointmentId);
                
                if ($cachedHtml && !empty($cachedHtml['html'])) {
                    debugLog("HTML erfolgreich aus Cache geladen");
                    echo json_encode([
                        'success' => true,
                        'html' => $cachedHtml['html'],
                        'message' => 'HTML-Daten aus Cache geladen'
                    ]);
                } else {
                    debugLog("Keine HTML-Daten im Cache gefunden");
                    echo json_encode([
                        'success' => false,
                        'error' => 'Keine HTML-Daten im Cache gefunden'
                    ]);
                }
                
            } catch (Exception $e) {
                debugLog("Fehler beim Laden der HTML-Daten aus Cache: " . $e->getMessage());
                echo json_encode([
                    'success' => false,
                    'error' => $e->getMessage()
                ]);
            }
            break;
            
        default:
            throw new Exception("Unbekannte Aktion: $action");
    }
    
} catch (Exception $e) {
    debugLog("API-Fehler: " . $e->getMessage());
    debugLog("Stack Trace: " . $e->getTraceAsString());
    
    http_response_code(500);
    $result = [
        'success' => false,
        'error' => $e->getMessage(),
        'debug' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'timestamp' => date('Y-m-d H:i:s')
        ]
    ];
    echo json_encode($result);
}

debugLog("=== BREAKTHROUGH RESY API Request Ended ===");
?>