<?php
/**
 * Verarbeitet RESY-Daten und speichert sie in der Datenbank
 * Wird vom BAT-Script aufgerufen mit: autohausId, JSON-String
 */

// DEBUG: Log-Datei für Fehler
$logDir = dirname(__DIR__) . '/logs';
if (!is_dir($logDir)) {
    mkdir($logDir, 0777, true);
}
$debugLog = $logDir . '/process_resy_data_' . date('Y-m-d') . '.log';

function debugLog($message, $logFile) {
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[{$timestamp}] {$message}\n", FILE_APPEND);
}

require_once dirname(__DIR__) . '/config/database.php';

// Kann sowohl als CLI als auch als Web-Request aufgerufen werden
if (php_sapi_name() !== 'cli') {
    header('Content-Type: application/json');
}

$autohausId = null;
$jsonData = null;

debugLog("=== START process_resy_data.php ===", $debugLog);
debugLog("argv: " . json_encode($argv ?? []), $debugLog);
debugLog("php_sapi_name: " . php_sapi_name(), $debugLog);

if (php_sapi_name() === 'cli') {
    // CLI-Modus: Parameter von argv
    $autohausId = $argv[1] ?? null;
    $jsonFile = $argv[2] ?? null;
    
    debugLog("CLI Mode - autohausId: {$autohausId}, jsonFile: {$jsonFile}", $debugLog);
    
    if ($jsonFile && file_exists($jsonFile)) {
        $jsonData = file_get_contents($jsonFile);
        debugLog("JSON-Datei gelesen: " . strlen($jsonData) . " bytes", $debugLog);
    } else {
        debugLog("WARNUNG: JSON-Datei nicht gefunden oder ungültig: {$jsonFile}", $debugLog);
        // Falls nicht als Datei, versuche direkt von stdin
        if (!feof(STDIN)) {
            $jsonData = stream_get_contents(STDIN);
            debugLog("JSON von STDIN gelesen: " . strlen($jsonData ?? '') . " bytes", $debugLog);
        }
    }
} else {
    // Web-Modus: Parameter von GET/POST
    $autohausId = $_GET['autohaus_id'] ?? $_POST['autohaus_id'] ?? null;
    $jsonData = file_get_contents('php://input');
    
    if (empty($jsonData)) {
        $jsonData = $_POST['data'] ?? null;
    }
    
    debugLog("Web Mode - autohausId: {$autohausId}", $debugLog);
}

if (!$autohausId) {
    debugLog("FEHLER: Keine Autohaus-ID angegeben", $debugLog);
    echo json_encode(['success' => false, 'error' => 'Keine Autohaus-ID angegeben']);
    exit(1);
}

if (empty($jsonData)) {
    debugLog("FEHLER: Keine Daten erhalten", $debugLog);
    echo json_encode(['success' => false, 'error' => 'Keine Daten erhalten']);
    exit(1);
}

debugLog("Autohaus ID: {$autohausId}, JSON-Länge: " . strlen($jsonData), $debugLog);

try {
    // Parse JSON
    $result = json_decode($jsonData, true);
    
    $jsonError = json_last_error();
    if ($jsonError !== JSON_ERROR_NONE) {
        $errorMsg = json_last_error_msg();
        debugLog("JSON Parse Error: {$errorMsg} (Code: {$jsonError})", $debugLog);
        echo json_encode(['success' => false, 'error' => "JSON Parse Error: {$errorMsg}"]);
        exit(1);
    }
    
    if (!$result || !isset($result['success']) || !$result['success']) {
        debugLog("FEHLER: Ungültige JSON-Daten - success nicht true", $debugLog);
        debugLog("JSON-Vorschau: " . substr($jsonData, 0, 200), $debugLog);
        echo json_encode(['success' => false, 'error' => 'Ungültige JSON-Daten']);
        exit(1);
    }
    
    $wheelsets = $result['wheelsets'] ?? [];
    debugLog("Wheelsets gefunden: " . count($wheelsets), $debugLog);
    
    if (empty($wheelsets)) {
        debugLog("WARNUNG: Keine Wheelsets zum Speichern", $debugLog);
        echo json_encode(['success' => true, 'count' => 0, 'message' => 'Keine Daten zum Speichern']);
        exit(0);
    }
    
    // Verbinde mit DB
    $db = new Database();
    $pdo = $db->getConnection();
    debugLog("Datenbank-Verbindung erfolgreich", $debugLog);
    
    // Lösche alte Daten für dieses Autohaus
    $stmt = $pdo->prepare("DELETE FROM stored_wheelsets_v2 WHERE autohaus_id = ?");
    $stmt->execute([$autohausId]);
    $deletedCount = $stmt->rowCount();
    debugLog("Alte Daten gelöscht: {$deletedCount} Einträge für Autohaus {$autohausId}", $debugLog);
    
    // Speichere neue Daten
    $seen = [];
    $count = 0;
    
    foreach ($wheelsets as $ws) {
        $kennzeichen = trim($ws['kennzeichen'] ?? '');
        
        if (empty($kennzeichen) || isset($seen[$kennzeichen])) {
            continue;
        }
        
        $seen[$kennzeichen] = true;
        
        try {
            $stmt = $pdo->prepare("
                INSERT INTO stored_wheelsets_v2 
                (autohaus_id, satznummer, haltername, kennzeichen, fin, autohaus, ziel_autohaus, 
                 eingangsdatum, rueckliefertermin, pruefbericht, reifenmodell, reifenart, saison, 
                 reifengroesse, lagerplatz, profiltiefe, empfehlung, rueckgabe_bemerkung, status, created_at, updated_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'gelagert', NOW(), NOW())
                ON DUPLICATE KEY UPDATE
                    satznummer = VALUES(satznummer),
                    haltername = VALUES(haltername),
                    fin = VALUES(fin),
                    autohaus = VALUES(autohaus),
                    ziel_autohaus = VALUES(ziel_autohaus),
                    eingangsdatum = VALUES(eingangsdatum),
                    rueckliefertermin = VALUES(rueckliefertermin),
                    reifenart = VALUES(reifenart),
                    saison = VALUES(saison),
                    reifengroesse = VALUES(reifengroesse),
                    profiltiefe = VALUES(profiltiefe),
                    empfehlung = VALUES(empfehlung),
                    updated_at = NOW()
            ");
            
            $stmt->execute([
                $autohausId,
                $ws['satznummer'] ?? '',
                $ws['haltername'] ?? '',
                $kennzeichen,
                $ws['fin'] ?? '',
                $ws['autohaus'] ?? '',
                $ws['ziel_autohaus'] ?? '',
                $ws['eingangsdatum'] ?? '',
                $ws['rueckliefertermin'] ?? '',
                $ws['pruefbericht'] ?? '',
                $ws['reifenmodell'] ?? '',
                $ws['reifenart'] ?? '',
                $ws['saison'] ?? '',
                $ws['reifengroesse'] ?? '',
                $ws['lagerplatz'] ?? '',
                $ws['profiltiefe'] ?? '',
                $ws['empfehlung'] ?? '',
                $ws['rueckgabe_bemerkung'] ?? ''
            ]);
            
            $count++;
        } catch (Exception $e) {
            error_log("Fehler beim Speichern von {$kennzeichen}: " . $e->getMessage());
        }
    }
    
    debugLog("=== ERFOLG: {$count} Reifensätze gespeichert für Autohaus {$autohausId} ===", $debugLog);
    
    echo json_encode([
        'success' => true,
        'count' => $count,
        'autohaus_id' => $autohausId
    ]);
    
} catch (Exception $e) {
    $errorMsg = $e->getMessage();
    $errorTrace = $e->getTraceAsString();
    debugLog("=== EXCEPTION: {$errorMsg} ===", $debugLog);
    debugLog("Stack Trace: {$errorTrace}", $debugLog);
    
    echo json_encode([
        'success' => false,
        'error' => $errorMsg
    ]);
    exit(1);
}

