<?php
/**
 * Professional Wheelsets API - RESY-Integration mit Fallback
 * Lädt Daten aus dem RESY-System oder der lokalen Datenbank
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once dirname(__DIR__) . '/config/database.php';
require_once __DIR__ . '/resy_selenium_simple.php';
require_once __DIR__ . '/resy_puppeteer.php';

class ProfessionalWheelsetsAPI {
    private $autohausId;
    private $autohausName;
    private $pdo;
    private $username;
    private $password;
    private $dealerId;
    
    public function __construct($autohausId) {
        $this->autohausId = $autohausId;
        $db = new Database();
        $this->pdo = $db->getConnection();
        $this->loadAutohausData();
    }
    
    private function loadAutohausData() {
        $stmt = $this->pdo->prepare("SELECT name, resy_username, resy_password FROM autohaus WHERE id = ?");
        $stmt->execute([$this->autohausId]);
        $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$autohaus) {
            throw new Exception("Autohaus nicht gefunden");
        }
        
        $this->autohausName = $autohaus['name'];
        $this->username = $autohaus['resy_username'] ?? null;
        $this->password = $autohaus['resy_password'] ?? null;
        $this->dealerId = $this->getDealerId();
    }
    
    private function getDealerId() {
        $mapping = [
            'B & E Volvo Kiel' => '453',
            'B & E Volvo Norderstedt' => '452',
            'B & E Volvo Glinde' => '454'
        ];
        return $mapping[$this->autohausName] ?? '453';
    }
    
    /**
     * Versucht Daten von RESY zu holen via Playwright
     */
    private function tryFetchFromResy() {
        if (!$this->username || !$this->password) {
            return null;
        }
        
        try {
            error_log("🔄 Starte automatischen RESY-Import via Playwright...");
            set_time_limit(300); // Mehr Zeit für alle 7 Seiten
            
            // Rufe Playwright auf
            $script = dirname(__DIR__) . '/resy_playwright.js';
            $command = "node \"$script\" 2>&1";
            
            $startTime = microtime(true);
            $output = shell_exec($command);
            $duration = round(microtime(true) - $startTime, 2);
            
            error_log("⏱️  Playwright-Ausführung: {$duration}s");
            
            // Parse JSON-Output
            $lines = explode("\n", $output);
            foreach ($lines as $line) {
                if (strpos($line, '{"success":') !== false) {
                    $result = json_decode($line, true);
                    
                    if ($result && $result['success'] && isset($result['wheelsets'])) {
                        $count = count($result['wheelsets']);
                        $this->saveWheelsetsToDatabase($result['wheelsets']);
                        error_log("✅ {$count} Reifensätze von RESY geladen ({$duration}s)");
                        return $result['wheelsets'];
                    }
                }
            }
            
            error_log("❌ Keine JSON-Daten von Playwright erhalten");
            return null;
            
        } catch (Exception $e) {
            error_log("❌ RESY-Fehler: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Parst RESY HTML und extrahiert Reifensätze
     * VERBESSERTE VERSION mit echter HTML-Parsing
     */
    private function parseResyHTML($html) {
        $wheelsets = [];
        
        // DOMDocument verwenden für besseres Parsing
        libxml_use_internal_errors(true);
        $dom = new DOMDocument();
        $dom->loadHTML($html);
        $xpath = new DOMXPath($dom);
        
        // Finde alle Tabellen-Zeilen
        $rows = $xpath->query('//tr');
        
        foreach ($rows as $row) {
            $cells = $xpath->query('./td', $row);
            
            if ($cells->length >= 5) {
                // Extrahiere Daten aus den Zellen
                $kennzeichen = trim($cells->item(0)->textContent ?? '');
                $satznummer = trim($cells->item(1)->textContent ?? '');
                $eingangsdatum = trim($cells->item(2)->textContent ?? '');
                
                // Nur speichern wenn Kennzeichen vorhanden ist
                if (!empty($kennzeichen) && strlen($kennzeichen) >= 3) {
                    $wheelsets[] = [
                        'kennzeichen' => $kennzeichen,
                        'satznummer' => $satznummer,
                        'eingangsdatum' => $eingangsdatum,
                        'marke' => 'RESY',
                        'modell' => $satznummer,
                        'oe_nummer' => $satznummer,
                        'reifengroesse' => '',
                        'status' => 'gelagert',
                        'archiviert' => 0
                    ];
                }
            }
        }
        
        return $wheelsets;
    }
    
    /**
     * Speichert Reifensätze in die Datenbank (ohne Duplikate)
     */
    private function saveWheelsetsToDatabase($wheelsets) {
        // Lösche alte Daten für dieses Autohaus zuerst
        $stmt = $this->pdo->prepare("DELETE FROM stored_wheelsets WHERE autohaus_id = ?");
        $stmt->execute([$this->autohausId]);
        
        // Speichere nur eindeutige Kennzeichen
        $seen = [];
        $count = 0;
        
        foreach ($wheelsets as $ws) {
            $kennzeichen = trim($ws['kennzeichen'] ?? '');
            
            // Überspringe leere oder doppelte Kennzeichen
            if (empty($kennzeichen) || isset($seen[$kennzeichen])) {
                continue;
            }
            
            $seen[$kennzeichen] = true;
            
            try {
                $stmt = $this->pdo->prepare("
                    INSERT INTO stored_wheelsets 
                    (autohaus_id, kennzeichen, marke, modell, oe_nummer, reifengroesse, status, created_at, updated_at)
                    VALUES (?, ?, ?, ?, ?, ?, 'gelagert', NOW(), NOW())
                ");
                
                $stmt->execute([
                    $this->autohausId,
                    $kennzeichen,
                    'RESY',  // Marke ist immer RESY
                    $ws['modell'] ?? '',  // Kundenname
                    $ws['oe_nummer'] ?? '',  // Satznummer
                    $ws['reifengroesse'] ?? '',  // Reifengröße aus RESY
                ]);
                
                $count++;
            } catch (Exception $e) {
                error_log("Fehler beim Speichern von {$kennzeichen}: " . $e->getMessage());
            }
        }
        
        error_log("💾 {$count} eindeutige Reifensätze gespeichert");
    }
    
    /**
     * Lädt Reifensätze - GELADENE Daten aus DB (kein Auto-Import!)
     */
    public function getStoredWheelsets() {
        try {
            // 2. Lade aus lokaler Datenbank (entweder frisch von RESY oder alte Daten)
               $stmt = $this->pdo->prepare("
                   SELECT 
                       w.id,
                       w.autohaus_id,
                       w.satznummer,
                       w.haltername,
                       w.kennzeichen,
                       w.fin,
                       w.autohaus,
                       w.ziel_autohaus,
                       w.eingangsdatum,
                       w.rueckliefertermin,
                       w.ausgangsdatum,
                       w.reifenmodell,
                       w.reifenart,
                       w.saison,
                       w.reifengroesse,
                       w.profiltiefe,
                       w.lagerplatz,
                       w.status,
                       w.empfehlung,
                       w.pruefbericht,
                       w.reifenvorschaden,
                       w.felgenvorschaden,
                       w.radkappenvorschaden,
                       w.rueckgabe_bemerkung,
                       w.bemerkung,
                       w.archiviert,
                       w.created_at,
                       w.updated_at,
                       CONCAT('Satz ', w.satznummer, ' - ', w.kennzeichen) as bezeichnung,
                       w.archiviert = 0 as active
                   FROM stored_wheelsets_v2 w
                   WHERE w.autohaus_id = ? AND w.archiviert = 0
                   ORDER BY w.updated_at DESC
               ");
            
            $stmt->execute([$this->autohausId]);
            $wheelsets = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return [
                'success' => true,
                'data' => $wheelsets,
                'wheelsets' => $wheelsets, // Für Frontend-Kompatibilität
                'count' => count($wheelsets),
                'source' => 'database',
                'autohaus' => $this->autohausName,
                'autohaus_id' => $this->autohausId,
                'fetched_from_resy' => false,
                'loaded_at' => date('Y-m-d H:i:s')
            ];
            
        } catch (Exception $e) {
               return [
                   'success' => true,
                   'data' => [],
                   'wheelsets' => [], // Für Frontend-Kompatibilität
                   'count' => 0,
                   'source' => 'database_v2',
                   'autohaus' => $this->autohausName,
                   'autohaus_id' => $this->autohausId,
                   'message' => 'Keine Reifensätze gefunden',
                   'loaded_at' => date('Y-m-d H:i:s')
               ];
           }
       }
}

// Haupt-Logik
try {
    $autohausId = $_GET['autohaus_id'] ?? $_POST['autohaus_id'] ?? 1;
    $api = new ProfessionalWheelsetsAPI($autohausId);
    $result = $api->getStoredWheelsets();
    
    echo json_encode($result, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
