<?php
/**
 * Reifenlager - Innovatives Split-View Design
 * Alle RESY REST API Features + Premium Erweiterungen
 * Master-Detail Layout mit Sidebar-Filters und Slide-in Panel
 */

session_start();

if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

require_once dirname(__DIR__) . '/config/database.php';

$db = new Database();
$conn = $db->getConnection();

// Lade Autohäuser
try {
    $stmt = $conn->prepare("SELECT id, name FROM autohaus WHERE is_active = 1 ORDER BY name");
    $stmt->execute();
    $autohauser = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $autohauser = [];
}

// User-Daten
$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_name = 'Admin';
$current_user_role = 'admin';

if ($current_user_id) {
    $stmt = $conn->prepare("
        SELECT u.full_name, u.username, u.role, u.autohaus_id, a.name as autohaus_name
        FROM admin_users u
        LEFT JOIN autohaus a ON u.autohaus_id = a.id
        WHERE u.id = ?
    ");
    $stmt->execute([$current_user_id]);
    $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
    $current_user_name = $user_data['full_name'] ?? $user_data['username'] ?? 'Benutzer';
    $current_user_role = $user_data['role'] ?? 'admin';
}

// Theme
$current_theme = 'light';
if ($current_user_id) {
    $stmt = $conn->prepare("SELECT theme FROM admin_users WHERE id = ?");
    $stmt->execute([$current_user_id]);
    $theme_result = $stmt->fetch();
    $current_theme = $theme_result['theme'] ?? 'light';
}
?>
<!DOCTYPE html>
<html lang="de" data-theme="<?php echo htmlspecialchars($current_theme); ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reifenlager - RESY Integration</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="assets/css/unified-sidebar.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://cdn.datatables.net/1.13.6/css/jquery.dataTables.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary: #3b82f6;
            --primary-dark: #2563eb;
            --secondary: #64748b;
            --success: #10b981;
            --warning: #f59e0b;
            --danger: #ef4444;
            --info: #06b6d4;
            
            --bg-main: #f1f5f9;
            --bg-panel: #ffffff;
            --bg-sidebar: #ffffff;
            --bg-hover: #f8fafc;
            --text-main: #0f172a;
            --text-secondary: #475569;
            --text-muted: #94a3b8;
            --border: #e2e8f0;
            --border-light: #f1f5f9;
            
            --shadow-sm: 0 1px 2px 0 rgba(0,0,0,0.05);
            --shadow: 0 1px 3px 0 rgba(0,0,0,0.1), 0 1px 2px 0 rgba(0,0,0,0.06);
            --shadow-md: 0 4px 6px -1px rgba(0,0,0,0.1), 0 2px 4px -1px rgba(0,0,0,0.06);
            --shadow-lg: 0 10px 15px -3px rgba(0,0,0,0.1), 0 4px 6px -2px rgba(0,0,0,0.05);
            
            --radius: 8px;
            --radius-lg: 12px;
        }

        [data-theme="dark"] {
            --bg-main: #0f172a;
            --bg-panel: #1e293b;
            --bg-sidebar: #1e293b;
            --bg-hover: #334155;
            --text-main: #f1f5f9;
            --text-secondary: #cbd5e1;
            --text-muted: #64748b;
            --border: #334155;
            --border-light: #1e293b;
        }

        * {
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', -apple-system, sans-serif;
            margin: 0;
            padding: 0;
            background: var(--bg-main);
            color: var(--text-main);
        }

        /* Main Layout - Split View */
        .storage-layout {
            display: flex;
            height: 100vh;
            overflow: hidden;
        }

        /* Left Sidebar */
        .filter-sidebar {
            width: 320px;
            background: var(--bg-sidebar);
            border-right: 1px solid var(--border);
            display: flex;
            flex-direction: column;
            overflow-y: auto;
        }

        .sidebar-header {
            padding: 1.5rem;
            border-bottom: 1px solid var(--border);
            background: var(--bg-hover);
        }

        .sidebar-title {
            font-size: 1.125rem;
            font-weight: 600;
            margin-bottom: 0.5rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .autohaus-select {
            width: 100%;
            padding: 0.625rem;
            border: 1px solid var(--border);
            border-radius: var(--radius);
            background: var(--bg-panel);
            color: var(--text-main);
            font-size: 0.875rem;
            margin-bottom: 1rem;
        }

        .filter-section {
            padding: 1.5rem;
            border-bottom: 1px solid var(--border);
        }

        .filter-title {
            font-size: 0.875rem;
            font-weight: 600;
            color: var(--text-secondary);
            margin-bottom: 1rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .filter-group {
            margin-bottom: 1.25rem;
        }

        .filter-label {
            display: block;
            font-size: 0.8125rem;
            color: var(--text-secondary);
            margin-bottom: 0.5rem;
        }

        .filter-input, .filter-select {
            width: 100%;
            padding: 0.5rem 0.75rem;
            border: 1px solid var(--border);
            border-radius: var(--radius);
            background: var(--bg-panel);
            color: var(--text-main);
            font-size: 0.875rem;
        }

        .filter-input:focus, .filter-select:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }

        .stats-compact {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 0.75rem;
            margin-top: 1rem;
        }

        .stat-mini {
            background: var(--bg-hover);
            padding: 0.75rem;
            border-radius: var(--radius);
            text-align: center;
        }

        .stat-mini-value {
            font-size: 1.25rem;
            font-weight: 700;
            color: var(--primary);
        }

        .stat-mini-label {
            font-size: 0.75rem;
            color: var(--text-secondary);
            margin-top: 0.25rem;
        }

        /* Main Content Area */
        .content-area {
            flex: 1;
            display: flex;
            flex-direction: column;
            overflow: hidden;
        }

        .content-header {
            padding: 1.25rem 1.5rem;
            background: var(--bg-panel);
            border-bottom: 1px solid var(--border);
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-shrink: 0;
        }

        .view-controls {
            display: flex;
            gap: 0.5rem;
        }

        .view-btn {
            padding: 0.5rem 1rem;
            border: 1px solid var(--border);
            background: var(--bg-panel);
            color: var(--text-secondary);
            border-radius: var(--radius);
            cursor: pointer;
            font-size: 0.875rem;
            transition: all 0.2s;
        }

        .view-btn:hover {
            background: var(--bg-hover);
        }

        .view-btn.active {
            background: var(--primary);
            color: white;
            border-color: var(--primary);
        }

        .content-body {
            flex: 1;
            overflow-y: auto;
            padding: 1.5rem;
        }

        /* Masonry Grid View */
        .masonry-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: 1.25rem;
        }

        .wheelset-item {
            background: var(--bg-panel);
            border: 1px solid var(--border);
            border-radius: var(--radius-lg);
            padding: 1.25rem;
            cursor: pointer;
            transition: all 0.2s;
            position: relative;
        }

        /* Empfehlungsstufen Farbcodierung */
        .wheelset-item.recommendation-ok {
            border-left: 4px solid var(--success);
        }

        .wheelset-item.recommendation-check {
            border-left: 4px solid var(--warning);
        }

        .wheelset-item.recommendation-replace {
            border-left: 4px solid var(--danger);
        }

        .wheelset-item:hover {
            box-shadow: var(--shadow-md);
            transform: translateY(-2px);
        }

        .wheelset-item.recommendation-ok:hover {
            border-color: var(--success);
        }

        .wheelset-item.recommendation-check:hover {
            border-color: var(--warning);
        }

        .wheelset-item.recommendation-replace:hover {
            border-color: var(--danger);
        }

        .wheelset-item.selected {
            border-width: 2px;
            background: rgba(59, 130, 246, 0.05);
        }

        .wheelset-item.recommendation-ok.selected {
            border-color: var(--success);
            background: rgba(16, 185, 129, 0.05);
        }

        .wheelset-item.recommendation-check.selected {
            border-color: var(--warning);
            background: rgba(245, 158, 11, 0.05);
        }

        .wheelset-item.recommendation-replace.selected {
            border-color: var(--danger);
            background: rgba(239, 68, 68, 0.05);
        }

        .item-header {
            display: flex;
            justify-content: space-between;
            align-items: start;
            margin-bottom: 1rem;
        }

        .item-number {
            font-size: 1.25rem;
            font-weight: 700;
            color: var(--primary);
        }

        .item-status {
            padding: 0.25rem 0.625rem;
            border-radius: 9999px;
            font-size: 0.75rem;
            font-weight: 500;
        }

        .status-stored { background: rgba(16, 185, 129, 0.1); color: var(--success); }
        .status-returned { background: rgba(6, 182, 212, 0.1); color: var(--info); }
        .status-warning { background: rgba(245, 158, 11, 0.1); color: var(--warning); }
        
        /* Order Status Badges */
        .order-status-badge {
            display: inline-flex;
            align-items: center;
            gap: 0.25rem;
            padding: 0.25rem 0.5rem;
            border-radius: 9999px;
            font-size: 0.6875rem;
            font-weight: 600;
            margin-left: 0.5rem;
            white-space: nowrap;
        }
        
        .order-status-badge.pickup {
            background: rgba(59, 130, 246, 0.15);
            color: var(--primary);
            border: 1px solid rgba(59, 130, 246, 0.3);
        }
        
        .order-status-badge.return {
            background: rgba(139, 92, 246, 0.15);
            color: #8b5cf6;
            border: 1px solid rgba(139, 92, 246, 0.3);
        }
        
        .order-status-badge.both {
            background: linear-gradient(135deg, rgba(59, 130, 246, 0.15) 0%, rgba(139, 92, 246, 0.15) 100%);
            color: var(--primary);
            border: 1px solid rgba(59, 130, 246, 0.3);
        }
        
        .order-status-tooltip {
            position: absolute;
            bottom: calc(100% + 8px);
            left: 50%;
            transform: translateX(-50%);
            background: var(--text-main);
            color: white;
            padding: 0.5rem 0.75rem;
            border-radius: var(--radius);
            font-size: 0.75rem;
            white-space: nowrap;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
            pointer-events: none;
            z-index: 1000;
            box-shadow: var(--shadow-md);
        }
        
        .order-status-tooltip::after {
            content: '';
            position: absolute;
            top: 100%;
            left: 50%;
            transform: translateX(-50%);
            border: 5px solid transparent;
            border-top-color: var(--text-main);
        }
        
        .order-status-badge:hover .order-status-tooltip {
            opacity: 1;
            visibility: visible;
        }
        
        /* Wheelset Item mit Order-Status */
        .wheelset-item.has-pickup {
            border-top: 3px solid var(--primary);
        }
        
        .wheelset-item.has-return {
            border-top: 3px solid var(--success);
        }
        
        .wheelset-item.has-both-orders {
            border-top: 3px solid var(--primary);
            border-bottom: 3px solid var(--success);
        }
        
        .list-item.has-pickup {
            border-left-color: var(--primary);
        }
        
        .list-item.has-return {
            border-left-color: var(--success);
        }
        
        .list-item.has-both-orders {
            border-left: 4px solid;
            border-image: linear-gradient(to bottom, var(--primary), var(--success)) 1;
        }

        .item-info {
            display: grid;
            gap: 0.5rem;
            font-size: 0.875rem;
        }

        .info-row {
            display: flex;
            justify-content: space-between;
        }

        .info-label {
            color: var(--text-secondary);
        }

        .info-value {
            color: var(--text-main);
            font-weight: 500;
        }

        .item-actions {
            margin-top: 1rem;
            padding-top: 1rem;
            border-top: 1px solid var(--border);
            display: flex;
            gap: 0.5rem;
        }

        .action-btn {
            flex: 1;
            padding: 0.5rem;
            border: 1px solid var(--border);
            background: var(--bg-panel);
            border-radius: var(--radius);
            cursor: pointer;
            font-size: 0.8125rem;
            transition: all 0.2s;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.25rem;
        }

        .action-btn:hover {
            background: var(--bg-hover);
        }

        .action-btn.primary { border-color: var(--primary); color: var(--primary); }
        .action-btn.success { border-color: var(--success); color: var(--success); }
        .action-btn.warning { border-color: var(--warning); color: var(--warning); }

        /* Compact List View */
        .compact-list {
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        }

        .list-item {
            background: var(--bg-panel);
            border: 1px solid var(--border);
            border-radius: var(--radius);
            padding: 1rem;
            display: grid;
            grid-template-columns: auto 1fr auto auto;
            gap: 1rem;
            align-items: center;
            cursor: pointer;
            transition: all 0.2s;
        }

        /* List Item Empfehlungsstufen */
        .list-item.recommendation-ok {
            border-left: 4px solid var(--success);
        }

        .list-item.recommendation-check {
            border-left: 4px solid var(--warning);
        }

        .list-item.recommendation-replace {
            border-left: 4px solid var(--danger);
        }

        .list-item:hover {
            background: var(--bg-hover);
        }

        .list-item.selected {
            border-color: var(--primary);
            background: rgba(59, 130, 246, 0.05);
        }

        .list-item.recommendation-ok.selected {
            border-color: var(--success);
            background: rgba(16, 185, 129, 0.05);
        }

        .list-item.recommendation-check.selected {
            border-color: var(--warning);
            background: rgba(245, 158, 11, 0.05);
        }

        .list-item.recommendation-replace.selected {
            border-color: var(--danger);
            background: rgba(239, 68, 68, 0.05);
        }

        .list-checkbox {
            width: 20px;
            height: 20px;
            cursor: pointer;
        }

        /* Right Detail Panel */
        .detail-panel {
            width: 0;
            background: var(--bg-panel);
            border-left: 1px solid var(--border);
            transition: width 0.3s ease, right 0.3s ease;
            overflow: hidden;
            position: relative;
        }

        .detail-panel.open {
            width: 420px;
        }
        
        /* Detail Panel verschieben wenn Orders Panel auch offen ist */
        .detail-panel.open.orders-panel-active {
            position: fixed;
            right: 803px; /* 800px Orders Panel + 3px Separator */
            top: 0;
            height: 100vh;
            z-index: 1500;
            box-shadow: 4px 0 20px rgba(0, 0, 0, 0.1);
        }
        
        /* Dezenter Separator zwischen den Panels */
        .panel-separator {
            position: fixed;
            right: 800px;
            top: 0;
            width: 3px;
            height: 100vh;
            background: linear-gradient(to bottom, 
                transparent 0%, 
                rgba(59, 130, 246, 0.15) 20%, 
                rgba(59, 130, 246, 0.15) 80%, 
                transparent 100%);
            z-index: 1499;
            display: none;
            pointer-events: none;
        }
        
        .panel-separator.active {
            display: block;
        }

        .panel-header {
            padding: 1.5rem;
            border-bottom: 1px solid var(--border);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .panel-header-left {
            display: flex;
            align-items: center;
            gap: 1rem;
            flex: 1;
        }

        .panel-header-icons {
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .panel-header-icon {
            width: 32px;
            height: 32px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            background: var(--bg-panel);
            border: 1.5px solid var(--border);
            color: var(--text-secondary);
            cursor: pointer;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            font-size: 0.875rem;
            position: relative;
        }

        .panel-header-icon:hover {
            background: var(--bg-hover);
            border-color: var(--primary);
            color: var(--primary);
            transform: scale(1.1);
            box-shadow: var(--shadow-sm);
        }

        .panel-header-icon.has-data {
            border-color: var(--info);
            color: var(--info);
            background: linear-gradient(135deg, rgba(6, 182, 212, 0.15) 0%, rgba(59, 130, 246, 0.1) 100%);
        }

        .panel-header-icon.has-data:hover {
            box-shadow: 0 0 0 3px rgba(6, 182, 212, 0.2);
        }

        .panel-header-icon-badge {
            position: absolute;
            top: -4px;
            right: -4px;
            width: 10px;
            height: 10px;
            background: var(--info);
            border: 2px solid var(--bg-panel);
            border-radius: 50%;
            animation: commentPulse 2s infinite;
        }

        .panel-header-icon.placeholder {
            opacity: 0.4;
            cursor: default;
        }

        .panel-header-icon.placeholder:hover {
            transform: none;
            border-color: var(--border);
            color: var(--text-muted);
            background: var(--bg-panel);
            box-shadow: none;
        }

        .panel-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            color: var(--text-secondary);
            cursor: pointer;
            width: 32px;
            height: 32px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: var(--radius);
        }

        .panel-close:hover {
            background: var(--bg-hover);
        }

        .panel-body {
            padding: 1.5rem;
            overflow-y: auto;
            height: calc(100vh - 80px);
        }

        /* Kompaktes innovatives Bemerkungs-Icon Design */
        .comment-badge-container {
            position: relative;
            display: inline-flex;
            align-items: center;
            margin-bottom: 1rem;
        }

        .comment-icon-wrapper {
            position: relative;
            display: inline-flex;
            align-items: center;
            cursor: pointer;
        }

        .comment-icon-btn {
            width: 36px;
            height: 36px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            background: var(--bg-panel);
            border: 1.5px solid var(--border);
            color: var(--text-secondary);
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            font-size: 1rem;
            position: relative;
        }

        .comment-icon-btn:hover {
            background: var(--bg-hover);
            border-color: var(--primary);
            color: var(--primary);
            transform: scale(1.1);
            box-shadow: var(--shadow-md);
        }

        .comment-icon-btn.has-comment {
            border-color: var(--info);
            color: var(--info);
            background: linear-gradient(135deg, rgba(6, 182, 212, 0.15) 0%, rgba(59, 130, 246, 0.1) 100%);
        }

        .comment-icon-btn.has-comment:hover {
            border-color: var(--info);
            color: var(--info);
            box-shadow: 0 0 0 4px rgba(6, 182, 212, 0.2);
        }

        /* Kleines X-Icon für Löschen beim Hover */
        .comment-delete-hint {
            position: absolute;
            top: -8px;
            left: -8px;
            width: 20px;
            height: 20px;
            background: var(--danger);
            border: 2px solid var(--bg-panel);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 0.625rem;
            opacity: 0;
            transform: scale(0);
            transition: all 0.2s ease;
            cursor: pointer;
            z-index: 10;
        }

        .comment-icon-wrapper:hover .comment-delete-hint {
            opacity: 1;
            transform: scale(1);
        }

        .comment-delete-hint:hover {
            transform: scale(1.15);
            box-shadow: var(--shadow-md);
        }

        /* Kompakter Notification-Badge */
        .comment-notification-badge {
            position: absolute;
            top: -2px;
            right: -2px;
            width: 12px;
            height: 12px;
            background: var(--info);
            border: 2px solid var(--bg-panel);
            border-radius: 50%;
            animation: commentPulse 2s infinite;
            box-shadow: 0 0 0 0 rgba(6, 182, 212, 0.4);
        }

        @keyframes commentPulse {
            0% {
                box-shadow: 0 0 0 0 rgba(6, 182, 212, 0.4);
                transform: scale(1);
            }
            50% {
                box-shadow: 0 0 0 4px rgba(6, 182, 212, 0);
                transform: scale(1.05);
            }
            100% {
                box-shadow: 0 0 0 0 rgba(6, 182, 212, 0);
                transform: scale(1);
            }
        }

        /* Dropdown-Menü für Aktionen */
        .comment-actions-menu {
            position: absolute;
            top: calc(100% + 8px);
            right: 0;
            background: var(--bg-panel);
            border: 1px solid var(--border);
            border-radius: var(--radius);
            box-shadow: var(--shadow-lg);
            min-width: 200px;
            opacity: 0;
            visibility: hidden;
            transform: translateY(-8px) scale(0.95);
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            z-index: 1000;
            overflow: hidden;
        }

        .comment-actions-menu.active {
            opacity: 1;
            visibility: visible;
            transform: translateY(0) scale(1);
        }

        .comment-menu-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem 1rem;
            cursor: pointer;
            transition: all 0.2s;
            font-size: 0.875rem;
            color: var(--text-main);
            border: none;
            background: none;
            width: 100%;
            text-align: left;
        }

        .comment-menu-item:hover {
            background: var(--bg-hover);
        }

        .comment-menu-item.delete {
            color: var(--danger);
        }

        .comment-menu-item.delete:hover {
            background: rgba(239, 68, 68, 0.1);
        }

        .comment-menu-divider {
            height: 1px;
            background: var(--border);
            margin: 0.25rem 0;
        }

        /* Tooltip für Kommentar-Vorschau */
        .comment-preview-tooltip {
            position: absolute;
            top: calc(100% + 8px);
            right: 0;
            background: var(--text-main);
            color: white;
            padding: 0.75rem 1rem;
            border-radius: var(--radius);
            font-size: 0.8125rem;
            white-space: pre-wrap;
            max-width: 280px;
            box-shadow: var(--shadow-lg);
            z-index: 999;
            opacity: 0;
            visibility: hidden;
            transform: translateY(-8px);
            transition: all 0.3s ease;
            pointer-events: none;
        }

        .comment-preview-tooltip::before {
            content: '';
            position: absolute;
            bottom: 100%;
            right: 1rem;
            border: 6px solid transparent;
            border-bottom-color: var(--text-main);
        }

        .comment-icon-wrapper:hover .comment-preview-tooltip {
            opacity: 1;
            visibility: visible;
            transform: translateY(0);
        }

        .detail-section {
            margin-bottom: 1.5rem;
        }

        .detail-title {
            font-size: 0.875rem;
            font-weight: 600;
            color: var(--text-secondary);
            margin-bottom: 0.75rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .detail-grid {
            display: grid;
            gap: 0.75rem;
        }

        .detail-row {
            display: flex;
            justify-content: space-between;
            padding: 0.625rem 0;
            border-bottom: 1px solid var(--border-light);
        }

        .detail-label {
            color: var(--text-secondary);
            font-size: 0.875rem;
        }

        .detail-value {
            color: var(--text-main);
            font-weight: 500;
            font-size: 0.875rem;
        }

        /* Kompakte Reifen-Übersicht im Detail-Panel */
        .wheels-overview-grid {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 0.75rem;
            margin-top: 1rem;
        }
        
        .wheel-overview-item {
            background: var(--bg-panel);
            border: 2px solid var(--border);
            border-radius: var(--radius);
            padding: 1rem;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .wheel-overview-item:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-md);
            border-color: var(--primary);
        }
        
        .wheel-overview-item.status-ok {
            border-color: var(--success);
            background: linear-gradient(135deg, var(--bg-panel) 0%, rgba(16, 185, 129, 0.05) 100%);
        }
        
        .wheel-overview-item.status-warning {
            border-color: var(--warning);
            background: linear-gradient(135deg, var(--bg-panel) 0%, rgba(245, 158, 11, 0.05) 100%);
        }
        
        .wheel-overview-item.status-danger {
            border-color: var(--danger);
            background: linear-gradient(135deg, var(--bg-panel) 0%, rgba(239, 68, 68, 0.05) 100%);
        }
        
        .wheel-overview-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 0.75rem;
            padding-bottom: 0.5rem;
            border-bottom: 1px solid var(--border);
        }
        
        .wheel-overview-position {
            font-size: 1.125rem;
            font-weight: 700;
            color: var(--primary);
            letter-spacing: 1px;
        }
        
        .wheel-overview-status {
            font-size: 1.25rem;
        }
        
        .wheel-overview-status.status-ok {
            color: var(--success);
        }
        
        .wheel-overview-status.status-warning {
            color: var(--warning);
        }
        
        .wheel-overview-status.status-danger {
            color: var(--danger);
        }
        
        .wheel-overview-info {
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        }
        
        .wheel-overview-brand {
            font-size: 0.875rem;
            font-weight: 600;
            color: var(--text-main);
        }
        
        .wheel-overview-details {
            display: flex;
            justify-content: space-between;
            align-items: center;
            font-size: 0.8125rem;
            color: var(--text-secondary);
        }
        
        .wheel-overview-details i {
            margin-right: 0.25rem;
            color: var(--primary);
        }
        
        .wheel-overview-recommendation {
            padding: 0.25rem 0.5rem;
            border-radius: var(--radius);
            font-size: 0.75rem;
            font-weight: 600;
        }
        
        .wheel-overview-recommendation.ok {
            background: rgba(16, 185, 129, 0.15);
            color: var(--success);
        }
        
        .wheel-overview-recommendation.warning {
            background: rgba(245, 158, 11, 0.15);
            color: var(--warning);
        }
        
        .wheel-overview-recommendation.danger {
            background: rgba(239, 68, 68, 0.15);
            color: var(--danger);
        }

        /* Auto-zentriertes Reifen-Layout (RESY-Style) für Modal */
        .wheels-car-layout {
            position: relative;
            width: 100%;
            min-height: auto;
            padding: 1.5rem;
            background: linear-gradient(135deg, var(--bg-hover) 0%, var(--bg-panel) 100%);
            border-radius: var(--radius-lg);
            margin-top: 0;
            overflow: visible;
            box-sizing: border-box;
        }
        
        .car-container {
            position: relative;
            width: 100%;
            max-width: 1600px;
            margin: 0 auto;
            box-sizing: border-box;
            display: flex;
            flex-direction: row;
            align-items: center;
            justify-content: center;
            gap: 2rem;
        }
        
        /* Auto-Bild Illustration - Zentriert in der Mitte */
        .car-illustration {
            flex: 0 0 auto;
            max-width: 500px;
            width: 100%;
            height: auto;
            max-height: 400px;
            object-fit: contain;
            margin: 0;
            display: block;
            z-index: 1;
        }
        
        /* 4 Positions-Panels horizontal neben dem Auto */
        .wheels-panels-grid {
            position: relative;
            width: 100%;
            display: flex;
            flex-direction: column;
            gap: 1.5rem;
            flex: 0 0 auto;
        }
        
        .wheels-panels-side {
            display: flex;
            flex-direction: column;
            gap: 1.5rem;
            justify-content: center;
            align-items: stretch;
            flex: 0 0 auto;
        }
        
        .wheels-panels-side.left {
            align-items: flex-end;
        }
        
        .wheels-panels-side.right {
            align-items: flex-start;
        }
        
        .wheel-position-panel {
            width: 100%;
            max-width: 420px;
            min-width: 380px;
            min-height: auto;
            background: var(--bg-panel);
            border: 2px solid var(--border);
            border-radius: var(--radius-lg);
            padding: 1.25rem;
            box-shadow: var(--shadow-md);
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            box-sizing: border-box;
            overflow: visible;
            position: relative;
            display: flex;
            flex-direction: column;
        }
        
        .wheel-position-panel.selectable {
            cursor: pointer;
        }
        
        .wheel-position-panel.selectable:hover {
            border-color: var(--primary);
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }
        
        .wheel-position-panel.selected {
            border-color: var(--primary) !important;
            border-width: 4px !important;
            box-shadow: 0 0 20px rgba(59, 130, 246, 0.5), var(--shadow-lg) !important;
            background: linear-gradient(135deg, var(--bg-panel) 0%, rgba(59, 130, 246, 0.1) 100%) !important;
            transform: translateY(-2px);
        }
        
        .wheel-position-panel.selected::before {
            content: '';
            position: absolute;
            top: 1rem;
            right: 1rem;
            width: 32px;
            height: 32px;
            background: var(--primary);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 10;
            box-shadow: 0 0 12px rgba(59, 130, 246, 0.6);
        }
        
        .wheel-position-panel.selected::after {
            content: '✓';
            position: absolute;
            top: 1rem;
            right: 1rem;
            width: 32px;
            height: 32px;
            color: white;
            font-weight: bold;
            font-size: 18px;
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 11;
        }
        
        /* Button für ausgewählte Reifen */
        .selected-wheels-button {
            position: fixed;
            bottom: 2rem;
            right: 2rem;
            background: var(--primary);
            color: white;
            border: none;
            border-radius: var(--radius-lg);
            padding: 1rem 1.5rem;
            font-size: 1rem;
            font-weight: 600;
            box-shadow: 0 8px 24px rgba(59, 130, 246, 0.4);
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 0.75rem;
            z-index: 10002;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            animation: slideInUp 0.3s ease-out;
        }
        
        @keyframes slideInUp {
            from {
                transform: translateY(100%);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        .selected-wheels-button:hover {
            transform: translateY(-3px);
            box-shadow: 0 12px 32px rgba(59, 130, 246, 0.5);
            background: var(--primary-hover, #2563eb);
        }
        
        .selected-wheels-button:active {
            transform: translateY(-1px);
        }
        
        #selectedWheelsCount {
            background: rgba(255, 255, 255, 0.25);
            padding: 0.375rem 0.75rem;
            border-radius: var(--radius);
            font-weight: 700;
            min-width: 2.5rem;
            text-align: center;
            font-size: 1.125rem;
        }
        
        /* Verbindungslinien zwischen Auto-Rädern und Panels */
        .wheel-connections-svg {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            pointer-events: none;
            z-index: 1;
        }
        
        .wheel-connection-line {
            stroke: var(--success);
            stroke-width: 3;
            stroke-dasharray: 1000;
            stroke-dashoffset: 1000;
            fill: none;
            opacity: 0.85;
            transition: all 0.6s ease-out;
            filter: drop-shadow(0 0 3px rgba(16, 185, 129, 0.4));
        }
        
        @keyframes drawLine {
            0% {
                stroke-dashoffset: 1000;
                opacity: 0;
            }
            30% {
                opacity: 0.5;
            }
            100% {
                stroke-dashoffset: 0;
                opacity: 0.85;
            }
        }
        
        /* Animierte Punkte entlang der Linie - DEUTLICH SICHTBAR für alle */
        .wheel-connection-dot {
            fill: #10b981 !important;
            r: 8 !important;
            opacity: 1 !important;
            filter: drop-shadow(0 0 5px rgba(16, 185, 129, 1)) !important;
            pointer-events: none;
        }
        
        /* Hover-Effekte */
        .wheel-position-panel:hover {
            transform: translateY(-3px);
            box-shadow: var(--shadow-lg);
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        /* Hover-Effekt auf Linien - via JavaScript */
        .wheel-connection-line.highlighted {
            opacity: 1 !important;
            stroke-width: 3.5 !important;
            filter: drop-shadow(0 0 6px var(--success)) !important;
        }
        
        .wheel-connection-glow.highlighted {
            opacity: 0.35 !important;
            stroke-width: 7 !important;
            filter: blur(5px) !important;
        }
        
        .wheel-connection-dot.highlighted {
            opacity: 1 !important;
            filter: drop-shadow(0 0 5px var(--success)) !important;
        }
        
        .car-container {
            position: relative;
        }
        
        .wheel-position-panel::-webkit-scrollbar {
            width: 6px;
        }
        
        .wheel-position-panel::-webkit-scrollbar-track {
            background: var(--bg-hover);
        }
        
        .wheel-position-panel::-webkit-scrollbar-thumb {
            background: var(--border);
            border-radius: 3px;
        }
        
        .wheel-position-panel:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
            border-color: var(--primary);
        }
        
        /* Panel Positionen - Keine Grid-Positionierung mehr, flexbox übernimmt */
        
        /* Panel Status-Border (farbkodiert) */
        .wheel-position-panel.status-ok {
            border-color: var(--success);
            border-width: 3px;
            background: linear-gradient(135deg, var(--bg-panel) 0%, rgba(16, 185, 129, 0.03) 100%);
        }
        
        .wheel-position-panel.status-warning {
            border-color: var(--warning);
            border-width: 3px;
            background: linear-gradient(135deg, var(--bg-panel) 0%, rgba(245, 158, 11, 0.03) 100%);
        }
        
        .wheel-position-panel.status-danger {
            border-color: var(--danger);
            border-width: 3px;
            background: linear-gradient(135deg, var(--bg-panel) 0%, rgba(239, 68, 68, 0.03) 100%);
        }
        
        /* Panel Header */
        .wheel-panel-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 0.5rem;
            padding-bottom: 0.375rem;
            border-bottom: 2px solid var(--border);
            flex-shrink: 0;
        }
        
        /* Kritische Infos: Profiltiefe und Druck - Immer sichtbar oben */
        .wheel-critical-info {
            margin-bottom: 0.75rem;
            padding: 0.75rem;
            background: var(--bg-hover);
            border-radius: var(--radius);
            flex-shrink: 0;
            border: 1px solid var(--border);
        }
        
        .critical-info-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1rem;
        }
        
        .critical-info-item {
            text-align: left;
            padding: 0.5rem;
            background: var(--bg-hover);
            border-radius: var(--radius);
        }
        
        .critical-info-label {
            font-size: 0.8125rem;
            font-weight: 600;
            color: var(--text-secondary);
            text-transform: uppercase;
            letter-spacing: 0.5px;
            margin-bottom: 0.5rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .critical-info-value {
            font-size: 1.25rem;
            font-weight: 700;
            font-family: 'Courier New', monospace;
            color: var(--text-main);
        }
        
        .critical-info-value.success {
            color: var(--success);
        }
        
        .critical-info-value.warning {
            color: var(--warning);
        }
        
        .critical-info-value.danger {
            color: var(--danger);
        }
        
        .critical-info-value.default {
            color: var(--text-muted);
        }
        
        .wheel-panel-position {
            font-size: 1.25rem;
            font-weight: 700;
            color: var(--primary);
            letter-spacing: 1px;
            display: flex;
            align-items: center;
            gap: 0.375rem;
        }
        
        .wheel-panel-position-icon {
            width: 28px;
            height: 28px;
            background: var(--primary);
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.875rem;
            font-weight: bold;
        }
        
        .wheel-panel-status {
            width: 28px;
            height: 28px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
            font-size: 0.875rem;
            flex-shrink: 0;
        }
        
        .wheel-panel-status.ok {
            background: var(--success);
            color: white;
        }
        
        .wheel-panel-status.warning {
            background: var(--warning);
            color: white;
        }
        
        .wheel-panel-status.danger {
            background: var(--danger);
            color: white;
        }
        
        /* Profiltiefen-Messungen (I, M, A) - Kompakt */
        .wheel-profile-measurements {
            margin-bottom: 0.5rem;
            flex-shrink: 0;
        }
        
        .profile-measures-grid {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 0.25rem;
        }
        
        .profile-measure-item {
            text-align: center;
            padding: 0.25rem;
            background: var(--bg-hover);
            border-radius: var(--radius);
            border: 1px solid var(--border);
            transition: all 0.2s;
        }
        
        .profile-measure-item:hover {
            border-color: var(--primary);
            transform: translateY(-2px);
        }
        
        .profile-measure-label {
            font-size: 0.625rem;
            font-weight: 700;
            color: var(--text-secondary);
            margin-bottom: 0.125rem;
            text-transform: uppercase;
        }
        
        .profile-measure-value {
            font-size: 0.75rem;
            font-weight: 700;
            font-family: 'Courier New', monospace;
            margin-bottom: 0;
        }
        
        .profile-measure-value.success {
            color: var(--success);
        }
        
        .profile-measure-value.warning {
            color: var(--warning);
        }
        
        .profile-measure-value.danger {
            color: var(--danger);
        }
        
        .profile-measure-unit {
            font-size: 0.75rem;
            color: var(--text-muted);
        }
        
        /* Empfehlungs-Badge */
        .wheel-recommendation-badge {
            display: inline-flex;
            align-items: center;
            gap: 0.375rem;
            padding: 0.4rem 0.75rem;
            border-radius: var(--radius);
            font-size: 0.8125rem;
            font-weight: 600;
            margin-bottom: 0.5rem;
            width: 100%;
            justify-content: center;
            flex-shrink: 0;
        }
        
        .wheel-recommendation-badge.ok {
            background: rgba(16, 185, 129, 0.15);
            color: var(--success);
            border: 2px solid var(--success);
        }
        
        .wheel-recommendation-badge.warning {
            background: rgba(245, 158, 11, 0.15);
            color: var(--warning);
            border: 2px solid var(--warning);
        }
        
        .wheel-recommendation-badge.danger {
            background: rgba(239, 68, 68, 0.15);
            color: var(--danger);
            border: 2px solid var(--danger);
        }
        
        /* Wheel Info Section */
        .wheel-panel-info {
            font-size: 0.8125rem;
            color: var(--text-secondary);
            margin-top: 0.5rem;
        }
        
        .wheel-info-section {
            margin-bottom: 0.375rem;
            padding: 0.375rem;
            background: var(--bg-hover);
            border-radius: var(--radius);
        }
        
        .wheel-info-section.compact {
            padding: 0.5rem;
            margin-bottom: 0.5rem;
        }
        
        .wheel-info-label {
            font-size: 0.625rem;
            font-weight: 600;
            color: var(--text-secondary);
            text-transform: uppercase;
            letter-spacing: 0.5px;
            margin-bottom: 0.25rem;
        }
        
        .wheel-info-label i {
            margin-right: 0.375rem;
        }
        
        .wheel-info-value {
            font-size: 0.75rem;
            font-weight: 600;
            color: var(--text-main);
            word-break: break-word;
        }
        
        .wheel-panel-brand {
            font-weight: 700;
            color: var(--text-main);
            font-size: 0.9375rem;
            margin-bottom: 0.25rem;
        }
        
        .wheel-panel-size {
            font-size: 0.8125rem;
            color: var(--text-secondary);
            font-family: monospace;
        }
        
        .wheel-panel-size-type {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            margin-top: 0.25rem;
            flex-wrap: wrap;
        }
        
        .wheel-panel-type {
            display: inline-flex;
            align-items: center;
            gap: 0.25rem;
            padding: 0.25rem 0.5rem;
            background: rgba(59, 130, 246, 0.1);
            color: var(--primary);
            border-radius: var(--radius);
            font-size: 0.6875rem;
            font-weight: 600;
        }
        
        .wheel-additional-info {
            display: flex;
            flex-wrap: wrap;
            gap: 0.25rem;
            justify-content: center;
        }
        
        .wheel-info-chip {
            display: inline-flex;
            align-items: center;
            gap: 0.25rem;
            padding: 0.2rem 0.4rem;
            background: rgba(59, 130, 246, 0.08);
            color: var(--text-secondary);
            border-radius: var(--radius);
            font-size: 0.625rem;
            font-weight: 600;
            border: 1px solid rgba(59, 130, 246, 0.2);
        }
        
        .wheel-info-chip i {
            font-size: 0.6rem;
        }
        
        /* Responsive Design */
        @media (max-width: 1024px) {
            .wheels-car-layout {
                padding: 1.5rem;
                min-height: auto;
            }
            
            .car-container {
                max-width: 100%;
            }
            
            .car-illustration {
                max-width: 500px;
                max-height: 350px;
            }
            
            .wheel-position-panel {
                min-height: 220px;
                max-width: 280px;
                padding: 0.875rem;
            }
        }
        
        @media (max-width: 768px) {
            .wheels-car-layout {
                padding: 1rem;
                min-height: auto;
            }
            
            .car-container {
                height: auto;
                min-height: auto;
            }
            
            .wheels-panels-grid {
                grid-template-columns: 1fr;
                grid-template-rows: auto auto auto auto;
                gap: 1rem;
            }
            
            .wheel-position-panel {
                position: relative;
                width: 100%;
                min-height: auto;
            }
            
            .wheel-position-panel.vl {
                grid-column: 1;
                grid-row: 1;
            }
            
            .wheel-position-panel.vr {
                grid-column: 1;
                grid-row: 2;
            }
            
            .wheel-position-panel.hl {
                grid-column: 1;
                grid-row: 3;
            }
            
            .wheel-position-panel.hr {
                grid-column: 1;
                grid-row: 4;
            }
            
            .car-illustration {
                width: 100%;
                max-width: 280px;
                height: auto;
                margin: 0 auto;
            }
        }

        /* Prüfbericht Viewer */
        .test-report-viewer {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(15, 23, 42, 0.95);
            backdrop-filter: blur(8px);
            z-index: 10001;
            display: none;
            flex-direction: column;
            opacity: 0;
            transition: opacity 0.3s ease;
        }

        .test-report-viewer.active {
            display: flex;
            opacity: 1;
        }

        .test-report-header {
            padding: 1.5rem;
            background: var(--bg-panel);
            border-bottom: 1px solid var(--border);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .test-report-title {
            font-size: 1.25rem;
            font-weight: 600;
        }

        .test-report-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            color: var(--text-secondary);
            cursor: pointer;
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: var(--radius);
        }

        .test-report-close:hover {
            background: var(--bg-hover);
        }

        .test-report-body {
            flex: 1;
            padding: 1.5rem;
            overflow: auto;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .test-report-content {
            width: 100%;
            max-width: 1200px;
            height: 100%;
            display: flex;
            flex-direction: column;
        }

        .test-report-iframe {
            width: 100%;
            flex: 1;
            border: 1px solid var(--border);
            border-radius: var(--radius);
            background: white;
        }

        .test-report-actions {
            padding: 1rem 1.5rem;
            background: var(--bg-panel);
            border-top: 1px solid var(--border);
            display: flex;
            gap: 0.75rem;
            justify-content: flex-end;
        }

        .btn-sm {
            padding: 0.5rem 1rem;
            font-size: 0.8125rem;
        }

        /* Slot Calendar Design */
        .slot-calendar {
            width: 100%;
            margin-top: 1rem;
            background: var(--bg-hover);
            border-radius: var(--radius-lg);
            padding: 1.25rem;
        }

        .slot-calendar-wrapper {
            position: relative;
        }

        .calendar-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1rem;
        }

        .calendar-nav {
            display: flex;
            gap: 0.5rem;
            align-items: center;
        }

        .calendar-nav-btn {
            width: 36px;
            height: 36px;
            border: 1px solid var(--border);
            background: var(--bg-panel);
            border-radius: var(--radius);
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            color: var(--text-main);
            transition: all 0.2s;
        }

        .calendar-nav-btn:hover {
            background: var(--bg-hover);
            border-color: var(--primary);
        }

        .calendar-month {
            font-weight: 600;
            font-size: 1rem;
            color: var(--text-main);
        }

        .calendar-grid {
            display: grid;
            grid-template-columns: repeat(7, 1fr);
            gap: 0.375rem;
            margin-bottom: 0.75rem;
        }

        .calendar-day-header {
            text-align: center;
            font-size: 0.75rem;
            font-weight: 600;
            color: var(--text-secondary);
            text-transform: uppercase;
            padding: 0.5rem;
        }

        .calendar-day {
            aspect-ratio: 1;
            border: 2px solid var(--border);
            border-radius: var(--radius);
            background: var(--bg-panel);
            cursor: pointer;
            transition: all 0.2s;
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            position: relative;
            padding: 0.375rem;
            min-height: 45px;
        }

        .calendar-day .capacity-tooltip {
            position: absolute;
            bottom: 100%;
            left: 50%;
            transform: translateX(-50%) translateY(-8px);
            background: var(--text-main);
            color: white;
            padding: 0.5rem 0.75rem;
            border-radius: var(--radius);
            font-size: 0.75rem;
            white-space: nowrap;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
            pointer-events: none;
            z-index: 1000;
            box-shadow: var(--shadow-md);
            margin-bottom: 0.5rem;
        }

        .calendar-day .capacity-tooltip::after {
            content: '';
            position: absolute;
            top: 100%;
            left: 50%;
            transform: translateX(-50%);
            border: 6px solid transparent;
            border-top-color: var(--text-main);
        }

        .calendar-day.available:hover .capacity-tooltip {
            opacity: 1;
            visibility: visible;
            transform: translateX(-50%) translateY(-12px);
        }

        .calendar-day:hover:not(.disabled):not(.selected) {
            border-color: var(--primary);
            background: var(--bg-hover);
        }

        .calendar-day.disabled {
            opacity: 0.3;
            cursor: not-allowed;
            background: var(--border-light);
        }

        .calendar-day.no-capacity {
            opacity: 0.4;
            cursor: not-allowed;
            background: var(--border-light);
            border-color: var(--border);
        }

        .calendar-day.no-capacity:hover {
            background: var(--border-light);
            border-color: var(--border);
        }

        .calendar-day.today {
            border-color: var(--info);
            background: rgba(6, 182, 212, 0.1);
        }

        .calendar-day.selected {
            border-color: var(--primary);
            background: var(--primary);
            color: white;
        }

        .calendar-day-number {
            font-size: 0.875rem;
            font-weight: 600;
            margin-bottom: 0.25rem;
        }

        .calendar-day-slots {
            font-size: 0.625rem;
            font-weight: 500;
            opacity: 0.8;
        }

        .selected-time {
            font-size: 0.75rem;
            font-weight: 700;
            color: white;
            margin-top: 0.25rem;
            padding: 0.125rem 0.375rem;
            background: rgba(81, 246, 59, 0.51);
            border-radius: var(--radius);
            border: 1px solid var(--primary);
        }

        .slot-list-container {
            position: absolute;
            width: 320px;
            max-height: 400px;
            overflow-y: auto;
            padding: 1rem;
            background: var(--bg-panel);
            border: 2px solid var(--primary);
            border-radius: var(--radius-lg);
            box-shadow: var(--shadow-lg);
            z-index: 1000;
            opacity: 0;
            visibility: hidden;
            transform: translateX(-10px);
            transition: all 0.3s ease;
        }

        .slot-list-container.active {
            opacity: 1;
            visibility: visible;
            transform: translateX(0);
        }

        .slot-calendar-wrapper {
            position: relative;
        }

        /* Fallback für Mobile - dann unter dem Kalender */
        @media (max-width: 768px) {
            .slot-list-container {
                position: relative;
                top: auto;
                left: auto !important;
                right: auto;
                width: 100%;
                margin-top: 1rem;
                max-height: 280px;
            }
        }

        .slot-date-header {
            font-weight: 600;
            font-size: 0.875rem;
            color: var(--text-main);
            margin-bottom: 0.75rem;
            padding-bottom: 0.5rem;
            border-bottom: 2px solid var(--border);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .slot-date-header .close-slots {
            background: none;
            border: none;
            color: var(--text-secondary);
            font-size: 1.25rem;
            cursor: pointer;
            padding: 0.25rem;
            width: 24px;
            height: 24px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: var(--radius);
            transition: all 0.2s;
        }

        .slot-date-header .close-slots:hover {
            background: var(--bg-hover);
            color: var(--danger);
        }

        .slots-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(120px, 1fr));
            gap: 0.75rem;
        }

        .slot-item {
            padding: 0.75rem;
            border: 2px solid var(--border);
            border-radius: var(--radius);
            background: var(--bg-panel);
            cursor: pointer;
            transition: all 0.2s;
            text-align: center;
        }

        .slot-item:hover:not(.disabled):not(.selected) {
            border-color: var(--primary);
            transform: translateY(-2px);
            box-shadow: var(--shadow);
        }

        .slot-item.disabled {
            opacity: 0.4;
            cursor: not-allowed;
            background: var(--border-light);
        }

        .slot-item.selected {
            border-color: var(--primary);
            background: var(--primary);
            color: white;
        }

        .slot-time {
            font-weight: 600;
            font-size: 0.875rem;
            margin-bottom: 0.25rem;
        }

        .slot-capacity {
            font-size: 0.75rem;
            opacity: 0.8;
        }

        .slot-item.selected .slot-capacity {
            opacity: 1;
        }

        .no-slots-message {
            text-align: center;
            padding: 2rem;
            color: var(--text-secondary);
            font-size: 0.875rem;
        }

        /* Quick Actions Bar */
        .quick-actions {
            position: fixed;
            bottom: 2rem;
            right: 2rem;
            display: flex;
            flex-direction: column;
            gap: 0.75rem;
            z-index: 1000;
            transition: right 0.3s ease;
        }

        /* Quick Actions verschieben, wenn Detail-Panel geöffnet ist */
        .quick-actions.panel-open {
            right: calc(420px + 2rem);
        }
        
        /* Quick Actions verschieben, wenn Orders Panel geöffnet ist */
        .quick-actions.orders-panel-open {
            right: calc(800px + 2rem);
        }
        
        /* Beide Panels offen - Detail Panel (420px) + Separator (3px) + Orders Panel (800px) = 1223px */
        .quick-actions.panel-open.orders-panel-open {
            right: calc(1223px + 2rem);
        }

        @media (max-width: 1024px) {
            .detail-panel.open.orders-panel-active {
                right: calc(800px + 3px);
            }
            .quick-actions.panel-open {
                right: calc(360px + 2rem);
            }
            .quick-actions.orders-panel-open {
                right: calc(800px + 2rem);
            }
            .quick-actions.panel-open.orders-panel-open {
                right: calc(1163px + 2rem); /* 360px + 3px + 800px */
            }
        }

        @media (max-width: 768px) {
            .quick-actions.panel-open,
            .quick-actions.orders-panel-open {
                right: 2rem;
            }
        }

        .fab {
            width: 56px;
            height: 56px;
            border-radius: 50%;
            border: none;
            background: var(--primary);
            color: white;
            font-size: 1.25rem;
            cursor: pointer;
            box-shadow: var(--shadow-lg);
            transition: all 0.2s;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .fab:hover {
            transform: scale(1.1);
            box-shadow: var(--shadow-lg);
        }

        .fab.secondary {
            background: var(--secondary);
        }

        .fab.success {
            background: var(--success);
        }

        /* Orders Management Panel (Slide-in von rechts, ähnlich Quick Actions aber anders) */
        .orders-panel {
            position: fixed;
            top: 0;
            right: -800px;
            width: 800px;
            height: 100vh;
            background: var(--bg-panel);
            border-left: 1px solid var(--border);
            box-shadow: -4px 0 20px rgba(0, 0, 0, 0.1);
            z-index: 2000;
            transition: right 0.3s ease;
            display: flex;
            flex-direction: column;
            overflow: hidden;
        }

        .orders-panel.open {
            right: 0;
        }

        .orders-panel-header {
            padding: 1.5rem;
            border-bottom: 1px solid var(--border);
            background: var(--bg-hover);
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-shrink: 0;
        }
        
        .orders-panel-body-wrapper {
            display: flex;
            flex-direction: row;
            flex: 1;
            overflow: hidden;
        }

        .orders-panel-title {
            font-size: 1.125rem;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .orders-panel-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            color: var(--text-secondary);
            cursor: pointer;
            width: 32px;
            height: 32px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: var(--radius);
            transition: all 0.2s;
        }

        .orders-panel-close:hover {
            background: var(--bg-hover);
            color: var(--text-main);
        }

        /* Orders Panel Sidebar */
        .orders-panel-sidebar {
            width: 200px;
            background: var(--bg-sidebar);
            border-right: 1px solid var(--border);
            display: flex;
            flex-direction: column;
            overflow-y: auto;
        }
        
        .orders-sidebar-header {
            padding: 1rem;
            border-bottom: 1px solid var(--border);
            background: var(--bg-hover);
        }
        
        .orders-sidebar-title {
            font-size: 0.875rem;
            font-weight: 600;
            color: var(--text-secondary);
            text-transform: uppercase;
            letter-spacing: 0.5px;
            margin-bottom: 0.5rem;
        }
        
        .orders-filter-toggle {
            width: 100%;
            padding: 0.5rem;
            background: var(--bg-panel);
            border: 1px solid var(--border);
            color: var(--text-secondary);
            cursor: pointer;
            font-size: 0.75rem;
            border-radius: var(--radius);
            transition: all 0.2s;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
        }
        
        .orders-filter-toggle:hover {
            background: var(--bg-hover);
            color: var(--primary);
            border-color: var(--primary);
        }
        
        .orders-sidebar-nav {
            flex: 1;
            padding: 0.5rem;
        }
        
        .orders-sidebar-item {
            width: 100%;
            padding: 0.75rem 1rem;
            border: none;
            background: none;
            border-left: 3px solid transparent;
            color: var(--text-secondary);
            font-size: 0.875rem;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.2s;
            position: relative;
            text-align: left;
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 0.25rem;
            border-radius: var(--radius);
        }

        .orders-sidebar-item:hover {
            background: var(--bg-hover);
            color: var(--text-main);
        }

        .orders-sidebar-item.active {
            color: var(--primary);
            border-left-color: var(--primary);
            background: rgba(59, 130, 246, 0.1);
        }
        
        .orders-sidebar-item.placeholder {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        .orders-sidebar-item.placeholder:hover {
            background: none;
            color: var(--text-secondary);
        }
        
        .orders-sidebar-item:disabled {
            pointer-events: none;
        }

        .orders-sidebar-badge {
            background: var(--danger);
            color: white;
            font-size: 0.625rem;
            font-weight: 600;
            padding: 0.125rem 0.375rem;
            border-radius: 9999px;
            min-width: 18px;
            text-align: center;
        }

        .orders-panel-content {
            flex: 1;
            display: flex;
            flex-direction: column;
            overflow: hidden;
        }
        
        .orders-panel-filters {
            border-bottom: 1px solid var(--border);
            background: var(--bg-panel);
            padding: 1rem;
            display: none;
        }
        
        .orders-panel-filters.active {
            display: block;
        }

        .orders-panel-body {
            flex: 1;
            overflow-y: auto;
            padding: 1rem;
        }

        .order-item {
            background: var(--bg-panel);
            border: 1px solid var(--border);
            border-radius: var(--radius);
            padding: 1rem;
            margin-bottom: 0.75rem;
            transition: all 0.2s;
        }

        .order-item:hover {
            box-shadow: var(--shadow-sm);
            border-color: var(--primary);
        }

        .order-item-header {
            display: flex;
            justify-content: space-between;
            align-items: start;
            margin-bottom: 0.75rem;
        }

        .order-item-title {
            font-weight: 600;
            font-size: 0.875rem;
            color: var(--text-main);
        }

        .order-status-code {
            padding: 0.25rem 0.5rem;
            border-radius: 9999px;
            font-size: 0.6875rem;
            font-weight: 500;
        }

        .order-status-code.success {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success);
        }

        .order-status-code.warning {
            background: rgba(245, 158, 11, 0.1);
            color: var(--warning);
        }

        .order-status-code.error {
            background: rgba(239, 68, 68, 0.1);
            color: var(--danger);
        }

        .order-item-details {
            display: grid;
            gap: 0.5rem;
            font-size: 0.8125rem;
            margin-bottom: 0.75rem;
        }

        .order-detail-row {
            display: flex;
            justify-content: space-between;
        }

        .order-detail-label {
            color: var(--text-secondary);
        }

        .order-detail-value {
            color: var(--text-main);
            font-weight: 500;
        }

        .order-item-actions {
            display: flex;
            gap: 0.5rem;
            padding-top: 0.75rem;
            border-top: 1px solid var(--border-light);
        }

        .order-action-btn {
            flex: 1;
            padding: 0.5rem;
            border: 1px solid var(--border);
            background: var(--bg-panel);
            border-radius: var(--radius);
            cursor: pointer;
            font-size: 0.75rem;
            transition: all 0.2s;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.25rem;
        }

        .order-action-btn:hover {
            background: var(--bg-hover);
        }

        .order-action-btn.danger {
            border-color: var(--danger);
            color: var(--danger);
        }

        .order-action-btn.danger:hover {
            background: rgba(239, 68, 68, 0.1);
        }

        .order-action-btn.primary {
            border-color: var(--primary);
            color: var(--primary);
        }

        .order-action-btn.primary:hover {
            background: rgba(59, 130, 246, 0.1);
        }

        /* Alerts */
        #alertContainer {
            position: fixed;
            top: 1rem;
            right: 1rem;
            z-index: 999999;
            pointer-events: none;
        }
        
        .alert {
            position: relative;
            padding: 1rem 1.5rem;
            border-radius: var(--radius-lg);
            box-shadow: var(--shadow-lg);
            z-index: 999999;
            max-width: 400px;
            animation: slideIn 0.3s ease;
            pointer-events: auto;
        }

        @keyframes slideIn {
            from {
                transform: translateX(100%);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }

        .alert-success { background: var(--success); color: white; }
        .alert-warning { background: var(--warning); color: white; }
        .alert-danger { background: var(--danger); color: white; }
        .alert-info { background: var(--info); color: white; }

        /* Recommendation Badges */
        .recommendation-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 9999px;
            font-size: 0.75rem;
            font-weight: 500;
        }

        .recommendation-badge.ok {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success);
        }

        .recommendation-badge.check {
            background: rgba(245, 158, 11, 0.1);
            color: var(--warning);
        }

        .recommendation-badge.replace {
            background: rgba(239, 68, 68, 0.1);
            color: var(--danger);
        }

        /* Modern Loading Wrapper */
        .loading-wrapper {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(15, 23, 42, 0.75);
            backdrop-filter: blur(8px);
            z-index: 99999;
            display: none;
            align-items: center;
            justify-content: center;
            opacity: 0;
            transition: opacity 0.3s ease, visibility 0.3s ease;
            visibility: hidden;
        }

        .loading-wrapper.active {
            display: flex;
            opacity: 1;
            visibility: visible;
        }

        .loading-content {
            background: var(--bg-panel);
            border-radius: var(--radius-lg);
            padding: 2.5rem;
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.3), 0 10px 10px -5px rgba(0, 0, 0, 0.2);
            min-width: 280px;
            max-width: 400px;
            text-align: center;
            animation: fadeInScale 0.3s ease;
        }

        @keyframes fadeInScale {
            from {
                opacity: 0;
                transform: scale(0.9);
            }
            to {
                opacity: 1;
                transform: scale(1);
            }
        }

        .loading-spinner {
            width: 64px;
            height: 64px;
            margin: 0 auto 1.5rem;
            position: relative;
        }

        .loading-spinner-inner {
            width: 100%;
            height: 100%;
            border: 4px solid var(--border-light);
            border-top-color: var(--primary);
            border-right-color: var(--primary);
            border-radius: 50%;
            animation: spin 0.8s linear infinite;
        }

        .loading-spinner-pulse {
            position: absolute;
            top: -4px;
            left: -4px;
            width: 100%;
            height: 100%;
            border: 4px solid var(--primary);
            border-radius: 50%;
            opacity: 0.3;
            animation: pulse 1.5s ease-in-out infinite;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }

        @keyframes pulse {
            0%, 100% {
                transform: scale(1);
                opacity: 0.3;
            }
            50% {
                transform: scale(1.2);
                opacity: 0;
            }
        }

        .loading-text {
            font-size: 1rem;
            font-weight: 500;
            color: var(--text-main);
            margin-bottom: 0.5rem;
        }

        .loading-subtext {
            font-size: 0.875rem;
            color: var(--text-secondary);
            margin-top: 0.5rem;
        }

        .loading-progress {
            width: 100%;
            height: 6px;
            background: var(--border-light);
            border-radius: 9999px;
            margin-top: 1.5rem;
            overflow: hidden;
        }

        .loading-progress-bar {
            height: 100%;
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            border-radius: 9999px;
            animation: progress 1.5s ease-in-out infinite;
        }

        @keyframes progress {
            0% {
                width: 0%;
                transform: translateX(0);
            }
            50% {
                width: 70%;
                transform: translateX(0);
            }
            100% {
                width: 100%;
                transform: translateX(0);
            }
        }

        /* Loading Dots Animation */
        .loading-dots {
            display: inline-flex;
            gap: 0.5rem;
            margin-top: 1rem;
        }

        .loading-dot {
            width: 10px;
            height: 10px;
            background: var(--primary);
            border-radius: 50%;
            animation: dotBounce 1.4s ease-in-out infinite;
        }

        .loading-dot:nth-child(1) {
            animation-delay: 0s;
        }

        .loading-dot:nth-child(2) {
            animation-delay: 0.2s;
        }

        .loading-dot:nth-child(3) {
            animation-delay: 0.4s;
        }

        @keyframes dotBounce {
            0%, 80%, 100% {
                transform: scale(0.8);
                opacity: 0.5;
            }
            40% {
                transform: scale(1.2);
                opacity: 1;
            }
        }

        /* Compact Loading (für kleine Bereiche) */
        .loading-compact {
            display: inline-flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem 1rem;
            background: var(--bg-panel);
            border-radius: var(--radius);
            box-shadow: var(--shadow-md);
        }

        .loading-compact-spinner {
            width: 20px;
            height: 20px;
            border: 2px solid var(--border-light);
            border-top-color: var(--primary);
            border-radius: 50%;
            animation: spin 0.8s linear infinite;
        }

        .loading-compact-text {
            font-size: 0.875rem;
            color: var(--text-main);
        }

        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 4rem 2rem;
            color: var(--text-secondary);
        }

        .empty-icon {
            font-size: 4rem;
            opacity: 0.3;
            margin-bottom: 1rem;
        }

        /* Modal Styles */
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            backdrop-filter: blur(4px);
            z-index: 10000;
            display: none;
            align-items: center;
            justify-content: center;
            padding: 2rem;
        }

        .modal-overlay.active {
            display: flex;
        }

        .modal {
            background: var(--bg-panel);
            border-radius: var(--radius-lg);
            box-shadow: var(--shadow-lg);
            max-width: 1500px;
            width: 95vw;
            max-height: 90vh;
            display: flex;
            flex-direction: column;
            overflow: hidden;
        }
        
        /* Spezifisches Styling für Wheel Details Modal - Breit und Leserlich */
        #wheelDetailsModal .modal {
            max-width: 98vw;
            width: 98vw;
            max-height: 90vh;
        }
        
        #wheelDetailsModal .modal-body {
            padding: 1.5rem;
            overflow-y: visible;
            max-height: calc(90vh - 100px);
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .modal-header {
            padding: 1.5rem;
            border-bottom: 1px solid var(--border);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .modal-title {
            font-size: 1.25rem;
            font-weight: 600;
        }

        .modal-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            color: var(--text-secondary);
            cursor: pointer;
            width: 32px;
            height: 32px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: var(--radius);
        }

        .modal-close:hover {
            background: var(--bg-hover);
        }

        .modal-body {
            padding: 1.5rem;
            overflow-y: auto;
            flex: 1;
        }

        .modal-footer {
            padding: 1.5rem;
            border-top: 1px solid var(--border);
            display: flex;
            gap: 0.75rem;
            justify-content: flex-end;
        }

        .form-group {
            margin-bottom: 1.5rem;
        }

        .modal-body .form-group:last-child {
            margin-bottom: 0;
        }

        .form-label {
            display: block;
            font-size: 0.875rem;
            font-weight: 500;
            color: var(--text-main);
            margin-bottom: 0.5rem;
        }

        .form-control {
            width: 100%;
            padding: 0.625rem 0.75rem;
            border: 1px solid var(--border);
            border-radius: var(--radius);
            background: var(--bg-panel);
            color: var(--text-main);
            font-size: 0.875rem;
        }

        .form-control:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }

        /* Neues Design für Select-Felder - Karten-Layout */
        .select-cards-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
            gap: 1rem;
            margin-top: 0.75rem;
        }

        .select-card {
            border: 2px solid var(--border);
            border-radius: var(--radius-lg);
            padding: 1rem;
            background: var(--bg-panel);
            cursor: pointer;
            transition: all 0.3s ease;
            position: relative;
            display: flex;
            flex-direction: column;
            align-items: center;
            text-align: center;
            min-height: 120px;
            justify-content: center;
        }

        .select-card:hover {
            border-color: var(--primary);
            background: rgba(59, 130, 246, 0.05);
            transform: translateY(-2px);
            box-shadow: var(--shadow-md);
        }

        .select-card.selected {
            border-color: var(--primary);
            background: rgba(59, 130, 246, 0.1);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.2);
        }

        .select-card-icon {
            font-size: 2rem;
            color: var(--primary);
            margin-bottom: 0.5rem;
            transition: transform 0.3s ease;
        }

        .select-card:hover .select-card-icon {
            transform: scale(1.1);
        }

        .select-card-title {
            font-weight: 600;
            font-size: 0.875rem;
            color: var(--text-main);
            margin-bottom: 0.25rem;
        }

        .select-card-description {
            font-size: 0.75rem;
            color: var(--text-secondary);
        }

        .select-card-check {
            position: absolute;
            top: 0.5rem;
            right: 0.5rem;
            width: 24px;
            height: 24px;
            background: var(--primary);
            color: white;
            border-radius: 50%;
            display: none;
            align-items: center;
            justify-content: center;
            font-size: 0.75rem;
        }

        .select-card.selected .select-card-check {
            display: flex;
        }

        .btn {
            padding: 0.625rem 1.25rem;
            border-radius: var(--radius);
            border: none;
            font-size: 0.875rem;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.2s;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--primary-dark);
        }

        .btn-secondary {
            background: var(--secondary);
            color: white;
        }

        .btn-secondary:hover {
            background: #475569;
        }

        /* Responsive */
        @media (max-width: 1024px) {
            .filter-sidebar {
                width: 280px;
            }
            
            .detail-panel.open {
                width: 360px;
            }
        }

        @media (max-width: 768px) {
            .storage-layout {
                flex-direction: column;
            }
            
            .filter-sidebar {
                width: 100%;
                max-height: 300px;
            }
            
            .detail-panel.open {
                position: fixed;
                right: 0;
                top: 0;
                height: 100vh;
                width: 100%;
                z-index: 1000;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <?php include 'includes/sidebar.php'; ?>

        <div class="admin-main">
            <div class="storage-layout">
                <!-- Left Filter Sidebar -->
                <div class="filter-sidebar">
                    <div class="sidebar-header">
                        <div class="sidebar-title">
                            <i class="fas fa-filter"></i>
                            Filter & Suche
                        </div>
                        <select id="autohausSelect" class="autohaus-select">
                            <option value="">-- Autohaus wählen --</option>
                            <?php foreach ($autohauser as $autohaus): ?>
                                <option value="<?= htmlspecialchars($autohaus['id']) ?>">
                                    <?= htmlspecialchars($autohaus['name']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="filter-section">
                        <div class="filter-title">Suche</div>
                        <div class="filter-group">
                            <input type="text" id="searchInput" class="filter-input" placeholder="Satznummer, Kennzeichen, FIN...">
                        </div>
                    </div>

                    <div class="filter-section">
                        <div class="filter-title">Status</div>
                        <div class="filter-group">
                            <select id="statusFilter" class="filter-select">
                                <option value="">Alle</option>
                                <option value="eingelagert">Eingelagert</option>
                                <option value="zurück">Zurück geliefert</option>
                            </select>
                        </div>
                        <div class="filter-group">
                            <label class="filter-label">Empfehlung</label>
                            <select id="recommendationFilter" class="filter-select">
                                <option value="">Alle</option>
                                <option value="In Ordnung">✅ In Ordnung</option>
                                <option value="Prüfung">⚠️ Prüfung empfohlen</option>
                                <option value="Austausch">🔴 Austausch notwendig</option>
                            </select>
                        </div>
                    </div>

                    <div class="filter-section">
                        <div class="filter-title">Statistiken</div>
                        <div class="stats-compact">
                            <div class="stat-mini">
                                <div class="stat-mini-value" id="statTotal">0</div>
                                <div class="stat-mini-label">Gesamt</div>
                            </div>
                            <div class="stat-mini">
                                <div class="stat-mini-value" id="statStored">0</div>
                                <div class="stat-mini-label">Eingelagert</div>
                            </div>
                            <div class="stat-mini">
                                <div class="stat-mini-value" id="statReturned">0</div>
                                <div class="stat-mini-label">Zurück</div>
                            </div>
                            <div class="stat-mini">
                                <div class="stat-mini-value" id="statRecommendation">0</div>
                                <div class="stat-mini-label">Warnung</div>
                            </div>
                        </div>
                    </div>

                    <div class="filter-section">
                        <button id="loadBtn" class="filter-input" style="background: var(--primary); color: white; border: none; cursor: pointer; font-weight: 600;" onclick="loadWheelsets()">
                            <i class="fas fa-sync-alt"></i> Daten laden
                        </button>
                    </div>
                </div>

                <!-- Main Content Area -->
                <div class="content-area">
                    <div class="content-header">
                        <div>
                            <h2 style="margin: 0; font-size: 1.25rem; font-weight: 600;">Reifensätze</h2>
                            <p style="margin: 0.25rem 0 0 0; font-size: 0.875rem; color: var(--text-secondary);" id="resultsCount">0 Ergebnisse</p>
                        </div>
                        <div class="view-controls">
                            <button class="view-btn active" onclick="setView('grid')" data-view="grid">
                                <i class="fas fa-th"></i> Grid
                            </button>
                            <button class="view-btn" onclick="setView('list')" data-view="list">
                                <i class="fas fa-list"></i> Liste
                            </button>
                        </div>
                    </div>

                    <div class="content-body" id="contentBody">
                        <div class="empty-state">
                            <div class="empty-icon">
                                <i class="fas fa-warehouse"></i>
                            </div>
                            <h3>Keine Daten geladen</h3>
                            <p>Wählen Sie ein Autohaus aus und klicken Sie auf "Daten laden"</p>
                        </div>
                    </div>
                </div>

                <!-- Right Detail Panel -->
                <div class="detail-panel" id="detailPanel">
                    <div class="panel-header">
                        <div class="panel-header-left">
                            <h3 style="margin: 0; font-size: 1.125rem; font-weight: 600;">Details</h3>
                            <div class="panel-header-icons" id="panelHeaderIcons">
                                <!-- Icons werden dynamisch durch JavaScript eingefügt -->
                            </div>
                        </div>
                        <button class="panel-close" onclick="closeDetailPanel()">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                    <div class="panel-body" id="panelBody">
                        <p style="color: var(--text-secondary);">Wählen Sie einen Reifensatz aus</p>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Panel Separator (erscheint zwischen Detail und Orders Panel) -->
        <div class="panel-separator" id="panelSeparator"></div>
    </div>

    <!-- Quick Actions FAB -->
    <div class="quick-actions">
        <button class="fab" onclick="showPickupModal()" title="Abholung bestellen">
            <i class="fas fa-truck"></i>
        </button>
        <button class="fab secondary" onclick="showReturnModal()" title="Rücklieferung bestellen">
            <i class="fas fa-arrow-left"></i>
        </button>
        <button class="fab success" onclick="showOrdersPanel()" title="Bestellungen verwalten" id="ordersPanelBtn" style="position: relative;">
            <i class="fas fa-list-alt"></i>
            <span id="ordersPanelBadge" style="position: absolute; top: -4px; right: -4px; background: var(--danger); color: white; font-size: 0.625rem; padding: 0.125rem 0.375rem; border-radius: 9999px; min-width: 18px; text-align: center; display: none;">0</span>
        </button>
        <button class="fab" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); position: relative;" onclick="showOffersNotificationsPanel()" title="Angebotsbenachrichtigungen" id="offersNotificationsBtn">
            <i class="fas fa-file-invoice-dollar"></i>
            <span id="offersNotificationsBadge" style="position: absolute; top: -4px; right: -4px; background: var(--danger); color: white; font-size: 0.625rem; padding: 0.125rem 0.375rem; border-radius: 9999px; min-width: 18px; text-align: center; display: none;">0</span>
        </button>
        <button class="fab" style="background: var(--info);" onclick="showToolsModal()" title="Tools & Validierung">
            <i class="fas fa-tools"></i>
        </button>
    </div>

    <!-- Orders Management Panel -->
    <div class="orders-panel" id="ordersPanel">
        <div class="orders-panel-header">
            <div class="orders-panel-title">
                <i class="fas fa-list-alt"></i>
                Bestellungen
            </div>
            <button class="orders-panel-close" onclick="closeOrdersPanel()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="orders-panel-body-wrapper">
            <div class="orders-panel-sidebar">
                <div class="orders-sidebar-header">
                    <div class="orders-sidebar-title">Navigation</div>
                    <button class="orders-filter-toggle" onclick="toggleOrdersFilters()" title="Filter anzeigen/ausblenden" id="ordersFilterToggle">
                        <i class="fas fa-filter"></i>
                        <span>Filter</span>
                    </button>
                </div>
                <div class="orders-sidebar-nav">
                <button class="orders-sidebar-item active" onclick="switchOrdersTab('pickup')" data-tab="pickup" id="ordersTabPickup">
                    <span>
                        <i class="fas fa-truck" style="margin-right: 0.5rem;"></i>
                        Abholung
                    </span>
                    <span class="orders-sidebar-badge" id="pickupTabBadge" style="display: none;">0</span>
                </button>
                <button class="orders-sidebar-item" onclick="switchOrdersTab('return')" data-tab="return" id="ordersTabReturn">
                    <span>
                        <i class="fas fa-arrow-left" style="margin-right: 0.5rem;"></i>
                        Rücklieferung
                    </span>
                    <span class="orders-sidebar-badge" id="returnTabBadge" style="display: none;">0</span>
                </button>
                <button class="orders-sidebar-item" onclick="switchOrdersTab('extra')" data-tab="extra" id="ordersTabExtra">
                    <span>
                        <i class="fas fa-shopping-cart" style="margin-right: 0.5rem;"></i>
                        Extra Orders
                    </span>
                    <span class="orders-sidebar-badge" id="extraTabBadge" style="display: none;">0</span>
                </button>
                <button class="orders-sidebar-item placeholder" data-tab="inventory" id="ordersTabInventory" disabled style="opacity: 0.5; cursor: not-allowed;" title="Coming Soon">
                    <span>
                        <i class="fas fa-boxes" style="margin-right: 0.5rem;"></i>
                        Inventar
                    </span>
                </button>
                <button class="orders-sidebar-item" onclick="switchOrdersTab('customers')" data-tab="customers" id="ordersTabCustomers">
                    <span>
                        <i class="fas fa-users" style="margin-right: 0.5rem;"></i>
                        Kunden
                    </span>
                </button>
                <button class="orders-sidebar-item placeholder" data-tab="global_inventory" id="ordersTabGlobalInventory" disabled style="opacity: 0.5; cursor: not-allowed;" title="Coming Soon">
                    <span>
                        <i class="fas fa-globe" style="margin-right: 0.5rem;"></i>
                        Globales Inventar
                    </span>
                </button>
                </div>
            </div>
            <div class="orders-panel-content">
            <div class="orders-panel-filters" id="ordersPanelFilters">
            <div style="padding: 1rem; border-bottom: 1px solid var(--border); background: var(--bg-hover);">
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 0.75rem; margin-bottom: 0.75rem;">
                    <div>
                        <label style="display: block; font-size: 0.75rem; font-weight: 600; color: var(--text-secondary); margin-bottom: 0.25rem;">Datum</label>
                        <input type="date" id="ordersDateFilter" class="filter-input" style="width: 100%; padding: 0.5rem; font-size: 0.875rem;" onchange="applyOrdersFilters()">
                    </div>
                    <div>
                        <label style="display: block; font-size: 0.75rem; font-weight: 600; color: var(--text-secondary); margin-bottom: 0.25rem;">Kunde</label>
                        <select id="ordersCustomerFilter" class="filter-select" style="width: 100%; padding: 0.5rem; font-size: 0.875rem;" onchange="applyOrdersFilters()">
                            <option value="">Alle</option>
                        </select>
                    </div>
                </div>
                <button onclick="clearOrdersFilters()" style="width: 100%; padding: 0.5rem; font-size: 0.875rem; background: var(--bg-hover); border: 1px solid var(--border); border-radius: var(--radius); color: var(--text-secondary); cursor: pointer; transition: all 0.2s;">
                    <i class="fas fa-times"></i> Filter zurücksetzen
                </button>
            </div>
            </div>
            <div class="orders-panel-body" id="ordersPanelBody">
            <!-- Wird dynamisch gefüllt -->
        </div>
    </div>

    <!-- Angebotsbenachrichtigungen Panel -->
    <div class="orders-panel" id="offersNotificationsPanel">
        <div class="orders-panel-header">
            <div class="orders-panel-title">
                <i class="fas fa-file-invoice-dollar"></i>
                Angebotsbenachrichtigungen
            </div>
            <button class="orders-panel-close" onclick="closeOffersNotificationsPanel()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="orders-panel-body-wrapper">
            <div class="orders-panel-sidebar">
                <div class="orders-sidebar-header">
                    <div class="orders-sidebar-title">Status</div>
                </div>
                <div class="orders-sidebar-nav">
                    <button class="orders-sidebar-item active" onclick="switchOffersNotificationsTab('accepted')" data-tab="accepted" id="offersTabAccepted">
                        <span>
                            <i class="fas fa-check-circle" style="margin-right: 0.5rem; color: #28a745;"></i>
                            Angenommen
                        </span>
                        <span class="orders-sidebar-badge" id="acceptedOffersBadge" style="display: none;">0</span>
                    </button>
                    <button class="orders-sidebar-item" onclick="switchOffersNotificationsTab('declined')" data-tab="declined" id="offersTabDeclined">
                        <span>
                            <i class="fas fa-times-circle" style="margin-right: 0.5rem; color: #dc3545;"></i>
                            Abgelehnt
                        </span>
                        <span class="orders-sidebar-badge" id="declinedOffersBadge" style="display: none;">0</span>
                    </button>
                    <button class="orders-sidebar-item" onclick="switchOffersNotificationsTab('new_request')" data-tab="new_request" id="offersTabNewRequest">
                        <span>
                            <i class="fas fa-redo" style="margin-right: 0.5rem; color: #17a2b8;"></i>
                            Neu angefragt
                        </span>
                        <span class="orders-sidebar-badge" id="newRequestOffersBadge" style="display: none;">0</span>
                    </button>
                    <button class="orders-sidebar-item" onclick="switchOffersNotificationsTab('pending')" data-tab="pending" id="offersTabPending">
                        <span>
                            <i class="fas fa-clock" style="margin-right: 0.5rem; color: #f59e0b;"></i>
                            Noch Offen
                        </span>
                        <span class="orders-sidebar-badge" id="pendingOffersBadge" style="display: none;">0</span>
                    </button>
                    <button class="orders-sidebar-item" onclick="switchOffersNotificationsTab('all')" data-tab="all" id="offersTabAll">
                        <span>
                            <i class="fas fa-list" style="margin-right: 0.5rem;"></i>
                            Alle
                        </span>
                        <span class="orders-sidebar-badge" id="offersTabAllBadge" style="display: none;">0</span>
                    </button>
                </div>
            </div>
            <div class="orders-panel-content">
                <div style="padding: 0.75rem 1rem; border-bottom: 1px solid var(--border); background: var(--bg-hover); display: flex; justify-content: space-between; align-items: center;">
                    <div style="font-size: 0.875rem; color: var(--text-secondary);">
                        <i class="fas fa-clock"></i> Angebotsbenachrichtigungen
                    </div>
                    <div style="font-size: 0.75rem; color: var(--text-secondary);" id="offersLastUpdateTime">
                        <i class="fas fa-sync"></i> Wird geladen...
                    </div>
                </div>
                <div class="orders-panel-body" id="offersNotificationsBody">
                    <!-- Wird dynamisch gefüllt -->
                </div>
            </div>
        </div>
    </div>

    <!-- Modern Loading Wrapper -->
    <div class="loading-wrapper" id="loadingOverlay">
        <div class="loading-content">
            <div class="loading-spinner">
                <div class="loading-spinner-pulse"></div>
                <div class="loading-spinner-inner"></div>
            </div>
            <div class="loading-text" id="loadingText">Wird geladen...</div>
            <div class="loading-subtext" id="loadingSubtext">Bitte warten Sie einen Moment</div>
            <div class="loading-progress">
                <div class="loading-progress-bar"></div>
            </div>
            <div class="loading-dots">
                <div class="loading-dot"></div>
                <div class="loading-dot"></div>
                <div class="loading-dot"></div>
            </div>
        </div>
    </div>

    <!-- Alert Container -->
    <div id="alertContainer"></div>

    <!-- Modal Container -->
    <div id="modalContainer"></div>

    <!-- Prüfbericht Viewer -->
    <div class="test-report-viewer" id="testReportViewer">
        <div class="test-report-header">
            <div class="test-report-title" id="testReportTitle">Prüfbericht anzeigen</div>
            <button class="test-report-close" onclick="closeTestReport()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="test-report-body">
            <div class="test-report-content">
                <iframe id="testReportIframe" class="test-report-iframe" frameborder="0"></iframe>
            </div>
        </div>
        <div class="test-report-actions">
            <button class="btn btn-secondary btn-sm" onclick="closeTestReport()">
                <i class="fas fa-times"></i> Schließen
            </button>
            <a id="testReportDownload" href="#" target="_blank" class="btn btn-primary btn-sm" style="text-decoration: none;">
                <i class="fas fa-download"></i> Herunterladen
            </a>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://code.jquery.com/jquery-3.7.0.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
    <script src="../assets/js/main.js"></script>
    <script src="assets/js/admin.js"></script>
    <script src="assets/js/unified-sidebar.js"></script>

    <script>
        // Global State
        let wheelsetsData = [];
        let filteredData = [];
        let currentView = 'grid';
        let selectedWheels = []; // Globale Variable für ausgewählte Reifen
        let selectedWheelsets = new Set();
        let pickupOrders = [];
        let returnOrders = [];
        let extraOrders = [];
        let inventoryData = [];

        // Initialize
        // Prüfe URL-Parameter beim Laden der Seite (für Teams-Buttons)
        function checkUrlParameters() {
            const urlParams = new URLSearchParams(window.location.search);
            const wheelset = urlParams.get('wheelset');
            const autohausId = urlParams.get('autohaus_id');
            const offerId = urlParams.get('offer_id');
            
            if (wheelset && autohausId) {
                // Setze Autohaus-Auswahl
                if (autohausId && $('#autohausSelect').val() !== autohausId) {
                    $('#autohausSelect').val(autohausId).trigger('change');
                }
                
                // Warte kurz bis Autohaus geladen ist
                setTimeout(() => {
                    // Öffne Wheelset-Details
                    viewDetails(wheelset);
                    
                    // Wenn offerId vorhanden, öffne auch das Angebot
                    if (offerId) {
                        setTimeout(() => {
                            openOfferModal(wheelset);
                            setTimeout(() => {
                                viewOfferDetails(parseInt(offerId), wheelset, null);
                            }, 500);
                        }, 500);
                    }
                    
                    // Entferne URL-Parameter aus Adressleiste
                    window.history.replaceState({}, document.title, window.location.pathname);
                }, 500);
            }
        }
        
        $(document).ready(function() {
            // Prüfe URL-Parameter
            checkUrlParameters();
            $('#autohausSelect').change(function() {
                if ($(this).val()) {
                    loadWheelsets();
                }
            });

            // Search
            $('#searchInput').on('input', debounce(applyFilters, 300));
            $('#statusFilter, #recommendationFilter').change(applyFilters);
        });

        function debounce(func, wait) {
            let timeout;
            return function executedFunction(...args) {
                const later = () => {
                    clearTimeout(timeout);
                    func(...args);
                };
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
            };
        }

        function loadWheelsets() {
            const autohausId = $('#autohausSelect').val();
            if (!autohausId) {
                showAlert('Bitte wählen Sie ein Autohaus aus', 'warning');
                return;
            }

            showLoading(true, 'Lade Reifensätze...', 'Bitte warten Sie einen Moment');

            // Lade Wheelsets, Orders und Inventar parallel
            Promise.all([
                fetch(`../api/get_resy_wheelsets.php?autohaus_id=${autohausId}`),
                fetch(`../api/resy_api_routes.php?action=get_pickup_orders&autohaus_id=${autohausId}`),
                fetch(`../api/resy_api_routes.php?action=get_return_orders&autohaus_id=${autohausId}`),
                fetch(`../api/resy_api_routes.php?action=get_all_extra_orders&autohaus_id=${autohausId}`).catch(() => ({ json: () => Promise.resolve({ success: false, extraOrders: [] }) })),
                fetch(`../api/resy_api_routes.php?action=get_inventory&autohaus_id=${autohausId}`).catch(() => ({ json: () => Promise.resolve({ success: false, inventory: [] }) }))
            ])
                .then(async ([wheelsetsResponse, pickupResponse, returnResponse, extraResponse, inventoryResponse]) => {
                    const wheelsetsData_result = await wheelsetsResponse.json();
                    const pickupData = await pickupResponse.json();
                    const returnData = await returnResponse.json();
                    const extraData = await extraResponse.json();
                    const inventoryResult = await inventoryResponse.json();
                    
                    showLoading(false);
                    
                    // Wheelsets
                    if (wheelsetsData_result.success && wheelsetsData_result.wheelsets) {
                        wheelsetsData = wheelsetsData_result.wheelsets;
                    } else {
                        wheelsetsData = [];
                    }
                    
                    // Pickup Orders
                    if (pickupData.success && pickupData.pickupOrders) {
                        pickupOrders = pickupData.pickupOrders;
                        // Erstelle Mapping: wheelsetNumber -> order
                        pickupOrders.forEach(order => {
                            const ws = wheelsetsData.find(w => w.satznummer == order.wheelSetNumber);
                            if (ws) {
                                if (!ws.orders) ws.orders = {};
                                ws.orders.pickup = order;
                            }
                        });
                    } else {
                        pickupOrders = [];
                    }
                    
                    // Return Orders
                    if (returnData.success && returnData.returnOrders) {
                        returnOrders = returnData.returnOrders;
                        // Erstelle Mapping: wheelsetNumber -> order
                        returnOrders.forEach(order => {
                            const ws = wheelsetsData.find(w => w.satznummer == order.wheelSetNumber);
                            if (ws) {
                                if (!ws.orders) ws.orders = {};
                                ws.orders.return = order;
                            }
                        });
                    } else {
                        returnOrders = [];
                    }
                    
                    // Extra Orders
                    if (extraData.success && extraData.extraOrders) {
                        extraOrders = extraData.extraOrders;
                    } else {
                        extraOrders = [];
                    }
                    
                    // Inventar
                    if (inventoryResult.success && inventoryResult.inventory) {
                        inventoryData = inventoryResult.inventory;
                    } else {
                        inventoryData = [];
                    }
                    
                    // Update Orders Panel Badges
                    updateOrdersBadges();
                    updateCustomerFilterOptions();
                    
                    applyFilters();
                    showAlert(`${wheelsetsData.length} Reifensätze geladen`, 'success');
                })
                .catch(error => {
                    showLoading(false);
                    showAlert('Fehler: ' + error.message, 'danger');
                    wheelsetsData = [];
                    pickupOrders = [];
                    returnOrders = [];
                    extraOrders = [];
                    inventoryData = [];
                    renderView();
                });
        }
        
        function viewInventoryDetails(inventoryType) {
            const item = inventoryData.find(i => i.inventoryType === inventoryType);
            if (!item || !item.inventoryNumbers || item.inventoryNumbers.length === 0) {
                showAlert('Keine Details verfügbar', 'warning');
                return;
            }
            
            let detailsHtml = `<div style="max-height: 400px; overflow-y: auto;"><div style="display: grid; gap: 0.5rem;">`;
            item.inventoryNumbers.forEach(num => {
                detailsHtml += `
                    <div style="display: flex; justify-content: space-between; align-items: center; padding: 0.5rem; background: var(--bg-hover); border-radius: var(--radius);">
                        <span>Satz #${num}</span>
                        <button class="btn btn-sm btn-primary" onclick="closeModal('inventoryDetailsModal'); viewDetails(${num})" style="padding: 0.25rem 0.75rem; font-size: 0.75rem;">
                            <i class="fas fa-eye"></i> Details
                        </button>
                    </div>
                `;
            });
            detailsHtml += '</div></div>';
            
            createModal('inventoryDetailsModal', `Inventar: ${inventoryType} (${item.inventoryNumbers.length} Stück)`, detailsHtml);
        }

        function applyFilters() {
            const search = $('#searchInput').val().toLowerCase();
            const status = $('#statusFilter').val().toLowerCase();
            const recommendation = $('#recommendationFilter').val().toLowerCase();

            filteredData = wheelsetsData.filter(ws => {
                const matchSearch = !search || 
                    (ws.satznummer && ws.satznummer.toString().includes(search)) ||
                    (ws.kennzeichen && ws.kennzeichen.toLowerCase().includes(search)) ||
                    (ws.fin && ws.fin.toLowerCase().includes(search)) ||
                    (ws.reifenmodell && ws.reifenmodell.toLowerCase().includes(search));

                const matchStatus = !status || (ws.status && ws.status.toLowerCase().includes(status));
                
                let matchRecommendation = true;
                if (recommendation) {
                    const empfehlung = (ws.empfehlung || '').toLowerCase();
                    if (recommendation === 'in ordnung') {
                        matchRecommendation = !empfehlung.includes('prüfung') && !empfehlung.includes('austausch');
                    } else if (recommendation === 'prüfung') {
                        matchRecommendation = empfehlung.includes('prüfung') || empfehlung.includes('empfohlen');
                    } else if (recommendation === 'austausch') {
                        matchRecommendation = empfehlung.includes('austausch') || empfehlung.includes('notwendig');
                    }
                }

                return matchSearch && matchStatus && matchRecommendation;
            });

            updateStats(filteredData);
            renderView();
        }

        function updateStats(data) {
            const total = data.length;
            const stored = data.filter(w => (w.status || '').toLowerCase().includes('eingelagert')).length;
            const returned = data.filter(w => (w.status || '').toLowerCase().includes('zurück')).length;
            const recommendation = data.filter(w => {
                const empfehlung = (w.empfehlung || '').toLowerCase();
                return empfehlung.includes('austausch') || empfehlung.includes('notwendig');
            }).length;

            $('#statTotal').text(total);
            $('#statStored').text(stored);
            $('#statReturned').text(returned);
            $('#statRecommendation').text(recommendation);
            $('#resultsCount').text(`${total} Ergebnis${total !== 1 ? 'se' : ''}`);
        }

        function setView(view) {
            currentView = view;
            $('.view-btn').removeClass('active');
            $(`.view-btn[data-view="${view}"]`).addClass('active');
            renderView();
        }

        function renderView() {
            const body = $('#contentBody');
            
            if (filteredData.length === 0) {
                body.html(`
                    <div class="empty-state">
                        <div class="empty-icon"><i class="fas fa-search"></i></div>
                        <h3>Keine Ergebnisse</h3>
                        <p>Passen Sie Ihre Filter an oder laden Sie Daten</p>
                    </div>
                `);
                return;
            }

            if (currentView === 'grid') {
                renderGridView();
            } else {
                renderListView();
            }
        }

        function getRecommendationClass(empfehlung) {
            if (!empfehlung) return 'recommendation-ok';
            const e = empfehlung.toLowerCase();
            if (e.includes('austausch') || e.includes('notwendig')) {
                return 'recommendation-replace';
            } else if (e.includes('prüfung') || e.includes('empfohlen')) {
                return 'recommendation-check';
            }
            return 'recommendation-ok';
        }

        function renderGridView() {
            const html = `
                <div class="masonry-grid">
                    ${filteredData.map(ws => {
                        const recClass = getRecommendationClass(ws.empfehlung);
                        const orderStatus = getOrderStatus(ws);
                        const orderStatusClass = getOrderStatusClass(orderStatus);
                        const orderBadge = getOrderStatusBadge(orderStatus);
                        return `
                        <div class="wheelset-item ${recClass} ${orderStatusClass} ${selectedWheelsets.has(ws.satznummer) ? 'selected' : ''}" 
                             onclick="selectWheelset('${ws.satznummer}', event)">
                            <div class="item-header">
                                <div class="item-number" style="display: flex; align-items: center; gap: 0.5rem;">
                                    #${ws.satznummer || '-'}
                                    ${orderBadge}
                                </div>
                                <span class="item-status status-${getStatusClass(ws.status)}">${ws.status || '-'}</span>
                            </div>
                            <div class="item-info">
                                <div class="info-row">
                                    <span class="info-label">Kennzeichen:</span>
                                    <span class="info-value">${ws.kennzeichen || '-'}</span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">FIN:</span>
                                    <span class="info-value">${ws.fin || '-'}</span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">Modell:</span>
                                    <span class="info-value">${ws.reifenmodell || '-'}</span>
                                </div>
                                <div class="info-row">
                                    <span class="info-label">Empfehlung:</span>
                                    <span class="recommendation-badge ${getRecommendationClass(ws.empfehlung)}">${ws.empfehlung || 'In Ordnung'}</span>
                                </div>
                            </div>
                            <div class="item-actions">
                                <button class="action-btn primary" onclick="event.stopPropagation(); viewDetails('${ws.satznummer || ws.id}')">
                                    <i class="fas fa-eye"></i> Details
                                </button>
                                <button class="action-btn success" onclick="event.stopPropagation(); quickPickup('${ws.satznummer || ws.id}')">
                                    <i class="fas fa-truck"></i>
                                </button>
                                <button class="action-btn warning" onclick="event.stopPropagation(); quickReturn('${ws.satznummer || ws.id}')">
                                    <i class="fas fa-arrow-left"></i>
                                </button>
                            </div>
                        </div>
                    `;
                    }).join('')}
                </div>
            `;
            $('#contentBody').html(html);
        }

        function renderListView() {
            const html = `
                <div class="compact-list">
                    ${filteredData.map(ws => {
                        const recClass = getRecommendationClass(ws.empfehlung);
                        const orderStatus = getOrderStatus(ws);
                        const orderStatusClass = getOrderStatusClass(orderStatus);
                        const orderBadge = getOrderStatusBadge(orderStatus);
                        return `
                        <div class="list-item ${recClass} ${orderStatusClass} ${selectedWheelsets.has(ws.satznummer) ? 'selected' : ''}" 
                             onclick="selectWheelset('${ws.satznummer}', event)">
                            <input type="checkbox" class="list-checkbox" ${selectedWheelsets.has(ws.satznummer) ? 'checked' : ''} 
                                   onclick="event.stopPropagation(); toggleSelection('${ws.satznummer}')">
                            <div>
                                <div style="font-weight: 600; margin-bottom: 0.25rem; display: flex; align-items: center; gap: 0.5rem;">
                                    Satz #${ws.satznummer || '-'}
                                    ${orderBadge}
                                </div>
                                <div style="font-size: 0.875rem; color: var(--text-secondary);">${ws.kennzeichen || '-'} | ${ws.fin || '-'}</div>
                            </div>
                            <div style="text-align: right;">
                                <div style="font-size: 0.875rem; color: var(--text-secondary);">${ws.reifenmodell || '-'}</div>
                                <div style="font-size: 0.75rem; color: var(--text-muted);">${ws.status || '-'}</div>
                            </div>
                            <div style="display: flex; gap: 0.5rem;">
                                <button class="action-btn primary" onclick="event.stopPropagation(); viewDetails('${ws.satznummer || ws.id}')">
                                    <i class="fas fa-eye"></i>
                                </button>
                            </div>
                        </div>
                    `;
                    }).join('')}
                </div>
            `;
            $('#contentBody').html(html);
        }

        function getStatusClass(status) {
            if (!status) return 'stored';
            const s = status.toLowerCase();
            if (s.includes('eingelagert')) return 'stored';
            if (s.includes('zurück')) return 'returned';
            return 'warning';
        }
        
        function getOrderStatus(wheelset) {
            const hasPickup = wheelset.orders && wheelset.orders.pickup;
            const hasReturn = wheelset.orders && wheelset.orders.return;
            
            if (hasPickup && hasReturn) {
                return { type: 'both', pickup: wheelset.orders.pickup, return: wheelset.orders.return };
            } else if (hasPickup) {
                return { type: 'pickup', pickup: wheelset.orders.pickup };
            } else if (hasReturn) {
                return { type: 'return', return: wheelset.orders.return };
            }
            return null;
        }
        
        function getOrderStatusBadge(orderStatus) {
            if (!orderStatus) return '';
            
            let badgeClass = '';
            let icon = '';
            let text = '';
            let tooltip = '';
            
            if (orderStatus.type === 'both') {
                badgeClass = 'both';
                icon = '<i class="fas fa-exchange-alt"></i>';
                text = 'Beide';
                const pickupDate = orderStatus.pickup.appointment ? new Date(orderStatus.pickup.appointment).toLocaleDateString('de-DE') : '-';
                const returnDate = orderStatus.return.date ? new Date(orderStatus.return.date).toLocaleDateString('de-DE') : '-';
                tooltip = `Abholung: ${pickupDate}\nRücklieferung: ${returnDate}`;
            } else if (orderStatus.type === 'pickup') {
                badgeClass = 'pickup';
                icon = '<i class="fas fa-truck"></i>';
                text = 'Abholung';
                const date = orderStatus.pickup.appointment ? new Date(orderStatus.pickup.appointment).toLocaleDateString('de-DE') : '-';
                const time = orderStatus.pickup.time || '-';
                tooltip = `Abholung geplant\nDatum: ${date}\nZeit: ${time}`;
            } else if (orderStatus.type === 'return') {
                badgeClass = 'return';
                icon = '<i class="fas fa-arrow-left"></i>';
                text = 'Rücklieferung';
                const date = orderStatus.return.date ? new Date(orderStatus.return.date).toLocaleDateString('de-DE') : '-';
                const time = orderStatus.return.time || '-';
                tooltip = `Rücklieferung geplant\nDatum: ${date}\nZeit: ${time}`;
            }
            
            return `
                <span class="order-status-badge ${badgeClass}" style="position: relative;" title="${tooltip.replace(/\n/g, ' ')}">
                    ${icon} ${text}
                    <span class="order-status-tooltip">${tooltip}</span>
                </span>
            `;
        }
        
        function getOrderStatusClass(orderStatus) {
            if (!orderStatus) return '';
            if (orderStatus.type === 'both') return 'has-both-orders';
            if (orderStatus.type === 'pickup') return 'has-pickup';
            if (orderStatus.type === 'return') return 'has-return';
            return '';
        }

        function selectWheelset(id, event) {
            if (event.target.tagName === 'BUTTON' || event.target.closest('button')) {
                return;
            }
            toggleSelection(id);
        }

        function toggleSelection(id) {
            if (selectedWheelsets.has(id)) {
                selectedWheelsets.delete(id);
            } else {
                selectedWheelsets.add(id);
            }
            renderView();
        }

        function viewDetails(id) {
            const autohausId = $('#autohausSelect').val();
            showLoading(true);
            
            // Lade Wheelset-Details, lokalen Kommentar und Angebote parallel
            Promise.all([
                fetch(`../api/resy_api_routes.php?action=get_wheelset_details&autohaus_id=${autohausId}&wheelset_number=${id}`),
                fetch(`../api/wheelset_comments.php?action=get&wheelset_number=${id}&autohaus_id=${autohausId}`),
                fetch(`../api/wheelset_offers.php?action=get&wheelset_number=${id}&autohaus_id=${autohausId}`)
            ])
                .then(async ([wheelsetResponse, commentResponse, offersResponse]) => {
                    const wheelsetData = await wheelsetResponse.json();
                    const commentData = await commentResponse.json();
                    const offersData = await offersResponse.json();
                    
                    showLoading(false);
                    
                    if (wheelsetData.success && wheelsetData.wheelset) {
                        // Füge lokalen Kommentar hinzu
                        if (commentData.success && commentData.comment) {
                            wheelsetData.wheelset.localComment = commentData.comment;
                        }
                        // Füge Angebote hinzu
                        if (offersData.success && offersData.offers) {
                            wheelsetData.wheelset.offers = offersData.offers;
                            wheelsetData.wheelset.offerCount = offersData.count || 0;
                        }
                        openDetailPanel(wheelsetData.wheelset);
                    } else {
                        showAlert('Details konnten nicht geladen werden', 'danger');
                    }
                })
                .catch(error => {
                    showLoading(false);
                    showAlert('Fehler: ' + error.message, 'danger');
                });
        }
        

        function editComment(wheelsetNumber) {
            const autohausId = $('#autohausSelect').val();
            
            // Lade aktuellen Kommentar vom Server
            showLoading(true);
            fetch(`../api/wheelset_comments.php?action=get&wheelset_number=${wheelsetNumber}&autohaus_id=${autohausId}`)
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    const currentComment = data.success && data.comment ? data.comment : '';
                    
                    // Lade auch RESY-Kommentar für Anzeige
                    fetch(`../api/resy_api_routes.php?action=get_wheelset_details&autohaus_id=${autohausId}&wheelset_number=${wheelsetNumber}`)
                        .then(response => response.json())
                        .then(wheelsetData => {
                            const resyComment = wheelsetData.success && wheelsetData.wheelset && wheelsetData.wheelset.comment ? wheelsetData.wheelset.comment : '';
                            
                            let commentContent = `
                                <div class="form-group">
                                    <label class="form-label">Lokale Bemerkung</label>
                                    <textarea id="commentInput_${wheelsetNumber}" class="form-control" rows="6" placeholder="Ihre Bemerkung zu diesem Radsatz...">${currentComment}</textarea>
                                    <small style="color: var(--text-secondary); margin-top: 0.25rem; display: block;">
                                        Diese Bemerkung wird nur lokal gespeichert und ist nicht Teil der RESY-Daten.
                                    </small>
                                </div>
                            `;
                            
                            // Zeige RESY-Kommentar falls vorhanden
                            if (resyComment && resyComment !== currentComment) {
                                commentContent += `
                                    <div class="form-group" style="margin-top: 1.5rem;">
                                        <label class="form-label" style="display: flex; align-items: center; gap: 0.5rem;">
                                            <i class="fas fa-info-circle" style="color: var(--info);"></i>
                                            RESY Bemerkung (schreibgeschützt)
                                        </label>
                                        <div style="background: rgba(6, 182, 212, 0.1); padding: 1rem; border-radius: var(--radius); border-left: 3px solid var(--info); white-space: pre-wrap; font-size: 0.875rem;">
                                            ${resyComment}
                                        </div>
                                    </div>
                                `;
                            }
                            
                            commentContent += `
                                <div style="display: flex; gap: 0.5rem; justify-content: flex-end; margin-top: 1.5rem;">
                                    <button class="btn btn-secondary" onclick="cancelEditComment('${wheelsetNumber}')">Abbrechen</button>
                                    <button class="btn btn-primary" onclick="saveComment('${wheelsetNumber}')">
                                        <i class="fas fa-save"></i> Speichern
                                    </button>
                                </div>
                            `;
                            
                            createModal(`commentModal_${wheelsetNumber}`, `Bemerkung für Radsatz #${wheelsetNumber}`, commentContent);
                        })
                        .catch(error => {
                            // Fallback wenn RESY-Details nicht geladen werden können
                            let commentContent = `
                                <div class="form-group">
                                    <label class="form-label">Bemerkung</label>
                                    <textarea id="commentInput_${wheelsetNumber}" class="form-control" rows="6" placeholder="Ihre Bemerkung zu diesem Radsatz...">${currentComment}</textarea>
                                </div>
                                <div style="display: flex; gap: 0.5rem; justify-content: flex-end; margin-top: 1.5rem;">
                                    <button class="btn btn-secondary" onclick="cancelEditComment('${wheelsetNumber}')">Abbrechen</button>
                                    <button class="btn btn-primary" onclick="saveComment('${wheelsetNumber}')">
                                        <i class="fas fa-save"></i> Speichern
                                    </button>
                                </div>
                            `;
                            createModal(`commentModal_${wheelsetNumber}`, `Bemerkung für Radsatz #${wheelsetNumber}`, commentContent);
                        });
                })
                .catch(error => {
                    showLoading(false);
                    showAlert('Fehler beim Laden der Bemerkung: ' + error.message, 'danger');
                });
        }
        
        function saveComment(wheelsetNumber) {
            const autohausId = $('#autohausSelect').val();
            const comment = $(`#commentInput_${wheelsetNumber}`).val();
            
            showLoading(true);
            
            fetch(`../api/wheelset_comments.php?action=save`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    wheelset_number: wheelsetNumber,
                    autohaus_id: autohausId,
                    comment: comment
                })
            })
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    if (data.success) {
                        showAlert('✅ Bemerkung gespeichert', 'success');
                        closeModal(`commentModal_${wheelsetNumber}`);
                        // Lade Details neu, um Icon zu aktualisieren
                        viewDetails(wheelsetNumber);
                    } else {
                        showAlert('❌ Fehler: ' + (data.error || 'Unbekannt'), 'danger');
                    }
                })
                .catch(error => {
                    showLoading(false);
                    showAlert('Fehler: ' + error.message, 'danger');
                });
        }
        
        function cancelEditComment(wheelsetNumber) {
            closeModal(`commentModal_${wheelsetNumber}`);
        }
        
        function deleteComment(wheelsetNumber) {
            const autohausId = $('#autohausSelect').val();
            
            if (!confirm('Möchten Sie die Bemerkung wirklich löschen?')) {
                return;
            }
            
            showLoading(true);
            
            fetch(`../api/wheelset_comments.php?action=delete`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    wheelset_number: wheelsetNumber,
                    autohaus_id: autohausId
                })
            })
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    if (data.success) {
                        showAlert('✅ Bemerkung gelöscht', 'success');
                        // Lade Details neu, um Icon zu aktualisieren
                        viewDetails(wheelsetNumber);
                    } else {
                        showAlert('❌ Fehler: ' + (data.error || 'Unbekannt'), 'danger');
                    }
                })
                .catch(error => {
                    showLoading(false);
                    showAlert('Fehler: ' + error.message, 'danger');
                });
        }

        function updatePanelHeaderIcons(wheelset) {
            const iconsContainer = $('#panelHeaderIcons');
            if (!iconsContainer.length) return;
            
            // Prüfe ob Kommentar vorhanden
            const hasLocalComment = wheelset.localComment && wheelset.localComment.trim();
            const hasResyComment = wheelset.comment && wheelset.comment.trim();
            const hasAnyComment = hasLocalComment || hasResyComment;
            
            // Kürze Kommentar für Preview (max 100 Zeichen)
            const commentPreview = hasLocalComment 
                ? (wheelset.localComment.length > 100 ? wheelset.localComment.substring(0, 100) + '...' : wheelset.localComment)
                : (hasResyComment 
                    ? (wheelset.comment.length > 100 ? wheelset.comment.substring(0, 100) + '...' : wheelset.comment)
                    : '');
            
            let iconsHtml = `
                <!-- Bemerkungs-Icon -->
                <div class="comment-icon-wrapper" style="position: relative;">
                    <div class="panel-header-icon ${hasAnyComment ? 'has-data' : ''}" 
                         onclick="editComment('${wheelset.wheelSetNumber || ''}')"
                         title="${hasAnyComment ? 'Bemerkung bearbeiten' : 'Bemerkung hinzufügen'}">
                        <i class="fas ${hasAnyComment ? 'fa-comment-dots' : 'fa-comment'}"></i>
                        ${hasAnyComment ? '<span class="panel-header-icon-badge"></span>' : ''}
                    </div>
                    ${hasAnyComment && commentPreview ? `
                    <div class="comment-preview-tooltip">
                        ${commentPreview.replace(/</g, '&lt;').replace(/>/g, '&gt;')}
                    </div>
                    ` : ''}
                    ${hasLocalComment ? `
                    <button class="comment-delete-hint" 
                            onclick="event.stopPropagation(); deleteComment('${wheelset.wheelSetNumber || ''}')"
                            title="Bemerkung löschen">
                        <i class="fas fa-times"></i>
                    </button>
                    ` : ''}
                </div>
                
                <!-- Angebot-Icon -->
                <div class="panel-header-icon ${(wheelset.offerCount || 0) > 0 ? 'has-data' : ''}" 
                     onclick="openOfferModal('${wheelset.wheelSetNumber || ''}')"
                     title="${(wheelset.offerCount || 0) > 0 ? `${wheelset.offerCount} Angebot${wheelset.offerCount > 1 ? 'e' : ''}` : 'Angebot erstellen'}">
                    <i class="fas fa-file-invoice-dollar"></i>
                    ${(wheelset.offerCount || 0) > 0 ? `<span class="panel-header-icon-badge">${wheelset.offerCount}</span>` : ''}
                </div>
                
                <!-- Rechnung-Icon (Platzhalter) -->
                <div class="panel-header-icon placeholder" 
                     title="Rechnung (Coming Soon)"
                     style="cursor: not-allowed;">
                    <i class="fas fa-receipt"></i>
                </div>
                
                <!-- Service-Historie-Icon (Platzhalter) -->
                <div class="panel-header-icon placeholder" 
                     title="Service-Historie (Coming Soon)"
                     style="cursor: not-allowed;">
                    <i class="fas fa-history"></i>
                </div>
            `;
            
            // Prüfe ob Pickup oder Return Order vorhanden
            const wheelsetNumber = wheelset.wheelSetNumber;
            const hasPickupOrder = pickupOrders.some(o => o.wheelSetNumber == wheelsetNumber);
            const hasReturnOrder = returnOrders.some(o => o.wheelSetNumber == wheelsetNumber);
            
            // Füge Pickup/Return Order Icons hinzu wenn vorhanden
            if (hasPickupOrder || hasReturnOrder) {
                iconsHtml = iconsHtml.replace('</div>', ''); // Entferne schließendes div
                
                if (hasPickupOrder) {
                    const pickupOrder = pickupOrders.find(o => o.wheelSetNumber == wheelsetNumber);
                    iconsHtml += `
                        <!-- Pickup Order Icon -->
                        <div class="panel-header-icon has-data" 
                             onclick="showPickupOrderModal('${wheelsetNumber}')"
                             title="Abholung bearbeiten">
                            <i class="fas fa-truck"></i>
                            <span class="panel-header-icon-badge"></span>
                        </div>
                    `;
                }
                
                if (hasReturnOrder) {
                    const returnOrder = returnOrders.find(o => o.wheelSetNumber == wheelsetNumber);
                    iconsHtml += `
                        <!-- Return Order Icon -->
                        <div class="panel-header-icon has-data" 
                             onclick="showReturnOrderModal('${wheelsetNumber}')"
                             title="Rücklieferung bearbeiten">
                            <i class="fas fa-arrow-left"></i>
                            <span class="panel-header-icon-badge"></span>
                        </div>
                    `;
                }
                
                iconsHtml += '</div>'; // Schließe wieder
            }
            
            iconsContainer.html(iconsHtml);
        }
        
        function showPickupOrderModal(wheelsetNumber) {
            const pickupOrder = pickupOrders.find(o => o.wheelSetNumber == wheelsetNumber);
            if (!pickupOrder) {
                showAlert('Abholung nicht gefunden', 'warning');
                return;
            }
            
            const appointmentDate = pickupOrder.appointment ? new Date(pickupOrder.appointment).toLocaleDateString('de-DE') : '-';
            const appointmentTime = pickupOrder.time || '-';
            const statusClass = pickupOrder.statusCode === 'SUCCESS' || pickupOrder.statusCode === 'OK' ? 'success' : 
                              pickupOrder.statusCode === 'WARNING' || pickupOrder.statusCode === 'PENDING' ? 'warning' : 'error';
            
            const modalContent = `
                <div style="max-height: 500px; overflow-y: auto;">
                    <div style="background: var(--bg-hover); padding: 1.5rem; border-radius: var(--radius); margin-bottom: 1rem;">
                        <h3 style="margin: 0 0 1rem 0; font-size: 1.125rem; font-weight: 600; color: var(--primary);">
                            <i class="fas fa-truck" style="margin-right: 0.5rem;"></i>Abholung - Satz #${wheelsetNumber}
                        </h3>
                        <div style="display: grid; gap: 0.75rem;">
                            <div>
                                <strong style="color: var(--text-secondary); font-size: 0.75rem;">Status:</strong>
                                <div style="font-size: 0.875rem; margin-top: 0.25rem;">
                                    <span class="order-status-code ${statusClass}">${pickupOrder.status || pickupOrder.statusCode || '-'}</span>
                                </div>
                            </div>
                            <div>
                                <strong style="color: var(--text-secondary); font-size: 0.75rem;">Datum:</strong>
                                <div style="font-size: 0.875rem; margin-top: 0.25rem;">${appointmentDate}</div>
                            </div>
                            <div>
                                <strong style="color: var(--text-secondary); font-size: 0.75rem;">Zeit:</strong>
                                <div style="font-size: 0.875rem; margin-top: 0.25rem;">${appointmentTime}</div>
                            </div>
                            <div>
                                <strong style="color: var(--text-secondary); font-size: 0.75rem;">Ladestelle:</strong>
                                <div style="font-size: 0.875rem; margin-top: 0.25rem;">${pickupOrder.location || '-'}</div>
                            </div>
                            <div>
                                <strong style="color: var(--text-secondary); font-size: 0.75rem;">Service-Paket:</strong>
                                <div style="font-size: 0.875rem; margin-top: 0.25rem;">${pickupOrder.servicePackageName || pickupOrder.servicePackageIdent || '-'}</div>
                            </div>
                            ${pickupOrder.licenceNo ? `
                            <div>
                                <strong style="color: var(--text-secondary); font-size: 0.75rem;">Kennzeichen:</strong>
                                <div style="font-size: 0.875rem; margin-top: 0.25rem;">${pickupOrder.licenceNo}</div>
                            </div>
                            ` : ''}
                        </div>
                    </div>
                    <div style="display: flex; gap: 0.5rem;">
                        <button class="btn btn-secondary" onclick="closeModal('pickupOrderDetailsModal')" style="flex: 1;">
                            <i class="fas fa-times"></i> Schließen
                        </button>
                        <button class="btn btn-danger" onclick="cancelPickupOrder(${wheelsetNumber}); closeModal('pickupOrderDetailsModal');" style="flex: 1;">
                            <i class="fas fa-times"></i> Stornieren
                        </button>
                    </div>
                </div>
            `;
            
            createModal('pickupOrderDetailsModal', `Abholung - Satz #${wheelsetNumber}`, modalContent);
        }
        
        function showReturnOrderModal(wheelsetNumber) {
            const returnOrder = returnOrders.find(o => o.wheelSetNumber == wheelsetNumber);
            if (!returnOrder) {
                showAlert('Rücklieferung nicht gefunden', 'warning');
                return;
            }
            
            const returnDate = returnOrder.date ? new Date(returnOrder.date).toLocaleDateString('de-DE') : '-';
            const returnTime = returnOrder.time || '-';
            const statusClass = returnOrder.statusCode === 'SUCCESS' || returnOrder.statusCode === 'OK' ? 'success' : 
                              returnOrder.statusCode === 'WARNING' || returnOrder.statusCode === 'PENDING' ? 'warning' : 'error';
            
            const modalContent = `
                <div style="max-height: 500px; overflow-y: auto;">
                    <div style="background: var(--bg-hover); padding: 1.5rem; border-radius: var(--radius); margin-bottom: 1rem;">
                        <h3 style="margin: 0 0 1rem 0; font-size: 1.125rem; font-weight: 600; color: #8b5cf6;">
                            <i class="fas fa-arrow-left" style="margin-right: 0.5rem;"></i>Rücklieferung - Satz #${wheelsetNumber}
                        </h3>
                        <div style="display: grid; gap: 0.75rem;">
                            <div>
                                <strong style="color: var(--text-secondary); font-size: 0.75rem;">Status:</strong>
                                <div style="font-size: 0.875rem; margin-top: 0.25rem;">
                                    <span class="order-status-code ${statusClass}">${returnOrder.status || returnOrder.statusCode || '-'}</span>
                                </div>
                            </div>
                            <div>
                                <strong style="color: var(--text-secondary); font-size: 0.75rem;">Datum:</strong>
                                <div style="font-size: 0.875rem; margin-top: 0.25rem;">${returnDate}</div>
                            </div>
                            <div>
                                <strong style="color: var(--text-secondary); font-size: 0.75rem;">Zeit:</strong>
                                <div style="font-size: 0.875rem; margin-top: 0.25rem;">${returnTime}</div>
                            </div>
                            <div>
                                <strong style="color: var(--text-secondary); font-size: 0.75rem;">Lieferoption:</strong>
                                <div style="font-size: 0.875rem; margin-top: 0.25rem;">${returnOrder.deliveryOption || '-'}</div>
                            </div>
                            ${returnOrder.rollShelfNo ? `
                            <div>
                                <strong style="color: var(--text-secondary); font-size: 0.75rem;">Rollenregal:</strong>
                                <div style="font-size: 0.875rem; margin-top: 0.25rem;">${returnOrder.rollShelfNo}</div>
                            </div>
                            ` : ''}
                            ${returnOrder.licenseNo ? `
                            <div>
                                <strong style="color: var(--text-secondary); font-size: 0.75rem;">Kennzeichen:</strong>
                                <div style="font-size: 0.875rem; margin-top: 0.25rem;">${returnOrder.licenseNo}</div>
                            </div>
                            ` : ''}
                        </div>
                    </div>
                    <div style="display: flex; gap: 0.5rem;">
                        <button class="btn btn-secondary" onclick="closeModal('returnOrderDetailsModal')" style="flex: 1;">
                            <i class="fas fa-times"></i> Schließen
                        </button>
                        <button class="btn btn-danger" onclick="cancelReturnOrder(${wheelsetNumber}); closeModal('returnOrderDetailsModal');" style="flex: 1;">
                            <i class="fas fa-times"></i> Stornieren
                        </button>
                    </div>
                </div>
            `;
            
            createModal('returnOrderDetailsModal', `Rücklieferung - Satz #${wheelsetNumber}`, modalContent);
        }
        
        function openOfferModal(wheelsetNumber) {
            const autohausId = $('#autohausSelect').val();
            
            // Lade aktuelles Wheelset mit Angeboten
            showLoading(true);
            Promise.all([
                fetch(`../api/resy_api_routes.php?action=get_wheelset_details&autohaus_id=${autohausId}&wheelset_number=${wheelsetNumber}`),
                fetch(`../api/wheelset_offers.php?action=get&wheelset_number=${wheelsetNumber}&autohaus_id=${autohausId}`)
            ])
                .then(async ([wheelsetResponse, offersResponse]) => {
                    const wheelsetData = await wheelsetResponse.json();
                    const offersData = await offersResponse.json();
                    
                    showLoading(false);
                    
                    if (!wheelsetData.success || !wheelsetData.wheelset) {
                        showAlert('Reifensatz-Daten konnten nicht geladen werden', 'danger');
                        return;
                    }
                    
                    const wheelset = wheelsetData.wheelset;
                    const offers = offersData.success ? (offersData.offers || []) : [];
                    
                    if (offers.length > 0) {
                        // Zeige Angebots-Übersicht
                        showOffersListModal(wheelsetNumber, wheelset, offers);
                    } else {
                        // Öffne direkt das Erstellungs-Modal
                        showCreateOfferModal(wheelsetNumber, wheelset);
                    }
                })
                .catch(error => {
                    showLoading(false);
                    showAlert('Fehler: ' + error.message, 'danger');
                });
        }
        
        function showOffersListModal(wheelsetNumber, wheelset, offers) {
            let offersListHtml = '';
            offers.forEach((offer, index) => {
                const createdDate = new Date(offer.created_at).toLocaleDateString('de-DE', {
                    day: '2-digit',
                    month: '2-digit',
                    year: 'numeric',
                    hour: '2-digit',
                    minute: '2-digit'
                });
                    const statusBadge = {
                        'draft': '<span style="background: #6c757d; color: white; padding: 0.25rem 0.5rem; border-radius: 4px; font-size: 0.75rem;">Entwurf</span>',
                        'sent': '<span style="background: #007bff; color: white; padding: 0.25rem 0.5rem; border-radius: 4px; font-size: 0.75rem;">Gesendet</span>',
                        'viewed': '<span style="background: #ffc107; color: #000; padding: 0.25rem 0.5rem; border-radius: 4px; font-size: 0.75rem;">Angesehen</span>',
                        'accepted': '<span style="background: #28a745; color: white; padding: 0.25rem 0.5rem; border-radius: 4px; font-size: 0.75rem;">Angenommen</span>',
                        'declined': '<span style="background: #dc3545; color: white; padding: 0.25rem 0.5rem; border-radius: 4px; font-size: 0.75rem;">Abgelehnt</span>',
                        'new_request': '<span style="background: #17a2b8; color: white; padding: 0.25rem 0.5rem; border-radius: 4px; font-size: 0.75rem;">Neu angefragt</span>'
                    }[offer.status] || '<span style="background: #6c757d; color: white; padding: 0.25rem 0.5rem; border-radius: 4px; font-size: 0.75rem;">Unbekannt (' + (offer.status || 'null') + ')</span>';
                
                const canEdit = offer.status === 'draft';
                const canView = true;
                
                offersListHtml += `
                    <div style="padding: 1rem; border: 1px solid var(--border); border-radius: var(--radius); margin-bottom: 0.75rem; background: var(--bg-secondary);">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 0.5rem;">
                            <div>
                                <strong>Angebot #${index + 1}</strong>
                                <div style="font-size: 0.75rem; color: var(--text-secondary); margin-top: 0.25rem;">Erstellt am: ${createdDate}</div>
                                ${offer.employee_name ? `<div style="font-size: 0.75rem; color: var(--text-secondary); margin-top: 0.25rem;"><i class="fas fa-user"></i> ${offer.employee_name}</div>` : ''}
                            </div>
                            ${statusBadge}
                        </div>
                        ${offer.total_amount ? `<div style="font-size: 0.875rem; color: var(--text-primary); margin-bottom: 0.5rem;"><strong>Gesamtbetrag:</strong> ${parseFloat(offer.total_amount).toFixed(2).replace('.', ',')} €</div>` : ''}
                        ${offer.email_sent ? `<div style="font-size: 0.75rem; color: var(--text-secondary); margin-bottom: 0.5rem;"><i class="fas fa-envelope"></i> Email versendet${offer.email_sent_at ? ' am ' + new Date(offer.email_sent_at).toLocaleDateString('de-DE') : ''}</div>` : ''}
                        <div style="display: flex; gap: 0.5rem; margin-top: 0.75rem;">
                            <button class="btn btn-sm btn-primary" onclick="viewOfferDetails(${offer.id}, '${wheelsetNumber}', ${JSON.stringify(wheelset).replace(/"/g, '&quot;')})" style="font-size: 0.75rem;">
                                <i class="fas fa-eye"></i> Ansehen
                            </button>
                            ${canEdit ? `
                            <button class="btn btn-sm btn-secondary" onclick="editOffer(${offer.id}, '${wheelsetNumber}', ${JSON.stringify(wheelset).replace(/"/g, '&quot;')})" style="font-size: 0.75rem;">
                                <i class="fas fa-edit"></i> Bearbeiten
                            </button>
                            <button class="btn btn-sm btn-primary" onclick="resendOfferFromList(${offer.id}, '${wheelsetNumber}')" style="font-size: 0.75rem;">
                                <i class="fas fa-envelope"></i> Versenden
                            </button>
                            ` : ''}
                        </div>
                    </div>
                `;
            });
            
            const modalContent = `
                <div style="max-height: 500px; overflow-y: auto;">
                    <div style="background: var(--bg-hover); padding: 1.5rem; border-radius: var(--radius); margin-bottom: 1rem;">
                        <h3 style="margin: 0 0 0.5rem 0; font-size: 1.125rem; font-weight: 600; color: var(--primary);">
                            <i class="fas fa-file-invoice-dollar" style="margin-right: 0.5rem;"></i>Angebote - Satz #${wheelsetNumber}
                        </h3>
                        <div style="font-size: 0.875rem; color: var(--text-secondary);">
                            ${offers.length} Angebot${offers.length > 1 ? 'e' : ''} vorhanden
                        </div>
                    </div>
                    
                    <div style="margin-bottom: 1rem;">
                        ${offersListHtml}
                    </div>
                    
                    <div style="display: flex; gap: 0.5rem;">
                        <button class="btn btn-secondary" onclick="closeModal('offersListModal')" style="flex: 1;">
                            <i class="fas fa-times"></i> Schließen
                        </button>
                    </div>
                </div>
            `;
            
            createModal('offersListModal', `Angebote - Satz #${wheelsetNumber}`, modalContent);
            
            // Füge Button "Neues Angebot" zum Modal-Header hinzu (rechts neben dem Titel)
            setTimeout(() => {
                const modalHeader = $(`#offersListModal .modal-header`);
                const modalTitle = $(`#offersListModal .modal-title`);
                if (modalHeader.length && modalTitle.length) {
                    // Stelle sicher dass Header Flexbox ist (ist bereits per CSS, aber für Sicherheit)
                    modalHeader.css({
                        'display': 'flex',
                        'align-items': 'center',
                        'justify-content': 'space-between'
                    });
                    
                    // Titel nimmt verfügbaren Platz
                    modalTitle.css('flex', '1');
                    
                    // Erstelle Button
                    const newOfferButton = $(`
                        <button type="button" class="btn btn-primary btn-sm" onclick="closeModal('offersListModal'); setTimeout(() => showCreateOfferModal('${wheelsetNumber}', ${JSON.stringify(wheelset).replace(/"/g, '&quot;')}), 100);" style="margin-left: auto; margin-right: 10px; white-space: nowrap;">
                            <i class="fas fa-plus"></i> Neues Angebot
                        </button>
                    `);
                    
                    // Füge Button vor dem Close-Button ein
                    const closeButton = modalHeader.find('.modal-close');
                    closeButton.css('margin-left', '10px');
                    newOfferButton.insertBefore(closeButton);
                }
            }, 50);
        }
        
        function viewOfferDetails(offerId, wheelsetNumber, wheelset) {
            const autohausId = $('#autohausSelect').val();
            
            showLoading(true);
            
            fetch(`../api/wheelset_offers.php?action=get&wheelset_number=${wheelsetNumber}&autohaus_id=${autohausId}`)
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    
                    if (!data.success || !data.offers) {
                        showAlert('Angebot nicht gefunden', 'danger');
                        return;
                    }
                    
                    const offer = data.offers.find(o => o.id == offerId);
                    if (!offer) {
                        showAlert('Angebot nicht gefunden', 'danger');
                        return;
                    }
                    
                    const offerData = offer.offer_data || {};
                    const newTires = offerData.new_tires || [];
                    const selectedServices = offerData.selected_services || [];
                    
                    const createdDate = new Date(offer.created_at).toLocaleDateString('de-DE', {
                        day: '2-digit',
                        month: '2-digit',
                        year: 'numeric',
                        hour: '2-digit',
                        minute: '2-digit'
                    });
                    
                    const statusBadge = {
                        'draft': '<span style="background: #6c757d; color: white; padding: 0.5rem 1rem; border-radius: 6px;">Entwurf</span>',
                        'sent': '<span style="background: #007bff; color: white; padding: 0.5rem 1rem; border-radius: 6px;">Gesendet</span>',
                        'viewed': '<span style="background: #ffc107; color: #000; padding: 0.5rem 1rem; border-radius: 6px;">Angesehen</span>',
                        'accepted': '<span style="background: #28a745; color: white; padding: 0.5rem 1rem; border-radius: 6px;">Angenommen</span>',
                        'declined': '<span style="background: #dc3545; color: white; padding: 0.5rem 1rem; border-radius: 6px;">Abgelehnt</span>',
                        'new_request': '<span style="background: #17a2b8; color: white; padding: 0.5rem 1rem; border-radius: 6px;">Neu angefragt</span>'
                    }[offer.status] || '<span style="background: #6c757d; color: white; padding: 0.5rem 1rem; border-radius: 6px;">Unbekannt (' + (offer.status || 'null') + ')</span>';
                    
                    let tiresHtml = '';
                    if (newTires.length > 0) {
                        tiresHtml = '<table style="width: 100%; border-collapse: collapse; margin-top: 1rem;">';
                        tiresHtml += '<thead><tr style="background: var(--bg-hover);"><th style="padding: 0.75rem; text-align: left; border-bottom: 2px solid var(--border);">Reifen</th><th style="padding: 0.75rem; text-align: left; border-bottom: 2px solid var(--border);">Dimension</th><th style="padding: 0.75rem; text-align: right; border-bottom: 2px solid var(--border);">Preis</th><th style="padding: 0.75rem; text-align: right; border-bottom: 2px solid var(--border);">Montage</th><th style="padding: 0.75rem; text-align: center; border-bottom: 2px solid var(--border);">Anzahl</th><th style="padding: 0.75rem; text-align: right; border-bottom: 2px solid var(--border);">Betrag</th></tr></thead><tbody>';
                        
                        newTires.forEach(tire => {
                            tiresHtml += `
                                <tr style="border-bottom: 1px solid var(--border);">
                                    <td style="padding: 0.75rem;">${tire.brand_model || '-'}</td>
                                    <td style="padding: 0.75rem;">${tire.dimension || '-'} ${tire.eu_label ? '(' + tire.eu_label + ')' : ''}</td>
                                    <td style="padding: 0.75rem; text-align: right;">${parseFloat(tire.price || 0).toFixed(2).replace('.', ',')} €</td>
                                    <td style="padding: 0.75rem; text-align: right;">${parseFloat(tire.mounting || 0).toFixed(2).replace('.', ',')} €</td>
                                    <td style="padding: 0.75rem; text-align: center;">${tire.quantity || 1}</td>
                                    <td style="padding: 0.75rem; text-align: right; font-weight: 600;">${parseFloat(tire.total || 0).toFixed(2).replace('.', ',')} €</td>
                                </tr>
                            `;
                        });
                        tiresHtml += '</tbody></table>';
                    } else {
                        tiresHtml = '<div style="color: var(--text-secondary); padding: 1rem;">Keine Reifen definiert</div>';
                    }
                    
                    // Lade Services
                    Promise.resolve().then(() => {
                        if (selectedServices.length > 0) {
                            return fetch(`../api/get_autohaus_services.php?autohaus_id=${autohausId}`)
                                .then(response => response.json())
                                .then(servicesData => {
                                    const allServices = servicesData.success ? (servicesData.services || []) : [];
                                    const services = allServices.filter(s => selectedServices.includes(s.id));
                                    return services;
                                });
                        }
                        return [];
                    }).then(services => {
                        let servicesHtml = '';
                        if (services.length > 0) {
                            services.forEach(service => {
                                servicesHtml += `
                                    <div style="display: flex; justify-content: space-between; padding: 0.75rem; border-bottom: 1px solid var(--border);">
                                        <div>${service.service_name}</div>
                                        <div style="font-weight: 600; color: var(--primary);">${parseFloat(service.price || 0).toFixed(2).replace('.', ',')} €</div>
                                    </div>
                                `;
                            });
                        } else if (selectedServices.length > 0) {
                            servicesHtml = '<div style="color: var(--text-secondary); padding: 1rem;">Serviceleistungen konnten nicht geladen werden</div>';
                        } else {
                            servicesHtml = '<div style="color: var(--text-secondary); padding: 1rem;">Keine Serviceleistungen ausgewählt</div>';
                        }
                        
                        // Parse customer_selection - kann bereits Objekt sein oder JSON-String
                        let customerSelection = null;
                        if (offer.customer_selection) {
                            if (typeof offer.customer_selection === 'string') {
                                try {
                                    customerSelection = JSON.parse(offer.customer_selection);
                                } catch (e) {
                                    console.error('Error parsing customer_selection:', e);
                                    customerSelection = null;
                                }
                            } else {
                                customerSelection = offer.customer_selection;
                            }
                        }
                        
                        let selectionHtml = '';
                        if (customerSelection) {
                            // Zeige alle relevanten Informationen der Kundenauswahl
                            let selectionDetails = '';
                            
                            if (customerSelection.action === 'accepted') {
                                selectionDetails += '<div style="margin-top: 0.75rem;"><strong style="color: #0056b3;"><i class="fas fa-check"></i> Status: Angenommen</strong></div>';
                            } else if (customerSelection.action === 'declined') {
                                selectionDetails += '<div style="margin-top: 0.75rem;"><strong style="color: #dc3545;"><i class="fas fa-times"></i> Status: Abgelehnt</strong></div>';
                                if (customerSelection.decline_reason) {
                                    selectionDetails += `<div style="margin-top: 0.5rem; padding: 0.75rem; background: #fff5f5; border-radius: 6px; border-left: 3px solid #dc3545;"><strong>Ablehnungsgrund:</strong><br>${customerSelection.decline_reason}</div>`;
                                }
                            } else if (customerSelection.action === 'new_request') {
                                selectionDetails += '<div style="margin-top: 0.75rem;"><strong style="color: #17a2b8;"><i class="fas fa-redo"></i> Status: Neu angefragt</strong></div>';
                                if (customerSelection.request_notes) {
                                    selectionDetails += `<div style="margin-top: 0.5rem; padding: 0.75rem; background: #e7f5ff; border-radius: 6px; border-left: 3px solid #17a2b8;"><strong>Anfrage-Wünsche:</strong><br>${customerSelection.request_notes}</div>`;
                                }
                            }
                            
                            // Zeige ausgewählte Reifen
                            if (customerSelection.tires && customerSelection.tires.length > 0) {
                                selectionDetails += '<div style="margin-top: 0.75rem;"><strong style="color: #0056b3;"><i class="fas fa-tire"></i> Ausgewählte Reifen:</strong></div>';
                                selectionDetails += '<ul style="margin-top: 0.5rem; padding-left: 1.5rem;">';
                                customerSelection.tires.forEach((tireIndex, idx) => {
                                    const tire = newTires[tireIndex];
                                    if (tire) {
                                        selectionDetails += `<li style="margin-top: 0.25rem;">${tire.brand_model || 'Reifen'} (${tire.dimension || ''}) - ${parseFloat(tire.total || 0).toFixed(2).replace('.', ',')} €</li>`;
                                    }
                                });
                                selectionDetails += '</ul>';
                            }
                            
                            // Zeige ausgewählte Services
                            if (customerSelection.services && customerSelection.services.length > 0) {
                                selectionDetails += '<div style="margin-top: 0.75rem;"><strong style="color: #0056b3;"><i class="fas fa-cog"></i> Ausgewählte Zusatzleistungen:</strong></div>';
                                selectionDetails += '<ul style="margin-top: 0.5rem; padding-left: 1.5rem;">';
                                customerSelection.services.forEach(serviceId => {
                                    const service = services.find(s => s.id == serviceId);
                                    if (service) {
                                        selectionDetails += `<li style="margin-top: 0.25rem;">${service.service_name} - ${parseFloat(service.price || 0).toFixed(2).replace('.', ',')} €</li>`;
                                    }
                                });
                                selectionDetails += '</ul>';
                            }
                            
                            // Zeige Anmerkungen (wichtig!)
                            if (customerSelection.notes) {
                                selectionDetails += '<div style="margin-top: 1rem; padding: 1rem; background: #fff9e6; border-radius: 8px; border-left: 4px solid #f59e0b;">';
                                selectionDetails += '<strong style="color: #92400e;"><i class="fas fa-comment"></i> Kundenanmerkungen:</strong>';
                                selectionDetails += `<div style="margin-top: 0.5rem; color: #78350f; white-space: pre-wrap;">${customerSelection.notes}</div>`;
                                selectionDetails += '</div>';
                            }
                            
                            selectionHtml = `
                                <div style="background: #e7f1ff; padding: 1.5rem; border-radius: 8px; margin-top: 1rem; border-left: 4px solid #007bff;">
                                    <strong style="color: #0056b3; font-size: 1.1rem;"><i class="fas fa-check-circle"></i> Kundenauswahl</strong>
                                    ${selectionDetails}
                                </div>
                            `;
                        }
                        
                        const modalContent = `
                            <div style="max-height: 70vh; overflow-y: auto;">
                                <div style="background: var(--bg-hover); padding: 1.5rem; border-radius: var(--radius); margin-bottom: 1rem;">
                                    <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 0.5rem;">
                                        <div>
                                            <h3 style="margin: 0; font-size: 1.125rem; font-weight: 600; color: var(--primary);">
                                                Angebot #${offer.id}
                                            </h3>
                                            <div style="font-size: 0.875rem; color: var(--text-secondary); margin-top: 0.25rem;">
                                                Erstellt am: ${createdDate}
                                            </div>
                                            ${offer.employee_name ? `<div style="font-size: 0.875rem; color: var(--text-secondary); margin-top: 0.25rem;"><i class="fas fa-user"></i> ${offer.employee_name}</div>` : ''}
                                        </div>
                                        ${statusBadge}
                                    </div>
                                </div>
                                
                                <div style="margin-bottom: 1.5rem;">
                                    <strong style="color: var(--text-primary);">Kundendaten</strong>
                                    <div style="background: var(--bg-hover); padding: 1rem; border-radius: var(--radius); margin-top: 0.5rem; font-size: 0.875rem;">
                                        <div><strong>Name:</strong> ${offer.customer_name || '-'}</div>
                                        <div><strong>Kennzeichen:</strong> ${offer.license_plate || '-'}</div>
                                        <div><strong>Satznummer:</strong> ${offer.wheelset_number || '-'}</div>
                                        ${offer.customer_no ? `<div><strong>Kundennummer:</strong> ${offer.customer_no}</div>` : ''}
                                        ${offer.vin ? `<div><strong>FIN/VIN:</strong> ${offer.vin}</div>` : ''}
                                    </div>
                                </div>
                                
                                <div style="margin-bottom: 1.5rem;">
                                    <strong style="color: var(--text-primary);">Reifen</strong>
                                    ${tiresHtml}
                                </div>
                                
                                ${services.length > 0 || selectedServices.length > 0 ? `
                                <div style="margin-bottom: 1.5rem;">
                                    <strong style="color: var(--text-primary);">Zusatzleistungen</strong>
                                    <div style="background: var(--bg-hover); padding: 1rem; border-radius: var(--radius); margin-top: 0.5rem;">
                                        ${servicesHtml}
                                    </div>
                                </div>
                                ` : ''}
                                
                                ${offer.total_amount ? `
                                <div style="background: #fef3c7; border: 2px solid #f59e0b; border-radius: 8px; padding: 1rem; margin-bottom: 1.5rem;">
                                    <div style="display: flex; justify-content: space-between; font-size: 1.1rem; font-weight: 700; color: #92400e;">
                                        <span>Gesamtbetrag:</span>
                                        <span>${parseFloat(offer.total_amount).toFixed(2).replace('.', ',')} €</span>
                                    </div>
                                </div>
                                ` : ''}
                                
                                ${offer.letter_content ? `
                                <div style="margin-bottom: 1.5rem;">
                                    <strong style="color: var(--text-primary);">Brieftext</strong>
                                    <div style="background: var(--bg-hover); padding: 1rem; border-radius: var(--radius); margin-top: 0.5rem; white-space: pre-wrap; font-size: 0.875rem;">
                                        ${offer.letter_content.replace(/\n/g, '<br>')}
                                    </div>
                                </div>
                                ` : ''}
                                
                                ${selectionHtml}
                                
                                ${offer.email_sent ? `
                                <div style="background: #d1fae5; padding: 1rem; border-radius: 8px; margin-top: 1rem; border-left: 4px solid #10b981;">
                                    <strong style="color: #065f46;"><i class="fas fa-envelope"></i> Email versendet</strong>
                                    ${offer.email_sent_at ? `<div style="font-size: 0.875rem; color: #065f46; margin-top: 0.25rem;">am ${new Date(offer.email_sent_at).toLocaleDateString('de-DE', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</div>` : ''}
                                </div>
                                ` : ''}
                                
                                <div style="display: flex; gap: 0.5rem; justify-content: flex-end; margin-top: 1.5rem;">
                                    ${offer.status === 'draft' ? `
                                    <button class="btn btn-primary" onclick="resendOffer(${offer.id}, '${wheelsetNumber}')">
                                        <i class="fas fa-envelope"></i> Erneut versenden
                                    </button>
                                    ` : ''}
                                    <button class="btn btn-secondary" onclick="closeModal('offerDetailsModal')">
                                        <i class="fas fa-times"></i> Schließen
                                    </button>
                                </div>
                            </div>
                        `;
                        
                        createModal('offerDetailsModal', `Angebot Details - Satz #${wheelsetNumber}`, modalContent);
                    });
                })
                .catch(error => {
                    showLoading(false);
                    showAlert('Fehler: ' + error.message, 'danger');
                });
        }
        
        function editOffer(offerId, wheelsetNumber, wheelset) {
            // Lade Angebot-Daten und öffne Bearbeitungs-Modal ähnlich wie createOffer
            const autohausId = $('#autohausSelect').val();
            
            showLoading(true);
            
            Promise.all([
                fetch(`../api/wheelset_offers.php?action=get&wheelset_number=${wheelsetNumber}&autohaus_id=${autohausId}`),
                fetch(`../api/get_autohaus_services.php?autohaus_id=${autohausId}`)
            ])
                .then(async ([offersResponse, servicesResponse]) => {
                    const offersData = await offersResponse.json();
                    const servicesData = await servicesResponse.json();
                    
                    showLoading(false);
                    
                    if (!offersData.success || !offersData.offers) {
                        showAlert('Angebot nicht gefunden', 'danger');
                        return;
                    }
                    
                    const offer = offersData.offers.find(o => o.id == offerId);
                    if (!offer) {
                        showAlert('Angebot nicht gefunden', 'danger');
                        return;
                    }
                    
                    if (offer.status !== 'draft') {
                        showAlert('Nur Entwürfe können bearbeitet werden', 'warning');
                        return;
                    }
                    
                    // Weiter mit Bearbeitungs-Modal ähnlich wie showCreateOfferModal
                    showEditOfferModal(offerId, offer, wheelsetNumber, wheelset, servicesData.success ? (servicesData.services || []) : []);
                })
                .catch(error => {
                    showLoading(false);
                    showAlert('Fehler: ' + error.message, 'danger');
                });
        }
        
        function showEditOfferModal(offerId, offer, wheelsetNumber, wheelset, services) {
            const autohausId = $('#autohausSelect').val();
            const employeeName = offer.employee_name || '<?php echo htmlspecialchars($current_user_name, ENT_QUOTES, "UTF-8"); ?>';
            const offerData = offer.offer_data || {};
            const newTires = offerData.new_tires || [];
            const selectedServices = offerData.selected_services || [];
            
            let servicesHtml = '';
            if (services.length > 0) {
                servicesHtml = '<div style="display: grid; gap: 0.75rem;">';
                services.forEach(service => {
                    const isSelected = selectedServices.includes(service.id);
                    servicesHtml += `
                        <label class="service-option-card" style="display: block; cursor: pointer; transition: all 0.3s;">
                            <input type="checkbox" class="offer-service-checkbox" value="${service.id}" data-price="${service.price}" ${isSelected ? 'checked' : ''} style="position: absolute; opacity: 0; pointer-events: none;">
                            <div class="service-card-content" style="display: flex; align-items: center; padding: 1rem; border: 2px solid ${isSelected ? 'var(--primary)' : 'var(--border)'}; border-radius: 12px; background: ${isSelected ? 'rgba(102, 126, 234, 0.05)' : 'var(--bg-secondary)'}; transition: all 0.3s; position: relative; overflow: hidden; box-shadow: ${isSelected ? '0 0 0 3px rgba(102, 126, 234, 0.1)' : 'none'};">
                                <div style="width: 48px; height: 48px; border-radius: 10px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); display: flex; align-items: center; justify-content: center; margin-right: 1rem; flex-shrink: 0;">
                                    <i class="fas fa-check" style="color: white; font-size: 1.25rem; opacity: ${isSelected ? '1' : '0'}; transition: opacity 0.2s;"></i>
                                </div>
                                <div style="flex: 1; min-width: 0;">
                                    <div style="font-weight: 600; color: var(--text-primary); font-size: 0.9375rem; margin-bottom: 0.25rem;">${service.service_name}</div>
                                    ${service.service_description ? `<div style="font-size: 0.8125rem; color: var(--text-secondary); line-height: 1.4;">${service.service_description}</div>` : ''}
                                </div>
                                ${service.show_price ? `<div style="font-weight: 700; color: var(--primary); font-size: 1.0625rem; margin-left: 1rem; white-space: nowrap;">${parseFloat(service.price).toFixed(2).replace('.', ',')} €</div>` : ''}
                            </div>
                        </label>
                    `;
                });
                servicesHtml += '</div>';
                
                // CSS für Service-Cards mit Checkbox-Interaktion
                servicesHtml += `
                    <style>
                        .service-option-card input:checked + .service-card-content {
                            border-color: var(--primary) !important;
                            background: rgba(102, 126, 234, 0.05) !important;
                            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1) !important;
                        }
                        .service-option-card input:checked + .service-card-content .fa-check {
                            opacity: 1 !important;
                        }
                        .service-option-card:hover .service-card-content {
                            border-color: var(--primary);
                            transform: translateY(-1px);
                            box-shadow: 0 4px 12px rgba(0,0,0,0.08);
                        }
                    </style>
                `;
            } else {
                servicesHtml = '<div style="color: var(--text-secondary); font-size: 0.875rem; padding: 1rem; text-align: center; background: var(--bg-hover); border-radius: var(--radius);">Keine Serviceleistungen verfügbar</div>';
            }
            
            let tireInputs = '';
            if (newTires.length > 0) {
                tireInputs = newTires.map((tire, index) => `
                    <div class="form-group tire-input-group" style="border: 1px solid var(--border); padding: 1rem; border-radius: var(--radius); margin-bottom: 1rem;">
                        <div style="display: grid; grid-template-columns: 2fr 1fr 1fr 1fr 1fr; gap: 0.5rem;">
                            <input type="text" class="form-control tire-brand-model" placeholder="Hersteller, Profilbezeichnung" value="${tire.brand_model || ''}" style="font-size: 0.875rem;">
                            <input type="text" class="form-control tire-eu-label" placeholder="EU-Label" maxlength="10" value="${tire.eu_label || ''}" style="font-size: 0.875rem;">
                            <input type="text" class="form-control tire-dimension" placeholder="Dimension" value="${tire.dimension || ''}" style="font-size: 0.875rem;">
                            <input type="text" class="form-control tire-rf" placeholder="RF" maxlength="5" value="${tire.rf || ''}" style="font-size: 0.875rem;">
                            <input type="number" class="form-control tire-price" placeholder="Preis" step="0.01" min="0" value="${tire.price || 0}" style="font-size: 0.875rem;">
                        </div>
                        <div style="display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 0.5rem; margin-top: 0.5rem;">
                            <input type="number" class="form-control tire-mounting" placeholder="Montage Felge" step="0.01" min="0" value="${tire.mounting || 0}" style="font-size: 0.875rem;">
                            <input type="number" class="form-control tire-quantity" placeholder="Anzahl" min="1" value="${tire.quantity || 1}" style="font-size: 0.875rem;">
                            <input type="text" class="form-control tire-amount" placeholder="Betrag" readonly value="${parseFloat(tire.total || 0).toFixed(2).replace('.', ',')} €" style="font-size: 0.875rem; background: var(--bg-hover);">
                        </div>
                        <button type="button" class="btn btn-sm btn-secondary" onclick="removeTireInput(this)" style="margin-top: 0.5rem;">
                            <i class="fas fa-trash"></i> Entfernen
                        </button>
                    </div>
                `).join('');
            } else {
                // Keine Reifen vorhanden, zeige leeres Eingabefeld
                tireInputs = `
                    <div class="form-group tire-input-group" style="border: 1px solid var(--border); padding: 1rem; border-radius: var(--radius); margin-bottom: 1rem;">
                        <div style="display: grid; grid-template-columns: 2fr 1fr 1fr 1fr 1fr; gap: 0.5rem;">
                            <input type="text" class="form-control tire-brand-model" placeholder="Hersteller, Profilbezeichnung" style="font-size: 0.875rem;">
                            <input type="text" class="form-control tire-eu-label" placeholder="EU-Label" maxlength="10" style="font-size: 0.875rem;">
                            <input type="text" class="form-control tire-dimension" placeholder="Dimension" style="font-size: 0.875rem;">
                            <input type="text" class="form-control tire-rf" placeholder="RF" maxlength="5" style="font-size: 0.875rem;">
                            <input type="number" class="form-control tire-price" placeholder="Preis" step="0.01" min="0" style="font-size: 0.875rem;">
                        </div>
                        <div style="display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 0.5rem; margin-top: 0.5rem;">
                            <input type="number" class="form-control tire-mounting" placeholder="Montage Felge" step="0.01" min="0" style="font-size: 0.875rem;">
                            <input type="number" class="form-control tire-quantity" placeholder="Anzahl" min="1" value="1" style="font-size: 0.875rem;">
                            <input type="text" class="form-control tire-amount" placeholder="Betrag" readonly style="font-size: 0.875rem; background: var(--bg-hover);">
                        </div>
                        <button type="button" class="btn btn-sm btn-secondary" onclick="removeTireInput(this)" style="margin-top: 0.5rem;">
                            <i class="fas fa-trash"></i> Entfernen
                        </button>
                    </div>
                `;
            }
            
            const modalContent = `
                <form id="editOfferForm" style="max-height: 70vh; overflow-y: auto;">
                    <input type="hidden" id="editOfferId" value="${offerId}">
                    
                    <div class="form-group">
                        <label class="form-label">Kundendaten</label>
                        <div style="background: var(--bg-hover); padding: 1rem; border-radius: var(--radius); font-size: 0.875rem;">
                            <div><strong>Name:</strong> ${wheelset.customerName || wheelset.licenseNo || '-'}</div>
                            <div><strong>Kennzeichen:</strong> ${wheelset.licenseNo || '-'}</div>
                            <div><strong>Satznummer:</strong> ${wheelsetNumber}</div>
                            ${wheelset.customerNo ? `<div><strong>Kundennummer:</strong> ${wheelset.customerNo}</div>` : ''}
                            ${wheelset.VIN ? `<div><strong>FIN/VIN:</strong> ${wheelset.VIN}</div>` : ''}
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Neue Reifen</label>
                        <div id="editTireInputsContainer">
                            ${tireInputs}
                        </div>
                        <button type="button" class="btn btn-sm btn-secondary" onclick="addTireInput('editTireInputsContainer')" style="margin-top: 0.5rem;">
                            <i class="fas fa-plus"></i> Weitere Reifen hinzufügen
                        </button>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Zusatzleistungen</label>
                        <div id="editServicesContainer">
                            ${servicesHtml}
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Brieftext</label>
                        <textarea id="editOfferLetterContent" class="form-control" rows="8" style="font-size: 0.875rem;">${offer.letter_content || ''}</textarea>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Status</label>
                        <select id="editOfferStatus" class="form-control" style="font-size: 0.875rem;">
                            <option value="draft" ${offer.status === 'draft' ? 'selected' : ''}>Entwurf</option>
                            <option value="sent" ${offer.status === 'sent' ? 'selected' : ''}>Gesendet</option>
                        </select>
                    </div>
                    
                    <div style="display: flex; gap: 0.5rem; justify-content: flex-end; margin-top: 1.5rem;">
                        <button type="button" class="btn btn-secondary" onclick="closeModal('editOfferModal')">Abbrechen</button>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save"></i> Änderungen speichern
                        </button>
                    </div>
                </form>
            `;
            
            createModal('editOfferModal', `Angebot bearbeiten - Satz #${wheelsetNumber}`, modalContent);
            
            // Event Handler für Berechnung
            $('#editOfferForm').on('input', '.tire-price, .tire-mounting, .tire-quantity', function() {
                calculateTireAmount($(this).closest('.tire-input-group'));
            });
            
            // Form Submit Handler
            $('#editOfferForm').off('submit').on('submit', function(e) {
                e.preventDefault();
                saveEditedOffer(offerId, wheelsetNumber, wheelset, employeeName);
            });
        }
        
        function addTireInput(containerId = 'tireInputsContainer') {
            const html = `
                <div class="form-group tire-input-group" style="border: 1px solid var(--border); padding: 1rem; border-radius: var(--radius); margin-bottom: 1rem;">
                    <div style="display: grid; grid-template-columns: 2fr 1fr 1fr 1fr 1fr; gap: 0.5rem;">
                        <input type="text" class="form-control tire-brand-model" placeholder="Hersteller, Profilbezeichnung" style="font-size: 0.875rem;">
                        <input type="text" class="form-control tire-eu-label" placeholder="EU-Label" maxlength="10" style="font-size: 0.875rem;">
                        <input type="text" class="form-control tire-dimension" placeholder="Dimension" style="font-size: 0.875rem;">
                        <input type="text" class="form-control tire-rf" placeholder="RF" maxlength="5" style="font-size: 0.875rem;">
                        <input type="number" class="form-control tire-price" placeholder="Preis" step="0.01" min="0" style="font-size: 0.875rem;">
                    </div>
                    <div style="display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 0.5rem; margin-top: 0.5rem;">
                        <input type="number" class="form-control tire-mounting" placeholder="Montage Felge" step="0.01" min="0" style="font-size: 0.875rem;">
                        <input type="number" class="form-control tire-quantity" placeholder="Anzahl" min="1" value="1" style="font-size: 0.875rem;">
                        <input type="text" class="form-control tire-amount" placeholder="Betrag" readonly style="font-size: 0.875rem; background: var(--bg-hover);">
                    </div>
                    <button type="button" class="btn btn-sm btn-secondary" onclick="removeTireInput(this)" style="margin-top: 0.5rem;">
                        <i class="fas fa-trash"></i> Entfernen
                    </button>
                </div>
            `;
            $(`#${containerId}`).append(html);
            $(`#${containerId}`).on('input', '.tire-price, .tire-mounting, .tire-quantity', function() {
                calculateTireAmount($(this).closest('.tire-input-group'));
            });
        }
        
        function saveEditedOffer(offerId, wheelsetNumber, wheelset, employeeName) {
            const autohausId = $('#autohausSelect').val();
            
            // Sammle Reifen-Daten
            const newTires = [];
            $('#editTireInputsContainer .tire-input-group').each(function() {
                const brandModel = $(this).find('.tire-brand-model').val();
                const dimension = $(this).find('.tire-dimension').val();
                const price = parseFloat($(this).find('.tire-price').val()) || 0;
                const mounting = parseFloat($(this).find('.tire-mounting').val()) || 0;
                const quantity = parseInt($(this).find('.tire-quantity').val()) || 1;
                const euLabel = $(this).find('.tire-eu-label').val();
                const rf = $(this).find('.tire-rf').val();
                
                if (brandModel && dimension && (price > 0 || mounting > 0)) {
                    newTires.push({
                        brand_model: brandModel,
                        dimension: dimension,
                        eu_label: euLabel,
                        rf: rf,
                        price: price,
                        mounting: mounting,
                        quantity: quantity,
                        total: (price + mounting) * quantity
                    });
                }
            });
            
            // Sammle ausgewählte Services
            const selectedServices = [];
            $('#editServicesContainer .offer-service-checkbox:checked').each(function() {
                selectedServices.push(parseInt($(this).val()));
            });
            
            const letterContent = $('#editOfferLetterContent').val();
            const status = $('#editOfferStatus').val();
            
            // Berechne Gesamtsumme
            let totalAmount = 0;
            newTires.forEach(tire => {
                totalAmount += parseFloat(tire.total || 0);
            });
            selectedServices.forEach(serviceId => {
                const serviceCheckbox = $(`#editServicesContainer .offer-service-checkbox[value="${serviceId}"]`);
                const price = parseFloat(serviceCheckbox.data('price')) || 0;
                totalAmount += price;
            });
            
            showLoading(true);
            
            // Update Angebot
            fetch('../api/wheelset_offers.php?action=update', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    offer_id: offerId,
                    wheelset_number: wheelsetNumber,
                    autohaus_id: autohausId,
                    new_tires: newTires,
                    selected_services: selectedServices,
                    letter_content: letterContent,
                    status: status,
                    total_amount: totalAmount
                })
            })
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    if (data.success) {
                        showAlert('✅ Angebot erfolgreich aktualisiert', 'success');
                        closeModal('editOfferModal');
                        // Lade Details neu
                        viewDetails(wheelsetNumber);
                    } else {
                        showAlert('❌ Fehler: ' + (data.error || 'Unbekannt'), 'danger');
                    }
                })
                .catch(error => {
                    showLoading(false);
                    showAlert('Fehler: ' + error.message, 'danger');
                });
        }
        
        function resendOffer(offerId, wheelsetNumber) {
            const customerEmail = prompt('Bitte geben Sie die E-Mail-Adresse des Kunden ein:');
            
            if (!customerEmail || !customerEmail.includes('@')) {
                showAlert('Bitte geben Sie eine gültige E-Mail-Adresse ein', 'warning');
                return;
            }
            
            if (!confirm(`Möchten Sie das Angebot wirklich an ${customerEmail} versenden?`)) {
                return;
            }
            
            sendOfferEmail(offerId, customerEmail, wheelsetNumber);
        }
        
        function resendOfferFromList(offerId, wheelsetNumber) {
            const customerEmail = prompt('Bitte geben Sie die E-Mail-Adresse des Kunden ein:');
            
            if (!customerEmail || !customerEmail.includes('@')) {
                showAlert('Bitte geben Sie eine gültige E-Mail-Adresse ein', 'warning');
                return;
            }
            
            if (!confirm(`Möchten Sie das Angebot wirklich an ${customerEmail} versenden?`)) {
                return;
            }
            
            sendOfferEmail(offerId, customerEmail, wheelsetNumber, true);
        }
        
        function sendOfferEmail(offerId, customerEmail, wheelsetNumber, fromList = false) {
            showLoading(true);
            
            fetch('../api/wheelset_offers.php?action=send_email', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    offer_id: offerId,
                    recipient_email: customerEmail
                })
            })
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    if (data.success) {
                        showAlert('✅ Angebot erfolgreich per E-Mail versendet', 'success');
                        if (fromList) {
                            closeModal('offersListModal');
                        } else {
                            closeModal('offerDetailsModal');
                        }
                        // Lade Details neu
                        viewDetails(wheelsetNumber);
                    } else {
                        showAlert('❌ Fehler: ' + (data.error || 'Unbekannt'), 'danger');
                    }
                })
                .catch(error => {
                    showLoading(false);
                    showAlert('Fehler: ' + error.message, 'danger');
                });
        }
        
        function showCreateOfferModal(wheelsetNumber, wheelset) {
            const autohausId = $('#autohausSelect').val();
            const employeeName = '<?php echo htmlspecialchars($current_user_name, ENT_QUOTES, "UTF-8"); ?>';
            
            // Vorgefertigter Text
            const defaultLetterContent = `Sehr geehrte Damen und Herren,

anbei erhalten Sie Ihr Reifenangebot für den kommenden Boxenstop.

Bei Fragen stehen wir Ihnen gerne zur Verfügung.

Mit freundlichen Grüßen
${employeeName}`;
            
            showLoading(true);
            
            // Lade Serviceleistungen
            fetch(`../api/get_autohaus_services.php?autohaus_id=${autohausId}`)
                .then(response => {
                    if (!response.ok) {
                        throw new Error(`HTTP error! status: ${response.status}`);
                    }
                    return response.json();
                })
                .then(servicesData => {
                    showLoading(false);
                    
                    console.log('Services Response:', servicesData); // Debug
                    
                    const services = servicesData.success ? (servicesData.services || []) : [];
                    
                    let servicesHtml = '';
                    if (services.length > 0) {
                        servicesHtml = '<div style="display: grid; gap: 0.75rem;">';
                        services.forEach(service => {
                            servicesHtml += `
                                <label class="service-option-card" style="display: block; cursor: pointer; transition: all 0.3s;">
                                    <input type="checkbox" class="offer-service-checkbox" value="${service.id}" data-price="${service.price}" style="position: absolute; opacity: 0; pointer-events: none;">
                                    <div class="service-card-content" style="display: flex; align-items: center; padding: 1rem; border: 2px solid var(--border); border-radius: 12px; background: var(--bg-secondary); transition: all 0.3s; position: relative; overflow: hidden;">
                                        <div style="width: 48px; height: 48px; border-radius: 10px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); display: flex; align-items: center; justify-content: center; margin-right: 1rem; flex-shrink: 0;">
                                            <i class="fas fa-check" style="color: white; font-size: 1.25rem; opacity: 0; transition: opacity 0.2s;"></i>
                                        </div>
                                        <div style="flex: 1; min-width: 0;">
                                            <div style="font-weight: 600; color: var(--text-primary); font-size: 0.9375rem; margin-bottom: 0.25rem;">${service.service_name}</div>
                                            ${service.service_description ? `<div style="font-size: 0.8125rem; color: var(--text-secondary); line-height: 1.4;">${service.service_description}</div>` : ''}
                                        </div>
                                        ${service.show_price ? `<div style="font-weight: 700; color: var(--primary); font-size: 1.0625rem; margin-left: 1rem; white-space: nowrap;">${parseFloat(service.price).toFixed(2).replace('.', ',')} €</div>` : ''}
                                    </div>
                                </label>
                            `;
                        });
                        servicesHtml += '</div>';
                        
                        // CSS für Service-Cards mit Checkbox-Interaktion
                        servicesHtml += `
                            <style>
                                .service-option-card input:checked + .service-card-content {
                                    border-color: var(--primary) !important;
                                    background: rgba(102, 126, 234, 0.05) !important;
                                    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
                                }
                                .service-option-card input:checked + .service-card-content .fa-check {
                                    opacity: 1 !important;
                                }
                                .service-option-card:hover .service-card-content {
                                    border-color: var(--primary);
                                    transform: translateY(-1px);
                                    box-shadow: 0 4px 12px rgba(0,0,0,0.08);
                                }
                            </style>
                        `;
                    } else {
                        servicesHtml = '<div style="color: var(--text-secondary); font-size: 0.875rem; padding: 1rem; text-align: center; background: var(--bg-hover); border-radius: var(--radius);">Keine Serviceleistungen verfügbar</div>';
                    }
                    
                    // Starte mit leerer Liste - "Neue Reifen" soll initial leer sein
                    // Füge einen ersten leeren Reifen-Eintrag hinzu
                    let tireInputs = `
                        <div class="form-group tire-input-group" style="border: 1px solid var(--border); padding: 1rem; border-radius: var(--radius); margin-bottom: 1rem;">
                            <div style="display: grid; grid-template-columns: 2fr 1fr 1fr 1fr 1fr; gap: 0.5rem;">
                                <input type="text" class="form-control tire-brand-model" placeholder="Hersteller, Profilbezeichnung" style="font-size: 0.875rem;">
                                <input type="text" class="form-control tire-eu-label" placeholder="EU-Label" maxlength="10" style="font-size: 0.875rem;">
                                <input type="text" class="form-control tire-dimension" placeholder="Dimension" style="font-size: 0.875rem;">
                                <input type="text" class="form-control tire-rf" placeholder="RF" maxlength="5" style="font-size: 0.875rem;">
                                <input type="number" class="form-control tire-price" placeholder="Preis" step="0.01" min="0" style="font-size: 0.875rem;">
                            </div>
                            <div style="display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 0.5rem; margin-top: 0.5rem;">
                                <input type="number" class="form-control tire-mounting" placeholder="Montage Felge" step="0.01" min="0" style="font-size: 0.875rem;">
                                <input type="number" class="form-control tire-quantity" placeholder="Anzahl" min="1" value="1" style="font-size: 0.875rem;">
                                <input type="text" class="form-control tire-amount" placeholder="Betrag" readonly style="font-size: 0.875rem; background: var(--bg-hover);">
                            </div>
                            <button type="button" class="btn btn-sm btn-secondary" onclick="removeTireInput(this)" style="margin-top: 0.5rem;">
                                <i class="fas fa-trash"></i> Entfernen
                            </button>
                        </div>
                    `;
                    
                    
                    const modalContent = `
                        <form id="createOfferForm" style="max-height: 70vh; overflow-y: auto;">
                            <div class="form-group">
                                <label class="form-label">Kundendaten</label>
                                <div style="background: var(--bg-hover); padding: 1rem; border-radius: var(--radius); font-size: 0.875rem;">
                                    <div><strong>Name:</strong> ${wheelset.customerName || wheelset.licenseNo || '-'}</div>
                                    <div><strong>Kennzeichen:</strong> ${wheelset.licenseNo || '-'}</div>
                                    <div><strong>Satznummer:</strong> ${wheelsetNumber}</div>
                                    ${wheelset.customerNo ? `<div><strong>Kundennummer:</strong> ${wheelset.customerNo}</div>` : ''}
                                    ${wheelset.VIN ? `<div><strong>FIN/VIN:</strong> ${wheelset.VIN}</div>` : ''}
                                </div>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">Neue Reifen</label>
                                <div id="tireInputsContainer">
                                    ${tireInputs}
                                </div>
                                <button type="button" class="btn btn-sm btn-secondary" onclick="addTireInput()" style="margin-top: 0.5rem;">
                                    <i class="fas fa-plus"></i> Weitere Reifen hinzufügen
                                </button>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">Zusatzleistungen</label>
                                <div id="servicesContainer">
                                    ${servicesHtml}
                                </div>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">Brieftext</label>
                                <textarea id="offerLetterContent" class="form-control" rows="8" style="font-size: 0.875rem;">${defaultLetterContent}</textarea>
                            </div>
                        </form>
                    `;
                    
                    createModal('createOfferModal', `Angebot erstellen - Satz #${wheelsetNumber}`, modalContent);
                    
                    // Passe Header an: Füge E-Mail-Feld und Submit-Button hinzu
                    const modalHeader = $(`#createOfferModal .modal-header`);
                    const modalTitle = modalHeader.find('.modal-title');
                    
                    // Ändere Header-Layout zu Column für zwei Zeilen
                    modalHeader.css({
                        'display': 'flex',
                        'flex-direction': 'column',
                        'align-items': 'stretch',
                        'gap': '1rem',
                        'padding': '1.5rem'
                    });
                    
                    // Erste Zeile: Titel + Buttons
                    const headerTopRow = $(`
                        <div style="display: flex; align-items: center; justify-content: space-between; width: 100%;">
                            <div class="modal-title" style="flex-grow: 1;">Angebot erstellen - Satz #${wheelsetNumber}</div>
                            <div style="display: flex; align-items: center; gap: 0.5rem;">
                                <button type="submit" form="createOfferForm" class="btn btn-primary" style="white-space: nowrap;">
                                    <i class="fas fa-envelope"></i> Angebot senden + speichern
                                </button>
                                <button class="modal-close" onclick="closeModal('createOfferModal')" style="margin-left: 0.5rem;">&times;</button>
                            </div>
                        </div>
                    `);
                    
                    // Zweite Zeile: E-Mail-Feld
                    const headerEmailRow = $(`
                        <div style="display: flex; flex-direction: column; gap: 0.25rem;">
                            <label style="font-size: 0.875rem; font-weight: 500; color: var(--text-secondary);">E-Mail-Adresse des Kunden</label>
                            <input type="email" id="offerCustomerEmail" form="createOfferForm" class="form-control" placeholder="kunde@beispiel.de" required style="font-size: 0.875rem;">
                            <small style="color: var(--text-secondary); font-size: 0.75rem;">
                                Das Angebot wird per E-Mail an diese Adresse gesendet.
                            </small>
                        </div>
                    `);
                    
                    // Ersetze Header-Inhalt
                    modalHeader.html('');
                    modalHeader.append(headerTopRow);
                    modalHeader.append(headerEmailRow);
                    
                    // Verstecke den Footer, da keine Buttons mehr benötigt werden
                    $(`#createOfferModal .modal-footer`).hide();
                    
                    // Event Handler für Berechnung
                    $('#createOfferForm').on('input', '.tire-price, .tire-mounting, .tire-quantity', function() {
                        calculateTireAmount($(this).closest('.tire-input-group'));
                    });
                    
                    // Form Submit Handler
                    $('#createOfferForm').off('submit').on('submit', function(e) {
                        e.preventDefault();
                        saveOffer(wheelsetNumber, wheelset, employeeName);
                    });
                    
                    // Lade E-Mail-Adresse aus appointments automatisch
                    loadAppointmentEmail(wheelsetNumber, wheelset, autohausId);
                })
                .catch(error => {
                    showLoading(false);
                    console.error('Error loading services:', error);
                    showAlert('Fehler beim Laden der Serviceleistungen: ' + error.message, 'danger');
                    // Zeige Modal trotzdem an, aber ohne Services
                    const servicesHtml = '<div style="color: var(--danger); font-size: 0.875rem; padding: 1rem; text-align: center; background: var(--bg-hover); border-radius: var(--radius);">Fehler beim Laden der Serviceleistungen</div>';
                    // ... Modal-Inhalt würde hier weitergehen, aber wir haben bereits die Funktion
                });
        }
        
        function addTireInput() {
            const html = `
                <div class="form-group tire-input-group" style="border: 1px solid var(--border); padding: 1rem; border-radius: var(--radius); margin-bottom: 1rem;">
                    <div style="display: grid; grid-template-columns: 2fr 1fr 1fr 1fr 1fr; gap: 0.5rem;">
                        <input type="text" class="form-control tire-brand-model" placeholder="Hersteller, Profilbezeichnung" style="font-size: 0.875rem;">
                        <input type="text" class="form-control tire-eu-label" placeholder="EU-Label" maxlength="10" style="font-size: 0.875rem;">
                        <input type="text" class="form-control tire-dimension" placeholder="Dimension" style="font-size: 0.875rem;">
                        <input type="text" class="form-control tire-rf" placeholder="RF" maxlength="5" style="font-size: 0.875rem;">
                        <input type="number" class="form-control tire-price" placeholder="Preis" step="0.01" min="0" style="font-size: 0.875rem;">
                    </div>
                    <div style="display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 0.5rem; margin-top: 0.5rem;">
                        <input type="number" class="form-control tire-mounting" placeholder="Montage Felge" step="0.01" min="0" style="font-size: 0.875rem;">
                        <input type="number" class="form-control tire-quantity" placeholder="Anzahl" min="1" value="1" style="font-size: 0.875rem;">
                        <input type="text" class="form-control tire-amount" placeholder="Betrag" readonly style="font-size: 0.875rem; background: var(--bg-hover);">
                    </div>
                    <button type="button" class="btn btn-sm btn-secondary" onclick="removeTireInput(this)" style="margin-top: 0.5rem;">
                        <i class="fas fa-trash"></i> Entfernen
                    </button>
                </div>
            `;
            $('#tireInputsContainer').append(html);
            $('#tireInputsContainer').on('input', '.tire-price, .tire-mounting, .tire-quantity', function() {
                calculateTireAmount($(this).closest('.tire-input-group'));
            });
        }
        
        function removeTireInput(button) {
            $(button).closest('.tire-input-group').remove();
        }
        
        function calculateTireAmount(container) {
            const price = parseFloat($(container).find('.tire-price').val()) || 0;
            const mounting = parseFloat($(container).find('.tire-mounting').val()) || 0;
            const quantity = parseInt($(container).find('.tire-quantity').val()) || 1;
            const total = (price + mounting) * quantity;
            $(container).find('.tire-amount').val(total.toFixed(2).replace('.', ',') + ' €');
        }
        
        /**
         * Lade E-Mail-Adresse aus appointments basierend auf Einlagerungsnummer oder Kennzeichen
         */
        function loadAppointmentEmail(wheelsetNumber, wheelset, autohausId) {
            const licensePlate = wheelset.licenseNo || wheelset.kennzeichen || '';
            
            // Baue URL-Parameter
            let url = `../api/get_appointment_email.php?autohaus_id=${autohausId}`;
            if (wheelsetNumber) {
                url += `&wheelset_number=${encodeURIComponent(wheelsetNumber)}`;
            }
            if (licensePlate) {
                url += `&license_plate=${encodeURIComponent(licensePlate)}`;
            }
            
            fetch(url)
                .then(response => response.json())
                .then(data => {
                    if (data.success && data.email) {
                        // E-Mail automatisch eintragen
                        const emailInput = $('#offerCustomerEmail');
                        if (emailInput.length && !emailInput.val()) {
                            emailInput.val(data.email);
                            
                            // Info-Meldung anzeigen
                            const sourceText = data.source === 'storage_number' ? 'Einlagerungsnummer' : 'Kennzeichen';
                            const customerName = data.customer_name ? ` (${data.customer_name})` : '';
                            
                            showAlert(
                                `✅ Daten übernommen: E-Mail-Adresse wurde aus dem Termin ${sourceText === 'Einlagerungsnummer' ? 'über die Einlagerungsnummer' : 'über das Kennzeichen'} geladen${customerName}`,
                                'info',
                                5000
                            );
                        }
                    }
                })
                .catch(error => {
                    // Fehler wird stillschweigend ignoriert - kein Alert nötig, da optional
                    console.log('Keine E-Mail aus Terminen gefunden:', error.message);
                });
        }
        
        function saveOffer(wheelsetNumber, wheelset, employeeName) {
            const autohausId = $('#autohausSelect').val();
            const customerEmail = $('#offerCustomerEmail').val();
            
            if (!customerEmail) {
                showAlert('Bitte geben Sie eine E-Mail-Adresse ein', 'warning');
                return;
            }
            
            // Sammle Reifen-Daten
            const newTires = [];
            $('#tireInputsContainer .tire-input-group').each(function() {
                const brandModel = $(this).find('.tire-brand-model').val();
                const dimension = $(this).find('.tire-dimension').val();
                const price = parseFloat($(this).find('.tire-price').val()) || 0;
                const mounting = parseFloat($(this).find('.tire-mounting').val()) || 0;
                const quantity = parseInt($(this).find('.tire-quantity').val()) || 1;
                const euLabel = $(this).find('.tire-eu-label').val();
                const rf = $(this).find('.tire-rf').val();
                
                // Nur vollständig ausgefüllte Reifen hinzufügen
                if (brandModel && dimension && (price > 0 || mounting > 0)) {
                    newTires.push({
                        brand_model: brandModel,
                        dimension: dimension,
                        eu_label: euLabel,
                        rf: rf,
                        price: price,
                        mounting: mounting,
                        quantity: quantity,
                        total: (price + mounting) * quantity
                    });
                }
            });
            
            // Sammle ausgewählte Services
            const selectedServices = [];
            $('.offer-service-checkbox:checked').each(function() {
                selectedServices.push(parseInt($(this).val()));
            });
            
            // Debug: Log Reifen-Daten
            console.log('Save Offer - Reifen:', newTires);
            console.log('Save Offer - Services:', selectedServices);
            
            // Prüfe ob mindestens Reifen ODER Services vorhanden sind
            if (newTires.length === 0 && selectedServices.length === 0) {
                showAlert('Bitte geben Sie mindestens einen Reifen ein oder wählen Sie eine Serviceleistung aus', 'warning');
                return;
            }
            
            const letterContent = $('#offerLetterContent').val();
            
            // Berechne Gesamtbetrag (für Debug)
            let totalAmount = 0;
            newTires.forEach(tire => {
                totalAmount += tire.total || 0;
            });
            console.log('Save Offer - Gesamtbetrag (Reifen):', totalAmount);
            
            showLoading(true);
            
            // Erstelle Angebot
            const requestData = {
                wheelset_number: wheelsetNumber,
                autohaus_id: autohausId,
                customer_name: wheelset.customerName || wheelset.licenseNo || '',
                customer_no: wheelset.customerNo || '',
                license_plate: wheelset.licenseNo || '',
                vin: wheelset.VIN || '',
                new_tires: newTires,
                selected_services: selectedServices,
                letter_content: letterContent,
                employee_name: employeeName
            };
            
            console.log('Save Offer - Request Data:', requestData);
            
            fetch('../api/wheelset_offers.php?action=create', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(requestData)
            })
                .then(async response => {
                    const contentType = response.headers.get('content-type');
                    if (!contentType || !contentType.includes('application/json')) {
                        const text = await response.text();
                        console.error('Non-JSON Response:', text);
                        throw new Error('Server-Antwort ist kein gültiges JSON. Möglicherweise ein PHP-Fehler.');
                    }
                    return response.json();
                })
                .then(data => {
                    if (data.success) {
                        // Versende Email
                        return fetch('../api/wheelset_offers.php?action=send_email', {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/json' },
                            body: JSON.stringify({
                                offer_id: data.offer_id,
                                recipient_email: customerEmail
                            })
                        });
                    } else {
                        throw new Error(data.error || 'Fehler beim Erstellen des Angebots');
                    }
                })
                .then(async response => {
                    const contentType = response.headers.get('content-type');
                    if (!contentType || !contentType.includes('application/json')) {
                        const text = await response.text();
                        console.error('Non-JSON Response (send_email):', text);
                        // Versuche trotzdem JSON zu parsen, falls es funktioniert hat
                        try {
                            return JSON.parse(text);
                        } catch (e) {
                            // Wenn Email trotzdem gesendet wurde, behandele es als Erfolg
                            return { success: true, message: 'E-Mail wurde versendet (Details siehe Server-Log)' };
                        }
                    }
                    return response.json();
                })
                .then(data => {
                    showLoading(false);
                    if (data.success) {
                        showAlert('✅ Angebot erfolgreich erstellt und per E-Mail versendet', 'success');
                        closeModal('createOfferModal');
                        // Lade Details neu
                        viewDetails(wheelsetNumber);
                    } else {
                        showAlert('❌ Fehler: ' + (data.error || 'Unbekannt'), 'danger');
                    }
                })
                .catch(error => {
                    showLoading(false);
                    console.error('Save Offer Error:', error);
                    showAlert('Fehler: ' + error.message, 'danger');
                });
        }

        function openDetailPanel(wheelset) {
            const panel = $('#detailPanel');
            const body = $('#panelBody');
            
            // Update Header Icons
            updatePanelHeaderIcons(wheelset);
            
            let html = `
                <div class="detail-section">
                    <div class="detail-title">Grundinformationen</div>
                    <div class="detail-grid">
                        <div class="detail-row">
                            <span class="detail-label">Satznummer</span>
                            <span class="detail-value">${wheelset.wheelSetNumber || '-'}</span>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label">Kennzeichen</span>
                            <span class="detail-value">${wheelset.licenseNo || '-'}</span>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label">FIN/VIN</span>
                            <span class="detail-value">${wheelset.VIN || '-'}</span>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label">Kundennummer</span>
                            <span class="detail-value">${wheelset.customerNo || '-'}</span>
                        </div>
                    </div>
                </div>

                <div class="detail-section">
                    <div class="detail-title">Service & Lager</div>
                    <div class="detail-grid">
                        <div class="detail-row">
                            <span class="detail-label">Service-Paket</span>
                            <span class="detail-value">${wheelset.servicePackageName || '-'}</span>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label">Lagerplatz</span>
                            <span class="detail-value">${wheelset.storagePlace || '-'}</span>
                        </div>
                        ${wheelset.testReportURL ? `
                        <div class="detail-row">
                            <span class="detail-label">Prüfbericht</span>
                            <span class="detail-value">
                                <button class="btn btn-sm btn-primary" onclick="viewTestReport('${wheelset.testReportURL}', '${wheelset.wheelSetNumber}')" style="padding: 0.375rem 0.75rem; font-size: 0.8125rem;">
                                    <i class="fas fa-eye"></i> Ansehen
                                </button>
                                <a href="${wheelset.testReportURL}" target="_blank" class="btn btn-sm btn-secondary" style="padding: 0.375rem 0.75rem; font-size: 0.8125rem; margin-left: 0.5rem; text-decoration: none;">
                                    <i class="fas fa-download"></i> Download
                                </a>
                            </span>
                        </div>
                        ` : ''}
                    </div>
                </div>
            `;

            if (wheelset.wheels && wheelset.wheels.length > 0) {
                // Organisiere Wheels nach Position (VL, VR, HL, HR)
                const wheelsByPosition = {
                    'VL': null,
                    'VR': null,
                    'HL': null,
                    'HR': null
                };
                
                wheelset.wheels.forEach(wheel => {
                    const pos = (wheel.position || '').toUpperCase().trim();
                    if (pos === 'VL' || pos === 'VR' || pos === 'HL' || pos === 'HR') {
                        wheelsByPosition[pos] = wheel;
                    } else {
                        // Fallback: Zuordnung nach Index falls Position nicht klar
                        const positions = ['VL', 'VR', 'HL', 'HR'];
                        const idx = wheelset.wheels.indexOf(wheel);
                        if (idx < 4 && !wheelsByPosition[positions[idx]]) {
                            wheelsByPosition[positions[idx]] = wheel;
                        }
                    }
                });
                
                // Berechne Status für jedes Rad basierend auf Profiltiefe
                const getWheelStatus = (wheel) => {
                    if (!wheel || wheel.profile === null || wheel.profile === undefined) {
                        return 'default';
                    }
                    const profile = parseFloat(wheel.profile);
                    if (profile < 1.6) return 'danger';
                    if (profile < 3) return 'warning';
                    return 'ok';
                };
                
                // Berechne Profiltiefen (I, M, A)
                const getProfileMeasurements = (wheel) => {
                    if (!wheel || !wheel.profile) return null;
                    const avgProfile = parseFloat(wheel.profile);
                    return {
                        i: avgProfile,
                        m: avgProfile,
                        a: avgProfile
                    };
                };
                
                const getMeasureStatus = (value) => {
                    if (value === null || value === undefined) return 'default';
                    if (value < 1.6) return 'danger';
                    if (value < 3) return 'warning';
                    return 'success';
                };
                
                const getRecommendation = (wheel) => {
                    if (!wheel || !wheel.profile) return 'In Ordnung';
                    const profile = parseFloat(wheel.profile);
                    if (profile < 1.6) return 'Austausch notwendig';
                    if (profile < 3) return 'Prüfung empfohlen';
                    return 'In Ordnung';
                };
                
                const getRecommendationShort = (wheel) => {
                    if (!wheel || !wheel.profile) return 'OK';
                    const profile = parseFloat(wheel.profile);
                    if (profile < 1.6) return 'Austausch';
                    if (profile < 3) return 'Prüfung';
                    return 'OK';
                };
                
                // Kompakte Reifen-Übersicht im Detail-Panel
                html += `
                    <div class="detail-section">
                        <div class="detail-title">Reifen (${wheelset.wheels.length})</div>
                        <div class="wheels-overview-grid">
                            ${['VL', 'VR', 'HL', 'HR'].map(pos => {
                                const wheel = wheelsByPosition[pos];
                                if (!wheel) {
                                    return `
                                        <div class="wheel-overview-item" style="opacity: 0.5;">
                                            <div class="wheel-overview-position">${pos}</div>
                                            <div class="wheel-overview-status" style="color: var(--text-muted);">
                                                <i class="fas fa-minus-circle"></i> Keine Daten
                                            </div>
                                        </div>
                                    `;
                                }
                                
                                const status = getWheelStatus(wheel);
                                const statusClass = status === 'danger' ? 'danger' : status === 'warning' ? 'warning' : 'ok';
                                const recommendation = getRecommendationShort(wheel);
                                const profile = wheel.profile ? parseFloat(wheel.profile).toFixed(1) : '-';
                                const tireType = wheel.tireType || 'UNKNOWN';
                                
                                const wheelsetNumber = wheelset.wheelSetNumber || wheelset.satznummer;
                                return `
                                    <div class="wheel-overview-item status-${statusClass}" onclick="showWheelDetailsModal('${wheelsetNumber}', '${pos}')" style="cursor: pointer;">
                                        <div class="wheel-overview-header">
                                            <div class="wheel-overview-position">${pos}</div>
                                            <div class="wheel-overview-status status-${statusClass}">
                                                ${status === 'ok' ? '<i class="fas fa-check-circle"></i>' : status === 'warning' ? '<i class="fas fa-exclamation-circle"></i>' : '<i class="fas fa-times-circle"></i>'}
                                            </div>
                                        </div>
                                        <div class="wheel-overview-info">
                                            <div class="wheel-overview-brand">${wheel.manufacturer || 'Unbekannt'} ${wheel.model || ''}</div>
                                            <div class="wheel-overview-details">
                                                <span><i class="fas fa-ruler"></i> ${profile} mm</span>
                                                <span class="wheel-overview-recommendation ${statusClass}">${recommendation}</span>
                                            </div>
                                        </div>
                                    </div>
                                `;
                            }).join('')}
                        </div>
                    </div>
                `;
            }

            // Kommentar wird nicht mehr hier angezeigt, sondern nur über das Icon
            // Order-Informationen werden jetzt über Icons im Header verwaltet

            body.html(html);
            panel.addClass('open');
            
            // Prüfe ob Orders Panel auch offen ist
            const ordersPanel = $('#ordersPanel');
            if (ordersPanel.hasClass('open')) {
                panel.addClass('orders-panel-active');
                $('#panelSeparator').addClass('active');
            }
            
            updateQuickActionsPosition();
        }

        function closeDetailPanel() {
            $('#detailPanel').removeClass('open orders-panel-active');
            $('#panelSeparator').removeClass('active');
            updateQuickActionsPosition();
        }
        
        function showWheelDetailsModal(wheelsetNumber, position) {
            // Finde das Wheelset in wheelsetsData
            const wheelset = wheelsetsData.find(ws => ws.satznummer == wheelsetNumber || ws.wheelSetNumber == wheelsetNumber);
            if (!wheelset) {
                showAlert('Reifensatz nicht gefunden', 'error');
                return;
            }
            
            // Wenn wheels noch nicht geladen sind, lade sie nach
            if (!wheelset.wheels && !wheelset.resy_wheels) {
                // Versuche Details zu laden
                fetch(`../api/resy_api_routes.php?action=get_wheelset_details&wheelset_number=${wheelsetNumber}`)
                    .then(response => response.json())
                    .then(data => {
                        if (data.success && data.wheelSet && data.wheelSet.wheels) {
                            wheelset.wheels = data.wheelSet.wheels;
                            showWheelDetailsModal(wheelsetNumber, position); // Rekursiver Aufruf mit geladenen Daten
                        } else {
                            showAlert('Reifendaten konnten nicht geladen werden', 'error');
                        }
                    })
                    .catch(error => {
                        showAlert('Fehler beim Laden der Reifendaten', 'error');
                    });
                return;
            }
            
            // Verwende resy_wheels falls vorhanden
            if (!wheelset.wheels && wheelset.resy_wheels) {
                wheelset.wheels = wheelset.resy_wheels;
            }
            
            // Organisiere Wheels nach Position
            const wheelsByPosition = {
                'VL': null,
                'VR': null,
                'HL': null,
                'HR': null
            };
            
            wheelset.wheels.forEach(wheel => {
                const pos = (wheel.position || '').toUpperCase().trim();
                if (pos === 'VL' || pos === 'VR' || pos === 'HL' || pos === 'HR') {
                    wheelsByPosition[pos] = wheel;
                } else {
                    const positions = ['VL', 'VR', 'HL', 'HR'];
                    const idx = wheelset.wheels.indexOf(wheel);
                    if (idx < 4 && !wheelsByPosition[positions[idx]]) {
                        wheelsByPosition[positions[idx]] = wheel;
                    }
                }
            });
            
            // Helper-Funktionen
            const getWheelStatus = (wheel) => {
                if (!wheel || wheel.profile === null || wheel.profile === undefined) return 'default';
                const profile = parseFloat(wheel.profile);
                if (profile < 1.6) return 'danger';
                if (profile < 3) return 'warning';
                return 'ok';
            };
            
            const getProfileMeasurements = (wheel) => {
                if (!wheel || !wheel.profile) return null;
                const avgProfile = parseFloat(wheel.profile);
                return { i: avgProfile, m: avgProfile, a: avgProfile };
            };
            
            const getMeasureStatus = (value) => {
                if (value === null || value === undefined) return 'default';
                if (value < 1.6) return 'danger';
                if (value < 3) return 'warning';
                return 'success';
            };
            
            const getRecommendation = (wheel) => {
                if (!wheel || !wheel.profile) return 'In Ordnung';
                const profile = parseFloat(wheel.profile);
                if (profile < 1.6) return 'Austausch notwendig';
                if (profile < 3) return 'Prüfung empfohlen';
                return 'In Ordnung';
            };
            
            // Modal-Content generieren
            let modalContent = `
                <div style="width: 100%; height: 100%; overflow: visible; margin: 0 auto;">
                    <div class="wheels-car-layout">
                        <div class="car-container">
                            <!-- Linke Seite: VL und HL -->
                            <div class="wheels-panels-side left">
                                <div class="wheels-panels-grid">
                                    ${['VL', 'HL'].map(pos => {
                                    const wheel = wheelsByPosition[pos];
                                    const isSelected = pos === position;
                                    
                                    // Prüfe ob dieser Reifen ausgewählt ist
                                    const wheelKey = `${wheelsetNumber}-${pos}`;
                                    const isWheelSelected = selectedWheels.some(sw => sw.key === wheelKey);
                                    
                                    if (!wheel) {
                                        return `
                                        <div class="wheel-position-panel ${pos.toLowerCase()} selectable ${isWheelSelected ? 'selected' : ''}" 
                                             data-wheel-key="${wheelKey}"
                                             data-position="${pos}"
                                             data-wheelset-number="${wheelsetNumber}"
                                             onclick="toggleWheelSelection('${wheelKey}', '${pos}', '${wheelsetNumber}')"
                                             style="opacity: 0.4; ${isSelected ? 'border-width: 4px; border-color: var(--primary);' : ''}">
                                            <div class="wheel-panel-header">
                                                <div class="wheel-panel-position">
                                                    <span class="wheel-panel-position-icon">${pos.charAt(1) === 'L' ? 'L' : 'R'}</span>
                                                    ${pos}
                                                </div>
                                                <div class="wheel-panel-status" style="background: var(--border); color: var(--text-muted);">
                                                    <i class="fas fa-minus"></i>
                                                </div>
                                            </div>
                                            <div style="text-align: center; color: var(--text-muted); font-size: 0.875rem; padding: 2rem 0;">
                                                Keine Daten
                                            </div>
                                        </div>
                                    `;
                                }
                                
                                const status = getWheelStatus(wheel);
                                const measurements = getProfileMeasurements(wheel);
                                const tireType = wheel.tireType || 'UNKNOWN';
                                const tireTypeIcon = tireType === 'WINTER' ? '<i class="fas fa-snowflake"></i>' : 
                                                     tireType === 'SUMMER' ? '<i class="fas fa-sun"></i>' : 
                                                     tireType === 'ALLSEASON' ? '<i class="fas fa-globe"></i>' : 
                                                     '<i class="fas fa-circle"></i>';
                                const statusIcon = status === 'danger' ? '<i class="fas fa-times-circle"></i>' : 
                                                   status === 'warning' ? '<i class="fas fa-exclamation-circle"></i>' : 
                                                   '<i class="fas fa-check-circle"></i>';
                                const statusClass = status === 'danger' ? 'danger' : status === 'warning' ? 'warning' : 'ok';
                                const recommendation = getRecommendation(wheel);
                                const recClass = recommendation.includes('Austausch') ? 'danger' : recommendation.includes('Prüfung') ? 'warning' : 'ok';
                                
                                // Profiltiefe und Druck immer anzeigen (mit Fallback)
                                const profileValue = wheel.profile !== null && wheel.profile !== undefined ? parseFloat(wheel.profile).toFixed(1) : '-';
                                const pressureValue = wheel.pressure !== null && wheel.pressure !== undefined ? parseFloat(wheel.pressure).toFixed(2) : '-';
                                const profileStatus = measurements ? getMeasureStatus(measurements.m || measurements.i || measurements.a || 0) : 'default';
                                
                                // Prüfe ob dieser Reifen ausgewählt ist (linke Seite)
                                const wheelKeyLeft = `${wheelsetNumber}-${pos}`;
                                const isWheelSelectedLeft = selectedWheels.some(sw => sw.key === wheelKeyLeft);
                                
                                return `
                                    <div class="wheel-position-panel ${pos.toLowerCase()} status-${statusClass} selectable ${isWheelSelectedLeft ? 'selected' : ''}" 
                                         data-wheel-key="${wheelKeyLeft}"
                                         data-position="${pos}"
                                         data-wheelset-number="${wheelsetNumber}"
                                         onclick="toggleWheelSelection('${wheelKeyLeft}', '${pos}', '${wheelsetNumber}')"
                                         style="${isSelected ? 'border-width: 4px; box-shadow: 0 0 20px rgba(59, 130, 246, 0.5);' : ''}">
                                        <div class="wheel-panel-header">
                                            <div class="wheel-panel-position">
                                                <span class="wheel-panel-position-icon">${pos.charAt(1) === 'L' ? 'L' : 'R'}</span>
                                                ${pos}
                                            </div>
                                            <div class="wheel-panel-status ${statusClass}">${statusIcon}</div>
                                        </div>
                                        
                                        <!-- Profiltiefe und Druck - Immer sichtbar oben -->
                                        <div class="wheel-critical-info">
                                            <div class="critical-info-row">
                                                <div class="critical-info-item">
                                                    <div class="critical-info-label"><i class="fas fa-ruler"></i> Profiltiefe</div>
                                                    <div class="critical-info-value ${profileStatus}">${profileValue} mm</div>
                                                </div>
                                                <div class="critical-info-item">
                                                    <div class="critical-info-label"><i class="fas fa-tachometer-alt"></i> Druck</div>
                                                    <div class="critical-info-value ${pressureValue !== '-' ? 'ok' : 'default'}">${pressureValue} ${pressureValue !== '-' ? 'bar' : ''}</div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        ${measurements && measurements.i && measurements.m && measurements.a ? `
                                        <div class="wheel-profile-measurements">
                                            <div class="profile-measures-grid">
                                                <div class="profile-measure-item">
                                                    <div class="profile-measure-label">I</div>
                                                    <div class="profile-measure-value ${getMeasureStatus(measurements.i)}">${measurements.i.toFixed(1)}</div>
                                                </div>
                                                <div class="profile-measure-item">
                                                    <div class="profile-measure-label">M</div>
                                                    <div class="profile-measure-value ${getMeasureStatus(measurements.m)}">${measurements.m.toFixed(1)}</div>
                                                </div>
                                                <div class="profile-measure-item">
                                                    <div class="profile-measure-label">A</div>
                                                    <div class="profile-measure-value ${getMeasureStatus(measurements.a)}">${measurements.a.toFixed(1)}</div>
                                                </div>
                                            </div>
                                        </div>
                                        ` : ''}
                                        
                                        <div class="wheel-recommendation-badge ${recClass}">
                                            ${recClass === 'ok' ? '<i class="fas fa-check"></i>' : recClass === 'warning' ? '<i class="fas fa-exclamation-triangle"></i>' : '<i class="fas fa-times"></i>'} ${recommendation}
                                        </div>
                                        
                                        <div class="wheel-panel-info">
                                            <div class="wheel-info-section compact">
                                                <div class="wheel-panel-brand">
                                                    ${wheel.manufacturer || 'Unbekannt'} ${wheel.model || ''}
                                                </div>
                                                <div class="wheel-panel-size-type">
                                                    <span class="wheel-panel-size">${wheel.width || '-'}/${wheel.ratio || '-'}R${wheel.diameter || '-'}</span>
                                                    <span class="wheel-panel-type">${tireTypeIcon} ${tireType}</span>
                                                </div>
                                            </div>
                                            
                                            ${(wheel.speedIndx || wheel.loadIndx1 || wheel.rimType) ? `
                                            <div class="wheel-info-section compact">
                                                <div class="wheel-additional-info">
                                                    ${wheel.speedIndx ? `<span class="wheel-info-chip"><i class="fas fa-gauge-high"></i> ${wheel.speedIndx}</span>` : ''}
                                                    ${wheel.loadIndx1 ? `<span class="wheel-info-chip"><i class="fas fa-weight"></i> ${wheel.loadIndx1}${wheel.loadIndx2 ? '/' + wheel.loadIndx2 : ''}</span>` : ''}
                                                    ${wheel.rimType ? `<span class="wheel-info-chip"><i class="fas fa-cog"></i> ${wheel.rimType}</span>` : ''}
                                                </div>
                                            </div>
                                            ` : ''}
                                            
                                            ${(wheel.pressureControl || wheel.runFlat || wheel.seal || wheel.dotNumber) ? `
                                            <div class="wheel-info-section compact">
                                                <div class="wheel-additional-info">
                                                    ${wheel.dotNumber ? `<span class="wheel-info-chip" title="DOT: ${wheel.dotNumber}"><i class="fas fa-barcode"></i> DOT</span>` : ''}
                                                    ${wheel.pressureControl ? '<span class="wheel-info-chip"><i class="fas fa-tachometer-alt"></i> TPMS</span>' : ''}
                                                    ${wheel.runFlat ? '<span class="wheel-info-chip"><i class="fas fa-shield-alt"></i> Run-Flat</span>' : ''}
                                                    ${wheel.seal ? '<span class="wheel-info-chip"><i class="fas fa-band-aid"></i> Seal</span>' : ''}
                                                </div>
                                            </div>
                                            ` : ''}
                                        </div>
                                    </div>
                                `;
                            }).join('')}
                                </div>
                            </div>
                            
                            <!-- Auto in der Mitte -->
                            <img src="../assets/images/xc90facelifetop.png?v=${Date.now()}" alt="Fahrzeug" class="car-illustration" onerror="this.style.display='none'; console.error('Bild nicht gefunden: xc90facelifetop.png');">
                            
                            <!-- Rechte Seite: VR und HR -->
                            <div class="wheels-panels-side right">
                                <div class="wheels-panels-grid">
                                    ${['VR', 'HR'].map(pos => {
                                const wheel = wheelsByPosition[pos];
                                const isSelected = pos === position;
                                
                                // Prüfe ob dieser Reifen ausgewählt ist (rechte Seite)
                                const wheelKeyRight = `${wheelsetNumber}-${pos}`;
                                const isWheelSelectedRight = selectedWheels.some(sw => sw.key === wheelKeyRight);
                                
                                if (!wheel) {
                                    return `
                                        <div class="wheel-position-panel ${pos.toLowerCase()} selectable ${isWheelSelectedRight ? 'selected' : ''}" 
                                             data-wheel-key="${wheelKeyRight}"
                                             data-position="${pos}"
                                             data-wheelset-number="${wheelsetNumber}"
                                             onclick="toggleWheelSelection('${wheelKeyRight}', '${pos}', '${wheelsetNumber}')"
                                             style="opacity: 0.4; ${isSelected ? 'border-width: 4px; border-color: var(--primary);' : ''}">
                                            <div class="wheel-panel-header">
                                                <div class="wheel-panel-position">
                                                    <span class="wheel-panel-position-icon">${pos.charAt(1) === 'L' ? 'L' : 'R'}</span>
                                                    ${pos}
                                                </div>
                                                <div class="wheel-panel-status" style="background: var(--border); color: var(--text-muted);">
                                                    <i class="fas fa-minus"></i>
                                                </div>
                                            </div>
                                            <div style="text-align: center; color: var(--text-muted); font-size: 0.875rem; padding: 2rem 0;">
                                                Keine Daten
                                            </div>
                                        </div>
                                    `;
                                }
                                
                                const status = getWheelStatus(wheel);
                                const measurements = getProfileMeasurements(wheel);
                                const tireType = wheel.tireType || 'UNKNOWN';
                                const tireTypeIcon = tireType === 'WINTER' ? '<i class="fas fa-snowflake"></i>' : 
                                                     tireType === 'SUMMER' ? '<i class="fas fa-sun"></i>' : 
                                                     tireType === 'ALLSEASON' ? '<i class="fas fa-globe"></i>' : 
                                                     '<i class="fas fa-circle"></i>';
                                const statusIcon = status === 'danger' ? '<i class="fas fa-times-circle"></i>' : 
                                                   status === 'warning' ? '<i class="fas fa-exclamation-circle"></i>' : 
                                                   '<i class="fas fa-check-circle"></i>';
                                const statusClass = status === 'danger' ? 'danger' : status === 'warning' ? 'warning' : 'ok';
                                const recommendation = getRecommendation(wheel);
                                const recClass = recommendation.includes('Austausch') ? 'danger' : recommendation.includes('Prüfung') ? 'warning' : 'ok';
                                
                                // Profiltiefe und Druck immer anzeigen (mit Fallback)
                                const profileValue = wheel.profile !== null && wheel.profile !== undefined ? parseFloat(wheel.profile).toFixed(1) : '-';
                                const pressureValue = wheel.pressure !== null && wheel.pressure !== undefined ? parseFloat(wheel.pressure).toFixed(2) : '-';
                                const profileStatus = measurements ? getMeasureStatus(measurements.m || measurements.i || measurements.a || 0) : 'default';
                                
                                return `
                                    <div class="wheel-position-panel ${pos.toLowerCase()} status-${statusClass} selectable ${isWheelSelectedRight ? 'selected' : ''}" 
                                         data-wheel-key="${wheelKeyRight}"
                                         data-position="${pos}"
                                         data-wheelset-number="${wheelsetNumber}"
                                         onclick="toggleWheelSelection('${wheelKeyRight}', '${pos}', '${wheelsetNumber}')"
                                         style="${isSelected ? 'border-width: 4px; box-shadow: 0 0 20px rgba(59, 130, 246, 0.5);' : ''}">
                                        <div class="wheel-panel-header">
                                            <div class="wheel-panel-position">
                                                <span class="wheel-panel-position-icon">${pos.charAt(1) === 'L' ? 'L' : 'R'}</span>
                                                ${pos}
                                            </div>
                                            <div class="wheel-panel-status ${statusClass}">${statusIcon}</div>
                                        </div>
                                        
                                        <!-- Profiltiefe und Druck - Immer sichtbar oben -->
                                        <div class="wheel-critical-info">
                                            <div class="critical-info-row">
                                                <div class="critical-info-item">
                                                    <div class="critical-info-label"><i class="fas fa-ruler"></i> Profiltiefe</div>
                                                    <div class="critical-info-value ${profileStatus}">${profileValue} mm</div>
                                                </div>
                                                <div class="critical-info-item">
                                                    <div class="critical-info-label"><i class="fas fa-tachometer-alt"></i> Druck</div>
                                                    <div class="critical-info-value ${pressureValue !== '-' ? 'ok' : 'default'}">${pressureValue} ${pressureValue !== '-' ? 'bar' : ''}</div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        ${measurements && measurements.i && measurements.m && measurements.a ? `
                                        <div class="wheel-profile-measurements">
                                            <div class="profile-measures-grid">
                                                <div class="profile-measure-item">
                                                    <div class="profile-measure-label">I</div>
                                                    <div class="profile-measure-value ${getMeasureStatus(measurements.i)}">${measurements.i.toFixed(1)}</div>
                                                </div>
                                                <div class="profile-measure-item">
                                                    <div class="profile-measure-label">M</div>
                                                    <div class="profile-measure-value ${getMeasureStatus(measurements.m)}">${measurements.m.toFixed(1)}</div>
                                                </div>
                                                <div class="profile-measure-item">
                                                    <div class="profile-measure-label">A</div>
                                                    <div class="profile-measure-value ${getMeasureStatus(measurements.a)}">${measurements.a.toFixed(1)}</div>
                                                </div>
                                            </div>
                                        </div>
                                        ` : ''}
                                        
                                        <div class="wheel-recommendation-badge ${recClass}">
                                            ${recClass === 'ok' ? '<i class="fas fa-check"></i>' : recClass === 'warning' ? '<i class="fas fa-exclamation-triangle"></i>' : '<i class="fas fa-times"></i>'} ${recommendation}
                                        </div>
                                        
                                        <div class="wheel-panel-info">
                                            <div class="wheel-info-section compact">
                                                <div class="wheel-panel-brand">
                                                    ${wheel.manufacturer || 'Unbekannt'} ${wheel.model || ''}
                                                </div>
                                                <div class="wheel-panel-size-type">
                                                    <span class="wheel-panel-size">${wheel.width || '-'}/${wheel.ratio || '-'}R${wheel.diameter || '-'}</span>
                                                    <span class="wheel-panel-type">${tireTypeIcon} ${tireType}</span>
                                                </div>
                                            </div>
                                            
                                            ${(wheel.speedIndx || wheel.loadIndx1 || wheel.rimType) ? `
                                            <div class="wheel-info-section compact">
                                                <div class="wheel-additional-info">
                                                    ${wheel.speedIndx ? `<span class="wheel-info-chip"><i class="fas fa-gauge-high"></i> ${wheel.speedIndx}</span>` : ''}
                                                    ${wheel.loadIndx1 ? `<span class="wheel-info-chip"><i class="fas fa-weight"></i> ${wheel.loadIndx1}${wheel.loadIndx2 ? '/' + wheel.loadIndx2 : ''}</span>` : ''}
                                                    ${wheel.rimType ? `<span class="wheel-info-chip"><i class="fas fa-cog"></i> ${wheel.rimType}</span>` : ''}
                                                </div>
                                            </div>
                                            ` : ''}
                                            
                                            ${(wheel.pressureControl || wheel.runFlat || wheel.seal || wheel.dotNumber) ? `
                                            <div class="wheel-info-section compact">
                                                <div class="wheel-additional-info">
                                                    ${wheel.dotNumber ? `<span class="wheel-info-chip" title="DOT: ${wheel.dotNumber}"><i class="fas fa-barcode"></i> DOT</span>` : ''}
                                                    ${wheel.pressureControl ? '<span class="wheel-info-chip"><i class="fas fa-tachometer-alt"></i> TPMS</span>' : ''}
                                                    ${wheel.runFlat ? '<span class="wheel-info-chip"><i class="fas fa-shield-alt"></i> Run-Flat</span>' : ''}
                                                    ${wheel.seal ? '<span class="wheel-info-chip"><i class="fas fa-band-aid"></i> Seal</span>' : ''}
                                                </div>
                                            </div>
                                            ` : ''}
                                        </div>
                                    </div>
                                    `;
                            }).join('')}
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            `;
            
            createModal('wheelDetailsModal', `Reifensatz #${wheelsetNumber} - Position ${position}`, modalContent);
            
            // Verbindungslinien nach dem Rendern zeichnen
            setTimeout(() => {
                drawWheelConnectionLines();
            }, 100);
            
            // Button für ausgewählte Reifen aktualisieren
            updateSelectedWheelsButton();
        }
        
        function toggleWheelSelection(wheelKey, position, wheelsetNumber) {
            // Finde das Wheelset
            const wheelset = wheelsetsData.find(ws => ws.satznummer == wheelsetNumber || ws.wheelSetNumber == wheelsetNumber);
            if (!wheelset) return;
            
            // Prüfe ob bereits ausgewählt
            const existingIndex = selectedWheels.findIndex(sw => sw.key === wheelKey);
            
            if (existingIndex >= 0) {
                // Entferne Auswahl
                selectedWheels.splice(existingIndex, 1);
                $(`.wheel-position-panel[data-wheel-key="${wheelKey}"]`).removeClass('selected');
            } else {
                // Füge Auswahl hinzu
                const wheels = wheelset.wheels || wheelset.resy_wheels || [];
                const wheel = wheels.find(w => w.position === position);
                
                selectedWheels.push({
                    key: wheelKey,
                    wheelsetNumber: wheelsetNumber,
                    position: position,
                    wheel: wheel || null,
                    wheelset: wheelset
                });
                
                $(`.wheel-position-panel[data-wheel-key="${wheelKey}"]`).addClass('selected');
            }
            
            updateSelectedWheelsButton();
        }
        
        function updateSelectedWheelsButton() {
            // Entferne vorhandenen Button
            $('#selectedWheelsButton').remove();
            
            if (selectedWheels.length === 0) {
                return;
            }
            
            // Erstelle Button
            const button = $(`
                <div id="selectedWheelsButton" class="selected-wheels-button" onclick="showSelectedWheelsModal()">
                    <i class="fas fa-check-circle"></i>
                    <span id="selectedWheelsCount">${selectedWheels.length}</span> Reifen ausgewählt
                    <i class="fas fa-chevron-right" style="margin-left: auto;"></i>
                </div>
            `);
            
            // Füge Button zum Body hinzu (fixed position)
            $('body').append(button);
        }
        
        function showSelectedWheelsModal() {
            if (selectedWheels.length === 0) {
                showAlert('Keine Reifen ausgewählt', 'warning');
                return;
            }
            
            // Sammle alle Wheelsets
            const wheelsetNumbers = [...new Set(selectedWheels.map(sw => sw.wheelsetNumber))];
            const wheelsets = wheelsetNumbers.map(wsNum => {
                return wheelsetsData.find(ws => 
                    ws.satznummer == wsNum || ws.wheelSetNumber == wsNum
                );
            }).filter(ws => ws);
            
            // Erstelle Modal-Content mit allen gesammelten Daten
            let modalContent = `
                <div style="padding: 1.5rem; max-height: 85vh; overflow-y: auto;">
                    <div style="background: var(--bg-hover); padding: 1.25rem; border-radius: var(--radius-lg); margin-bottom: 1.5rem; border: 2px solid var(--border);">
                        <h3 style="margin: 0 0 1rem 0; color: var(--text-main); font-size: 1.25rem;">
                            <i class="fas fa-list-check"></i> Ausgewählte Reifen - Datenübersicht
                        </h3>
                        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem;">
                            <div>
                                <div style="font-size: 0.75rem; color: var(--text-secondary); margin-bottom: 0.5rem; text-transform: uppercase; letter-spacing: 0.5px;">Anzahl Reifen</div>
                                <div style="font-size: 2rem; font-weight: 700; color: var(--primary);">${selectedWheels.length}</div>
                            </div>
                            <div>
                                <div style="font-size: 0.75rem; color: var(--text-secondary); margin-bottom: 0.5rem; text-transform: uppercase; letter-spacing: 0.5px;">Reifensätze</div>
                                <div style="font-size: 2rem; font-weight: 700; color: var(--text-main);">${wheelsetNumbers.length}</div>
                            </div>
                        </div>
                    </div>
                    
                    ${wheelsets.map(wheelset => {
                        const wsWheels = selectedWheels.filter(sw => 
                            sw.wheelsetNumber == (wheelset.satznummer || wheelset.wheelSetNumber)
                        );
                        
                        if (wsWheels.length === 0) return '';
                        
                        return `
                            <div style="background: var(--bg-panel); border: 2px solid var(--border); border-radius: var(--radius-lg); padding: 1.5rem; margin-bottom: 1.5rem; box-shadow: var(--shadow-md);">
                                <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 1.25rem; padding-bottom: 1rem; border-bottom: 2px solid var(--border);">
                                    <div style="flex: 1;">
                                        <h4 style="margin: 0 0 1rem 0; color: var(--text-main); font-size: 1.125rem;">
                                            <i class="fas fa-car"></i> Reifensatz #${wheelset.satznummer || wheelset.wheelSetNumber || '-'}
                                        </h4>
                                        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(180px, 1fr)); gap: 1rem; font-size: 0.875rem;">
                                            <div>
                                                <div style="color: var(--text-secondary); margin-bottom: 0.375rem; font-weight: 600; text-transform: uppercase; font-size: 0.75rem; letter-spacing: 0.5px;">Kunde</div>
                                                <div style="font-weight: 600; color: var(--text-main); font-size: 1rem;">
                                                    ${wheelset.haltername || wheelset.customerNo || wheelset.customerName || '-'}
                                                </div>
                                            </div>
                                            <div>
                                                <div style="color: var(--text-secondary); margin-bottom: 0.375rem; font-weight: 600; text-transform: uppercase; font-size: 0.75rem; letter-spacing: 0.5px;">Kennzeichen</div>
                                                <div style="font-weight: 600; color: var(--text-main); font-size: 1rem; font-family: monospace;">
                                                    ${wheelset.kennzeichen || wheelset.licenseNo || wheelset.licensePlate || '-'}
                                                </div>
                                            </div>
                                            <div>
                                                <div style="color: var(--text-secondary); margin-bottom: 0.375rem; font-weight: 600; text-transform: uppercase; font-size: 0.75rem; letter-spacing: 0.5px;">Fahrgestellnummer</div>
                                                <div style="font-weight: 600; color: var(--text-main); font-size: 1rem; font-family: monospace;">
                                                    ${wheelset.fin || wheelset.VIN || wheelset.vin || '-'}
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <div>
                                    <div style="font-size: 0.875rem; color: var(--text-secondary); margin-bottom: 1rem; font-weight: 600; text-transform: uppercase; letter-spacing: 0.5px;">
                                        Ausgewählte Reifen (${wsWheels.length}):
                                    </div>
                                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(280px, 1fr)); gap: 1rem;">
                                        ${wsWheels.map(sw => {
                                            const wheel = sw.wheel || {};
                                            const profile = wheel.profile ? parseFloat(wheel.profile).toFixed(1) + ' mm' : '-';
                                            const manufacturer = wheel.manufacturer || '-';
                                            const model = wheel.model || '-';
                                            const size = wheel.width && wheel.diameter ? 
                                                `${wheel.width || '-'}/${wheel.ratio || '-'}R${wheel.diameter || '-'}` : '-';
                                            
                                            return `
                                                <div style="background: var(--bg-hover); padding: 1rem; border-radius: var(--radius-lg); border-left: 4px solid var(--primary); box-shadow: var(--shadow-sm);">
                                                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 0.75rem;">
                                                        <div style="font-weight: 700; font-size: 1.25rem; color: var(--primary); letter-spacing: 1px;">${sw.position}</div>
                                                        <div style="font-size: 0.875rem; color: var(--text-secondary);">
                                                            Profil: <strong style="color: var(--text-main); font-size: 1rem;">${profile}</strong>
                                                        </div>
                                                    </div>
                                                    <div style="font-size: 0.875rem; color: var(--text-main); margin-bottom: 0.5rem; font-weight: 600;">
                                                        ${manufacturer} ${model}
                                                    </div>
                                                    <div style="font-size: 0.8125rem; color: var(--text-secondary); font-family: monospace;">
                                                        ${size}
                                                    </div>
                                                </div>
                                            `;
                                        }).join('')}
                                    </div>
                                </div>
                            </div>
                        `;
                    }).join('')}
                </div>
            `;
            
            createModal('selectedWheelsModal', 'Ausgewählte Reifen - Datenübersicht', modalContent);
        }
        
        function drawWheelConnectionLines() {
            const carContainer = document.querySelector('#wheelDetailsModal .car-container');
            const carImg = document.querySelector('#wheelDetailsModal .car-illustration');
            
            if (!carContainer || !carImg) return;
            
            // Entferne vorhandenes SVG falls vorhanden
            const existingSvg = carContainer.querySelector('.wheel-connections-svg');
            if (existingSvg) {
                existingSvg.remove();
            }
            
            // Erstelle SVG für Verbindungslinien
            const svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
            svg.setAttribute('class', 'wheel-connections-svg');
            carContainer.appendChild(svg);
            
            // Räder-Positionen relativ zum Auto (geschätzt basierend auf Auto-Bild)
            const wheelPositions = {
                'VL': { x: 0.25, y: 0.25 },  // Vorder Links - oben links
                'VR': { x: 0.75, y: 0.25 },  // Vorder Rechts - oben rechts
                'HL': { x: 0.25, y: 0.75 },  // Hinter Links - unten links
                'HR': { x: 0.75, y: 0.75 }   // Hinter Rechts - unten rechts
            };
            
            // Hole Container-Dimensionen (relativ zum Container, nicht Viewport)
            const containerWidth = carContainer.offsetWidth;
            const containerHeight = carContainer.offsetHeight;
            const imgOffsetLeft = carImg.offsetLeft;
            const imgOffsetTop = carImg.offsetTop;
            const imgWidth = carImg.offsetWidth;
            const imgHeight = carImg.offsetHeight;
            
            // Setze SVG Dimensionen
            svg.setAttribute('width', containerWidth);
            svg.setAttribute('height', containerHeight);
            svg.setAttribute('viewBox', `0 0 ${containerWidth} ${containerHeight}`);
            
            // Zeichne Linien für jedes Rad
            ['VL', 'VR', 'HL', 'HR'].forEach(pos => {
                const panel = document.querySelector(`#wheelDetailsModal .wheel-position-panel.${pos.toLowerCase()}`);
                if (!panel) return;
                
                // Berechne Rad-Position auf dem Auto (relativ zum Container)
                const wheelX = imgOffsetLeft + (wheelPositions[pos].x * imgWidth);
                const wheelY = imgOffsetTop + (wheelPositions[pos].y * imgHeight);
                
                // Hole Panel-Position relativ zum Container
                const panelRect = panel.getBoundingClientRect();
                const containerRect = carContainer.getBoundingClientRect();
                const panelLeft = panelRect.left - containerRect.left;
                const panelTop = panelRect.top - containerRect.top;
                const panelWidth = panelRect.width;
                const panelHeight = panelRect.height;
                
                // Bestimme Verbindungspunkt basierend auf Position
                let panelX, panelY;
                const panelParent = panel.closest('.wheels-panels-side');
                
                if (panelParent && panelParent.classList.contains('left')) {
                    // VL und HL: Verbinde mittig an der RECHTEN Seite des Panels (Auto-seitig)
                    panelX = panelLeft + panelWidth;
                    panelY = panelTop + (panelHeight / 2);
                } else {
                    // VR und HR: Verbinde mittig an der LINKEN Seite des Panels (Auto-seitig)
                    panelX = panelLeft;
                    panelY = panelTop + (panelHeight / 2);
                }
                
                // Erstelle gekrümmte Linie (Bezier-Kurve) für professionelleren Look
                const path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
                
                // Berechne Kontrollpunkte für sanfte Kurve
                const dx = panelX - wheelX;
                const dy = panelY - wheelY;
                const distance = Math.sqrt(dx * dx + dy * dy);
                const controlOffset = distance * 0.4; // Kurven-Stärke basierend auf Distanz
                
                // Kontrollpunkte für sanfte S-Kurve
                const cp1x = wheelX + (panelParent && panelParent.classList.contains('left') ? -controlOffset : controlOffset);
                const cp1y = wheelY + (dy > 0 ? controlOffset * 0.3 : -controlOffset * 0.3);
                const cp2x = panelX + (panelParent && panelParent.classList.contains('left') ? -controlOffset : controlOffset);
                const cp2y = panelY - (dy > 0 ? controlOffset * 0.3 : -controlOffset * 0.3);
                
                // Erstelle Bezier-Kurve
                const pathData = `M ${wheelX} ${wheelY} C ${cp1x} ${cp1y}, ${cp2x} ${cp2y}, ${panelX} ${panelY}`;
                path.setAttribute('d', pathData);
                path.setAttribute('class', 'wheel-connection-line');
                path.setAttribute('data-wheel', pos);
                path.setAttribute('fill', 'none');
                
                // Berechne Path-Länge für Animation
                const pathLength = path.getTotalLength();
                path.style.strokeDasharray = pathLength + ' ' + pathLength;
                path.style.strokeDashoffset = pathLength;
                
                // Animation zum Zeichnen der Linie (für ALLE sichtbar)
                const drawDelay = 200 * (['VL', 'VR', 'HL', 'HR'].indexOf(pos));
                setTimeout(() => {
                    path.style.transition = 'stroke-dashoffset 2.5s ease-out';
                    path.style.strokeDashoffset = '0';
                }, drawDelay); // Gestaffelte Animation
                
                svg.appendChild(path);
                
                // Füge einen DEUTLICH sichtbaren animierten Punkt hinzu für JEDEN Reifen
                const dot = document.createElementNS('http://www.w3.org/2000/svg', 'circle');
                dot.setAttribute('cx', wheelX);
                dot.setAttribute('cy', wheelY);
                dot.setAttribute('r', '8'); // Größer für bessere Sichtbarkeit
                dot.setAttribute('class', 'wheel-connection-dot');
                dot.setAttribute('data-wheel', pos);
                dot.setAttribute('fill', '#10b981'); // Direkte Farbe
                
                // Start-Delay basierend auf Position - alle bekommen ihre Animation
                const posIndex = ['VL', 'VR', 'HL', 'HR'].indexOf(pos);
                const startDelay = 3.0 + (posIndex * 0.5); // Gestaffelt, aber alle sichtbar
                
                // Animation entlang des Pfades - DEUTLICH sichtbar für ALLE
                const animate = document.createElementNS('http://www.w3.org/2000/svg', 'animateMotion');
                animate.setAttribute('dur', '4s');
                animate.setAttribute('repeatCount', 'indefinite');
                animate.setAttribute('begin', startDelay + 's');
                animate.setAttribute('path', pathData);
                
                // Opacity Animation - Start sichtbar, dann sanftes Fade
                const animateOpacity = document.createElementNS('http://www.w3.org/2000/svg', 'animate');
                animateOpacity.setAttribute('attributeName', 'opacity');
                animateOpacity.setAttribute('values', '0;1;1;0.8;0.8;0'); // Deutlicher sichtbar
                animateOpacity.setAttribute('dur', '4s');
                animateOpacity.setAttribute('begin', startDelay + 's');
                animateOpacity.setAttribute('repeatCount', 'indefinite');
                
                // Zusätzlich: Kleine Größen-Animation für bessere Sichtbarkeit
                const animateSize = document.createElementNS('http://www.w3.org/2000/svg', 'animate');
                animateSize.setAttribute('attributeName', 'r');
                animateSize.setAttribute('values', '7;9;9;7');
                animateSize.setAttribute('dur', '4s');
                animateSize.setAttribute('begin', startDelay + 's');
                animateSize.setAttribute('repeatCount', 'indefinite');
                
                dot.appendChild(animate);
                dot.appendChild(animateOpacity);
                dot.appendChild(animateSize);
                svg.appendChild(dot);
                
                // Sanfter Glow-Effekt mit einem unsichtbaren Pfad (reduziert)
                const glowPath = document.createElementNS('http://www.w3.org/2000/svg', 'path');
                glowPath.setAttribute('d', pathData);
                glowPath.setAttribute('class', 'wheel-connection-glow');
                glowPath.setAttribute('data-wheel', pos);
                glowPath.setAttribute('fill', 'none');
                glowPath.setAttribute('stroke', 'url(#glowGradient)');
                glowPath.setAttribute('stroke-width', '5');
                glowPath.setAttribute('opacity', '0.15');
                glowPath.style.filter = 'blur(3px)';
                svg.appendChild(glowPath);
                
                // Erstelle Gradient für Glow-Effekt (nur einmal pro SVG)
                let defs = svg.querySelector('defs');
                if (!defs) {
                    defs = document.createElementNS('http://www.w3.org/2000/svg', 'defs');
                    svg.insertBefore(defs, svg.firstChild);
                }
                
                // Prüfe ob Gradient bereits existiert
                if (!svg.querySelector('#glowGradient')) {
                    const gradient = document.createElementNS('http://www.w3.org/2000/svg', 'linearGradient');
                    gradient.setAttribute('id', 'glowGradient');
                    gradient.setAttribute('x1', '0%');
                    gradient.setAttribute('y1', '0%');
                    gradient.setAttribute('x2', '100%');
                    gradient.setAttribute('y2', '100%');
                    
                    // Hole CSS-Variable für Erfolgsfarbe
                    const successColor = getComputedStyle(document.documentElement).getPropertyValue('--success').trim() || '#10b981';
                    
                    const stop1 = document.createElementNS('http://www.w3.org/2000/svg', 'stop');
                    stop1.setAttribute('offset', '0%');
                    stop1.setAttribute('stop-color', successColor);
                    stop1.setAttribute('stop-opacity', '0.8');
                    
                    const stop2 = document.createElementNS('http://www.w3.org/2000/svg', 'stop');
                    stop2.setAttribute('offset', '100%');
                    stop2.setAttribute('stop-color', successColor);
                    stop2.setAttribute('stop-opacity', '0.2');
                    
                    gradient.appendChild(stop1);
                    gradient.appendChild(stop2);
                    defs.appendChild(gradient);
                }
                
                // Hover-Effekt für Panel hinzufügen
                panel.addEventListener('mouseenter', () => {
                    path.classList.add('highlighted');
                    glowPath.classList.add('highlighted');
                    // Alle Dots für dieses Rad hervorheben
                    svg.querySelectorAll(`.wheel-connection-dot[data-wheel="${pos}"]`).forEach(d => {
                        d.classList.add('highlighted');
                    });
                });
                
                panel.addEventListener('mouseleave', () => {
                    path.classList.remove('highlighted');
                    glowPath.classList.remove('highlighted');
                    // Alle Dots für dieses Rad normalisieren
                    svg.querySelectorAll(`.wheel-connection-dot[data-wheel="${pos}"]`).forEach(d => {
                        d.classList.remove('highlighted');
                    });
                });
            });
            
            // Linien beim Resize aktualisieren
            const resizeObserver = new ResizeObserver(() => {
                setTimeout(() => {
                    const existingSvg = carContainer.querySelector('.wheel-connections-svg');
                    if (existingSvg) {
                        drawWheelConnectionLines();
                    }
                }, 100);
            });
            
            resizeObserver.observe(carContainer);
            resizeObserver.observe(document.querySelector('#wheelDetailsModal .modal-body'));
        }

        function updateQuickActionsPosition() {
            const panel = $('#detailPanel');
            const ordersPanel = $('#ordersPanel');
            const quickActions = $('.quick-actions');
            const separator = $('#panelSeparator');
            
            quickActions.removeClass('panel-open orders-panel-open');
            
            const isDetailOpen = panel.hasClass('open');
            const isOrdersOpen = ordersPanel.hasClass('open');
            
            // Wenn beide Panels offen sind, verschiebe Detail Panel und zeige Separator
            if (isDetailOpen && isOrdersOpen) {
                panel.addClass('orders-panel-active');
                separator.addClass('active');
            } else {
                panel.removeClass('orders-panel-active');
                separator.removeClass('active');
            }
            
            if (isDetailOpen) {
                quickActions.addClass('panel-open');
            }
            
            if (isOrdersOpen) {
                quickActions.addClass('orders-panel-open');
            }
        }
        
        function updateOrdersBadges() {
            const totalOrders = pickupOrders.length + returnOrders.length + extraOrders.length;
            const badge = $('#ordersPanelBadge');
            if (totalOrders > 0) {
                badge.text(totalOrders).show();
            } else {
                badge.hide();
            }
            
            // Sidebar Badges
            const pickupBadge = $('#pickupTabBadge');
            if (pickupOrders.length > 0) {
                pickupBadge.text(pickupOrders.length).show();
            } else {
                pickupBadge.hide();
            }
            
            const returnBadge = $('#returnTabBadge');
            if (returnOrders.length > 0) {
                returnBadge.text(returnOrders.length).show();
            } else {
                returnBadge.hide();
            }
            
            const extraBadge = $('#extraTabBadge');
            if (extraOrders.length > 0) {
                extraBadge.text(extraOrders.length).show();
            } else {
                extraBadge.hide();
            }
        }
        
        function showOrdersPanel() {
            $('#ordersPanel').addClass('open');
            
            // Prüfe ob Detail Panel bereits offen ist
            const detailPanel = $('#detailPanel');
            if (detailPanel.hasClass('open')) {
                detailPanel.addClass('orders-panel-active');
                $('#panelSeparator').addClass('active');
            }
            
            // Lade ersten Tab
            switchOrdersTab('pickup');
            updateQuickActionsPosition();
        }
        
        function closeOrdersPanel() {
            $('#ordersPanel').removeClass('open');
            // Entferne auch Detail Panel Separator wenn Orders Panel geschlossen wird
            $('#detailPanel').removeClass('orders-panel-active');
            $('#panelSeparator').removeClass('active');
            updateQuickActionsPosition();
        }
        
        // Angebotsbenachrichtigungen Panel
        function showOffersNotificationsPanel() {
            $('#offersNotificationsPanel').addClass('open');
            
            // Prüfe ob Detail Panel bereits offen ist
            const detailPanel = $('#detailPanel');
            if (detailPanel.hasClass('open')) {
                detailPanel.addClass('orders-panel-active');
                $('#panelSeparator').addClass('active');
            }
            
            // Lade ersten Tab (Noch Offen)
            switchOffersNotificationsTab('pending');
            updateQuickActionsPosition();
            
            // Starte Auto-Reload alle 3 Minuten (180000 ms)
            if (offersNotificationsInterval) {
                clearInterval(offersNotificationsInterval);
            }
            offersNotificationsInterval = setInterval(() => {
                // Nur reloaden wenn Panel offen ist und Tab aktiv
                if ($('#offersNotificationsPanel').hasClass('open')) {
                    loadOffersNotifications(currentOffersNotificationsTab);
                }
            }, 180000); // 3 Minuten
        }
        
        function closeOffersNotificationsPanel() {
            $('#offersNotificationsPanel').removeClass('open');
            // Entferne auch Detail Panel Separator wenn Panel geschlossen wird
            $('#detailPanel').removeClass('orders-panel-active');
            $('#panelSeparator').removeClass('active');
            updateQuickActionsPosition();
            
            // Stoppe Auto-Reload wenn Panel geschlossen wird
            if (offersNotificationsInterval) {
                clearInterval(offersNotificationsInterval);
                offersNotificationsInterval = null;
            }
        }
        
        let currentOffersNotificationsTab = 'pending';
        let offersNotificationsInterval = null; // Deklariere Variable für Auto-Reload
        
        function switchOffersNotificationsTab(tab) {
            currentOffersNotificationsTab = tab;
            
            // Update Sidebar Items
            $('.orders-sidebar-item', '#offersNotificationsPanel').removeClass('active');
            $(`.orders-sidebar-item[data-tab="${tab}"]`, '#offersNotificationsPanel').addClass('active');
            
            // Lade Angebote für den Tab (Badges werden dort aktualisiert)
            loadOffersNotifications(tab);
        }
        
        function loadOffersNotifications(statusFilter = 'pending') {
            const autohausId = $('#autohausSelect').val();
            if (!autohausId) {
                $('#offersNotificationsBody').html('<div style="padding: 2rem; text-align: center; color: var(--text-secondary);">Bitte wählen Sie ein Autohaus aus</div>');
                return;
            }
            
            // Zeige nur Loading wenn Panel offen ist (um Server nicht zu überlasten bei Background-Reloads)
            const showLoadingIndicator = $('#offersNotificationsPanel').hasClass('open');
            if (showLoadingIndicator) {
                showLoading(true);
            }
            
            // Hole alle Angebote basierend auf Filter
            let filterParam = '';
            if (statusFilter === 'all') {
                filterParam = '&filter=all';
            } else if (statusFilter === 'pending') {
                filterParam = '&filter=pending';
            } else if (['accepted', 'declined', 'new_request'].includes(statusFilter)) {
                filterParam = `&filter=${statusFilter}`;
            } else {
                filterParam = '&filter=all'; // Default
            }
            
            fetch(`../api/wheelset_offers.php?action=get_all&autohaus_id=${autohausId}${filterParam}`)
                .then(response => {
                    if (!response.ok) {
                        throw new Error(`HTTP error! status: ${response.status}`);
                    }
                    return response.json();
                })
                .then(data => {
                    if (showLoadingIndicator) {
                        showLoading(false);
                    }
                    
                    if (!data.success) {
                        $('#offersNotificationsBody').html(`<div style="padding: 2rem; text-align: center; color: var(--danger);">Fehler: ${data.error || 'Unbekannter Fehler'}</div>`);
                        return;
                    }
                    
                    if (!data.offers || data.offers.length === 0) {
                        $('#offersNotificationsBody').html('<div style="padding: 2rem; text-align: center; color: var(--text-secondary);">Keine Angebote gefunden</div>');
                        return;
                    }
                    
                    console.log('All Offers:', data.offers); // Debug
                    console.log('Offer Statuses:', data.offers.map(o => o.status)); // Debug
                    
                    // Filtere nach Status (Backend filtert bereits, aber für Sicherheit)
                    let filteredOffers = data.offers;
                    
                    // Keine zusätzliche Frontend-Filterung mehr, da Backend bereits filtert
                    // Backend liefert bereits die korrekten Angebote basierend auf Filter-Parameter
                    
                    console.log('Filtered Offers:', filteredOffers); // Debug
                    
                    // Sortiere nach Datum (neueste zuerst)
                    filteredOffers.sort((a, b) => new Date(b.updated_at || b.created_at) - new Date(a.updated_at || a.created_at));
                    
                    // Update Badges - hole ALLE Angebote für korrekte Badge-Berechnung
                    // (nicht nur die gefilterten, damit alle Badges korrekt sind)
                    const autohausIdForBadges = $('#autohausSelect').val();
                    if (autohausIdForBadges) {
                        fetch(`../api/wheelset_offers.php?action=get_all&autohaus_id=${autohausIdForBadges}&filter=all`)
                            .then(response => response.json())
                            .then(badgeData => {
                                if (badgeData.success && badgeData.offers) {
                                    updateOffersNotificationsBadges(badgeData.offers);
                                }
                            })
                            .catch(error => {
                                console.error('Error loading badges:', error);
                                // Fallback: verwende die aktuellen Daten
                                updateOffersNotificationsBadges(data.offers);
                            });
                    } else {
                        updateOffersNotificationsBadges(data.offers);
                    }
                    
                    // Update letzte Aktualisierungszeit
                    updateOffersLastUpdateTime();
                    
                    if (filteredOffers.length === 0) {
                        $('#offersNotificationsBody').html('<div style="padding: 2rem; text-align: center; color: var(--text-secondary);">Keine Angebote für diesen Status gefunden</div>');
                        return;
                    }
                    
                    // Erstelle HTML
                    let html = '';
                    filteredOffers.forEach(offer => {
                        const statusConfig = {
                            'accepted': { color: '#28a745', icon: 'fa-check-circle', label: 'Angenommen' },
                            'declined': { color: '#dc3545', icon: 'fa-times-circle', label: 'Abgelehnt' },
                            'new_request': { color: '#17a2b8', icon: 'fa-redo', label: 'Neu angefragt' },
                            'sent': { color: '#007bff', icon: 'fa-envelope', label: 'Gesendet' },
                            'viewed': { color: '#ffc107', icon: 'fa-eye', label: 'Angesehen' }
                        }[offer.status] || { color: '#6c757d', icon: 'fa-question', label: 'Unbekannt' };
                        
                        const updatedDate = new Date(offer.updated_at || offer.created_at).toLocaleDateString('de-DE', {
                            day: '2-digit',
                            month: '2-digit',
                            year: 'numeric',
                            hour: '2-digit',
                            minute: '2-digit'
                        });
                        
                        // Parse customer_selection
                        let customerSelection = null;
                        if (offer.customer_selection) {
                            if (typeof offer.customer_selection === 'string') {
                                try {
                                    customerSelection = JSON.parse(offer.customer_selection);
                                } catch (e) {
                                    customerSelection = null;
                                }
                            } else {
                                customerSelection = offer.customer_selection;
                            }
                        }
                        
                        html += `
                            <div style="padding: 1rem; border: 1px solid var(--border); border-radius: var(--radius); margin-bottom: 0.75rem; background: white; cursor: pointer; transition: all 0.2s;" onclick="openOfferFromNotification('${offer.wheelset_number}', ${offer.id})" onmouseover="this.style.background='var(--bg-hover)'; this.style.borderColor='${statusConfig.color}';" onmouseout="this.style.background='white'; this.style.borderColor='var(--border)';">
                                <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 0.5rem;">
                                    <div>
                                        <div style="font-weight: 600; color: var(--text-primary); margin-bottom: 0.25rem;">
                                            <i class="fas ${statusConfig.icon}" style="color: ${statusConfig.color}; margin-right: 0.5rem;"></i>
                                            Satz #${offer.wheelset_number} - ${statusConfig.label}
                                        </div>
                                        <div style="font-size: 0.875rem; color: var(--text-secondary);">
                                            ${offer.customer_name || '-'} - ${offer.license_plate || '-'}
                                        </div>
                                    </div>
                                    <div style="text-align: right;">
                                        <div style="font-size: 0.75rem; color: var(--text-secondary);">${updatedDate}</div>
                                        ${offer.total_amount ? `<div style="font-weight: 600; color: var(--primary); margin-top: 0.25rem;">${parseFloat(offer.total_amount).toFixed(2).replace('.', ',')} €</div>` : ''}
                                    </div>
                                </div>
                                ${customerSelection ? `
                                    <div style="font-size: 0.875rem; color: var(--text-secondary); margin-top: 0.5rem; padding-top: 0.5rem; border-top: 1px solid var(--border);">
                                        ${customerSelection.notes ? `<div><i class="fas fa-comment"></i> ${customerSelection.notes}</div>` : ''}
                                        ${customerSelection.decline_reason ? `<div style="color: #dc3545;"><i class="fas fa-exclamation-triangle"></i> ${customerSelection.decline_reason}</div>` : ''}
                                        ${customerSelection.request_notes ? `<div style="color: #17a2b8;"><i class="fas fa-info-circle"></i> ${customerSelection.request_notes}</div>` : ''}
                                    </div>
                                ` : ''}
                                <div style="margin-top: 0.75rem; display: flex; gap: 0.5rem;">
                                    <button type="button" class="btn btn-sm btn-primary" onclick="event.stopPropagation(); openOfferFromNotification('${offer.wheelset_number}', ${offer.id})" style="font-size: 0.75rem;">
                                        <i class="fas fa-eye"></i> Öffnen
                                    </button>
                                </div>
                            </div>
                        `;
                    });
                    
                    $('#offersNotificationsBody').html(html);
                })
                .catch(error => {
                    if (showLoadingIndicator) {
                        showLoading(false);
                    }
                    // Nur Fehler anzeigen wenn Panel offen ist
                    if ($('#offersNotificationsPanel').hasClass('open')) {
                        $('#offersNotificationsBody').html('<div style="padding: 2rem; text-align: center; color: var(--danger);">Fehler beim Laden der Angebote</div>');
                    }
                    console.error('Error loading offers:', error);
                });
        }
        
        function updateOffersNotificationsBadges(offers) {
            const accepted = offers.filter(o => o.status === 'accepted').length;
            const declined = offers.filter(o => o.status === 'declined').length;
            const newRequest = offers.filter(o => o.status === 'new_request').length;
            // "Noch Offen" = noch nicht entschieden (sent oder viewed)
            const pending = offers.filter(o => o.status === 'sent' || o.status === 'viewed').length;
            // "Alle" = accepted, declined, new_request, sent, viewed
            const all = offers.filter(o => ['accepted', 'declined', 'new_request', 'sent', 'viewed'].includes(o.status)).length;
            
            // Pending Badge - immer anzeigen
            if (pending > 0) {
                $('#offersTabPendingBadge').text(pending).show();
            } else {
                $('#offersTabPendingBadge').hide();
            }
            
            // Accepted Badge - IMMER anzeigen (nicht nur wenn Tab aktiv)
            if (accepted > 0) {
                $('#acceptedOffersBadge').text(accepted).show();
            } else {
                $('#acceptedOffersBadge').hide();
            }
            
            // Declined Badge - IMMER anzeigen (nicht nur wenn Tab aktiv)
            if (declined > 0) {
                $('#declinedOffersBadge').text(declined).show();
            } else {
                $('#declinedOffersBadge').hide();
            }
            
            // New Request Badge - IMMER anzeigen (nicht nur wenn Tab aktiv)
            if (newRequest > 0) {
                $('#newRequestOffersBadge').text(newRequest).show();
            } else {
                $('#newRequestOffersBadge').hide();
            }
            
            // All Badge - IMMER anzeigen (nicht nur wenn Tab aktiv)
            if (all > 0) {
                $('#offersTabAllBadge').text(all).show();
            } else {
                $('#offersTabAllBadge').hide();
            }
            
            // Haupt-Badge für Panel-Button (zeigt "Alle")
            if (all > 0) {
                $('#offersNotificationsBadge').text(all).show();
            } else {
                $('#offersNotificationsBadge').hide();
            }
        }
        
        function updateOffersLastUpdateTime() {
            const now = new Date();
            const timeString = now.toLocaleTimeString('de-DE', {
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            });
            $('#offersLastUpdateTime').html(`<i class="fas fa-sync"></i> Zuletzt aktualisiert: ${timeString}`);
        }
        
        function openOfferFromNotification(wheelsetNumber, offerId) {
            // Schließe Panel
            closeOffersNotificationsPanel();
            
            // Hole Wheelset-Daten zuerst - verwende resy_api_routes.php
            showLoading(true);
            const autohausId = $('#autohausSelect').val();
            
            fetch(`../api/resy_api_routes.php?action=get_wheelset_details&wheelset_number=${wheelsetNumber}&autohaus_id=${autohausId}`)
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    
                    if (data.success && data.wheelset) {
                        // Öffne Wheelset Details
                        viewDetails(wheelsetNumber);
                        
                        // Öffne Angebotsliste nach kurzer Verzögerung
                        setTimeout(() => {
                            openOfferModal(wheelsetNumber);
                            // Nach nochmaliger Verzögerung das spezifische Angebot anzeigen
                            setTimeout(() => {
                                viewOfferDetails(offerId, wheelsetNumber, data.wheelset);
                            }, 500);
                        }, 300);
                    } else {
                        showAlert('Fehler beim Laden des Reifensatzes', 'danger');
                    }
                })
                .catch(error => {
                    showLoading(false);
                    showAlert('Fehler: ' + error.message, 'danger');
                });
        }
        
        // Lade Badge-Count beim Seitenaufruf
        function loadOffersNotificationsBadge() {
            const autohausId = $('#autohausSelect').val();
            if (!autohausId) {
                $('#offersNotificationsBadge').hide();
                return;
            }
            
            fetch(`../api/wheelset_offers.php?action=get_all&autohaus_id=${autohausId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success && data.offers) {
                        updateOffersNotificationsBadges(data.offers);
                    }
                })
                .catch(error => {
                    console.error('Error loading offers badge:', error);
                });
        }
        
        // Lade Badge wenn Autohaus gewechselt wird
        $(document).on('change', '#autohausSelect', function() {
            loadOffersNotificationsBadge();
        });
        
        // Initial badge load
        setTimeout(() => {
            loadOffersNotificationsBadge();
        }, 1000);
        
        let currentOrdersTab = 'pickup';
        
        function switchOrdersTab(tab) {
            // Verhindere Klick auf disabled/placeholder Items
            if (tab === 'inventory' || tab === 'global_inventory') {
                return;
            }
            
            currentOrdersTab = tab;
            
            // Update Sidebar Items
            $('.orders-sidebar-item').removeClass('active');
            $(`.orders-sidebar-item[data-tab="${tab}"]`).addClass('active');
            
            // Render Content
            renderOrdersTab(tab);
        }
        
        function toggleOrdersFilters() {
            const filters = $('#ordersPanelFilters');
            filters.toggleClass('active');
            const isVisible = filters.hasClass('active');
            $('#ordersFilterToggle').attr('title', isVisible ? 'Filter ausblenden' : 'Filter anzeigen');
        }
        
        function applyOrdersFilters() {
            const dateFilter = $('#ordersDateFilter').val();
            const customerFilter = $('#ordersCustomerFilter').val();
            
            // Render aktuellen Tab mit Filtern
            renderOrdersTab(currentOrdersTab, dateFilter, customerFilter);
        }
        
        function clearOrdersFilters() {
            $('#ordersDateFilter').val('');
            $('#ordersCustomerFilter').val('');
            applyOrdersFilters();
        }
        
        function updateCustomerFilterOptions() {
            const customerSelect = $('#ordersCustomerFilter');
            const customers = new Set();
            
            // Sammle alle eindeutigen Customer Numbers aus allen Orders
            pickupOrders.forEach(order => {
                if (order.customerNo) customers.add(order.customerNo);
            });
            returnOrders.forEach(order => {
                if (order.customerNo) customers.add(order.customerNo);
            });
            extraOrders.forEach(order => {
                if (order.customerNo) customers.add(order.customerNo);
            });
            
            // Update Dropdown (behalte "Alle" Option)
            const currentValue = customerSelect.val();
            customerSelect.html('<option value="">Alle</option>');
            Array.from(customers).sort().forEach(custNo => {
                customerSelect.append(`<option value="${custNo}">Kunde ${custNo}</option>`);
            });
            if (currentValue) customerSelect.val(currentValue);
        }
        
        function renderOrdersTab(tab, dateFilter = null, customerFilter = null) {
            const body = $('#ordersPanelBody');
            let html = '';
            let filteredOrders = [];
            
            // Wende Filter an
            if (tab === 'pickup') {
                filteredOrders = pickupOrders.filter(order => {
                    if (dateFilter) {
                        const orderDate = order.appointment ? new Date(order.appointment).toISOString().split('T')[0] : null;
                        if (orderDate !== dateFilter) return false;
                    }
                    if (customerFilter) {
                        if (order.customerNo !== customerFilter) return false;
                    }
                    return true;
                });
                
                if (filteredOrders.length === 0) {
                    html = '<div class="empty-state" style="padding: 3rem 2rem; text-align: center; color: var(--text-secondary);"><i class="fas fa-truck" style="font-size: 3rem; opacity: 0.3; margin-bottom: 1rem;"></i><p>' + (dateFilter || customerFilter ? 'Keine Abholungen für diese Filter gefunden' : 'Keine Abholungen vorhanden') + '</p></div>';
                } else {
                    filteredOrders.forEach(order => {
                        const appointmentDate = order.appointment ? new Date(order.appointment).toLocaleDateString('de-DE') : '-';
                        const appointmentTime = order.time || '-';
                        const statusClass = order.statusCode === 'SUCCESS' || order.statusCode === 'OK' ? 'success' : 
                                          order.statusCode === 'WARNING' || order.statusCode === 'PENDING' ? 'warning' : 'error';
                        html += `
                            <div class="order-item">
                                <div class="order-item-header">
                                    <div class="order-item-title">
                                        <i class="fas fa-truck" style="color: var(--primary); margin-right: 0.5rem;"></i>
                                        Satz #${order.wheelSetNumber || '-'}
                                    </div>
                                    ${order.statusCode ? `<span class="order-status-code ${statusClass}">${order.status || order.statusCode || ''}</span>` : ''}
                                </div>
                                <div class="order-item-details">
                                    <div class="order-detail-row">
                                        <span class="order-detail-label">Datum:</span>
                                        <span class="order-detail-value">${appointmentDate}</span>
                                    </div>
                                    <div class="order-detail-row">
                                        <span class="order-detail-label">Zeit:</span>
                                        <span class="order-detail-value">${appointmentTime}</span>
                                    </div>
                                    <div class="order-detail-row">
                                        <span class="order-detail-label">Ladestelle:</span>
                                        <span class="order-detail-value">${order.location || '-'}</span>
                                    </div>
                                    <div class="order-detail-row">
                                        <span class="order-detail-label">Service-Paket:</span>
                                        <span class="order-detail-value">${order.servicePackageName || order.servicePackageIdent || '-'}</span>
                                    </div>
                                    ${order.licenceNo ? `
                                    <div class="order-detail-row">
                                        <span class="order-detail-label">Kennzeichen:</span>
                                        <span class="order-detail-value">${order.licenceNo}</span>
                                    </div>
                                    ` : ''}
                                </div>
                                <div class="order-item-actions">
                                    <button class="order-action-btn primary" onclick="viewWheelsetFromOrder(${order.wheelSetNumber})">
                                        <i class="fas fa-eye"></i> Details
                                    </button>
                                    <button class="order-action-btn danger" onclick="cancelPickupOrder(${order.wheelSetNumber})">
                                        <i class="fas fa-times"></i> Stornieren
                                    </button>
                                </div>
                            </div>
                        `;
                    });
                }
            } else if (tab === 'return') {
                filteredOrders = returnOrders.filter(order => {
                    if (dateFilter) {
                        const orderDate = order.date ? new Date(order.date).toISOString().split('T')[0] : null;
                        if (orderDate !== dateFilter) return false;
                    }
                    if (customerFilter) {
                        if (order.customerNo !== customerFilter) return false;
                    }
                    return true;
                });
                
                if (filteredOrders.length === 0) {
                    html = '<div class="empty-state" style="padding: 3rem 2rem; text-align: center; color: var(--text-secondary);"><i class="fas fa-arrow-left" style="font-size: 3rem; opacity: 0.3; margin-bottom: 1rem;"></i><p>' + (dateFilter || customerFilter ? 'Keine Rücklieferungen für diese Filter gefunden' : 'Keine Rücklieferungen vorhanden') + '</p></div>';
                } else {
                    filteredOrders.forEach(order => {
                        const returnDate = order.date ? new Date(order.date).toLocaleDateString('de-DE') : '-';
                        const returnTime = order.time || '-';
                        const statusClass = order.statusCode === 'SUCCESS' || order.statusCode === 'OK' ? 'success' : 
                                          order.statusCode === 'WARNING' || order.statusCode === 'PENDING' ? 'warning' : 'error';
                        html += `
                            <div class="order-item">
                                <div class="order-item-header">
                                    <div class="order-item-title">
                                        <i class="fas fa-arrow-left" style="color: var(--success); margin-right: 0.5rem;"></i>
                                        Satz #${order.wheelSetNumber || '-'}
                                    </div>
                                    ${order.statusCode ? `<span class="order-status-code ${statusClass}">${order.status || order.statusCode || ''}</span>` : ''}
                                </div>
                                <div class="order-item-details">
                                    <div class="order-detail-row">
                                        <span class="order-detail-label">Datum:</span>
                                        <span class="order-detail-value">${returnDate}</span>
                                    </div>
                                    <div class="order-detail-row">
                                        <span class="order-detail-label">Zeit:</span>
                                        <span class="order-detail-value">${returnTime}</span>
                                    </div>
                                    <div class="order-detail-row">
                                        <span class="order-detail-label">Lieferoption:</span>
                                        <span class="order-detail-value">${order.deliveryOption || '-'}</span>
                                    </div>
                                    ${order.rollShelfNo ? `
                                    <div class="order-detail-row">
                                        <span class="order-detail-label">Rollenregal:</span>
                                        <span class="order-detail-value">${order.rollShelfNo}</span>
                                    </div>
                                    ` : ''}
                                    ${order.licenseNo ? `
                                    <div class="order-detail-row">
                                        <span class="order-detail-label">Kennzeichen:</span>
                                        <span class="order-detail-value">${order.licenseNo}</span>
                                    </div>
                                    ` : ''}
                                </div>
                                <div class="order-item-actions">
                                    <button class="order-action-btn primary" onclick="viewWheelsetFromOrder(${order.wheelSetNumber})">
                                        <i class="fas fa-eye"></i> Details
                                    </button>
                                    <button class="order-action-btn danger" onclick="cancelReturnOrder(${order.wheelSetNumber})">
                                        <i class="fas fa-times"></i> Stornieren
                                    </button>
                                </div>
                            </div>
                        `;
                    });
                }
            } else if (tab === 'extra') {
                filteredOrders = extraOrders.filter(order => {
                    if (dateFilter) {
                        const orderDate = order.date ? new Date(order.date).toISOString().split('T')[0] : null;
                        if (orderDate !== dateFilter) return false;
                    }
                    if (customerFilter) {
                        if (order.customerNo !== customerFilter) return false;
                    }
                    return true;
                });
                
                if (filteredOrders.length === 0) {
                    html = '<div class="empty-state" style="padding: 3rem 2rem; text-align: center; color: var(--text-secondary);"><i class="fas fa-shopping-cart" style="font-size: 3rem; opacity: 0.3; margin-bottom: 1rem;"></i><p>' + (dateFilter || customerFilter ? 'Keine Extra Orders für diese Filter gefunden' : 'Keine Extra Orders vorhanden') + '</p><button class="btn btn-primary" onclick="showExtraOrderModal()" style="margin-top: 1rem;"><i class="fas fa-plus"></i> Extra Order erstellen</button></div>';
                } else {
                    filteredOrders.forEach(order => {
                        const orderDate = order.date ? new Date(order.date).toLocaleDateString('de-DE') : '-';
                        html += `
                            <div class="order-item">
                                <div class="order-item-header">
                                    <div class="order-item-title">
                                        <i class="fas fa-shopping-cart" style="color: var(--warning); margin-right: 0.5rem;"></i>
                                        ${order.name || 'Extra Order'}
                                    </div>
                                </div>
                                <div class="order-item-details">
                                    <div class="order-detail-row">
                                        <span class="order-detail-label">Menge:</span>
                                        <span class="order-detail-value">${order.quantity || '-'}</span>
                                    </div>
                                    <div class="order-detail-row">
                                        <span class="order-detail-label">Kunde:</span>
                                        <span class="order-detail-value">${order.customerNo || '-'}</span>
                                    </div>
                                    <div class="order-detail-row">
                                        <span class="order-detail-label">Datum:</span>
                                        <span class="order-detail-value">${orderDate}</span>
                                    </div>
                                </div>
                            </div>
                        `;
                    });
                    html += '<div style="margin-top: 1rem;"><button class="btn btn-primary" onclick="showExtraOrderModal()" style="width: 100%;"><i class="fas fa-plus"></i> Neue Extra Order</button></div>';
                }
            } else if (tab === 'inventory') {
                if (!inventoryData || inventoryData.length === 0) {
                    html = '<div class="empty-state" style="padding: 3rem 2rem; text-align: center; color: var(--text-secondary);"><i class="fas fa-boxes" style="font-size: 3rem; opacity: 0.3; margin-bottom: 1rem;"></i><p>Keine Inventar-Daten verfügbar</p></div>';
                } else {
                    inventoryData.forEach(item => {
                        const count = item.inventoryNumbers ? item.inventoryNumbers.length : 0;
                        html += `
                            <div class="order-item">
                                <div class="order-item-header">
                                    <div class="order-item-title">
                                        <i class="fas fa-boxes" style="color: var(--info); margin-right: 0.5rem;"></i>
                                        ${item.inventoryType || 'Unbekannt'}
                                    </div>
                                    <span class="order-status-code success">${count} Stück</span>
                                </div>
                                ${item.inventoryNumbers && item.inventoryNumbers.length > 0 ? `
                                <div class="order-item-details">
                                    <div class="order-detail-row">
                                        <span class="order-detail-label">Wheelset-Nummern:</span>
                                        <span class="order-detail-value" style="font-size: 0.75rem; max-width: 200px; overflow: hidden; text-overflow: ellipsis;">
                                            ${item.inventoryNumbers.slice(0, 5).join(', ')}${item.inventoryNumbers.length > 5 ? '...' : ''}
                                        </span>
                                    </div>
                                </div>
                                <div class="order-item-actions">
                                    <button class="order-action-btn primary" onclick="viewInventoryDetails('${item.inventoryType || ''}')" style="width: 100%;">
                                        <i class="fas fa-eye"></i> Alle anzeigen (${item.inventoryNumbers.length})
                                    </button>
                                </div>
                                ` : '<p style="color: var(--text-secondary); font-size: 0.875rem; margin-top: 0.5rem;">Keine Details verfügbar</p>'}
                            </div>
                        `;
                    });
                }
            } else if (tab === 'customers') {
                // Lade Kunden-Liste asynchron
                body.html('<div style="text-align: center; padding: 2rem; color: var(--text-secondary);"><i class="fas fa-spinner fa-spin"></i> Lade Kunden...</div>');
                
                const autohausId = $('#autohausSelect').val();
                fetch(`../api/resy_api_routes.php?action=get_allowed_customers&autohaus_id=${autohausId}`)
                    .then(response => response.json())
                    .then(data => {
                        let customerHtml = '';
                        
                        if (data.success && data.customers && data.customers.length > 0) {
                            data.customers.forEach(customer => {
                                customerHtml += `
                                    <div class="order-item" style="cursor: pointer;" onclick="viewCustomerDetails('${customer.customerNo}')">
                                        <div class="order-item-header">
                                            <div class="order-item-title">
                                                <i class="fas fa-building" style="color: var(--info); margin-right: 0.5rem;"></i>
                                                ${customer.name || 'Unbekannt'}
                                            </div>
                                        </div>
                                        <div class="order-item-details">
                                            <div class="order-detail-row">
                                                <span class="order-detail-label">Kundennummer:</span>
                                                <span class="order-detail-value">${customer.customerNo || '-'}</span>
                                            </div>
                                        </div>
                                        <div class="order-item-actions">
                                            <button class="order-action-btn primary" onclick="event.stopPropagation(); viewCustomerDetails('${customer.customerNo}')" style="width: 100%;">
                                                <i class="fas fa-eye"></i> Details anzeigen
                                            </button>
                                        </div>
                                    </div>
                                `;
                            });
                        } else {
                            customerHtml = '<div class="empty-state" style="padding: 3rem 2rem; text-align: center; color: var(--text-secondary);"><i class="fas fa-users" style="font-size: 3rem; opacity: 0.3; margin-bottom: 1rem;"></i><p>Keine Kunden verfügbar</p></div>';
                        }
                        
                        body.html(customerHtml);
                    })
                    .catch(error => {
                        body.html(`<div class="empty-state" style="padding: 3rem 2rem; text-align: center; color: var(--text-secondary);"><i class="fas fa-exclamation-triangle" style="font-size: 3rem; opacity: 0.3; margin-bottom: 1rem;"></i><p>Fehler beim Laden der Kunden: ${error.message}</p></div>`);
                    });
                    
                return; // Early return, da async
            } else if (tab === 'global_inventory') {
                // Lade globales Inventar (alle Kunden)
                body.html('<div style="text-align: center; padding: 2rem; color: var(--text-secondary);"><i class="fas fa-spinner fa-spin"></i> Lade globales Inventar...</div>');
                
                const autohausId = $('#autohausSelect').val();
                fetch(`../api/resy_api_routes.php?action=get_global_inventory&autohaus_id=${autohausId}`)
                    .then(response => response.json())
                    .then(data => {
                        let inventoryHtml = '';
                        
                        if (data.success && data.inventory && data.inventory.length > 0) {
                            // Gruppiere nach Customer
                            const inventoryByCustomer = {};
                            data.inventory.forEach(item => {
                                if (!inventoryByCustomer[item.customerNo]) {
                                    inventoryByCustomer[item.customerNo] = {
                                        customerNo: item.customerNo,
                                        totalCount: 0,
                                        items: []
                                    };
                                }
                                const itemCount = item.inventoryCount || 0;
                                inventoryByCustomer[item.customerNo].totalCount += itemCount;
                                inventoryByCustomer[item.customerNo].items.push(item);
                            });
                            
                            // Sortiere nach totalCount (absteigend)
                            const sortedCustomers = Object.values(inventoryByCustomer).sort((a, b) => b.totalCount - a.totalCount);
                            
                            sortedCustomers.forEach(customerData => {
                                inventoryHtml += `
                                    <div class="order-item">
                                        <div class="order-item-header">
                                            <div class="order-item-title">
                                                <i class="fas fa-building" style="color: var(--info); margin-right: 0.5rem;"></i>
                                                Kunde ${customerData.customerNo}
                                            </div>
                                            <span class="order-status-code success">${customerData.totalCount} Stück</span>
                                        </div>
                                        <div class="order-item-details">
                                            ${customerData.items.map(item => `
                                                <div class="order-detail-row">
                                                    <span class="order-detail-label">Inventar:</span>
                                                    <span class="order-detail-value">${item.inventoryCount || 0} Stück</span>
                                                </div>
                                            `).join('')}
                                        </div>
                                    </div>
                                `;
                            });
                        } else {
                            inventoryHtml = '<div class="empty-state" style="padding: 3rem 2rem; text-align: center; color: var(--text-secondary);"><i class="fas fa-boxes" style="font-size: 3rem; opacity: 0.3; margin-bottom: 1rem;"></i><p>Keine globalen Inventar-Daten verfügbar</p></div>';
                        }
                        
                        body.html(inventoryHtml);
                    })
                    .catch(error => {
                        body.html(`<div class="empty-state" style="padding: 3rem 2rem; text-align: center; color: var(--text-secondary);"><i class="fas fa-exclamation-triangle" style="font-size: 3rem; opacity: 0.3; margin-bottom: 1rem;"></i><p>Fehler beim Laden: ${error.message}</p></div>`);
                    });
                    
                return; // Early return, da async
            }
            
            body.html(html);
        }
        
        function viewCustomerDetails(customerNo) {
            const autohausId = $('#autohausSelect').val();
            showLoading(true, 'Lade Kundendetails...');
            
            fetch(`../api/resy_api_routes.php?action=get_customer_details&autohaus_id=${autohausId}&customer_no=${customerNo}`)
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    
                    if (data.success && data.customer) {
                        const customer = data.customer;
                        let detailsHtml = `
                            <div style="max-height: 500px; overflow-y: auto;">
                                <div style="background: var(--bg-hover); padding: 1.5rem; border-radius: var(--radius); margin-bottom: 1rem;">
                                    <h3 style="margin: 0 0 1rem 0; font-size: 1.125rem; font-weight: 600; color: var(--primary);">
                                        <i class="fas fa-building" style="margin-right: 0.5rem;"></i>${customer.name1 || ''} ${customer.name2 || ''}
                                    </h3>
                                    <div style="display: grid; gap: 0.75rem;">
                                        <div>
                                            <strong style="color: var(--text-secondary); font-size: 0.75rem;">Kundennummer:</strong>
                                            <div style="font-size: 0.875rem; margin-top: 0.25rem;">${customer.customerNo || '-'}</div>
                                        </div>
                                        <div>
                                            <strong style="color: var(--text-secondary); font-size: 0.75rem;">Adresse:</strong>
                                            <div style="font-size: 0.875rem; margin-top: 0.25rem;">${customer.address || '-'}</div>
                                        </div>
                                        <div>
                                            <strong style="color: var(--text-secondary); font-size: 0.75rem;">PLZ & Stadt:</strong>
                                            <div style="font-size: 0.875rem; margin-top: 0.25rem;">${customer.postalcode || ''} ${customer.city || ''}</div>
                                        </div>
                                    </div>
                                </div>
                                <div style="margin-top: 1rem;">
                                    <button class="btn btn-secondary" onclick="closeModal('customerDetailsModal')" style="width: 100%;">
                                        <i class="fas fa-times"></i> Schließen
                                    </button>
                                </div>
                            </div>
                        `;
                        
                        createModal('customerDetailsModal', `Kundendetails - ${customer.customerNo}`, detailsHtml);
                    } else {
                        showAlert('Kundendetails konnten nicht geladen werden', 'danger');
                    }
                })
                .catch(error => {
                    showLoading(false);
                    showAlert('Fehler: ' + error.message, 'danger');
                });
        }
        
        function viewWheelsetFromOrder(wheelsetNumber) {
            closeOrdersPanel();
            viewDetails(wheelsetNumber);
        }
        
        function cancelPickupOrder(wheelsetNumber) {
            if (!confirm(`Möchten Sie die Abholung für Satz #${wheelsetNumber} wirklich stornieren?`)) {
                return;
            }
            
            const autohausId = $('#autohausSelect').val();
            showLoading(true);
            
            fetch(`../api/resy_api_routes.php?action=delete_pickup_order&autohaus_id=${autohausId}`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ wheelset_number: wheelsetNumber })
            })
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    if (data.success) {
                        showAlert('✅ Abholung erfolgreich storniert', 'success');
                        loadWheelsets(); // Lade alles neu
                        setTimeout(() => {
                            if ($('#ordersPanel').hasClass('open')) {
                                renderOrdersTab(currentOrdersTab); // Update aktuelle Ansicht
                            }
                        }, 500);
                    } else {
                        showAlert('❌ Fehler: ' + (data.error || data.result?.status || 'Unbekannt'), 'danger');
                    }
                })
                .catch(error => {
                    showLoading(false);
                    showAlert('Fehler: ' + error.message, 'danger');
                });
        }
        
        function cancelReturnOrder(wheelsetNumber) {
            if (!confirm(`Möchten Sie die Rücklieferung für Satz #${wheelsetNumber} wirklich stornieren?`)) {
                return;
            }
            
            const autohausId = $('#autohausSelect').val();
            showLoading(true);
            
            fetch(`../api/resy_api_routes.php?action=delete_return_order&autohaus_id=${autohausId}`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ wheelset_number: wheelsetNumber })
            })
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    if (data.success) {
                        showAlert('✅ Rücklieferung erfolgreich storniert', 'success');
                        loadWheelsets(); // Lade alles neu
                        setTimeout(() => {
                            if ($('#ordersPanel').hasClass('open')) {
                                renderOrdersTab(currentOrdersTab); // Update aktuelle Ansicht
                            }
                        }, 500);
                    } else {
                        showAlert('❌ Fehler: ' + (data.error || data.result?.status || 'Unbekannt'), 'danger');
                    }
                })
                .catch(error => {
                    showLoading(false);
                    showAlert('Fehler: ' + error.message, 'danger');
                });
        }
        
        function showExtraOrderModal() {
            const autohausId = $('#autohausSelect').val();
            if (!autohausId) {
                showAlert('Bitte wählen Sie ein Autohaus aus', 'warning');
                return;
            }
            
            // Lade mögliche Extra Orders
            showLoading(true);
            fetch(`../api/resy_api_routes.php?action=get_customer_extra_orders&autohaus_id=${autohausId}`)
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    let extraOrderOptions = '<option value="">-- Extra Order wählen --</option>';
                    if (data.success && data.extraOrders && data.extraOrders.length > 0) {
                        data.extraOrders.forEach(option => {
                            extraOrderOptions += `<option value="${option.name || ''}">${option.name || ''}</option>`;
                        });
                    }
                    
                    const modalContent = `
                        <div class="form-group">
                            <label class="form-label">Extra Order *</label>
                            <select id="extraOrderName" class="form-control" required>
                                ${extraOrderOptions}
                            </select>
                        </div>
                        <div class="form-group">
                            <label class="form-label">Menge *</label>
                            <input type="number" id="extraOrderQuantity" class="form-control" min="1" value="1" required>
                        </div>
                        <div class="form-group">
                            <label class="form-label">Kundennummer</label>
                            <input type="text" id="extraOrderCustomerNo" class="form-control" placeholder="Wird automatisch ausgewählt" readonly>
                        </div>
                    `;
                    
                    createModal('extraOrderModal', 'Extra Order erstellen', modalContent, true, 'submitExtraOrder', '<i class="fas fa-check"></i> Erstellen');
                })
                .catch(error => {
                    showLoading(false);
                    showAlert('Fehler beim Laden der Extra Orders: ' + error.message, 'danger');
                });
        }
        
        function submitExtraOrder() {
            const autohausId = $('#autohausSelect').val();
            const name = $('#extraOrderName').val();
            const quantity = parseInt($('#extraOrderQuantity').val());
            
            if (!name || !quantity || quantity < 1) {
                showAlert('Bitte füllen Sie alle Felder aus', 'warning');
                return;
            }
            
            showLoading(true);
            
            fetch(`../api/resy_api_routes.php?action=create_extra_order&autohaus_id=${autohausId}`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    name: name,
                    quantity: quantity
                    // customerNo wird vom Backend automatisch gesetzt
                })
            })
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    if (data.success) {
                        showAlert('✅ Extra Order erfolgreich erstellt', 'success');
                        closeModal('extraOrderModal');
                        loadWheelsets(); // Lade alles neu
                        if ($('#ordersPanel').hasClass('open')) {
                            renderOrdersTab('extra'); // Update Extra Orders Tab
                        }
                    } else {
                        showAlert('❌ Fehler: ' + (data.error || 'Unbekannt'), 'danger');
                    }
                })
                .catch(error => {
                    showLoading(false);
                    showAlert('Fehler: ' + error.message, 'danger');
                });
        }

        function quickPickup(id) {
            selectedWheelsets.clear();
            selectedWheelsets.add(id);
            showPickupModal();
        }

        function quickReturn(id) {
            selectedWheelsets.clear();
            selectedWheelsets.add(id);
            showReturnModal();
        }

        function showPickupModal() {
            const autohausId = $('#autohausSelect').val();
            if (!autohausId) {
                showAlert('Bitte wählen Sie ein Autohaus aus', 'warning');
                return;
            }

            if (selectedWheelsets.size === 0) {
                showAlert('Bitte wählen Sie mindestens einen Reifensatz aus', 'warning');
                return;
            }

            createModal('pickupModal', 'Abholung bestellen', getPickupModalContent());
            loadPickupOrderData(autohausId);
            // Auto-Fill erweiterte Parameter nach Modal-Erstellung
            setTimeout(() => {
                autoFillPickupParameters();
            }, 500);
        }
        
        function autoFillPickupParameters() {
            // Wenn nur ein Wheelset ausgewählt, fülle Daten automatisch
            if (selectedWheelsets.size === 1) {
                const wsNum = Array.from(selectedWheelsets)[0];
                const ws = wheelsetsData.find(w => w.satznummer == wsNum);
                
                if (ws) {
                    // Lade Details wenn nicht vorhanden
                    if (!ws.wheels || ws.wheels.length === 0) {
                        const autohausId = $('#autohausSelect').val();
                        fetch(`../api/resy_api_routes.php?action=get_wheelset_details&autohaus_id=${autohausId}&wheelset_number=${wsNum}`)
                            .then(response => response.json())
                            .then(data => {
                                if (data.success && data.wheelset && data.wheelset.wheels) {
                                    // Auto-Fill mit geladenen Daten
                                    fillPickupParametersFromWheelset(data.wheelset);
                                }
                            })
                            .catch(() => {
                                // Fehler ignorieren, Nutzer kann manuell eingeben
                            });
                    } else {
                        // Auto-Fill mit vorhandenen Daten
                        fillPickupParametersFromWheelset(ws);
                    }
                }
            }
        }
        
        function fillPickupParametersFromWheelset(wheelset) {
            if (!wheelset.wheels || wheelset.wheels.length === 0) return;
            
            // Reifenzahl - Auto-Fill und aktivieren für Submit
            const tireCount = wheelset.wheels.length;
            $('#pickupTireCount').val(tireCount).prop('disabled', false).prop('readonly', false);
            
            // Reifenart (nehme die des ersten Reifens)
            const firstTireType = wheelset.wheels[0].tireType;
            if (firstTireType) {
                $('#pickupTireType').val(firstTireType).prop('disabled', false);
            }
        }

        function showReturnModal() {
            const autohausId = $('#autohausSelect').val();
            if (!autohausId) {
                showAlert('Bitte wählen Sie ein Autohaus aus', 'warning');
                return;
            }

            if (selectedWheelsets.size === 0) {
                showAlert('Bitte wählen Sie mindestens einen Reifensatz aus', 'warning');
                return;
            }

            createModal('returnModal', 'Rücklieferung bestellen', getReturnModalContent());
            loadReturnOrderData(autohausId);
        }

        function getPickupModalContent() {
            const selectedList = Array.from(selectedWheelsets).map(id => {
                const ws = wheelsetsData.find(w => w.satznummer == id);
                return ws ? `#${ws.satznummer} (${ws.kennzeichen || '-'})` : `#${id}`;
            }).join(', ');

            return `
                <div class="form-group">
                    <label class="form-label">Ausgewählte Reifensätze</label>
                    <div style="background: var(--bg-hover); padding: 0.75rem; border-radius: var(--radius); font-size: 0.875rem;">
                        ${selectedList || 'Keine ausgewählt'}
                    </div>
                </div>
                <div class="form-group">
                    <label class="form-label">Abholdatum & Zeit *</label>
                    <div class="slot-calendar-wrapper">
                        <div id="pickupSlotCalendar" class="slot-calendar">
                            <div style="text-align: center; padding: 2rem; color: var(--text-secondary);">
                                <i class="fas fa-spinner fa-spin"></i> Lade verfügbare Termine...
                            </div>
                        </div>
                    </div>
                    <input type="hidden" id="pickupSelectedDate" value="">
                    <input type="hidden" id="pickupSelectedTime" value="">
                </div>
                <div class="form-group">
                    <label class="form-label">Service-Paket *</label>
                    <div id="servicePackageCards" class="select-cards-grid">
                        <div style="text-align: center; padding: 2rem; color: var(--text-secondary);">
                            <i class="fas fa-spinner fa-spin"></i> Lade Service-Pakete...
                        </div>
                    </div>
                    <select id="servicePackageSelect" class="form-control" style="display: none;">
                        <option value="">-- Service-Paket wählen --</option>
                    </select>
                </div>
                <div class="form-group">
                    <label class="form-label">Ladestelle *</label>
                    <div id="loadingPointCards" class="select-cards-grid">
                        <div style="text-align: center; padding: 2rem; color: var(--text-secondary);">
                            <i class="fas fa-spinner fa-spin"></i> Lade Ladestellen...
                        </div>
                    </div>
                    <select id="loadingPointSelect" class="form-control" style="display: none;">
                        <option value="">-- Ladestelle wählen --</option>
                    </select>
                </div>
                <div class="form-group">
                    <label class="form-label" style="display: flex; align-items: center; gap: 0.5rem;">
                        <i class="fas fa-info-circle" style="color: var(--info); font-size: 0.875rem;"></i>
                        Erweiterte Parameter (optional, automatisch befüllt)
                    </label>
                    <div style="background: var(--bg-hover); padding: 1rem; border-radius: var(--radius); border-left: 3px solid var(--info); opacity: 0.8;">
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 0.75rem; margin-bottom: 0.75rem;">
                            <div>
                                <label style="display: block; font-size: 0.75rem; color: var(--text-secondary); margin-bottom: 0.25rem;">Reifenzahl</label>
                                <input type="number" id="pickupTireCount" class="form-control" min="0" max="20" placeholder="Auto" style="font-size: 0.875rem;" readonly disabled>
                                <small style="color: var(--text-muted); font-size: 0.6875rem;">Wird automatisch aus Wheelset-Details befüllt</small>
                            </div>
                            <div>
                                <label style="display: block; font-size: 0.75rem; color: var(--text-secondary); margin-bottom: 0.25rem;">Radkappenzahl</label>
                                <input type="number" id="pickupTireCapCount" class="form-control" min="0" max="20" placeholder="Auto" style="font-size: 0.875rem;" readonly disabled>
                                <small style="color: var(--text-muted); font-size: 0.6875rem;">Wird automatisch aus Wheelset-Details befüllt</small>
                            </div>
                        </div>
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 0.75rem;">
                            <div>
                                <label style="display: block; font-size: 0.75rem; color: var(--text-secondary); margin-bottom: 0.25rem;">Reifenart</label>
                                <select id="pickupTireType" class="form-control" style="font-size: 0.875rem;" disabled>
                                    <option value="">-- Auto aus Wheelset --</option>
                                    <option value="SUMMER">☀️ Sommer</option>
                                    <option value="WINTER">❄️ Winter</option>
                                    <option value="ALLSEASON">🌍 Ganzjahres</option>
                                </select>
                            </div>
                            <div>
                                <label style="display: block; font-size: 0.75rem; color: var(--text-secondary); margin-bottom: 0.25rem;">Verschiebung (Tage)</label>
                                <input type="number" id="pickupOffset" class="form-control" min="-7" max="7" value="0" style="font-size: 0.875rem;" placeholder="0" readonly disabled>
                                <small style="color: var(--text-muted); font-size: 0.6875rem;">± Tage vom gewählten Datum</small>
                            </div>
                        </div>
                    </div>
                </div>
            `;
        }

        function getReturnModalContent() {
            const selectedList = Array.from(selectedWheelsets).map(id => {
                const ws = wheelsetsData.find(w => w.satznummer == id);
                return ws ? `#${ws.satznummer} (${ws.kennzeichen || '-'})` : `#${id}`;
            }).join(', ');

            return `
                <div class="form-group">
                    <label class="form-label">Ausgewählte Reifensätze</label>
                    <div style="background: var(--bg-hover); padding: 0.75rem; border-radius: var(--radius); font-size: 0.875rem;">
                        ${selectedList || 'Keine ausgewählt'}
                    </div>
                </div>
                <div class="form-group">
                    <label class="form-label">Rücklieferdatum & Zeit *</label>
                    <div class="slot-calendar-wrapper">
                        <div id="returnSlotCalendar" class="slot-calendar">
                            <div style="text-align: center; padding: 2rem; color: var(--text-secondary);">
                                <i class="fas fa-spinner fa-spin"></i> Lade verfügbare Termine...
                            </div>
                        </div>
                    </div>
                    <input type="hidden" id="returnSelectedDate" value="">
                    <input type="hidden" id="returnSelectedTime" value="">
                </div>
                <div class="form-group">
                    <label class="form-label">Lieferoption *</label>
                    <div id="deliveryOptionCards" class="select-cards-grid">
                        <div style="text-align: center; padding: 2rem; color: var(--text-secondary);">
                            <i class="fas fa-spinner fa-spin"></i> Lade Lieferoptionen...
                        </div>
                    </div>
                    <select id="deliveryOptionSelect" class="form-control" style="display: none;">
                        <option value="">-- Lieferoption wählen --</option>
                    </select>
                </div>
                <div class="form-group">
                    <label class="form-label">Kommentar (optional)</label>
                    <textarea id="returnComment" class="form-control" rows="3" placeholder="Optionale Anmerkungen..."></textarea>
                </div>
                <div class="form-group">
                    <label class="form-label" style="display: flex; align-items: center; gap: 0.5rem;">
                        <i class="fas fa-info-circle" style="color: var(--info); font-size: 0.875rem;"></i>
                        Erweiterte Parameter (optional, automatisch befüllt)
                    </label>
                    <div style="background: var(--bg-hover); padding: 1rem; border-radius: var(--radius); border-left: 3px solid var(--info); opacity: 0.8;">
                        <div>
                            <label style="display: block; font-size: 0.75rem; color: var(--text-secondary); margin-bottom: 0.25rem;">Verschiebung (Tage)</label>
                            <input type="number" id="returnOffset" class="form-control" min="-7" max="7" value="0" style="font-size: 0.875rem;" placeholder="0" readonly disabled>
                            <small style="color: var(--text-muted); font-size: 0.6875rem;">± Tage vom gewählten Datum (z.B. -1 = einen Tag vor dem Termin)</small>
                        </div>
                    </div>
                </div>
            `;
        }

        function loadPickupOrderData(autohausId) {
            console.log('[loadPickupOrderData] Loading pickup order dates for autohaus:', autohausId);
            // Lade Termine
            fetch(`../api/resy_api_routes.php?action=get_pickup_order_dates&autohaus_id=${autohausId}`)
                .then(response => {
                    console.log('[loadPickupOrderData] Response status:', response.status);
                    return response.json();
                })
                .then(data => {
                    console.log('[loadPickupOrderData] Received data:', data);
                    if (data.success && data.orderDays && Array.isArray(data.orderDays) && data.orderDays.length > 0) {
                        console.log('[loadPickupOrderData] OrderDays count:', data.orderDays.length);
                        renderSlotCalendar('pickupSlotCalendar', data.orderDays, 'pickup');
                    } else {
                        console.warn('[loadPickupOrderData] No orderDays or empty array');
                        $('#pickupSlotCalendar').html('<div class="no-slots-message">Keine verfügbaren Termine gefunden</div>');
                    }
                })
                .catch(error => {
                    console.error('[loadPickupOrderData] Fehler beim Laden der Termine:', error);
                    $('#pickupSlotCalendar').html(`<div class="no-slots-message">Fehler beim Laden der Termine: ${error.message}</div>`);
                });

            // Lade Service-Pakete
            fetch(`../api/resy_api_routes.php?action=get_service_packages&autohaus_id=${autohausId}`)
                .then(response => response.json())
                .then(data => {
                    const cardsContainer = $('#servicePackageCards');
                    if (data.success && data.servicePackages && data.servicePackages.length > 0) {
                        let cardsHtml = '';
                        data.servicePackages.forEach(pkg => {
                            cardsHtml += `
                                <div class="select-card" data-value="${pkg.servicePackageIdent}" onclick="selectServicePackage('${pkg.servicePackageIdent}')">
                                    <div class="select-card-check"><i class="fas fa-check"></i></div>
                                    <div class="select-card-icon"><i class="fas fa-box"></i></div>
                                    <div class="select-card-title">${pkg.servicePackageName || 'Service-Paket'}</div>
                                    <div class="select-card-description">ID: ${pkg.servicePackageIdent}</div>
                                </div>
                            `;
                        });
                        cardsContainer.html(cardsHtml);
                        
                        // Update hidden select
                        let options = '<option value="">-- Service-Paket wählen --</option>';
                        data.servicePackages.forEach(pkg => {
                            options += `<option value="${pkg.servicePackageIdent}">${pkg.servicePackageName}</option>`;
                        });
                        $('#servicePackageSelect').html(options);
                    } else {
                        cardsContainer.html('<div style="text-align: center; padding: 2rem; color: var(--text-secondary);">Keine Service-Pakete verfügbar</div>');
                    }
                })
                .catch(error => {
                    console.error('Fehler beim Laden der Service-Pakete:', error);
                    $('#servicePackageCards').html(`<div style="text-align: center; padding: 2rem; color: var(--danger);">Fehler beim Laden</div>`);
                });

            // Lade Ladestellen
            fetch(`../api/resy_api_routes.php?action=get_loading_points&autohaus_id=${autohausId}`)
                .then(response => response.json())
                .then(data => {
                    const cardsContainer = $('#loadingPointCards');
                    if (data.success && data.loadingPoints && data.loadingPoints.length > 0) {
                        let cardsHtml = '';
                        data.loadingPoints.forEach(point => {
                            cardsHtml += `
                                <div class="select-card" data-value="${point.name}" onclick="selectLoadingPoint('${point.name.replace(/'/g, "\\'")}')">
                                    <div class="select-card-check"><i class="fas fa-check"></i></div>
                                    <div class="select-card-icon"><i class="fas fa-map-marker-alt"></i></div>
                                    <div class="select-card-title">${point.name || 'Ladestelle'}</div>
                                    <div class="select-card-description">Standort</div>
                                </div>
                            `;
                        });
                        cardsContainer.html(cardsHtml);
                        
                        // Update hidden select
                        let options = '<option value="">-- Ladestelle wählen --</option>';
                        data.loadingPoints.forEach(point => {
                            options += `<option value="${point.name}">${point.name}</option>`;
                        });
                        $('#loadingPointSelect').html(options);
                    } else {
                        cardsContainer.html('<div style="text-align: center; padding: 2rem; color: var(--text-secondary);">Keine Ladestellen verfügbar</div>');
                    }
                })
                .catch(error => {
                    console.error('Fehler beim Laden der Ladestellen:', error);
                    $('#loadingPointCards').html(`<div style="text-align: center; padding: 2rem; color: var(--danger);">Fehler beim Laden</div>`);
                });
        }

        function loadReturnOrderData(autohausId) {
            console.log('[loadReturnOrderData] Loading return order dates for autohaus:', autohausId);
            // Lade Termine
            fetch(`../api/resy_api_routes.php?action=get_return_order_dates&autohaus_id=${autohausId}`)
                .then(response => {
                    console.log('[loadReturnOrderData] Response status:', response.status);
                    return response.json();
                })
                .then(data => {
                    console.log('[loadReturnOrderData] Received data:', data);
                    if (data.success && data.orderDays && Array.isArray(data.orderDays) && data.orderDays.length > 0) {
                        console.log('[loadReturnOrderData] OrderDays count:', data.orderDays.length);
                        renderSlotCalendar('returnSlotCalendar', data.orderDays, 'return');
                    } else {
                        console.warn('[loadReturnOrderData] No orderDays or empty array');
                        $('#returnSlotCalendar').html('<div class="no-slots-message">Keine verfügbaren Termine gefunden</div>');
                    }
                })
                .catch(error => {
                    console.error('[loadReturnOrderData] Fehler beim Laden der Termine:', error);
                    $('#returnSlotCalendar').html(`<div class="no-slots-message">Fehler beim Laden der Termine: ${error.message}</div>`);
                });

            // Lade Lieferoptionen
            fetch(`../api/resy_api_routes.php?action=get_delivery_options&autohaus_id=${autohausId}`)
                .then(response => response.json())
                .then(data => {
                    const cardsContainer = $('#deliveryOptionCards');
                    if (data.success && data.deliveryOptions && data.deliveryOptions.length > 0) {
                        let cardsHtml = '';
                        data.deliveryOptions.forEach(option => {
                            const icon = option.name.toLowerCase().includes('express') ? 'fa-shipping-fast' : 
                                       option.name.toLowerCase().includes('standard') ? 'fa-truck' : 'fa-box-open';
                            cardsHtml += `
                                <div class="select-card" data-value="${option.name}" onclick="selectDeliveryOption('${option.name.replace(/'/g, "\\'")}')">
                                    <div class="select-card-check"><i class="fas fa-check"></i></div>
                                    <div class="select-card-icon"><i class="fas ${icon}"></i></div>
                                    <div class="select-card-title">${option.name || 'Lieferoption'}</div>
                                    <div class="select-card-description">Lieferart</div>
                                </div>
                            `;
                        });
                        cardsContainer.html(cardsHtml);
                        
                        // Update hidden select
                        let options = '<option value="">-- Lieferoption wählen --</option>';
                        data.deliveryOptions.forEach(option => {
                            options += `<option value="${option.name}">${option.name}</option>`;
                        });
                        $('#deliveryOptionSelect').html(options);
                    } else {
                        cardsContainer.html('<div style="text-align: center; padding: 2rem; color: var(--text-secondary);">Keine Lieferoptionen verfügbar</div>');
                    }
                })
                .catch(error => {
                    console.error('Fehler beim Laden der Lieferoptionen:', error);
                    $('#deliveryOptionCards').html(`<div style="text-align: center; padding: 2rem; color: var(--danger);">Fehler beim Laden</div>`);
                });
        }

        // Slot Calendar Rendering - Separate Monate für Pickup und Return
        let pickupMonth = new Date().getMonth();
        let pickupYear = new Date().getFullYear();
        let returnMonth = new Date().getMonth();
        let returnYear = new Date().getFullYear();

        function renderSlotCalendar(containerId, orderDays, type) {
            console.log('[renderSlotCalendar] Called with:', { containerId, orderDays, type });
            const container = $(`#${containerId}`);
            
            // Initialisiere daysByDate auch wenn keine Daten kommen
            const daysByDate = {};
            
            if (orderDays && Array.isArray(orderDays) && orderDays.length > 0) {
                // Gruppiere nach Datum
                orderDays.forEach(day => {
                    console.log('[renderSlotCalendar] Processing day:', day);
                    let dateKey = day.date || day.Date || day.dateStr;
                    if (!dateKey) {
                        console.warn('[renderSlotCalendar] Day object missing date:', day);
                        return;
                    }
                    
                    // Entferne Zeitzone (+01:00, +02:00, etc.) und konvertiere zu ISO Format YYYY-MM-DD
                    dateKey = dateKey.split('+')[0].split('T')[0];
                    
                    if (!daysByDate[dateKey]) {
                        daysByDate[dateKey] = [];
                    }
                    daysByDate[dateKey].push(day);
                });

                console.log('[renderSlotCalendar] daysByDate:', daysByDate);

                // Finde min/max Datum für Kalender
                const dates = Object.keys(daysByDate).sort();
                if (dates.length > 0) {
                    console.log('[renderSlotCalendar] First date:', dates[0]);
                    const firstDateStr = dates[0];
                    // Parse ISO date string (YYYY-MM-DD)
                    const dateParts = firstDateStr.split('-');
                    if (dateParts.length === 3) {
                        const firstDate = new Date(parseInt(dateParts[0]), parseInt(dateParts[1]) - 1, parseInt(dateParts[2]));
                        console.log('[renderSlotCalendar] Parsed firstDate:', firstDate);
                        
                        // Setze aktuellen Monat zum ersten verfügbaren Datum (nur wenn gültig)
                        if (!isNaN(firstDate.getTime())) {
                            if (type === 'pickup') {
                                pickupMonth = firstDate.getMonth();
                                pickupYear = firstDate.getFullYear();
                                console.log('[renderSlotCalendar] Set pickup month:', pickupMonth, 'year:', pickupYear);
                            } else {
                                returnMonth = firstDate.getMonth();
                                returnYear = firstDate.getFullYear();
                                console.log('[renderSlotCalendar] Set return month:', returnMonth, 'year:', returnYear);
                            }
                        }
                    }
                } else {
                    console.warn('[renderSlotCalendar] No valid dates found in orderDays');
                    // Verwende aktuellen Monat als Fallback
                }
                
                console.log('[renderSlotCalendar] Final daysByDate:', daysByDate);
            } else {
                console.warn('[renderSlotCalendar] No orderDays data - using current month');
                // Verwende aktuellen Monat als Fallback
                if (type === 'pickup') {
                    pickupMonth = new Date().getMonth();
                    pickupYear = new Date().getFullYear();
                } else {
                    returnMonth = new Date().getMonth();
                    returnYear = new Date().getFullYear();
                }
            }

            // Stelle sicher, dass Monat und Jahr gültig sind
            if (type === 'pickup') {
                if (isNaN(pickupMonth) || isNaN(pickupYear)) {
                    pickupMonth = new Date().getMonth();
                    pickupYear = new Date().getFullYear();
                }
            } else {
                if (isNaN(returnMonth) || isNaN(returnYear)) {
                    returnMonth = new Date().getMonth();
                    returnYear = new Date().getFullYear();
                }
            }

            renderCalendar(containerId, daysByDate, type);
        }

        function renderCalendar(containerId, daysByDate, type) {
            console.log('[renderCalendar] Called with:', { containerId, daysByDate, type });
            const container = $(`#${containerId}`);
            const monthNames = ['Januar', 'Februar', 'März', 'April', 'Mai', 'Juni', 'Juli', 'August', 'September', 'Oktober', 'November', 'Dezember'];
            const dayNames = ['So', 'Mo', 'Di', 'Mi', 'Do', 'Fr', 'Sa'];
            
            let currentMonth = type === 'pickup' ? pickupMonth : returnMonth;
            let currentYear = type === 'pickup' ? pickupYear : returnYear;
            
            console.log('[renderCalendar] Using month:', currentMonth, 'year:', currentYear);
            
            // Validate month/year
            if (isNaN(currentMonth) || isNaN(currentYear)) {
                console.error('[renderCalendar] Invalid month/year:', { currentMonth, currentYear });
                currentMonth = new Date().getMonth();
                currentYear = new Date().getFullYear();
                if (type === 'pickup') {
                    pickupMonth = currentMonth;
                    pickupYear = currentYear;
                } else {
                    returnMonth = currentMonth;
                    returnYear = currentYear;
                }
                console.log('[renderCalendar] Reset to current date:', { currentMonth, currentYear });
            }
            
            const firstDay = new Date(currentYear, currentMonth, 1);
            const lastDay = new Date(currentYear, currentMonth + 1, 0);
            const daysInMonth = lastDay.getDate();
            const startingDayOfWeek = firstDay.getDay();
            
            console.log('[renderCalendar] Calendar info:', { firstDay, lastDay, daysInMonth, startingDayOfWeek });

            let html = `
                <div class="calendar-header">
                    <div class="calendar-nav">
                        <button class="calendar-nav-btn" onclick="changeMonth('${containerId}', '${type}', -1)">
                            <i class="fas fa-chevron-left"></i>
                        </button>
                        <button class="calendar-nav-btn" onclick="changeMonth('${containerId}', '${type}', 1)">
                            <i class="fas fa-chevron-right"></i>
                        </button>
                    </div>
                    <div class="calendar-month">${monthNames[currentMonth]} ${currentYear}</div>
                    <button class="calendar-nav-btn" onclick="showCustomDateRange('${containerId}', '${type}')" title="Erweiterte Suche" style="margin-left: 0.5rem;">
                        <i class="fas fa-search"></i>
                    </button>
                </div>
                <div class="calendar-grid">
            `;

            console.log('[renderCalendar] Generated header HTML, now generating day headers...');
            console.log('[renderCalendar] monthNames[currentMonth]:', monthNames[currentMonth], 'currentYear:', currentYear);

            // Wochentage-Header
            dayNames.forEach(day => {
                html += `<div class="calendar-day-header">${day}</div>`;
            });

            // Leere Zellen für Tage vor Monatsbeginn
            for (let i = 0; i < startingDayOfWeek; i++) {
                html += `<div class="calendar-day disabled"></div>`;
            }

            // Kalender-Tage
            const today = new Date();
            today.setHours(0, 0, 0, 0);

            for (let day = 1; day <= daysInMonth; day++) {
                const date = new Date(currentYear, currentMonth, day);
                date.setHours(0, 0, 0, 0);
                const dateStr = date.toISOString().split('T')[0];
                
                // Prüfe ob dieses Datum in daysByDate existiert (mit verschiedenen Format-Varianten)
                let hasSlots = daysByDate[dateStr];
                if (!hasSlots) {
                    // Versuche auch mit Zeitzone-Format
                    const dateStrWithTz = dateStr + '+01:00';
                    hasSlots = daysByDate[dateStrWithTz];
                }
                
                const isToday = date.getTime() === today.getTime();
                const isPast = date < today;

                let dayClass = 'calendar-day';
                if (isPast) dayClass += ' disabled';
                if (isToday) dayClass += ' today';
                
                let slotCount = 0;
                let totalCapacity = 0;
                
                if (hasSlots && Array.isArray(hasSlots)) {
                    slotCount = hasSlots.length;
                    hasSlots.forEach(slot => {
                        const cap = parseInt(slot.freeCapacity || slot.free_capacity || slot.capacity || 0);
                        if (!isNaN(cap) && cap > 0) {
                            totalCapacity += cap;
                        }
                    });
                }
                
                // Wenn keine Slots oder keine Kapazität, markiere als nicht verfügbar
                if (!hasSlots || totalCapacity === 0) {
                    dayClass += ' no-capacity';
                } else {
                    dayClass += ' available';
                }

                const onClickHandler = (isPast || !hasSlots || totalCapacity === 0) ? '' : `selectDate('${dateStr}', '${containerId}', '${type}'); event.stopPropagation();`;
                const capacityText = `${totalCapacity} Platz${totalCapacity !== 1 ? 'e' : ''} verfügbar`;
                
                html += `
                    <div class="${dayClass}" 
                         data-date="${dateStr}" 
                         ${onClickHandler ? `onclick="${onClickHandler}"` : ''}
                         style="${onClickHandler ? 'cursor: pointer;' : 'cursor: not-allowed;'}">
                        <div class="calendar-day-number">${day}</div>
                        ${hasSlots && totalCapacity > 0 ? `<div class="calendar-day-slots">${totalCapacity >= 999 ? 'Verfügbar' : totalCapacity + ' frei'}</div>` : ''}
                        ${hasSlots && totalCapacity > 0 ? `<div class="capacity-tooltip">${capacityText}</div>` : ''}
                    </div>
                `;
            }

            html += `</div>`;

            container.html(html);
            
            // Füge Slot-Container hinzu (wird dynamisch positioniert)
            if (!container.find('.slot-list-container').length) {
                container.append(`<div class="slot-list-container" id="${containerId}Slots"></div>`);
            }
        }

        function changeMonth(containerId, type, direction) {
            if (type === 'pickup') {
                pickupMonth += direction;
                if (pickupMonth < 0) {
                    pickupMonth = 11;
                    pickupYear--;
                } else if (pickupMonth > 11) {
                    pickupMonth = 0;
                    pickupYear++;
                }
            } else {
                returnMonth += direction;
                if (returnMonth < 0) {
                    returnMonth = 11;
                    returnYear--;
                } else if (returnMonth > 11) {
                    returnMonth = 0;
                    returnYear++;
                }
            }

            // Lade Daten neu und rendere nur den Kalender neu (nicht die Slots)
            const autohausId = $('#autohausSelect').val();
            const action = type === 'pickup' ? 'get_pickup_order_dates' : 'get_return_order_dates';
            
            fetch(`../api/resy_api_routes.php?action=${action}&autohaus_id=${autohausId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success && data.orderDays) {
                        const daysByDate = {};
                        data.orderDays.forEach(day => {
                            const dateKey = (day.date || day.Date || day.dateStr || '').split('+')[0].split('T')[0];
                            if (!daysByDate[dateKey]) {
                                daysByDate[dateKey] = [];
                            }
                            daysByDate[dateKey].push(day);
                        });
                        renderCalendar(containerId, daysByDate, type);
                    }
                })
                .catch(error => console.error('Fehler:', error));
        }
        
        function showCustomDateRange(containerId, type) {
            const modalContent = `
                <div class="form-group">
                    <label class="form-label">Ab Datum</label>
                    <input type="date" id="customStartDate" class="form-control" value="${new Date().toISOString().split('T')[0]}" required>
                </div>
                <div class="form-group">
                    <label class="form-label">Anzahl Tage voraus</label>
                    <input type="number" id="customForwardDays" class="form-control" min="1" max="90" value="30" required>
                    <small style="color: var(--text-muted); font-size: 0.75rem;">Wie viele Tage ab dem Startdatum sollen durchsucht werden?</small>
                </div>
                <div style="display: flex; gap: 0.5rem; justify-content: flex-end; margin-top: 1.5rem;">
                    <button class="btn btn-secondary" onclick="closeModal('customDateRangeModal')">Abbrechen</button>
                    <button class="btn btn-primary" onclick="loadCustomDateRange('${containerId}', '${type}')">
                        <i class="fas fa-search"></i> Suchen
                    </button>
                </div>
            `;
            
            createModal('customDateRangeModal', `Erweiterte Terminsuche - ${type === 'pickup' ? 'Abholung' : 'Rücklieferung'}`, modalContent);
        }
        
        function loadCustomDateRange(containerId, type) {
            const startDate = $('#customStartDate').val();
            const forwardDays = parseInt($('#customForwardDays').val());
            
            if (!startDate || !forwardDays || forwardDays < 1) {
                showAlert('Bitte geben Sie ein gültiges Datum und Anzahl Tage ein', 'warning');
                return;
            }
            
            const autohausId = $('#autohausSelect').val();
            const action = type === 'pickup' ? 'get_pickup_order_dates_custom' : 'get_return_order_dates_custom';
            
            showLoading(true, 'Suche Termine...', 'Lade verfügbare Termine für den gewählten Zeitraum');
            
            fetch(`../api/resy_api_routes.php?action=${action}&autohaus_id=${autohausId}&date=${startDate}&forwardDays=${forwardDays}`)
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    closeModal('customDateRangeModal');
                    
                    if (data.success && data.orderDays && data.orderDays.length > 0) {
                        const daysByDate = {};
                        data.orderDays.forEach(day => {
                            const dateKey = (day.date || day.Date || day.dateStr || '').split('+')[0].split('T')[0];
                            if (!daysByDate[dateKey]) {
                                daysByDate[dateKey] = [];
                            }
                            daysByDate[dateKey].push(day);
                        });
                        
                        // Setze Monat zum ersten verfügbaren Datum
                        const dates = Object.keys(daysByDate).sort();
                        if (dates.length > 0) {
                            const firstDateStr = dates[0];
                            const dateParts = firstDateStr.split('-');
                            if (dateParts.length === 3) {
                                const firstDate = new Date(parseInt(dateParts[0]), parseInt(dateParts[1]) - 1, parseInt(dateParts[2]));
                                if (!isNaN(firstDate.getTime())) {
                                    if (type === 'pickup') {
                                        pickupMonth = firstDate.getMonth();
                                        pickupYear = firstDate.getFullYear();
                                    } else {
                                        returnMonth = firstDate.getMonth();
                                        returnYear = firstDate.getFullYear();
                                    }
                                }
                            }
                        }
                        
                        renderCalendar(containerId, daysByDate, type);
                        showAlert(`✅ ${data.orderDays.length} Termine gefunden`, 'success');
                    } else {
                        showAlert('Keine Termine im gewählten Zeitraum gefunden', 'warning');
                        // Lade Standard-Termine neu
                        const standardAction = type === 'pickup' ? 'get_pickup_order_dates' : 'get_return_order_dates';
                        fetch(`../api/resy_api_routes.php?action=${standardAction}&autohaus_id=${autohausId}`)
                            .then(response => response.json())
                            .then(data => {
                                if (data.success && data.orderDays) {
                                    renderSlotCalendar(containerId, data.orderDays, type);
                                }
                            });
                    }
                })
                .catch(error => {
                    showLoading(false);
                    showAlert('Fehler beim Laden der Termine: ' + error.message, 'danger');
                });
        }

        function selectDate(dateStr, containerId, type) {
            console.log('[selectDate] Called with:', { dateStr, containerId, type });
            
            // Markiere ausgewähltes Datum zuerst
            $(`#${containerId} .calendar-day`).removeClass('selected');
            const selectedDay = $(`#${containerId} .calendar-day[data-date="${dateStr}"]`);
            
            // Prüfe ob Tag verfügbar ist
            if (selectedDay.hasClass('no-capacity') || selectedDay.hasClass('disabled')) {
                showAlert('Dieses Datum ist nicht verfügbar', 'warning');
                return;
            }
            
            selectedDay.addClass('selected');
            
            // Positioniere Slot-Container neben dem ausgewählten Tag
            const calendarWrapper = $(`#${containerId}`).closest('.slot-calendar-wrapper');
            let slotContainer = $(`#${containerId}Slots`);
            
            if (slotContainer.length === 0) {
                if (calendarWrapper.length) {
                    calendarWrapper.append(`<div class="slot-list-container" id="${containerId}Slots"></div>`);
                } else {
                    $(`#${containerId}`).closest('.slot-calendar').after(`<div class="slot-list-container" id="${containerId}Slots"></div>`);
                }
                slotContainer = $(`#${containerId}Slots`);
            }
            
            // Berechne Position relativ zum Kalender-Container
            setTimeout(() => {
                const dayPosition = selectedDay.position();
                const calendarPosition = $(`#${containerId}`).position();
                
                if (dayPosition && calendarPosition) {
                    const leftPosition = dayPosition.left + selectedDay.outerWidth() + 10;
                    const topPosition = dayPosition.top;
                    
                    slotContainer.css({
                        position: 'absolute',
                        left: leftPosition + 'px',
                        top: topPosition + 'px',
                        right: 'auto'
                    });
                }
            }, 50);
            
            // Lade Slots für diesen Tag
            const autohausId = $('#autohausSelect').val();
            const action = type === 'pickup' ? 'get_pickup_order_dates' : 'get_return_order_dates';
            
            fetch(`../api/resy_api_routes.php?action=${action}&autohaus_id=${autohausId}`)
                .then(response => response.json())
                .then(data => {
                    console.log('[selectDate] Received data:', data);
                    if (data.success && data.orderDays) {
                        // Filtere nach Datum - beachte verschiedene Formate
                        const daySlots = data.orderDays.filter(day => {
                            let dayDate = day.date || day.Date || day.dateStr || '';
                            // Normalisiere Datum (entferne Zeitzone)
                            dayDate = dayDate.split('+')[0].split('T')[0];
                            return dayDate === dateStr;
                        });
                        
                        console.log('[selectDate] Filtered daySlots for date:', dateStr, daySlots);
                        renderSlots(containerId + 'Slots', daySlots, dateStr, type, containerId);
                    } else {
                        console.warn('[selectDate] No data or orderDays in response');
                        slotContainer.html('<div class="no-slots-message">Keine Daten verfügbar</div>').addClass('active');
                    }
                })
                .catch(error => {
                    console.error('[selectDate] Fehler:', error);
                    slotContainer.html(`<div class="no-slots-message">Fehler beim Laden der Slots: ${error.message}</div>`).addClass('active');
                });
        }

        function renderSlots(containerId, slots, dateStr, type, calendarContainerId) {
            const container = $(`#${containerId}`);
            
            // Debug: Log was wir bekommen
            console.log('renderSlots called with:', { containerId, slots, dateStr, type });
            
            // slots ist ein Array von orderDays für dieses Datum
            if (!slots || slots.length === 0) {
                container.html('<div class="no-slots-message">Keine Slots verfügbar für dieses Datum</div>');
                return;
            }

            const selectedDateInput = type === 'pickup' ? '#pickupSelectedDate' : '#returnSelectedDate';
            $(selectedDateInput).val(dateStr);

            // Hole die freie Kapazität für dieses Datum - slots ist Array von orderDays Objekten
            const daySlot = slots[0]; // Erstes Slot-Objekt für dieses Datum
            console.log('[renderSlots] daySlot:', daySlot);
            
            let freeCapacity = 0;
            if (daySlot) {
                // Versuche verschiedene mögliche Eigenschaftsnamen
                const cap = daySlot.freeCapacity || daySlot.free_capacity || daySlot.capacity || 0;
                freeCapacity = parseInt(cap);
                if (isNaN(freeCapacity) || freeCapacity < 0) {
                    freeCapacity = 0;
                }
                // Wenn freeCapacity sehr groß ist (2147483647 = max int), zeige "Verfügbar"
                if (freeCapacity > 1000000) {
                    freeCapacity = 999; // Zeige hohe Zahl aber nicht unrealistisch
                }
            }
            
            console.log('[renderSlots] freeCapacity:', freeCapacity);

            const formattedDate = new Date(dateStr + 'T00:00:00').toLocaleDateString('de-DE', {
                weekday: 'long',
                year: 'numeric',
                month: 'long',
                day: 'numeric'
            });

            // Hole verfügbare Zeiten von der API
            const autohausId = $('#autohausSelect').val();
            fetch(`../api/resy_api_routes.php?action=get_order_times&autohaus_id=${autohausId}`)
                .then(response => response.json())
                .then(timeData => {
                    console.log('timeData:', timeData);
                    let availableTimes = ['08:00', '09:00', '10:00', '11:00', '12:00', '13:00', '14:00', '15:00', '16:00', '17:00'];
                    
                    if (timeData.success && timeData.orderTimes && timeData.orderTimes.length > 0) {
                        availableTimes = timeData.orderTimes;
                    }

                    const capacityText = `${freeCapacity} Platz${freeCapacity !== 1 ? 'e' : ''}`;
                    
                    let html = `
                        <div class="slot-date-header">
                            <div>
                                <i class="fas fa-calendar-day"></i> ${formattedDate}
                                ${freeCapacity > 0 ? `<span style="margin-left: 1rem; color: var(--primary); font-size: 0.875rem;">${capacityText} verfügbar</span>` : ''}
                            </div>
                            <button class="close-slots" onclick="closeSlotPanel('${containerId}')" title="Schließen">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                        <div class="slots-grid">
                    `;

                    if (freeCapacity > 0 && availableTimes.length > 0) {
                        availableTimes.forEach(time => {
                            html += `
                                <div class="slot-item" 
                                     data-date="${dateStr}"
                                     data-time="${time}"
                                     onclick="selectSlot('${dateStr}', '${time}', '${type}')">
                                    <div class="slot-time">${time}</div>
                                    <div class="slot-capacity">Verfügbar</div>
                                </div>
                            `;
                        });
                    } else {
                        html += '<div class="no-slots-message">Keine freien Plätze verfügbar für dieses Datum</div>';
                    }

                    html += `</div>`;
                    container.html(html).addClass('active');
                    
                    // Positioniere Container neu nach dem Rendern
                    if (calendarContainerId) {
                        setTimeout(() => {
                            const selectedDay = $(`#${calendarContainerId} .calendar-day.selected`);
                            if (selectedDay.length) {
                                const dayPosition = selectedDay.position();
                                const calendarPosition = $(`#${calendarContainerId}`).position();
                                
                                if (dayPosition && calendarPosition) {
                                    const leftPosition = dayPosition.left + selectedDay.outerWidth() + 10;
                                    const topPosition = dayPosition.top;
                                    container.css({
                                        position: 'absolute',
                                        left: leftPosition + 'px',
                                        top: topPosition + 'px',
                                        right: 'auto'
                                    });
                                }
                            }
                        }, 50);
                    }
                })
                .catch(error => {
                    console.error('[renderSlots] Fehler beim Laden der Zeiten:', error);
                    // Fallback: Zeige Standard-Zeiten
                    const defaultTimes = ['08:00', '09:00', '10:00', '11:00', '12:00', '13:00', '14:00', '15:00', '16:00', '17:00'];
                    const capacityTextFallback = `${freeCapacity} Platz${freeCapacity !== 1 ? 'e' : ''}`;
                    
                    let html = `
                        <div class="slot-date-header">
                            <div>
                                <i class="fas fa-calendar-day"></i> ${formattedDate}
                                ${freeCapacity > 0 ? `<span style="margin-left: 1rem; color: var(--primary); font-size: 0.875rem;">${capacityTextFallback} verfügbar</span>` : ''}
                            </div>
                            <button class="close-slots" onclick="closeSlotPanel('${containerId}')" title="Schließen">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                        <div class="slots-grid">
                    `;

                    if (freeCapacity > 0 && defaultTimes.length > 0) {
                        defaultTimes.forEach(time => {
                            html += `
                                <div class="slot-item" 
                                     data-date="${dateStr}"
                                     data-time="${time}"
                                     onclick="selectSlot('${dateStr}', '${time}', '${type}')">
                                    <div class="slot-time">${time}</div>
                                    <div class="slot-capacity">Verfügbar</div>
                                </div>
                            `;
                        });
                    } else {
                        html += '<div class="no-slots-message">Keine freien Plätze verfügbar für dieses Datum</div>';
                    }

                    html += `</div>`;
                    container.html(html).addClass('active');
                });
        }

        function selectSlot(date, time, type) {
            const dateInput = type === 'pickup' ? '#pickupSelectedDate' : '#returnSelectedDate';
            const timeInput = type === 'pickup' ? '#pickupSelectedTime' : '#returnSelectedTime';
            const containerId = type === 'pickup' ? 'pickupSlotCalendar' : 'returnSlotCalendar';
            
            $(dateInput).val(date);
            $(timeInput).val(time);
            
            // Markiere ausgewählten Slot
            $(`.slot-item`).removeClass('selected');
            $(`.slot-item[data-date="${date}"][data-time="${time}"]`).addClass('selected');
            
            // Zeige Zeit im Kalendertag an
            const selectedDay = $(`#${containerId} .calendar-day[data-date="${date}"]`);
            if (selectedDay.length) {
                // Entferne vorherige Zeit-Anzeige
                selectedDay.find('.selected-time').remove();
                // Füge neue Zeit-Anzeige hinzu
                selectedDay.append(`<div class="selected-time">${time}</div>`);
            }
            
            // Schließe Slot-Panel
            closeSlotPanel(containerId + 'Slots');
        }

        function selectServicePackage(value) {
            $('.select-card[data-value="' + value + '"]').each(function() {
                const isTarget = $(this).parent().attr('id') === 'servicePackageCards';
                if (isTarget) {
                    $('#servicePackageCards .select-card').removeClass('selected');
                    $(this).addClass('selected');
                    $('#servicePackageSelect').val(value);
                }
            });
        }

        function selectLoadingPoint(value) {
            $('.select-card[data-value="' + value + '"]').each(function() {
                const isTarget = $(this).parent().attr('id') === 'loadingPointCards';
                if (isTarget) {
                    $('#loadingPointCards .select-card').removeClass('selected');
                    $(this).addClass('selected');
                    $('#loadingPointSelect').val(value);
                }
            });
        }

        function selectDeliveryOption(value) {
            $('.select-card[data-value="' + value + '"]').each(function() {
                const isTarget = $(this).parent().attr('id') === 'deliveryOptionCards';
                if (isTarget) {
                    $('#deliveryOptionCards .select-card').removeClass('selected');
                    $(this).addClass('selected');
                    $('#deliveryOptionSelect').val(value);
                }
            });
        }

        function closeSlotPanel(containerId) {
            $(`#${containerId}`).removeClass('active');
        }

        // Schließe Slot-Panel beim Klick außerhalb
        $(document).on('click', function(e) {
            if (!$(e.target).closest('.slot-list-container, .calendar-day.available').length) {
                $('.slot-list-container').removeClass('active');
            }
        });

        function submitPickupOrder() {
            const autohausId = $('#autohausSelect').val();
            const date = $('#pickupSelectedDate').val();
            const time = $('#pickupSelectedTime').val();
            const servicePackage = $('#servicePackageSelect').val();
            const location = $('#loadingPointSelect').val();
            
            // Erweiterte Parameter - nur verwenden wenn Auto-Fill aktiv war
            const tireCount = $('#pickupTireCount').val() && !$('#pickupTireCount').prop('disabled') ? parseInt($('#pickupTireCount').val()) : null;
            const tireCapCount = $('#pickupTireCapCount').val() && !$('#pickupTireCapCount').prop('disabled') ? parseInt($('#pickupTireCapCount').val()) : null;
            const tireType = $('#pickupTireType').val() && !$('#pickupTireType').prop('disabled') ? $('#pickupTireType').val() : null;
            const offset = $('#pickupOffset').val() && !$('#pickupOffset').prop('disabled') ? parseInt($('#pickupOffset').val()) : 0;

            if (!date || !time || !servicePackage || !location) {
                showAlert('Bitte wählen Sie Datum, Zeit und alle weiteren Felder aus', 'warning');
                return;
            }

            const orders = [];
            selectedWheelsets.forEach(wsNum => {
                const ws = wheelsetsData.find(w => w.satznummer == wsNum);
                if (ws) {
                    // Auto-Fill aus Wheelset-Details wenn verfügbar
                    let finalTireCount = tireCount;
                    let finalTireCapCount = tireCapCount;
                    let finalTireType = tireType;
                    
                    // Lade Wheelset-Details für Auto-Fill
                    const wheelsetDetails = wheelsetsData.find(w => w.satznummer == wsNum);
                    if (wheelsetDetails && !finalTireCount && wheelsetDetails.wheels) {
                        finalTireCount = wheelsetDetails.wheels.length;
                    }
                    if (wheelsetDetails && !finalTireType && wheelsetDetails.wheels && wheelsetDetails.wheels.length > 0) {
                        const firstTireType = wheelsetDetails.wheels[0].tireType;
                        if (firstTireType) {
                            finalTireType = firstTireType;
                        }
                    }
                    
                    const orderData = {
                        wheelSetNumber: parseInt(wsNum),
                        appointment: date,
                        location: location,
                        servicePackageIdent: servicePackage,
                        licenceNo: ws.kennzeichen || '',
                        VIN: ws.fin || ''
                    };
                    
                    // Füge erweiterte Parameter hinzu wenn gesetzt
                    if (finalTireCount !== null) orderData.tireCount = finalTireCount;
                    if (finalTireCapCount !== null) orderData.tireCapCount = finalTireCapCount;
                    if (finalTireType) orderData.tireType = finalTireType;
                    if (offset !== 0) orderData.offset = offset;
                    
                    orders.push(orderData);
                }
            });

            showLoading(true);
            fetch(`../api/resy_api_routes.php?action=create_pickup_order&autohaus_id=${autohausId}`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ pickupOrders: orders })
            })
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    if (data.success) {
                        showAlert('✅ Abholung erfolgreich bestellt!', 'success');
                        closeModal('pickupModal');
                        loadWheelsets();
                        // Update Orders Panel falls offen
                        if ($('#ordersPanel').hasClass('open')) {
                            setTimeout(() => {
                                renderOrdersTab('pickup');
                            }, 500);
                        }
                    } else {
                        showAlert('❌ Fehler: ' + (data.error || 'Unbekannt'), 'danger');
                    }
                })
                .catch(error => {
                    showLoading(false);
                    showAlert('Fehler: ' + error.message, 'danger');
                });
        }

        function submitReturnOrder() {
            const autohausId = $('#autohausSelect').val();
            const date = $('#returnSelectedDate').val();
            const time = $('#returnSelectedTime').val();
            const deliveryOption = $('#deliveryOptionSelect').val();
            const comment = $('#returnComment').val();
            const offset = $('#returnOffset').val() && !$('#returnOffset').prop('disabled') ? parseInt($('#returnOffset').val()) : 0;

            if (!date || !time || !deliveryOption) {
                showAlert('Bitte wählen Sie Datum, Zeit und Lieferoption aus', 'warning');
                return;
            }

            const orderData = {
                wheelSetNumbers: Array.from(selectedWheelsets).map(n => parseInt(n)),
                date: date,
                deliveryOption: deliveryOption,
                comment: comment || ''
            };
            
            // Füge offset hinzu wenn gesetzt
            if (offset !== 0) {
                orderData.offset = offset;
            }
            
            // Füge time hinzu wenn vorhanden
            if (time) {
                orderData.time = time;
            }
            
            const orders = [orderData];

            showLoading(true);
            fetch(`../api/resy_api_routes.php?action=create_return_order&autohaus_id=${autohausId}`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ returnOrders: orders })
            })
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    if (data.success) {
                        showAlert('✅ Rücklieferung erfolgreich bestellt!', 'success');
                        closeModal('returnModal');
                        loadWheelsets();
                        // Update Orders Panel falls offen
                        if ($('#ordersPanel').hasClass('open')) {
                            setTimeout(() => {
                                renderOrdersTab('return');
                            }, 500);
                        }
                    } else {
                        showAlert('❌ Fehler: ' + (data.error || 'Unbekannt'), 'danger');
                    }
                })
                .catch(error => {
                    showLoading(false);
                    showAlert('Fehler: ' + error.message, 'danger');
                });
        }

        function createModal(id, title, content, hasSubmitButton = false, submitFunction = null, submitLabel = 'Speichern') {
            let footerButtons = `<button class="btn btn-secondary" onclick="closeModal('${id}')">Abbrechen</button>`;
            
            if (id === 'pickupModal') {
                footerButtons += '<button class="btn btn-primary" onclick="submitPickupOrder()">Bestellen</button>';
            } else if (id === 'returnModal') {
                footerButtons += '<button class="btn btn-primary" onclick="submitReturnOrder()">Bestellen</button>';
            } else if (hasSubmitButton && submitFunction) {
                footerButtons += `<button class="btn btn-primary" onclick="${submitFunction}()">${submitLabel}</button>`;
            }
            
            const modal = $(`
                <div class="modal-overlay active" id="${id}" onclick="if(event.target.id === '${id}') closeModal('${id}')">
                    <div class="modal">
                        <div class="modal-header">
                            <div class="modal-title">${title}</div>
                            <button class="modal-close" onclick="closeModal('${id}')">&times;</button>
                        </div>
                        <div class="modal-body">${content}</div>
                        <div class="modal-footer">
                            ${footerButtons}
                        </div>
                    </div>
                </div>
            `);
            $('#modalContainer').html(modal);
        }

        function closeModal(id) {
            $(`#${id}`).remove();
        }

        function showResyFeatures() {
            const autohausId = $('#autohausSelect').val();
            if (!autohausId) {
                showAlert('Bitte wählen Sie ein Autohaus aus', 'warning');
                return;
            }

            const featuresContent = `
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem; margin-bottom: 1.5rem;">
                    <button class="btn btn-primary" onclick="checkServiceStatus()" style="width: 100%;">
                        <i class="fas fa-heartbeat"></i> Service Status
                    </button>
                    <button class="btn btn-primary" onclick="getServiceVersion()" style="width: 100%;">
                        <i class="fas fa-code-branch"></i> Version
                    </button>
                    <button class="btn btn-primary" onclick="getAllPickupOrders()" style="width: 100%;">
                        <i class="fas fa-truck"></i> Alle Abholungen
                    </button>
                    <button class="btn btn-primary" onclick="getAllReturnOrders()" style="width: 100%;">
                        <i class="fas fa-arrow-left"></i> Alle Rücklieferungen
                    </button>
                    <button class="btn btn-primary" onclick="getInventory()" style="width: 100%;">
                        <i class="fas fa-boxes"></i> Inventar
                    </button>
                    <button class="btn btn-primary" onclick="getExtraOrders()" style="width: 100%;">
                        <i class="fas fa-shopping-cart"></i> Extra Bestellungen
                    </button>
                    <button class="btn btn-primary" onclick="getCustomerInfo()" style="width: 100%;">
                        <i class="fas fa-building"></i> Kundeninfo
                    </button>
                    <button class="btn btn-primary" onclick="getAllowedCustomers()" style="width: 100%;">
                        <i class="fas fa-users"></i> Kundenliste
                    </button>
                    <button class="btn btn-primary" onclick="searchByVehicle()" style="width: 100%;">
                        <i class="fas fa-search"></i> Fahrzeug suchen
                    </button>
                </div>
                <div id="resyFeaturesResult"></div>
            `;

            createModal('resyFeaturesModal', 'RESY API Features', featuresContent);
        }

        function showToolsModal() {
            const toolsContent = `
                <div style="display: grid; gap: 1.5rem;">
                    <div class="form-group">
                        <label class="form-label">
                            <i class="fas fa-check-circle"></i> Wheelset-Nummer prüfen
                        </label>
                        <div style="display: flex; gap: 0.5rem;">
                            <input type="text" id="checkWheelsetInput" class="form-control" placeholder="z.B. 103379" maxlength="10">
                            <button class="btn btn-primary" onclick="checkWheelsetNumber()">
                                <i class="fas fa-check"></i> Prüfen
                            </button>
                        </div>
                        <div id="checkWheelsetResult" style="margin-top: 0.75rem;"></div>
                    </div>

                    <div class="form-group">
                        <label class="form-label">
                            <i class="fas fa-car"></i> Kennzeichen prüfen
                        </label>
                        <div style="display: flex; gap: 0.5rem;">
                            <input type="text" id="checkLicenseInput" class="form-control" placeholder="z.B. M-UH999" maxlength="20">
                            <button class="btn btn-primary" onclick="checkLicensePlate()">
                                <i class="fas fa-check"></i> Prüfen
                            </button>
                        </div>
                        <div id="checkLicenseResult" style="margin-top: 0.75rem;"></div>
                    </div>

                    <div class="form-group">
                        <label class="form-label">
                            <i class="fas fa-barcode"></i> VIN prüfen
                        </label>
                        <div style="display: flex; gap: 0.5rem;">
                            <input type="text" id="checkVINInput" class="form-control" placeholder="z.B. W0L000051T2125000" maxlength="17">
                            <button class="btn btn-primary" onclick="checkVIN()">
                                <i class="fas fa-check"></i> Prüfen
                            </button>
                        </div>
                        <div id="checkVINResult" style="margin-top: 0.75rem;"></div>
                    </div>
                </div>
            `;

            createModal('toolsModal', 'Tools & Validierung', toolsContent);
        }

        function checkWheelsetNumber() {
            const number = $('#checkWheelsetInput').val();
            if (!number) {
                $('#checkWheelsetResult').html('<div class="alert alert-warning">Bitte geben Sie eine Nummer ein</div>');
                return;
            }
            
            const autohausId = $('#autohausSelect').val();
            if (!autohausId) {
                $('#checkWheelsetResult').html('<div class="alert alert-warning">Bitte wählen Sie ein Autohaus aus</div>');
                return;
            }

            $('#checkWheelsetResult').html('<div style="text-align: center;"><div class="spinner" style="width: 24px; height: 24px; margin: 0 auto;"></div></div>');
            
            fetch(`../api/resy_api_routes.php?action=check_wheelset&autohaus_id=${autohausId}&wheelset_number=${number}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success && data.result && data.result.result) {
                        $('#checkWheelsetResult').html('<div class="alert alert-success">✅ <strong>Wheelset-Nummer ist gültig</strong><br><small>Nummer: ' + number + '</small></div>');
                    } else {
                        $('#checkWheelsetResult').html('<div class="alert alert-danger">❌ <strong>Wheelset-Nummer ist ungültig</strong><br><small>Nummer: ' + number + '</small></div>');
                    }
                })
                .catch(error => {
                    $('#checkWheelsetResult').html(`<div class="alert alert-danger">Fehler: ${error.message}</div>`);
                });
        }

        function checkLicensePlate() {
            const plate = $('#checkLicenseInput').val();
            if (!plate) {
                $('#checkLicenseResult').html('<div class="alert alert-warning">Bitte geben Sie ein Kennzeichen ein</div>');
                return;
            }
            
            const autohausId = $('#autohausSelect').val();
            if (!autohausId) {
                $('#checkLicenseResult').html('<div class="alert alert-warning">Bitte wählen Sie ein Autohaus aus</div>');
                return;
            }

            $('#checkLicenseResult').html('<div style="text-align: center;"><div class="spinner" style="width: 24px; height: 24px; margin: 0 auto;"></div></div>');
            
            fetch(`../api/resy_api_routes.php?action=check_license_plate&autohaus_id=${autohausId}&license_plate=${encodeURIComponent(plate)}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success && data.result && data.result.result) {
                        const normalized = data.result.normalizedLicencePlate || plate;
                        $('#checkLicenseResult').html(`<div class="alert alert-success">✅ <strong>Kennzeichen ist gültig</strong><br><small>Original: ${plate}<br>Normalisiert: ${normalized}</small></div>`);
                    } else {
                        $('#checkLicenseResult').html(`<div class="alert alert-danger">❌ <strong>Kennzeichen ist ungültig</strong><br><small>${plate}</small></div>`);
                    }
                })
                .catch(error => {
                    $('#checkLicenseResult').html(`<div class="alert alert-danger">Fehler: ${error.message}</div>`);
                });
        }

        function checkVIN() {
            const vin = $('#checkVINInput').val();
            if (!vin) {
                $('#checkVINResult').html('<div class="alert alert-warning">Bitte geben Sie eine VIN ein</div>');
                return;
            }
            
            const autohausId = $('#autohausSelect').val();
            if (!autohausId) {
                $('#checkVINResult').html('<div class="alert alert-warning">Bitte wählen Sie ein Autohaus aus</div>');
                return;
            }

            $('#checkVINResult').html('<div style="text-align: center;"><div class="spinner" style="width: 24px; height: 24px; margin: 0 auto;"></div></div>');
            
            fetch(`../api/resy_api_routes.php?action=check_vin&autohaus_id=${autohausId}&vin=${encodeURIComponent(vin)}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success && data.result && data.result.result) {
                        const normalized = data.result.normalizedVIN || vin;
                        $('#checkVINResult').html(`<div class="alert alert-success">✅ <strong>VIN ist gültig</strong><br><small>Original: ${vin}<br>Normalisiert: ${normalized}</small></div>`);
                    } else {
                        $('#checkVINResult').html(`<div class="alert alert-danger">❌ <strong>VIN ist ungültig</strong><br><small>${vin}</small></div>`);
                    }
                })
                .catch(error => {
                    $('#checkVINResult').html(`<div class="alert alert-danger">Fehler: ${error.message}</div>`);
                });
        }

        function checkServiceStatus() {
            showLoading(true);
            fetch(`http://145.253.121.150:8087/resy-rest-extern/amialive/`)
                .then(response => response.text())
                .then(text => {
                    showLoading(false);
                    $('#resyFeaturesResult').html(`<div class="alert alert-success"><strong>Status:</strong> ${text}</div>`);
                })
                .catch(error => {
                    showLoading(false);
                    $('#resyFeaturesResult').html(`<div class="alert alert-danger">Fehler: ${error.message}</div>`);
                });
        }

        function getServiceVersion() {
            showLoading(true);
            fetch(`http://145.253.121.150:8087/resy-rest-extern/serviceversion/`)
                .then(response => response.text())
                .then(text => {
                    showLoading(false);
                    $('#resyFeaturesResult').html(`<div class="alert alert-info"><strong>Version:</strong> ${text}</div>`);
                })
                .catch(error => {
                    showLoading(false);
                    $('#resyFeaturesResult').html(`<div class="alert alert-danger">Fehler: ${error.message}</div>`);
                });
        }

        function getAllPickupOrders() {
            const autohausId = $('#autohausSelect').val();
            showLoading(true);
            fetch(`../api/resy_api_routes.php?action=get_pickup_orders&autohaus_id=${autohausId}`)
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    if (data.success) {
                        let html = `<div class="alert alert-info"><strong>Abholungen:</strong> ${data.count || 0}</div>`;
                        if (data.pickupOrders && data.pickupOrders.length > 0) {
                            html += '<table style="width:100%; margin-top: 1rem;"><thead><tr><th>Satznummer</th><th>Datum</th><th>Ort</th><th>Service</th></tr></thead><tbody>';
                            data.pickupOrders.forEach(order => {
                                html += `<tr><td>${order.wheelSetNumber}</td><td>${order.appointment || '-'}</td><td>${order.location || '-'}</td><td>${order.servicePackageName || '-'}</td></tr>`;
                            });
                            html += '</tbody></table>';
                        }
                        $('#resyFeaturesResult').html(html);
                    } else {
                        $('#resyFeaturesResult').html(`<div class="alert alert-danger">Fehler: ${data.error}</div>`);
                    }
                })
                .catch(error => {
                    showLoading(false);
                    $('#resyFeaturesResult').html(`<div class="alert alert-danger">Fehler: ${error.message}</div>`);
                });
        }

        function getAllReturnOrders() {
            const autohausId = $('#autohausSelect').val();
            showLoading(true);
            fetch(`../api/resy_api_routes.php?action=get_return_orders&autohaus_id=${autohausId}`)
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    if (data.success) {
                        let html = `<div class="alert alert-info"><strong>Rücklieferungen:</strong> ${data.count || 0}</div>`;
                        if (data.returnOrders && data.returnOrders.length > 0) {
                            html += '<table style="width:100%; margin-top: 1rem;"><thead><tr><th>Satznummer</th><th>Datum</th><th>Option</th></tr></thead><tbody>';
                            data.returnOrders.forEach(order => {
                                html += `<tr><td>${order.wheelSetNumber}</td><td>${order.date || '-'}</td><td>${order.deliveryOption || '-'}</td></tr>`;
                            });
                            html += '</tbody></table>';
                        }
                        $('#resyFeaturesResult').html(html);
                    } else {
                        $('#resyFeaturesResult').html(`<div class="alert alert-danger">Fehler: ${data.error}</div>`);
                    }
                })
                .catch(error => {
                    showLoading(false);
                    $('#resyFeaturesResult').html(`<div class="alert alert-danger">Fehler: ${error.message}</div>`);
                });
        }

        function getInventory() {
            const autohausId = $('#autohausSelect').val();
            showLoading(true);
            fetch(`../api/resy_api_routes.php?action=get_inventory&autohaus_id=${autohausId}`)
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    if (data.success && data.inventory) {
                        let html = '<div class="alert alert-info"><strong>Inventar:</strong></div>';
                        data.inventory.forEach(item => {
                            const count = item.inventoryNumbers ? item.inventoryNumbers.length : 0;
                            html += `<div style="background: var(--bg-hover); padding: 1rem; border-radius: var(--radius); margin-top: 0.5rem;">
                                <strong>${item.inventoryType || 'Unbekannt'}:</strong> ${count} Stück
                            </div>`;
                        });
                        $('#resyFeaturesResult').html(html);
                    } else {
                        $('#resyFeaturesResult').html(`<div class="alert alert-danger">Fehler: ${data.error || 'Keine Daten'}</div>`);
                    }
                })
                .catch(error => {
                    showLoading(false);
                    $('#resyFeaturesResult').html(`<div class="alert alert-danger">Fehler: ${error.message}</div>`);
                });
        }

        function getExtraOrders() {
            const autohausId = $('#autohausSelect').val();
            showLoading(true);
            fetch(`../api/resy_api_routes.php?action=get_customer_details&autohaus_id=${autohausId}`)
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    $('#resyFeaturesResult').html(`<div class="alert alert-info">Extra Orders Feature - Implementierung folgt</div>`);
                })
                .catch(error => {
                    showLoading(false);
                    $('#resyFeaturesResult').html(`<div class="alert alert-danger">Fehler: ${error.message}</div>`);
                });
        }

        function getCustomerInfo() {
            const autohausId = $('#autohausSelect').val();
            showLoading(true);
            fetch(`../api/resy_api_routes.php?action=get_customer_details&autohaus_id=${autohausId}`)
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    if (data.success && data.customer) {
                        const customer = data.customer;
                        const html = `
                            <div class="alert alert-info"><strong>Kundeninformationen</strong></div>
                            <div style="background: var(--bg-hover); padding: 1rem; border-radius: var(--radius); margin-top: 0.5rem;">
                                <div style="margin-bottom: 0.5rem;"><strong>Kundennummer:</strong> ${customer.customerNo || '-'}</div>
                                <div style="margin-bottom: 0.5rem;"><strong>Name:</strong> ${customer.name1 || ''} ${customer.name2 || ''}</div>
                                <div style="margin-bottom: 0.5rem;"><strong>Adresse:</strong> ${customer.address || '-'}</div>
                                <div style="margin-bottom: 0.5rem;"><strong>PLZ:</strong> ${customer.postalcode || '-'}</div>
                                <div><strong>Stadt:</strong> ${customer.city || '-'}</div>
                            </div>
                        `;
                        $('#resyFeaturesResult').html(html);
                    } else {
                        $('#resyFeaturesResult').html(`<div class="alert alert-danger">Kundeninformationen konnten nicht geladen werden</div>`);
                    }
                })
                .catch(error => {
                    showLoading(false);
                    $('#resyFeaturesResult').html(`<div class="alert alert-danger">Fehler: ${error.message}</div>`);
                });
        }

        function getAllowedCustomers() {
            showLoading(true);
            const autohausId = $('#autohausSelect').val();
            const username = 'BUE1';
            const password = 'resytest';
            const auth = btoa(username + ':' + password);

            fetch(`http://145.253.121.150:8087/resy-rest-extern/v1/customer/`, {
                headers: { 'Authorization': 'Basic ' + auth }
            })
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    if (data.customers) {
                        let html = `<div class="alert alert-info"><strong>Kunden:</strong> ${data.customers.length}</div>`;
                        html += '<div style="max-height: 300px; overflow-y: auto; margin-top: 0.5rem;">';
                        data.customers.forEach(customer => {
                            html += `<div style="background: var(--bg-hover); padding: 0.75rem; border-radius: var(--radius); margin-bottom: 0.5rem;">
                                <strong>${customer.customerNo}:</strong> ${customer.name || '-'}
                            </div>`;
                        });
                        html += '</div>';
                        $('#resyFeaturesResult').html(html);
                    } else {
                        $('#resyFeaturesResult').html(`<div class="alert alert-danger">Keine Daten</div>`);
                    }
                })
                .catch(error => {
                    showLoading(false);
                    $('#resyFeaturesResult').html(`<div class="alert alert-danger">Fehler: ${error.message}</div>`);
                });
        }

        function searchByVehicle() {
            const searchContent = `
                <div class="form-group">
                    <label class="form-label">Suche nach VIN oder Kennzeichen</label>
                    <div style="display: flex; gap: 0.5rem;">
                        <input type="text" id="vehicleVIN" class="form-control" placeholder="VIN (17 Zeichen)">
                        <input type="text" id="vehicleLicense" class="form-control" placeholder="Kennzeichen">
                    </div>
                    <button class="btn btn-primary" onclick="performVehicleSearch()" style="width: 100%; margin-top: 0.75rem;">
                        <i class="fas fa-search"></i> Suchen
                    </button>
                </div>
                <div id="vehicleSearchResult"></div>
            `;
            $('#resyFeaturesResult').html(searchContent);
        }

        function performVehicleSearch() {
            const autohausId = $('#autohausSelect').val();
            const vin = $('#vehicleVIN').val();
            const license = $('#vehicleLicense').val();

            if (!vin && !license) {
                $('#vehicleSearchResult').html('<div class="alert alert-warning">Bitte geben Sie VIN oder Kennzeichen ein</div>');
                return;
            }

            showLoading(true);
            fetch(`../api/resy_api_routes.php?action=get_wheelsets_by_vehicle&autohaus_id=${autohausId}&vin=${encodeURIComponent(vin || '')}&license_no=${encodeURIComponent(license || '')}`)
                .then(response => response.json())
                .then(data => {
                    showLoading(false);
                    if (data.success && data.wheelsets) {
                        let html = `<div class="alert alert-success"><strong>Gefunden:</strong> ${data.wheelsets.length} Wheelset(s)</div>`;
                        if (data.wheelsets.length > 0) {
                            html += '<div style="margin-top: 1rem;">';
                            data.wheelsets.forEach(ws => {
                                html += `<div style="background: var(--bg-hover); padding: 1rem; border-radius: var(--radius); margin-bottom: 0.5rem;">
                                    <strong>Satznummer:</strong> ${ws.wheelSetNumber || '-'}<br>
                                    <strong>Abholdatum:</strong> ${ws.pickupDate || '-'}<br>
                                    <strong>Rückgabedatum:</strong> ${ws.returnDate || '-'}
                                </div>`;
                            });
                            html += '</div>';
                        }
                        $('#vehicleSearchResult').html(html);
                    } else {
                        $('#vehicleSearchResult').html('<div class="alert alert-warning">Keine Wheelsets für dieses Fahrzeug gefunden</div>');
                    }
                })
                .catch(error => {
                    showLoading(false);
                    $('#vehicleSearchResult').html(`<div class="alert alert-danger">Fehler: ${error.message}</div>`);
                });
        }

        function showAlert(message, type, duration = 3000) {
            const alert = $(`
                <div class="alert alert-${type}">
                    ${message}
                </div>
            `);
            $('#alertContainer').html(alert);
            setTimeout(() => alert.fadeOut(() => alert.remove()), duration);
        }

        function getPositionIcon(position) {
            const pos = (position || '').toUpperCase();
            if (pos.includes('VORNE') || pos.includes('FRONT') || pos.includes('VORDER')) {
                return '⬆️';
            } else if (pos.includes('HINTEN') || pos.includes('REAR') || pos.includes('HECK')) {
                return '⬇️';
            } else if (pos.includes('LINKS') || pos.includes('LEFT')) {
                return '⬅️';
            } else if (pos.includes('RECHTS') || pos.includes('RIGHT')) {
                return '➡️';
            }
            return '🛞';
        }

        function viewTestReport(url, wheelsetNumber) {
            const viewer = $('#testReportViewer');
            const iframe = $('#testReportIframe');
            const downloadLink = $('#testReportDownload');
            const title = $('#testReportTitle');
            
            title.text(`Prüfbericht - Satz #${wheelsetNumber}`);
            iframe.attr('src', url);
            downloadLink.attr('href', url);
            
            viewer.addClass('active');
            document.body.style.overflow = 'hidden';
        }

        function closeTestReport() {
            const viewer = $('#testReportViewer');
            const iframe = $('#testReportIframe');
            
            viewer.removeClass('active');
            setTimeout(() => {
                iframe.attr('src', '');
            }, 300);
            document.body.style.overflow = '';
        }

        // ESC-Taste zum Schließen
        $(document).keydown(function(e) {
            if (e.key === 'Escape' && $('#testReportViewer').hasClass('active')) {
                closeTestReport();
            }
        });

        function showLoading(show, text = null, subtext = null) {
            const overlay = $('#loadingOverlay');
            const loadingText = $('#loadingText');
            const loadingSubtext = $('#loadingSubtext');
            
            if (show) {
                if (text) loadingText.text(text);
                else loadingText.text('Wird geladen...');
                
                if (subtext) loadingSubtext.text(subtext);
                else loadingSubtext.text('Bitte warten Sie einen Moment');
                
                overlay.addClass('active');
            } else {
                overlay.removeClass('active');
                // Reset nach Animation
                setTimeout(() => {
                    loadingText.text('Wird geladen...');
                    loadingSubtext.text('Bitte warten Sie einen Moment');
                }, 300);
            }
        }

        // Compact Loading Helper (für kleine Bereiche)
        function showCompactLoading(element, show, text = 'Lädt...') {
            if (!element) return;
            
            const $el = $(element);
            
            if (show) {
                if ($el.find('.loading-compact').length > 0) return;
                
                const loading = $(`
                    <div class="loading-compact" style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); z-index: 1000;">
                        <div class="loading-compact-spinner"></div>
                        <div class="loading-compact-text">${text}</div>
                    </div>
                `);
                
                $el.css('position', 'relative');
                $el.append(loading);
            } else {
                $el.find('.loading-compact').fadeOut(200, function() {
                    $(this).remove();
                });
            }
        }
    </script>
</body>
</html>
