<?php
session_start();
require_once 'config/database.php';

// Get token from URL
$token = $_GET['token'] ?? '';

if (empty($token)) {
    die('Ungültiger Token');
}

// Validate token and get offer
$offer = null;
$tokenValid = false;
$tokenExpired = false;
$tokenUsed = false;

try {
    $database = new Database();
    $conn = $database->getConnection();
    
    if (!$conn) {
        throw new Exception("Keine Datenbankverbindung möglich");
    }
    
    // Validate token
    $stmt = $conn->prepare("
        SELECT * FROM wheelset_offers
        WHERE token = ? AND token_expires_at > NOW() AND token_used_at IS NULL
    ");
    $stmt->execute([$token]);
    $tokenData = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$tokenData) {
        // Check if token exists but is expired or used
        $stmt = $conn->prepare("SELECT * FROM wheelset_offers WHERE token = ?");
        $stmt->execute([$token]);
        $expiredToken = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($expiredToken) {
            if ($expiredToken['token_used_at']) {
                $tokenUsed = true;
            } else {
                $tokenExpired = true;
            }
        }
        
        // Show appropriate message
        if ($tokenUsed) {
            die('
            <!DOCTYPE html>
            <html lang="de">
            <head>
                <meta charset="UTF-8">
                <meta name="viewport" content="width=device-width, initial-scale=1.0">
                <title>Angebot bereits angenommen</title>
                <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
                <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
                <style>
                    body { font-family: Inter, sans-serif; background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%); margin: 0; padding: 20px; min-height: 100vh; display: flex; align-items: center; justify-content: center; }
                    .container { max-width: 600px; margin: 0 auto; background: white; border-radius: 16px; padding: 40px; text-align: center; box-shadow: 0 10px 40px rgba(0,0,0,0.1); }
                    .icon { font-size: 4rem; color: #28a745; margin-bottom: 20px; }
                    h1 { color: #2c3e50; margin-bottom: 20px; font-weight: 700; }
                    p { color: #6c757d; margin-bottom: 30px; line-height: 1.6; }
                </style>
            </head>
            <body>
                <div class="container">
                    <div class="icon"><i class="fas fa-check-circle"></i></div>
                    <h1>Angebot bereits angenommen</h1>
                    <p>Sie haben das Angebot bereits angenommen. Falls Sie sich umentschieden haben, kontaktieren Sie uns gerne!</p>
                </div>
            </body>
            </html>');
        } else {
            die('
            <!DOCTYPE html>
            <html lang="de">
            <head>
                <meta charset="UTF-8">
                <meta name="viewport" content="width=device-width, initial-scale=1.0">
                <title>Token abgelaufen</title>
                <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
                <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
                <style>
                    body { font-family: Inter, sans-serif; background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%); margin: 0; padding: 20px; min-height: 100vh; display: flex; align-items: center; justify-content: center; }
                    .container { max-width: 600px; margin: 0 auto; background: white; border-radius: 16px; padding: 40px; text-align: center; box-shadow: 0 10px 40px rgba(0,0,0,0.1); }
                    .icon { font-size: 4rem; color: #dc3545; margin-bottom: 20px; }
                    h1 { color: #2c3e50; margin-bottom: 20px; font-weight: 700; }
                    p { color: #6c757d; margin-bottom: 30px; line-height: 1.6; }
                </style>
            </head>
            <body>
                <div class="container">
                    <div class="icon"><i class="fas fa-clock"></i></div>
                    <h1>Token abgelaufen</h1>
                    <p>Der Link ist abgelaufen (24 Stunden gültig). Bitte kontaktieren Sie uns für ein neues Angebot.</p>
                </div>
            </body>
            </html>');
        }
    }
    
    $offer = $tokenData;
    $tokenValid = true;
    
    // Decode JSON fields
    $offerData = json_decode($offer['offer_data'], true);
    $newTires = $offerData['new_tires'] ?? [];
    $selectedServices = $offerData['selected_services'] ?? [];
    
    // Get services details
    $services = [];
    if (!empty($selectedServices)) {
        $placeholders = str_repeat('?,', count($selectedServices) - 1) . '?';
        $stmt = $conn->prepare("
            SELECT id, service_name, service_description, price, show_price
            FROM autohaus_services 
            WHERE id IN ($placeholders) AND is_active = 1
        ");
        $stmt->execute($selectedServices);
        $services = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Get autohaus data
    $stmt = $conn->prepare("SELECT name FROM autohaus WHERE id = ?");
    $stmt->execute([$offer['autohaus_id']]);
    $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
    
} catch (Exception $e) {
    die('Fehler: ' . $e->getMessage());
}

// Handle form submission
$success = false;
$error = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    try {
        // Direkt SQL ausführen statt über API
        $stmt = $conn->prepare("
            SELECT * FROM wheelset_offers
            WHERE token = ? AND token_expires_at > NOW() AND token_used_at IS NULL
        ");
        $stmt->execute([$token]);
        $offerCheck = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$offerCheck) {
            throw new Exception('Token ungültig oder abgelaufen');
        }
        
        $status = 'viewed';
        $customerSelection = [];
        
        if ($action === 'decline') {
            // Angebot ablehnen
            $status = 'declined';
            $customerSelection = [
                'action' => 'declined',
                'decline_reason' => $_POST['decline_reason'] ?? ''
            ];
        } elseif ($action === 'new_request') {
            // Neues Angebot anfragen
            $status = 'new_request';
            $customerSelection = [
                'action' => 'new_request',
                'request_notes' => $_POST['request_notes'] ?? ''
            ];
        } else {
            // Normale Auswahl
            $customerTireSelection = $_POST['tire_selection'] ?? [];
            $customerServiceSelection = $_POST['service_selection'] ?? [];
            $customerNotes = $_POST['customer_notes'] ?? '';
            
            if (!empty($customerTireSelection) || !empty($customerServiceSelection)) {
                $status = 'accepted';
                $customerSelection['action'] = 'accepted';
            }
            
            $customerSelection['tires'] = $customerTireSelection;
            $customerSelection['services'] = $customerServiceSelection;
            $customerSelection['notes'] = $customerNotes;
        }
        
        // Markiere Token als verwendet und speichere Auswahl
        $stmt = $conn->prepare("
            UPDATE wheelset_offers 
            SET customer_selection = ?, token_used_at = NOW(), status = ?
            WHERE id = ?
        ");
        $stmt->execute([json_encode($customerSelection), $status, $offerCheck['id']]);
        
        // Sende Teams-Benachrichtigung
        if (in_array($status, ['accepted', 'declined', 'new_request'])) {
            require_once __DIR__ . '/api/send_wheelset_offer_teams_notification.php';
            $teamsResult = sendWheelsetOfferTeamsNotification($offerCheck['id'], $status);
            // Log aber wirf keinen Fehler wenn Teams nicht funktioniert
            if (!$teamsResult['success']) {
                error_log("Teams Notification Failed: " . $teamsResult['error']);
            }
        }
        
        $success = true;
        
    } catch (Exception $e) {
        $error = 'Fehler: ' . $e->getMessage();
    }
}

// Company Settings
$stmt = $conn->prepare("SELECT setting_key, setting_value FROM settings WHERE setting_key IN ('company_name', 'company_logo', 'app_name')");
$stmt->execute();
$companySettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
$companyName = $companySettings['company_name'] ?? 'B&E Gruppe';
$companyLogo = $companySettings['company_logo'] ?? 'assets/images/BE-Logo-FINAL_Schwarz-1.png';
$companyLogo = ltrim($companyLogo, '/');
$appName = $companySettings['app_name'] ?? 'B&E Gruppe';
?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Ihr Reifenangebot - <?php echo htmlspecialchars($companyName); ?></title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&family=Space+Grotesk:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="assets/css/style.css">
    <link rel="stylesheet" href="assets/css/volvo-theme.css">
    <style>
        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            min-height: 100vh;
            padding: 0;
            margin: 0;
        }
        
        .navbar {
            background: white;
            box-shadow: 0 2px 10px rgba(0,0,0,0.08);
            padding: 1rem 0;
            position: sticky;
            top: 0;
            z-index: 1000;
        }
        
        .nav-container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 2rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .nav-logo {
            height: 40px;
        }
        
        .hero-section {
            background: linear-gradient(135deg, #003057 0%, #0056b3 100%);
            color: white;
            padding: 4rem 2rem;
            text-align: center;
        }
        
        .hero-title {
            font-size: 2.5rem;
            font-weight: 700;
            margin-bottom: 1rem;
        }
        
        .hero-subtitle {
            font-size: 1.25rem;
            opacity: 0.9;
        }
        
        .container {
            max-width: 1000px;
            margin: 0 auto;
            padding: 3rem 2rem;
        }
        
        .offer-card {
            background: white;
            border-radius: 16px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.1);
            overflow: hidden;
            margin-bottom: 2rem;
        }
        
        .card-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 2rem;
        }
        
        .card-content {
            padding: 2rem;
        }
        
        .info-box {
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            padding: 1.5rem;
            border-radius: 12px;
            margin-bottom: 2rem;
            border-left: 4px solid #667eea;
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
        }
        
        .info-item {
            display: flex;
            flex-direction: column;
        }
        
        .info-label {
            font-size: 0.875rem;
            color: #6c757d;
            margin-bottom: 0.25rem;
        }
        
        .info-value {
            font-weight: 600;
            color: #2c3e50;
            font-size: 1.1rem;
        }
        
        .section {
            margin-bottom: 2rem;
            padding: 2rem;
            border: 1px solid #e9ecef;
            border-radius: 12px;
            background: white;
        }
        
        .section-title {
            font-size: 1.5rem;
            font-weight: 700;
            color: #2c3e50;
            margin-bottom: 1.5rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .section-title i {
            color: #667eea;
            font-size: 1.75rem;
        }
        
        .tire-grid {
            display: grid;
            gap: 1rem;
        }
        
        .tire-item {
            border: 2px solid #e9ecef;
            border-radius: 12px;
            padding: 1.5rem;
            cursor: pointer;
            transition: all 0.3s ease;
            background: white;
            position: relative;
            overflow: hidden;
        }
        
        .tire-item::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 4px;
            height: 100%;
            background: #667eea;
            transform: scaleY(0);
            transition: transform 0.3s ease;
        }
        
        .tire-item:hover {
            border-color: #667eea;
            background: #f8f9ff;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(102, 126, 234, 0.15);
        }
        
        .tire-item:hover::before {
            transform: scaleY(1);
        }
        
        .tire-item.selected {
            border-color: #667eea;
            background: linear-gradient(135deg, #e7f1ff 0%, #f0f4ff 100%);
            box-shadow: 0 4px 20px rgba(102, 126, 234, 0.2);
        }
        
        .tire-item.selected::before {
            transform: scaleY(1);
        }
        
        .tire-item label {
            display: block;
            cursor: pointer;
        }
        
        .tire-item input[type="checkbox"] {
            position: absolute;
            opacity: 0;
            pointer-events: none;
        }
        
        .tire-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 1rem;
        }
        
        .tire-name {
            font-weight: 700;
            font-size: 1.1rem;
            color: #2c3e50;
            margin-bottom: 0.5rem;
        }
        
        .tire-details {
            display: flex;
            gap: 1rem;
            color: #6c757d;
            font-size: 0.9rem;
            flex-wrap: wrap;
        }
        
        .tire-detail-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .tire-price-section {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: 1rem;
            padding-top: 1rem;
            border-top: 1px solid #e9ecef;
        }
        
        .tire-price-label {
            font-size: 0.875rem;
            color: #6c757d;
        }
        
        .tire-price-value {
            font-weight: 700;
            font-size: 1.5rem;
            color: #667eea;
        }
        
        .service-grid {
            display: grid;
            gap: 0.75rem;
        }
        
        .service-item {
            border: 2px solid #e9ecef;
            border-radius: 10px;
            padding: 1.25rem;
            display: flex;
            align-items: center;
            gap: 1rem;
            cursor: pointer;
            transition: all 0.3s ease;
            background: white;
        }
        
        .service-item:hover {
            border-color: #667eea;
            background: #f8f9ff;
            transform: translateX(4px);
        }
        
        .service-item input[type="checkbox"] {
            width: 24px;
            height: 24px;
            cursor: pointer;
            accent-color: #667eea;
        }
        
        .service-content {
            flex: 1;
        }
        
        .service-name {
            font-weight: 600;
            color: #2c3e50;
            margin-bottom: 0.25rem;
        }
        
        .service-description {
            font-size: 0.875rem;
            color: #6c757d;
        }
        
        .service-price {
            font-weight: 700;
            font-size: 1.25rem;
            color: #667eea;
        }
        
        .notes-section {
            margin-bottom: 2rem;
        }
        
        .notes-area {
            width: 100%;
            min-height: 120px;
            padding: 1rem;
            border: 2px solid #e9ecef;
            border-radius: 10px;
            font-family: inherit;
            font-size: 0.95rem;
            resize: vertical;
            transition: border-color 0.3s ease;
        }
        
        .notes-area:focus {
            outline: none;
            border-color: #667eea;
        }
        
        .summary-box {
            background: linear-gradient(135deg, #fff3c7 0%, #ffeaa7 100%);
            border: 2px solid #f59e0b;
            border-radius: 12px;
            padding: 2rem;
            margin: 2rem 0;
            display: none;
        }
        
        .summary-box.active {
            display: block;
        }
        
        .summary-title {
            font-weight: 700;
            font-size: 1.25rem;
            color: #92400e;
            margin-bottom: 1.5rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .summary-items {
            margin-bottom: 1.5rem;
        }
        
        .summary-item {
            display: flex;
            justify-content: space-between;
            padding: 0.75rem 0;
            border-bottom: 1px solid rgba(242, 153, 74, 0.3);
            color: #78350f;
        }
        
        .summary-item:last-child {
            border-bottom: none;
        }
        
        .summary-total {
            display: flex;
            justify-content: space-between;
            font-weight: 700;
            font-size: 1.5rem;
            padding-top: 1.5rem;
            border-top: 2px solid #f59e0b;
            color: #78350f;
        }
        
        .btn-actions {
            display: grid;
            grid-template-columns: 1fr 1fr 1fr;
            gap: 1rem;
            margin-top: 2rem;
        }
        
        .btn-submit, .btn-decline, .btn-new-request {
            padding: 1.25rem 2rem;
            color: white;
            border: none;
            border-radius: 12px;
            font-size: 1.1rem;
            font-weight: 700;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.75rem;
        }
        
        .btn-submit {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
        }
        
        .btn-submit:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.4);
        }
        
        .btn-decline {
            background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
            box-shadow: 0 4px 15px rgba(220, 53, 69, 0.3);
        }
        
        .btn-decline:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(220, 53, 69, 0.4);
        }
        
        .btn-new-request {
            background: linear-gradient(135deg, #17a2b8 0%, #138496 100%);
            box-shadow: 0 4px 15px rgba(23, 162, 184, 0.3);
        }
        
        .btn-new-request:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(23, 162, 184, 0.4);
        }
        
        .btn-submit:disabled, .btn-decline:disabled, .btn-new-request:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
        }
        
        .action-modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.5);
            z-index: 10000;
            align-items: center;
            justify-content: center;
        }
        
        .action-modal.active {
            display: flex;
        }
        
        .action-modal-content {
            background: white;
            border-radius: 16px;
            padding: 2rem;
            max-width: 500px;
            width: 90%;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
        }
        
        .action-modal-content h3 {
            margin-top: 0;
            margin-bottom: 1rem;
            font-size: 1.5rem;
        }
        
        .action-modal-content p {
            color: #6c757d;
            margin-bottom: 1.5rem;
        }
        
        .action-modal-textarea {
            width: 100%;
            min-height: 120px;
            padding: 1rem;
            border: 2px solid #e9ecef;
            border-radius: 10px;
            font-family: inherit;
            font-size: 0.95rem;
            resize: vertical;
            margin-bottom: 1.5rem;
        }
        
        .action-modal-textarea:focus {
            outline: none;
            border-color: #667eea;
        }
        
        .action-modal-buttons {
            display: flex;
            gap: 1rem;
            justify-content: flex-end;
        }
        
        .btn-modal-cancel {
            padding: 0.75rem 1.5rem;
            background: #6c757d;
            color: white;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
        }
        
        .btn-modal-submit {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 600;
            color: white;
        }
        
        .btn-modal-submit.decline {
            background: #dc3545;
        }
        
        .btn-modal-submit.request {
            background: #17a2b8;
        }
        
        @media (max-width: 768px) {
            .btn-actions {
                grid-template-columns: 1fr;
            }
        }
        
        .success-message {
            background: linear-gradient(135deg, #d1fae5 0%, #a7f3d0 100%);
            border: 2px solid #10b981;
            border-radius: 16px;
            padding: 3rem 2rem;
            text-align: center;
            margin: 2rem 0;
        }
        
        .success-message .icon {
            font-size: 4rem;
            color: #10b981;
            margin-bottom: 1.5rem;
        }
        
        .success-message h2 {
            color: #065f46;
            margin-bottom: 1rem;
            font-weight: 700;
        }
        
        .success-message p {
            color: #047857;
            font-size: 1.1rem;
            line-height: 1.6;
        }
        
        .error-message {
            background: linear-gradient(135deg, #fee2e2 0%, #fecaca 100%);
            border: 2px solid #ef4444;
            border-radius: 12px;
            padding: 1.5rem;
            margin: 1.5rem 0;
            color: #991b1b;
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .error-message i {
            font-size: 1.5rem;
        }
        
        @media (max-width: 768px) {
            .hero-title {
                font-size: 1.75rem;
            }
            
            .hero-subtitle {
                font-size: 1rem;
            }
            
            .container {
                padding: 2rem 1rem;
            }
            
            .card-content {
                padding: 1.5rem;
            }
            
            .section {
                padding: 1.5rem;
            }
            
            .info-box {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar">
        <div class="nav-container">
            <div style="display: flex; align-items: center; gap: 1rem;">
                <img src="<?php echo htmlspecialchars($companyLogo); ?>" alt="<?php echo htmlspecialchars($companyName); ?>" class="nav-logo">
                <span style="font-weight: 600; color: #2c3e50;"><?php echo htmlspecialchars($appName); ?></span>
            </div>
        </div>
    </nav>

    <!-- Hero Section -->
    <section class="hero-section">
        <div class="hero-title">
            <i class="fas fa-file-invoice-dollar"></i> Ihr Reifenangebot
        </div>
        <div class="hero-subtitle">Bitte treffen Sie Ihre Auswahl</div>
    </section>

    <div class="container">
        <?php if ($success): ?>
            <div class="success-message">
                <div class="icon"><i class="fas fa-check-circle"></i></div>
                <h2>Auswahl erfolgreich übermittelt!</h2>
                <p>Vielen Dank für Ihre Auswahl. Wir werden uns in Kürze bei Ihnen melden.</p>
            </div>
        <?php else: ?>
            
            <div class="offer-card">
                <div class="card-content">
                    <div class="info-box">
                        <div class="info-item">
                            <span class="info-label"><i class="fas fa-car"></i> Kennzeichen</span>
                            <span class="info-value"><?php echo htmlspecialchars($offer['license_plate'] ?? '-'); ?></span>
                        </div>
                        <div class="info-item">
                            <span class="info-label"><i class="fas fa-hashtag"></i> Satznummer</span>
                            <span class="info-value"><?php echo htmlspecialchars($offer['wheelset_number'] ?? '-'); ?></span>
                        </div>
                        <?php if ($offer['customer_name']): ?>
                        <div class="info-item">
                            <span class="info-label"><i class="fas fa-user"></i> Kunde</span>
                            <span class="info-value"><?php echo htmlspecialchars($offer['customer_name']); ?></span>
                        </div>
                        <?php endif; ?>
                        <?php if ($autohaus): ?>
                        <div class="info-item">
                            <span class="info-label"><i class="fas fa-building"></i> Autohaus</span>
                            <span class="info-value"><?php echo htmlspecialchars($autohaus['name']); ?></span>
                        </div>
                        <?php endif; ?>
                    </div>
                    
                    <?php if ($error): ?>
                        <div class="error-message">
                            <i class="fas fa-exclamation-circle"></i>
                            <span><?php echo htmlspecialchars($error); ?></span>
                        </div>
                    <?php endif; ?>
                    
                    <form method="POST" id="selectionForm">
                        <!-- Reifen-Auswahl -->
                        <?php if (!empty($newTires)): ?>
                            <div class="section">
                                <div class="section-title">
                                    <i class="fas fa-tire"></i> Reifen auswählen
                                </div>
                                <div class="tire-grid">
                                    <?php foreach ($newTires as $index => $tire): ?>
                                        <div class="tire-item" data-tire-index="<?php echo $index; ?>">
                                            <label>
                                                <input type="checkbox" name="tire_selection[]" value="<?php echo $index; ?>" class="tire-checkbox">
                                                <div class="tire-header">
                                                    <div>
                                                        <div class="tire-name"><?php echo htmlspecialchars($tire['brand_model'] ?? ''); ?></div>
                                                        <div class="tire-details">
                                                            <?php if (!empty($tire['dimension'])): ?>
                                                                <div class="tire-detail-item">
                                                                    <i class="fas fa-ruler"></i>
                                                                    <span><?php echo htmlspecialchars($tire['dimension']); ?></span>
                                                                </div>
                                                            <?php endif; ?>
                                                            <?php if (!empty($tire['eu_label'])): ?>
                                                                <div class="tire-detail-item">
                                                                    <i class="fas fa-tag"></i>
                                                                    <span><?php echo htmlspecialchars($tire['eu_label']); ?></span>
                                                                </div>
                                                            <?php endif; ?>
                                                            <?php if (!empty($tire['rf'])): ?>
                                                                <div class="tire-detail-item">
                                                                    <i class="fas fa-info-circle"></i>
                                                                    <span>RF: <?php echo htmlspecialchars($tire['rf']); ?></span>
                                                                </div>
                                                            <?php endif; ?>
                                                            <div class="tire-detail-item">
                                                                <i class="fas fa-box"></i>
                                                                <span><?php echo htmlspecialchars($tire['quantity'] ?? 1); ?>x</span>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="tire-price-section">
                                                    <div>
                                                        <div class="tire-price-label">Einzelpreis</div>
                                                        <div style="color: #6c757d; font-size: 0.9rem;"><?php echo number_format($tire['price'] ?? 0, 2, ',', '.'); ?> €</div>
                                                    </div>
                                                    <div>
                                                        <div class="tire-price-label">Gesamt</div>
                                                        <div class="tire-price-value"><?php echo number_format($tire['total'] ?? 0, 2, ',', '.'); ?> €</div>
                                                    </div>
                                                </div>
                                            </label>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        <?php endif; ?>
                        
                        <!-- Service-Auswahl -->
                        <?php if (!empty($services)): ?>
                            <div class="section">
                                <div class="section-title">
                                    <i class="fas fa-cog"></i> Zusatzleistungen
                                </div>
                                <div class="service-grid">
                                    <?php foreach ($services as $service): ?>
                                        <div class="service-item">
                                            <input type="checkbox" name="service_selection[]" value="<?php echo $service['id']; ?>" class="service-checkbox">
                                            <div class="service-content">
                                                <div class="service-name"><?php echo htmlspecialchars($service['service_name']); ?></div>
                                                <?php if (!empty($service['service_description'])): ?>
                                                    <div class="service-description"><?php echo htmlspecialchars($service['service_description']); ?></div>
                                                <?php endif; ?>
                                            </div>
                                            <?php if ($service['show_price']): ?>
                                                <div class="service-price"><?php echo number_format($service['price'], 2, ',', '.'); ?> €</div>
                                            <?php endif; ?>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        <?php endif; ?>
                        
                        <!-- Anmerkungen -->
                        <div class="section notes-section">
                            <div class="section-title">
                                <i class="fas fa-comment"></i> Anmerkungen
                            </div>
                            <textarea name="customer_notes" class="notes-area" placeholder="Optionale Anmerkungen oder besondere Wünsche..."></textarea>
                        </div>
                        
                        <!-- Zusammenfassung -->
                        <div class="summary-box" id="summaryBox">
                            <div class="summary-title">
                                <i class="fas fa-calculator"></i> Zusammenfassung
                            </div>
                            <div class="summary-items" id="summaryContent"></div>
                            <div class="summary-total">
                                <span>Gesamt:</span>
                                <span id="totalAmount">0,00 €</span>
                            </div>
                        </div>
                        
                        <!-- Action Buttons -->
                        <div class="btn-actions">
                            <button type="submit" class="btn-submit" id="submitBtn">
                                <i class="fas fa-check"></i>
                                <span>Annehmen</span>
                            </button>
                            <button type="button" class="btn-decline" id="declineBtn">
                                <i class="fas fa-times"></i>
                                <span>Ablehnen</span>
                            </button>
                            <button type="button" class="btn-new-request" id="newRequestBtn">
                                <i class="fas fa-redo"></i>
                                <span>Neu anfragen</span>
                            </button>
                        </div>
                        
                        <!-- Decline Modal -->
                        <div id="declineModal" class="action-modal">
                            <div class="action-modal-content">
                                <h3 style="color: #dc3545;"><i class="fas fa-times-circle"></i> Angebot ablehnen</h3>
                                <p>Bitte geben Sie einen Grund für die Ablehnung an (optional):</p>
                                <textarea id="declineReason" class="action-modal-textarea" placeholder="Grund für die Ablehnung..."></textarea>
                                <div class="action-modal-buttons">
                                    <button type="button" class="btn-modal-cancel" onclick="closeDeclineModal()">Abbrechen</button>
                                    <button type="button" class="btn-modal-submit decline" onclick="submitDecline()">
                                        <i class="fas fa-times"></i> Ablehnen
                                    </button>
                                </div>
                            </div>
                        </div>
                        
                        <!-- New Request Modal -->
                        <div id="newRequestModal" class="action-modal">
                            <div class="action-modal-content">
                                <h3 style="color: #17a2b8;"><i class="fas fa-redo"></i> Neues Angebot anfragen</h3>
                                <p>Bitte beschreiben Sie Ihre Wünsche für ein neues Angebot:</p>
                                <textarea id="requestNotes" class="action-modal-textarea" placeholder="Ihre Wünsche für das neue Angebot..."></textarea>
                                <div class="action-modal-buttons">
                                    <button type="button" class="btn-modal-cancel" onclick="closeNewRequestModal()">Abbrechen</button>
                                    <button type="button" class="btn-modal-submit request" onclick="submitNewRequest()">
                                        <i class="fas fa-paper-plane"></i> Anfragen
                                    </button>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Confirmation Modal -->
                        <div id="confirmModal" class="action-modal">
                            <div class="action-modal-content">
                                <h3 style="color: #667eea;"><i class="fas fa-question-circle"></i> Bestätigung</h3>
                                <p id="confirmMessage">Möchten Sie Ihre Auswahl wirklich bestätigen?</p>
                                <div class="action-modal-buttons">
                                    <button type="button" class="btn-modal-cancel" onclick="closeConfirmModal()">Abbrechen</button>
                                    <button type="button" class="btn-modal-submit" style="background: #667eea;" onclick="confirmSubmit()">
                                        <i class="fas fa-check"></i> Bestätigen
                                    </button>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        <?php endif; ?>
    </div>
    
    <script>
        // Update tire item visual state
        document.querySelectorAll('.tire-checkbox').forEach(checkbox => {
            checkbox.addEventListener('change', function() {
                const item = this.closest('.tire-item');
                if (this.checked) {
                    item.classList.add('selected');
                } else {
                    item.classList.remove('selected');
                }
                updateSummary();
            });
        });
        
        // Calculate summary
        function updateSummary() {
            const selectedTires = Array.from(document.querySelectorAll('.tire-checkbox:checked')).map(cb => {
                const item = cb.closest('.tire-item');
                const priceText = item.querySelector('.tire-price-value').textContent;
                const price = parseFloat(priceText.replace(/[^\d,]/g, '').replace(',', '.'));
                const name = item.querySelector('.tire-name').textContent.trim();
                return { name, price };
            });
            
            const selectedServices = Array.from(document.querySelectorAll('.service-checkbox:checked')).map(cb => {
                const item = cb.closest('.service-item');
                const priceText = item.querySelector('.service-price')?.textContent || '0,00 €';
                const price = parseFloat(priceText.replace(/[^\d,]/g, '').replace(',', '.') || '0');
                const name = item.querySelector('.service-name').textContent.trim();
                return { name, price };
            });
            
            let total = 0;
            let html = '';
            
            selectedTires.forEach(tire => {
                html += `<div class="summary-item"><span>${tire.name}</span><span>${tire.price.toFixed(2).replace('.', ',')} €</span></div>`;
                total += tire.price;
            });
            
            selectedServices.forEach(service => {
                html += `<div class="summary-item"><span>${service.name}</span><span>${service.price.toFixed(2).replace('.', ',')} €</span></div>`;
                total += service.price;
            });
            
            document.getElementById('summaryContent').innerHTML = html;
            document.getElementById('totalAmount').textContent = total.toFixed(2).replace('.', ',') + ' €';
            
            const summaryBox = document.getElementById('summaryBox');
            if (html) {
                summaryBox.classList.add('active');
            } else {
                summaryBox.classList.remove('active');
            }
        }
        
        // Add event listeners for services
        document.querySelectorAll('.service-checkbox').forEach(cb => {
            cb.addEventListener('change', updateSummary);
        });
        
        // Initial summary update
        updateSummary();
        
        // Decline button handler
        document.getElementById('declineBtn').addEventListener('click', function() {
            document.getElementById('declineModal').classList.add('active');
        });
        
        function closeDeclineModal() {
            document.getElementById('declineModal').classList.remove('active');
        }
        
        function submitDecline() {
            const form = document.getElementById('selectionForm');
            const actionInput = document.createElement('input');
            actionInput.type = 'hidden';
            actionInput.name = 'action';
            actionInput.value = 'decline';
            form.appendChild(actionInput);
            
            const declineReason = document.getElementById('declineReason').value;
            const reasonInput = document.createElement('input');
            reasonInput.type = 'hidden';
            reasonInput.name = 'decline_reason';
            reasonInput.value = declineReason;
            form.appendChild(reasonInput);
            
            disableAllButtons();
            form.submit();
        }
        
        // New Request button handler
        document.getElementById('newRequestBtn').addEventListener('click', function() {
            document.getElementById('newRequestModal').classList.add('active');
        });
        
        function closeNewRequestModal() {
            document.getElementById('newRequestModal').classList.remove('active');
        }
        
        function submitNewRequest() {
            const form = document.getElementById('selectionForm');
            const actionInput = document.createElement('input');
            actionInput.type = 'hidden';
            actionInput.name = 'action';
            actionInput.value = 'new_request';
            form.appendChild(actionInput);
            
            const requestNotes = document.getElementById('requestNotes').value;
            const notesInput = document.createElement('input');
            notesInput.type = 'hidden';
            notesInput.name = 'request_notes';
            notesInput.value = requestNotes;
            form.appendChild(notesInput);
            
            disableAllButtons();
            form.submit();
        }
        
        function disableAllButtons() {
            const submitBtn = document.getElementById('submitBtn');
            const declineBtn = document.getElementById('declineBtn');
            const newRequestBtn = document.getElementById('newRequestBtn');
            
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> <span>Wird übermittelt...</span>';
            declineBtn.disabled = true;
            newRequestBtn.disabled = true;
        }
        
        // Form validation
        let pendingSubmit = false;
        
        document.getElementById('selectionForm').addEventListener('submit', function(e) {
            // Prüfe ob es ein normaler Submit ist (nicht Decline/NewRequest)
            const action = e.submitter?.dataset?.action || '';
            if (action === 'decline' || action === 'new_request') {
                return; // Wird von den Modal-Funktionen behandelt
            }
            
            // Verhindere Default-Submit
            e.preventDefault();
            
            const hasSelection = document.querySelectorAll('.tire-checkbox:checked, .service-checkbox:checked').length > 0;
            
            if (!hasSelection) {
                // Zeige Modal für keine Auswahl
                showConfirmModal(
                    'Sie haben keine Auswahl getroffen.',
                    'Möchten Sie trotzdem fortfahren? Ihre Auswahl kann danach nicht mehr geändert werden.'
                );
                return false;
            }
            
            // Zeige Bestätigungs-Modal
            showConfirmModal(
                'Auswahl bestätigen',
                'Möchten Sie Ihre Auswahl wirklich bestätigen? Diese kann danach nicht mehr geändert werden.'
            );
        });
        
        function showConfirmModal(title, message) {
            const modal = document.getElementById('confirmModal');
            const titleElement = modal.querySelector('h3');
            const messageElement = document.getElementById('confirmMessage');
            
            if (title) {
                titleElement.innerHTML = '<i class="fas fa-question-circle"></i> ' + title;
            }
            messageElement.textContent = message;
            
            modal.classList.add('active');
        }
        
        function closeConfirmModal() {
            document.getElementById('confirmModal').classList.remove('active');
            pendingSubmit = false;
        }
        
        function confirmSubmit() {
            const form = document.getElementById('selectionForm');
            closeConfirmModal();
            disableAllButtons();
            
            // Führe tatsächlichen Submit aus
            pendingSubmit = true;
            form.submit();
        }
        
        // Close modals on outside click
        document.getElementById('declineModal')?.addEventListener('click', function(e) {
            if (e.target === this) {
                closeDeclineModal();
            }
        });
        
        document.getElementById('newRequestModal')?.addEventListener('click', function(e) {
            if (e.target === this) {
                closeNewRequestModal();
            }
        });
        
        document.getElementById('confirmModal')?.addEventListener('click', function(e) {
            if (e.target === this) {
                closeConfirmModal();
            }
        });
    </script>
</body>
</html>
