<?php
/**
 * Sicherer Dokumenten-Viewer für Verifikationsdokumente
 * Prüft Authentifizierung (Admin oder User) vor dem Anzeigen von Dokumenten
 */

session_start();
require_once 'config/database.php';

// Prüfe Authentifizierung: Admin ODER User (der das Dokument gehört)
$is_admin = isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'];
$is_user = isset($_SESSION['user_logged_in']) && $_SESSION['user_logged_in'];
$user_id = $_SESSION['user_id'] ?? null;
$admin_user_id = $_SESSION['admin_user_id'] ?? null;

if (!$is_admin && !$is_user) {
    http_response_code(401);
    die('Nicht autorisiert');
}

$file_path = $_GET['file'] ?? null;

if (!$file_path) {
    http_response_code(400);
    die('Dateipfad erforderlich');
}

// URL-dekodiere den Pfad
$file_path = urldecode($file_path);

// Sicherheitsprüfung: Nur Dateien aus dem verification_documents Verzeichnis erlauben
$base_upload_dir = realpath(__DIR__ . '/uploads/verification_documents/');

if (!$base_upload_dir || !is_dir($base_upload_dir)) {
    http_response_code(500);
    die('Upload-Verzeichnis nicht gefunden');
}

// Konvertiere relativen Pfad zu absolutem Pfad falls nötig
if (substr($file_path, 0, 1) !== '/' && substr($file_path, 0, 2) !== 'C:') {
    // Relativer Pfad - füge Base-Directory hinzu
    // Entferne "uploads/verification_documents/" falls vorhanden
    $clean_path = str_replace('uploads/verification_documents/', '', $file_path);
    
    // Entferne "../" Pfade für Sicherheit
    $clean_path = str_replace('../', '', $clean_path);
    
    // Entferne führende "/" falls vorhanden
    $clean_path = ltrim($clean_path, '/');
    
    $full_path = $base_upload_dir . '/' . $clean_path;
    $requested_path = realpath($full_path);
    
    // Falls realpath fehlschlägt, verwende den konstruierten Pfad
    if (!$requested_path) {
        $requested_path = $full_path;
    }
} else {
    // Absoluter Pfad - prüfe ob er im erlaubten Verzeichnis liegt
    $requested_path = realpath($file_path);
    
    // Falls realpath fehlschlägt, verwende den ursprünglichen Pfad
    if (!$requested_path) {
        $requested_path = $file_path;
    }
}

// Sicherheitsprüfung: Pfad muss innerhalb des erlaubten Verzeichnisses sein
if (!$requested_path || substr($requested_path, 0, strlen($base_upload_dir)) !== $base_upload_dir) {
    http_response_code(403);
    die('Zugriff verweigert: Ungültiger Dateipfad');
}

// Prüfe ob Datei existiert
if (!file_exists($requested_path)) {
    http_response_code(404);
    die('Datei nicht gefunden');
}

// Wenn User (nicht Admin): Prüfe ob das Dokument dem User gehört
if ($is_user && !$is_admin && $user_id) {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        // Extrahiere Dateinamen aus dem Pfad
        $filename = basename($requested_path);
        
        // Prüfe ob das Dokument in der users-Tabelle zu diesem User gehört
        $stmt = $conn->prepare("SELECT id FROM users WHERE id = ? AND verification_document LIKE ?");
        $stmt->execute([$user_id, '%' . $filename]);
        $user_doc = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user_doc) {
            http_response_code(403);
            die('Zugriff verweigert: Sie haben keine Berechtigung für dieses Dokument');
        }
    } catch (Exception $e) {
        http_response_code(500);
        die('Fehler bei der Berechtigungsprüfung');
    }
}

// Bestimme MIME-Type
$mime_type = mime_content_type($requested_path);
if (!$mime_type) {
    // Fallback basierend auf Dateierweiterung
    $ext = strtolower(pathinfo($requested_path, PATHINFO_EXTENSION));
    $mime_types = [
        'pdf' => 'application/pdf',
        'jpg' => 'image/jpeg',
        'jpeg' => 'image/jpeg',
        'png' => 'image/png',
        'gif' => 'image/gif'
    ];
    $mime_type = $mime_types[$ext] ?? 'application/octet-stream';
}

$allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'application/pdf', 'text/plain'];

if (!in_array($mime_type, $allowed_types)) {
    http_response_code(403);
    die('Dateityp nicht erlaubt: ' . $mime_type);
}

// Setze entsprechende Header
header('Content-Type: ' . $mime_type);
header('Content-Length: ' . filesize($requested_path));
header('Cache-Control: private, max-age=3600'); // 1 Stunde Cache
header('Content-Disposition: inline; filename="' . basename($requested_path) . '"');

// Output der Datei
readfile($requested_path);
exit;

