<?php
// CLI only
if (php_sapi_name() !== 'cli') { echo "CLI only\n"; exit(1); }
require_once __DIR__ . '/../config/database.php';

function execMulti(PDO $conn, string $sql): void {
  foreach (array_filter(array_map('trim', explode(';', $sql))) as $stmt) {
    if ($stmt === '' || str_starts_with($stmt, '--')) continue;
    $conn->exec($stmt);
  }
}

try {
  $db = new Database();
  $conn = $db->getConnection();

  // 1) Tabellen sicherstellen
  $ddl = file_get_contents(__DIR__ . '/../database/create_api_security_tables.sql');
  execMulti($conn, $ddl);

  // 2) Prüfung auf Spalten (zur Sicherheit, falls alte Tabelle existierte)
  $columns = ['name','key_hash','secret_hash','scopes','is_active','rate_limit_per_min','owner_type','owner_id','created_at','last_used_at'];
  foreach ($columns as $col) {
    $stmt = $conn->prepare("SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME='api_keys' AND COLUMN_NAME=?");
    $stmt->execute([$col]);
    if (!$stmt->fetch()) {
      // Minimaler Fallback (nur wenn wirklich fehlt)
      if ($col === 'name') $conn->exec("ALTER TABLE api_keys ADD COLUMN name VARCHAR(100) NOT NULL");
      if ($col === 'key_hash') $conn->exec("ALTER TABLE api_keys ADD COLUMN key_hash CHAR(64) NOT NULL UNIQUE");
      if ($col === 'secret_hash') $conn->exec("ALTER TABLE api_keys ADD COLUMN secret_hash CHAR(64) NOT NULL");
      if ($col === 'scopes') $conn->exec("ALTER TABLE api_keys ADD COLUMN scopes TEXT NOT NULL");
      if ($col === 'is_active') $conn->exec("ALTER TABLE api_keys ADD COLUMN is_active TINYINT(1) NOT NULL DEFAULT 1");
      if ($col === 'rate_limit_per_min') $conn->exec("ALTER TABLE api_keys ADD COLUMN rate_limit_per_min INT NOT NULL DEFAULT 60");
      if ($col === 'owner_type') $conn->exec("ALTER TABLE api_keys ADD COLUMN owner_type VARCHAR(20) DEFAULT 'admin'");
      if ($col === 'owner_id') $conn->exec("ALTER TABLE api_keys ADD COLUMN owner_id INT NULL");
      if ($col === 'created_at') $conn->exec("ALTER TABLE api_keys ADD COLUMN created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP");
      if ($col === 'last_used_at') $conn->exec("ALTER TABLE api_keys ADD COLUMN last_used_at TIMESTAMP NULL");
    }
  }

  // 3) Key/Secret generieren
  $apiKey = bin2hex(random_bytes(16));
  $apiSecret = bin2hex(random_bytes(32));
  $keyHash = hash('sha256', $apiKey);
  $secretHash = hash('sha256', $apiSecret);

  // 4) Standard-Scopes und Rate
  $name = 'default-mobile';
  $scopes = json_encode(["slots:read","appointments:write"], JSON_UNESCAPED_UNICODE);
  $rate = 120;

  // 5) Einfügen (falls gleicher Hash schon vorhanden, abbrechen)
  $stmt = $conn->prepare("SELECT id FROM api_keys WHERE key_hash = ?");
  $stmt->execute([$keyHash]);
  if ($stmt->fetch()) {
    echo "Key collision (extrem unwahrscheinlich). Bitte erneut ausführen.\n";
    exit(1);
  }

  $stmt = $conn->prepare("INSERT INTO api_keys (name, key_hash, secret_hash, scopes, is_active, rate_limit_per_min) VALUES (?, ?, ?, ?, 1, ?)");
  $stmt->execute([$name, $keyHash, $secretHash, $scopes, $rate]);

  echo "API_KEY=" . $apiKey . "\n";
  echo "API_SECRET=" . $apiSecret . "\n";
  echo "Name=" . $name . " Scopes=" . $scopes . " RateLimit/min=" . $rate . "\n";
  echo "Hinweis: Bewahre API_SECRET sicher auf (ENV/KMS).\n";

} catch (Throwable $e) {
  echo "ERROR: " . $e->getMessage() . "\n";
  exit(1);
}
