<?php
/**
 * Test-Script für die behobene HTML-Generierung
 */

echo "<h1>Test der behobenen HTML-Generierung</h1>\n";

// Test-Parameter
$testStorageNumber = '9.329.579'; // Storage Number die HTML-Daten fehlen
$testAppointmentId = '101'; // Appointment ID die HTML-Daten fehlen
$testLgsId = '9329579'; // LGS_ID für die HTML-Generierung

echo "<h2>Test-Parameter:</h2>\n";
echo "<ul>\n";
echo "<li>Storage Number: $testStorageNumber</li>\n";
echo "<li>Appointment ID: $testAppointmentId</li>\n";
echo "<li>LGS_ID: $testLgsId</li>\n";
echo "</ul>\n";

// Test 1: Aktueller Cache-Status prüfen
echo "<h2>Test 1: Aktueller Cache-Status</h2>\n";
try {
    require_once __DIR__ . '/config/database.php';
    $db = new Database();
    $conn = $db->getConnection();
    
    // Prüfe Data Cache
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM resy_data_cache WHERE einlagerungsnummer = ?");
    $stmt->execute([$testStorageNumber]);
    $dataCacheCount = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
    
    // Prüfe HTML Cache
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM resy_html_cache WHERE einlagerungsnummer = ?");
    $stmt->execute([$testStorageNumber]);
    $htmlCacheCount = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
    
    echo "<p><strong>Data Cache Einträge:</strong> $dataCacheCount</p>\n";
    echo "<p><strong>HTML Cache Einträge:</strong> $htmlCacheCount</p>\n";
    
    if ($dataCacheCount > 0 && $htmlCacheCount == 0) {
        echo "<p style='color: orange;'><strong>⚠️ Perfekter Test-Fall:</strong> Data Cache vorhanden, aber HTML Cache fehlt</p>\n";
    } elseif ($dataCacheCount > 0 && $htmlCacheCount > 0) {
        echo "<p style='color: green;'><strong>✅ Beide Caches vorhanden:</strong> Test wird Cache-Hit simulieren</p>\n";
    } else {
        echo "<p style='color: red;'><strong>❌ Keine Cache-Daten:</strong> Test wird Cache-Miss simulieren</p>\n";
    }
    
} catch (Exception $e) {
    echo "<p style='color: red;'>Datenbankfehler: " . htmlspecialchars($e->getMessage()) . "</p>\n";
}

// Test 2: API-Aufruf mit der neuen Logik
echo "<h2>Test 2: API-Aufruf mit neuer HTML-Generierung</h2>\n";
$apiUrl = "http://localhost:8080/api/resy_data_extraction_breakthrough.php";

$testData = [
    'action' => 'breakthrough',
    'appointment_id' => $testAppointmentId,
    'lgsId' => $testLgsId
];

echo "<h3>API-Anfrage:</h3>\n";
echo "<pre>" . json_encode($testData, JSON_PRETTY_PRINT) . "</pre>\n";

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $apiUrl);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($testData));
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    'Content-Type: application/json'
]);
curl_setopt($ch, CURLOPT_TIMEOUT, 60); // 60 Sekunden Timeout für RESY-Aufrufe

echo "<p><strong>Starte API-Aufruf...</strong></p>\n";
$startTime = microtime(true);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$error = curl_error($ch);
$totalTime = microtime(true) - $startTime;

curl_close($ch);

echo "<p><strong>API-Antwort (nach " . round($totalTime, 2) . " Sekunden):</strong></p>\n";
echo "<p><strong>HTTP Code:</strong> $httpCode</p>\n";
echo "<p><strong>Error:</strong> " . ($error ?: 'Keine') . "</p>\n";

if ($httpCode === 200) {
    $result = json_decode($response, true);
    if ($result && $result['success']) {
        echo "<p style='color: green;'>✅ API-Aufruf erfolgreich</p>\n";
        echo "<p><strong>Message:</strong> " . htmlspecialchars($result['message']) . "</p>\n";
        echo "<p><strong>Cached:</strong> " . ($result['cached'] ? 'Ja' : 'Nein') . "</p>\n";
        echo "<p><strong>HTML Cached:</strong> " . ($result['html_cached'] ? 'Ja' : 'Nein') . "</p>\n";
        
        if (!empty($result['detailHtml'])) {
            echo "<p style='color: green;'><strong>✅ HTML-Daten verfügbar:</strong> " . strlen($result['detailHtml']) . " Zeichen</p>\n";
            
            // Zeige HTML-Vorschau
            $htmlPreview = substr($result['detailHtml'], 0, 500);
            echo "<h3>HTML-Vorschau (erste 500 Zeichen):</h3>\n";
            echo "<pre style='background: #f5f5f5; padding: 10px; border: 1px solid #ddd; max-height: 200px; overflow-y: auto;'>\n";
            echo htmlspecialchars($htmlPreview) . "\n";
            echo "</pre>\n";
        } else {
            echo "<p style='color: red;'><strong>❌ HTML-Daten fehlen immer noch</strong></p>\n";
        }
        
        // Zeige Fahrzeugdaten
        if (!empty($result['vehicle_data'])) {
            echo "<h3>Fahrzeugdaten:</h3>\n";
            echo "<ul>\n";
            foreach ($result['vehicle_data'] as $key => $value) {
                echo "<li><strong>$key:</strong> " . htmlspecialchars($value) . "</li>\n";
            }
            echo "</ul>\n";
        }
        
        // Zeige Reifendaten
        if (!empty($result['tire_data'])) {
            echo "<h3>Reifendaten:</h3>\n";
            echo "<ul>\n";
            foreach ($result['tire_data'] as $key => $value) {
                if (is_array($value)) {
                    echo "<li><strong>$key:</strong> " . json_encode($value) . "</li>\n";
                } else {
                    echo "<li><strong>$key:</strong> " . htmlspecialchars($value) . "</li>\n";
                }
            }
            echo "</ul>\n";
        }
        
    } else {
        echo "<p style='color: red;'>❌ API-Fehler: " . htmlspecialchars($result['error'] ?? 'Unbekannter Fehler') . "</p>\n";
    }
} else {
    echo "<p style='color: red;'>❌ HTTP-Fehler: $httpCode</p>\n";
    echo "<pre>" . htmlspecialchars($response) . "</pre>\n";
}

// Test 3: Cache-Status nach dem Test prüfen
echo "<h2>Test 3: Cache-Status nach dem Test</h2>\n";
try {
    // Prüfe Data Cache
    $stmt = $conn->prepare("SELECT COUNT(*) as count, MAX(updated_at) as last_update FROM resy_data_cache WHERE einlagerungsnummer = ?");
    $stmt->execute([$testStorageNumber]);
    $dataCacheResult = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Prüfe HTML Cache
    $stmt = $conn->prepare("SELECT COUNT(*) as count, MAX(updated_at) as last_update FROM resy_html_cache WHERE einlagerungsnummer = ?");
    $stmt->execute([$testStorageNumber]);
    $htmlCacheResult = $stmt->fetch(PDO::FETCH_ASSOC);
    
    echo "<p><strong>Data Cache Einträge:</strong> " . $dataCacheResult['count'] . " (letzte Aktualisierung: " . $dataCacheResult['last_update'] . ")</p>\n";
    echo "<p><strong>HTML Cache Einträge:</strong> " . $htmlCacheResult['count'] . " (letzte Aktualisierung: " . $htmlCacheResult['last_update'] . ")</p>\n";
    
    if ($htmlCacheResult['count'] > 0) {
        echo "<p style='color: green;'><strong>✅ HTML-Cache erfolgreich erstellt!</strong></p>\n";
    } else {
        echo "<p style='color: red;'><strong>❌ HTML-Cache wurde nicht erstellt</strong></p>\n";
    }
    
} catch (Exception $e) {
    echo "<p style='color: red;'>Datenbankfehler: " . htmlspecialchars($e->getMessage()) . "</p>\n";
}

// Test 4: Log-Dateien prüfen
echo "<h2>Test 4: Log-Dateien prüfen</h2>\n";
$logFile = __DIR__ . '/logs/resy_extraction.log';
if (file_exists($logFile)) {
    $logContent = file_get_contents($logFile);
    $lines = explode("\n", $logContent);
    $recentLines = array_slice($lines, -30); // Letzte 30 Zeilen
    
    echo "<h3>Letzte 30 Log-Einträge:</h3>\n";
    echo "<pre style='background: #f5f5f5; padding: 10px; border: 1px solid #ddd; max-height: 400px; overflow-y: auto;'>\n";
    foreach ($recentLines as $line) {
        if (!empty(trim($line))) {
            echo htmlspecialchars($line) . "\n";
        }
    }
    echo "</pre>\n";
} else {
    echo "<p style='color: red;'>Log-Datei nicht gefunden: $logFile</p>\n";
}

echo "<h2>Zusammenfassung</h2>\n";
echo "<p>Die Verbesserungen umfassen:</p>\n";
echo "<ul>\n";
echo "<li>✅ Cache-Hit mit HTML-Generierung: Bei Cache-Hit wird trotzdem versucht, HTML von RESY zu generieren</li>\n";
echo "<li>✅ Automatische HTML-Cache-Erstellung: Generierte HTML-Daten werden automatisch gecacht</li>\n";
echo "<li>✅ Bessere Fehlerbehandlung: Detaillierte Logging-Informationen für alle Schritte</li>\n";
echo "<li>✅ Fallback-Mechanismus: Mehrere Versuche, HTML-Daten zu erhalten</li>\n";
echo "</ul>\n";

echo "<p><strong>Nächste Schritte:</strong></p>\n";
echo "<ol>\n";
echo "<li>Führen Sie 'Reifen prüfen' in der appointments.php durch</li>\n";
echo "<li>Prüfen Sie, ob HTML-Daten jetzt verfügbar sind</li>\n";
echo "<li>Testen Sie die Bildladung nach den HTML-Verbesserungen</li>\n";
echo "</ol>\n";

echo "<p style='color: green;'><strong>✅ Das Problem 'HTML-Inhalt fehlt' sollte jetzt behoben sein!</strong></p>\n";
?>
