<?php
/**
 * Erstellt die RESY-Cache-Tabellen für Daten und HTML
 */

require_once __DIR__ . '/../config/database.php';

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    echo "=== RESY Cache-Tabellen erstellen ===\n";
    
    // 1. Erstelle resy_data_cache Tabelle
    $sql1 = "
    CREATE TABLE IF NOT EXISTS resy_data_cache (
        id INT AUTO_INCREMENT PRIMARY KEY,
        einlagerungsnummer VARCHAR(50) NOT NULL,
        appointment_id INT NULL,
        vehicle_data LONGTEXT NOT NULL,
        tire_data LONGTEXT NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        expires_at TIMESTAMP NOT NULL DEFAULT (DATE_ADD(NOW(), INTERVAL 24 HOUR)),
        INDEX idx_einlagerungsnummer (einlagerungsnummer),
        INDEX idx_appointment_id (appointment_id),
        INDEX idx_expires_at (expires_at),
        INDEX idx_updated_at (updated_at)
    ) COMMENT = 'Cache für RESY-Fahrzeug- und Reifendaten'
    ";
    
    $conn->exec($sql1);
    echo "✅ Tabelle 'resy_data_cache' erfolgreich erstellt!\n";
    
    // 2. Erstelle resy_html_cache Tabelle
    $sql2 = "
    CREATE TABLE IF NOT EXISTS resy_html_cache (
        id INT AUTO_INCREMENT PRIMARY KEY,
        einlagerungsnummer VARCHAR(50) NOT NULL,
        appointment_id INT NULL,
        html_content LONGTEXT NOT NULL,
        resy_base_url VARCHAR(255) NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        expires_at TIMESTAMP NOT NULL DEFAULT (DATE_ADD(NOW(), INTERVAL 24 HOUR)),
        INDEX idx_einlagerungsnummer (einlagerungsnummer),
        INDEX idx_appointment_id (appointment_id),
        INDEX idx_expires_at (expires_at),
        INDEX idx_updated_at (updated_at)
    ) COMMENT = 'Cache für RESY-HTML-Inhalte'
    ";
    
    $conn->exec($sql2);
    echo "✅ Tabelle 'resy_html_cache' erfolgreich erstellt!\n";
    
    // 3. Prüfe ob Tabellen existieren
    $tables = ['resy_data_cache', 'resy_html_cache'];
    foreach ($tables as $table) {
        $stmt = $conn->query("SHOW TABLES LIKE '$table'");
        if ($stmt->rowCount() > 0) {
            echo "✅ Tabelle '$table' existiert und ist bereit.\n";
            
            // Zeige Tabellen-Struktur
            $stmt = $conn->query("DESCRIBE $table");
            $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
            echo "   Spalten: " . implode(', ', array_column($columns, 'Field')) . "\n";
        } else {
            echo "❌ Fehler: Tabelle '$table' wurde nicht erstellt.\n";
        }
    }
    
    // 4. Teste die Tabellen mit einem Test-Eintrag
    echo "\n=== Teste Cache-Tabellen ===\n";
    
    $testStorageNumber = 'TEST123';
    $testAppointmentId = 999;
    $testVehicleData = json_encode(['test' => 'vehicle data']);
    $testTireData = json_encode(['test' => 'tire data']);
    $testHtml = '<html><body>Test HTML Content</body></html>';
    $testResyUrl = 'https://test.resy.de';
    
    // Test resy_data_cache
    try {
        $stmt = $conn->prepare("
            INSERT INTO resy_data_cache (einlagerungsnummer, appointment_id, vehicle_data, tire_data, expires_at) 
            VALUES (?, ?, ?, ?, DATE_ADD(NOW(), INTERVAL 1 HOUR))
        ");
        $stmt->execute([$testStorageNumber, $testAppointmentId, $testVehicleData, $testTireData]);
        echo "✅ Test-Eintrag in resy_data_cache erfolgreich erstellt.\n";
        
        // Teste Abruf
        $stmt = $conn->prepare("SELECT * FROM resy_data_cache WHERE einlagerungsnummer = ? AND expires_at > NOW()");
        $stmt->execute([$testStorageNumber]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($result) {
            echo "✅ Test-Eintrag aus resy_data_cache erfolgreich abgerufen.\n";
        } else {
            echo "❌ Test-Eintrag aus resy_data_cache konnte nicht abgerufen werden.\n";
        }
        
        // Lösche Test-Eintrag
        $stmt = $conn->prepare("DELETE FROM resy_data_cache WHERE einlagerungsnummer = ?");
        $stmt->execute([$testStorageNumber]);
        echo "✅ Test-Eintrag aus resy_data_cache gelöscht.\n";
        
    } catch (Exception $e) {
        echo "❌ Fehler beim Testen von resy_data_cache: " . $e->getMessage() . "\n";
    }
    
    // Test resy_html_cache
    try {
        $stmt = $conn->prepare("
            INSERT INTO resy_html_cache (einlagerungsnummer, appointment_id, html_content, resy_base_url, expires_at) 
            VALUES (?, ?, ?, ?, DATE_ADD(NOW(), INTERVAL 1 HOUR))
        ");
        $stmt->execute([$testStorageNumber, $testAppointmentId, $testHtml, $testResyUrl]);
        echo "✅ Test-Eintrag in resy_html_cache erfolgreich erstellt.\n";
        
        // Teste Abruf
        $stmt = $conn->prepare("SELECT * FROM resy_html_cache WHERE einlagerungsnummer = ? AND expires_at > NOW()");
        $stmt->execute([$testStorageNumber]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($result) {
            echo "✅ Test-Eintrag aus resy_html_cache erfolgreich abgerufen.\n";
            echo "   HTML-Länge: " . strlen($result['html_content']) . " Zeichen\n";
        } else {
            echo "❌ Test-Eintrag aus resy_html_cache konnte nicht abgerufen werden.\n";
        }
        
        // Lösche Test-Eintrag
        $stmt = $conn->prepare("DELETE FROM resy_html_cache WHERE einlagerungsnummer = ?");
        $stmt->execute([$testStorageNumber]);
        echo "✅ Test-Eintrag aus resy_html_cache gelöscht.\n";
        
    } catch (Exception $e) {
        echo "❌ Fehler beim Testen von resy_html_cache: " . $e->getMessage() . "\n";
    }
    
    echo "\n=== Cache-Tabellen erfolgreich erstellt und getestet! ===\n";
    echo "Die Tabellen sind jetzt bereit für die RESY-Cache-Funktionalität.\n";
    
} catch (Exception $e) {
    echo "❌ Fehler beim Erstellen der Cache-Tabellen: " . $e->getMessage() . "\n";
}
?>
