<?php
/**
 * Debug-Skript: Prüft RESY API für alle Autohäuser
 * Zeigt Credentials und API-Status
 */

require_once 'config/database.php';

$db = new Database();
$conn = $db->getConnection();

// Lade alle Autohäuser
$stmt = $conn->prepare("
    SELECT 
        id, 
        name, 
        resy_username, 
        resy_password, 
        resy_dealer_id,
        CASE 
            WHEN EXISTS (
                SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS 
                WHERE TABLE_SCHEMA = DATABASE() 
                AND TABLE_NAME = 'autohaus' 
                AND COLUMN_NAME = 'resy_customer_no'
            ) THEN resy_customer_no
            ELSE NULL
        END as resy_customer_no
    FROM autohaus 
    WHERE is_active = 1 
    ORDER BY name
");

$stmt->execute();
$autohauser = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Prüfe ob resy_customer_no Spalte existiert
$stmt = $conn->query("SHOW COLUMNS FROM autohaus LIKE 'resy_customer_no'");
$hasCustomerNoColumn = $stmt->rowCount() > 0;

?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>RESY API Debug - Autohäuser</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            max-width: 1200px;
            margin: 20px auto;
            padding: 20px;
            background: #f5f5f5;
        }
        .header {
            background: #007bff;
            color: white;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        .autohaus-card {
            background: white;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .autohaus-card h3 {
            margin-top: 0;
            color: #333;
            border-bottom: 2px solid #007bff;
            padding-bottom: 10px;
        }
        .info-row {
            display: flex;
            padding: 8px 0;
            border-bottom: 1px solid #eee;
        }
        .info-label {
            font-weight: bold;
            width: 200px;
            color: #666;
        }
        .info-value {
            flex: 1;
        }
        .status-ok { color: #28a745; font-weight: bold; }
        .status-error { color: #dc3545; font-weight: bold; }
        .status-warning { color: #ffc107; font-weight: bold; }
        .test-btn {
            background: #007bff;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 5px;
            cursor: pointer;
            margin-top: 10px;
        }
        .test-btn:hover {
            background: #0056b3;
        }
        .result-box {
            margin-top: 15px;
            padding: 15px;
            border-radius: 5px;
            background: #f8f9fa;
            border-left: 4px solid #007bff;
            display: none;
        }
        .result-box.show {
            display: block;
        }
        .result-box.error {
            border-left-color: #dc3545;
            background: #fff5f5;
        }
        .result-box.success {
            border-left-color: #28a745;
            background: #f0fff4;
        }
        pre {
            background: #2d2d2d;
            color: #f8f8f2;
            padding: 15px;
            border-radius: 5px;
            overflow-x: auto;
            font-size: 12px;
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>🔍 RESY API Debug - Autohäuser Übersicht</h1>
        <p>Prüft Credentials und API-Verbindung für alle Autohäuser</p>
        <p><small>Spalte 'resy_customer_no' existiert: <?= $hasCustomerNoColumn ? '✅ JA' : '❌ NEIN' ?></small></p>
    </div>

    <?php foreach ($autohauser as $autohaus): ?>
        <div class="autohaus-card">
            <h3><?= htmlspecialchars($autohaus['name']) ?> (ID: <?= $autohaus['id'] ?>)</h3>
            
            <div class="info-row">
                <div class="info-label">RESY Username:</div>
                <div class="info-value">
                    <?= $autohaus['resy_username'] ? '<span class="status-ok">✅ ' . htmlspecialchars($autohaus['resy_username']) . '</span>' : '<span class="status-error">❌ Nicht gesetzt</span>' ?>
                </div>
            </div>
            
            <div class="info-row">
                <div class="info-label">RESY Password:</div>
                <div class="info-value">
                    <?= $autohaus['resy_password'] ? '<span class="status-ok">✅ Gesetzt (' . strlen($autohaus['resy_password']) . ' Zeichen)</span>' : '<span class="status-error">❌ Nicht gesetzt</span>' ?>
                </div>
            </div>
            
            <div class="info-row">
                <div class="info-label">RESY Dealer ID:</div>
                <div class="info-value">
                    <?= $autohaus['resy_dealer_id'] ? '<span class="status-ok">✅ ' . htmlspecialchars($autohaus['resy_dealer_id']) . '</span>' : '<span class="status-warning">⚠️ Nicht gesetzt</span>' ?>
                </div>
            </div>
            
            <?php if ($hasCustomerNoColumn): ?>
            <div class="info-row">
                <div class="info-label">RESY Customer No:</div>
                <div class="info-value">
                    <?php
                    $customerNo = $autohaus['resy_customer_no'];
                    if ($customerNo) {
                        echo '<span class="status-ok">✅ ' . htmlspecialchars($customerNo) . '</span>';
                    } else {
                        // Fallback: Versuche aus dealer_id zu extrahieren
                        if ($autohaus['resy_dealer_id'] && is_numeric($autohaus['resy_dealer_id'])) {
                            echo '<span class="status-warning">⚠️ Nicht gesetzt (Fallback: ' . htmlspecialchars($autohaus['resy_dealer_id']) . ')</span>';
                        } else {
                            echo '<span class="status-error">❌ Nicht gesetzt</span>';
                        }
                    }
                    ?>
                </div>
            </div>
            <?php endif; ?>
            
            <button class="test-btn" onclick="testAPI(<?= $autohaus['id'] ?>, '<?= htmlspecialchars($autohaus['name'], ENT_QUOTES) ?>')">
                🧪 API Test durchführen
            </button>
            
            <div class="result-box" id="result-<?= $autohaus['id'] ?>">
                <strong>Test-Ergebnis:</strong>
                <div id="result-content-<?= $autohaus['id'] ?>"></div>
            </div>
        </div>
    <?php endforeach; ?>

    <script>
        function testAPI(autohausId, autohausName) {
            const resultBox = document.getElementById('result-' + autohausId);
            const resultContent = document.getElementById('result-content-' + autohausId);
            
            resultBox.className = 'result-box show';
            resultContent.innerHTML = '<p>⏳ Teste API-Verbindung...</p>';
            
            fetch(`api/get_resy_wheelsets.php?autohaus_id=${autohausId}`)
                .then(response => {
                    console.log(`HTTP Status für ${autohausName}:`, response.status);
                    
                    // Prüfe Content-Type
                    const contentType = response.headers.get('content-type');
                    if (!contentType || !contentType.includes('application/json')) {
                        return response.text().then(text => {
                            throw new Error(`Unerwartete Antwort: HTML statt JSON. Status: ${response.status}. Beginn: ${text.substring(0, 200)}`);
                        });
                    }
                    
                    return response.json();
                })
                .then(data => {
                    if (data.success) {
                        resultBox.className = 'result-box show success';
                        resultContent.innerHTML = `
                            <p><strong>✅ Erfolgreich!</strong></p>
                            <p>Geladene Reifensätze: <strong>${data.count || 0}</strong></p>
                            <p>Kundennummer: <strong>${data.customer_no || 'N/A'}</strong></p>
                            <p>Autohaus: <strong>${data.autohaus || autohausName}</strong></p>
                            <details>
                                <summary>Vollständige Antwort anzeigen</summary>
                                <pre>${JSON.stringify(data, null, 2)}</pre>
                            </details>
                        `;
                    } else {
                        resultBox.className = 'result-box show error';
                        resultContent.innerHTML = `
                            <p><strong>❌ Fehler!</strong></p>
                            <p><strong>Fehlermeldung:</strong> ${data.error || 'Unbekannter Fehler'}</p>
                            ${data.debug_info ? `
                                <p><strong>Datei:</strong> ${data.debug_info.file || 'N/A'}</p>
                                <p><strong>Zeile:</strong> ${data.debug_info.line || 'N/A'}</p>
                            ` : ''}
                            <details>
                                <summary>Vollständige Fehler-Details anzeigen</summary>
                                <pre>${JSON.stringify(data, null, 2)}</pre>
                            </details>
                        `;
                    }
                })
                .catch(error => {
                    resultBox.className = 'result-box show error';
                    resultContent.innerHTML = `
                        <p><strong>❌ Netzwerk-Fehler!</strong></p>
                        <p>${error.message}</p>
                    `;
                });
        }
    </script>
</body>
</html>

