-- 🚀 INTELLIGENT DASHBOARD SETUP
-- Erstellt alle notwendigen Tabellen für das intelligente Dashboard-System

-- User Dashboard Configuration Table
CREATE TABLE IF NOT EXISTS `user_dashboard_config` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `user_id` int(11) NOT NULL,
    `config_data` JSON NOT NULL,
    `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    UNIQUE KEY `user_id` (`user_id`),
    KEY `idx_user_id` (`user_id`),
    KEY `idx_updated_at` (`updated_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Quick Actions Table (falls nicht vorhanden)
CREATE TABLE IF NOT EXISTS `quick_actions` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `title` varchar(255) NOT NULL,
    `url` varchar(500) NOT NULL,
    `icon` varchar(100) NOT NULL,
    `color` varchar(7) NOT NULL DEFAULT '#007AFF',
    `description` text,
    `is_active` tinyint(1) NOT NULL DEFAULT 1,
    `position_order` int(11) NOT NULL DEFAULT 0,
    `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_is_active` (`is_active`),
    KEY `idx_position_order` (`position_order`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert default quick actions
INSERT IGNORE INTO `quick_actions` (`title`, `url`, `icon`, `color`, `description`, `position_order`) VALUES
('Neues Event', 'events.php?action=create', 'fas fa-plus', '#007AFF', 'Ein neues Event erstellen', 1),
('Termine', 'appointments.php', 'fas fa-calendar-check', '#34C759', 'Termine verwalten', 2),
('Lager', 'storage.php', 'fas fa-warehouse', '#FF9500', 'Lager verwalten', 3),
('Benutzer', 'users.php', 'fas fa-users', '#AF52DE', 'Benutzer verwalten', 4),
('Einstellungen', 'settings.php', 'fas fa-cog', '#5856D6', 'System-Einstellungen', 5),
('Frontend', '../index.php', 'fas fa-external-link-alt', '#30B0C7', 'Frontend anzeigen', 6);

-- User Activity Log Table
CREATE TABLE IF NOT EXISTS `user_activity_log` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `user_id` int(11) NOT NULL,
    `action_type` varchar(100) NOT NULL,
    `action_description` text NOT NULL,
    `target_id` int(11) DEFAULT NULL,
    `target_type` varchar(100) DEFAULT NULL,
    `ip_address` varchar(45) DEFAULT NULL,
    `user_agent` text,
    `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_user_id` (`user_id`),
    KEY `idx_action_type` (`action_type`),
    KEY `idx_created_at` (`created_at`),
    KEY `idx_target` (`target_type`, `target_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Dashboard Widgets Configuration
CREATE TABLE IF NOT EXISTS `dashboard_widgets` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `widget_key` varchar(100) NOT NULL,
    `title` varchar(255) NOT NULL,
    `description` text,
    `icon` varchar(100) NOT NULL,
    `color` varchar(7) NOT NULL DEFAULT '#007AFF',
    `category` enum('essential','operational','analytics') NOT NULL DEFAULT 'essential',
    `importance` enum('high','medium','low') NOT NULL DEFAULT 'medium',
    `is_active` tinyint(1) NOT NULL DEFAULT 1,
    `position_order` int(11) NOT NULL DEFAULT 0,
    `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    UNIQUE KEY `widget_key` (`widget_key`),
    KEY `idx_category` (`category`),
    KEY `idx_importance` (`importance`),
    KEY `idx_is_active` (`is_active`),
    KEY `idx_position_order` (`position_order`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert default widgets
INSERT IGNORE INTO `dashboard_widgets` (`widget_key`, `title`, `description`, `icon`, `color`, `category`, `importance`, `position_order`) VALUES
('today_appointments', 'Heutige Termine', 'Zeigt alle Termine für heute an', 'fas fa-calendar-day', '#007AFF', 'essential', 'high', 1),
('upcoming_events', 'Kommende Events', 'Zeigt die nächsten Events an', 'fas fa-calendar-alt', '#34C759', 'essential', 'high', 2),
('quick_actions', 'Schnellaktionen', 'Ihre personalisierten Schnellzugriffe', 'fas fa-rocket', '#AF52DE', 'essential', 'high', 3),
('recent_storage', 'Letzte Lageraktivitäten', 'Zeigt die neuesten Einlagerungen an', 'fas fa-warehouse', '#FF9500', 'operational', 'medium', 4),
('performance_metrics', 'Performance-Metriken', 'Zeigt Statistiken und Kennzahlen an', 'fas fa-chart-line', '#5856D6', 'analytics', 'medium', 5),
('user_activity', 'Benutzeraktivität', 'Zeigt die letzten Benutzeraktionen an', 'fas fa-users-cog', '#FFCC00', 'analytics', 'low', 6),
('system_status', 'System Status', 'Zeigt System-Informationen an', 'fas fa-server', '#FF3B30', 'analytics', 'low', 7);

-- Dashboard Layouts Configuration
CREATE TABLE IF NOT EXISTS `dashboard_layouts` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `layout_key` varchar(100) NOT NULL,
    `name` varchar(255) NOT NULL,
    `description` text,
    `best_for` varchar(255) DEFAULT NULL,
    `screen_size` varchar(255) DEFAULT NULL,
    `grid_config` JSON DEFAULT NULL,
    `is_active` tinyint(1) NOT NULL DEFAULT 1,
    `position_order` int(11) NOT NULL DEFAULT 0,
    `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    UNIQUE KEY `layout_key` (`layout_key`),
    KEY `idx_is_active` (`is_active`),
    KEY `idx_position_order` (`position_order`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert default layouts
INSERT IGNORE INTO `dashboard_layouts` (`layout_key`, `name`, `description`, `best_for`, `screen_size`, `grid_config`, `position_order`) VALUES
('default', 'Standard Layout', 'Zwei-Spalten Layout mit Hauptinhalt und Sidebar', 'Tägliche Arbeit, ausgewogene Übersicht', 'Desktop & Tablet', '{"columns": "2fr 1fr", "areas": ["header header", "main sidebar", "widgets sidebar"]}', 1),
('compact', 'Kompaktes Layout', 'Ein-Spalten Layout für fokussierte Arbeit', 'Mobile Geräte, fokussierte Aufgaben', 'Mobile & Tablet', '{"columns": "1fr", "areas": ["header", "main", "widgets"]}', 2),
('wide', 'Breites Layout', 'Drei-Spalten Layout für maximale Übersicht', 'Administratoren, große Bildschirme', 'Desktop & Ultrawide', '{"columns": "1fr 1fr 1fr", "areas": ["header header header", "main main sidebar", "widgets widgets sidebar"]}', 3);

-- User Preferences Table
CREATE TABLE IF NOT EXISTS `user_preferences` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `user_id` int(11) NOT NULL,
    `preference_key` varchar(100) NOT NULL,
    `preference_value` text,
    `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    UNIQUE KEY `user_preference` (`user_id`, `preference_key`),
    KEY `idx_user_id` (`user_id`),
    KEY `idx_preference_key` (`preference_key`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Performance Metrics Cache Table
CREATE TABLE IF NOT EXISTS `performance_metrics_cache` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `metric_key` varchar(100) NOT NULL,
    `metric_value` decimal(10,2) DEFAULT NULL,
    `metric_data` JSON DEFAULT NULL,
    `calculated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `expires_at` timestamp NOT NULL,
    PRIMARY KEY (`id`),
    UNIQUE KEY `metric_key` (`metric_key`),
    KEY `idx_expires_at` (`expires_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Add theme column to admin_users if not exists
ALTER TABLE `admin_users` 
ADD COLUMN IF NOT EXISTS `theme` enum('light','dark') NOT NULL DEFAULT 'light' AFTER `role`;

-- Add full_name column to admin_users if not exists
ALTER TABLE `admin_users` 
ADD COLUMN IF NOT EXISTS `full_name` varchar(255) DEFAULT NULL AFTER `username`;

-- Create indexes for better performance
CREATE INDEX IF NOT EXISTS `idx_user_dashboard_config_user_id` ON `user_dashboard_config` (`user_id`);
CREATE INDEX IF NOT EXISTS `idx_user_dashboard_config_updated_at` ON `user_dashboard_config` (`updated_at`);
CREATE INDEX IF NOT EXISTS `idx_quick_actions_active_order` ON `quick_actions` (`is_active`, `position_order`);
CREATE INDEX IF NOT EXISTS `idx_user_activity_log_user_created` ON `user_activity_log` (`user_id`, `created_at`);
CREATE INDEX IF NOT EXISTS `idx_dashboard_widgets_category_importance` ON `dashboard_widgets` (`category`, `importance`);
CREATE INDEX IF NOT EXISTS `idx_performance_metrics_cache_expires` ON `performance_metrics_cache` (`expires_at`);

-- Insert default dashboard configuration for existing users
INSERT IGNORE INTO `user_dashboard_config` (`user_id`, `config_data`)
SELECT 
    au.id,
    JSON_OBJECT(
        'layout', 'default',
        'widgets', JSON_ARRAY('today_appointments', 'upcoming_events', 'quick_actions'),
        'quick_actions', JSON_ARRAY('events', 'appointments', 'storage', 'users', 'settings', 'frontend'),
        'theme', COALESCE(au.theme, 'light'),
        'show_welcome', true,
        'auto_refresh', false,
        'refresh_interval', 300,
        'widget_positions', JSON_OBJECT()
    )
FROM `admin_users` au
WHERE au.id NOT IN (SELECT user_id FROM user_dashboard_config);

-- Create view for dashboard statistics
CREATE OR REPLACE VIEW `dashboard_stats_view` AS
SELECT 
    COUNT(DISTINCT CASE WHEN DATE(a.slot_date) = CURDATE() THEN a.id END) as today_appointments,
    COUNT(DISTINCT CASE WHEN a.status = 'completed' THEN a.id END) as completed_appointments,
    COUNT(DISTINCT CASE WHEN a.status = 'cancelled' THEN a.id END) as cancelled_appointments,
    COUNT(DISTINCT e.id) as total_events,
    COUNT(DISTINCT a.id) as total_appointments,
    AVG(CASE WHEN a.status = 'completed' THEN 1 ELSE 0 END) as completion_rate,
    COUNT(DISTINCT CASE WHEN DATE(a.created_at) = CURDATE() THEN a.id END) as today_new_appointments
FROM appointments a
LEFT JOIN events e ON a.event_id = e.id
WHERE a.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY);

-- Create stored procedure for updating user score
DELIMITER //
CREATE PROCEDURE IF NOT EXISTS UpdateUserScore(IN user_id INT)
BEGIN
    DECLARE user_score DECIMAL(5,2) DEFAULT 0;
    DECLARE completed_appointments INT DEFAULT 0;
    DECLARE active_events INT DEFAULT 0;
    DECLARE completion_rate DECIMAL(5,2) DEFAULT 0;
    
    -- Calculate metrics
    SELECT 
        COUNT(CASE WHEN a.status = 'completed' THEN 1 END),
        COUNT(DISTINCT e.id),
        AVG(CASE WHEN a.status = 'completed' THEN 1 ELSE 0 END)
    INTO completed_appointments, active_events, completion_rate
    FROM appointments a
    LEFT JOIN events e ON a.event_id = e.id
    WHERE a.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY);
    
    -- Calculate score (max 100)
    SET user_score = LEAST(100, GREATEST(0, 
        (completed_appointments * 2) + 
        (active_events * 5) + 
        (completion_rate * 20)
    ));
    
    -- Update user preferences
    INSERT INTO user_preferences (user_id, preference_key, preference_value)
    VALUES (user_id, 'user_score', user_score)
    ON DUPLICATE KEY UPDATE 
    preference_value = user_score,
    updated_at = CURRENT_TIMESTAMP;
    
    SELECT user_score as calculated_score;
END //
DELIMITER ;

-- Create trigger to log user activities
DELIMITER //
CREATE TRIGGER IF NOT EXISTS log_appointment_activity
AFTER INSERT ON appointments
FOR EACH ROW
BEGIN
    INSERT INTO user_activity_log (user_id, action_type, action_description, target_id, target_type)
    VALUES (NEW.created_by, 'appointment_created', CONCAT('Termin erstellt für ', NEW.customer_name), NEW.id, 'appointment');
END //
DELIMITER ;

-- Create trigger to log event activities
DELIMITER //
CREATE TRIGGER IF NOT EXISTS log_event_activity
AFTER INSERT ON events
FOR EACH ROW
BEGIN
    INSERT INTO user_activity_log (user_id, action_type, action_description, target_id, target_type)
    VALUES (NEW.created_by, 'event_created', CONCAT('Event erstellt: ', NEW.name), NEW.id, 'event');
END //
DELIMITER ;

-- Grant necessary permissions (adjust as needed)
-- GRANT SELECT, INSERT, UPDATE, DELETE ON boxxenstopp.user_dashboard_config TO 'your_app_user'@'localhost';
-- GRANT SELECT, INSERT, UPDATE, DELETE ON boxxenstopp.quick_actions TO 'your_app_user'@'localhost';
-- GRANT SELECT, INSERT, UPDATE, DELETE ON boxxenstopp.user_activity_log TO 'your_app_user'@'localhost';
-- GRANT SELECT, INSERT, UPDATE, DELETE ON boxxenstopp.dashboard_widgets TO 'your_app_user'@'localhost';
-- GRANT SELECT, INSERT, UPDATE, DELETE ON boxxenstopp.dashboard_layouts TO 'your_app_user'@'localhost';
-- GRANT SELECT, INSERT, UPDATE, DELETE ON boxxenstopp.user_preferences TO 'your_app_user'@'localhost';
-- GRANT SELECT, INSERT, UPDATE, DELETE ON boxxenstopp.performance_metrics_cache TO 'your_app_user'@'localhost';
-- GRANT EXECUTE ON PROCEDURE boxxenstopp.UpdateUserScore TO 'your_app_user'@'localhost';

-- Show completion message
SELECT 'Intelligent Dashboard Setup completed successfully!' as message;
