/**
 * Boxenstop - Booking JavaScript
 */

document.addEventListener('DOMContentLoaded', function() {
    const bookingForm = document.getElementById('booking-form');
    const dateInput = document.getElementById('booking_date');
    const timeSlotsContainer = document.getElementById('time-slots');
    const submitBtn = document.getElementById('submit-btn');
    const bringsWheelsRadios = document.querySelectorAll('input[name="brings_wheels"]');
    const storageNumberGroup = document.getElementById('storage_number_group');
    const storageNumberInput = document.getElementById('storage_number');
    const tireTypeGroup = document.getElementById('tire_type_group');
    
    let selectedSlot = null;
    let eventId = getEventIdFromDom();
    
    // Add event listeners to service checkboxes
    const serviceCheckboxes = document.querySelectorAll('input[name^="service_"]');
    serviceCheckboxes.forEach(checkbox => {
        checkbox.addEventListener('change', updatePricing);
    });
    let pricing = {};
    
    // Initialize booking form
    initBookingForm();
    
    /**
     * Initialize booking form
     */
    function initBookingForm() {
        // Load time slots when date changes
        dateInput.addEventListener('change', loadTimeSlots);
        // For single-day events, lock date to event date
        if (dateInput.hasAttribute('max') && dateInput.getAttribute('max') === dateInput.getAttribute('min')) {
            dateInput.value = dateInput.getAttribute('min');
        }
        
        // Handle brings wheels radio change
        bringsWheelsRadios.forEach(radio => {
            radio.addEventListener('change', handleBringsWheelsChange);
        });
        
        // Wiper blades checkbox wurde entfernt
        
        // Handle seasonal check radio change
        const seasonalCheckRadios = document.querySelectorAll('input[name="seasonal_check"]');
        seasonalCheckRadios.forEach(radio => {
            radio.addEventListener('change', updatePricing);
        });
        
        // Add event listeners to all form inputs for real-time pricing updates
        const allInputs = document.querySelectorAll('input[type="radio"], input[type="checkbox"]');
        allInputs.forEach(input => {
            input.addEventListener('change', updatePricing);
        });
        
        // Handle form submission
        bookingForm.addEventListener('submit', handleFormSubmit);
        
        // Load initial time slots
        loadTimeSlots();
        
        // Load pricing settings
        loadPricingSettings();
        
        // Handle brings wheels change on page load
        handleBringsWheelsChange();
        
        // Update pricing on page load
        updatePricing();
    }
    
    /**
     * Handle brings wheels radio change
     */
    function handleBringsWheelsChange() {
        const bringsWheels = document.querySelector('input[name="brings_wheels"]:checked').value === '1';
        
        if (bringsWheels) {
            // If bringing wheels, hide storage number and show tire type
            storageNumberGroup.style.display = 'none';
            storageNumberInput.required = false;
            if (tireTypeGroup) tireTypeGroup.style.display = 'block';
        } else {
            // If not bringing wheels, show storage number and hide tire type
            storageNumberGroup.style.display = 'block';
            storageNumberInput.required = true;
            if (tireTypeGroup) tireTypeGroup.style.display = 'none';
            
            // Auto-generate storage number if empty
            if (!storageNumberInput.value) {
                generateStorageNumber();
            }
        }
        
        // Update pricing
        updatePricing();
    }
    
    /**
     * Generate a storage number
     */
    function generateStorageNumber() {
        // Generate a random 12-digit number
        const randomNumber = Math.floor(100000000000 + Math.random() * 900000000000);
        // Format as 9.xxx.xxx.xxx
        const formattedNumber = '9.' + randomNumber.toString().substring(0, 3) + '.' + randomNumber.toString().substring(3, 6) + '.' + randomNumber.toString().substring(6, 9);
        storageNumberInput.value = formattedNumber;
    }
    
    /**
     * Get event ID from URL
     */
    function getEventIdFromDom() {
        const hidden = document.getElementById('event_id');
        if (hidden && hidden.value) return hidden.value;
        const urlParams = new URLSearchParams(window.location.search);
        return urlParams.get('event');
    }
    
    /**
     * Load available time slots
     */
    function loadTimeSlots() {
        const selectedDate = dateInput.value;
        
        if (!selectedDate || !eventId) {
            return;
        }
        
        timeSlotsContainer.innerHTML = '<p class="loading">Lade verfügbare Zeitslots...</p>';
        
        // Use the new dedicated API endpoint for slots
        fetch(`api/get_available_slots.php?event_id=${eventId}&date=${selectedDate}`)
            .then(response => response.json())
                .then(data => {
                if (data.success) {
                    console.log('API Response:', data);
                    displayTimeSlots(data.available_slots);
                    renderOccupancy(eventId, selectedDate, data);
                } else {
                    console.error('API Error:', data.error);
                    showError('Fehler beim Laden der Zeitslots: ' + data.error);
                }
            })
            .catch(error => {
                console.error('Error loading time slots:', error);
                showError('Fehler beim Laden der Zeitslots');
            });
    }
    
    /**
     * Display time slots
     */
    function displayTimeSlots(slots) {
        if (slots.length === 0) {
            timeSlotsContainer.innerHTML = '<p class="no-slots">Keine verfügbaren Zeitslots für dieses Datum.</p>';
            return;
        }
        
        timeSlotsContainer.innerHTML = '';
        
        slots.forEach(slot => {
            const slotElement = createSlotElement(slot);
            timeSlotsContainer.appendChild(slotElement);
        });
    }
    
    /**
     * Create slot element
     */
    function createSlotElement(slot) {
        const slotDiv = document.createElement('div');
        const isAvailable = slot.available && slot.available_count > 0;
        slotDiv.className = `time-slot ${!isAvailable ? 'unavailable' : ''}`;
        
        if (isAvailable) {
            slotDiv.addEventListener('click', () => selectSlot(slot, slotDiv));
        }
        
        slotDiv.innerHTML = `
            <div class="time">${slot.formatted_time}</div>
            <div class="availability">
                ${isAvailable ? `${slot.available_count}/${slot.max_count} frei` : 'Ausgebucht'}
            </div>
        `;
        
        return slotDiv;
    }
    
    /**
     * Select time slot
     */
    function selectSlot(slot, element) {
        // Remove previous selection
        document.querySelectorAll('.time-slot.selected').forEach(el => {
            el.classList.remove('selected');
        });
        
        // Select new slot
        element.classList.add('selected');
        selectedSlot = slot;
        
        // Enable submit button
        submitBtn.disabled = false;
        
        // Add hidden input for slot time
        let slotTimeInput = document.querySelector('input[name="slot_time"]');
        if (!slotTimeInput) {
            slotTimeInput = document.createElement('input');
            slotTimeInput.type = 'hidden';
            slotTimeInput.name = 'slot_time';
            bookingForm.appendChild(slotTimeInput);
        }
        slotTimeInput.value = slot.time;
        
        // Show success message
        showSuccess(`Zeitslot ${slot.formatted_time} ausgewählt`);

        // Update summary card
        const summary = document.getElementById('summary-slot');
        if (summary) summary.textContent = slot.formatted_time;
    }
    
    /**
     * Wiper blades functionality wurde entfernt
     */
    
    /**
     * Load pricing settings
     */
    function loadPricingSettings() {
        console.log('Loading autohaus pricing settings...');
        
        // Verwende Autohaus-spezifische Preise falls verfügbar
        if (window.autohausPricing) {
            pricing = window.autohausPricing;
            console.log('Autohaus pricing loaded successfully:', pricing);
            updatePricing();
            return;
        }
        
        // Fallback zu globalen Preisen
        console.log('Loading global pricing settings...');
        fetch('api/settings.php')
            .then(response => {
                console.log('Settings API response status:', response.status);
                return response.json();
            })
            .then(data => {
                console.log('Settings API response data:', data);
                if (data.success) {
                    pricing = data.settings;
                    console.log('Global pricing loaded successfully:', pricing);
                    updatePricing();
                } else {
                    console.error('Settings API returned error:', data.error);
                    setFallbackPricing();
                }
            })
            .catch(error => {
                console.error('Error loading pricing settings:', error);
                setFallbackPricing();
            });
    }
    
    /**
     * Set fallback pricing
     */
    function setFallbackPricing() {
        pricing = {
            price_wheel_change: 'auf_anfrage',
            price_storage: 'auf_anfrage',
            price_spring_check: 'auf_anfrage',
            price_winter_check: 'auf_anfrage'
        };
        console.log('Using fallback pricing:', pricing);
        updatePricing();
    }
    
    function loadServices() {
        if (window.autohausServices && window.autohausServices.length > 0) {
            console.log('Loaded services:', window.autohausServices);
            return window.autohausServices;
        }
        return [];
    }
    
    /**
     * Update pricing display
     */
    function updatePricing() {
        console.log('updatePricing called, pricing object:', pricing);
        
        if (!pricing || !pricing.price_wheel_change) {
            console.log('Pricing not loaded yet, using fallback');
            // Use fallback pricing if not loaded yet
            pricing = {
                price_wheel_change: 'auf_anfrage',
                price_storage: 'auf_anfrage',
                price_spring_check: 'auf_anfrage',
                price_winter_check: 'auf_anfrage'
            };
        }
        
        const summaryPricing = document.getElementById('summary-pricing');
        const summaryTotal = document.getElementById('summary-total');
        const priceBreakdown = document.getElementById('price-breakdown');
        const totalPrice = document.getElementById('total-price');
        
        if (!summaryPricing || !summaryTotal || !priceBreakdown || !totalPrice) {
            console.log('Required elements not found:', {
                summaryPricing: !!summaryPricing,
                summaryTotal: !!summaryTotal,
                priceBreakdown: !!priceBreakdown,
                totalPrice: !!totalPrice
            });
            return;
        }
        
        let total = 0;
        let breakdown = [];
        
        // Base price for wheel change
        if (pricing.price_wheel_change === 'auf_anfrage') {
            breakdown.push(`Räderwechsel: ${pricing.price_wheel_change}`);
        } else {
            total += parseFloat(pricing.price_wheel_change);
            breakdown.push(`Räderwechsel: ${pricing.price_wheel_change}€`);
        }
        console.log('Base wheel change price added:', pricing.price_wheel_change);
        
        // Storage price if not bringing wheels
        const bringsWheels = document.querySelector('input[name="brings_wheels"]:checked');
        console.log('Brings wheels radio:', bringsWheels ? bringsWheels.value : 'not found');
        if (bringsWheels && bringsWheels.value === '0') {
            if (pricing.price_storage === 'auf_anfrage') {
                breakdown.push(`Einlagerung: ${pricing.price_storage}`);
            } else {
                total += parseFloat(pricing.price_storage);
                breakdown.push(`Einlagerung: ${pricing.price_storage}€`);
            }
            console.log('Added storage price:', pricing.price_storage);
        }
        
        // Dynamische Services
        const services = loadServices();
        services.forEach(service => {
            const serviceCheckbox = document.querySelector(`input[name="service_${service.id}"]`);
            if (serviceCheckbox && serviceCheckbox.checked) {
                if (service.show_price && service.price) {
                    total += parseFloat(service.price);
                    breakdown.push(`${service.service_name}: ${service.price}€`);
                } else {
                    breakdown.push(`${service.service_name}: auf Anfrage`);
                }
                console.log('Added service:', service.service_name, service.price);
            }
        });
        
        // Seasonal check - check both radio buttons and select dropdown
        const seasonalCheckRadio = document.querySelector('input[name="seasonal_check"]:checked');
        const seasonalCheckSelect = document.querySelector('select[name="seasonal_check"]');
        const seasonalCheckValue = seasonalCheckRadio ? seasonalCheckRadio.value : (seasonalCheckSelect ? seasonalCheckSelect.value : 'none');
        
        console.log('Seasonal check value:', seasonalCheckValue);
        if (seasonalCheckValue && seasonalCheckValue !== 'none') {
            const checkPrice = seasonalCheckValue === 'spring' ? pricing.price_spring_check : pricing.price_winter_check;
            const checkName = seasonalCheckValue === 'spring' ? 'Frühjahres-Check' : 'Winter-Check';
            
            if (checkPrice === 'auf_anfrage') {
                breakdown.push(`${checkName}: ${checkPrice}`);
            } else {
                total += parseFloat(checkPrice);
                breakdown.push(`${checkName}: ${checkPrice}€`);
            }
            console.log('Added seasonal check price:', checkPrice);
        }
        
        // Update display
        console.log('Final breakdown:', breakdown);
        console.log('Final total:', total);
        
        if (breakdown.length > 0) {
            priceBreakdown.innerHTML = breakdown.map(item => `<div class="price-item">${item}</div>`).join('');
            summaryPricing.style.display = 'block';
            summaryTotal.style.display = 'block';
            
            // Check if all prices are "auf_anfrage"
            const hasAufAnfrage = breakdown.some(item => item.includes('auf_anfrage'));
            if (hasAufAnfrage && total === 0) {
                totalPrice.textContent = 'auf Anfrage';
            } else {
                totalPrice.textContent = `${total.toFixed(2)}€`;
            }
        } else {
            summaryPricing.style.display = 'none';
            summaryTotal.style.display = 'block';
            totalPrice.textContent = 'auf Anfrage';
        }
    }
    
    /**
     * Format storage number as user types (9.xxx.xxx.xxx)
     */
    if (storageNumberInput) {
        storageNumberInput.addEventListener('input', function(e) {
            // Remove all non-digit characters
            let value = this.value.replace(/\D/g, '');
            
            // Add dots after first digit and after next 3 digits
            if (value.length > 1) {
                value = value.charAt(0) + '.' + value.substring(1, 4) + (value.length > 4 ? '.' + value.substring(4, 7) : '') + (value.length > 7 ? '.' + value.substring(7, 10) : '');
            }
            
            // Update the input value
            this.value = value;
        });
    }
    
    /**
     * Handle form submission
     */
    function handleFormSubmit(e) {
        e.preventDefault();
        if (!selectedSlot) {
            showError('Bitte wählen Sie einen Zeitslot aus');
            return;
        }
        
        // Validate form before submission
        if (!validateForm()) {
            return;
        }
        
        // Set loading state
        setButtonState('loading');
        
        // Collect form data
        const formData = new FormData(bookingForm);
        const data = Object.fromEntries(formData.entries());
        
        // Convert boolean values
        data.brings_wheels = data.brings_wheels === '1';
        data.has_wheel_lock = data.has_wheel_lock === '1';
        
        // Email is no longer required since we removed the email system

        // Submit form
        // Ensure event_id, slot_date and slot_time are present
        data.event_id = eventId;
        data.slot_date = data.slot_date || dateInput.value;
        data.slot_time = selectedSlot.time;
        
        // CSRF-Token hinzufügen
        data.csrf_token = document.querySelector('meta[name="csrf-token"]')?.content || '';

        fetch('api/appointments.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(data)
        })
        .then(response => {
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            return response.json();
        })
        .then(result => {
            if (result.success) {
                setButtonState('success');
                
                // Redirect to index with success parameter
                window.location.href = 'index.php?success=request';
            } else {
                setButtonState('error');
                showError(result.error || 'Fehler beim Buchen des Termins');
            }
        })
        .catch(error => {
            console.error('Error booking appointment:', error);
            setButtonState('error');
            showError('Fehler beim Buchen des Termins. Bitte versuchen Sie es erneut.');
        });
    }

    /**
     * Render simple occupancy indicator for selected date
     */
    function renderOccupancy(eventId, date, apiData) {
        const container = document.getElementById('occupancy-calendar');
        if (!container) return;

        // Use API data for occupancy calculation
        const slots = apiData.available_slots || [];
        let totalPlaces = 0;
        let bookedPlaces = 0;
        let availableSlots = 0;
        let totalSlots = slots.length;
        
        // Calculate total capacity and booked slots
        slots.forEach(slot => {
            totalPlaces += (slot.max_count || 0);
            bookedPlaces += (slot.booked || 0);
            if (slot.available && slot.available_count > 0) {
                availableSlots++;
            }
        });
        
        // Calculate occupancy ratio
        const occupancyRatio = totalPlaces > 0 ? bookedPlaces / totalPlaces : 0;
        const occupancyPercentage = Math.round(occupancyRatio * 100);
        
        // Calculate slot utilization
        const slotUtilization = totalSlots > 0 ? (totalSlots - availableSlots) / totalSlots : 0;
        const slotUtilizationPercentage = Math.round(slotUtilization * 100);
        
        // Get status based on occupancy
        const statusClass = getLoadClass(occupancyRatio);
        const statusText = getOccupancyStatusText(occupancyPercentage);
        
        container.innerHTML = `
            <div class="occupancy-card">
                <div class="occupancy-header">
                    <h4><i class="fas fa-calendar-check"></i> Auslastung</h4>
                    <span class="occupancy-badge ${statusClass}">${occupancyPercentage}%</span>
                </div>
                
                <div class="occupancy-stats">
                    <div class="stat-item">
                        <div class="stat-label">
                            <i class="fas fa-users"></i>
                            Plätze
                        </div>
                        <div class="stat-value">
                            <span class="booked">${bookedPlaces}</span> / 
                            <span class="total">${totalPlaces}</span> belegt
                        </div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-label">
                            <i class="fas fa-clock"></i>
                            Zeitslots
                        </div>
                        <div class="stat-value">
                            <span class="available">${availableSlots}</span> von 
                            <span class="total">${totalSlots}</span> verfügbar
                        </div>
                    </div>
                </div>
                
                <div class="occupancy-bar">
                    <div class="occupancy-fill ${statusClass}" style="width:${occupancyPercentage}%"></div>
                </div>
                
                <div class="occupancy-status">
                    <i class="fas fa-${getOccupancyIcon(occupancyPercentage)}"></i>
                    <span class="status-text">${statusText}</span>
                </div>
            </div>
        `;
    }
    
    /**
     * Get occupancy status text
     */
    function getOccupancyStatusText(percentage) {
        if (percentage === 0) return 'Keine Buchungen';
        if (percentage < 25) return 'Geringe Auslastung';
        if (percentage < 50) return 'Moderate Auslastung';
        if (percentage < 75) return 'Hohe Auslastung';
        if (percentage < 100) return 'Sehr hohe Auslastung';
        return 'Ausgebucht';
    }
    
    /**
     * Get occupancy icon
     */
    function getOccupancyIcon(percentage) {
        if (percentage === 0) return 'leaf';
        if (percentage < 25) return 'check-circle';
        if (percentage < 50) return 'exclamation-circle';
        if (percentage < 75) return 'clock';
        if (percentage < 100) return 'exclamation-triangle';
        return 'times-circle';
    }

    function getLoadClass(ratio){
        if (ratio === 1) return 'full';
        if (ratio >= 0.75) return 'high';
        if (ratio >= 0.4) return 'medium';
        return 'low';
    }

    /**
     * Set button state with visual feedback
     */
    function setButtonState(state) {
        const icon = submitBtn.querySelector('i');
        const text = submitBtn.querySelector('span');
        
        // Remove all state classes
        submitBtn.classList.remove('btn-loading', 'btn-success', 'btn-error');
        
        switch (state) {
            case 'loading':
                submitBtn.disabled = true;
                submitBtn.classList.add('btn-loading');
                icon.className = 'fas fa-spinner fa-spin';
                text.textContent = 'Sende Anfrage...';
                break;
                
            case 'success':
                submitBtn.disabled = true;
                submitBtn.classList.add('btn-success');
                icon.className = 'fas fa-check-circle';
                text.textContent = 'Anfrage gesendet!';
                break;
                
            case 'error':
                submitBtn.disabled = false;
                submitBtn.classList.add('btn-error');
                icon.className = 'fas fa-exclamation-circle';
                text.textContent = 'Fehler - Erneut versuchen';
                // Reset to normal state after 3 seconds
                setTimeout(() => {
                    setButtonState('normal');
                }, 3000);
                break;
                
            case 'normal':
            default:
                submitBtn.disabled = false;
                icon.className = 'fas fa-paper-plane';
                text.textContent = 'Terminanfrage senden';
                break;
        }
    }
    
    /**
     * Disable form after successful booking
     */
    function disableForm() {
        const formInputs = bookingForm.querySelectorAll('input, select, textarea, button');
        formInputs.forEach(input => {
            if (input !== submitBtn) {
                input.disabled = true;
            }
        });
        
        // Disable time slot selection
        const timeSlots = document.querySelectorAll('.time-slot');
        timeSlots.forEach(slot => {
            slot.style.pointerEvents = 'none';
            slot.style.opacity = '0.6';
        });
    }
    
    /**
     * Show success message
     */
    function showSuccess(message) {
        const alert = document.createElement('div');
        alert.className = 'alert alert-success';
        alert.innerHTML = `<i class="fas fa-check-circle"></i> ${message}`;
        
        const container = document.querySelector('.booking-section .section-container');
        const existingAlert = container.querySelector('.alert');
        if (existingAlert) {
            existingAlert.remove();
        }
        
        container.insertBefore(alert, container.querySelector('.booking-layout'));
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            if (alert.parentNode) {
                alert.remove();
            }
        }, 5000);
    }
    
    /**
     * Show success message with calendar download option
     */
    function showSuccessWithCalendar(eventDetails) {
        const container = document.querySelector('.booking-section .section-container');
        const existingAlert = container.querySelector('.alert');
        if (existingAlert) {
            existingAlert.remove();
        }
        
        // Create success alert with calendar download
        const alert = document.createElement('div');
        alert.className = 'alert alert-success';
        alert.style.cssText = `
            background: #d1ecf1;
            border: 2px solid #bee5eb;
            border-radius: 12px;
            padding: 2rem;
            margin-bottom: 2rem;
            text-align: center;
        `;
        
        alert.innerHTML = `
            <div style="margin-bottom: 1.5rem;">
                <i class="fas fa-check-circle" style="font-size: 3rem; color: #28a745; margin-bottom: 1rem;"></i>
                <h3 style="color: #155724; margin-bottom: 0.5rem;">Termin erfolgreich gebucht!</h3>
                <p style="color: #0c5460;">Sie erhalten eine Bestätigung per E-Mail.</p>
            </div>
            
            <div style="background: white; padding: 1.5rem; border-radius: 8px; margin-bottom: 1.5rem; text-align: left;">
                <h4 style="color: #003057; margin-bottom: 1rem;"><i class="fas fa-calendar-alt"></i> Termin-Details</h4>
                <div style="display: grid; gap: 0.75rem;">
                    <div style="display: flex; justify-content: space-between;">
                        <span style="color: #6c757d; font-weight: 500;">Event:</span>
                        <span style="color: #003057; font-weight: 600;">${eventDetails.title}</span>
                    </div>
                    <div style="display: flex; justify-content: space-between;">
                        <span style="color: #6c757d; font-weight: 500;">Datum:</span>
                        <span style="color: #003057; font-weight: 600;">${eventDetails.date}</span>
                    </div>
                    <div style="display: flex; justify-content: space-between;">
                        <span style="color: #6c757d; font-weight: 500;">Zeit:</span>
                        <span style="color: #003057; font-weight: 600;">${eventDetails.time}</span>
                    </div>
                    <div style="display: flex; justify-content: space-between;">
                        <span style="color: #6c757d; font-weight: 500;">Kennzeichen:</span>
                        <span style="color: #003057; font-weight: 600;">${eventDetails.license_plate}</span>
                    </div>
                </div>
            </div>
            
            <div style="background: #fff3cd; border: 2px solid #ffc107; border-radius: 8px; padding: 1.25rem; margin-bottom: 1.5rem;">
                <h4 style="color: #856404; margin-bottom: 0.75rem;"><i class="fas fa-download"></i> In Kalender eintragen</h4>
                <p style="color: #856404; margin: 0 0 1rem 0;">Laden Sie die Kalenderdatei herunter und fügen Sie den Termin in Ihren Kalender ein (Google Calendar, Outlook, iCal, etc.).</p>
                <button onclick="downloadCalendar(${eventDetails.appointment_id})" style="
                    background: linear-gradient(135deg, #003057 0%, #0056b3 100%);
                    color: white;
                    border: none;
                    padding: 0.75rem 1.5rem;
                    border-radius: 8px;
                    cursor: pointer;
                    font-weight: 600;
                    font-size: 1rem;
                    transition: all 0.3s ease;
                    box-shadow: 0 4px 12px rgba(0, 48, 87, 0.3);
                    display: inline-flex;
                    align-items: center;
                    gap: 0.5rem;
                " onmouseover="this.style.transform='translateY(-2px)'; this.style.boxShadow='0 6px 16px rgba(0, 48, 87, 0.4)'" onmouseout="this.style.transform=''; this.style.boxShadow='0 4px 12px rgba(0, 48, 87, 0.3)'">
                    <i class="fas fa-download"></i>
                    Kalenderdatei herunterladen
                </button>
            </div>
            
            <div style="display: flex; gap: 1rem; justify-content: center;">
                <button onclick="window.location.href='index.php'" style="
                    background: #6c757d;
                    color: white;
                    border: none;
                    padding: 0.75rem 1.5rem;
                    border-radius: 8px;
                    cursor: pointer;
                    font-weight: 500;
                ">
                    <i class="fas fa-home"></i> Zur Startseite
                </button>
                <button onclick="window.location.href='profile.php'" style="
                    background: #0056b3;
                    color: white;
                    border: none;
                    padding: 0.75rem 1.5rem;
                    border-radius: 8px;
                    cursor: pointer;
                    font-weight: 500;
                ">
                    <i class="fas fa-user"></i> Zum Profil
                </button>
            </div>
        `;
        
        container.insertBefore(alert, container.querySelector('.booking-layout'));
        
        // Disable form
        disableForm();
        
        // Hide booking form
        bookingForm.style.display = 'none';
    }
    
    /**
     * Download calendar file
     */
    window.downloadCalendar = function(appointmentId) {
        window.location.href = `download_appointment_ics.php?id=${appointmentId}`;
    }
    
    /**
     * Show error message
     */
    function showError(message) {
        const alert = document.createElement('div');
        alert.className = 'alert alert-error';
        alert.innerHTML = `<i class="fas fa-exclamation-circle"></i> ${message}`;
        
        const container = document.querySelector('.booking-section .section-container');
        const existingAlert = container.querySelector('.alert');
        if (existingAlert) {
            existingAlert.remove();
        }
        
        container.insertBefore(alert, container.querySelector('.booking-layout'));
        
        // Auto remove after 8 seconds
        setTimeout(() => {
            if (alert.parentNode) {
                alert.remove();
            }
        }, 8000);
    }
    
    /**
     * Validate form before submission
     */
    function validateForm() {
        const requiredFields = bookingForm.querySelectorAll('[required]');
        let isValid = true;
        
        requiredFields.forEach(field => {
            if (!field.value.trim()) {
                field.classList.add('error');
                isValid = false;
            } else {
                field.classList.remove('error');
            }
        });
        
        if (!selectedSlot) {
            showError('Bitte wählen Sie einen Zeitslot aus');
            isValid = false;
        }
        
        return isValid;
    }
    
    // Add CSS for booking specific styles
    const bookingStyle = document.createElement('style');
    bookingStyle.textContent = `
        .no-slots {
            text-align: center;
            color: #666;
            font-style: italic;
            padding: 2rem;
            background: #f8f9fa;
            border-radius: 8px;
            border: 2px dashed #dee2e6;
        }
        
        .time-slot {
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .time-slot:hover:not(.unavailable) {
            transform: translateY(-2px);
        }
        
        .time-slot.unavailable {
            cursor: not-allowed;
            opacity: 0.6;
        }
        
        .time-slot.selected {
            transform: scale(1.05);
        }
        
        .field-error {
            color: #e10600;
            font-size: 0.875rem;
            margin-top: 0.25rem;
        }
        
        input.error {
            border-color: #e10600;
            box-shadow: 0 0 0 3px rgba(225, 6, 0, 0.1);
        }
        
        .btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }
        
        .btn-loading {
            background: linear-gradient(135deg, #007bff, #0056b3);
            color: white;
            position: relative;
            overflow: hidden;
        }
        
        .btn-loading::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
            animation: loading-shimmer 1.5s infinite;
        }
        
        .btn-success {
            background: linear-gradient(135deg, #28a745, #1e7e34);
            color: white;
            border-color: #28a745;
            animation: success-pulse 0.6s ease-out;
        }
        
        .btn-error {
            background: linear-gradient(135deg, #dc3545, #c82333);
            color: white;
            border-color: #dc3545;
            animation: error-shake 0.5s ease-in-out;
        }
        
        .fa-spin {
            animation: fa-spin 1s infinite linear;
        }
        
        @keyframes fa-spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        @keyframes loading-shimmer {
            0% { left: -100%; }
            100% { left: 100%; }
        }
        
        @keyframes success-pulse {
            0% { transform: scale(1); }
            50% { transform: scale(1.05); }
            100% { transform: scale(1); }
        }
        
        @keyframes error-shake {
            0%, 100% { transform: translateX(0); }
            25% { transform: translateX(-5px); }
            75% { transform: translateX(5px); }
        }
        
        .alert {
            margin-bottom: 1rem;
            padding: 1rem;
            border-radius: 8px;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-weight: 500;
            animation: slideDown 0.3s ease-out;
        }
        
        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
    `;
    document.head.appendChild(bookingStyle);
});
