<?php
// upload_verification_document.php

// CORS/JSON-Header
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Api-Key, X-Timestamp, X-Nonce, X-Signature');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(200); exit; }

// Log-Datei-Funktion
function logToFile($message, $data = null) {
    $logDir = __DIR__ . '/../logs';
    if (!is_dir($logDir)) {
        @mkdir($logDir, 0755, true);
    }
    $logFile = $logDir . '/upload_verification_' . date('Y-m-d') . '.log';
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] $message";
    if ($data !== null) {
        $logMessage .= "\n" . (is_string($data) ? $data : json_encode($data, JSON_PRETTY_PRINT));
    }
    $logMessage .= "\n" . str_repeat('-', 80) . "\n";
    @file_put_contents($logFile, $logMessage, FILE_APPEND);
    // Auch in error_log schreiben (für Server-Logs)
    error_log($message);
}

// Session-Konfiguration: Stelle sicher, dass Cookies korrekt gesetzt werden
ini_set('session.cookie_httponly', 0); // Erlaube JavaScript-Zugriff (falls nötig)
ini_set('session.use_cookies', 1);
ini_set('session.use_only_cookies', 1);
session_start();

// Log Session-Info
logToFile("===== UPLOAD VERIFICATION REQUEST START =====");
logToFile("Session ID", session_id());
logToFile("Session status", session_status() === PHP_SESSION_ACTIVE ? 'ACTIVE' : 'INACTIVE');
logToFile("Session user_id (vor Prüfung)", $_SESSION['user_id'] ?? 'NOT SET');
logToFile("Session komplett", $_SESSION);
logToFile("Request Method", $_SERVER['REQUEST_METHOD'] ?? 'NOT SET');
logToFile("Content-Type", $_SERVER['CONTENT_TYPE'] ?? 'NOT SET');
logToFile("HTTP Cookies (Request)", $_SERVER['HTTP_COOKIE'] ?? 'NOT SET');
logToFile("All Headers", getallheaders());

// Debug-Logging am Anfang
logToFile("_POST keys", empty($_POST) ? 'EMPTY' : array_keys($_POST));
logToFile("_POST vollständig", $_POST);
logToFile("_FILES keys", empty($_FILES) ? 'EMPTY' : array_keys($_FILES));
if (!empty($_POST['email'])) {
    logToFile("_POST['email'] gefunden", $_POST['email']);
}
if (!empty($_POST['password'])) {
    logToFile("_POST['password'] gefunden", "SET (length: " . strlen($_POST['password']) . ")");
}

// 1) User ermitteln: Session ODER Bearer-Token akzeptieren
function getBearerToken(): ?string {
    $hdrs = getallheaders();
    if (!$hdrs) return null;
    foreach ($hdrs as $k => $v) {
        if (strtolower($k) === 'authorization' && stripos($v, 'bearer ') === 0) {
            return trim(substr($v, 7));
        }
    }
    return null;
}

// Database connection helper - verwendet config.php wie login_user.php
function getDbConnection() {
    static $pdo = null;
    if ($pdo !== null) return $pdo;
    
    logToFile("getDbConnection: Versuche DB-Verbindung herzustellen");
    
    try {
        // Methode 1: Versuche config.php (wie login_user.php es macht)
        $configPath = __DIR__ . '/config.php';
        logToFile("getDbConnection: Prüfe config.php", $configPath);
        
        if (file_exists($configPath)) {
            logToFile("getDbConnection: config.php gefunden, lade...");
            require_once $configPath;
            
            // Nach require_once sollte $pdo verfügbar sein (globale Variable)
            if (isset($pdo) && $pdo instanceof PDO) {
                logToFile("getDbConnection SUCCESS", "Verbindung aus config.php geladen (via \$pdo)");
                return $pdo;
            }
            
            // Prüfe globale Variablen
            if (isset($GLOBALS['pdo']) && $GLOBALS['pdo'] instanceof PDO) {
                logToFile("getDbConnection SUCCESS", "Verbindung aus config.php geladen (via \$GLOBALS['pdo'])");
                $pdo = $GLOBALS['pdo'];
                return $pdo;
            }
        } else {
            logToFile("getDbConnection WARNING", "config.php nicht gefunden: $configPath");
        }
        
        // Methode 2: Versuche config/database.php (wie index.php und booking.php)
        $dbConfigPath = __DIR__ . '/../config/database.php';
        logToFile("getDbConnection: Prüfe config/database.php", $dbConfigPath);
        
        if (file_exists($dbConfigPath)) {
            logToFile("getDbConnection: config/database.php gefunden, lade...");
            require_once $dbConfigPath;
            
            // Versuche Database-Klasse
            if (class_exists('Database')) {
                try {
                    $db = new Database();
                    $pdo = $db->getConnection();
                    if ($pdo instanceof PDO) {
                        logToFile("getDbConnection SUCCESS", "Verbindung via Database-Klasse hergestellt");
                        return $pdo;
                    }
                } catch (Exception $e) {
                    logToFile("getDbConnection WARNING", "Database-Klasse gefunden, aber getConnection() fehlgeschlagen: " . $e->getMessage());
                }
            }
        }
        
        // Methode 3: Fallback mit Constants
        if (defined('DB_HOST') && defined('DB_NAME') && defined('DB_USER')) {
            $host = DB_HOST;
            $dbname = DB_NAME;
            $user = DB_USER;
            $pass = defined('DB_PASS') ? DB_PASS : '';
            logToFile("getDbConnection: Verwende Constants", "Host: $host, DB: $dbname, User: $user");
            try {
                $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $user, $pass);
                $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
                logToFile("getDbConnection SUCCESS", "Verbindung via Constants hergestellt");
                return $pdo;
            } catch (PDOException $e) {
                logToFile("getDbConnection ERROR", "Constants-Verbindung fehlgeschlagen: " . $e->getMessage());
            }
        }
        
        logToFile("getDbConnection ERROR", "Alle Verbindungsversuche fehlgeschlagen");
        return null;
        
    } catch (Exception $e) {
        logToFile("getDbConnection ERROR", "Exception: " . $e->getMessage());
        logToFile("getDbConnection ERROR", "Stack trace: " . $e->getTraceAsString());
        return null;
    }
}

// Auth token -> user_id
function userIdFromToken(string $token) {
    $pdo = getDbConnection();
    if (!$pdo) return null;
    try {
        $stmt = $pdo->prepare("SELECT id FROM users WHERE auth_token = ? AND is_active = 1 LIMIT 1");
        $stmt->execute([$token]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return $row ? (int)$row['id'] : null;
    } catch (Exception $e) {
        error_log("userIdFromToken error: " . $e->getMessage());
        return null;
    }
}

// Email/Password -> user_id (für pending-User ohne Token)
function userIdFromCredentials(string $email, string $password) {
    // Verwende config.php wie login_user.php
    try {
        // Lade config.php falls noch nicht geladen
        if (!isset($GLOBALS['pdo'])) {
            $configPath = __DIR__ . '/config.php';
            if (file_exists($configPath)) {
                require_once $configPath;
            }
        }
        
        // Verwende globale $pdo Variable (wie in login_user.php)
        global $pdo;
        
        if (!isset($pdo) || !($pdo instanceof PDO)) {
            // Fallback: Versuche getDbConnection()
            $pdo = getDbConnection();
        }
        
        if (!$pdo || !($pdo instanceof PDO)) {
            logToFile("userIdFromCredentials ERROR", "Keine DB-Verbindung verfügbar");
            return null;
        }
        
        logToFile("userIdFromCredentials: Suche User", $email);
        // WICHTIG: login_user.php verwendet 'password_hash', nicht 'password'
        $stmt = $pdo->prepare("SELECT id, password_hash FROM users WHERE email = ? AND is_active = 1 LIMIT 1");
        $stmt->execute([$email]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$row) {
            logToFile("userIdFromCredentials ERROR", "User nicht gefunden für Email: $email");
            return null;
        }
        
        $userId = (int)$row['id'];
        $storedHash = $row['password_hash']; // WICHTIG: login_user.php verwendet 'password_hash'
        logToFile("userIdFromCredentials: User gefunden", "User ID: $userId, Hash-Länge: " . strlen($storedHash));
        
        // Passwort-Prüfung - genau wie in login_user.php
        // login_user.php verwendet: password_verify($input['password'], $user['password_hash'])
        if (!$storedHash || !password_verify($password, $storedHash)) {
            logToFile("userIdFromCredentials ERROR", "Passwort-Hash stimmt nicht überein");
            logToFile("userIdFromCredentials DEBUG", [
                'Hash-Länge' => strlen($storedHash ?? ''),
                'Hash-Präfix' => substr($storedHash ?? '', 0, 20),
                'Password-Länge' => strlen($password)
            ]);
            return null;
        }
        
        logToFile("userIdFromCredentials SUCCESS", "password_verify stimmt überein");
        return $userId;
        
    } catch (Exception $e) {
        logToFile("userIdFromCredentials ERROR", "Exception: " . $e->getMessage());
        return null;
    }
}

$userId = $_SESSION['user_id'] ?? null;
logToFile("Initial Session user_id", $userId ?? 'NOT SET');
logToFile("Session ID beim Start", session_id());
logToFile("Alle Session-Daten", $_SESSION);

// Prüfe auch, ob Session-Cookie im Request vorhanden ist
$cookieHeader = $_SERVER['HTTP_COOKIE'] ?? '';
if (!empty($cookieHeader)) {
    logToFile("Cookie-Header im Request", $cookieHeader);
    // Extrahiere PHPSESSID aus Cookie-Header
    if (preg_match('/PHPSESSID=([^;]+)/i', $cookieHeader, $matches)) {
        $requestSessionId = trim($matches[1]);
        logToFile("Session-ID aus Cookie-Header", $requestSessionId);
        if ($requestSessionId !== session_id()) {
            logToFile("WARNING", "Session-ID im Cookie stimmt nicht mit aktueller Session überein!");
            logToFile("Cookie Session-ID", $requestSessionId);
            logToFile("Aktuelle Session-ID", session_id());
            // Versuche Session mit der ID aus dem Cookie zu laden
            session_write_close();
            session_id($requestSessionId);
            session_start();
            logToFile("Session neu geladen mit Cookie-ID", "User ID in Session: " . ($_SESSION['user_id'] ?? 'NOT SET'));
            $userId = $_SESSION['user_id'] ?? null;
        }
    }
}

if (!$userId) {
    // 1. Versuche Bearer Token
    $token = getBearerToken();
    if ($token) {
        logToFile("Bearer Token gefunden", substr($token, 0, 20) . '...');
        $userId = userIdFromToken($token);
        if ($userId) { 
            $_SESSION['user_id'] = $userId;
            logToFile("User-ID aus Token", "User ID: $userId");
        } else {
            logToFile("Token ungültig", "Token konnte nicht zu User-ID aufgelöst werden");
        }
    } else {
        logToFile("Kein Bearer Token", "Authorization-Header nicht gefunden");
    }
}

// 2. Wenn immer noch kein User: Prüfe Email/Password im Multipart-Form-Data
if (!$userId) {
    logToFile("DEBUG: Kein User-ID gefunden. Prüfe Email/Password...");
    
    // Prüfe alle möglichen Quellen für Email/Password
    $email = null;
    $password = null;
    
    // 1. Versuche _POST (normal bei multipart/form-data)
    if (!empty($_POST['email'])) {
        $email = trim($_POST['email']);
        logToFile("Email aus _POST gefunden", $email);
    }
    if (!empty($_POST['password'])) {
        $password = $_POST['password'];
        logToFile("Password aus _POST gefunden", "SET (length: " . strlen($password) . ")");
    }
    
    // 2. Falls nicht in _POST, könnte es ein Problem mit multipart/form-data Parsing sein
    // Bei multipart/form-data sollte PHP die Daten automatisch in $_POST packen
    // ABER: Manchmal sind die Daten nur in $_FILES oder müssen manuell geparst werden
    if (!$email || !$password) {
        logToFile("WARNING: Email/Password nicht vollständig in _POST!");
        logToFile("_POST vollständig", $_POST);
        logToFile("CONTENT_TYPE", $_SERVER['CONTENT_TYPE'] ?? 'NOT SET');
        logToFile("REQUEST_METHOD", $_SERVER['REQUEST_METHOD'] ?? 'NOT SET');
        logToFile("_FILES keys", array_keys($_FILES));
        
        // Versuche manuelle Extraktion aus php://input (nur wenn multipart noch nicht geparst wurde)
        // ACHTUNG: php://input ist bei multipart/form-data normalerweise leer, ABER
        // manchmal kann man trotzdem darauf zugreifen, wenn die Daten noch nicht geparst wurden
        $contentType = $_SERVER['CONTENT_TYPE'] ?? '';
        if (strpos($contentType, 'multipart/form-data') !== false && (!$email || !$password)) {
            logToFile("DEBUG: Versuche manuelle Parsing von multipart/form-data...");
            
            // Versuche, die Boundary aus Content-Type zu extrahieren
            if (preg_match('/boundary=(.+)/i', $contentType, $matches)) {
                $boundary = '--' . trim($matches[1]);
                logToFile("Boundary gefunden", $boundary);
                
                // Lese den gesamten Request-Body
                $input = file_get_contents('php://input');
                if (!empty($input)) {
                    logToFile("php://input verfügbar", "Länge: " . strlen($input));
                    logToFile("php://input preview", substr($input, 0, 500));
                    
                    // Suche nach email-Feld
                    if (preg_match('/Content-Disposition:\s*form-data[^;]*;\s*name=["\']?email["\']?\s*\r?\n\r?\n([^\r\n]+)/i', $input, $emailMatch)) {
                        $email = trim($emailMatch[1]);
                        logToFile("Email aus manuellem Parsing", $email);
                    }
                    
                    // Suche nach password-Feld
                    if (preg_match('/Content-Disposition:\s*form-data[^;]*;\s*name=["\']?password["\']?\s*\r?\n\r?\n([^\r\n]+)/i', $input, $passMatch)) {
                        $password = trim($passMatch[1]);
                        logToFile("Password aus manuellem Parsing", "SET (length: " . strlen($password) . ")");
                    }
                } else {
                    logToFile("php://input ist leer", "normal bei bereits geparstem multipart");
                }
            }
        }
    }
    
    // 3. Authentifizierung versuchen
    if ($email && $password) {
        logToFile("DEBUG: Versuche Authentifizierung für", $email);
        $userId = userIdFromCredentials($email, $password);
        if ($userId) {
            $_SESSION['user_id'] = $userId;
            logToFile("SUCCESS: Authentifizierung via Email/Password erfolgreich", "User ID: $userId");
        } else {
            logToFile("ERROR: Authentifizierung via Email/Password fehlgeschlagen", $email);
        }
    } else {
        logToFile("ERROR: Email oder Password konnten nicht extrahiert werden");
        logToFile("Final email", $email ?: 'NULL');
        logToFile("Final password", $password ? 'SET' : 'NULL');
    }
} else {
    logToFile("User-ID bereits vorhanden", "User ID: $userId (aus Session oder Token)");
}

// Ohne identifizierten User -> 401
if (!$userId) {
    logToFile("ERROR: Keine User-ID gefunden - Upload abgelehnt");
    logToFile("Session nach allen Versuchen", $_SESSION);
    logToFile("===== REQUEST END (401) =====");
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Nicht angemeldet']);
    exit;
}

logToFile("SUCCESS: User-ID gefunden", "User ID: $userId");

// 2) Upload-Feld finden (verification_document|document|file|upload)
$inputNames = ['verification_document', 'document', 'file', 'upload'];
$file = null;
$nameUsed = null;
foreach ($inputNames as $n) {
    if (!empty($_FILES[$n]) && is_uploaded_file($_FILES[$n]['tmp_name'])) {
        $file = $_FILES[$n];
        $nameUsed = $n;
        break;
    }
}
if (!$file) {
    logToFile("ERROR: Kein Upload-Feld gefunden", [
        '_FILES keys' => array_keys($_FILES),
        '_FILES vollständig' => $_FILES,
        'Content-Type' => $_SERVER['CONTENT_TYPE'] ?? 'NOT SET',
        'Request Method' => $_SERVER['REQUEST_METHOD'] ?? 'NOT SET'
    ]);
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Kein Upload gefunden (erwarte Feld: verification_document|document|file|upload)']);
    exit;
}

// 3) Validierung (Typ/Größe)
$allowedMime = ['image/jpeg', 'image/png', 'application/pdf'];
$maxSize = 10 * 1024 * 1024; // 10MB
$mime = mime_content_type($file['tmp_name']) ?: $file['type'];
if (!in_array($mime, $allowedMime, true)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Ungültiger Dateityp (erlaubt: JPG, PNG, PDF)']);
    exit;
}
if ($file['size'] > $maxSize) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Datei zu groß (max. 10MB)']);
    exit;
}

// 4) Zielpfad vorbereiten
$uploadDir = __DIR__ . '/../uploads/verification_documents';
if (!is_dir($uploadDir)) { @mkdir($uploadDir, 0775, true); }
if (!is_writable($uploadDir)) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Upload-Verzeichnis nicht beschreibbar']);
    exit;
}

// 5) Sicherer Dateiname
$ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
if ($mime === 'application/pdf' && $ext !== 'pdf') $ext = 'pdf';
if ($mime === 'image/jpeg'  && !in_array($ext, ['jpg','jpeg'], true)) $ext = 'jpg';
if ($mime === 'image/png'   && $ext !== 'png') $ext = 'png';

$baseName = 'verif_' . $userId . '_' . date('YmdHis') . '_' . bin2hex(random_bytes(4));
$destFile = $baseName . '.' . $ext;
$destPath = $uploadDir . '/' . $destFile;

// 6) Move & in DB speichern
if (!move_uploaded_file($file['tmp_name'], $destPath)) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Upload fehlgeschlagen (move_uploaded_file)']);
    exit;
}

// 7) Pfad in Datenbank speichern (users Tabelle)
try {
    // Lade config.php falls noch nicht geladen
    if (!isset($GLOBALS['pdo'])) {
        $configPath = __DIR__ . '/config.php';
        if (file_exists($configPath)) {
            require_once $configPath;
        }
    }
    
    global $pdo;
    
    if (!isset($pdo) || !($pdo instanceof PDO)) {
        $pdo = getDbConnection();
    }
    
    if ($pdo && $pdo instanceof PDO) {
        // Relativer Pfad (ohne führenden Slash) - wie in Datenbank: 'uploads/verification_documents/user_XX_XXX.png'
        $relativePath = 'uploads/verification_documents/' . $destFile;
        
        // Bestimme Dateityp
        $docType = 'image';
        if ($mime === 'application/pdf') {
            $docType = 'pdf';
        } elseif (in_array($mime, ['image/jpeg', 'image/png'], true)) {
            $docType = 'image';
        }
        
        // UPDATE: Setze verification_document und verification_document_type
        // verification_status bleibt 'pending' (oder wird von Admin geändert)
        $stmt = $pdo->prepare("
            UPDATE users 
            SET verification_document = ?, 
                verification_document_type = ?,
                updated_at = NOW()
            WHERE id = ?
        ");
        
        $result = $stmt->execute([$relativePath, $docType, $userId]);
        
        if ($result) {
            logToFile("SUCCESS: Pfad in Datenbank gespeichert", [
                'user_id' => $userId,
                'verification_document' => $relativePath,
                'verification_document_type' => $docType
            ]);
        } else {
            logToFile("WARNING: Upload erfolgreich, aber DB-Update fehlgeschlagen", "User ID: $userId");
        }
    } else {
        logToFile("WARNING: Upload erfolgreich, aber keine DB-Verbindung für Update", "User ID: $userId");
    }
} catch (Exception $e) {
    logToFile("ERROR: Exception beim DB-Update", $e->getMessage());
    // Upload war erfolgreich, nur DB-Update fehlgeschlagen
}

// 8) Antwort
logToFile("SUCCESS: Upload erfolgreich", [
    'user_id' => $userId,
    'filename' => $destFile,
    'file_path' => 'uploads/verification_documents/' . $destFile
]);
logToFile("===== REQUEST END (SUCCESS) =====");

echo json_encode([
    'success'   => true,
    'file_path' => 'uploads/verification_documents/' . $destFile,
    'message'   => 'Dokument gespeichert und zur Prüfung eingereicht'
]);
