<?php
/**
 * Intelligenter Parser für Reifensätze
 * 
 * Features:
 * - Multiple Parsing-Strategien
 * - Automatische Format-Erkennung
 * - Robuste Fehlerbehandlung
 * - Performance-Optimierung
 */

class SmartParser {
    private $parsingStrategies = [];
    private $formatDetector;
    
    public function __construct() {
        $this->initializeStrategies();
        $this->formatDetector = new FormatDetector();
    }
    
    /**
     * Initialisiert verschiedene Parsing-Strategien
     */
    private function initializeStrategies() {
        $this->parsingStrategies = [
            'lgs_id' => new LgsIdParser(),
            'table_based' => new TableBasedParser(),
            'div_based' => new DivBasedParser(),
            'json_based' => new JsonBasedParser(),
            'fallback' => new FallbackParser()
        ];
    }
    
    /**
     * Hauptmethode zum Parsen von Reifensätzen
     */
    public function parseWheelsets($htmlData) {
        if (empty($htmlData)) {
            return [];
        }
        
        // 1. Erkenne HTML-Format
        $format = $this->formatDetector->detectFormat($htmlData);
        error_log("Erkanntes Format: " . $format);
        
        // 2. Wähle optimale Parsing-Strategie
        $strategy = $this->selectStrategy($format);
        
        // 3. Parse mit gewählter Strategie
        $wheelsets = $strategy->parse($htmlData);
        
        // 4. Validiere und bereinige Ergebnisse
        return $this->validateResults($wheelsets);
    }
    
    /**
     * Fallback-Parsing mit mehreren Strategien
     */
    public function parseWithFallback($htmlData) {
        $allResults = [];
        
        foreach ($this->parsingStrategies as $name => $strategy) {
            try {
                $results = $strategy->parse($htmlData);
                if (!empty($results)) {
                    $allResults = array_merge($allResults, $results);
                    error_log("Strategie '$name' fand " . count($results) . " Reifensätze");
                }
            } catch (Exception $e) {
                error_log("Strategie '$name' fehlgeschlagen: " . $e->getMessage());
            }
        }
        
        // Entferne Duplikate
        return $this->removeDuplicates($allResults);
    }
    
    /**
     * Wählt die optimale Parsing-Strategie
     */
    private function selectStrategy($format) {
        $strategyMap = [
            'lgs_id' => 'lgs_id',
            'table' => 'table_based',
            'div' => 'div_based',
            'json' => 'json_based',
            'unknown' => 'fallback'
        ];
        
        $strategyName = $strategyMap[$format] ?? 'fallback';
        return $this->parsingStrategies[$strategyName];
    }
    
    /**
     * Validiert und bereinigt Parsing-Ergebnisse
     */
    private function validateResults($wheelsets) {
        $validated = [];
        
        foreach ($wheelsets as $wheelset) {
            if ($this->isValidWheelset($wheelset)) {
                $validated[] = $this->cleanWheelset($wheelset);
            }
        }
        
        return $validated;
    }
    
    /**
     * Prüft ob ein Reifensatz gültig ist
     */
    private function isValidWheelset($wheelset) {
        return !empty($wheelset['satznummer']) && 
               !empty($wheelset['kennzeichen']) &&
               $wheelset['satznummer'] !== '&nbsp;' &&
               strpos($wheelset['satznummer'], 'Anz=') === false;
    }
    
    /**
     * Bereinigt Reifensatz-Daten
     */
    private function cleanWheelset($wheelset) {
        return [
            'satznummer' => trim(strip_tags($wheelset['satznummer'] ?? '')),
            'kennzeichen' => trim(strip_tags($wheelset['kennzeichen'] ?? '')),
            'fahrzeug' => trim(strip_tags($wheelset['fahrzeug'] ?? '')),
            'halter' => trim(strip_tags($wheelset['halter'] ?? 'Unbekannt')),
            'eingangsdatum' => trim(strip_tags($wheelset['eingangsdatum'] ?? '')),
            'status' => 'eingelagert',
            'prufdoc_id' => $this->extractPrufdocId($wheelset)
        ];
    }
    
    /**
     * Entfernt Duplikate basierend auf Satznummer und Kennzeichen
     */
    private function removeDuplicates($wheelsets) {
        $seen = [];
        $unique = [];
        
        foreach ($wheelsets as $wheelset) {
            $key = $wheelset['satznummer'] . '_' . $wheelset['kennzeichen'];
            if (!isset($seen[$key])) {
                $seen[$key] = true;
                $unique[] = $wheelset;
            }
        }
        
        return $unique;
    }
    
    /**
     * Extrahiert PRUFDOC_ID für Prüfbericht-Download
     */
    private function extractPrufdocId($wheelset) {
        // Implementierung für PRUFDOC_ID Extraktion
        return $wheelset['prufdoc_id'] ?? null;
    }
}

/**
 * Format-Detektor für HTML-Inhalte
 */
class FormatDetector {
    public function detectFormat($html) {
        // Prüfe auf LGS_ID-Format
        if (strpos($html, 'LGS_ID') !== false) {
            return 'lgs_id';
        }
        
        // Prüfe auf Tabellen-Format
        if (preg_match('/<table[^>]*>.*?<\/table>/s', $html)) {
            return 'table';
        }
        
        // Prüfe auf Div-Format
        if (preg_match('/<div[^>]*class="[^"]*table[^"]*"[^>]*>/', $html)) {
            return 'div';
        }
        
        // Prüfe auf JSON-Format
        if (strpos($html, '{') !== false && strpos($html, '}') !== false) {
            return 'json';
        }
        
        return 'unknown';
    }
}

/**
 * LGS_ID-basierter Parser (RESY-spezifisch)
 */
class LgsIdParser {
    public function parse($html) {
        $wheelsets = [];
        
        // Suche nach LGS_ID-Einträgen
        if (preg_match_all('/LGS_ID[^>]*>([^<]+)</', $html, $matches)) {
            $lgsData = $matches[1];
            
            // Gruppiere in Blöcke von 5 (jeder Reifensatz hat 5 LGS_ID-Einträge)
            $chunks = array_chunk($lgsData, 5);
            
            foreach ($chunks as $index => $chunk) {
                if ($index === 0) continue; // Überspringe Header
                
                if (count($chunk) >= 5) {
                    $wheelset = [
                        'satznummer' => trim($chunk[0]),
                        'kennzeichen' => trim($chunk[1]),
                        'fahrzeug' => trim($chunk[2]),
                        'halter' => $this->findCustomerName($html, trim($chunk[0])),
                        'eingangsdatum' => trim($chunk[4]),
                        'status' => 'eingelagert'
                    ];
                    
                    if (!empty($wheelset['satznummer'])) {
                        $wheelsets[] = $wheelset;
                    }
                }
            }
        }
        
        return $wheelsets;
    }
    
    private function findCustomerName($html, $satznummer) {
        // Vereinfachte Kundenname-Suche
        $patterns = [
            '/[^>]*' . preg_quote($satznummer, '/') . '[^<]*<[^>]*>([^<]*(?:GmbH|AG|UG|eG|Dr\.|Prof\.|Herr|Frau|von|zu)[^<]*)</i',
            '/<div[^>]*>([^<]*(?:GmbH|AG|UG|eG|Dr\.|Prof\.|Herr|Frau|von|zu)[^<]*)</i'
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $html, $matches)) {
                $candidate = trim(strip_tags($matches[1]));
                if (strlen($candidate) > 3 && strlen($candidate) < 100) {
                    return $candidate;
                }
            }
        }
        
        return 'Unbekannt';
    }
}

/**
 * Tabellen-basierter Parser
 */
class TableBasedParser {
    public function parse($html) {
        $wheelsets = [];
        
        // Suche nach Tabellen
        if (preg_match_all('/<table[^>]*>(.*?)<\/table>/s', $html, $tables)) {
            foreach ($tables[1] as $tableHtml) {
                if (preg_match_all('/<tr[^>]*>(.*?)<\/tr>/s', $tableHtml, $rows)) {
                    foreach ($rows[1] as $index => $row) {
                        if ($index === 0) continue; // Überspringe Header
                        
                        $wheelset = $this->parseTableRow($row);
                        if ($wheelset) {
                            $wheelsets[] = $wheelset;
                        }
                    }
                }
            }
        }
        
        return $wheelsets;
    }
    
    private function parseTableRow($row) {
        if (preg_match_all('/<td[^>]*>(.*?)<\/td>/s', $row, $cells)) {
            $cells = $cells[1];
            
            if (count($cells) >= 5) {
                return [
                    'satznummer' => trim(strip_tags($cells[0])),
                    'kennzeichen' => trim(strip_tags($cells[1])),
                    'fahrzeug' => trim(strip_tags($cells[2])),
                    'halter' => trim(strip_tags($cells[3])),
                    'eingangsdatum' => trim(strip_tags($cells[4])),
                    'status' => 'eingelagert'
                ];
            }
        }
        
        return null;
    }
}

/**
 * Div-basierter Parser
 */
class DivBasedParser {
    public function parse($html) {
        $wheelsets = [];
        
        // Suche nach div-basierten Strukturen
        if (preg_match_all('/<div[^>]*class="[^"]*table[^"]*"[^>]*>(.*?)<\/div>/s', $html, $divs)) {
            foreach ($divs[1] as $divHtml) {
                $wheelset = $this->parseDiv($divHtml);
                if ($wheelset) {
                    $wheelsets[] = $wheelset;
                }
            }
        }
        
        return $wheelsets;
    }
    
    private function parseDiv($divHtml) {
        // Vereinfachte Div-Parsing-Logik
        $wheelset = [
            'satznummer' => '',
            'kennzeichen' => '',
            'fahrzeug' => '',
            'halter' => '',
            'eingangsdatum' => '',
            'status' => 'eingelagert'
        ];
        
        // Suche nach verschiedenen Patterns
        if (preg_match('/LGS_ID[^>]*>([^<]+)</', $divHtml, $matches)) {
            $wheelset['satznummer'] = trim($matches[1]);
        }
        
        if (preg_match('/Kennzeichen[^>]*>([^<]+)</', $divHtml, $matches)) {
            $wheelset['kennzeichen'] = trim($matches[1]);
        }
        
        return !empty($wheelset['satznummer']) ? $wheelset : null;
    }
}

/**
 * JSON-basierter Parser
 */
class JsonBasedParser {
    public function parse($html) {
        $wheelsets = [];
        
        // Versuche JSON zu extrahieren
        if (preg_match('/\{.*\}/s', $html, $matches)) {
            $json = json_decode($matches[0], true);
            if ($json && isset($json['wheelsets'])) {
                return $json['wheelsets'];
            }
        }
        
        return $wheelsets;
    }
}

/**
 * Fallback-Parser
 */
class FallbackParser {
    public function parse($html) {
        $wheelsets = [];
        
        // Suche nach allen möglichen Reifensatz-Indikatoren
        $patterns = [
            '/Satznummer[^>]*>([^<]+)</',
            '/LGS[^>]*>([^<]+)</',
            '/<span[^>]*>([0-9]+\.[0-9]+\.[0-9]+)<\/span>/'
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match_all($pattern, $html, $matches)) {
                foreach ($matches[1] as $satznummer) {
                    $satznummer = trim(strip_tags($satznummer));
                    if (!empty($satznummer) && $satznummer !== '&nbsp;') {
                        $wheelsets[] = [
                            'satznummer' => $satznummer,
                            'kennzeichen' => '',
                            'fahrzeug' => '',
                            'halter' => 'Unbekannt',
                            'eingangsdatum' => '',
                            'status' => 'eingelagert'
                        ];
                    }
                }
            }
        }
        
        return $wheelsets;
    }
}
?>

