<?php
/**
 * Teams-Benachrichtigung für Wheelset-Angebote
 */

function sendWheelsetOfferTeamsNotification($offerId, $status) {
    require_once dirname(__DIR__) . '/config/database.php';
    
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        // Lade Angebot mit Autohaus-Daten
        $stmt = $conn->prepare("
            SELECT wo.*, 
                   ah.name as autohaus_name,
                   ah.teams_webhook_url,
                   ah.teams_channel_name,
                   ah.teams_notifications_enabled,
                   ah.teams_mention_users
            FROM wheelset_offers wo
            JOIN autohaus ah ON wo.autohaus_id = ah.id
            WHERE wo.id = ?
        ");
        $stmt->execute([$offerId]);
        $offer = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$offer) {
            return ['success' => false, 'error' => 'Angebot nicht gefunden'];
        }
        
        // Prüfe ob Teams-Benachrichtigungen aktiviert sind
        if (!$offer['teams_notifications_enabled'] || empty($offer['teams_webhook_url'])) {
            return ['success' => false, 'error' => 'Teams-Benachrichtigungen sind nicht konfiguriert'];
        }
        
        // Parse Daten
        $offerData = json_decode($offer['offer_data'], true) ?? [];
        $customerSelection = null;
        if ($offer['customer_selection']) {
            if (is_string($offer['customer_selection'])) {
                $customerSelection = json_decode($offer['customer_selection'], true);
            } else {
                $customerSelection = $offer['customer_selection'];
            }
        }
        
        // Erstelle Teams-Nachricht basierend auf Status
        $teamsMessage = createWheelsetOfferTeamsMessage($offer, $offerData, $customerSelection, $status, $offer['teams_channel_name']);
        
        // Sende an Teams
        $result = sendToTeams($offer['teams_webhook_url'], $teamsMessage);
        
        if ($result['success']) {
            error_log("=== WHEELSET OFFER TEAMS NOTIFICATION SUCCESS ===");
            error_log("Offer ID: $offerId");
            error_log("Status: $status");
            error_log("Autohaus: " . $offer['autohaus_name']);
            
            return ['success' => true, 'message' => 'Teams-Benachrichtigung gesendet'];
        } else {
            error_log("=== WHEELSET OFFER TEAMS NOTIFICATION FAILED ===");
            error_log("Offer ID: $offerId");
            error_log("Error: " . $result['error']);
            
            return ['success' => false, 'error' => $result['error']];
        }
        
    } catch (Exception $e) {
        error_log("=== WHEELSET OFFER TEAMS NOTIFICATION EXCEPTION ===");
        error_log("Offer ID: $offerId");
        error_log("Exception: " . $e->getMessage());
        
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

function createWheelsetOfferTeamsMessage($offer, $offerData, $customerSelection, $status, $channelName) {
    $baseUrl = getBaseUrl();
    $adminUrl = $baseUrl . "/admin/storage.php";
    $offerUrl = $baseUrl . "/admin/storage.php?wheelset=" . urlencode($offer['wheelset_number']) . "&autohaus_id=" . $offer['autohaus_id'] . "&offer_id=" . $offer['id'];
    
    // Verarbeite Benutzer-Markierungen
    $mentionText = '';
    if (!empty($offer['teams_mention_users'])) {
        $mentions = array_map('trim', explode(',', $offer['teams_mention_users']));
        $mentionText = implode(' ', $mentions) . "\n\n";
    }
    
    $newTires = $offerData['new_tires'] ?? [];
    $selectedServices = $offerData['selected_services'] ?? [];
    
    // Erstelle Reifen-Liste
    $tiresText = '';
    if (!empty($newTires)) {
        foreach ($newTires as $tire) {
            $brand = htmlspecialchars($tire['brand_model'] ?? 'Unbekannt');
            $dimension = htmlspecialchars($tire['dimension'] ?? '');
            $price = number_format($tire['total'] ?? 0, 2, ',', '.');
            $quantity = $tire['quantity'] ?? 1;
            $tiresText .= "• **{$brand}** ({$dimension}) - {$quantity}x - {$price}€\n";
        }
    }
    
    // Erstelle Services-Liste
    $servicesText = '';
    if ($customerSelection && isset($customerSelection['services']) && !empty($customerSelection['services']) && is_array($customerSelection['services'])) {
        // Hole Service-Details
        require_once dirname(__DIR__) . '/config/database.php';
        $db = new Database();
        $conn = $db->getConnection();
        
        $placeholders = str_repeat('?,', count($customerSelection['services']) - 1) . '?';
        $stmt = $conn->prepare("
            SELECT service_name, price, show_price 
            FROM autohaus_services 
            WHERE id IN ($placeholders)
        ");
        $stmt->execute($customerSelection['services']);
        $services = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($services as $service) {
            $price = $service['show_price'] && $service['price'] ? number_format($service['price'], 2, ',', '.') . "€" : "auf Anfrage";
            $servicesText .= "• **{$service['service_name']}** - {$price}\n";
        }
    }
    
    // Status-spezifische Nachrichten
    $declineReason = isset($customerSelection['decline_reason']) && !empty($customerSelection['decline_reason']) 
        ? htmlspecialchars($customerSelection['decline_reason']) 
        : '';
    
    $requestNotes = isset($customerSelection['request_notes']) && !empty($customerSelection['request_notes']) 
        ? htmlspecialchars($customerSelection['request_notes']) 
        : '';
    
    $statusConfig = [
        'accepted' => [
            'title' => '🎉 Angebot angenommen!',
            'subtitle' => '**' . htmlspecialchars($offer['customer_name'] ?? 'Kunde') . '** hat das Reifenangebot angenommen',
            'color' => '28a745',
            'icon' => 'https://img.icons8.com/color/48/000000/checked--v1.png',
            'text' => ''
        ],
        'declined' => [
            'title' => '❌ Angebot abgelehnt',
            'subtitle' => '**' . htmlspecialchars($offer['customer_name'] ?? 'Kunde') . '** hat das Reifenangebot abgelehnt',
            'color' => 'dc3545',
            'icon' => 'https://img.icons8.com/color/48/000000/cancel.png',
            'text' => $declineReason ? "\n**Grund:** " . $declineReason : ''
        ],
        'new_request' => [
            'title' => '🔄 Neues Angebot angefragt',
            'subtitle' => '**' . htmlspecialchars($offer['customer_name'] ?? 'Kunde') . '** möchte ein neues Angebot',
            'color' => '17a2b8',
            'icon' => 'https://img.icons8.com/color/48/000000/refresh.png',
            'text' => $requestNotes ? "\n**Wünsche:** " . $requestNotes : ''
        ],
        'viewed' => [
            'title' => '👁️ Angebot angesehen',
            'subtitle' => '**' . htmlspecialchars($offer['customer_name'] ?? 'Kunde') . '** hat das Angebot angesehen',
            'color' => 'ffc107',
            'icon' => 'https://img.icons8.com/color/48/000000/visible.png',
            'text' => ''
        ]
    ];
    
    $config = $statusConfig[$status] ?? $statusConfig['viewed'];
    
    $message = [
        "@type" => "MessageCard",
        "@context" => "http://schema.org/extensions",
        "themeColor" => $config['color'],
        "summary" => $config['title'] . " - " . htmlspecialchars($offer['license_plate'] ?? ''),
        "sections" => [
            [
                "activityTitle" => $config['title'],
                "activitySubtitle" => $mentionText . $config['subtitle'],
                "activityImage" => $config['icon'],
                "facts" => [
                    [
                        "name" => "🚗 Kennzeichen",
                        "value" => "**" . htmlspecialchars($offer['license_plate'] ?? 'Nicht angegeben') . "**"
                    ],
                    [
                        "name" => "🔢 Satznummer",
                        "value" => "**" . htmlspecialchars($offer['wheelset_number'] ?? '') . "**"
                    ],
                    [
                        "name" => "👤 Kunde",
                        "value" => "**" . htmlspecialchars($offer['customer_name'] ?? 'Nicht angegeben') . "**"
                    ],
                    [
                        "name" => "🏢 Autohaus",
                        "value" => "**" . htmlspecialchars($offer['autohaus_name']) . "**"
                    ],
                    [
                        "name" => "📅 Erstellt",
                        "value" => date('d.m.Y H:i', strtotime($offer['created_at']))
                    ]
                ],
                "markdown" => true
            ]
        ],
        "potentialAction" => [
            [
                "@type" => "OpenUri",
                "name" => "📋 Angebot öffnen",
                "targets" => [
                    [
                        "os" => "default",
                        "uri" => $offerUrl
                    ]
                ]
            ],
            [
                "@type" => "OpenUri",
                "name" => "✅ Als bearbeitet markieren",
                "targets" => [
                    [
                        "os" => "default",
                        "uri" => $baseUrl . "/api/wheelset_offer_status_update.php?offer_id=" . $offer['id'] . "&action=mark_processed&status=processed"
                    ]
                ]
            ],
            [
                "@type" => "OpenUri",
                "name" => "📧 Email erneut senden",
                "targets" => [
                    [
                        "os" => "default",
                        "uri" => $baseUrl . "/api/wheelset_offer_status_update.php?offer_id=" . $offer['id'] . "&action=resend_email"
                    ]
                ]
            ],
            [
                "@type" => "OpenUri",
                "name" => "📊 Alle Angebote anzeigen",
                "targets" => [
                    [
                        "os" => "default",
                        "uri" => $adminUrl . "#offers-notifications"
                    ]
                ]
            ]
        ]
    ];
    
    // Füge Kundenauswahl hinzu wenn vorhanden
    if ($customerSelection && (isset($customerSelection['tires']) || isset($customerSelection['services']))) {
        $selectionText = "\n\n**Kundenauswahl:**\n";
        
        if (!empty($tiresText)) {
            $selectionText .= "\n**Reifen:**\n" . $tiresText;
        }
        
        if (!empty($servicesText)) {
            $selectionText .= "\n**Zusatzleistungen:**\n" . $servicesText;
        }
        
        if (isset($customerSelection['notes']) && !empty($customerSelection['notes'])) {
            $selectionText .= "\n**Anmerkungen:** " . htmlspecialchars($customerSelection['notes']);
        }
        
        $message["sections"][0]["text"] = $config['text'] . $selectionText;
    } else if (!empty($config['text'])) {
        $message["sections"][0]["text"] = $config['text'];
    }
    
    // Füge zusätzliche Sections hinzu für bessere Übersicht
    if (!empty($tiresText)) {
        $message["sections"][] = [
            "title" => "🛞 **Ausgewählte Reifen**",
            "text" => $tiresText,
            "markdown" => true
        ];
    }
    
    if (!empty($servicesText)) {
        $message["sections"][] = [
            "title" => "⚙️ **Zusatzleistungen**",
            "text" => $servicesText,
            "markdown" => true
        ];
    }
    
    // Füge Gesamtbetrag hinzu wenn vorhanden
    if (!empty($offer['total_amount'])) {
        $message["sections"][] = [
            "title" => "💰 **Gesamtbetrag**",
            "text" => "**" . number_format($offer['total_amount'], 2, ',', '.') . " €**",
            "markdown" => true
        ];
    }
    
    // Füge Kundenanmerkungen hinzu wenn vorhanden
    if ($customerSelection && isset($customerSelection['notes']) && !empty($customerSelection['notes'])) {
        $message["sections"][] = [
            "title" => "💬 **Kundenanmerkungen**",
            "text" => "> " . htmlspecialchars($customerSelection['notes']),
            "markdown" => true
        ];
    }
    
    // Füge Ablehnungsgrund hinzu wenn vorhanden
    if ($status === 'declined' && !empty($declineReason)) {
        $message["sections"][] = [
            "title" => "❌ **Ablehnungsgrund**",
            "text" => "> " . $declineReason,
            "markdown" => true
        ];
    }
    
    // Füge Anfrage-Wünsche hinzu wenn vorhanden
    if ($status === 'new_request' && !empty($requestNotes)) {
        $message["sections"][] = [
            "title" => "🔄 **Anfrage-Wünsche**",
            "text" => "> " . $requestNotes,
            "markdown" => true
        ];
    }
    
    // Füge Zusammenfassung hinzu
    $totalItems = 0;
    if ($customerSelection) {
        $totalItems = (isset($customerSelection['tires']) ? count($customerSelection['tires']) : 0) + 
                      (isset($customerSelection['services']) ? count($customerSelection['services']) : 0);
    }
    
    $summaryText = "📊 **Status:** " . $statusConfig[$status]['title'] . "\n";
    if ($totalItems > 0) {
        $summaryText .= "✅ **" . $totalItems . "** Artikel ausgewählt\n";
    }
    $summaryText .= "📋 Angebot #" . $offer['id'] . "\n";
    $summaryText .= "📅 Erstellt: " . date('d.m.Y H:i', strtotime($offer['created_at'])) . "\n";
    if (!empty($offer['updated_at']) && $offer['updated_at'] !== $offer['created_at']) {
        $summaryText .= "🔄 Aktualisiert: " . date('d.m.Y H:i', strtotime($offer['updated_at']));
    }
    
    $message["sections"][] = [
        "title" => "📊 **Zusammenfassung**",
        "text" => $summaryText,
        "markdown" => true
    ];
    
    // Füge Status-Update-Hinweis hinzu
    $message["sections"][] = [
        "title" => "🔧 **Status-Updates**",
        "text" => "Verwenden Sie die Buttons unten, um Aktionen auszuführen:\n\n" .
                  "📋 **Angebot öffnen** - Öffnet das Angebot im Admin-Bereich\n" .
                  "✅ **Als bearbeitet markieren** - Markiert das Angebot als bearbeitet\n" .
                  "📧 **Email erneut senden** - Sendet die Angebots-Email erneut",
        "markdown" => true
    ];
    
    return $message;
}

function sendToTeams($webhookUrl, $message) {
    $ch = curl_init();
    
    curl_setopt($ch, CURLOPT_URL, $webhookUrl);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($message));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json'
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        return ['success' => false, 'error' => $error];
    }
    
    if ($httpCode >= 200 && $httpCode < 300) {
        return ['success' => true, 'response' => $response];
    } else {
        return ['success' => false, 'error' => "HTTP $httpCode: $response"];
    }
}

function getBaseUrl() {
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    
    if (strpos($host, ':') !== false) {
        return $protocol . '://' . $host;
    }
    
    $port = $_SERVER['SERVER_PORT'] ?? '';
    $port = ($port && $port != '80' && $port != '443') ? ':' . $port : '';
    return $protocol . '://' . $host . $port;
}

