<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../config/database.php';

try {
    $database = new Database();
    $conn = $database->getConnection();
    
    if (!$conn) {
        throw new Exception("Keine Datenbankverbindung möglich");
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['appointmentId'])) {
        throw new Exception('Appointment ID ist erforderlich');
    }
    
    $appointmentId = $input['appointmentId'];
    
    // Hole Autohaus-Daten mit Teams-Einstellungen
    $stmt = $conn->prepare("
        SELECT 
            a.*,
            ah.name as autohaus_name,
            ah.teams_webhook_url,
            ah.teams_channel_name,
            ah.teams_notifications_enabled,
            ah.teams_mention_users
        FROM appointments a
        JOIN autohaus ah ON a.autohaus_id = ah.id
        WHERE a.id = ?
    ");
    
    $stmt->execute([$appointmentId]);
    $appointment = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$appointment) {
        throw new Exception('Termin nicht gefunden');
    }
    
    // Prüfe ob Teams-Benachrichtigungen aktiviert sind
    if (!$appointment['teams_notifications_enabled'] || empty($appointment['teams_webhook_url'])) {
        throw new Exception('Teams-Benachrichtigungen sind nicht konfiguriert');
    }
    
    // Hole Kundenauswahl
    $stmt = $conn->prepare("
        SELECT * FROM customer_selections 
        WHERE appointment_id = ? 
        ORDER BY created_at DESC 
        LIMIT 1
    ");
    $stmt->execute([$appointmentId]);
    $customerSelection = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$customerSelection) {
        throw new Exception('Keine Kundenauswahl gefunden');
    }
    
    // Hole Angebot-Details
    $stmt = $conn->prepare("
        SELECT * FROM offer_tracking 
        WHERE appointment_id = ? AND offer_type = 'sent'
        ORDER BY created_at DESC 
        LIMIT 1
    ");
    $stmt->execute([$appointmentId]);
    $offer = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$offer) {
        throw new Exception('Kein Angebot gefunden');
    }
    
    $offerData = json_decode($offer['offer_data'], true);
    
    // Parse Kundenauswahl
    $tireSelection = json_decode($customerSelection['tire_selection'] ?? '[]', true);
    $serviceSelection = json_decode($customerSelection['service_selection'] ?? '[]', true);
    $customerNotes = $customerSelection['customer_notes'] ?? '';
    
    // Hole Services-Details
    $services = [];
    if (!empty($serviceSelection)) {
        $placeholders = str_repeat('?,', count($serviceSelection) - 1) . '?';
        $stmt = $conn->prepare("
            SELECT service_name, price, show_price 
            FROM autohaus_services 
            WHERE id IN ($placeholders)
        ");
        $stmt->execute($serviceSelection);
        $services = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Erstelle Teams-Nachricht
    $teamsMessage = createTeamsMessage($appointment, $offerData, $tireSelection, $services, $customerNotes, $appointment['teams_channel_name']);
    
    // Sende an Teams
    $result = sendToTeams($appointment['teams_webhook_url'], $teamsMessage);
    
    if ($result['success']) {
        // Log erfolgreiche Nachricht
        error_log("Teams-Benachrichtigung erfolgreich gesendet für Appointment $appointmentId. HTTP Code: " . ($result['http_code'] ?? 'unknown'));
        
        echo json_encode([
            'success' => true,
            'message' => 'Teams-Benachrichtigung erfolgreich gesendet',
            'channel' => $appointment['teams_channel_name'],
            'http_code' => $result['http_code'] ?? null
        ]);
    } else {
        // Log Fehler für Debugging
        error_log("Teams-Benachrichtigung fehlgeschlagen für Appointment $appointmentId: " . $result['error']);
        
        throw new Exception('Teams-Fehler: ' . $result['error']);
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}

function createTeamsMessage($appointment, $offerData, $tireSelection, $services, $customerNotes, $channelName) {
    $appointmentDate = date('d.m.Y', strtotime($appointment['appointment_date'] ?? date('Y-m-d')));
    $appointmentTime = date('H:i', strtotime($appointment['appointment_time'] ?? '12:00:00'));
    
    // Verarbeite Benutzer-Markierungen
    $mentionText = '';
    if (!empty($appointment['teams_mention_users'])) {
        $mentions = array_map('trim', explode(',', $appointment['teams_mention_users']));
        $mentionText = implode(' ', $mentions) . "\n\n";
    }
    
    // Erstelle Reifen-Liste
    $tiresText = '';
    if (!empty($offerData['selectedTires'])) {
        foreach ($offerData['selectedTires'] as $tire) {
            $brand = $tire['brand'] ?? 'Unbekannt';
            $model = $tire['model'] ?? 'Unbekannt';
            $size = $tire['size'] ?? 'Unbekannt';
            $recommendation = $tire['recommendation'] ?? 'Unbekannt';
            $tiresText .= "• **{$brand} {$model}** ({$size}) - {$recommendation}\n";
        }
    }
    
    if (!empty($offerData['newTires'])) {
        foreach ($offerData['newTires'] as $tire) {
            $brand = $tire['brand'] ?? 'Unbekannt';
            $model = $tire['model'] ?? 'Unbekannt';
            $size = $tire['size'] ?? 'Unbekannt';
            $price = $tire['price'] ?? '0';
            $tiresText .= "• **NEU: {$brand} {$model}** ({$size}) - {$price}€\n";
        }
    }
    
    // Erstelle Services-Liste
    $servicesText = '';
    if (!empty($services)) {
        foreach ($services as $service) {
            $price = $service['show_price'] && $service['price'] ? " - {$service['price']}€" : " - auf Anfrage";
            $servicesText .= "• **{$service['service_name']}**{$price}\n";
        }
    }
    
    // Erstelle Admin-Link
    $adminUrl = getBaseUrl() . "/admin/appointments.php";
    
    $message = [
        "@type" => "MessageCard",
        "@context" => "http://schema.org/extensions",
        "themeColor" => "28a745",
        "summary" => "🎉 Angebot angenommen von " . htmlspecialchars($appointment['customer_name']),
        "sections" => [
            [
                "activityTitle" => "🎉 Angebot erfolgreich angenommen!",
                "activitySubtitle" => $mentionText . "**" . htmlspecialchars($appointment['customer_name']) . "** hat das Reifenangebot angenommen",
                "activityImage" => "https://img.icons8.com/color/48/000000/checked--v1.png",
                "facts" => [
                    [
                        "name" => "📅 Termin",
                        "value" => "**" . $appointmentDate . "** um **" . $appointmentTime . " Uhr**"
                    ],
                    [
                        "name" => "🚗 Kennzeichen",
                        "value" => "**" . htmlspecialchars($appointment['license_plate'] ?? 'Nicht angegeben') . "**"
                    ],
                    [
                        "name" => "🏢 Autohaus",
                        "value" => "**" . htmlspecialchars($appointment['autohaus_name']) . "**"
                    ],
                    [
                        "name" => "📞 Telefon",
                        "value" => htmlspecialchars($appointment['phone'] ?? 'Nicht angegeben')
                    ],
                    [
                        "name" => "📧 E-Mail",
                        "value" => htmlspecialchars($appointment['email'] ?? 'Nicht angegeben')
                    ]
                ],
                "markdown" => true
            ]
        ],
        "potentialAction" => [
            [
                "@type" => "OpenUri",
                "name" => "🔗 Zum Admin-Dashboard",
                "targets" => [
                    [
                        "os" => "default",
                        "uri" => $adminUrl
                    ]
                ]
            ],
            [
                "@type" => "OpenUri",
                "name" => "🛞 Neue Reifen bestellt",
                "targets" => [
                    [
                        "os" => "default",
                        "uri" => getBaseUrl() . "/api/teams_reaction.php?appointment_id=" . $appointment['id'] . "&reaction_type=tire_ordered&user=Teams_User"
                    ]
                ]
            ],
            [
                "@type" => "OpenUri",
                "name" => "✅ Arbeit abgeschlossen",
                "targets" => [
                    [
                        "os" => "default",
                        "uri" => getBaseUrl() . "/api/teams_reaction.php?appointment_id=" . $appointment['id'] . "&reaction_type=work_completed&user=Teams_User"
                    ]
                ]
            ],
            [
                "@type" => "OpenUri",
                "name" => "📧 Kunde benachrichtigt",
                "targets" => [
                    [
                        "os" => "default",
                        "uri" => getBaseUrl() . "/api/teams_reaction.php?appointment_id=" . $appointment['id'] . "&reaction_type=customer_notified&user=Teams_User"
                    ]
                ]
            ]
        ]
    ];
    
    // Füge Reifen-Sektion hinzu
    if ($tiresText) {
        $message["sections"][] = [
            "title" => "🛞 **Ausgewählte Reifen**",
            "text" => $tiresText,
            "markdown" => true
        ];
    }
    
    // Füge Services-Sektion hinzu
    if ($servicesText) {
        $message["sections"][] = [
            "title" => "⚙️ **Zusatzleistungen**",
            "text" => $servicesText,
            "markdown" => true
        ];
    }
    
    // Füge Kundenanmerkungen hinzu
    if ($customerNotes) {
        $message["sections"][] = [
            "title" => "💬 **Kundenanmerkungen**",
            "text" => "> " . htmlspecialchars($customerNotes),
            "markdown" => true
        ];
    }
    
    // Füge Zusammenfassung hinzu
    $totalItems = count($offerData['selectedTires'] ?? []) + count($offerData['newTires'] ?? []) + count($services ?? []);
    $message["sections"][] = [
        "title" => "📊 **Zusammenfassung**",
        "text" => "✅ **" . $totalItems . "** Artikel ausgewählt\n📋 Angebot vollständig bearbeitet\n🎯 Kunde hat sich entschieden",
        "markdown" => true
    ];
    
    // Füge Status-Updates Hinweis hinzu
    $message["sections"][] = [
        "title" => "🔧 **Status-Updates**",
        "text" => "Klicken Sie auf die Buttons unten, um den Status zu aktualisieren:",
        "markdown" => true
    ];
    
    
    return $message;
}

function sendToTeams($webhookUrl, $message, $maxRetries = 3) {
    $retryCount = 0;
    $lastError = '';
    
    while ($retryCount < $maxRetries) {
        $ch = curl_init();
        
        curl_setopt($ch, CURLOPT_URL, $webhookUrl);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($message));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json'
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            $lastError = "cURL Fehler: $error";
            $retryCount++;
            if ($retryCount < $maxRetries) {
                sleep(2 * $retryCount); // Exponential backoff
                continue;
            }
            return ['success' => false, 'error' => $lastError];
        }
        
        if ($httpCode >= 200 && $httpCode < 300) {
            return ['success' => true, 'response' => $response, 'http_code' => $httpCode];
        } else {
            $lastError = "HTTP $httpCode: $response";
            $retryCount++;
            if ($retryCount < $maxRetries) {
                sleep(2 * $retryCount); // Exponential backoff
                continue;
            }
            return ['success' => false, 'error' => $lastError];
        }
    }
    
    return ['success' => false, 'error' => "Maximale Wiederholungen erreicht. Letzter Fehler: $lastError"];
}

function getBaseUrl() {
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    
    if (strpos($host, ':') !== false) {
        return $protocol . '://' . $host;
    }
    
    $port = $_SERVER['SERVER_PORT'] ?? '';
    $port = ($port && $port != '80' && $port != '443') ? ':' . $port : '';
    return $protocol . '://' . $host . $port;
}
?>
