<?php
/**
 * RESY Selenium Driver (Einfache Version - OHNE Composer)
 * Nutzt cURL zur Selenium-Steuerung
 */

class ResySeleniumDriver {
    private $username;
    private $password;
    private $dealerId;
    private $sessionId = null;
    
    public function __construct($username, $password, $dealerId) {
        $this->username = $username;
        $this->password = $password;
        $this->dealerId = $dealerId;
    }
    
    /**
     * Erstellt eine neue Selenium-Session
     */
    private function createSession() {
        if ($this->sessionId) {
            echo "Session bereits vorhanden: $this->sessionId\n";
            return true;
        }
        
        echo "Erstelle neue Selenium-Session...\n";
        
        $ch = curl_init('http://localhost:4444/wd/hub/session');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json'
        ]);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
            'capabilities' => [
                'alwaysMatch' => [
                    'browserName' => 'chrome',
                    'goog:chromeOptions' => [
                        'args' => ['--no-sandbox', '--disable-dev-shm-usage']
                    ]
                ]
            ]
        ]));
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);
        
        echo "HTTP Code: $httpCode\n";
        
        if ($curlError) {
            echo "CURL Error: $curlError\n";
            return false;
        }
        
        if ($httpCode === 200) {
            $data = json_decode($response, true);
            if ($data && isset($data['value']['sessionId'])) {
                $this->sessionId = $data['value']['sessionId'];
                echo "✅ Session erstellt: $this->sessionId\n";
                return true;
            } else {
                echo "❌ Keine Session-ID in Response\n";
                echo "Response: " . substr($response, 0, 500) . "\n";
                return false;
            }
        } else {
            echo "❌ HTTP Fehler: $httpCode\n";
            echo "Response: " . substr($response, 0, 500) . "\n";
            return false;
        }
    }
    
    /**
     * Führt eine Selenium-Aktion aus
     */
    private function execute($action, $data = []) {
        if (!$this->sessionId) {
            if (!$this->createSession()) {
                return false;
            }
        }
        
        $ch = curl_init("http://localhost:4444/wd/hub/session/{$this->sessionId}/{$action}");
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        
        if (!empty($data)) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        $responseData = json_decode($response, true);
        
        if ($httpCode >= 200 && $httpCode < 300 && isset($responseData['value'])) {
            return $responseData['value'];
        }
        
        if ($httpCode === 404) {
            // Session abgelaufen - neu erstellen
            $this->sessionId = null;
            return $this->execute($action, $data);
        }
        
        error_log("❌ Selenium-Aktion '{$action}' fehlgeschlagen: " . $response);
        return false;
    }
    
    /**
     * Navigiert zu einer URL
     */
    private function navigate($url) {
        return $this->execute('url', ['url' => $url]);
    }
    
    /**
     * Findet ein Element
     */
    private function findElement($strategy, $selector) {
        // Konvertiere 'id' zu 'css selector'
        if ($strategy === 'id') {
            $strategy = 'css selector';
            $selector = "#$selector";
        } elseif ($strategy === 'name') {
            $strategy = 'css selector';
            $selector = "[name=\"$selector\"]";
        }
        
        $result = $this->execute('element', [
            'using' => $strategy,
            'value' => $selector
        ]);
        
        if ($result && isset($result['ELEMENT'])) {
            return $result['ELEMENT'];
        }
        
        return false;
    }
    
    /**
     * Klickt auf ein Element
     */
    private function clickElement($elementId) {
        return $this->execute("element/{$elementId}/click", []);
    }
    
    /**
     * Sendet Text an ein Element
     */
    private function sendKeys($elementId, $text) {
        return $this->execute("element/{$elementId}/value", [
            'text' => $text
        ]);
    }
    
    /**
     * Wartet bis ein Element sichtbar ist
     */
    private function waitForElement($strategy, $selector, $timeout = 10) {
        $startTime = time();
        
        while (time() - $startTime < $timeout) {
            $element = $this->findElement($strategy, $selector);
            if ($element) {
                return $element;
            }
            sleep(1); // 1 Sekunde
        }
        
        return false;
    }
    
    /**
     * Führt Login durch
     */
    public function login() {
        try {
            echo "1. Navigiere zur Login-Seite...\n";
            // 1. Navigiere zur Login-Seite
            $this->navigate('https://resy.dtm-reifen.de/index.php?m=login&a=login');
            echo "✅ Login-Seite geladen\n";
            
            // 2. Warte auf Username-Feld
            echo "2. Warte auf Username-Feld...\n";
            $usernameElementId = $this->waitForElement('id', 'InputUser', 10);
            if (!$usernameElementId) {
                throw new Exception("Username-Feld nicht gefunden");
            }
            echo "✅ Username-Feld gefunden\n";
            
            // 3. Fülle Username
            echo "3. Fülle Username ein...\n";
            $this->sendKeys($usernameElementId, $this->username);
            echo "✅ Username eingegeben\n";
            
            // 4. Fülle Password
            echo "4. Fülle Password ein...\n";
            $passwordElementId = $this->findElement('id', 'InputPass');
            if (!$passwordElementId) {
                throw new Exception("Password-Feld nicht gefunden");
            }
            $this->sendKeys($passwordElementId['ELEMENT'], $this->password);
            echo "✅ Password eingegeben\n";
            
            // 5. Submit
            echo "5. Klicke Login-Button...\n";
            $submitElementId = $this->findElement('name', 'button');
            if (!$submitElementId) {
                throw new Exception("Submit-Button nicht gefunden");
            }
            $this->clickElement($submitElementId['ELEMENT']);
            echo "✅ Login-Button geklickt\n";
            
            // 6. Warte auf Seitenwechsel (Login erfolgreich)
            sleep(2);
            
            // Prüfe ob wir auf der Startseite sind
            $currentUrl = $this->execute('url', []);
            if (strpos($currentUrl['value'], 'resy.dtm-reifen.de') !== false) {
                error_log("✅ Login erfolgreich! URL: " . $currentUrl['value']);
                return true;
            }
            
            return false;
            
        } catch (Exception $e) {
            error_log("❌ Login-Fehler: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Lädt Reifensätze
     */
    public function loadWheelsets() {
        try {
            $searchUrl = 'https://resy.dtm-reifen.de/index.php?m=wheelset&a=search&FN=WheelsetSearch&KD_ID=' . $this->dealerId . '&STATUS_ID=20';
            $this->navigate($searchUrl);
            error_log("📄 Suchseite geladen");
            
            // Warte etwas
            sleep(3);
            
            // Hole HTML
            $htmlResponse = $this->execute('source', []);
            if ($htmlResponse) {
                $html = $htmlResponse['value'];
                error_log("✅ Reifensätze geladen (" . strlen($html) . " Zeichen)");
                return $html;
            }
            
            return null;
            
        } catch (Exception $e) {
            error_log("❌ Fehler beim Laden: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Beendet die Session
     */
    public function close() {
        if ($this->sessionId) {
            $ch = curl_init("http://localhost:4444/wd/hub/session/{$this->sessionId}");
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "DELETE");
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_exec($ch);
            curl_close($ch);
            error_log("✅ Selenium-Session beendet");
        }
    }
    
    /**
     * Prüft ob Selenium verfügbar ist
     */
    public static function isAvailable() {
        $ch = curl_init('http://localhost:4444/wd/hub/status');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 2);
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        return $httpCode === 200;
    }
}

