<?php
/**
 * RESY Data Extraction API - Fixed Version
 */

// Sicherstellen, dass nur JSON ausgegeben wird
ob_start();

// Headers setzen
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Error handling
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Debug-Logging-Funktion
function debugLog($message) {
    $logFile = __DIR__ . '/../logs/resy_extraction.log';
    $timestamp = date('Y-m-d H:i:s');
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

// Hauptverarbeitung
try {
    debugLog("=== API Request Started ===");
    debugLog("Request Method: " . ($_SERVER['REQUEST_METHOD'] ?? 'unknown'));
    debugLog("Content Type: " . ($_SERVER['CONTENT_TYPE'] ?? 'unknown'));
    
    // JSON-Input lesen
    $rawInput = file_get_contents('php://input');
    debugLog("Raw Input Length: " . strlen($rawInput));
    debugLog("Raw Input: " . substr($rawInput, 0, 500));
    
    if (empty($rawInput)) {
        throw new Exception('Keine Eingabedaten empfangen');
    }
    
    $input = json_decode($rawInput, true);
    
    if (!$input) {
        $error = json_last_error_msg();
        debugLog("JSON Decode Error: " . $error);
        throw new Exception('Keine gültigen JSON-Daten empfangen: ' . $error);
    }
    
    $action = $input['action'] ?? '';
    debugLog("Empfangene Aktion: $action");
    
    switch ($action) {
        case 'test':
            debugLog("Test-Aktion empfangen");
            $result = [
                'success' => true,
                'message' => 'API funktioniert korrekt',
                'timestamp' => date('Y-m-d H:i:s'),
                'input' => $input,
                'php_version' => phpversion(),
                'server_info' => [
                    'REQUEST_METHOD' => $_SERVER['REQUEST_METHOD'] ?? 'unknown',
                    'CONTENT_TYPE' => $_SERVER['CONTENT_TYPE'] ?? 'unknown',
                    'HTTP_HOST' => $_SERVER['HTTP_HOST'] ?? 'unknown',
                    'SERVER_NAME' => $_SERVER['SERVER_NAME'] ?? 'unknown'
                ]
            ];
            echo json_encode($result, JSON_PRETTY_PRINT);
            break;
            
        case 'login':
            $appointmentId = $input['appointment_id'] ?? null;
            debugLog("Login für Appointment ID: $appointmentId");
            
            if (!$appointmentId) {
                throw new Exception('Appointment ID fehlt');
            }
            
            // Simuliere erfolgreichen Login
            $result = [
                'success' => true,
                'sessionData' => [
                    'baseUrl' => 'https://resy.dtm-reifen.de',
                    'loggedIn' => true,
                    'timestamp' => time(),
                    'username' => 'D.ULUDAG',
                    'appointmentId' => $appointmentId
                ],
                'message' => 'Login erfolgreich simuliert'
            ];
            
            debugLog("Login erfolgreich: " . json_encode($result));
            echo json_encode($result, JSON_PRETTY_PRINT);
            break;
            
        case 'navigate_to_search':
            debugLog("Navigate to search");
            $result = [
                'success' => true,
                'message' => 'Navigation zur Suchseite simuliert',
                'sessionData' => $input['sessionData'] ?? []
            ];
            echo json_encode($result, JSON_PRETTY_PRINT);
            break;
            
        case 'search_lgs_id':
            $lgsId = $input['lgsId'] ?? null;
            debugLog("Search LGS ID: $lgsId");
            
            if (!$lgsId) {
                throw new Exception('LGS_ID fehlt');
            }
            
            $result = [
                'success' => true,
                'message' => "Suche nach LGS_ID $lgsId simuliert",
                'sessionData' => $input['sessionData'] ?? []
            ];
            echo json_encode($result, JSON_PRETTY_PRINT);
            break;
            
        case 'navigate_to_detail':
            $lgsId = $input['lgsId'] ?? null;
            debugLog("Navigate to detail for LGS ID: $lgsId");
            
            if (!$lgsId) {
                throw new Exception('LGS_ID fehlt');
            }
            
            $result = [
                'success' => true,
                'message' => "Navigation zur Detailseite für LGS_ID $lgsId simuliert",
                'sessionData' => $input['sessionData'] ?? []
            ];
            echo json_encode($result, JSON_PRETTY_PRINT);
            break;
            
        case 'extract_field':
            $field = $input['field'] ?? null;
            debugLog("Extract field: $field");
            
            if (!$field) {
                throw new Exception('Feld-Name fehlt');
            }
            
            // Simuliere Feld-Extraktion
            $mockValues = [
                'fahrzeughalter' => 'Max Mustermann',
                'kennzeichen' => 'KI-AB 123',
                'fin' => 'WBA12345678901234',
                'eingangsnummer' => '12345',
                'ladestelle' => 'Lager A-15',
                'bemerkung' => 'Reifen in gutem Zustand',
                'ruecklieferungsdatum' => '2025-09-20',
                'rueckgabe_bemerkung' => 'Keine Besonderheiten'
            ];
            
            $value = $mockValues[$field] ?? 'Test-Wert für ' . $field;
            
            $result = [
                'success' => true,
                'value' => $value,
                'field' => $field,
                'message' => "Feld '$field' erfolgreich extrahiert (simuliert)"
            ];
            echo json_encode($result, JSON_PRETTY_PRINT);
            break;
            
        default:
            throw new Exception("Unbekannte Aktion: $action");
    }
    
} catch (Exception $e) {
    debugLog("API-Fehler: " . $e->getMessage());
    debugLog("Stack Trace: " . $e->getTraceAsString());
    
    // Sicherstellen, dass nur JSON ausgegeben wird
    http_response_code(500);
    $result = [
        'success' => false,
        'error' => $e->getMessage(),
        'debug' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'timestamp' => date('Y-m-d H:i:s')
        ]
    ];
    echo json_encode($result, JSON_PRETTY_PRINT);
}

// Alle Output-Buffer leeren um sicherzustellen, dass nur JSON ausgegeben wird
$output = ob_get_clean();
if (!empty($output) && !json_decode($output)) {
    // Falls der Output nicht JSON ist, als Fehler behandeln
    $result = [
        'success' => false,
        'error' => 'Unerwarteter Output: ' . substr($output, 0, 200),
        'raw_output' => $output
    ];
    echo json_encode($result, JSON_PRETTY_PRINT);
}

debugLog("=== API Request Ended ===");
?>
