<?php
/**
 * RESY Data Extraction API - Final Solution
 * Verwendet die bewährte tire_order.php Logik und interpretiert die Suche korrekt
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Error handling
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Debug-Logging-Funktion
function debugLog($message) {
    $logFile = __DIR__ . '/../logs/resy_extraction.log';
    $timestamp = date('Y-m-d H:i:s');
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

// HTTP-Helper-Funktion mit verbessertem Session-Management
function httpRequest($url, $options = []) {
    $defaultOptions = [
        'method' => 'GET',
        'headers' => [],
        'data' => null,
        'timeout' => 30,
        'cookies' => null,
        'follow_redirects' => true,
        'cookie_file' => sys_get_temp_dir() . '/resy_cookies_final.txt'
    ];
    
    $options = array_merge($defaultOptions, $options);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, $options['follow_redirects']);
    curl_setopt($ch, CURLOPT_TIMEOUT, $options['timeout']);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36');
    
    // Verbessertes Session-Management
    curl_setopt($ch, CURLOPT_COOKIEJAR, $options['cookie_file']);
    curl_setopt($ch, CURLOPT_COOKIEFILE, $options['cookie_file']);
    curl_setopt($ch, CURLOPT_COOKIESESSION, false); // Verwende persistente Cookies
    curl_setopt($ch, CURLOPT_FRESH_CONNECT, false); // Verwende bestehende Verbindung
    
    // Zusätzliche Session-Optionen
    curl_setopt($ch, CURLOPT_COOKIELIST, ''); // Leere Cookie-Liste für bessere Kontrolle
    curl_setopt($ch, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_1); // HTTP/1.1 für bessere Kompatibilität
    
    if ($options['cookies']) {
        curl_setopt($ch, CURLOPT_COOKIE, $options['cookies']);
    }
    
    if ($options['method'] === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($options['data']) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $options['data']);
        }
    }
    
    if (!empty($options['headers'])) {
        curl_setopt($ch, CURLOPT_HTTPHEADER, $options['headers']);
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    
    curl_close($ch);
    
    if ($error) {
        throw new Exception("cURL Error: $error");
    }
    
    return [
        'body' => $response,
        'http_code' => $httpCode,
        'url' => $url
    ];
}

// RESY Login mit bewährter Logik
function performFinalLogin($baseUrl, $username, $password) {
    debugLog("=== FINAL LOGIN START ===");
    debugLog("Base URL: $baseUrl");
    debugLog("Username: $username");
    
    // Cookie-Datei für persistente Session
    $cookieFile = sys_get_temp_dir() . '/resy_cookies_final.txt';
    
    // Login-Seite laden
    $loginUrl = "$baseUrl/index.php?m=login&a=login";
    debugLog("Lade Login-Seite: $loginUrl");
    
    $response = httpRequest($loginUrl, [
        'cookie_file' => $cookieFile,
        'headers' => [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Connection: keep-alive'
        ]
    ]);
    
    if ($response['http_code'] !== 200) {
        throw new Exception("Login-Seite nicht erreichbar: HTTP " . $response['http_code']);
    }
    
    debugLog("Login-Seite erfolgreich geladen");
    
    // Login-Daten senden
    $postData = [
        'FN' => 'login',
        'UserName' => $username,
        'Password' => $password,
        'button' => 'submit'
    ];
    
    $loginDataString = http_build_query($postData);
    debugLog("Login-Daten: " . str_replace($password, '***', $loginDataString));
    
    $loginResponse = httpRequest($loginUrl, [
        'method' => 'POST',
        'data' => $loginDataString,
        'headers' => [
            'Content-Type: application/x-www-form-urlencoded',
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Connection: keep-alive',
            'Referer: ' . $loginUrl
        ],
        'cookie_file' => $cookieFile
    ]);
    
    debugLog("Login-Response Code: " . $loginResponse['http_code']);
    
    // Login-Erfolg prüfen - nur auf RESY-Content prüfen
    $responseBody = $loginResponse['body'];
    
    $hasResyContent = strpos($responseBody, 'RESY') !== false ||
                     strpos($responseBody, 'Übersicht') !== false ||
                     strpos($responseBody, 'Wheel') !== false;
    
    debugLog("Login-Validierung - RESY-Content: " . ($hasResyContent ? 'Ja' : 'Nein') . 
            ", Length: " . strlen($responseBody));
    
    if ($hasResyContent) {
        debugLog("✅ Login erfolgreich! (RESY-Content gefunden)");
        debugLog("💾 Session-Cookies gespeichert in: $cookieFile");
        debugLog("=== FINAL LOGIN END ===");
        return $loginResponse;
    }
    
    throw new Exception("Login fehlgeschlagen - kein RESY-Content gefunden");
}

// Suche mit tire_order.php Style und korrekter Interpretation
function searchTireSetFinal($baseUrl, $storageNumber) {
    debugLog("=== SEARCH TIRE SET FINAL START ===");
    debugLog("Storage Number: $storageNumber");
    
    $cookieFile = sys_get_temp_dir() . '/resy_cookies_final.txt';
    
    // Satznummer suchen mit GET-Methode (genau wie tire_order.php)
    $searchUrl = $baseUrl . '/index.php?m=wheelset&a=search&LGS_ID=' . urlencode($storageNumber) . '&button=submit&FN=WheelsetSearch';
    debugLog("Suche URL: $searchUrl");
    
    $response = httpRequest($searchUrl, [
        'method' => 'GET',
        'headers' => [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Referer: ' . $baseUrl . '/index.php?m=wheelset&a=search'
        ],
        'cookie_file' => $cookieFile
    ]);
    
    debugLog("Suche-Response Code: " . $response['http_code'] . ", Length: " . strlen($response['body']));
    
    // Debug: Speichere die Suchantwort
    file_put_contents(sys_get_temp_dir() . '/resy_search_final.html', $response['body']);
    
    // Prüfen ob Satznummer gefunden wurde (genau wie tire_order.php)
    if (strpos($response['body'], 'Keine Ergebnisse gefunden') !== false || 
        strpos($response['body'], 'Nicht gefunden') !== false ||
        strpos($response['body'], 'keine Treffer') !== false ||
        strpos($response['body'], 'Keine Treffer') !== false ||
        strpos($response['body'], 'Keine Sätze gefunden') !== false ||
        strpos($response['body'], 'keine Sätze gefunden') !== false) {
        throw new Exception('Einlagerungsnummer ' . $storageNumber . ' nicht im RESY-System gefunden');
    }
    
    // Zusätzliche Prüfung: Antwort muss eine gültige Satznummer-Tabelle enthalten
    if (strpos($response['body'], 'LGS_ID=') === false && strpos($response['body'], 'Satznummer') === false) {
        throw new Exception('Einlagerungsnummer ' . $storageNumber . ' nicht im RESY-System gefunden (keine gültige Antwort)');
    }
    
    // LGS_ID aus der Antwort extrahieren (genau wie tire_order.php)
    $lgsId = null;
    
    // Muster 1: LGS_ID=1234567 in der Antwort suchen
    if (preg_match('/LGS_ID=(\d+)/', $response['body'], $matches)) {
        $lgsId = $matches[1];
        debugLog("LGS_ID gefunden: " . $lgsId . " für Einlagerungsnummer " . $storageNumber);
    } else {
        debugLog("Keine LGS_ID in Antwort gefunden für Einlagerungsnummer " . $storageNumber);
    }
    
    // KEIN Fallback mehr! Wenn keine LGS_ID gefunden wird, ist die Satznummer nicht gültig
    if (!$lgsId) {
        throw new Exception('Einlagerungsnummer ' . $storageNumber . ' nicht im RESY-System gefunden (keine LGS_ID in Antwort)');
    }
    
    debugLog("✅ Suche erfolgreich! LGS_ID: $lgsId");
    debugLog("=== SEARCH TIRE SET FINAL END ===");
    
    return [
        'success' => true,
        'lgs_id' => $lgsId,
        'message' => 'Satznummer gefunden',
        'response' => $response
    ];
}

// Navigiere zur Detailseite mit tire_order.php Style
function navigateToDetailPageFinal($baseUrl, $lgsId) {
    debugLog("=== NAVIGATE TO DETAIL PAGE FINAL START ===");
    debugLog("LGS_ID: $lgsId");
    
    $cookieFile = sys_get_temp_dir() . '/resy_cookies_final.txt';
    
    // Direkt zur Detailseite navigieren (genau wie tire_order.php)
    $detailUrl = $baseUrl . '/index.php?m=wheelset&a=edit&LGS_ID=' . $lgsId . '&FN=WheelsetResult';
    debugLog("Detail URL: $detailUrl");
    
    $response = httpRequest($detailUrl, [
        'headers' => [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Referer: ' . $baseUrl . '/index.php?m=wheelset&a=search'
        ],
        'cookie_file' => $cookieFile
    ]);
    
    debugLog("Detail-Response Code: " . $response['http_code'] . ", Length: " . strlen($response['body']));
    
    // Debug: Speichere die Detail-Antwort
    file_put_contents(sys_get_temp_dir() . '/resy_detail_final.html', $response['body']);
    
    // Prüfe ob es eine Detailseite ist
    $hasWheelHaltername = strpos($response['body'], 'WheelHALTERNAME') !== false;
    $hasLoginForm = strpos($response['body'], 'Anmeldename') !== false;
    $hasWheelKennzeichen = strpos($response['body'], 'WheelKENNZEICHEN') !== false;
    $hasWheelFahrgestnr = strpos($response['body'], 'WheelFAHRGESTNR') !== false;
    
    debugLog("Detailseite - WheelHALTERNAME: " . ($hasWheelHaltername ? 'Ja' : 'Nein') . 
            ", Login-Form: " . ($hasLoginForm ? 'Ja' : 'Nein') . 
            ", WheelKENNZEICHEN: " . ($hasWheelKennzeichen ? 'Ja' : 'Nein') . 
            ", WheelFAHRGESTNR: " . ($hasWheelFahrgestnr ? 'Ja' : 'Nein'));
    
    if ($hasLoginForm) {
        throw new Exception("Session verloren - Login-Formular auf Detailseite gefunden");
    }
    
    if ($hasWheelHaltername && $hasWheelKennzeichen && $hasWheelFahrgestnr) {
        debugLog("✅ Detailseite erfolgreich gefunden!");
        debugLog("=== NAVIGATE TO DETAIL PAGE FINAL END ===");
        return $response;
    }
    
    throw new Exception("Keine Detailseite für LGS_ID $lgsId gefunden");
}

// HTML-Daten extrahieren
function extractDataFromHtml($html, $field) {
    $dom = new DOMDocument();
    @$dom->loadHTML($html);
    $xpath = new DOMXPath($dom);
    
    $fieldMapping = [
        'fahrzeughalter' => 'WheelHALTERNAME',
        'kennzeichen' => 'WheelKENNZEICHEN',
        'fin' => 'WheelFAHRGESTNR',
        'eingangsnummer' => 'WheelLGS_ID',
        'ladestelle' => 'WheelKD_ID',
        'bemerkung' => 'KDBEMERK',
        'ruecklieferungsdatum' => 'WheelRUECKLIEFERTERMIN',
        'rueckgabe_bemerkung' => 'WheelRUECKAKTBEMERKUNG'
    ];
    
    $htmlId = $fieldMapping[$field] ?? null;
    if (!$htmlId) {
        throw new Exception("Unbekanntes Feld: $field");
    }
    
    $element = $xpath->query("//*[@id='$htmlId']")->item(0);
    
    if (!$element) {
        return '';
    }
    
    $value = trim($element->textContent);
    
    if ($field === 'bemerkung') {
        $textarea = $xpath->query("//textarea[@name='KDBEMERK']")->item(0);
        if ($textarea) {
            $value = trim($textarea->textContent);
        }
    }
    
    if (empty($value)) {
        $boldElement = $xpath->query(".//b", $element)->item(0);
        if ($boldElement) {
            $value = trim($boldElement->textContent);
        }
    }
    
    return $value;
}

// Hauptverarbeitung
try {
    debugLog("=== FINAL RESY API Request Started ===");
    
    $rawInput = file_get_contents('php://input');
    debugLog("Raw Input Length: " . strlen($rawInput));
    
    if (empty($rawInput)) {
        throw new Exception('Keine Eingabedaten empfangen');
    }
    
    $input = json_decode($rawInput, true);
    
    if (!$input) {
        $error = json_last_error_msg();
        debugLog("JSON Decode Error: " . $error);
        throw new Exception('Keine gültigen JSON-Daten empfangen: ' . $error);
    }
    
    $action = $input['action'] ?? '';
    debugLog("Empfangene Aktion: $action");
    
    // Datenbankverbindung
    try {
        require_once __DIR__ . '/../config/database.php';
        $db = new Database();
        $conn = $db->getConnection();
        debugLog("Datenbankverbindung erfolgreich");
    } catch (Exception $e) {
        debugLog("Datenbankfehler: " . $e->getMessage());
        throw new Exception('Datenbankfehler: ' . $e->getMessage());
    }
    
    switch ($action) {
        case 'final':
            $appointmentId = $input['appointment_id'] ?? null;
            $lgsId = $input['lgsId'] ?? null;
            debugLog("Final für Appointment ID: $appointmentId, LGS_ID: $lgsId");
            
            if (!$appointmentId || !$lgsId) {
                throw new Exception('Appointment ID oder LGS_ID fehlt');
            }
            
            // Autohaus-Daten aus Appointment laden
            $stmt = $conn->prepare("
                SELECT a.resy_base_url, a.resy_username, a.resy_password 
                FROM autohaus a 
                JOIN appointments ap ON a.id = ap.autohaus_id 
                WHERE ap.id = ?
            ");
            $stmt->execute([$appointmentId]);
            $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$autohaus) {
                throw new Exception('Autohaus-Daten nicht gefunden für Appointment ID: ' . $appointmentId);
            }
            
            debugLog("Autohaus gefunden: " . json_encode($autohaus));
            
            // Schritt 1: Login mit bewährter Logik durchführen
            performFinalLogin($autohaus['resy_base_url'], $autohaus['resy_username'], $autohaus['resy_password']);
            
            // Schritt 2: Suche mit tire_order.php Style durchführen
            $searchResult = searchTireSetFinal($autohaus['resy_base_url'], $lgsId);
            
            // Schritt 3: Zur Detailseite navigieren
            $detailResponse = navigateToDetailPageFinal($autohaus['resy_base_url'], $searchResult['lgs_id']);
            
            $result = [
                'success' => true,
                'detailHtml' => $detailResponse['body'],
                'url' => $detailResponse['url'],
                'approach' => 'final',
                'message' => 'Erfolgreich eingeloggt und zur Detailseite navigiert mit tire_order.php Style',
                'lgs_id' => $searchResult['lgs_id']
            ];
            
            echo json_encode($result);
            break;
            
        case 'extract_field':
            $field = $input['field'] ?? null;
            $html = $input['html'] ?? null;
            debugLog("Extract field: $field");
            
            if (!$field) {
                throw new Exception('Feld-Name fehlt');
            }
            
            if (!$html) {
                throw new Exception('HTML-Inhalt fehlt');
            }
            
            try {
                $value = extractDataFromHtml($html, $field);
                
                $result = [
                    'success' => true,
                    'value' => $value,
                    'field' => $field,
                    'message' => "Feld '$field' erfolgreich extrahiert"
                ];
                
                echo json_encode($result);
                
            } catch (Exception $e) {
                debugLog("Feld-Extraktion fehlgeschlagen: " . $e->getMessage());
                $result = [
                    'success' => false,
                    'error' => $e->getMessage(),
                    'field' => $field
                ];
                echo json_encode($result);
            }
            break;
            
        default:
            throw new Exception("Unbekannte Aktion: $action");
    }
    
} catch (Exception $e) {
    debugLog("API-Fehler: " . $e->getMessage());
    debugLog("Stack Trace: " . $e->getTraceAsString());
    
    http_response_code(500);
    $result = [
        'success' => false,
        'error' => $e->getMessage(),
        'debug' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'timestamp' => date('Y-m-d H:i:s')
        ]
    ];
    echo json_encode($result);
}

debugLog("=== FINAL RESY API Request Ended ===");
?>