<?php
/**
 * RESY REST API Integration - Echte API-Anbindung
 * Nutzt die offizielle RESY REST API (http://145.253.121.150:8087/resy-rest-extern/)
 * 
 * Endpoints:
 * - GET /v1/customer/:customerNo/inventory/ - Inventarliste
 * - GET /v1/wheelset/:wheelsetnumber/ - Wheelset-Details
 * - GET /v1/wheelset/?VIN=... oder ?licenseNo=... - Wheelsets für Fahrzeug
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once dirname(__DIR__) . '/config/database.php';
require_once __DIR__ . '/resy_rest_api_complete.php';

class ResyRestAPI {
    private $baseUrl = 'http://145.253.121.150:8087/resy-rest-extern';
    private $username;
    private $password;
    private $customerNo;
    private $autohausId;
    private $autohausName;
    
    public function __construct($autohausId) {
        $this->autohausId = $autohausId;
        $this->loadAutohausCredentials();
    }
    
    /**
     * Lädt RESY-Credentials aus der Datenbank
     */
    private function loadAutohausCredentials() {
        $db = new Database();
        $conn = $db->getConnection();
        
        // ROBUST: Versuche zuerst mit resy_customer_no, dann ohne
        try {
            $stmt = $conn->prepare("
                SELECT name, resy_username, resy_password, resy_dealer_id, resy_customer_no
                FROM autohaus 
                WHERE id = ?
            ");
            $stmt->execute([$this->autohausId]);
            $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($autohaus) {
                $this->customerNo = $autohaus['resy_customer_no'] ?? null;
            }
        } catch (PDOException $e) {
            // Falls Spalte nicht existiert, ohne resy_customer_no versuchen
            if (strpos($e->getMessage(), 'resy_customer_no') !== false) {
                $stmt = $conn->prepare("
                    SELECT name, resy_username, resy_password, resy_dealer_id
                    FROM autohaus 
                    WHERE id = ?
                ");
                $stmt->execute([$this->autohausId]);
                $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
                $this->customerNo = null;
            } else {
                throw $e;
            }
        }
        
        if (!$autohaus) {
            throw new Exception("Autohaus mit ID {$this->autohausId} nicht gefunden");
        }
        
        $this->autohausName = $autohaus['name'];
        $this->username = $autohaus['resy_username'] ?? null;
        $this->password = $autohaus['resy_password'] ?? null;
        
        // WICHTIG: Customer No Mapping basierend auf Autohaus-Namen
        if (!$this->customerNo) {
            $autohausName = $this->autohausName ?? '';
            
            $customerNoMapping = [
                'Kiel' => '453',
                'Norderstedt' => '452',
                'Norderstadt' => '452',
                'Glinde' => '454'
            ];
            
            foreach ($customerNoMapping as $key => $customerNo) {
                if (stripos($autohausName, $key) !== false) {
                    $this->customerNo = $customerNo;
                    error_log("RESY: Customer No {$customerNo} für Autohaus '{$autohausName}' ermittelt");
                    break;
                }
            }
            
            // Fallback aus resy_dealer_id
            if (!$this->customerNo && isset($autohaus['resy_dealer_id'])) {
                $dealerId = trim($autohaus['resy_dealer_id']);
                foreach ($customerNoMapping as $key => $customerNo) {
                    if (stripos($dealerId, $key) !== false) {
                        $this->customerNo = $customerNo;
                        break;
                    }
                }
                if (!$this->customerNo && is_numeric($dealerId)) {
                    $this->customerNo = $dealerId;
                }
            }
        }
        
        // Test-Credentials als Fallback
        if (!$this->username || !$this->password) {
            if ($this->autohausId == 1 || $this->autohausId == 201) {
                $this->username = 'BUE1';
                $this->password = 'resytest';
                if (!$this->customerNo) {
                    $this->customerNo = '201';
                }
            } else {
                throw new Exception("RESY-Credentials für Autohaus {$this->autohausName} nicht konfiguriert");
            }
        }
        
        if (!$this->customerNo) {
            error_log("WARNING: Keine Customer No für '{$this->autohausName}' - verwende Fallback 201");
            $this->customerNo = '201';
        }
        
        error_log("RESY API - Autohaus: {$this->autohausName} (ID: {$this->autohausId}), Customer No: {$this->customerNo}");
    }
    
    /**
     * HTTP Request mit Basic Auth
     */
    private function makeRequest($endpoint, $method = 'GET', $data = null) {
        $url = $this->baseUrl . $endpoint;
        
        $ch = curl_init($url);
        
        // Basic Authentication
        $auth = base64_encode($this->username . ':' . $this->password);
        
        $headers = [
            'Authorization: Basic ' . $auth,
            'Accept: application/json',
            'Content-Type: application/json'
        ];
        
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_CONNECTTIMEOUT => 10
        ]);
        
        if ($method === 'POST' && $data) {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            throw new Exception("CURL Fehler: " . $error);
        }
        
        if ($httpCode >= 400) {
            throw new Exception("HTTP Fehler {$httpCode}: " . substr($response, 0, 200));
        }
        
        $decoded = json_decode($response, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception("JSON Parse Fehler: " . json_last_error_msg());
        }
        
        return $decoded;
    }
    
    /**
     * Lädt Inventarliste für den Kunden
     */
    public function getInventory() {
        try {
            $response = $this->makeRequest("/v1/customer/{$this->customerNo}/inventory/");
            return $response['inventory'] ?? [];
        } catch (Exception $e) {
            error_log("RESY Inventory Fehler: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Lädt alle Wheelset-Nummern für den Kunden
     * Nutzt verschiedene Methoden:
     * 1. Inventory-Endpoint
     * 2. Pickup-Orders (falls Inventory leer)
     */
    public function getAllWheelsetNumbers() {
        $wheelsetNumbers = [];
        
        // Methode 1: Inventory
        $inventory = $this->getInventory();
        foreach ($inventory as $item) {
            if (isset($item['inventoryNumbers']) && is_array($item['inventoryNumbers'])) {
                $wheelsetNumbers = array_merge($wheelsetNumbers, $item['inventoryNumbers']);
            }
        }
        
        // Methode 2: Pickup Orders (wenn Inventory leer)
        if (empty($wheelsetNumbers)) {
            try {
                $pickupOrders = $this->makeRequest("/v1/pickuporder/{$this->customerNo}/");
                if (isset($pickupOrders['pickupOrders']) && is_array($pickupOrders['pickupOrders'])) {
                    foreach ($pickupOrders['pickupOrders'] as $order) {
                        if (isset($order['wheelSetNumber'])) {
                            $wheelsetNumbers[] = $order['wheelSetNumber'];
                        }
                    }
                }
            } catch (Exception $e) {
                error_log("Pickup Orders Fehler: " . $e->getMessage());
            }
        }
        
        return array_unique($wheelsetNumbers);
    }
    
    /**
     * Lädt Details für einen Wheelset
     */
    public function getWheelsetDetails($wheelsetNumber) {
        try {
            $response = $this->makeRequest("/v1/wheelset/{$wheelsetNumber}/");
            return $response['wheelSet'] ?? null;
        } catch (Exception $e) {
            error_log("Wheelset {$wheelsetNumber} Details Fehler: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Transformiert RESY-Daten in unser Format
     */
    private function transformWheelset($resyWheelset) {
        if (!$resyWheelset) {
            return null;
        }
        
        // Extrahiere Reifendaten (nehme erstes Rad als Beispiel)
        $firstWheel = isset($resyWheelset['wheels']) && count($resyWheelset['wheels']) > 0 
            ? $resyWheelset['wheels'][0] 
            : [];
        
        // Berechne Reifengröße
        $width = $firstWheel['width'] ?? '';
        $ratio = $firstWheel['ratio'] ?? '';
        $diameter = $firstWheel['diameter'] ?? '';
        $reifengroesse = '';
        if ($width && $diameter) {
            $reifengroesse = "{$width}/{$ratio}R{$diameter}";
            if (!$ratio) {
                $reifengroesse = "{$width}R{$diameter}";
            }
        } else {
            $reifengroesse = $firstWheel['normalizedWheelSize'] ?? '';
        }
        
        // Bestimme Reifenart und Saison
        $tireType = $firstWheel['tireType'] ?? '';
        $reifenart = '';
        $saison = '';
        
        switch (strtoupper($tireType)) {
            case 'WINTER':
                $reifenart = 'Winterreifen';
                $saison = 'Winter';
                break;
            case 'SUMMER':
                $reifenart = 'Sommerreifen';
                $saison = 'Sommer';
                break;
            case 'ALLSEASON':
                $reifenart = 'Ganzjahresreifen';
                $saison = 'Ganzjahres';
                break;
            default:
                $reifenart = $tireType ?: 'Unbekannt';
                $saison = 'Unbekannt';
        }
        
        // Reifenmodell
        $manufacturer = $firstWheel['manufacturer'] ?? '';
        $model = $firstWheel['model'] ?? '';
        $reifenmodell = trim("{$manufacturer} {$model}");
        
        // Profiltiefe (aus erstem Rad, wenn verfügbar)
        $profiltiefe = $firstWheel['profile'] ?? null;
        if ($profiltiefe === null) {
            $profiltiefe = '-';
        } else {
            $profiltiefe = round($profiltiefe, 2) . ' mm';
        }
        
        // Empfehlung basierend auf Profiltiefe
        $empfehlung = 'In Ordnung';
        if (is_numeric($profiltiefe)) {
            $profiltiefeValue = floatval($profiltiefe);
            if ($profiltiefeValue < 1.6) {
                $empfehlung = 'Austausch notwendig';
            } elseif ($profiltiefeValue < 3) {
                $empfehlung = 'Prüfung empfohlen';
            }
        }
        
        // Status
        $status = 'eingelagert'; // Standard für gespeicherte Wheelsets
        
        return [
            'satznummer' => (string)($resyWheelset['wheelSetNumber'] ?? ''),
            'kennzeichen' => $resyWheelset['licenseNo'] ?? '-',
            'fin' => $resyWheelset['VIN'] ?? '-',
            'haltername' => '-', // Nicht in RESY API verfügbar
            'reifenmodell' => $reifenmodell ?: '-',
            'reifenart' => $reifenart,
            'saison' => $saison,
            'reifengroesse' => $reifengroesse ?: '-',
            'profiltiefe' => $profiltiefe,
            'empfehlung' => $empfehlung,
            'autohaus' => $this->autohausName,
            'ziel_autohaus' => '-',
            'eingangsdatum' => '-', // Kann aus pickupDate kommen, wenn verfügbar
            'rueckliefertermin' => '-',
            'lagerplatz' => $resyWheelset['storagePlace'] ?? '-',
            'status' => $status,
            'pruefbericht' => !empty($resyWheelset['testReportURL']) ? $resyWheelset['testReportURL'] : '-',
            'servicePackageName' => $resyWheelset['servicePackageName'] ?? '-',
            'comment' => $resyWheelset['comment'] ?? '',
            // Zusätzliche RESY-Daten
            'resy_customer_no' => $resyWheelset['customerNo'] ?? $this->customerNo,
            'resy_service_package' => $resyWheelset['servicePackageIdent'] ?? '',
            'resy_wheels' => $resyWheelset['wheels'] ?? []
        ];
    }
    
    /**
     * Lädt alle Wheelsets für das Autohaus (nutzt jetzt die komplette API)
     */
    public function getAllWheelsets() {
        try {
            // Nutze die komplette API-Klasse mit Smart-Loading
            $completeAPI = new ResyRestAPIComplete($this->autohausId);
            $wheelsets = $completeAPI->getAllWheelsetsSmart();
            
            return [
                'success' => true,
                'wheelsets' => $wheelsets,
                'count' => count($wheelsets),
                'autohaus_id' => $this->autohausId,
                'autohaus' => $this->autohausName,
                'customer_no' => $this->customerNo,
                'source' => 'resy_rest_api_complete',
                'loaded_at' => date('Y-m-d H:i:s')
            ];
            
        } catch (Exception $e) {
            error_log("RESY API Fehler: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'wheelsets' => [],
                'count' => 0
            ];
        }
    }
}

// Haupt-Logik
try {
    $autohausId = $_GET['autohaus_id'] ?? $_POST['autohaus_id'] ?? 1;
    
    // Debug: Log Autohaus-Info
    error_log("RESY API Request - Autohaus ID: {$autohausId}");
    
    $api = new ResyRestAPI($autohausId);
    $result = $api->getAllWheelsets();
    
    // Log Ergebnis
    error_log("RESY API Response - Success: " . ($result['success'] ? 'true' : 'false') . ", Count: " . ($result['count'] ?? 0));
    
    echo json_encode($result, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    // Detaillierte Fehlerbehandlung
    $errorDetails = [
        'success' => false,
        'error' => $e->getMessage(),
        'error_type' => get_class($e),
        'autohaus_id' => $_GET['autohaus_id'] ?? $_POST['autohaus_id'] ?? 'unknown',
        'wheelsets' => [],
        'count' => 0,
        'debug_info' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => explode("\n", $e->getTraceAsString())
        ]
    ];
    
    error_log("RESY API ERROR: " . $e->getMessage() . " in " . $e->getFile() . ":" . $e->getLine());
    
    http_response_code(500);
    echo json_encode($errorDetails, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
}

