<?php
require_once __DIR__ . '/_guard_bootstrap.php';
/**
 * API Endpoint: Get Available Time Slots
 * 
 * This API provides available time slots for an event on a specific date.
 * It can be used by mobile apps, web apps, or other services.
 * 
 * Parameters:
 * - event_id (required): The event ID
 * - date (required): The date in YYYY-MM-DD format
 * 
 * Returns:
 * - success: boolean
 * - event: object with event details
 * - available_slots: array of available time slots
 * - date: the requested date
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

require_once __DIR__ . '/../config/database.php';

try {
    $event_id = $_GET['event_id'] ?? null;
    $date = $_GET['date'] ?? null;
    
    if (!$event_id || !$date) {
        http_response_code(400);
        echo json_encode([
            'success' => false, 
            'error' => 'Event ID and date are required',
            'message' => 'Please provide both event_id and date parameters'
        ]);
        exit;
    }
    
    // Validate date format
    $dateObj = DateTime::createFromFormat('Y-m-d', $date);
    if (!$dateObj || $dateObj->format('Y-m-d') !== $date) {
        http_response_code(400);
        echo json_encode([
            'success' => false, 
            'error' => 'Invalid date format',
            'message' => 'Date must be in YYYY-MM-DD format'
        ]);
        exit;
    }
    
    $db = new Database();
    $conn = $db->getConnection();
    
    // Get event details
    $stmt = $conn->prepare("
        SELECT 
            e.id as event_id,
            e.name as event_name,
            e.event_date,
            e.start_time,
            e.end_time,
            e.slot_duration,
            e.max_monteure,
            e.pause_start,
            e.pause_end,
            e.blocked_start,
            e.blocked_end
        FROM events e 
        WHERE e.id = ? AND e.is_active = 1
    ");
    $stmt->execute([$event_id]);
    $event = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$event) {
        http_response_code(404);
        echo json_encode([
            'success' => false, 
            'error' => 'Event not found',
            'message' => 'The requested event does not exist or is inactive'
        ]);
        exit;
    }
    
    // Get booked appointments for the date (including all active statuses)
    $stmt = $conn->prepare("
        SELECT slot_time, COUNT(*) as booked_count
        FROM appointments 
        WHERE event_id = ? AND slot_date = ? AND status IN ('pending', 'approved', 'booked', 'vorbereitet', 'completed')
        GROUP BY slot_time
    ");
    $stmt->execute([$event_id, $date]);
    $bookedSlots = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
    
    // Generate available time slots
    $availableSlots = generateTimeSlots($event, $bookedSlots);
    
    // Return success response
    echo json_encode([
        'success' => true,
        'event' => [
            'id' => $event['event_id'],
            'name' => $event['event_name'],
            'event_date' => $event['event_date'],
            'start_time' => $event['start_time'],
            'end_time' => $event['end_time'],
            'slot_duration' => $event['slot_duration'],
            'max_monteure' => $event['max_monteure']
        ],
        'available_slots' => $availableSlots,
        'date' => $date,
        'total_slots' => count($availableSlots),
        'total_available' => array_sum(array_column($availableSlots, 'available_count'))
    ]);
    
} catch (Exception $e) {
    error_log("Error in get_available_slots.php: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false, 
        'error' => 'Internal server error',
        'message' => $e->getMessage()
    ]);
}

/**
 * Generate time slots for an event
 */
function generateTimeSlots($event, $bookedSlots) {
    $slots = [];
    $startTime = new DateTime($event['start_time']);
    $endTime = new DateTime($event['end_time']);
    
    $slotDuration = $event['slot_duration'] ?? 30; // minutes
    $maxCount = $event['max_monteure'] ?? 5;
    
    // Pause times
    $pauseStart = $event['pause_start'] ? new DateTime($event['pause_start']) : null;
    $pauseEnd = $event['pause_end'] ? new DateTime($event['pause_end']) : null;
    $blockedStart = $event['blocked_start'] ? new DateTime($event['blocked_start']) : null;
    $blockedEnd = $event['blocked_end'] ? new DateTime($event['blocked_end']) : null;
    
    $currentTime = clone $startTime;
    
    while ($currentTime < $endTime) {
        $timeStr = $currentTime->format('H:i:s');
        $booked = $bookedSlots[$timeStr] ?? 0;
        $availableCount = max(0, $maxCount - $booked);
        
        // Skip if in pause time
        if ($pauseStart && $pauseEnd) {
            if ($currentTime >= $pauseStart && $currentTime < $pauseEnd) {
                $currentTime->modify("+{$slotDuration} minutes");
                continue;
            }
        }
        
        // Skip if in blocked time
        if ($blockedStart && $blockedEnd) {
            if ($currentTime >= $blockedStart && $currentTime < $blockedEnd) {
                $currentTime->modify("+{$slotDuration} minutes");
                continue;
            }
        }
        
        $slots[] = [
            'time' => $timeStr,
            'formatted_time' => $currentTime->format('H:i'),
            'available' => $availableCount > 0,
            'available_count' => $availableCount,
            'booked' => $booked,
            'max_count' => $maxCount
        ];
        
        $currentTime->modify("+{$slotDuration} minutes");
    }
    
    return $slots;
}
?>

