<?php
/**
 * Cache Manager für RESY-Daten
 * Verwaltet mehrstufiges Caching für bessere Performance
 */

class CacheManager {
    private $cacheDir;
    private $db;
    
    public function __construct($database) {
        $this->db = $database;
        $this->cacheDir = __DIR__ . '/../cache/';
        
        // Erstelle Cache-Verzeichnis falls nicht vorhanden
        if (!is_dir($this->cacheDir)) {
            mkdir($this->cacheDir, 0755, true);
        }
    }
    
    /**
     * Generiert einen Cache-Key basierend auf Parametern
     */
    private function generateCacheKey($autohausId, $statusId = 20) {
        return 'wheelsets_' . $autohausId . '_' . $statusId . '_' . date('Y-m-d-H');
    }
    
    /**
     * Prüft ob Cache-Daten vorhanden und gültig sind
     */
    public function isCacheValid($autohausId, $statusId = 20, $ttlMinutes = 30) {
        $cacheKey = $this->generateCacheKey($autohausId, $statusId);
        
        // Prüfe Datenbank-Cache
        $stmt = $this->db->prepare("
            SELECT created_at FROM wheelsets_cache 
            WHERE cache_key = ? AND created_at > DATE_SUB(NOW(), INTERVAL ? MINUTE)
        ");
        $stmt->execute([$cacheKey, $ttlMinutes]);
        
        return $stmt->fetch() !== false;
    }
    
    /**
     * Lädt gecachte Reifensätze aus der Datenbank
     */
    public function getCachedWheelsets($autohausId, $statusId = 20) {
        $cacheKey = $this->generateCacheKey($autohausId, $statusId);
        
        $stmt = $this->db->prepare("
            SELECT data FROM wheelsets_cache 
            WHERE cache_key = ? AND created_at > DATE_SUB(NOW(), INTERVAL 30 MINUTE)
            ORDER BY created_at DESC LIMIT 1
        ");
        $stmt->execute([$cacheKey]);
        
        $result = $stmt->fetch();
        if ($result) {
            $data = json_decode($result['data'], true);
            error_log("Cache HIT für Autohaus $autohausId: " . count($data) . " Reifensätze");
            return $data;
        }
        
        error_log("Cache MISS für Autohaus $autohausId");
        return null;
    }
    
    /**
     * Speichert Reifensätze im Cache
     */
    public function setCachedWheelsets($autohausId, $statusId, $wheelsets) {
        $cacheKey = $this->generateCacheKey($autohausId, $statusId);
        $data = json_encode($wheelsets);
        
        // Lösche alte Einträge
        $stmt = $this->db->prepare("DELETE FROM wheelsets_cache WHERE cache_key = ?");
        $stmt->execute([$cacheKey]);
        
        // Speichere neue Daten
        $stmt = $this->db->prepare("
            INSERT INTO wheelsets_cache (cache_key, data, created_at) 
            VALUES (?, ?, NOW())
        ");
        $stmt->execute([$cacheKey, $data]);
        
        error_log("Cache gespeichert für Autohaus $autohausId: " . count($wheelsets) . " Reifensätze");
    }
    
    /**
     * Speichert HTML-Response im Datei-Cache
     */
    public function setCachedHtml($autohausId, $html) {
        $filename = $this->cacheDir . 'resy_response_' . $autohausId . '_' . date('Y-m-d-H') . '.html';
        file_put_contents($filename, $html);
        error_log("HTML-Cache gespeichert: $filename");
    }
    
    /**
     * Lädt HTML-Response aus dem Datei-Cache
     */
    public function getCachedHtml($autohausId) {
        $filename = $this->cacheDir . 'resy_response_' . $autohausId . '_' . date('Y-m-d-H') . '.html';
        
        if (file_exists($filename) && (time() - filemtime($filename)) < 3600) { // 1 Stunde TTL
            $html = file_get_contents($filename);
            error_log("HTML-Cache HIT: $filename");
            return $html;
        }
        
        error_log("HTML-Cache MISS: $filename");
        return null;
    }
    
    /**
     * Invalidiert Cache für ein bestimmtes Autohaus
     */
    public function invalidateCache($autohausId) {
        // Lösche Datenbank-Cache
        $stmt = $this->db->prepare("DELETE FROM wheelsets_cache WHERE cache_key LIKE ?");
        $stmt->execute(['wheelsets_' . $autohausId . '_%']);
        
        // Lösche HTML-Cache-Dateien
        $pattern = $this->cacheDir . 'resy_response_' . $autohausId . '_*.html';
        foreach (glob($pattern) as $file) {
            unlink($file);
        }
        
        error_log("Cache invalidiert für Autohaus $autohausId");
    }
    
    /**
     * Bereinigt abgelaufene Cache-Einträge
     */
    public function cleanupExpiredCache() {
        // Lösche abgelaufene Datenbank-Einträge
        $stmt = $this->db->prepare("DELETE FROM wheelsets_cache WHERE created_at < DATE_SUB(NOW(), INTERVAL 2 HOUR)");
        $stmt->execute();
        
        // Lösche abgelaufene HTML-Dateien
        $pattern = $this->cacheDir . 'resy_response_*.html';
        foreach (glob($pattern) as $file) {
            if ((time() - filemtime($file)) > 3600) { // 1 Stunde TTL
                unlink($file);
            }
        }
        
        error_log("Abgelaufene Cache-Einträge bereinigt");
    }
    
    /**
     * Gibt Cache-Statistiken zurück
     */
    public function getCacheStats() {
        $stmt = $this->db->prepare("
            SELECT 
                COUNT(*) as total_entries,
                COUNT(CASE WHEN created_at > DATE_SUB(NOW(), INTERVAL 30 MINUTE) THEN 1 END) as recent_entries,
                MIN(created_at) as oldest_entry,
                MAX(created_at) as newest_entry
            FROM wheelsets_cache
        ");
        $stmt->execute();
        
        return $stmt->fetch();
    }
}
?>
