<?php
/**
 * HTML Response Analyzer für RESY-Suchversuche
 * Analysiert die gespeicherten HTML-Antworten der Suchversuche
 */

$tempDir = sys_get_temp_dir();
echo "=== RESY HTML RESPONSE ANALYZER ===\n";
echo "Temp Directory: $tempDir\n\n";

// Finde alle HTML-Dateien
$htmlFiles = glob($tempDir . '/resy_search_updated_*.html');
$htmlFiles = array_merge($htmlFiles, glob($tempDir . '/resy_alternative_*.html'));
$htmlFiles = array_merge($htmlFiles, glob($tempDir . '/resy_fallback_*.html'));

echo "Gefundene HTML-Dateien: " . count($htmlFiles) . "\n\n";

foreach ($htmlFiles as $file) {
    echo "--- Analysiere: " . basename($file) . " ---\n";
    
    if (!file_exists($file)) {
        echo "❌ Datei nicht gefunden\n\n";
        continue;
    }
    
    $content = file_get_contents($file);
    $fileSize = strlen($content);
    
    echo "Dateigröße: $fileSize Bytes\n";
    
    // Prüfe ob die Datei komprimiert ist
    if (substr($content, 0, 2) === "\x1f\x8b") {
        echo "⚠️ Datei ist gzip-komprimiert\n";
        $content = gzdecode($content);
        if ($content === false) {
            echo "❌ Kann gzip-Daten nicht dekodieren\n\n";
            continue;
        }
        echo "✅ Gzip-Daten erfolgreich dekodiert\n";
    }
    
    $contentLength = strlen($content);
    echo "Dekodierte Länge: $contentLength Zeichen\n";
    
    // Analysiere den Inhalt
    $analysis = analyzeHtmlContent($content);
    
    foreach ($analysis as $key => $value) {
        echo "$key: $value\n";
    }
    
    echo "\n";
}

function analyzeHtmlContent($html) {
    $analysis = [];
    
    // Grundlegende Analyse
    $analysis['Enthält HTML'] = strpos($html, '<html') !== false ? 'Ja' : 'Nein';
    $analysis['Enthält DOCTYPE'] = strpos($html, '<!DOCTYPE') !== false ? 'Ja' : 'Nein';
    $analysis['Enthält RESY'] = strpos($html, 'RESY') !== false ? 'Ja' : 'Nein';
    $analysis['Enthält Login'] = strpos($html, 'Anmeldename') !== false ? 'Ja' : 'Nein';
    $analysis['Enthält Wheelset'] = strpos($html, 'Wheelset') !== false ? 'Ja' : 'Nein';
    $analysis['Enthält LGS_ID'] = strpos($html, 'LGS_ID') !== false ? 'Ja' : 'Nein';
    
    // Suche nach Fehlermeldungen
    $errorPatterns = [
        'nicht gefunden' => 'nicht gefunden',
        'keine Ergebnisse' => 'keine Ergebnisse',
        'error' => 'error',
        'fehler' => 'fehler',
        'not found' => 'not found',
        'no results' => 'no results'
    ];
    
    foreach ($errorPatterns as $pattern => $label) {
        if (stripos($html, $pattern) !== false) {
            $analysis["Fehlermeldung ($label)"] = 'Gefunden';
        }
    }
    
    // Suche nach Suchformularen
    if (strpos($html, '<form') !== false) {
        $analysis['Enthält Formular'] = 'Ja';
        
        // Suche nach Suchfeldern
        if (strpos($html, 'name="LGS_ID"') !== false) {
            $analysis['LGS_ID Feld'] = 'Gefunden';
        }
        if (strpos($html, 'name="search"') !== false) {
            $analysis['Search Feld'] = 'Gefunden';
        }
    }
    
    // Suche nach Tabellen oder Listen
    if (strpos($html, '<table') !== false) {
        $analysis['Enthält Tabelle'] = 'Ja';
    }
    if (strpos($html, '<ul') !== false || strpos($html, '<ol') !== false) {
        $analysis['Enthält Liste'] = 'Ja';
    }
    
    // Suche nach Links
    preg_match_all('/href="[^"]*"/', $html, $links);
    $analysis['Anzahl Links'] = count($links[0]);
    
    // Suche nach spezifischen RESY-Elementen
    $resyElements = [
        'WheelHALTERNAME' => 'WheelHALTERNAME',
        'WheelKENNZEICHEN' => 'WheelKENNZEICHEN',
        'WheelFAHRGESTNR' => 'WheelFAHRGESTNR',
        'WheelsetView' => 'WheelsetView',
        'WheelsetSearch' => 'WheelsetSearch'
    ];
    
    foreach ($resyElements as $element => $label) {
        if (strpos($html, $element) !== false) {
            $analysis["RESY Element ($label)"] = 'Gefunden';
        }
    }
    
    // Extrahiere den Titel
    if (preg_match('/<title[^>]*>(.*?)<\/title>/i', $html, $matches)) {
        $analysis['Titel'] = trim($matches[1]);
    }
    
    // Extrahiere Meta-Description
    if (preg_match('/<meta[^>]*name=["\']description["\'][^>]*content=["\']([^"\']*)["\'][^>]*>/i', $html, $matches)) {
        $analysis['Meta Description'] = trim($matches[1]);
    }
    
    return $analysis;
}

echo "=== ANALYSE ABGESCHLOSSEN ===\n";
?>
