<?php
/**
 * 🔗 PUBLIC SHARE PAGE
 * Öffentliche Seite zum Anzeigen und Importieren von geteilten Schnellaktionen
 */

require_once '../config/database_optimized.php';

$db = DatabaseOptimized::getInstance();
$conn = $db->getConnection();

$share_token = $_GET['token'] ?? '';
$action = null;
$error = '';

if (!empty($share_token)) {
    try {
        $stmt = $conn->prepare("
            SELECT id, title, description, url, icon, background_color, text_color, target, created_at
            FROM quick_actions 
            WHERE share_token = ? AND is_active = 1
        ");
        $stmt->execute([$share_token]);
        $action = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$action) {
            $error = 'Diese geteilte Aktion wurde nicht gefunden oder ist nicht mehr verfügbar.';
        }
    } catch (Exception $e) {
        $error = 'Fehler beim Laden der Aktion.';
    }
} else {
    $error = 'Kein gültiger Teilungslink angegeben.';
}
?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $action ? htmlspecialchars($action['title']) : 'Geteilte Aktion'; ?> - Boxenstop</title>
    
    <!-- Apple-Volvo Design System CSS -->
    <link rel="stylesheet" href="../assets/css/style.css">
    
    <!-- Typography -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=SF+Pro+Display:wght@200;300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    
    <!-- Icons -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    
    <style>
        :root {
            --system-blue: #007AFF;
            --system-green: #30D158;
            --system-red: #FF453A;
            --volvo-blue: #003057;
            --volvo-navy: #001529;
            --label-primary: #1D1D1F;
            --label-secondary: #48484A;
            --label-tertiary: #8E8E93;
            --background-primary: #FFFFFF;
            --background-secondary: #F9F9F9;
            --separator-non-opaque: #3C3C4349;
            --shadow-medium: 0 2px 8px rgba(0, 0, 0, 0.08);
            --shadow-large: 0 4px 16px rgba(0, 0, 0, 0.12);
            --radius-medium: 10px;
            --radius-large: 14px;
            --space-md: 12px;
            --space-lg: 18px;
            --space-xl: 24px;
            --space-xxl: 36px;
            --font-system: -apple-system, BlinkMacSystemFont, 'SF Pro Display', 'SF Pro Text', system-ui, sans-serif;
        }

        [data-theme="dark"] {
            --label-primary: #FFFFFF;
            --label-secondary: #EBEBF5;
            --label-tertiary: #8E8E93;
            --background-primary: #000000;
            --background-secondary: #0A0A0A;
            --separator-non-opaque: #54545899;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            -webkit-font-smoothing: antialiased;
            -moz-osx-font-smoothing: grayscale;
        }

        body {
            font-family: var(--font-system);
            background: var(--background-secondary);
            color: var(--label-primary);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: var(--space-lg);
        }

        .share-container {
            background: var(--background-primary);
            border: 1px solid var(--separator-non-opaque);
            border-radius: var(--radius-large);
            box-shadow: var(--shadow-medium);
            padding: var(--space-xxl);
            max-width: 420px;
            width: 100%;
            text-align: center;
        }

        .action-preview {
            background: var(--background-secondary);
            border: 1px solid var(--separator-non-opaque);
            border-radius: var(--radius-medium);
            padding: var(--space-lg);
            margin: var(--space-lg) 0;
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: var(--space-md);
        }

        .action-icon {
            width: 52px;
            height: 52px;
            border-radius: var(--radius-medium);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 20px;
            color: white;
        }

        .action-title {
            font-size: 20px;
            font-weight: 600;
            color: var(--label-primary);
            margin-bottom: var(--space-sm);
        }

        .action-description {
            font-size: 15px;
            color: var(--label-secondary);
            margin-bottom: var(--space-lg);
            line-height: 1.4;
        }

        .action-url {
            font-size: 12px;
            color: var(--label-tertiary);
            background: var(--background-primary);
            padding: var(--space-md);
            border-radius: var(--radius-small);
            word-break: break-all;
            font-family: 'SF Mono', Consolas, monospace;
        }

        .btn {
            background: var(--system-blue);
            color: white;
            border: none;
            border-radius: var(--radius-medium);
            padding: var(--space-md) var(--space-lg);
            font-family: var(--font-system);
            font-size: 15px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.15s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: var(--space-md);
            margin: var(--space-md) var(--space-sm);
        }

        .btn:hover {
            background: #0056CC;
        }

        .btn-secondary {
            background: var(--background-secondary);
            color: var(--label-primary);
            border: 1px solid var(--separator-non-opaque);
        }

        .btn-secondary:hover {
            background: var(--label-tertiary);
        }

        .error-message {
            background: rgba(255, 59, 48, 0.1);
            border: 1px solid rgba(255, 59, 48, 0.3);
            color: var(--system-red);
            padding: var(--space-lg);
            border-radius: 12px;
            margin: var(--space-lg) 0;
        }

        .success-message {
            background: rgba(52, 199, 89, 0.1);
            border: 1px solid rgba(52, 199, 89, 0.3);
            color: var(--system-green);
            padding: var(--space-lg);
            border-radius: 12px;
            margin: var(--space-lg) 0;
        }

        .brand-logo {
            margin-bottom: var(--space-lg);
        }

        .brand-logo h1 {
            font-size: 28px;
            font-weight: 600;
            color: var(--label-primary);
            margin-bottom: var(--space-sm);
            letter-spacing: -0.01em;
        }

        .brand-logo p {
            color: var(--label-tertiary);
            font-size: 15px;
        }

        @media (max-width: 768px) {
            .share-container {
                padding: var(--space-lg);
            }
            
            .action-title {
                font-size: 20px;
            }
            
            .btn {
                width: 100%;
                justify-content: center;
            }
        }
    </style>
</head>
<body>
    <div class="share-container">
        <div class="brand-logo">
            <h1>Boxenstop</h1>
            <p>Geteilte Schnellaktion</p>
        </div>

        <?php if ($error): ?>
            <div class="error-message">
                <i class="fas fa-exclamation-triangle"></i>
                <p><?php echo htmlspecialchars($error); ?></p>
            </div>
            <a href="../index.php" class="btn btn-secondary">
                <i class="fas fa-home"></i>
                Zur Startseite
            </a>
        <?php elseif ($action): ?>
            <div class="action-preview">
                <div class="action-icon" style="background: <?php echo htmlspecialchars($action['background_color']); ?>;">
                    <i class="<?php echo htmlspecialchars($action['icon']); ?>"></i>
                </div>
                <h2 class="action-title"><?php echo htmlspecialchars($action['title']); ?></h2>
                <?php if (!empty($action['description'])): ?>
                <p class="action-description"><?php echo htmlspecialchars($action['description']); ?></p>
                <?php endif; ?>
                <div class="action-url"><?php echo htmlspecialchars($action['url']); ?></div>
            </div>

            <div class="action-buttons">
                <a href="<?php echo htmlspecialchars($action['url']); ?>" 
                   class="btn" 
                   <?php echo $action['target'] === '_blank' ? 'target="_blank"' : ''; ?>>
                    <i class="fas fa-external-link-alt"></i>
                    Aktion öffnen
                </a>
                <a href="../index.php" class="btn btn-secondary">
                    <i class="fas fa-home"></i>
                    Zur Startseite
                </a>
            </div>

            <div class="import-info">
                <p style="color: var(--label-tertiary); font-size: 14px; margin-top: var(--space-lg);">
                    <i class="fas fa-info-circle"></i>
                    Um diese Aktion zu deinen Schnellaktionen hinzuzufügen, melde dich im Admin-Bereich an.
                </p>
            </div>
        <?php endif; ?>
    </div>
</body>
</html>
