<?php
/**
 * 🚀 QUICK ACTIONS MANAGER - Premium Apple-Volvo Design
 * Completely redesigned with authentic Apple-Volvo aesthetics
 */

session_start();
require_once '../config/database_optimized.php';
require_once '../includes/performance_monitor.php';

// 🔐 Security Check
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

$db = DatabaseOptimized::getInstance();
$conn = $db->getConnection();

// 👤 User Data & Theme
$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_name = 'Administrator';
$current_user_role = 'admin';
$current_theme = $_SESSION['current_theme'] ?? 'light';

if ($current_user_id) {
    $user_data = $db->getUserData($current_user_id);
    if ($user_data) {
        $current_user_name = $user_data['full_name'] ?? $user_data['username'] ?? 'Administrator';
        $current_user_role = $user_data['role'] ?? 'admin';
        $current_theme = $user_data['theme'] ?? $_SESSION['current_theme'] ?? 'light';
    }
}

// 🎨 Theme Toggle Handler
if (isset($_POST['toggle_theme'])) {
    $new_theme = ($current_theme === 'light') ? 'dark' : 'light';
    
    if ($current_user_id) {
        try {
            $stmt = $conn->prepare("UPDATE admin_users SET theme = ? WHERE id = ?");
            $stmt->execute([$new_theme, $current_user_id]);
            $current_theme = $new_theme;
            $_SESSION['current_theme'] = $current_theme;
        } catch (Exception $e) {
            $current_theme = $new_theme;
            $_SESSION['current_theme'] = $current_theme;
        }
    } else {
        $current_theme = $new_theme;
        $_SESSION['current_theme'] = $current_theme;
    }
}

// 📝 Messages
$message = '';
$message_type = '';

// 🔧 CRUD Operations
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        try {
            switch ($_POST['action']) {
                case 'create':
                    $stmt = $conn->prepare("
                        INSERT INTO quick_actions (title, url, icon, background_color, user_id, created_at) 
                        VALUES (?, ?, ?, ?, ?, NOW())
                    ");
                    $stmt->execute([
                        $_POST['title'],
                        $_POST['url'],
                        $_POST['icon'],
                        $_POST['color'],
                        $current_user_id
                    ]);
                    $message = 'Schnellaktion erfolgreich erstellt!';
                    $message_type = 'success';
                    break;

                case 'update':
                    $stmt = $conn->prepare("
                        UPDATE quick_actions 
                        SET title = ?, url = ?, icon = ?, background_color = ?, updated_at = NOW() 
                        WHERE id = ? AND user_id = ?
                    ");
                    $stmt->execute([
                        $_POST['title'],
                        $_POST['url'],
                        $_POST['icon'],
                        $_POST['color'],
                        $_POST['id'],
                        $current_user_id
                    ]);
                    $message = 'Schnellaktion erfolgreich aktualisiert!';
                    $message_type = 'success';
                    break;

                case 'delete':
                    $stmt = $conn->prepare("DELETE FROM quick_actions WHERE id = ? AND user_id = ?");
                    $stmt->execute([$_POST['id'], $current_user_id]);
                    $message = 'Schnellaktion erfolgreich gelöscht!';
                    $message_type = 'success';
                    break;

                case 'toggle_active':
                    $stmt = $conn->prepare("
                        UPDATE quick_actions 
                        SET is_active = NOT is_active, updated_at = NOW() 
                        WHERE id = ? AND user_id = ?
                    ");
                    $stmt->execute([$_POST['id'], $current_user_id]);
                    $message = 'Status erfolgreich geändert!';
                    $message_type = 'success';
                    break;

                case 'update_positions':
                    if (isset($_POST['positions']) && is_array($_POST['positions'])) {
                        foreach ($_POST['positions'] as $position => $id) {
                            $stmt = $conn->prepare("
                                UPDATE quick_actions 
                                SET position_order = ?, updated_at = NOW() 
                                WHERE id = ? AND user_id = ?
                            ");
                            $stmt->execute([$position, $id, $current_user_id]);
                        }
                        echo json_encode(['success' => true]);
                        exit;
                    }
                    break;
            }
        } catch (Exception $e) {
            $message = 'Fehler: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
}

// 📊 Get Quick Actions
$stmt = $conn->prepare("
    SELECT * FROM quick_actions 
    WHERE user_id = ? OR user_id IS NULL 
    ORDER BY user_id DESC, position_order ASC, id ASC
");
$stmt->execute([$current_user_id]);
$quick_actions = $stmt->fetchAll(PDO::FETCH_ASSOC);

// 🎨 Available Icons & Colors - Premium Collection
$available_icons = [
    'fas fa-home', 'fas fa-dashboard', 'fas fa-chart-bar', 'fas fa-users',
    'fas fa-calendar-alt', 'fas fa-tasks', 'fas fa-envelope', 'fas fa-phone',
    'fas fa-car', 'fas fa-tools', 'fas fa-warehouse', 'fas fa-clipboard-list',
    'fas fa-search', 'fas fa-download', 'fas fa-upload', 'fas fa-print',
    'fas fa-share-alt', 'fas fa-edit', 'fas fa-trash-alt', 'fas fa-star',
    'fas fa-heart', 'fas fa-bookmark', 'fas fa-flag', 'fas fa-rocket',
    'fas fa-cog', 'fas fa-user-cog', 'fas fa-database', 'fas fa-server'
];

$available_colors = [
    '#007AFF', '#5856D6', '#AF52DE', '#FF2D92', '#FF3B30', '#FF9500',
    '#FFCC00', '#34C759', '#00C7BE', '#64D2FF', '#007AFF', '#5856D6',
    '#1D1D1F', '#86868B', '#F5F5F7', '#FFFFFF'
];
?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, user-scalable=no">
    <title>Schnellaktionen Manager • Boxenstop</title>
    
    <!-- Apple-Volvo Design System CSS -->
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="assets/css/modern-sidebar.css">
    <link rel="stylesheet" href="assets/css/admin-inline-styles.css">
    
    <!-- Typography -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=SF+Pro+Display:wght@200;300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    
    <!-- Icons -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/sortablejs@1.15.0/Sortable.min.css">
    
    <style>
        /* 🎨 APPLE-VOLVO DESIGN SYSTEM - PREMIUM EDITION */
        :root {
            /* Apple System Colors */
            --system-blue: #007AFF;
            --system-green: #34C759;
            --system-indigo: #5856D6;
            --system-orange: #FF9500;
            --system-pink: #FF2D92;
            --system-purple: #AF52DE;
            --system-red: #FF3B30;
            --system-teal: #30B0C7;
            --system-yellow: #FFCC00;
            
            /* Volvo Brand Colors */
            --volvo-blue: #005AAA;
            --volvo-navy: #1A365D;
            --volvo-silver: #C1C8CE;
            --volvo-gold: #FFD700;
            
            /* Semantic Colors - Light */
            --label-primary: #000000;
            --label-secondary: #3C3C43;
            --label-tertiary: #3C3C4399;
            --label-quaternary: #3C3C432E;
            
            --fill-primary: #78788033;
            --fill-secondary: #78788028;
            --fill-tertiary: #7676801F;
            --fill-quaternary: #74748014;
            
            --background-primary: #FFFFFF;
            --background-secondary: #F2F2F7;
            --background-tertiary: #FFFFFF;
            
            --grouped-background-primary: #F2F2F7;
            --grouped-background-secondary: #FFFFFF;
            --grouped-background-tertiary: #F2F2F7;
            
            --separator-opaque: #C6C6C8;
            --separator-non-opaque: #3C3C4349;
            
            /* Glass Effects */
            --glass-background: rgba(255, 255, 255, 0.8);
            --glass-border: rgba(255, 255, 255, 0.18);
            
            /* Shadows */
            --shadow-small: 0 1px 3px rgba(0, 0, 0, 0.12);
            --shadow-medium: 0 4px 16px rgba(0, 0, 0, 0.12);
            --shadow-large: 0 8px 32px rgba(0, 0, 0, 0.16);
            --shadow-xlarge: 0 16px 64px rgba(0, 0, 0, 0.20);
            
            /* Border Radius */
            --radius-small: 8px;
            --radius-medium: 12px;
            --radius-large: 16px;
            --radius-xlarge: 20px;
            --radius-xxlarge: 24px;
            
            /* Spacing */
            --space-xs: 4px;
            --space-sm: 8px;
            --space-md: 16px;
            --space-lg: 24px;
            --space-xl: 32px;
            --space-xxl: 48px;
            
            /* Typography */
            --font-system: -apple-system, BlinkMacSystemFont, 'SF Pro Display', system-ui, sans-serif;
            --font-monospace: 'SF Mono', Consolas, 'Liberation Mono', Menlo, monospace;
        }

        /* Dark Theme */
        [data-theme="dark"] {
            --label-primary: #FFFFFF;
            --label-secondary: #EBEBF5;
            --label-tertiary: #EBEBF599;
            --label-quaternary: #EBEBF52E;
            
            --fill-primary: #7878805C;
            --fill-secondary: #78788052;
            --fill-tertiary: #7676803D;
            --fill-quaternary: #74748029;
            
            --background-primary: #000000;
            --background-secondary: #1C1C1E;
            --background-tertiary: #2C2C2E;
            
            --grouped-background-primary: #000000;
            --grouped-background-secondary: #1C1C1E;
            --grouped-background-tertiary: #2C2C2E;
            
            --separator-opaque: #38383A;
            --separator-non-opaque: #54545899;
            
            --glass-background: rgba(28, 28, 30, 0.8);
            --glass-border: rgba(255, 255, 255, 0.1);
        }

        /* Reset & Base */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            -webkit-font-smoothing: antialiased;
            -moz-osx-font-smoothing: grayscale;
        }

        body {
            font-family: var(--font-system);
            background: var(--background-primary);
            color: var(--label-primary);
            line-height: 1.5;
        }

        /* Main Layout */
        .admin-main {
            padding: var(--space-xl);
            background: var(--grouped-background-primary);
            min-height: 100vh;
        }

        /* Header Section */
        .page-header {
            background: var(--glass-background);
            backdrop-filter: blur(40px);
            -webkit-backdrop-filter: blur(40px);
            border: 1px solid var(--glass-border);
            border-radius: var(--radius-xxlarge);
            padding: var(--space-xl);
            margin-bottom: var(--space-xl);
            box-shadow: var(--shadow-medium);
        }

        .page-header-content {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            gap: var(--space-lg);
        }

        .page-title {
            font-size: 34px;
            font-weight: 700;
            color: var(--label-primary);
            letter-spacing: -0.02em;
            margin-bottom: var(--space-xs);
        }

        .page-subtitle {
            font-size: 17px;
            color: var(--label-secondary);
            font-weight: 400;
            line-height: 1.4;
        }

        /* Theme Toggle */
        .theme-toggle {
            background: var(--fill-secondary);
            border: 1px solid var(--separator-non-opaque);
            border-radius: var(--radius-large);
            padding: var(--space-sm) var(--space-md);
            color: var(--label-primary);
            font-family: var(--font-system);
            font-size: 15px;
            font-weight: 500;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: var(--space-sm);
            transition: all 0.2s cubic-bezier(0.25, 0.46, 0.45, 0.94);
        }

        .theme-toggle:hover {
            background: var(--fill-primary);
            transform: scale(1.02);
        }

        .theme-toggle:active {
            transform: scale(0.98);
        }

        /* Content Section */
        .content-section {
            margin-bottom: var(--space-xl);
        }

        .section-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: var(--space-lg);
        }

        .section-title {
            font-size: 28px;
            font-weight: 600;
            color: var(--label-primary);
            letter-spacing: -0.01em;
        }

        /* Add Button */
        .add-action-btn {
            background: linear-gradient(135deg, var(--system-blue) 0%, var(--volvo-blue) 100%);
            color: white;
            border: none;
            border-radius: var(--radius-large);
            padding: var(--space-md) var(--space-lg);
            font-family: var(--font-system);
            font-size: 15px;
            font-weight: 600;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: var(--space-sm);
            box-shadow: var(--shadow-small);
            transition: all 0.3s cubic-bezier(0.25, 0.46, 0.45, 0.94);
        }

        .add-action-btn:hover {
            transform: translateY(-1px);
            box-shadow: var(--shadow-medium);
        }

        .add-action-btn:active {
            transform: translateY(0);
        }

        /* Actions Grid */
        .actions-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: var(--space-lg);
        }

        /* Action Card - Premium Design */
        .action-card {
            background: var(--glass-background);
            backdrop-filter: blur(40px);
            -webkit-backdrop-filter: blur(40px);
            border: 1px solid var(--glass-border);
            border-radius: var(--radius-xxlarge);
            padding: var(--space-lg);
            cursor: grab;
            transition: all 0.4s cubic-bezier(0.25, 0.46, 0.45, 0.94);
            position: relative;
            overflow: hidden;
        }

        .action-card:hover {
            transform: translateY(-4px);
            box-shadow: var(--shadow-large);
            border-color: rgba(0, 122, 255, 0.3);
        }

        .action-card:active {
            transform: translateY(-2px);
        }

        .action-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 3px;
            background: linear-gradient(90deg, var(--system-blue), var(--system-purple), var(--system-pink));
            opacity: 0;
            transition: opacity 0.3s ease;
        }

        .action-card:hover::before {
            opacity: 1;
        }

        /* Action Icon */
        .action-icon {
            width: 52px;
            height: 52px;
            border-radius: var(--radius-medium);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 22px;
            color: white;
            margin-bottom: var(--space-md);
            box-shadow: var(--shadow-small);
            position: relative;
            background: var(--system-blue) !important;
        }

        .action-icon::after {
            content: '';
            position: absolute;
            inset: 0;
            border-radius: inherit;
            background: linear-gradient(135deg, rgba(255,255,255,0.3) 0%, transparent 50%, rgba(0,0,0,0.1) 100%);
        }

        .action-icon i {
            position: relative;
            z-index: 1;
        }

        /* Action Content */
        .action-content {
            margin-bottom: var(--space-md);
        }

        .action-title {
            font-size: 19px;
            font-weight: 600;
            color: var(--label-primary);
            margin-bottom: var(--space-xs);
            letter-spacing: -0.01em;
        }

        .action-url {
            font-size: 13px;
            color: var(--label-tertiary);
            font-family: var(--font-monospace);
            background: var(--fill-tertiary);
            padding: var(--space-xs) var(--space-sm);
            border-radius: var(--radius-small);
            word-break: break-all;
        }

        /* Action Controls */
        .action-controls {
            display: flex;
            gap: var(--space-sm);
            padding-top: var(--space-md);
            border-top: 1px solid var(--separator-non-opaque);
        }

        .control-btn {
            background: var(--fill-secondary);
            border: 1px solid var(--separator-non-opaque);
            border-radius: var(--radius-small);
            padding: var(--space-xs) var(--space-sm);
            color: var(--label-secondary);
            font-family: var(--font-system);
            font-size: 13px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.2s ease;
        }

        .control-btn:hover {
            background: var(--fill-primary);
            color: var(--label-primary);
            transform: scale(1.02);
        }

        .control-btn.delete {
            color: var(--system-red);
        }

        .control-btn.delete:hover {
            background: rgba(255, 59, 48, 0.1);
            border-color: var(--system-red);
        }

        /* Modal */
        .modal-overlay {
            position: fixed;
            inset: 0;
            background: rgba(0, 0, 0, 0.4);
            backdrop-filter: blur(20px);
            -webkit-backdrop-filter: blur(20px);
            z-index: 1000;
            display: none;
            align-items: center;
            justify-content: center;
            opacity: 0;
            transition: all 0.3s ease;
        }

        .modal-overlay.active {
            display: flex;
            opacity: 1;
        }

        .modal {
            background: var(--glass-background);
            backdrop-filter: blur(40px);
            -webkit-backdrop-filter: blur(40px);
            border: 1px solid var(--glass-border);
            border-radius: var(--radius-xxlarge);
            width: 90%;
            max-width: 480px;
            max-height: 90vh;
            overflow-y: auto;
            box-shadow: var(--shadow-xlarge);
            transform: scale(0.9);
            transition: all 0.3s cubic-bezier(0.34, 1.56, 0.64, 1);
        }

        .modal-overlay.active .modal {
            transform: scale(1);
        }

        .modal-header {
            padding: var(--space-xl) var(--space-xl) var(--space-lg);
            border-bottom: 1px solid var(--separator-non-opaque);
        }

        .modal-title {
            font-size: 24px;
            font-weight: 700;
            color: var(--label-primary);
            letter-spacing: -0.02em;
        }

        .modal-body {
            padding: var(--space-lg) var(--space-xl) var(--space-xl);
        }

        /* Form Styles */
        .form-group {
            margin-bottom: var(--space-lg);
        }

        .form-label {
            display: block;
            font-size: 15px;
            font-weight: 600;
            color: var(--label-primary);
            margin-bottom: var(--space-sm);
        }

        .form-input {
            width: 100%;
            padding: var(--space-md);
            border: 1px solid var(--separator-non-opaque);
            border-radius: var(--radius-medium);
            font-family: var(--font-system);
            font-size: 16px;
            color: var(--label-primary);
            background: var(--background-secondary);
            transition: all 0.2s ease;
        }

        .form-input:focus {
            outline: none;
            border-color: var(--system-blue);
            box-shadow: 0 0 0 3px rgba(0, 122, 255, 0.1);
            background: var(--background-primary);
        }

        /* Icon & Color Grids */
        .selection-grid {
            display: grid;
            gap: var(--space-sm);
            margin-top: var(--space-sm);
        }

        .icon-grid {
            grid-template-columns: repeat(auto-fill, minmax(44px, 1fr));
        }

        .color-grid {
            grid-template-columns: repeat(auto-fill, minmax(36px, 1fr));
        }

        .selection-option {
            aspect-ratio: 1;
            border: 1px solid var(--separator-non-opaque);
            border-radius: var(--radius-small);
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.2s ease;
            position: relative;
        }

        .selection-option:hover {
            transform: scale(1.1);
            box-shadow: var(--shadow-small);
        }

        .selection-option.selected {
            border-color: var(--system-blue);
            box-shadow: 0 0 0 2px rgba(0, 122, 255, 0.3);
        }

        .icon-option {
            color: var(--label-secondary);
            font-size: 16px;
        }

        .icon-option.selected {
            background: var(--system-blue);
            color: white;
        }

        /* Form Actions */
        .form-actions {
            display: flex;
            gap: var(--space-md);
            justify-content: flex-end;
            margin-top: var(--space-xl);
            padding-top: var(--space-lg);
            border-top: 1px solid var(--separator-non-opaque);
        }

        .btn {
            padding: var(--space-md) var(--space-lg);
            border-radius: var(--radius-medium);
            font-family: var(--font-system);
            font-size: 15px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.2s ease;
            display: flex;
            align-items: center;
            gap: var(--space-sm);
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--system-blue) 0%, var(--volvo-blue) 100%);
            color: white;
            border: none;
            box-shadow: var(--shadow-small);
        }

        .btn-primary:hover {
            transform: translateY(-1px);
            box-shadow: var(--shadow-medium);
        }

        .btn-secondary {
            background: var(--fill-secondary);
            color: var(--label-primary);
            border: 1px solid var(--separator-non-opaque);
        }

        .btn-secondary:hover {
            background: var(--fill-primary);
        }

        /* Alert Messages */
        .alert {
            padding: var(--space-md) var(--space-lg);
            border-radius: var(--radius-medium);
            margin-bottom: var(--space-lg);
            display: flex;
            align-items: center;
            gap: var(--space-sm);
            font-weight: 500;
            backdrop-filter: blur(20px);
            -webkit-backdrop-filter: blur(20px);
        }

        .alert-success {
            background: rgba(52, 199, 89, 0.1);
            border: 1px solid rgba(52, 199, 89, 0.3);
            color: var(--system-green);
        }

        .alert-error {
            background: rgba(255, 59, 48, 0.1);
            border: 1px solid rgba(255, 59, 48, 0.3);
            color: var(--system-red);
        }

        /* Sortable States */
        .sortable-ghost {
            opacity: 0.4;
        }

        .sortable-drag {
            transform: rotate(5deg);
            z-index: 1000;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .admin-main {
                padding: var(--space-lg);
            }

            .page-header-content {
                flex-direction: column;
                gap: var(--space-md);
            }

            .section-header {
                flex-direction: column;
                gap: var(--space-md);
                align-items: flex-start;
            }

            .actions-grid {
                grid-template-columns: 1fr;
            }

            .modal {
                width: 95%;
                margin: var(--space-md);
            }

            .form-actions {
                flex-direction: column;
            }
        }

        /* Accessibility */
        @media (prefers-reduced-motion: reduce) {
            * {
                animation-duration: 0.01ms !important;
                animation-iteration-count: 1 !important;
                transition-duration: 0.01ms !important;
            }
        }

        /* High contrast mode */
        @media (prefers-contrast: high) {
            .action-card {
                border-width: 2px;
            }
        }
    </style>
</head>
<body class="admin-body" data-theme="<?php echo htmlspecialchars($current_theme); ?>" data-user-id="<?php echo htmlspecialchars($current_user_id); ?>">
    <div class="admin-container">
        <!-- Sidebar -->
        <aside class="modern-sidebar" id="modern-sidebar">
            <!-- Sidebar Header -->
            <div class="sidebar-header">
                <div class="logo-container">
                    <div class="logo-icon">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M12 2L2 7l10 5 10-5-10-5zM2 17l10 5 10-5M2 12l10 5 10-5"/>
                        </svg>
                    </div>
                    <div class="logo-text">
                        <h2>Boxenstop</h2>
                        <span>Admin Panel</span>
                    </div>
                </div>
                <button class="sidebar-toggle-btn" id="sidebar-toggle-btn">
                    <i class="fas fa-chevron-left"></i>
                </button>
            </div>

            <!-- Navigation Menu -->
            <nav class="sidebar-nav">
                <!-- Main Navigation -->
                <div class="nav-section">
                    <div class="nav-section-title">
                        <span>Übersicht</span>
                    </div>
                    <div class="nav-items">
                        <a href="index.php" class="nav-item">
                            <div class="nav-icon">
                                <i class="fas fa-tachometer-alt"></i>
                            </div>
                            <span class="nav-text">Dashboard</span>
                            <div class="nav-indicator"></div>
                        </a>
                        <?php if ($current_user_role === 'admin'): ?>
                        <a href="performance.php" class="nav-item">
                            <div class="nav-icon">
                                <i class="fas fa-chart-line"></i>
                            </div>
                            <span class="nav-text">Performance</span>
                            <div class="nav-indicator"></div>
                        </a>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Events & Appointments -->
                <div class="nav-section">
                    <div class="nav-section-title">
                        <span>Terminplanung</span>
                    </div>
                    <div class="nav-items">
                        <a href="events.php" class="nav-item">
                            <div class="nav-icon">
                                <i class="fas fa-calendar-alt"></i>
                            </div>
                            <span class="nav-text">Events</span>
                            <div class="nav-indicator"></div>
                        </a>
                        <a href="appointments.php" class="nav-item">
                            <div class="nav-icon">
                                <i class="fas fa-calendar-check"></i>
                            </div>
                            <span class="nav-text">Termine</span>
                            <div class="nav-indicator"></div>
                        </a>
                    </div>
                </div>

                <!-- Storage & Inventory -->
                <div class="nav-section">
                    <div class="nav-section-title">
                        <span>Lagerverwaltung</span>
                    </div>
                    <div class="nav-items">
                        <a href="storage.php" class="nav-item">
                            <div class="nav-icon">
                                <i class="fas fa-warehouse"></i>
                            </div>
                            <span class="nav-text">Einlagerungen</span>
                            <div class="nav-indicator"></div>
                        </a>
                    </div>
                </div>

                <!-- System Management -->
                <div class="nav-section">
                    <div class="nav-section-title">
                        <span>System</span>
                    </div>
                    <div class="nav-items">
                        <a href="profile.php" class="nav-item">
                            <div class="nav-icon">
                                <i class="fas fa-user"></i>
                            </div>
                            <span class="nav-text">Profil</span>
                            <div class="nav-indicator"></div>
                        </a>
                        <a href="settings.php" class="nav-item">
                            <div class="nav-icon">
                                <i class="fas fa-cog"></i>
                            </div>
                            <span class="nav-text">Einstellungen</span>
                            <div class="nav-indicator"></div>
                        </a>
                        <a href="users.php" class="nav-item">
                            <div class="nav-icon">
                                <i class="fas fa-users"></i>
                            </div>
                            <span class="nav-text">Benutzer</span>
                            <div class="nav-indicator"></div>
                        </a>
                        <a href="quick_actions_manager.php" class="nav-item active">
                            <div class="nav-icon">
                                <i class="fas fa-rocket"></i>
                            </div>
                            <span class="nav-text">Schnellaktionen</span>
                            <div class="nav-indicator"></div>
                        </a>
                    </div>
                </div>
            </nav>

            <!-- Sidebar Footer -->
            <div class="sidebar-footer">
                <!-- Theme Toggle -->
                <div class="theme-toggle-container">
                    <button class="theme-toggle-btn" id="theme-toggle-btn-sidebar" title="Design wechseln">
                        <div class="theme-icon">
                            <i class="fas fa-moon" id="theme-icon"></i>
                        </div>
                        <span class="theme-text" id="theme-text">Dunkel</span>
                    </button>
                </div>

                <!-- User Menu -->
                <div class="user-menu-container">
                    <button class="user-menu-btn" id="user-menu-btn">
                        <div class="user-avatar">
                            <i class="fas fa-user"></i>
                        </div>
                        <div class="user-info">
                            <span class="user-name"><?php echo htmlspecialchars($current_user_name); ?></span>
                            <span class="user-role">Administrator</span>
                        </div>
                        <i class="fas fa-chevron-down user-chevron"></i>
                    </button>
                    
                    <div class="user-dropdown" id="user-dropdown">
                        <a href="profile.php" class="user-dropdown-item">
                            <i class="fas fa-user"></i>
                            <span>Mein Profil</span>
                        </a>
                        <a href="settings.php" class="user-dropdown-item">
                            <i class="fas fa-cog"></i>
                            <span>Einstellungen</span>
                        </a>
                        <a href="users.php" class="user-dropdown-item">
                            <i class="fas fa-users"></i>
                            <span>Benutzer</span>
                        </a>
                        <div class="user-dropdown-divider"></div>
                        <a href="logout.php" class="user-dropdown-item logout">
                            <i class="fas fa-sign-out-alt"></i>
                            <span>Abmelden</span>
                        </a>
                    </div>
                </div>
            </div>
        </aside>

        <!-- Main Content -->
        <main class="admin-main" style="margin-left: 280px; transition: margin-left 0.3s cubic-bezier(0.4, 0, 0.2, 1);">
            <!-- Header -->
            <header class="page-header">
                <div class="page-header-content">
                    <div class="header-text">
                        <h1 class="page-title">Schnellaktionen</h1>
                        <p class="page-subtitle">Verwalte deine personalisierten Dashboard-Aktionen mit Apple-Volvo Design</p>
                    </div>
                    <form method="POST" style="display: inline;">
                        <button type="submit" name="toggle_theme" class="theme-toggle" title="Theme wechseln">
                            <i class="fas fa-<?php echo $current_theme === 'light' ? 'moon' : 'sun'; ?>"></i>
                            <?php echo $current_theme === 'light' ? 'Dark Mode' : 'Light Mode'; ?>
                        </button>
                    </form>
                </div>
            </header>

            <!-- Messages -->
            <?php if ($message): ?>
            <div class="alert alert-<?php echo $message_type; ?>">
                <i class="fas fa-<?php echo $message_type === 'success' ? 'check-circle' : 'exclamation-triangle'; ?>"></i>
                <?php echo htmlspecialchars($message); ?>
            </div>
            <?php endif; ?>

            <!-- Content Section -->
            <div class="content-section">
                <div class="section-header">
                    <h2 class="section-title">Deine Aktionen</h2>
                    <button class="add-action-btn" onclick="openModal()">
                        <i class="fas fa-plus"></i>
                        Neue Aktion
                    </button>
                </div>
                
                <!-- Actions Grid -->
                <div class="actions-grid" id="sortable-actions">
                    <?php foreach ($quick_actions as $action): ?>
                    <div class="action-card" data-id="<?php echo $action['id']; ?>">
                        <div class="action-icon" style="background: <?php echo htmlspecialchars($action['background_color'] ?? '#007AFF'); ?> !important;">
                            <i class="<?php echo htmlspecialchars($action['icon'] ?? 'fas fa-star'); ?>"></i>
                        </div>
                        <div class="action-content">
                            <div class="action-title"><?php echo htmlspecialchars($action['title']); ?></div>
                            <div class="action-url"><?php echo htmlspecialchars($action['url']); ?></div>
                        </div>
                        <div class="action-controls">
                            <button class="control-btn" onclick="editAction(<?php echo $action['id']; ?>)">
                                <i class="fas fa-edit"></i> Bearbeiten
                            </button>
                            <form method="POST" style="display: inline;">
                                <input type="hidden" name="action" value="toggle_active">
                                <input type="hidden" name="id" value="<?php echo $action['id']; ?>">
                                <button type="submit" class="control-btn">
                                    <i class="fas fa-<?php echo $action['is_active'] ? 'eye-slash' : 'eye'; ?>"></i>
                                    <?php echo $action['is_active'] ? 'Aus' : 'Ein'; ?>
                                </button>
                            </form>
                            <?php if ($action['user_id'] == $current_user_id): ?>
                            <form method="POST" style="display: inline;" onsubmit="return confirm('Wirklich löschen?')">
                                <input type="hidden" name="action" value="delete">
                                <input type="hidden" name="id" value="<?php echo $action['id']; ?>">
                                <button type="submit" class="control-btn delete">
                                    <i class="fas fa-trash-alt"></i> Löschen
                                </button>
                            </form>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </main>
    </div>

    <!-- Modal -->
    <div class="modal-overlay" id="actionModal">
        <div class="modal">
            <div class="modal-header">
                <h2 class="modal-title" id="modalTitle">Neue Schnellaktion</h2>
            </div>
            <div class="modal-body">
                <form method="POST" id="actionForm">
                    <input type="hidden" name="action" id="actionType" value="create">
                    <input type="hidden" name="id" id="actionId">
                    
                    <div class="form-group">
                        <label class="form-label" for="actionTitle">Titel</label>
                        <input type="text" class="form-input" id="actionTitle" name="title" required placeholder="z.B. Neuer Termin">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="actionUrl">URL</label>
                        <input type="url" class="form-input" id="actionUrl" name="url" required placeholder="https://example.com">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Icon auswählen</label>
                        <input type="hidden" id="actionIcon" name="icon" value="fas fa-rocket">
                        <div class="selection-grid icon-grid">
                            <?php foreach ($available_icons as $icon): ?>
                            <div class="selection-option icon-option" data-icon="<?php echo $icon; ?>" onclick="selectIcon('<?php echo $icon; ?>')">
                                <i class="<?php echo $icon; ?>"></i>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Farbe auswählen</label>
                        <input type="hidden" id="actionColor" name="color" value="#007AFF">
                        <div class="selection-grid color-grid">
                            <?php foreach ($available_colors as $color): ?>
                            <div class="selection-option" style="background: <?php echo $color; ?>;" data-color="<?php echo $color; ?>" onclick="selectColor('<?php echo $color; ?>')"></div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    
                    <div class="form-actions">
                        <button type="button" class="btn btn-secondary" onclick="closeModal()">
                            <i class="fas fa-times"></i> Abbrechen
                        </button>
                        <button type="submit" class="btn btn-primary" id="submitBtn">
                            <i class="fas fa-check"></i> Erstellen
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/sortablejs@1.15.0/Sortable.min.js"></script>
    <script src="assets/js/modern-sidebar.js"></script>
    <script>
        // Initialize sortable with premium animations
        const sortable = new Sortable(document.getElementById('sortable-actions'), {
            animation: 200,
            easing: 'cubic-bezier(0.25, 0.46, 0.45, 0.94)',
            ghostClass: 'sortable-ghost',
            dragClass: 'sortable-drag',
            onEnd: function(evt) {
                const positions = Array.from(evt.to.children).map((el, index) => el.dataset.id);
                
                fetch(window.location.href, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: new URLSearchParams({
                        action: 'update_positions',
                        positions: positions
                    })
                });
            }
        });

        // Modal functions with premium animations
        function openModal() {
            const modal = document.getElementById('actionModal');
            modal.classList.add('active');
            document.getElementById('modalTitle').textContent = 'Neue Schnellaktion';
            document.getElementById('actionType').value = 'create';
            document.getElementById('submitBtn').innerHTML = '<i class="fas fa-check"></i> Erstellen';
            document.getElementById('actionForm').reset();
            selectIcon('fas fa-rocket');
            selectColor('#007AFF');
            
            // Focus first input
            setTimeout(() => {
                document.getElementById('actionTitle').focus();
            }, 300);
        }

        function closeModal() {
            const modal = document.getElementById('actionModal');
            modal.classList.remove('active');
        }

        function editAction(id) {
            const card = document.querySelector(`[data-id="${id}"]`);
            const title = card.querySelector('.action-title').textContent;
            const url = card.querySelector('.action-url').textContent;
            const icon = card.querySelector('.action-icon i').className;
            const colorStyle = card.querySelector('.action-icon').style.background;
            
            // Extract color from background style
            let color = '#007AFF';
            if (colorStyle.includes('rgb')) {
                color = rgbToHex(colorStyle);
            } else if (colorStyle.includes('#')) {
                color = colorStyle.match(/#[a-fA-F0-9]{6}/)?.[0] || '#007AFF';
            }
            
            document.getElementById('modalTitle').textContent = 'Schnellaktion bearbeiten';
            document.getElementById('actionType').value = 'update';
            document.getElementById('actionId').value = id;
            document.getElementById('actionTitle').value = title;
            document.getElementById('actionUrl').value = url;
            document.getElementById('submitBtn').innerHTML = '<i class="fas fa-save"></i> Speichern';
            
            selectIcon(icon);
            selectColor(color);
            
            document.getElementById('actionModal').classList.add('active');
            
            // Focus title input
            setTimeout(() => {
                document.getElementById('actionTitle').focus();
                document.getElementById('actionTitle').select();
            }, 300);
        }

        function selectColor(color) {
            document.getElementById('actionColor').value = color;
            document.querySelectorAll('.selection-option[data-color]').forEach(opt => {
                opt.classList.remove('selected');
            });
            
            const colorOption = document.querySelector(`[data-color="${color}"]`);
            if (colorOption) {
                colorOption.classList.add('selected');
            }
        }

        function selectIcon(icon) {
            document.getElementById('actionIcon').value = icon;
            document.querySelectorAll('.icon-option').forEach(opt => {
                opt.classList.remove('selected');
            });
            
            const iconOption = document.querySelector(`[data-icon="${icon}"]`);
            if (iconOption) {
                iconOption.classList.add('selected');
            }
        }

        // Helper function to convert RGB to HEX
        function rgbToHex(rgb) {
            const result = rgb.match(/\d+/g);
            if (result && result.length >= 3) {
                return "#" + ((1 << 24) + (parseInt(result[0]) << 16) + (parseInt(result[1]) << 8) + parseInt(result[2])).toString(16).slice(1);
            }
            return '#007AFF';
        }
        
        // Keyboard shortcuts
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                closeModal();
            }
            
            if ((e.metaKey || e.ctrlKey) && e.key === 'n') {
                e.preventDefault();
                openModal();
            }
        });
        
        // Close modal on background click
        document.getElementById('actionModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeModal();
            }
        });

        // Initialize default selections
        document.addEventListener('DOMContentLoaded', function() {
            selectIcon('fas fa-rocket');
            selectColor('#007AFF');
        });
    </script>
</body>
</html>