<?php
/**
 * 🚀 INTELLIGENT ADMIN DASHBOARD - Next Generation
 * Ultra-modernes Dashboard mit KI-gestützten Empfehlungen und individueller Konfiguration
 */

session_start();
require_once '../config/database_optimized.php';

// Check if user is logged in
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

$db = DatabaseOptimized::getInstance();
$conn = $db->getConnection();

// Get current user data
$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_name = 'Admin';
$current_user_role = 'admin';
$current_user_autohaus_id = null;
$current_user_autohaus_name = null;
$current_theme = 'light';

if ($current_user_id) {
    $user_data = $db->getUserData($current_user_id);
    if ($user_data) {
        $current_user_name = $user_data['full_name'] ?? $user_data['username'] ?? 'Benutzer';
        $current_user_role = $user_data['role'] ?? 'admin';
        $current_user_autohaus_id = $user_data['autohaus_id'];
        $current_user_autohaus_name = $user_data['autohaus_name'];
        $current_theme = $user_data['theme'] ?? 'light';
    }
}

// Check if user needs dashboard setup
$needs_setup = false;
$user_dashboard_config = null;
$show_setup_modal = false;

if ($current_user_id) {
    // First check if user has the needs_dashboard_setup flag
    $stmt = $conn->prepare("SELECT needs_dashboard_setup FROM admin_users WHERE id = ?");
    $stmt->execute([$current_user_id]);
    $user_needs_setup = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($user_needs_setup && $user_needs_setup['needs_dashboard_setup'] == 1) {
        $needs_setup = true;
        $show_setup_modal = true;
    } else {
        // Check if user has dashboard configuration
        $stmt = $conn->prepare("SELECT config_data FROM user_dashboard_config WHERE user_id = ?");
        $stmt->execute([$current_user_id]);
        $result = $stmt->fetch();
        
        if ($result) {
            $user_dashboard_config = json_decode($result['config_data'], true);
        } else {
            $needs_setup = true;
            $show_setup_modal = true;
        }
    }
}

// Handle setup form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'save_dashboard_config') {
    try {
        $config = [
            'layout' => $_POST['layout'] ?? 'default',
            'widgets' => json_decode($_POST['widgets'] ?? '[]', true),
            'quick_actions' => [], // Quick Actions are now managed via quick_actions_manager.php
            'theme' => $_POST['theme'] ?? 'light',
            'show_welcome' => isset($_POST['show_welcome']) ? 1 : 0,
            'auto_refresh' => isset($_POST['auto_refresh']) ? 1 : 0,
            'refresh_interval' => $_POST['refresh_interval'] ?? 300
        ];
        
        $stmt = $conn->prepare("
            INSERT INTO user_dashboard_config (user_id, config_data, created_at, updated_at) 
            VALUES (?, ?, NOW(), NOW())
            ON DUPLICATE KEY UPDATE 
            config_data = VALUES(config_data), updated_at = NOW()
        ");
        $stmt->execute([$current_user_id, json_encode($config)]);
        
        $stmt = $conn->prepare("UPDATE admin_users SET theme = ?, needs_dashboard_setup = 0 WHERE id = ?");
        $stmt->execute([$config['theme'], $current_user_id]);
        
        // Reload page to show new configuration
        header('Location: index.php');
        exit;
    } catch (Exception $e) {
        $error_message = 'Fehler: ' . $e->getMessage();
    }
}

// Get user's dashboard configuration
$dashboard_layout = $user_dashboard_config['layout'] ?? 'default';
$enabled_widgets = $user_dashboard_config['widgets'] ?? ['today_appointments', 'upcoming_events', 'quick_actions'];
// Quick Actions are now managed via quick_actions_manager.php
$enabled_quick_actions = [];
$show_welcome = $user_dashboard_config['show_welcome'] ?? true;
$auto_refresh = $user_dashboard_config['auto_refresh'] ?? false;
$refresh_interval = $user_dashboard_config['refresh_interval'] ?? 300;

// Apply user theme
$current_theme = $user_dashboard_config['theme'] ?? $current_theme;

// Helper function to check if widget is enabled
function isWidgetEnabled($widget_key, $user_config) {
    if (!isset($user_config['widgets']) || !is_array($user_config['widgets'])) {
        return false;
    }
    return in_array($widget_key, $user_config['widgets']);
}

// Get dashboard statistics
$stats = $db->getDashboardStats($current_user_autohaus_id, 'admin', true);

// Get today's appointments
$todayAppointments = $db->fetchAll("
    SELECT a.*, e.name as event_name
    FROM appointments a
    LEFT JOIN events e ON a.event_id = e.id
    WHERE a.slot_date = CURDATE()
    ORDER BY a.slot_time ASC
    LIMIT 10
");

// Get upcoming events
$upcomingEvents = $db->fetchAll("
    SELECT e.*, 
           COUNT(a.id) as appointment_count,
           COUNT(CASE WHEN a.status = 'booked' THEN 1 END) as booked_count
    FROM events e
    LEFT JOIN appointments a ON e.id = a.event_id
    WHERE e.event_date >= CURDATE() AND e.is_active = 1
    GROUP BY e.id
    ORDER BY e.event_date ASC
    LIMIT 5
");

// Get performance metrics
// REMOVED: Performance metrics query has been removed as requested

// Get recent storage activities
$recentStorage = $db->fetchAll("
    SELECT * FROM storage 
    ORDER BY created_at DESC 
    LIMIT 5
");

// Get Quick Actions from database
// Get user's selected quick actions from quick_actions_manager
$quick_actions = [];
if ($current_user_id) {
    // Load user's personal quick actions first
    $stmt = $conn->prepare("
        SELECT * FROM quick_actions 
        WHERE user_id = ? AND is_active = 1
        ORDER BY position_order ASC, id ASC
    ");
    $stmt->execute([$current_user_id]);
    $user_quick_actions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Load global quick actions (user_id IS NULL)
    $stmt = $conn->prepare("
        SELECT * FROM quick_actions 
        WHERE user_id IS NULL AND is_active = 1
        ORDER BY position_order ASC, id ASC
    ");
    $stmt->execute();
    $global_quick_actions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Combine user actions first, then global actions
    $quick_actions = array_merge($user_quick_actions, $global_quick_actions);
} else {
    // Fallback: Load all active quick actions
    $stmt = $conn->prepare("SELECT * FROM quick_actions WHERE is_active = 1 ORDER BY position_order ASC");
    $stmt->execute();
    $quick_actions = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Intelligent User Scoring System
// REMOVED: calculateIntelligentScore function has been removed as requested

// Calculate intelligent user score for current user (Deniz Uludag)
// REMOVED: Scoring system has been removed as requested

// Get user activity log
$userActivity = $db->fetchAll("
    SELECT 
        'appointment' as type,
        'Termin erstellt' as action,
        customer_name as description,
        created_at
    FROM appointments 
    WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
    ORDER BY created_at DESC
    LIMIT 10
");

// Setup data for modal
$available_widgets = [
    'today_appointments' => [
        'title' => 'Heutige Termine',
        'description' => 'Zeigt alle Termine für heute an',
        'icon' => 'fas fa-calendar-day',
        'color' => '#007AFF',
        'category' => 'essential',
        'importance' => 'high'
    ],
    'upcoming_events' => [
        'title' => 'Kommende Events',
        'description' => 'Zeigt die nächsten Events an',
        'icon' => 'fas fa-calendar-alt',
        'color' => '#34C759',
        'category' => 'operational',
        'importance' => 'medium'
    ],
    'quick_actions' => [
        'title' => 'Schnellaktionen',
        'description' => 'Ihre personalisierten Schnellzugriffe',
        'icon' => 'fas fa-rocket',
        'color' => '#AF52DE',
        'category' => 'essential',
        'importance' => 'high'
    ],
    'recent_storage' => [
        'title' => 'Letzte Lageraktivitäten',
        'description' => 'Zeigt die neuesten Einlagerungen an',
        'icon' => 'fas fa-warehouse',
        'color' => '#FF9500',
        'category' => 'operational',
        'importance' => 'medium'
    ],
    'performance_metrics' => [
        'title' => 'Performance-Metriken',
        'description' => 'Zeigt Statistiken und Kennzahlen an',
        'icon' => 'fas fa-chart-line',
        'color' => '#5856D6',
        'category' => 'analytics',
        'importance' => 'low'
    ]
];

$available_layouts = [
    'default' => [
        'name' => 'Standard Layout',
        'description' => 'Zwei-Spalten Layout mit Hauptinhalt und Sidebar',
        'best_for' => 'Tägliche Arbeit, ausgewogene Übersicht',
        'screen_size' => 'Desktop & Tablet'
    ],
    'compact' => [
        'name' => 'Kompaktes Layout',
        'description' => 'Ein-Spalten Layout für fokussierte Arbeit',
        'best_for' => 'Mobile Geräte, fokussierte Aufgaben',
        'screen_size' => 'Mobile & Tablet'
    ],
    'wide' => [
        'name' => 'Breites Layout',
        'description' => 'Drei-Spalten Layout für maximale Übersicht',
        'best_for' => 'Administratoren, große Bildschirme',
        'screen_size' => 'Desktop & Ultrawide'
    ]
];

// REMOVED: Quick action templates are now managed via quick_actions_manager.php

// Default config for new users
if (!$user_dashboard_config) {
    $user_dashboard_config = [
        'layout' => 'default',
        'widgets' => ['today_appointments', 'upcoming_events', 'quick_actions'],
        'quick_actions' => [], // Quick Actions are now managed via quick_actions_manager.php
        'theme' => 'light',
        'show_welcome' => true,
        'auto_refresh' => false,
        'refresh_interval' => 300
    ];
}

// Ensure user_dashboard_config is always available for the modal
$modal_config = $user_dashboard_config;
?>

<!DOCTYPE html>
<html lang="de" data-theme="<?php echo htmlspecialchars($current_theme); ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, user-scalable=no">
    <meta name="robots" content="noindex, nofollow">
    <title>🚀 Intelligent Dashboard • Boxenstop</title>
    
    <!-- Ultra Modern Design System -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    <link rel="stylesheet" href="assets/css/unified-sidebar.css">
    
    <style>
        /* 🚀 ULTRA MODERN DESIGN SYSTEM - 2024 */
        :root {
            /* Modern Color Palette */
            --primary: #6366f1;
            --primary-dark: #4f46e5;
            --secondary: #8b5cf6;
            --accent: #06b6d4;
            --success: #10b981;
            --warning: #f59e0b;
            --error: #ef4444;
            
            /* Neutral Colors */
            --gray-50: #f9fafb;
            --gray-100: #f3f4f6;
            --gray-200: #e5e7eb;
            --gray-300: #d1d5db;
            --gray-400: #9ca3af;
            --gray-500: #6b7280;
            --gray-600: #4b5563;
            --gray-700: #374151;
            --gray-800: #1f2937;
            --gray-900: #111827;
            
            /* Glass Effects */
            --glass-bg: rgba(255, 255, 255, 0.1);
            --glass-border: rgba(255, 255, 255, 0.2);
            --glass-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            
            /* Spacing */
            --space-1: 0.25rem;
            --space-2: 0.5rem;
            --space-3: 0.75rem;
            --space-4: 1rem;
            --space-5: 1.25rem;
            --space-6: 1.5rem;
            --space-8: 2rem;
            --space-10: 2.5rem;
            --space-12: 3rem;
            --space-16: 4rem;
            --space-20: 5rem;
            
            /* Border Radius */
            --radius-sm: 0.375rem;
            --radius-md: 0.5rem;
            --radius-lg: 0.75rem;
            --radius-xl: 1rem;
            --radius-2xl: 1.5rem;
            --radius-3xl: 2rem;
            
            /* Shadows */
            --shadow-sm: 0 1px 2px rgba(0, 0, 0, 0.05);
            --shadow-md: 0 4px 6px rgba(0, 0, 0, 0.1);
            --shadow-lg: 0 10px 15px rgba(0, 0, 0, 0.1);
            --shadow-xl: 0 20px 25px rgba(0, 0, 0, 0.1);
            --shadow-2xl: 0 25px 50px rgba(0, 0, 0, 0.25);
        }

        /* Light Theme */
        [data-theme="light"] {
            --glass-bg: rgba(255, 255, 255, 0.15);
            --glass-border: rgba(255, 255, 255, 0.25);
            --glass-shadow: 0 8px 32px rgba(0, 0, 0, 0.15);
        }

        /* Dark Theme */
        [data-theme="dark"] {
            --gray-50: #111827;
            --gray-100: #1f2937;
            --gray-200: #374151;
            --gray-300: #4b5563;
            --gray-400: #6b7280;
            --gray-500: #9ca3af;
            --gray-600: #d1d5db;
            --gray-700: #e5e7eb;
            --gray-800: #f3f4f6;
            --gray-900: #f9fafb;
            
            --glass-bg: rgba(0, 0, 0, 0.2);
            --glass-border: rgba(255, 255, 255, 0.1);
        }

        /* Reset & Base */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            min-height: 100vh;
            color: #1e293b;
            line-height: 1.6;
            overflow-x: hidden;
        }

        [data-theme="dark"] body {
            background: linear-gradient(135deg, #0f172a 0%, #1e293b 100%);
            color: #f1f5f9;
        }

        /* Main Container */
        .admin-container {
            display: flex;
            min-height: 100vh;
        }

        .admin-main {
            flex: 1;
            padding: var(--space-8);
            background: transparent;
            min-height: 100vh;
            transition: margin-left 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }

        .modern-sidebar.collapsed + .admin-main {
            margin-left: 80px;
        }

        /* Dashboard Layouts */
        .dashboard-grid {
            display: grid;
            gap: var(--space-6);
            min-height: calc(100vh - var(--space-16));
        }

        .dashboard-grid.layout-default {
            grid-template-columns: 2fr 1fr;
            grid-template-areas: 
                "header header"
                "main sidebar"
                "widgets sidebar";
        }

        .dashboard-grid.layout-compact {
            grid-template-columns: 1fr;
            grid-template-areas: 
                "header"
                "main"
                "widgets";
        }

        .dashboard-grid.layout-wide {
            grid-template-columns: 1fr 1fr 1fr;
            grid-template-areas: 
                "header header header"
                "main main sidebar"
                "widgets widgets sidebar";
        }

        /* Header */
        .dashboard-header {
            grid-area: header;
            background: rgba(255, 255, 255, 0.8);
            backdrop-filter: blur(20px);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-2xl);
            padding: var(--space-8);
            margin-bottom: var(--space-6);
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            position: relative;
            overflow: hidden;
        }

        [data-theme="dark"] .dashboard-header {
            background: rgba(0, 0, 0, 0.3);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .dashboard-header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, var(--primary), var(--secondary), var(--accent));
            opacity: 0.8;
        }

        .header-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: relative;
            z-index: 1;
        }

        .header-left h1 {
            font-size: clamp(2rem, 4vw, 3rem);
            font-weight: 800;
            color: #1e293b;
            margin-bottom: var(--space-2);
            letter-spacing: -0.02em;
        }

        [data-theme="dark"] .header-left h1 {
            color: #f1f5f9;
        }

        .header-left p {
            color: #64748b;
            font-size: 1.1rem;
        }

        [data-theme="dark"] .header-left p {
            color: #94a3b8;
        }

        .user-info {
            display: flex;
            align-items: center;
            gap: var(--space-4);
            background: rgba(255, 255, 255, 0.4);
            padding: var(--space-4) var(--space-6);
            border-radius: var(--radius-xl);
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.3);
        }

        [data-theme="dark"] .user-info {
            background: rgba(0, 0, 0, 0.3);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .user-avatar {
            width: 48px;
            height: 48px;
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.25rem;
            color: white;
            font-weight: 700;
        }

        .user-details h3 {
            color: #1e293b;
            font-weight: 600;
            margin-bottom: var(--space-1);
        }

        [data-theme="dark"] .user-details h3 {
            color: #f1f5f9;
        }

        .user-details p {
            color: #64748b;
            font-size: 0.875rem;
        }

        [data-theme="dark"] .user-details p {
            color: #94a3b8;
        }

        .header-actions {
            display: flex;
            gap: var(--space-3);
        }

        .btn-header {
            background: rgba(255, 255, 255, 0.6);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-lg);
            padding: var(--space-3) var(--space-5);
            color: #1e293b;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            backdrop-filter: blur(10px);
            display: flex;
            align-items: center;
            gap: var(--space-2);
            text-decoration: none;
        }

        [data-theme="dark"] .btn-header {
            background: rgba(0, 0, 0, 0.3);
            border: 1px solid rgba(255, 255, 255, 0.1);
            color: #f1f5f9;
        }

        .btn-header:hover {
            background: rgba(255, 255, 255, 0.8);
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }

        [data-theme="dark"] .btn-header:hover {
            background: rgba(0, 0, 0, 0.4);
        }

        /* Main Content Area */
        .main-content {
            grid-area: main;
            display: grid;
            gap: var(--space-6);
        }

        .sidebar-content {
            grid-area: sidebar;
            display: grid;
            gap: var(--space-6);
        }

        /* Widget System */
        .widget {
            background: rgba(255, 255, 255, 0.8);
            backdrop-filter: blur(20px);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-2xl);
            padding: var(--space-6);
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            position: relative;
            overflow: hidden;
            transition: all 0.3s ease;
        }

        [data-theme="dark"] .widget {
            background: rgba(0, 0, 0, 0.3);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .widget:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-xl);
        }

        .widget::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 3px;
            background: linear-gradient(90deg, var(--primary), var(--secondary));
            opacity: 0;
            transition: opacity 0.3s ease;
        }

        .widget:hover::before {
            opacity: 1;
        }

        .widget-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: var(--space-4);
        }

        .widget-title {
            font-size: 1.25rem;
            font-weight: 700;
            color: #1e293b;
            display: flex;
            align-items: center;
            gap: var(--space-3);
        }

        [data-theme="dark"] .widget-title {
            color: #f1f5f9;
        }

        .widget-icon {
            width: 32px;
            height: 32px;
            border-radius: var(--radius-lg);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1rem;
            color: white;
        }

        .widget-actions {
            display: flex;
            gap: var(--space-2);
        }

        .widget-btn {
            width: 32px;
            height: 32px;
            border: none;
            border-radius: var(--radius-md);
            background: rgba(255, 255, 255, 0.1);
            color: white;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .widget-btn:hover {
            background: rgba(255, 255, 255, 0.2);
        }

        /* Statistics Cards */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: var(--space-4);
            margin-bottom: var(--space-6);
        }

        .stat-card {
            background: rgba(255, 255, 255, 0.6);
            border-radius: var(--radius-xl);
            padding: var(--space-5);
            text-align: center;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
            border: 1px solid rgba(255, 255, 255, 0.3);
        }

        [data-theme="dark"] .stat-card {
            background: rgba(0, 0, 0, 0.3);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .stat-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: linear-gradient(135deg, rgba(255, 255, 255, 0.1), rgba(255, 255, 255, 0.05));
            opacity: 0;
            transition: opacity 0.3s ease;
        }

        .stat-card:hover::before {
            opacity: 1;
        }

        .stat-card:hover {
            transform: translateY(-4px);
            box-shadow: var(--shadow-lg);
        }

        .stat-number {
            font-size: 2.5rem;
            font-weight: 800;
            color: #1e293b;
            margin-bottom: var(--space-2);
            position: relative;
            z-index: 1;
        }

        [data-theme="dark"] .stat-number {
            color: #f1f5f9;
        }

        .stat-label {
            color: #64748b;
            font-weight: 600;
            position: relative;
            z-index: 1;
        }

        [data-theme="dark"] .stat-label {
            color: #94a3b8;
        }

        .stat-icon {
            position: absolute;
            top: var(--space-4);
            right: var(--space-4);
            width: 40px;
            height: 40px;
            border-radius: var(--radius-lg);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.25rem;
            color: white;
            opacity: 0.3;
        }

        /* Quick Actions */
        .quick-actions-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
            gap: var(--space-4);
        }

        .quick-action {
            background: rgba(255, 255, 255, 0.6);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-xl);
            padding: var(--space-5);
            text-align: center;
            transition: all 0.3s ease;
            text-decoration: none;
            color: inherit;
            position: relative;
            overflow: hidden;
        }

        [data-theme="dark"] .quick-action {
            background: rgba(0, 0, 0, 0.3);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .quick-action::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: linear-gradient(135deg, rgba(255, 255, 255, 0.1), rgba(255, 255, 255, 0.05));
            opacity: 0;
            transition: opacity 0.3s ease;
        }

        .quick-action:hover {
            transform: translateY(-4px);
            box-shadow: var(--shadow-lg);
            text-decoration: none;
            color: inherit;
        }

        .quick-action:hover::before {
            opacity: 1;
        }

        .quick-action-icon {
            width: 48px;
            height: 48px;
            border-radius: var(--radius-lg);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            color: white;
            margin: 0 auto var(--space-3);
            position: relative;
            z-index: 1;
        }

        .quick-action-title {
            font-weight: 600;
            color: #1e293b;
            position: relative;
            z-index: 1;
        }

        [data-theme="dark"] .quick-action-title {
            color: #f1f5f9;
        }

        /* Quick Action Empty State */
        .quick-action-empty {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            padding: var(--space-6);
            text-align: center;
            color: var(--label-secondary);
            border: 2px dashed var(--separator-non-opaque);
            border-radius: var(--radius-lg);
            transition: all 0.2s ease;
        }

        .quick-action-empty:hover {
            border-color: var(--system-blue);
            color: var(--system-blue);
        }

        .quick-action-empty .quick-action-icon {
            margin-bottom: var(--space-3);
            opacity: 0.6;
        }

        .quick-action-empty .quick-action-title {
            font-size: 14px;
            font-weight: 500;
        }

.user-select option {
    background: #1e293b;
    color: white;
    padding: var(--space-2);
}


/* Modal Styles */
.modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100vw;
    height: 100vh;
    background: rgba(0, 0, 0, 0.7);
    backdrop-filter: blur(10px);
    z-index: 10000;
    display: flex;
    align-items: center;
    justify-content: center;
    padding: var(--space-4);
    box-sizing: border-box;
}

.modal-content {
    background: rgba(255, 255, 255, 0.95);
    border-radius: var(--radius-xl);
    border: 1px solid rgba(255, 255, 255, 0.3);
    box-shadow: var(--shadow-xl);
    max-width: 90vw;
    max-height: 90vh;
    width: auto;
    min-width: 600px;
    overflow-y: auto;
    backdrop-filter: blur(20px);
    margin: auto;
    position: relative;
}

[data-theme="dark"] .modal-content {
    background: rgba(0, 0, 0, 0.8);
    border: 1px solid rgba(255, 255, 255, 0.1);
}

.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: var(--space-5);
    border-bottom: 1px solid rgba(255, 255, 255, 0.2);
}

.modal-header h3 {
    font-size: 1.25rem;
    font-weight: 700;
    color: #1e293b;
    margin: 0;
}

[data-theme="dark"] .modal-header h3 {
    color: #f1f5f9;
}

.modal-close {
    background: none;
    border: none;
    font-size: 1.5rem;
    color: #64748b;
    cursor: pointer;
    padding: var(--space-2);
    border-radius: var(--radius-md);
    transition: all 0.3s ease;
}

.modal-close:hover {
    background: rgba(255, 255, 255, 0.2);
    color: #1e293b;
}

[data-theme="dark"] .modal-close {
    color: #94a3b8;
}

[data-theme="dark"] .modal-close:hover {
    background: rgba(255, 255, 255, 0.1);
    color: #f1f5f9;
}

.modal-form {
    padding: 0;
}

/* Responsive Modal */
@media (max-width: 768px) {
    .modal {
        padding: var(--space-2);
    }
    
    .modal-content {
        min-width: 95vw;
        max-width: 95vw;
        max-height: 95vh;
    }
}

@media (max-width: 480px) {
    .modal-content {
        min-width: 100vw;
        max-width: 100vw;
        max-height: 100vh;
        border-radius: 0;
    }
}

/* Score Details Modal */
.score-details-content {
    padding: var(--space-4);
}

        .score-category {
            background: rgba(255, 255, 255, 0.6);
            border-radius: var(--radius-xl);
            padding: var(--space-5);
            margin-bottom: var(--space-4);
            border: 1px solid rgba(255, 255, 255, 0.3);
            transition: all 0.3s ease;
        }

        [data-theme="dark"] .score-category {
            background: rgba(0, 0, 0, 0.3);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .score-category:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }

        .category-header {
            display: flex;
            align-items: center;
            gap: var(--space-4);
            margin-bottom: var(--space-4);
        }

        .category-icon {
            width: 48px;
            height: 48px;
            border-radius: var(--radius-lg);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.25rem;
            color: white;
            flex-shrink: 0;
        }

        .category-info {
            flex: 1;
        }

        .category-info h4 {
            font-size: 1.125rem;
            font-weight: 700;
            color: #1e293b;
            margin-bottom: var(--space-1);
        }

        [data-theme="dark"] .category-info h4 {
            color: #f1f5f9;
        }

        .category-info p {
            color: #64748b;
            font-size: 0.875rem;
        }

        [data-theme="dark"] .category-info p {
            color: #94a3b8;
        }

        .category-score {
            font-size: 1.5rem;
            font-weight: 800;
            color: #1e293b;
            background: rgba(255, 255, 255, 0.8);
            padding: var(--space-2) var(--space-4);
            border-radius: var(--radius-lg);
            border: 1px solid rgba(255, 255, 255, 0.3);
        }

        [data-theme="dark"] .category-score {
            color: #f1f5f9;
            background: rgba(0, 0, 0, 0.3);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .category-details {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: var(--space-3);
        }

        .detail-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: var(--space-2) var(--space-3);
            background: rgba(255, 255, 255, 0.4);
            border-radius: var(--radius-md);
            border: 1px solid rgba(255, 255, 255, 0.2);
        }

        [data-theme="dark"] .detail-item {
            background: rgba(0, 0, 0, 0.2);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .detail-label {
            font-size: 0.875rem;
            color: #64748b;
            font-weight: 500;
        }

        [data-theme="dark"] .detail-label {
            color: #94a3b8;
        }

        .detail-value {
            font-size: 0.875rem;
            font-weight: 700;
            color: #1e293b;
        }

        [data-theme="dark"] .detail-value {
            color: #f1f5f9;
        }

        .score-tips {
            background: linear-gradient(135deg, #10b981, #059669);
            border-radius: var(--radius-xl);
            padding: var(--space-5);
            color: white;
            margin-top: var(--space-6);
        }

        .score-tips h4 {
            font-size: 1.125rem;
            font-weight: 700;
            margin-bottom: var(--space-3);
            display: flex;
            align-items: center;
            gap: var(--space-2);
        }

        .score-tips ul {
            list-style: none;
            padding: 0;
        }

        .score-tips li {
            padding: var(--space-2) 0;
            border-bottom: 1px solid rgba(255, 255, 255, 0.2);
        }

        .score-tips li:last-child {
            border-bottom: none;
        }

        .score-tips strong {
            color: rgba(255, 255, 255, 0.9);
        }

        /* Activity Feed */
        .activity-item {
            display: flex;
            align-items: center;
            gap: var(--space-3);
            padding: var(--space-3);
            background: rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-lg);
            margin-bottom: var(--space-2);
            transition: all 0.3s ease;
            border: 1px solid rgba(255, 255, 255, 0.2);
        }

        .activity-item:hover {
            background: rgba(255, 255, 255, 0.4);
        }

        [data-theme="dark"] .activity-item {
            background: rgba(0, 0, 0, 0.2);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        [data-theme="dark"] .activity-item:hover {
            background: rgba(0, 0, 0, 0.3);
        }

        .activity-icon {
            width: 32px;
            height: 32px;
            border-radius: var(--radius-md);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.875rem;
            color: white;
        }

        .activity-content {
            flex: 1;
        }

        .activity-title {
            color: #1e293b;
            font-weight: 600;
            font-size: 0.875rem;
            margin-bottom: var(--space-1);
        }

        [data-theme="dark"] .activity-title {
            color: #f1f5f9;
        }

        .activity-description {
            color: #64748b;
            font-size: 0.75rem;
        }

        [data-theme="dark"] .activity-description {
            color: #94a3b8;
        }

        .activity-time {
            color: #94a3b8;
            font-size: 0.75rem;
        }

        [data-theme="dark"] .activity-time {
            color: #64748b;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .admin-main {
                padding: var(--space-4);
            }

            .dashboard-grid.layout-default,
            .dashboard-grid.layout-wide {
                grid-template-columns: 1fr;
                grid-template-areas: 
                    "header"
                    "main"
                    "sidebar"
                    "widgets";
            }

            .header-content {
                flex-direction: column;
                gap: var(--space-4);
                text-align: center;
            }

            .stats-grid {
                grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            }

            .quick-actions-grid {
                grid-template-columns: repeat(auto-fit, minmax(100px, 1fr));
            }
        }

        /* Loading States */
        .loading {
            opacity: 0.6;
            pointer-events: none;
        }

        .loading::after {
            content: '';
            position: absolute;
            top: 50%;
            left: 50%;
            width: 20px;
            height: 20px;
            margin: -10px 0 0 -10px;
            border: 2px solid rgba(255, 255, 255, 0.3);
            border-top: 2px solid white;
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        /* Animations */
        .fade-in {
            animation: fadeIn 0.6s ease-out;
        }

        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .slide-in {
            animation: slideIn 0.6s ease-out;
        }

        @keyframes slideIn {
            from { opacity: 0; transform: translateX(-20px); }
            to { opacity: 1; transform: translateX(0); }
        }

        /* Welcome Message */
        .welcome-message {
            background: linear-gradient(135deg, #10b981, #059669);
            border-radius: var(--radius-2xl);
            padding: var(--space-6);
            margin-bottom: var(--space-6);
            color: white;
            position: relative;
            overflow: hidden;
        }

        .welcome-message::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="welcome" width="30" height="30" patternUnits="userSpaceOnUse"><circle cx="15" cy="15" r="2" fill="rgba(255,255,255,0.1)"/></pattern></defs><rect width="100" height="100" fill="url(%23welcome)"/></svg>');
            opacity: 0.3;
        }

        .welcome-content {
            position: relative;
            z-index: 1;
            text-align: center;
        }

        .welcome-title {
            font-size: 1.5rem;
            font-weight: 700;
            margin-bottom: var(--space-2);
        }

        .welcome-text {
            opacity: 0.9;
            margin-bottom: var(--space-4);
        }

        .welcome-actions {
            display: flex;
            gap: var(--space-3);
            justify-content: center;
        }

        .btn-welcome {
            background: rgba(255, 255, 255, 0.2);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-lg);
            padding: var(--space-3) var(--space-5);
            color: white;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            backdrop-filter: blur(10px);
            text-decoration: none;
            display: flex;
            align-items: center;
            gap: var(--space-2);
        }

        .btn-welcome:hover {
            background: rgba(255, 255, 255, 0.3);
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
            text-decoration: none;
            color: white;
        }

        /* Setup Modal */
        .setup-modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.8);
            backdrop-filter: blur(10px);
            z-index: 9999;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: var(--space-4);
            animation: fadeIn 0.3s ease-out;
        }

        .setup-modal {
            background: rgba(255, 255, 255, 0.9);
            backdrop-filter: blur(20px);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-2xl);
            max-width: 1000px;
            width: 100%;
            max-height: 90vh;
            overflow-y: auto;
            box-shadow: var(--shadow-2xl);
            position: relative;
            animation: slideIn 0.3s ease-out;
        }

        [data-theme="dark"] .setup-modal {
            background: rgba(0, 0, 0, 0.4);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .setup-modal-header {
            padding: var(--space-8);
            text-align: center;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }

        .setup-modal-header h2 {
            font-size: 2rem;
            font-weight: 800;
            color: #1e293b;
            margin-bottom: var(--space-2);
        }

        [data-theme="dark"] .setup-modal-header h2 {
            color: #f1f5f9;
        }

        .setup-modal-header p {
            color: #64748b;
            font-size: 1.1rem;
        }

        [data-theme="dark"] .setup-modal-header p {
            color: #94a3b8;
        }

        .setup-section {
            padding: var(--space-6);
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }

        .setup-section:last-child {
            border-bottom: none;
        }

        .setup-section h3 {
            font-size: 1.5rem;
            font-weight: 700;
            color: #1e293b;
            margin-bottom: var(--space-4);
        }

        [data-theme="dark"] .setup-section h3 {
            color: #f1f5f9;
        }

        .setup-section .layout-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: var(--space-4);
        }

        .setup-section .layout-card {
            background: rgba(255, 255, 255, 0.6);
            border: 2px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-xl);
            padding: var(--space-5);
            cursor: pointer;
            transition: all 0.3s ease;
            position: relative;
            user-select: none;
            pointer-events: auto;
        }

        [data-theme="dark"] .setup-section .layout-card {
            background: rgba(0, 0, 0, 0.3);
            border: 2px solid rgba(255, 255, 255, 0.1);
        }

        .setup-section .layout-card:hover {
            border-color: #3b82f6;
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(59, 130, 246, 0.2);
        }

        .setup-section .layout-card:active {
            transform: translateY(0);
            box-shadow: 0 4px 12px rgba(59, 130, 246, 0.3);
        }

        .setup-section .layout-card.selected {
            border-color: #3b82f6;
            background: rgba(59, 130, 246, 0.1);
            box-shadow: 0 4px 12px rgba(59, 130, 246, 0.2);
        }

        .setup-section .layout-card.selected::after {
            content: '✓';
            position: absolute;
            top: var(--space-3);
            right: var(--space-3);
            width: 24px;
            height: 24px;
            background: var(--primary);
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
            font-size: 0.875rem;
        }

        .setup-section .layout-preview {
            width: 100%;
            height: 80px;
            background: rgba(255, 255, 255, 0.1);
            border-radius: var(--radius-lg);
            margin-bottom: var(--space-3);
            position: relative;
            overflow: hidden;
        }

        /* Layout Preview Visualizations */
        .setup-section .layout-preview.default::before {
            content: '';
            position: absolute;
            top: 10px;
            left: 10px;
            right: 10px;
            bottom: 10px;
            background: linear-gradient(90deg, 
                rgba(59, 130, 246, 0.3) 0%, 
                rgba(59, 130, 246, 0.3) 65%, 
                rgba(59, 130, 246, 0.1) 65%, 
                rgba(59, 130, 246, 0.1) 100%);
            border-radius: var(--radius-sm);
        }

        .setup-section .layout-preview.compact::before {
            content: '';
            position: absolute;
            top: 10px;
            left: 10px;
            right: 10px;
            bottom: 10px;
            background: rgba(59, 130, 246, 0.3);
            border-radius: var(--radius-sm);
        }

        .setup-section .layout-preview.wide::before {
            content: '';
            position: absolute;
            top: 10px;
            left: 10px;
            right: 10px;
            bottom: 10px;
            background: linear-gradient(90deg, 
                rgba(59, 130, 246, 0.3) 0%, 
                rgba(59, 130, 246, 0.3) 30%, 
                rgba(59, 130, 246, 0.2) 30%, 
                rgba(59, 130, 246, 0.2) 65%, 
                rgba(59, 130, 246, 0.1) 65%, 
                rgba(59, 130, 246, 0.1) 100%);
            border-radius: var(--radius-sm);
        }

        .setup-section .layout-name {
            font-size: 1.125rem;
            font-weight: 600;
            color: #1e293b;
            margin-bottom: var(--space-2);
        }

        [data-theme="dark"] .setup-section .layout-name {
            color: #f1f5f9;
        }

        .setup-section .layout-description {
            color: #64748b;
            font-size: 0.875rem;
            margin-bottom: var(--space-2);
        }

        [data-theme="dark"] .setup-section .layout-description {
            color: #94a3b8;
        }

        .setup-section .layout-meta {
            display: flex;
            justify-content: space-between;
            align-items: center;
            font-size: 0.75rem;
            color: rgba(255, 255, 255, 0.6);
        }

        .setup-section .layout-tag {
            background: rgba(255, 255, 255, 0.2);
            padding: var(--space-1) var(--space-2);
            border-radius: var(--radius-sm);
        }

        .setup-section .widget-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: var(--space-4);
        }

        .setup-section .widget-card {
            background: rgba(255, 255, 255, 0.6);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-lg);
            padding: var(--space-4);
            cursor: pointer;
            transition: all 0.3s ease;
            position: relative;
            user-select: none;
            pointer-events: auto;
        }

        [data-theme="dark"] .setup-section .widget-card {
            background: rgba(0, 0, 0, 0.3);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .setup-section .widget-card:hover {
            border-color: #3b82f6;
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(59, 130, 246, 0.2);
        }

        .setup-section .widget-card:active {
            transform: translateY(0);
            box-shadow: 0 4px 12px rgba(59, 130, 246, 0.3);
        }

        .setup-section .widget-card.selected {
            border-color: #3b82f6;
            background: rgba(59, 130, 246, 0.1);
            box-shadow: 0 4px 12px rgba(59, 130, 246, 0.2);
        }

        .setup-section .widget-card.selected::after {
            content: '✓';
            position: absolute;
            top: var(--space-3);
            right: var(--space-3);
            width: 20px;
            height: 20px;
            background: var(--primary);
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
            font-size: 0.75rem;
        }

        .setup-section .widget-header {
            display: flex;
            align-items: center;
            gap: var(--space-3);
        }

        .setup-section .widget-icon {
            width: 40px;
            height: 40px;
            border-radius: var(--radius-lg);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1rem;
            color: white;
        }

        .setup-section .widget-info h4 {
            font-size: 1rem;
            font-weight: 600;
            color: #1e293b;
            margin-bottom: var(--space-1);
        }

        [data-theme="dark"] .setup-section .widget-info h4 {
            color: #f1f5f9;
        }

        .setup-section .widget-info p {
            color: #64748b;
            font-size: 0.875rem;
        }

        [data-theme="dark"] .setup-section .widget-info p {
            color: #94a3b8;
        }

        .setup-section .widget-importance {
            margin-left: auto;
            padding: var(--space-1) var(--space-2);
            border-radius: var(--radius-sm);
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: uppercase;
        }

        .setup-section .importance-high {
            background: rgba(239, 68, 68, 0.2);
            color: #fca5a5;
        }

        .setup-section .importance-medium {
            background: rgba(245, 158, 11, 0.2);
            color: #fcd34d;
        }

        .setup-section .importance-low {
            background: rgba(107, 114, 128, 0.2);
            color: #d1d5db;
        }

        .setup-section .quick-actions-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: var(--space-4);
        }

        .setup-section .quick-action-card {
            background: rgba(255, 255, 255, 0.6);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-lg);
            padding: var(--space-4);
            cursor: pointer;
            transition: all 0.3s ease;
            text-align: center;
            position: relative;
            user-select: none;
            pointer-events: auto;
        }

        [data-theme="dark"] .setup-section .quick-action-card {
            background: rgba(0, 0, 0, 0.3);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .setup-section .quick-action-card:hover {
            border-color: #3b82f6;
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(59, 130, 246, 0.2);
        }

        .setup-section .quick-action-card:active {
            transform: translateY(0);
            box-shadow: 0 4px 12px rgba(59, 130, 246, 0.3);
        }

        .setup-section .quick-action-card.selected {
            border-color: #3b82f6;
            background: rgba(59, 130, 246, 0.1);
            box-shadow: 0 4px 12px rgba(59, 130, 246, 0.2);
        }

        .setup-section .quick-action-card.selected::after {
            content: '✓';
            position: absolute;
            top: var(--space-3);
            right: var(--space-3);
            width: 20px;
            height: 20px;
            background: var(--primary);
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
            font-size: 0.75rem;
        }

        .setup-section .quick-action-icon {
            width: 48px;
            height: 48px;
            border-radius: var(--radius-lg);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.25rem;
            color: white;
            margin: 0 auto var(--space-3);
        }

        .setup-section .quick-action-title {
            font-size: 0.875rem;
            font-weight: 600;
            color: #1e293b;
        }

        [data-theme="dark"] .setup-section .quick-action-title {
            color: #f1f5f9;
        }

        .setup-section .form-group {
            margin-bottom: var(--space-4);
        }

        .setup-section .form-group label {
            display: block;
            font-size: 0.875rem;
            font-weight: 600;
            color: #1e293b;
            margin-bottom: var(--space-2);
        }

        [data-theme="dark"] .setup-section .form-group label {
            color: #f1f5f9;
        }

        .setup-section .form-control {
            width: 100%;
            padding: var(--space-3);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-lg);
            background: rgba(255, 255, 255, 0.6);
            color: #1e293b;
            font-size: 0.875rem;
            transition: all 0.3s ease;
        }

        .setup-section .form-control:hover {
            border-color: #3b82f6;
            background: rgba(255, 255, 255, 0.8);
        }

        [data-theme="dark"] .setup-section .form-control {
            background: rgba(0, 0, 0, 0.3);
            border: 1px solid rgba(255, 255, 255, 0.1);
            color: #f1f5f9;
        }

        [data-theme="dark"] .setup-section .form-control:hover {
            border-color: #3b82f6;
            background: rgba(0, 0, 0, 0.4);
        }

        .setup-section .form-control:focus {
            outline: none;
            border-color: #3b82f6;
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
            background: rgba(255, 255, 255, 0.8);
        }

        [data-theme="dark"] .setup-section .form-control:focus {
            background: rgba(0, 0, 0, 0.4);
        }

        .setup-section .checkbox-group {
            display: flex;
            align-items: center;
            gap: var(--space-3);
            margin-bottom: var(--space-3);
        }

        .setup-section .checkbox {
            width: 24px;
            height: 24px;
            border: 2px solid rgba(255, 255, 255, 0.3);
            border-radius: var(--radius-sm);
            cursor: pointer;
            position: relative;
            transition: all 0.3s ease;
            flex-shrink: 0;
            background: rgba(255, 255, 255, 0.1);
            user-select: none;
            pointer-events: auto;
        }

        .setup-section .checkbox.checked {
            background: #3b82f6;
            border-color: #3b82f6;
            box-shadow: 0 4px 12px rgba(59, 130, 246, 0.3);
        }

        .setup-section .checkbox:hover {
            border-color: #3b82f6;
            background: rgba(59, 130, 246, 0.1);
        }

        .setup-section .checkbox.checked::after {
            content: '✓';
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            color: white;
            font-size: 0.875rem;
            font-weight: 700;
        }

        .setup-section .checkbox-label {
            color: #1e293b;
            cursor: pointer;
            font-weight: 500;
            font-size: 0.875rem;
        }

        [data-theme="dark"] .setup-section .checkbox-label {
            color: #f1f5f9;
        }

        .setup-section .setup-actions {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: var(--space-6);
            padding-top: var(--space-4);
            border-top: 1px solid rgba(255, 255, 255, 0.1);
        }

        .setup-section .btn {
            padding: var(--space-4) var(--space-6);
            border-radius: var(--radius-lg);
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: var(--space-2);
            text-decoration: none;
            border: none;
            font-size: 1rem;
            min-height: 48px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        }

        /* Button Classes */
        .btn {
            padding: var(--space-4) var(--space-6);
            border-radius: var(--radius-lg);
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: var(--space-2);
            text-decoration: none;
            border: none;
            font-size: 1rem;
            min-height: 48px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        }

        /* Ensure buttons are properly styled */
        button.btn {
            padding: var(--space-4) var(--space-6);
            border-radius: var(--radius-lg);
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: var(--space-2);
            text-decoration: none;
            border: none;
            font-size: 1rem;
            min-height: 48px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        }

        .btn-primary {
            background: linear-gradient(135deg, #3b82f6, #1d4ed8);
            color: white;
            box-shadow: var(--shadow-lg);
        }

        button.btn-primary {
            background: linear-gradient(135deg, #3b82f6, #1d4ed8);
            color: white;
            box-shadow: var(--shadow-lg);
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(59, 130, 246, 0.4);
            background: linear-gradient(135deg, #2563eb, #1e40af);
        }

        button.btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(59, 130, 246, 0.4);
            background: linear-gradient(135deg, #2563eb, #1e40af);
        }

        .btn-secondary {
            background: rgba(255, 255, 255, 0.6);
            color: #1e293b;
            border: 1px solid rgba(255, 255, 255, 0.3);
        }

        button.btn-secondary {
            background: rgba(255, 255, 255, 0.6);
            color: #1e293b;
            border: 1px solid rgba(255, 255, 255, 0.3);
        }

        [data-theme="dark"] .btn-secondary {
            background: rgba(0, 0, 0, 0.3);
            color: #f1f5f9;
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        [data-theme="dark"] button.btn-secondary {
            background: rgba(0, 0, 0, 0.3);
            color: #f1f5f9;
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .btn-secondary:hover {
            background: rgba(255, 255, 255, 0.8);
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
        }

        button.btn-secondary:hover {
            background: rgba(255, 255, 255, 0.8);
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
        }

        [data-theme="dark"] .btn-secondary:hover {
            background: rgba(0, 0, 0, 0.4);
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.3);
        }

        [data-theme="dark"] button.btn-secondary:hover {
            background: rgba(0, 0, 0, 0.4);
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.3);
        }

        .setup-section .btn-primary {
            background: linear-gradient(135deg, #3b82f6, #1d4ed8);
            color: white;
            box-shadow: var(--shadow-lg);
        }

        .setup-section .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(59, 130, 246, 0.4);
            background: linear-gradient(135deg, #2563eb, #1e40af);
        }

        .setup-section .btn-secondary {
            background: rgba(255, 255, 255, 0.6);
            color: #1e293b;
            border: 1px solid rgba(255, 255, 255, 0.3);
        }

        [data-theme="dark"] .setup-section .btn-secondary {
            background: rgba(0, 0, 0, 0.3);
            color: #f1f5f9;
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .setup-section .btn-secondary:hover {
            background: rgba(255, 255, 255, 0.8);
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
        }

        [data-theme="dark"] .setup-section .btn-secondary:hover {
            background: rgba(0, 0, 0, 0.4);
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.3);
        }

        @media (max-width: 768px) {
            .setup-modal {
                margin: var(--space-4);
                max-height: calc(100vh - var(--space-8));
            }

            .setup-section .layout-grid,
            .setup-section .widget-grid,
            .setup-section .quick-actions-grid {
                grid-template-columns: 1fr;
            }

            .setup-section .setup-actions {
                flex-direction: column;
                gap: var(--space-3);
            }

            .setup-section .btn {
                width: 100%;
                justify-content: center;
            }
        }
    </style>
</head>
<body class="admin-body" data-theme="<?php echo htmlspecialchars($current_theme); ?>">
    <div class="admin-container">
        <!-- Include Unified Sidebar -->
        <?php include 'includes/sidebar.php'; ?>

        <!-- Main Content -->
        <main class="admin-main">
            <div class="dashboard-grid layout-<?php echo $dashboard_layout; ?>">
                <!-- Header -->
                <header class="dashboard-header fade-in">
                    <div class="header-content">
                        <div class="header-left">
                            <h1>🚀 Intelligent Dashboard</h1>
                            <p>Willkommen zurück, <?php echo htmlspecialchars($current_user_name); ?>!</p>
                        </div>
                        <div class="header-actions">
                            <button class="btn-header" onclick="openSetupModal()">
                                <i class="fas fa-cog"></i>
                                Anpassen
                            </button>
                            <button class="btn-header" onclick="toggleTheme()">
                                <i class="fas fa-moon"></i>
                                Theme
                            </button>
                        </div>
                    </div>
                </header>

                <!-- Welcome Message -->
                <?php if ($show_welcome): ?>
                    <div class="welcome-message slide-in">
                        <div class="welcome-content">
                            <h2 class="welcome-title">🎉 Willkommen in Ihrem intelligenten Dashboard!</h2>
                            <p class="welcome-text">Ihr Dashboard wurde basierend auf Ihrer Rolle als <strong><?php echo ucfirst($current_user_role); ?></strong> optimiert.</p>
                            <div class="welcome-actions">
                                <a href="setup_user_dashboard.php" class="btn-welcome">
                                    <i class="fas fa-magic"></i>
                                    KI-Empfehlungen anzeigen
                                </a>
                                <button class="btn-welcome" onclick="hideWelcome()">
                                    <i class="fas fa-times"></i>
                                    Ausblenden
                                </button>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>

                <!-- Main Content Area -->
                <div class="main-content">
                    <!-- Statistics -->
                    <div class="widget fade-in">
                        <div class="widget-header">
                            <h3 class="widget-title">
                                <div class="widget-icon" style="background: var(--primary);">
                                    <i class="fas fa-chart-line"></i>
                                </div>
                                Statistiken
                            </h3>
                            <div class="widget-actions">
                                <button class="widget-btn" onclick="refreshStats()">
                                    <i class="fas fa-sync-alt"></i>
                                </button>
                            </div>
                        </div>
                        <div class="stats-grid">
                            <div class="stat-card">
                                <div class="stat-icon" style="background: var(--success);">
                                    <i class="fas fa-calendar-day"></i>
                                </div>
                                <div class="stat-number"><?php echo $stats['today_appointments']; ?></div>
                                <div class="stat-label">Heutige Termine</div>
                            </div>
                            <div class="stat-card">
                                <div class="stat-icon" style="background: var(--primary);">
                                    <i class="fas fa-calendar-alt"></i>
                                </div>
                                <div class="stat-number"><?php echo $stats['total_events']; ?></div>
                                <div class="stat-label">Aktive Events</div>
                            </div>
                            <div class="stat-card">
                                <div class="stat-icon" style="background: var(--accent);">
                                    <i class="fas fa-calendar-check"></i>
                                </div>
                                <div class="stat-number"><?php echo $stats['total_appointments']; ?></div>
                                <div class="stat-label">Alle Termine</div>
                            </div>
                            <div class="stat-card">
                                <div class="stat-icon" style="background: var(--warning);">
                                    <i class="fas fa-warehouse"></i>
                                </div>
                                <div class="stat-number">-</div>
                                <div class="stat-label">Lageraktivitäten</div>
                            </div>
                        </div>
                    </div>

                    <!-- Today's Appointments -->
                    <?php if (isWidgetEnabled('today_appointments', $user_dashboard_config)): ?>
                        <div class="widget fade-in">
                            <div class="widget-header">
                                <h3 class="widget-title">
                                    <div class="widget-icon" style="background: var(--success);">
                                        <i class="fas fa-calendar-day"></i>
                                    </div>
                                    Heutige Termine
                                </h3>
                                <div class="widget-actions">
                                    <a href="appointments.php?filter=today" class="widget-btn">
                                        <i class="fas fa-external-link-alt"></i>
                                    </a>
                                </div>
                            </div>
                            <?php if (empty($todayAppointments)): ?>
                                <div style="text-align: center; padding: var(--space-8); color: rgba(255, 255, 255, 0.8);">
                                    <i class="fas fa-calendar-times" style="font-size: 2rem; margin-bottom: var(--space-4); opacity: 0.5;"></i>
                                    <p>Keine Termine für heute</p>
                                </div>
                            <?php else: ?>
                                <?php foreach ($todayAppointments as $appointment): ?>
                                    <div class="activity-item">
                                        <div class="activity-icon" style="background: var(--success);">
                                            <i class="fas fa-user"></i>
                                        </div>
                                        <div class="activity-content">
                                            <div class="activity-title"><?php echo htmlspecialchars($appointment['customer_name']); ?></div>
                                            <div class="activity-description"><?php echo htmlspecialchars($appointment['license_plate']); ?></div>
                                        </div>
                                        <div class="activity-time">
                                            <?php echo date('H:i', strtotime($appointment['slot_time'])); ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>

                    <!-- Upcoming Events -->
                    <?php if (isWidgetEnabled('upcoming_events', $user_dashboard_config)): ?>
                        <div class="widget fade-in">
                            <div class="widget-header">
                                <h3 class="widget-title">
                                    <div class="widget-icon" style="background: var(--primary);">
                                        <i class="fas fa-calendar-alt"></i>
                                    </div>
                                    Kommende Events
                                </h3>
                                <div class="widget-actions">
                                    <a href="events.php" class="widget-btn">
                                        <i class="fas fa-external-link-alt"></i>
                                    </a>
                                </div>
                            </div>
                            <?php if (empty($upcomingEvents)): ?>
                                <div style="text-align: center; padding: var(--space-8); color: rgba(255, 255, 255, 0.8);">
                                    <i class="fas fa-calendar-plus" style="font-size: 2rem; margin-bottom: var(--space-4); opacity: 0.5;"></i>
                                    <p>Keine kommenden Events</p>
                                    <a href="events.php?action=create" class="btn-welcome" style="margin-top: var(--space-4);">
                                        <i class="fas fa-plus"></i>
                                        Event erstellen
                                    </a>
                                </div>
                            <?php else: ?>
                                <?php foreach ($upcomingEvents as $event): ?>
                                    <div class="activity-item">
                                        <div class="activity-icon" style="background: var(--primary);">
                                            <i class="fas fa-calendar"></i>
                                        </div>
                                        <div class="activity-content">
                                            <div class="activity-title"><?php echo htmlspecialchars($event['name']); ?></div>
                                            <div class="activity-description"><?php echo date('d.m.Y', strtotime($event['event_date'])); ?></div>
                                        </div>
                                        <div class="activity-time">
                                            <?php echo $event['booked_count']; ?>/<?php echo $event['appointment_count']; ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                </div>

                <!-- Sidebar Content -->
                <div class="sidebar-content">
        <!-- REMOVED: Scoreboard has been removed as requested -->

                    <!-- Quick Actions -->
                    <?php if (isWidgetEnabled('quick_actions', $user_dashboard_config)): ?>
                        <div class="widget slide-in">
                            <div class="widget-header">
                                <h3 class="widget-title">
                                    <div class="widget-icon" style="background: var(--secondary);">
                                        <i class="fas fa-rocket"></i>
                                    </div>
                                    Schnellaktionen
                                </h3>
                                <div class="widget-actions">
                                    <a href="quick_actions_manager.php" class="widget-btn" title="Schnellaktionen verwalten">
                                        <i class="fas fa-cog"></i>
                                    </a>
                                </div>
                            </div>
                            <div class="quick-actions-grid">
                                <?php if (empty($quick_actions)): ?>
                                    <div class="quick-action-empty">
                                        <div class="quick-action-icon" style="background: var(--secondary);">
                                            <i class="fas fa-plus"></i>
                                        </div>
                                        <div class="quick-action-title">
                                            <a href="quick_actions_manager.php" style="color: inherit; text-decoration: none;">
                                                Erste Aktion hinzufügen
                                            </a>
                                        </div>
                                    </div>
                                <?php else: ?>
                                    <?php foreach ($quick_actions as $action): ?>
                                        <a href="<?php echo htmlspecialchars($action['url']); ?>" class="quick-action">
                                            <div class="quick-action-icon" style="background: <?php echo htmlspecialchars($action['background_color'] ?? $action['color'] ?? '#007AFF'); ?>;">
                                                <i class="<?php echo htmlspecialchars($action['icon']); ?>"></i>
                                            </div>
                                            <div class="quick-action-title"><?php echo htmlspecialchars($action['title']); ?></div>
                                        </a>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endif; ?>

                    <!-- REMOVED: Performance Metrics widget has been removed as requested -->

                    <!-- Recent Storage -->
                    <?php if (isWidgetEnabled('recent_storage', $user_dashboard_config)): ?>
                        <div class="widget slide-in">
                            <div class="widget-header">
                                <h3 class="widget-title">
                                    <div class="widget-icon" style="background: var(--warning);">
                                        <i class="fas fa-warehouse"></i>
                                    </div>
                                    Letzte Lageraktivitäten
                                </h3>
                                <div class="widget-actions">
                                    <a href="storage.php" class="widget-btn">
                                        <i class="fas fa-external-link-alt"></i>
                                    </a>
                                </div>
                            </div>
                            <?php if (empty($recentStorage)): ?>
                                <div style="text-align: center; padding: var(--space-8); color: rgba(255, 255, 255, 0.8);">
                                    <i class="fas fa-warehouse" style="font-size: 2rem; margin-bottom: var(--space-4); opacity: 0.5;"></i>
                                    <p>Keine Lageraktivitäten</p>
                                </div>
                            <?php else: ?>
                                <?php foreach ($recentStorage as $storage): ?>
                                    <div class="activity-item">
                                        <div class="activity-icon" style="background: var(--warning);">
                                            <i class="fas fa-box"></i>
                                        </div>
                                        <div class="activity-content">
                                            <div class="activity-title"><?php echo htmlspecialchars($storage['item_name'] ?? 'Unbekannt'); ?></div>
                                            <div class="activity-description">Einlagerung</div>
                                        </div>
                                        <div class="activity-time">
                                            <?php echo date('d.m', strtotime($storage['created_at'])); ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>

    <!-- Setup Modal -->
    <?php if ($show_setup_modal): ?>
        <div id="setupModal" class="setup-modal-overlay">
            <div class="setup-modal">
                <div class="setup-modal-header">
                    <h2>🚀 Dashboard Setup</h2>
                    <p>Lassen Sie uns Ihr persönliches Dashboard einrichten!</p>
                </div>
                
                <form method="POST" id="dashboardSetupForm">
                    <input type="hidden" name="action" value="save_dashboard_config">
                    
                    <!-- Layout Selection -->
                    <div class="setup-section">
                        <h3>Layout auswählen</h3>
                        <div class="layout-grid">
                            <?php foreach ($available_layouts as $layout_id => $layout): ?>
                                <div class="layout-card <?php echo $modal_config['layout'] === $layout_id ? 'selected' : ''; ?>" 
                                     data-layout="<?php echo $layout_id; ?>">
                                    <div class="layout-preview <?php echo $layout_id; ?>"></div>
                                    <div class="layout-name"><?php echo htmlspecialchars($layout['name']); ?></div>
                                    <div class="layout-description"><?php echo htmlspecialchars($layout['description']); ?></div>
                                    <div class="layout-meta">
                                        <span class="layout-tag"><?php echo htmlspecialchars($layout['best_for']); ?></span>
                                        <span><?php echo htmlspecialchars($layout['screen_size']); ?></span>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                        <input type="hidden" name="layout" id="selectedLayout" value="<?php echo htmlspecialchars($modal_config['layout']); ?>">
                    </div>

                    <!-- Widget Selection -->
                    <div class="setup-section">
                        <h3>Widgets auswählen</h3>
                        <div class="widget-grid">
                            <?php foreach ($available_widgets as $widget_id => $widget): ?>
                                <div class="widget-card <?php echo in_array($widget_id, $modal_config['widgets']) ? 'selected' : ''; ?>" 
                                     data-widget="<?php echo $widget_id; ?>">
                                    <div class="widget-header">
                                        <div class="widget-icon" style="background: <?php echo htmlspecialchars($widget['color']); ?>;">
                                            <i class="<?php echo htmlspecialchars($widget['icon']); ?>"></i>
                                        </div>
                                        <div class="widget-info">
                                            <h4><?php echo htmlspecialchars($widget['title']); ?></h4>
                                            <p><?php echo htmlspecialchars($widget['description']); ?></p>
                                        </div>
                                        <span class="widget-importance importance-<?php echo $widget['importance']; ?>">
                                            <?php echo $widget['importance']; ?>
                                        </span>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                        <input type="hidden" name="widgets" id="selectedWidgets" value="<?php echo htmlspecialchars(json_encode($modal_config['widgets'])); ?>">
                    </div>

                    <!-- Quick Actions Setup -->
                    <!-- Quick Actions Manager Link -->
                    <div class="setup-section">
                        <h3>Schnellaktionen verwalten</h3>
                        <p>Verwalte deine personalisierten Schnellaktionen über den Quick Actions Manager.</p>
                        <a href="quick_actions_manager.php" class="btn btn-primary" style="display: inline-flex; align-items: center; gap: var(--space-2);">
                            <i class="fas fa-rocket"></i>
                            Schnellaktionen Manager öffnen
                        </a>
                    </div>

                    <!-- Additional Settings -->
                    <div class="setup-section">
                        <h3>Weitere Einstellungen</h3>
                        
                        <div class="form-group">
                            <label>Design-Theme</label>
                            <select name="theme" class="form-control">
                                <option value="light" <?php echo $modal_config['theme'] === 'light' ? 'selected' : ''; ?>>Hell</option>
                                <option value="dark" <?php echo $modal_config['theme'] === 'dark' ? 'selected' : ''; ?>>Dunkel</option>
                            </select>
                        </div>

                        <div class="checkbox-group">
                            <div class="checkbox <?php echo $modal_config['show_welcome'] ? 'checked' : ''; ?>" data-checkbox="show_welcome"></div>
                            <label class="checkbox-label">Willkommensnachricht anzeigen</label>
                        </div>
                        <input type="hidden" name="show_welcome" id="showWelcome" value="<?php echo $modal_config['show_welcome'] ? '1' : '0'; ?>">

                        <div class="checkbox-group">
                            <div class="checkbox <?php echo $modal_config['auto_refresh'] ? 'checked' : ''; ?>" data-checkbox="auto_refresh"></div>
                            <label class="checkbox-label">Automatische Aktualisierung</label>
                        </div>
                        <input type="hidden" name="auto_refresh" id="autoRefresh" value="<?php echo $modal_config['auto_refresh'] ? '1' : '0'; ?>">

                        <div class="form-group">
                            <label>Aktualisierungsintervall (Sekunden)</label>
                            <select name="refresh_interval" class="form-control">
                                <option value="60" <?php echo $modal_config['refresh_interval'] == 60 ? 'selected' : ''; ?>>1 Minute</option>
                                <option value="300" <?php echo $modal_config['refresh_interval'] == 300 ? 'selected' : ''; ?>>5 Minuten</option>
                                <option value="600" <?php echo $modal_config['refresh_interval'] == 600 ? 'selected' : ''; ?>>10 Minuten</option>
                                <option value="1800" <?php echo $modal_config['refresh_interval'] == 1800 ? 'selected' : ''; ?>>30 Minuten</option>
                            </select>
                        </div>
                    </div>

                    <!-- Action Buttons -->
                    <div class="setup-actions">
                        <button type="button" class="btn btn-secondary" onclick="closeSetupModal()">
                            <i class="fas fa-times"></i>
                            Überspringen
                        </button>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save"></i>
                            Dashboard einrichten
                        </button>
                    </div>
                </form>
            </div>
        </div>
    <?php endif; ?>

    <!-- Score Details Modal -->
    <div id="scoreDetailsModal" class="modal" style="display: none;">
        <div class="modal-content" style="max-width: 800px;">
                    <div class="modal-header">
                        <h3>🎯 Ihr persönlicher Performance-Score</h3>
                        <button class="modal-close" onclick="closeScoreDetails()">&times;</button>
                    </div>
            <div class="modal-form">
                <div class="score-details-content">
                    <!-- Termine Details -->
                    <div class="score-category">
                        <div class="category-header">
                            <div class="category-icon" style="background: #10b981;">
                                <i class="fas fa-calendar-check"></i>
                            </div>
                            <div class="category-info">
                                <h4>📅 Termine & Kundenbetreuung</h4>
                                <p>Wie gut verwalten Sie Termine und betreuen Ihre Kunden?</p>
                            </div>
                            <div class="category-score"><?php echo round($score_breakdown['appointments'], 1); ?>/40</div>
                        </div>
                        <div class="category-details">
                            <div class="detail-item">
                                <span class="detail-label">📊 Alle Termine (letzte 3 Monate):</span>
                                <span class="detail-value"><?php echo $score_details['appointments']['total_appointments'] ?? 0; ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">✅ Erfolgreich abgeschlossen:</span>
                                <span class="detail-value"><?php echo $score_details['appointments']['completed_appointments'] ?? 0; ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">🎯 Erfolgsquote:</span>
                                <span class="detail-value"><?php echo $score_details['appointments']['completion_rate'] ?? 0; ?>%</span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">🕒 Aktuelle Aktivität (letzter Monat):</span>
                                <span class="detail-value"><?php echo $score_details['appointments']['recent_appointments'] ?? 0; ?></span>
                            </div>
                        </div>
                    </div>

                    <!-- Events Details -->
                    <div class="score-category">
                        <div class="category-header">
                            <div class="category-icon" style="background: #3b82f6;">
                                <i class="fas fa-calendar-alt"></i>
                            </div>
                            <div class="category-info">
                                <h4>🎪 Events & Veranstaltungen</h4>
                                <p>Wie aktiv sind Sie bei der Organisation von Events?</p>
                            </div>
                            <div class="category-score"><?php echo round($score_breakdown['events'], 1); ?>/25</div>
                        </div>
                        <div class="category-details">
                            <div class="detail-item">
                                <span class="detail-label">📊 Alle Events (letzte 3 Monate):</span>
                                <span class="detail-value"><?php echo $score_details['events']['total_events'] ?? 0; ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">🟢 Aktuell laufende Events:</span>
                                <span class="detail-value"><?php echo $score_details['events']['active_events'] ?? 0; ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">📅 Geplante Events:</span>
                                <span class="detail-value"><?php echo $score_details['events']['upcoming_events'] ?? 0; ?></span>
                            </div>
                        </div>
                    </div>

                    <!-- Lager Details -->
                    <div class="score-category">
                        <div class="category-header">
                            <div class="category-icon" style="background: #f59e0b;">
                                <i class="fas fa-warehouse"></i>
                            </div>
                            <div class="category-info">
                                <h4>📦 Lager & Einlagerungen</h4>
                                <p>Wie gut organisieren Sie das Lager und die Einlagerungen?</p>
                            </div>
                            <div class="category-score"><?php echo round($score_breakdown['storage'], 1); ?>/15</div>
                        </div>
                        <div class="category-details">
                            <div class="detail-item">
                                <span class="detail-label">📊 Alle Lageraktivitäten (letzte 3 Monate):</span>
                                <span class="detail-value"><?php echo $score_details['storage']['total_activities'] ?? 0; ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">🕒 Aktuelle Aktivitäten (letzter Monat):</span>
                                <span class="detail-value"><?php echo $score_details['storage']['recent_activities'] ?? 0; ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">🏷️ Verschiedene Lagerplätze:</span>
                                <span class="detail-value"><?php echo $score_details['storage']['unique_storage_numbers'] ?? 0; ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">👥 Verschiedene Kunden:</span>
                                <span class="detail-value"><?php echo $score_details['storage']['unique_customers'] ?? 0; ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">📥 Eingelagerte Artikel:</span>
                                <span class="detail-value"><?php echo $score_details['storage']['stored_items'] ?? 0; ?></span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">📤 Abgeholte Artikel:</span>
                                <span class="detail-value"><?php echo $score_details['storage']['retrieved_items'] ?? 0; ?></span>
                            </div>
                        </div>
                    </div>

                    <!-- Effizienz Details -->
                    <div class="score-category">
                        <div class="category-header">
                            <div class="category-icon" style="background: #8b5cf6;">
                                <i class="fas fa-tachometer-alt"></i>
                            </div>
                            <div class="category-info">
                                <h4>⚡ Arbeits-Effizienz</h4>
                                <p>Wie schnell und qualitativ arbeiten Sie?</p>
                            </div>
                            <div class="category-score"><?php echo round($score_breakdown['efficiency'], 1); ?>/10</div>
                        </div>
                        <div class="category-details">
                            <div class="detail-item">
                                <span class="detail-label">🎯 Aufgaben-Abschlussrate:</span>
                                <span class="detail-value"><?php echo $score_details['efficiency']['completion_rate'] ?? 0; ?>%</span>
                            </div>
                            <div class="detail-item">
                                <span class="detail-label">🕒 Aktuelle Arbeitsaktivität:</span>
                                <span class="detail-value"><?php echo $score_details['efficiency']['recent_activity'] ?? 0; ?> Termine</span>
                            </div>
                        </div>
                    </div>

                    <!-- Konsistenz Details -->
                    <div class="score-category">
                        <div class="category-header">
                            <div class="category-icon" style="background: #06b6d4;">
                                <i class="fas fa-chart-line"></i>
                            </div>
                            <div class="category-info">
                                <h4>📈 Arbeits-Konsistenz</h4>
                                <p>Wie regelmäßig und kontinuierlich arbeiten Sie?</p>
                            </div>
                            <div class="category-score"><?php echo round($score_breakdown['consistency'], 1); ?>/10</div>
                        </div>
                        <div class="category-details">
                            <div class="detail-item">
                                <span class="detail-label">📊 Regelmäßige Arbeitsaktivität:</span>
                                <span class="detail-value"><?php echo $score_details['consistency']['activity_consistency'] ?? 0; ?>%</span>
                            </div>
                        </div>
                    </div>

                    <!-- Score Tipps -->
                    <div class="score-tips">
                        <h4>💡 So verbessern Sie Ihren Score:</h4>
                        <ul>
                            <li><strong>📅 Termine & Kundenbetreuung:</strong> Schließen Sie mehr Termine erfolgreich ab und reduzieren Sie Stornierungen. Betreuen Sie Ihre Kunden aktiv.</li>
                            <li><strong>🎪 Events & Veranstaltungen:</strong> Erstellen Sie regelmäßig neue Events und halten Sie sie aktiv. Planen Sie Veranstaltungen im Voraus.</li>
                            <li><strong>📦 Lager & Einlagerungen:</strong> Dokumentieren Sie alle Lageraktivitäten regelmäßig. Organisieren Sie das Lager effizient.</li>
                            <li><strong>⚡ Arbeits-Effizienz:</strong> Arbeiten Sie kontinuierlich und schließen Sie Aufgaben schnell ab. Verbessern Sie Ihre Arbeitsgeschwindigkeit.</li>
                            <li><strong>📈 Arbeits-Konsistenz:</strong> Bleiben Sie regelmäßig aktiv und arbeiten Sie kontinuierlich. Vermeiden Sie längere Pausen.</li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- JavaScript -->
    <script src="../assets/js/main.js"></script>
    <script src="assets/js/admin.js"></script>
    <script src="assets/js/unified-sidebar.js"></script>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            console.log('🚀 Intelligent Dashboard loaded');
            
            // Auto-refresh if enabled
            <?php if ($auto_refresh): ?>
                setInterval(() => {
                    refreshDashboard();
                }, <?php echo $refresh_interval * 1000; ?>);
            <?php endif; ?>
            
            // Animate elements on load
            const elements = document.querySelectorAll('.fade-in, .slide-in');
            elements.forEach((element, index) => {
                element.style.animationDelay = `${index * 0.1}s`;
            });
            
            // Initialize setup modal for new users
            <?php if ($show_setup_modal): ?>
                console.log('New user detected, initializing setup modal...');
                setTimeout(() => {
                    initializeSetupModal();
                }, 500);
            <?php endif; ?>
        });

        function refreshDashboard() {
            location.reload();
        }

        function refreshStats() {
            const btn = event.target.closest('.widget-btn');
            btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
            
            setTimeout(() => {
                location.reload();
            }, 1000);
        }

        function toggleTheme() {
            const currentTheme = document.documentElement.getAttribute('data-theme');
            const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
            
            document.documentElement.setAttribute('data-theme', newTheme);
            
            // Save theme preference
            fetch('save_theme_preference.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `theme=${newTheme}`
            });
        }

        function hideWelcome() {
            const welcomeMessage = document.querySelector('.welcome-message');
            welcomeMessage.style.animation = 'fadeOut 0.3s ease-in';
            setTimeout(() => {
                welcomeMessage.remove();
            }, 300);
        }

        // Add CSS for fadeOut animation
        const style = document.createElement('style');
        style.textContent = `
            @keyframes fadeOut {
                from { opacity: 1; transform: translateY(0); }
                to { opacity: 0; transform: translateY(-20px); }
            }
        `;
        document.head.appendChild(style);

        // Setup Modal Functions
        function openSetupModal() {
            // Get current user config from PHP
            const currentConfig = {
                layout: '<?php echo $user_dashboard_config['layout']; ?>',
                widgets: <?php echo json_encode($user_dashboard_config['widgets']); ?>,
                quick_actions: [], // Quick Actions are now managed via quick_actions_manager.php
                theme: '<?php echo $user_dashboard_config['theme']; ?>',
                show_welcome: <?php echo $user_dashboard_config['show_welcome'] ? 'true' : 'false'; ?>,
                auto_refresh: <?php echo $user_dashboard_config['auto_refresh'] ? 'true' : 'false'; ?>,
                refresh_interval: <?php echo $user_dashboard_config['refresh_interval']; ?>
            };
            
            // Create modal dynamically with current config
            const modalHTML = `
                <div id="setupModal" class="setup-modal-overlay">
                    <div class="setup-modal">
                        <div class="setup-modal-header">
                            <h2>⚙️ Dashboard Konfiguration</h2>
                            <p>Passen Sie Ihr Dashboard nach Ihren Wünschen an</p>
                        </div>
                        
                        <form method="POST" id="dashboardSetupForm">
                            <input type="hidden" name="action" value="save_dashboard_config">
                            
                            <!-- Layout Selection -->
                            <div class="setup-section">
                                <h3>Layout auswählen</h3>
                                <div class="layout-grid">
                                    <div class="layout-card ${currentConfig.layout === 'default' ? 'selected' : ''}" data-layout="default">
                                        <div class="layout-preview default"></div>
                                        <div class="layout-name">Standard Layout</div>
                                        <div class="layout-description">Zwei-Spalten Layout mit Hauptinhalt und Sidebar</div>
                                        <div class="layout-meta">
                                            <span class="layout-tag">Tägliche Arbeit</span>
                                            <span>Desktop & Tablet</span>
                                        </div>
                                    </div>
                                    <div class="layout-card ${currentConfig.layout === 'compact' ? 'selected' : ''}" data-layout="compact">
                                        <div class="layout-preview compact"></div>
                                        <div class="layout-name">Kompaktes Layout</div>
                                        <div class="layout-description">Ein-Spalten Layout für fokussierte Arbeit</div>
                                        <div class="layout-meta">
                                            <span class="layout-tag">Mobile Geräte</span>
                                            <span>Mobile & Tablet</span>
                                        </div>
                                    </div>
                                    <div class="layout-card ${currentConfig.layout === 'wide' ? 'selected' : ''}" data-layout="wide">
                                        <div class="layout-preview wide"></div>
                                        <div class="layout-name">Breites Layout</div>
                                        <div class="layout-description">Drei-Spalten Layout für maximale Übersicht</div>
                                        <div class="layout-meta">
                                            <span class="layout-tag">Administratoren</span>
                                            <span>Desktop & Ultrawide</span>
                                        </div>
                                    </div>
                                </div>
                                <input type="hidden" name="layout" id="selectedLayout" value="${currentConfig.layout}">
                            </div>

                            <!-- Widget Selection -->
                            <div class="setup-section">
                                <h3>Widgets auswählen</h3>
                                <div class="widget-grid">
                                    <div class="widget-card ${currentConfig.widgets.includes('today_appointments') ? 'selected' : ''}" data-widget="today_appointments">
                                        <div class="widget-header">
                                            <div class="widget-icon" style="background: #007AFF;">
                                                <i class="fas fa-calendar-day"></i>
                                            </div>
                                            <div class="widget-info">
                                                <h4>Heutige Termine</h4>
                                                <p>Zeigt alle Termine für heute an</p>
                                            </div>
                                            <span class="widget-importance importance-high">high</span>
                                        </div>
                                    </div>
                                    <div class="widget-card ${currentConfig.widgets.includes('upcoming_events') ? 'selected' : ''}" data-widget="upcoming_events">
                                        <div class="widget-header">
                                            <div class="widget-icon" style="background: #34C759;">
                                                <i class="fas fa-calendar-alt"></i>
                                            </div>
                                            <div class="widget-info">
                                                <h4>Kommende Events</h4>
                                                <p>Zeigt die nächsten Events an</p>
                                            </div>
                                            <span class="widget-importance importance-medium">medium</span>
                                        </div>
                                    </div>
                                    <div class="widget-card ${currentConfig.widgets.includes('quick_actions') ? 'selected' : ''}" data-widget="quick_actions">
                                        <div class="widget-header">
                                            <div class="widget-icon" style="background: #AF52DE;">
                                                <i class="fas fa-rocket"></i>
                                            </div>
                                            <div class="widget-info">
                                                <h4>Schnellaktionen</h4>
                                                <p>Ihre personalisierten Schnellzugriffe</p>
                                            </div>
                                            <span class="widget-importance importance-high">high</span>
                                        </div>
                                    </div>
                                    <div class="widget-card ${currentConfig.widgets.includes('recent_storage') ? 'selected' : ''}" data-widget="recent_storage">
                                        <div class="widget-header">
                                            <div class="widget-icon" style="background: #FF9500;">
                                                <i class="fas fa-warehouse"></i>
                                            </div>
                                            <div class="widget-info">
                                                <h4>Letzte Lageraktivitäten</h4>
                                                <p>Zeigt die neuesten Einlagerungen an</p>
                                            </div>
                                            <span class="widget-importance importance-medium">medium</span>
                                        </div>
                                    </div>
                                    <div class="widget-card ${currentConfig.widgets.includes('performance_metrics') ? 'selected' : ''}" data-widget="performance_metrics">
                                        <div class="widget-header">
                                            <div class="widget-icon" style="background: #5856D6;">
                                                <i class="fas fa-chart-line"></i>
                                            </div>
                                            <div class="widget-info">
                                                <h4>Performance-Metriken</h4>
                                                <p>Zeigt Statistiken und Kennzahlen an</p>
                                            </div>
                                            <span class="widget-importance importance-low">low</span>
                                        </div>
                                    </div>
                                </div>
                                <input type="hidden" name="widgets" id="selectedWidgets" value='${JSON.stringify(currentConfig.widgets)}'>
                            </div>

                            <!-- Quick Actions Setup -->
                            <div class="setup-section">
                                <!-- Quick Actions Manager Link -->
                                <div class="setup-section">
                                    <h3>Schnellaktionen verwalten</h3>
                                    <p>Verwalte deine personalisierten Schnellaktionen über den Quick Actions Manager.</p>
                                    <a href="quick_actions_manager.php" class="btn btn-primary" style="display: inline-flex; align-items: center; gap: var(--space-2);">
                                        <i class="fas fa-rocket"></i>
                                        Schnellaktionen Manager öffnen
                                    </a>
                                </div>
                            </div>

                            <!-- Additional Settings -->
                            <div class="setup-section">
                                <h3>Weitere Einstellungen</h3>
                                
                                <div class="form-group">
                                    <label>Design-Theme</label>
                                    <select name="theme" class="form-control">
                                        <option value="light" ${currentConfig.theme === 'light' ? 'selected' : ''}>Hell</option>
                                        <option value="dark" ${currentConfig.theme === 'dark' ? 'selected' : ''}>Dunkel</option>
                                    </select>
                                </div>

                                <div class="checkbox-group">
                                    <div class="checkbox ${currentConfig.show_welcome ? 'checked' : ''}" data-checkbox="show_welcome"></div>
                                    <label class="checkbox-label">Willkommensnachricht anzeigen</label>
                                </div>
                                <input type="hidden" name="show_welcome" id="showWelcome" value="${currentConfig.show_welcome ? '1' : '0'}">

                                <div class="checkbox-group">
                                    <div class="checkbox ${currentConfig.auto_refresh ? 'checked' : ''}" data-checkbox="auto_refresh"></div>
                                    <label class="checkbox-label">Automatische Aktualisierung</label>
                                </div>
                                <input type="hidden" name="auto_refresh" id="autoRefresh" value="${currentConfig.auto_refresh ? '1' : '0'}">

                                <div class="form-group">
                                    <label>Aktualisierungsintervall (Sekunden)</label>
                                    <select name="refresh_interval" class="form-control">
                                        <option value="60" ${currentConfig.refresh_interval == 60 ? 'selected' : ''}>1 Minute</option>
                                        <option value="300" ${currentConfig.refresh_interval == 300 ? 'selected' : ''}>5 Minuten</option>
                                        <option value="600" ${currentConfig.refresh_interval == 600 ? 'selected' : ''}>10 Minuten</option>
                                        <option value="1800" ${currentConfig.refresh_interval == 1800 ? 'selected' : ''}>30 Minuten</option>
                                    </select>
                                </div>
                            </div>

                            <!-- Action Buttons -->
                            <div class="setup-actions">
                                <button type="button" class="btn btn-secondary" onclick="closeSetupModal()">
                                    <i class="fas fa-times"></i>
                                    Abbrechen
                                </button>
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save"></i>
                                    Änderungen speichern
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            `;
            
            document.body.insertAdjacentHTML('beforeend', modalHTML);
            
            // Initialize modal event listeners
            initializeSetupModal();
        }

        function closeSetupModal() {
            const modal = document.getElementById('setupModal');
            if (modal) {
                modal.style.animation = 'fadeOut 0.3s ease-in';
                setTimeout(() => {
                    modal.remove();
                }, 300);
            }
        }

        function initializeSetupModal() {
            const modal = document.getElementById('setupModal');
            if (!modal) {
                console.error('Setup modal not found!');
                return;
            }

            console.log('Initializing setup modal...');
            console.log('Modal found:', modal);
            console.log('Modal children:', modal.children.length);

            // Layout selection
            const layoutCards = modal.querySelectorAll('.layout-card');
            const selectedLayoutInput = modal.querySelector('#selectedLayout');
            
            console.log('Found layout cards:', layoutCards.length);
            
            layoutCards.forEach((card, index) => {
                console.log(`Setting up layout card ${index}:`, card);
                // Remove existing listeners
                card.removeEventListener('click', handleLayoutClick);
                // Add new listener
                card.addEventListener('click', handleLayoutClick);
                console.log(`Layout card ${index} event listener added`);
            });

            function handleLayoutClick(e) {
                e.preventDefault();
                e.stopPropagation();
                console.log('Layout card clicked:', this.dataset.layout);
                layoutCards.forEach(c => c.classList.remove('selected'));
                this.classList.add('selected');
                selectedLayoutInput.value = this.dataset.layout;
            }

            // Widget selection
            const widgetCards = modal.querySelectorAll('.widget-card');
            const selectedWidgetsInput = modal.querySelector('#selectedWidgets');
            let selectedWidgets = JSON.parse(selectedWidgetsInput.value || '[]');
            
            console.log('Found widget cards:', widgetCards.length);
            
            widgetCards.forEach((card, index) => {
                console.log(`Setting up widget card ${index}:`, card);
                card.removeEventListener('click', handleWidgetClick);
                card.addEventListener('click', handleWidgetClick);
                console.log(`Widget card ${index} event listener added`);
            });

            function handleWidgetClick(e) {
                e.preventDefault();
                e.stopPropagation();
                const widgetId = this.dataset.widget;
                console.log('Widget card clicked:', widgetId);
                
                if (this.classList.contains('selected')) {
                    this.classList.remove('selected');
                    selectedWidgets = selectedWidgets.filter(id => id !== widgetId);
                } else {
                    this.classList.add('selected');
                    selectedWidgets.push(widgetId);
                }
                
                selectedWidgetsInput.value = JSON.stringify(selectedWidgets);
            }

            // Quick actions selection
            const quickActionCards = modal.querySelectorAll('.quick-action-card');
            const selectedQuickActionsInput = modal.querySelector('#selectedQuickActions');
            let selectedQuickActions = JSON.parse(selectedQuickActionsInput.value || '[]');
            
            console.log('Found quick action cards:', quickActionCards.length);
            
            quickActionCards.forEach((card, index) => {
                console.log(`Setting up quick action card ${index}:`, card);
                card.removeEventListener('click', handleQuickActionClick);
                card.addEventListener('click', handleQuickActionClick);
                console.log(`Quick action card ${index} event listener added`);
            });

            function handleQuickActionClick(e) {
                e.preventDefault();
                e.stopPropagation();
                const actionId = this.dataset.action;
                console.log('Quick action card clicked:', actionId);
                
                if (this.classList.contains('selected')) {
                    this.classList.remove('selected');
                    selectedQuickActions = selectedQuickActions.filter(id => id !== actionId);
                } else {
                    this.classList.add('selected');
                    selectedQuickActions.push(actionId);
                }
                
                selectedQuickActionsInput.value = JSON.stringify(selectedQuickActions);
            }

            // Checkbox handling
            const checkboxes = modal.querySelectorAll('.checkbox');
            console.log('Found checkboxes:', checkboxes.length);
            
            checkboxes.forEach((checkbox, index) => {
                console.log(`Setting up checkbox ${index}:`, checkbox);
                checkbox.removeEventListener('click', handleCheckboxClick);
                checkbox.addEventListener('click', handleCheckboxClick);
                console.log(`Checkbox ${index} event listener added`);
            });

            function handleCheckboxClick(e) {
                e.preventDefault();
                e.stopPropagation();
                const checkboxType = this.dataset.checkbox;
                const input = modal.querySelector(`#${checkboxType}`);
                console.log('Checkbox clicked:', checkboxType);
                
                if (this.classList.contains('checked')) {
                    this.classList.remove('checked');
                    input.value = '0';
                } else {
                    this.classList.add('checked');
                    input.value = '1';
                }
            }

            // Form submission
            const form = modal.querySelector('#dashboardSetupForm');
            if (form) {
                form.addEventListener('submit', function(e) {
                    e.preventDefault();
                    
                    const submitBtn = this.querySelector('button[type="submit"]');
                    const originalText = submitBtn.innerHTML;
                    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Speichern...';
                    submitBtn.disabled = true;
                    
                    setTimeout(() => {
                        this.submit();
                    }, 500);
                });
            }

            // Close modal on overlay click
            modal.addEventListener('click', function(e) {
                if (e.target === modal) {
                    console.log('Overlay clicked, closing modal');
                    closeSetupModal();
                }
            });

            console.log('Setup modal initialized successfully');
        }

        // Score Details Functions - Nur für Admin Deniz Uludag
        function showScoreDetails() {
            // Check if current user is admin Deniz Uludag
            <?php if ($current_user_role !== 'admin' || $current_user_name !== 'Deniz Uludag'): ?>
            console.log('❌ Access denied: Only admin Deniz Uludag can view score details');
            alert('Zugriff verweigert: Nur Admin Deniz Uludag kann Score-Details anzeigen');
            return;
            <?php endif; ?>
            
            console.log('🎯 Opening score details modal...');
            const modal = document.getElementById('scoreDetailsModal');
            if (modal) {
                console.log('✅ Modal found:', modal);
                modal.style.display = 'block';
                modal.style.opacity = '0';
                modal.style.transform = 'scale(0.9)';
                
                // Smooth fade in animation
                setTimeout(() => {
                    modal.style.transition = 'all 0.3s ease-out';
                    modal.style.opacity = '1';
                    modal.style.transform = 'scale(1)';
                }, 10);
                
                // Animate score breakdown bars
                setTimeout(() => {
                    const breakdownBars = modal.querySelectorAll('.breakdown-fill');
                    console.log('📊 Animating breakdown bars:', breakdownBars.length);
                    breakdownBars.forEach((bar, index) => {
                        const width = bar.style.width;
                        bar.style.width = '0%';
                        bar.style.transition = 'width 1s ease-out';
                        setTimeout(() => {
                            bar.style.width = width;
                        }, index * 100);
                    });
                }, 300);
            } else {
                console.error('❌ Score details modal not found!');
            }
        }

        function closeScoreDetails() {
            console.log('🎯 Closing score details modal...');
            const modal = document.getElementById('scoreDetailsModal');
            if (modal) {
                modal.style.transition = 'all 0.3s ease-in';
                modal.style.opacity = '0';
                modal.style.transform = 'scale(0.9)';
                setTimeout(() => {
                    modal.style.display = 'none';
                }, 300);
            }
        }

        // Close modal when clicking outside
        window.addEventListener('click', function(event) {
            const modal = document.getElementById('scoreDetailsModal');
            if (event.target === modal) {
                closeScoreDetails();
            }
        });

        // Ensure button click works
        document.addEventListener('DOMContentLoaded', function() {
            const scoreDetailsBtn = document.querySelector('.btn-score-details');
            if (scoreDetailsBtn) {
                console.log('✅ Score details button found');
                scoreDetailsBtn.addEventListener('click', function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    console.log('🎯 Button clicked!');
                    showScoreDetails();
                });
            } else {
                console.log('⚠️ Score details button not found');
            }
        });

        // REMOVED: Scoring JavaScript functions have been removed as requested

    </script>
</body>
</html>
