# SQL Ambiguous Column Fix

## Problem
```
Fatal error: Uncaught PDOException: SQLSTATE[23000]: Integrity constraint violation: 1052 Column 'created_at' in field list is ambiguous
```

## Ursache
Die Spalte `created_at` existiert in beiden Tabellen (`appointments` und `events`), wodurch sie mehrdeutig wird, wenn beide Tabellen in einem JOIN verwendet werden.

## Lösung

### Vorher (fehlerhaft):
```sql
SELECT 
    COUNT(*) as total_appointments,
    COUNT(CASE WHEN status = 'completed' THEN 1 END) as completed_appointments,
    COUNT(CASE WHEN status = 'cancelled' THEN 1 END) as cancelled_appointments,
    COUNT(CASE WHEN DATE(created_at) >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN 1 END) as recent_appointments,
    AVG(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completion_rate
FROM appointments a
LEFT JOIN events e ON a.event_id = e.id
WHERE a.created_at >= DATE_SUB(NOW(), INTERVAL 90 DAY)
AND (e.autohaus_id = ? OR e.autohaus_id IS NULL)
```

### Nachher (korrekt):
```sql
SELECT 
    COUNT(*) as total_appointments,
    COUNT(CASE WHEN a.status = 'completed' THEN 1 END) as completed_appointments,
    COUNT(CASE WHEN a.status = 'cancelled' THEN 1 END) as cancelled_appointments,
    COUNT(CASE WHEN DATE(a.created_at) >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN 1 END) as recent_appointments,
    AVG(CASE WHEN a.status = 'completed' THEN 1 ELSE 0 END) as completion_rate
FROM appointments a
LEFT JOIN events e ON a.event_id = e.id
WHERE a.created_at >= DATE_SUB(NOW(), INTERVAL 90 DAY)
AND (e.autohaus_id = ? OR e.autohaus_id IS NULL)
```

## Änderungen
- ✅ **Tabellen-Aliase hinzugefügt**: `a.status` statt `status`
- ✅ **Spalten-Aliase hinzugefügt**: `a.created_at` statt `created_at`
- ✅ **Eindeutige Referenzen**: Alle Spalten beziehen sich jetzt eindeutig auf die `appointments` Tabelle

## Warum war das nötig?
Beim JOIN von zwei Tabellen mit gleichen Spaltennamen muss explizit angegeben werden, welche Tabelle gemeint ist. Ohne Tabellen-Alias ist die Spalte `created_at` mehrdeutig.

**Das SQL-Problem ist behoben! ✅**
